import { Fragment, ReactNode, RefObject, useState } from "react";
import { Dialog, Transition } from "@headlessui/react";
import classNames from "classnames";
import { XMarkIcon } from "@heroicons/react/24/solid";

export type ModalProps = {
  open: boolean;
  setOpen: (open: boolean) => void;
  children: ReactNode;
  classNames?: string;
  initialFocus?: RefObject<HTMLButtonElement>;
};

export default function Modal(props: ModalProps) {
  return (
    <Transition.Root show={props.open} as={Fragment}>
      <Dialog
        as="div"
        className="relative z-30"
        onClose={props.setOpen}
        initialFocus={props.initialFocus ?? undefined}
      >
        <Transition.Child
          as={Fragment}
          enter="ease-out duration-300"
          enterFrom="opacity-0"
          enterTo="opacity-100"
          leave="ease-in duration-200"
          leaveFrom="opacity-100"
          leaveTo="opacity-0"
        >
          <div className="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity" />
        </Transition.Child>

        <div className="fixed inset-0 z-10 overflow-y-auto" id={"scroll-modal"}>
          <div className="flex min-h-full items-end justify-center p-4 text-center sm:items-center sm:p-0">
            <Transition.Child
              as={Fragment}
              enter="ease-out duration-300"
              enterFrom="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95"
              enterTo="opacity-100 translate-y-0 sm:scale-100"
              leave="ease-in duration-200"
              leaveFrom="opacity-100 translate-y-0 sm:scale-100"
              leaveTo="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95"
            >
              <Dialog.Panel
                className={classNames(
                  "relative transform overflow-hidden rounded-lg bg-gray-800 px-4 pb-4 pt-5 text-left shadow-xl transition-all sm:my-8 sm:w-full sm:p-6",
                  props.classNames ?? "",
                )}
              >
                <button
                  className="z-10 absolute top-2.5 right-2.5 bg-transparent"
                  onClick={() => props.setOpen(false)}
                >
                  <XMarkIcon className="h-8 w-8 text-gray-300 rounded-md hover:bg-gray-900 p-1 transition" />
                </button>
                {props.children}
              </Dialog.Panel>
            </Transition.Child>
          </div>
        </div>
      </Dialog>
    </Transition.Root>
  );
}
