import { describe, expect, it } from "@jest/globals";
import { deduplicateArray, filterKeys } from "../lib/utils";

describe("deduplicateArray", () => {
  it("No duplicate properties:", () => {
    const arr1 = [
      { name: "Bob", age: 30 },
      { name: "Alice", age: 25 },
      { name: "John", age: 18 },
    ];
    const result1 = deduplicateArray(arr1);
    expect(result1).toStrictEqual([
      { name: "Bob", age: 30 },
      { name: "Alice", age: 25 },
      { name: "John", age: 18 },
    ]);
  });

  it("With duplicate properties:", () => {
    const arr2 = [
      { name: "Bob", age: 30 },
      { name: "Bob", age: 30 },
      { name: "Bob", age: 30 },
    ];
    const result2 = deduplicateArray(arr2);
    expect(result2).toStrictEqual({ name: "Bob", age: 30 });
  });

  it("Some duplicate properties:", () => {
    const arr3 = [
      { name: "Bob", age: 30 },
      { name: "Bob", age: 25 },
      { name: "Bob", age: 18 },
    ];
    const result3 = deduplicateArray(arr3);
    expect(result3).toStrictEqual({
      name: "Bob",
      items: [{ age: 30 }, { age: 25 }, { age: 18 }],
    });
  });
  it("Empty array:", () => {
    const arr4: any[] = [];
    const result4 = deduplicateArray(arr4);
    expect(result4).toStrictEqual({});
  });
  it("Numbered array:", () => {
    const arr5 = [1, 2, 3];
    const result5 = deduplicateArray(arr5);
    expect(result5).toStrictEqual([1, 2, 3]);
  });
  it("Complicated array, real data", () => {
    const arr = [
      {
        idproject: 82,
        idclaim: 769,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        mitigationDate: "2022-12-21",
        typeDamage: "Retarp",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInvoice: 103,
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        createdDate: "2023-01-24T15:06:21",
        dueDate: null,
        notes:
          'The PA Bulldog Isabel Solorzano, (954) 637-2669 isabel@bulldogadjusters.com, advised "See scope attached while I continue to fight for more."',
        collectoragent: "Jaris Alvarado",
      },
      {
        idproject: 82,
        idclaim: 463,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        mitigationDate: "2022-12-09",
        typeDamage: "Mitigation",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInvoice: 61,
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        createdDate: "2023-01-24T15:06:28",
        dueDate: null,
        notes:
          'The PA Bulldog Isabel Solorzano, (954) 637-2669 isabel@bulldogadjusters.com, advised "See scope attached while I continue to fight for more."',
        collectoragent: "Jaris Alvarado",
      },
      {
        idproject: 82,
        idclaim: 769,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        mitigationDate: "2022-12-21",
        typeDamage: "Retarp",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInvoice: 103,
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        createdDate: "2023-01-24T15:31:03",
        dueDate: null,
        notes:
          "I called the&nbsp;Columbia Lloyds Insurance Company&nbsp;800-275-6768, the agent gave me the contact information for the adjuster, Angela Refuge 800-2756768 ext 182. arefuge@mdowinsurance.com",
        collectoragent: "Jaris Alvarado",
      },
      {
        idproject: 82,
        idclaim: 463,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        mitigationDate: "2022-12-09",
        typeDamage: "Mitigation",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInvoice: 61,
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        createdDate: "2023-01-24T15:31:07",
        dueDate: null,
        notes:
          "I called the&nbsp;Columbia Lloyds Insurance Company&nbsp;800-275-6768, the agent gave me the contact information for the adjuster, Angela Refuge 800-2756768 ext 182. arefuge@mdowinsurance.com",
        collectoragent: "Jaris Alvarado",
      },
      {
        idproject: 82,
        idclaim: 463,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        mitigationDate: "2022-12-09",
        typeDamage: "Mitigation",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInvoice: 61,
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        createdDate: "2023-01-31T16:28:20",
        dueDate: null,
        notes:
          "I called the adjuster Angela, she asked me to contact the insured regarding payment, she will not discuss any payment with us even though we have a signed contract from the insured.",
        collectoragent: "Jaris Alvarado",
      },
      {
        idproject: 82,
        idclaim: 769,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        mitigationDate: "2022-12-21",
        typeDamage: "Retarp",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInvoice: 103,
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        createdDate: "2023-01-31T16:28:26",
        dueDate: null,
        notes:
          "I called the adjuster Angela, she asked me to contact the insured regarding payment, she will not discuss any payment with us even though we have a signed contract from the insured.",
        collectoragent: "Jaris Alvarado",
      },
      {
        idproject: 82,
        idclaim: 463,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        mitigationDate: "2022-12-09",
        typeDamage: "Mitigation",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInvoice: 61,
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        createdDate: "2023-02-20T11:29:47",
        dueDate: null,
        notes:
          "I called the insurance adjuster Angela Refuge, she told me that I should call the insured directly for any matters related to the claim.&nbsp;",
        collectoragent: "Jaris Alvarado",
      },
      {
        idproject: 82,
        idclaim: 769,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        mitigationDate: "2022-12-21",
        typeDamage: "Retarp",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInvoice: 103,
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        createdDate: "2023-02-20T11:29:52",
        dueDate: null,
        notes:
          "I called the insurance adjuster Angela Refuge, she told me that I should call the insured directly for any matters related to the claim.&nbsp;",
        collectoragent: "Jaris Alvarado",
      },
    ];
    const result4 = deduplicateArray(arr);
    expect(result4).toEqual({
      idproject: 82,
      idcustomer: 3290,
      firstName: "Fernando",
      lastName: "Alvarez",
      fullname: "Fernando Alvarez",
      idpubliccompany: 12,
      idinsurancecompany: 180,
      personalPhone: "(214) 677-3306",
      personalEmail: "Alvarez800do@gmail.com",
      idClaim1: "22-F246419",
      policyNumber: "TXP_F302492-22",
      dol: "2022-04-12",
      propertyAddress: "3332",
      propertyAddress2: "Half Moon Drive",
      unit: "",
      propertyCity: "Fort Worth",
      propertyState: "Texas",
      propertyPostalCode: "76111",
      propertyCountry: "United States",
      insuranceCompany: "Columbia Lloyds Insurance Company",
      publicCompany: "Bulldog Adjusters",
      negotiatedAmount: "",
      receiptAmount: "",
      underpaidAmount: "",
      statusInvoice: "PROCESSING",
      idInsuranceAdjuster: null,
      idPublicAdjuster: null,
      perilName: "Wind & Hail",
      internalId: "RC005-8372805",
      idCompany: "2",
      refBy: "Leonardo Macias",
      idRefBy: 3098,
      registerDate: "2022-11-13",
      statusProject: "approved",
      dueDate: null,
      collectoragent: "Jaris Alvarado",
      items: [
        {
          idclaim: 769,
          mitigationDate: "2022-12-21",
          typeDamage: "Retarp",
          idInvoice: 103,
          createdDate: "2023-01-24T15:06:21",
          notes:
            'The PA Bulldog Isabel Solorzano, (954) 637-2669 isabel@bulldogadjusters.com, advised "See scope attached while I continue to fight for more."',
        },
        {
          idclaim: 463,
          mitigationDate: "2022-12-09",
          typeDamage: "Mitigation",
          idInvoice: 61,
          createdDate: "2023-01-24T15:06:28",
          notes:
            'The PA Bulldog Isabel Solorzano, (954) 637-2669 isabel@bulldogadjusters.com, advised "See scope attached while I continue to fight for more."',
        },
        {
          idclaim: 769,
          mitigationDate: "2022-12-21",
          typeDamage: "Retarp",
          idInvoice: 103,
          createdDate: "2023-01-24T15:31:03",
          notes:
            "I called the&nbsp;Columbia Lloyds Insurance Company&nbsp;800-275-6768, the agent gave me the contact information for the adjuster, Angela Refuge 800-2756768 ext 182. arefuge@mdowinsurance.com",
        },
        {
          idclaim: 463,
          mitigationDate: "2022-12-09",
          typeDamage: "Mitigation",
          idInvoice: 61,
          createdDate: "2023-01-24T15:31:07",
          notes:
            "I called the&nbsp;Columbia Lloyds Insurance Company&nbsp;800-275-6768, the agent gave me the contact information for the adjuster, Angela Refuge 800-2756768 ext 182. arefuge@mdowinsurance.com",
        },
        {
          createdDate: "2023-01-31T16:28:20",
          idInvoice: 61,
          idclaim: 463,
          mitigationDate: "2022-12-09",
          notes:
            "I called the adjuster Angela, she asked me to contact the insured regarding payment, she will not discuss any payment with us even though we have a signed contract from the insured.",
          typeDamage: "Mitigation",
        },
        {
          idclaim: 769,
          mitigationDate: "2022-12-21",
          typeDamage: "Retarp",
          idInvoice: 103,
          createdDate: "2023-01-31T16:28:26",
          notes:
            "I called the adjuster Angela, she asked me to contact the insured regarding payment, she will not discuss any payment with us even though we have a signed contract from the insured.",
        },
        {
          idclaim: 463,
          mitigationDate: "2022-12-09",
          typeDamage: "Mitigation",
          idInvoice: 61,
          createdDate: "2023-02-20T11:29:47",
          notes:
            "I called the insurance adjuster Angela Refuge, she told me that I should call the insured directly for any matters related to the claim.&nbsp;",
        },
        {
          idclaim: 769,
          mitigationDate: "2022-12-21",
          typeDamage: "Retarp",
          idInvoice: 103,
          createdDate: "2023-02-20T11:29:52",
          notes:
            "I called the insurance adjuster Angela Refuge, she told me that I should call the insured directly for any matters related to the claim.&nbsp;",
        },
      ],
    });
  });
});

describe("cutUnecessaryKeys", () => {
  it("basic example", () => {
    const out = filterKeys({ name: "Bob", age: 20, address: "123 Main St." }, [
      "age",
      "address",
    ]);
    expect(out).toEqual({ age: 20, address: "123 Main St." });
  });
  it("basic example + array", () => {
    const out = filterKeys(
      { name: "Bob", age: 20, address: "123 Main St.", items: [1, 2, 3, 4, 5] },
      ["age", "address"],
    );
    expect(out).toEqual({
      age: 20,
      address: "123 Main St.",
      // TODO: We keep all array values which don't contain objects
      items: [1, 2, 3, 4, 5],
    });
  });
  it("basic example + array with nulls", () => {
    const out = filterKeys(
      {
        name: "Bob",
        age: 20,
        address: "123 Main St.",
        items: [1, null, 3, 4, 5],
      },
      ["age", "address"],
    );
    expect(out).toEqual({
      age: 20,
      address: "123 Main St.",
      items: [1, null, 3, 4, 5],
    });
  });
  it("basic example + array of objects", () => {
    const out = filterKeys(
      {
        name: "Bob",
        age: 20,
        address: "123 Main St.",
        items: [{ age: 1 }, { age: 55 }],
      },
      ["age", "address"],
    );
    expect(out).toEqual({
      age: 20,
      address: "123 Main St.",
      items: [{ age: 1 }, { age: 55 }],
    });
  });
  it("basic example + array of objects with keys to be removed", () => {
    const out = filterKeys(
      {
        name: "Bob",
        age: 20,
        address: "123 Main St.",
        items: [
          { age: 1, space: "is cool" },
          { age: 55, notAProblem: "okay" },
        ],
      },
      ["age", "address"],
    );
    expect(out).toEqual({
      age: 20,
      address: "123 Main St.",
      items: [{ age: 1 }, { age: 55 }],
    });
  });
  it("real world example (will probably explode)", () => {
    const out = filterKeys(
      {
        idproject: 82,
        idcustomer: 3290,
        firstName: "Fernando",
        lastName: "Alvarez",
        fullname: "Fernando Alvarez",
        idpubliccompany: 12,
        idinsurancecompany: 180,
        personalPhone: "(214) 677-3306",
        personalEmail: "Alvarez800do@gmail.com",
        idClaim1: "22-F246419",
        policyNumber: "TXP_F302492-22",
        dol: "2022-04-12",
        propertyAddress: "3332",
        propertyAddress2: "Half Moon Drive",
        unit: "",
        propertyCity: "Fort Worth",
        propertyState: "Texas",
        propertyPostalCode: "76111",
        propertyCountry: "United States",
        insuranceCompany: "Columbia Lloyds Insurance Company",
        publicCompany: "Bulldog Adjusters",
        negotiatedAmount: "",
        receiptAmount: "",
        underpaidAmount: "",
        statusInvoice: "PROCESSING",
        idInsuranceAdjuster: null,
        idPublicAdjuster: null,
        perilName: "Wind & Hail",
        internalId: "RC005-8372805",
        idCompany: "2",
        refBy: "Leonardo Macias",
        idRefBy: 3098,
        registerDate: "2022-11-13",
        statusProject: "approved",
        dueDate: null,
        collectoragent: "Jaris Alvarado",
        items: [
          {
            idclaim: 769,
            mitigationDate: "2022-12-21",
            typeDamage: "Retarp",
            idInvoice: 103,
            createdDate: "2023-01-24T15:06:21",
            notes:
              'The PA Bulldog Isabel Solorzano, (954) 637-2669 isabel@bulldogadjusters.com, advised "See scope attached while I continue to fight for more."',
          },
          {
            idclaim: 463,
            mitigationDate: "2022-12-09",
            typeDamage: "Mitigation",
            idInvoice: 61,
            createdDate: "2023-01-24T15:06:28",
            notes:
              'The PA Bulldog Isabel Solorzano, (954) 637-2669 isabel@bulldogadjusters.com, advised "See scope attached while I continue to fight for more."',
          },
          {
            idclaim: 769,
            mitigationDate: "2022-12-21",
            typeDamage: "Retarp",
            idInvoice: 103,
            createdDate: "2023-01-24T15:31:03",
            notes:
              "I called the&nbsp;Columbia Lloyds Insurance Company&nbsp;800-275-6768, the agent gave me the contact information for the adjuster, Angela Refuge 800-2756768 ext 182. arefuge@mdowinsurance.com",
          },
          {
            idclaim: 463,
            mitigationDate: "2022-12-09",
            typeDamage: "Mitigation",
            idInvoice: 61,
            createdDate: "2023-01-24T15:31:07",
            notes:
              "I called the&nbsp;Columbia Lloyds Insurance Company&nbsp;800-275-6768, the agent gave me the contact information for the adjuster, Angela Refuge 800-2756768 ext 182. arefuge@mdowinsurance.com",
          },
          {
            createdDate: "2023-01-31T16:28:20",
            idInvoice: 61,
            idclaim: 463,
            mitigationDate: "2022-12-09",
            notes:
              "I called the adjuster Angela, she asked me to contact the insured regarding payment, she will not discuss any payment with us even though we have a signed contract from the insured.",
            typeDamage: "Mitigation",
          },
          {
            idclaim: 769,
            mitigationDate: "2022-12-21",
            typeDamage: "Retarp",
            idInvoice: 103,
            createdDate: "2023-01-31T16:28:26",
            notes:
              "I called the adjuster Angela, she asked me to contact the insured regarding payment, she will not discuss any payment with us even though we have a signed contract from the insured.",
          },
          {
            idclaim: 463,
            mitigationDate: "2022-12-09",
            typeDamage: "Mitigation",
            idInvoice: 61,
            createdDate: "2023-02-20T11:29:47",
            notes:
              "I called the insurance adjuster Angela Refuge, she told me that I should call the insured directly for any matters related to the claim.&nbsp;",
          },
          {
            idclaim: 769,
            mitigationDate: "2022-12-21",
            typeDamage: "Retarp",
            idInvoice: 103,
            createdDate: "2023-02-20T11:29:52",
            notes:
              "I called the insurance adjuster Angela Refuge, she told me that I should call the insured directly for any matters related to the claim.&nbsp;",
          },
        ],
      },
      [
        "fullname",
        "dol",
        "propertyAddress",
        "propertyAddress2",
        "propertyCity",
        "propertyState",
        "insuranceCompany",
        "publicCompany",
        "negotiatedAmount",
        "receiptAmount",
        "underpaidAmount",
        "statusInvoice",
        "perilName",
        "refBy",
        "statusProject",
        "dueDate",
        "mitigationDate",
        "typeDamage",
        "createdDate",
        "notes",
        "collectoragent",
      ],
    );
    expect(out).toEqual({
      fullname: "Fernando Alvarez",
      dol: "2022-04-12",
      propertyAddress: "3332",
      propertyAddress2: "Half Moon Drive",
      propertyCity: "Fort Worth",
      propertyState: "Texas",
      insuranceCompany: "Columbia Lloyds Insurance Company",
      publicCompany: "Bulldog Adjusters",
      negotiatedAmount: "",
      receiptAmount: "",
      underpaidAmount: "",
      statusInvoice: "PROCESSING",
      perilName: "Wind & Hail",
      refBy: "Leonardo Macias",
      statusProject: "approved",
      dueDate: null,
      collectoragent: "Jaris Alvarado",
      items: [
        {
          mitigationDate: "2022-12-21",
          typeDamage: "Retarp",
          createdDate: "2023-01-24T15:06:21",
          notes:
            'The PA Bulldog Isabel Solorzano, (954) 637-2669 isabel@bulldogadjusters.com, advised "See scope attached while I continue to fight for more."',
        },
        {
          mitigationDate: "2022-12-09",
          typeDamage: "Mitigation",
          createdDate: "2023-01-24T15:06:28",
          notes:
            'The PA Bulldog Isabel Solorzano, (954) 637-2669 isabel@bulldogadjusters.com, advised "See scope attached while I continue to fight for more."',
        },
        {
          mitigationDate: "2022-12-21",
          typeDamage: "Retarp",
          createdDate: "2023-01-24T15:31:03",
          notes:
            "I called the&nbsp;Columbia Lloyds Insurance Company&nbsp;800-275-6768, the agent gave me the contact information for the adjuster, Angela Refuge 800-2756768 ext 182. arefuge@mdowinsurance.com",
        },
        {
          mitigationDate: "2022-12-09",
          typeDamage: "Mitigation",
          createdDate: "2023-01-24T15:31:07",
          notes:
            "I called the&nbsp;Columbia Lloyds Insurance Company&nbsp;800-275-6768, the agent gave me the contact information for the adjuster, Angela Refuge 800-2756768 ext 182. arefuge@mdowinsurance.com",
        },
        {
          createdDate: "2023-01-31T16:28:20",
          mitigationDate: "2022-12-09",
          notes:
            "I called the adjuster Angela, she asked me to contact the insured regarding payment, she will not discuss any payment with us even though we have a signed contract from the insured.",
          typeDamage: "Mitigation",
        },
        {
          mitigationDate: "2022-12-21",
          typeDamage: "Retarp",
          createdDate: "2023-01-31T16:28:26",
          notes:
            "I called the adjuster Angela, she asked me to contact the insured regarding payment, she will not discuss any payment with us even though we have a signed contract from the insured.",
        },
        {
          mitigationDate: "2022-12-09",
          typeDamage: "Mitigation",
          createdDate: "2023-02-20T11:29:47",
          notes:
            "I called the insurance adjuster Angela Refuge, she told me that I should call the insured directly for any matters related to the claim.&nbsp;",
        },
        {
          mitigationDate: "2022-12-21",
          typeDamage: "Retarp",
          createdDate: "2023-02-20T11:29:52",
          notes:
            "I called the insurance adjuster Angela Refuge, she told me that I should call the insured directly for any matters related to the claim.&nbsp;",
        },
      ],
    });
  });
  it("real world example 2", () => {
    const out = filterKeys(
      {
        status: "success",
        message: "Supplier retrieved successfully",
        data: {
          count: 14,
          currentPage: 1,
          totalPages: 1,
          dataHistory: [
            {
              id: "f9211360-32fe-4a69-b7f1-33956242303e",
              username: "TheChefSupplyInc.",
              contact_person: null,
              phone_number: null,
              total_spend: "10565.68",
            },
            {
              id: "3d98755f-eb3b-441c-9908-3757551e78ee",
              username: "AndrewTestVendor",
              contact_person: null,
              phone_number: null,
              total_spend: "2.00",
            },
            {
              id: "df7411c1-8d21-43e2-b8e1-4f02ced07f55",
              username: "CueSuppliesInc",
              contact_person: null,
              phone_number: null,
              total_spend: "7584.42",
            },
            {
              id: "c9de3e16-9696-4879-87c8-8b11c474c72b",
              username: "RoselandProduce",
              contact_person: null,
              phone_number: null,
              total_spend: "64384.86",
            },
            {
              id: "5739de7d-4c05-4092-8429-52dff2c1cf14",
              username: "TripleBogey",
              contact_person: null,
              phone_number: null,
              total_spend: "0.00",
            },
            {
              id: "a58231d1-4da0-41e5-a7ab-82bf9ae29353",
              username: "Sysco",
              contact_person: null,
              phone_number: null,
              total_spend: "1176097.38",
            },
            {
              id: "78620543-8fc2-44c0-ac90-8ece849f497d",
              username: "Stem",
              contact_person: null,
              phone_number: null,
              total_spend: "99559.13",
            },
            {
              id: "72791bc3-1a30-4ae3-a54e-6432b7519d22",
              username: "Andrew'sTestVendor",
              contact_person: null,
              phone_number: null,
              total_spend: "1.00",
            },
            {
              id: "90adaa5b-1c85-4521-96b3-024d64475d54",
              username: null,
              contact_person: null,
              phone_number: null,
              total_spend: "759.17",
            },
            {
              id: "d65f28a0-c46a-49bd-818c-f84a2266928a",
              username: "Triple Bogey",
              contact_person: null,
              phone_number: null,
              total_spend: "10733.11",
            },
            {
              id: "a483ac5f-aaa6-4c25-80ed-527d60305e8b",
              username: "Middlesex",
              contact_person: null,
              phone_number: null,
              total_spend: "80659.92",
            },
            {
              id: "e4535669-0088-4a06-be25-167a59660909",
              username: "Woodward Meats",
              contact_person: null,
              phone_number: null,
              total_spend: "155436.84",
            },
            {
              id: "ca4406df-cd67-417d-825a-3ad7a72c89c2",
              username: "Tab Commerce Services Inc.",
              contact_person: "Tab Commerce Services Inc",
              phone_number: "6474680067",
              total_spend: "0.00",
            },
            {
              id: "822b90ed-bfd4-4b7b-99be-1a33d81a737f",
              username: "WillowRunFoods",
              contact_person: "Tab Commerce Services Inc",
              phone_number: "",
              total_spend: "239.52",
            },
          ],
        },
      },
      ["id", "username"],
    );
    expect(out).toEqual({
      data: {
        dataHistory: [
          {
            id: "f9211360-32fe-4a69-b7f1-33956242303e",
            username: "TheChefSupplyInc.",
          },
          {
            id: "3d98755f-eb3b-441c-9908-3757551e78ee",
            username: "AndrewTestVendor",
          },
          {
            id: "df7411c1-8d21-43e2-b8e1-4f02ced07f55",
            username: "CueSuppliesInc",
          },
          {
            id: "c9de3e16-9696-4879-87c8-8b11c474c72b",
            username: "RoselandProduce",
          },
          {
            id: "5739de7d-4c05-4092-8429-52dff2c1cf14",
            username: "TripleBogey",
          },
          {
            id: "a58231d1-4da0-41e5-a7ab-82bf9ae29353",
            username: "Sysco",
          },
          {
            id: "78620543-8fc2-44c0-ac90-8ece849f497d",
            username: "Stem",
          },
          {
            id: "72791bc3-1a30-4ae3-a54e-6432b7519d22",
            username: "Andrew'sTestVendor",
          },
          {
            id: "90adaa5b-1c85-4521-96b3-024d64475d54",
            username: null,
          },
          {
            id: "d65f28a0-c46a-49bd-818c-f84a2266928a",
            username: "Triple Bogey",
          },
          {
            id: "a483ac5f-aaa6-4c25-80ed-527d60305e8b",
            username: "Middlesex",
          },
          {
            id: "e4535669-0088-4a06-be25-167a59660909",
            username: "Woodward Meats",
          },
          {
            id: "ca4406df-cd67-417d-825a-3ad7a72c89c2",
            username: "Tab Commerce Services Inc.",
          },
          {
            id: "822b90ed-bfd4-4b7b-99be-1a33d81a737f",
            username: "WillowRunFoods",
          },
        ],
      },
    });
  });
});
