---
layout: document
title: "Routing"
date: 2017-03-15 17:28:56
categories: Components
---

## Routing

There are a few mechanisms for routing within the Staple framework.

### MVC Routing

The default routing mechanism in STAPLE is the MVC routing. To access
a controller action you route to it by using the name of the controller
with out the `Controller` suffix and the method name of a public method
in the controller object.

Given the controller:

```php?start_inline=1
class PostsController extends Controller
{
    // URI: /posts/all
    public function all()
    {
        // Show all posts
    }

    // URI: /posts/my-posts
    public function myPosts()
    {
        // Show only my posts.
    }
    
    // URI: /posts/for-user/1849
    public function ForUser($userId)
    {
        // Show only my posts.
    }
}
```

There are three routable actions here. The first to show all posts
is accessible by the URI of `/posts/all`.

The second method is accessible at `posts/my-posts`. The capital
is converted to a lowercase letter and signified with a dash to keep
URIs compatible with different server systems. 

**Note:** All uppercase characters, with the exception of the first 
letter in the name of the method will be converted this way.

For the third method which includes a parameter, you can specify a user
ID for the selected user. This is accessible for example by the URI 
`/posts/for-user/3734`. Any value can be placed in the final section of
the URI, so you are responsible for verifying the data that is supplied.

### RESTful Routing

RESTful Routing occurs when the request is forwarded on to a provider
that exists within your application. The routing mechanism is very 
similar to MVC Routing, however, each method is prefixed with the HTTP
verb that was used to make the request. For more information check out
the Providers page for more information.

### Static Content Routes

You can route to static content simply my placing files in the
`/application/static` folder. These files must have the `.phtml`
file extension. This is very useful for pages that simply have text
or images and do not need authentication or the overhead of a
controller attached to them.

### Functional Routing

New to the STAPLE framework is Functional Routing. This routing
mechanism is useful when you want to have the full power of the
STAPLE framework available, but you don't want to route through
controller.

#### Adding a Functional Route

Although it is possible to add functional routes from anywhere in
the application it is preferred to use the `routes.php` file included
in the base `/application` folder.

Add a new functional route with `Route::add(string $route, callable $func)`

```php?start_inline=1
Route::add('my-route', function() {
	//Do something
	return 'Test';
});
```

You can use the same method return mechanism that is used for MVC style
routes or provider routes.

```php?start_inline=1
Route::add('terms-of-service', function() {
	return View::create('terms','policy');
});
```

#### Adding Parameters to a Functional Route

To add parameters to your functional routes, you can wrap a string
in mustaches `{` and `}`. The string is arbitrary and does not have to
match the variable name. If multiple parameters are included in a route
they will be added into the function parameter list in order from left
to right.

```php?start_inline=1
Route::add('product/{id}/details', function($id) {
	$product = ProductModel::find($id);
	return View::create('details','product')
		->data(['product'=>$product]);
});
```

#### Before Action Callback Functions

Before a functional route is executed you can register at callback 
function to execute. You can add as many callbacks as you need. **Note
that they will execute prior to EVERY Functional Route.**

```php?start_inline=1
Route::before(function() {
    //Do Something Here
});
```

#### After Action Callback Functions

After a functional route is executed you can register at callback 
function to execute. You can add as many callbacks as you need. **Note
that they will execute after to EVERY Functional Route.**

```php?start_inline=1
Route::after(function() {
    //Do Something Here
});
```