---
layout: document
title: "Models"
date: 2017-03-15 17:28:56
categories: Components
---

## Models

### Create a New Dynamic Data Model

To create a new data model, simply extend the `Model` base class. 

```php?start_inline=1
use Staple\Model;

class CustomerModel extends Model
{
    //Any custom model code here.
}
```

In the above example it will assume that the model relates to a table called `customers`
in the data structure specified by the `Connection` configuration.

### New Model Instance
```php?start_inline=1
$customer = CustomerModel::create();
```

### Assigning Data

Data can be assigned to the model by setting dynamic properties on the model. 
These will be the database columns and values that will get saved to the data store.

```php?start_inline=1
$customer = CustomerModel::create();
$customer->first_name = 'Larry';
$customer->last_name = 'Smith';
```

### Saving Models

To save a model call the `save()` method on the object. If a primary key is specified
for the model object and `UPDATE` will take place. Otherwise the model will perform
and `INSERT` into the database.

```php?start_inline=1
//Returns booleon true/false
$customer->save();
```

## Getting Data From Models

### Find a Single Model

To return a single instance of a model use the `find()` method and specify the primary
key value

```php?start_inline=1
$user = UserModel::find(1);
```

The `$user` variable will contain an instance of `UserModel` for the specified key.

**Note:** The `find()` method will throw a `ModelNotFoundException` if the specified key
is not found in the data store.

### Find Where Equal

You can use custom where class shortcuts to find all models that match specific values.

```php?start_inline=1
//Find all locked users
$users = UserModel::findWhereEqual('locked',1);
```

This returns an `array` of `UserModel` objects. If nothing is found a `ModelNotFoundException`
is thrown.

### Find Where In

Find all model objects that match an array of values.

```php?start_inline=1
//Find all locked users
$users = UserModel::findWhereIn('first_name', ['Tom', 'Dick', 'Harry']);
```

This returns an `array` of `UserModel` objects. If nothing is found a `ModelNotFoundException`
is thrown.

## Changing Defaults

### Custom Table Name

To change the default table name a model relates to you can specify the value of the
`_table` property when inheriting.

```php?start_inline=1
use Staple\Model;

class UserModel extends Model
{
    protected $_table = 'logins';
}
```

### Custom Primary Key

If you would like the change the column name for the primary key you can change the
value of the `_primaryKey` property when inheriting.

```php?start_inline=1
use Staple\Model;

class AccountModel extends Model
{
    protected $_primaryKey = 'guid';
}
```

## Model Queries

You can extend the functionality of the model system by using Model Queries. This allows
you to write custom queries using the query builder functionality and to return instances
of the models that you expect.

```php?start_inline=1
//Returns array of ClientModels
$clients = ClientModel::query()
    ->whereEqual('last_name','Smith')
    ->whereNull('deleted_at')
    ->limit(20)
    ->get();
```

The above code will return all of the `ClientModel` objects that have the last name of `Smith`
and have not been soft deleted, limiting to 20 total records.

You can learn more about the Model Query functionality by visiting the Model Query page of the
documentation.