package page_test

import (
	"net/http"
	"net/url"
	"testing"

	"github.com/stjudewashere/seonaut/internal/issues/errors"
	"github.com/stjudewashere/seonaut/internal/issues/page"
	"github.com/stjudewashere/seonaut/internal/models"

	"golang.org/x/net/html"
)

// Test the LittleContent reporter with a pageReport that does not
// have a little content issue. The reporter should not report the issue.
func TestLittelContentNoIssues(t *testing.T) {
	pageReport := &models.PageReport{
		Crawled:    true,
		MediaType:  "text/html",
		StatusCode: 200,
		Words:      300,
	}

	reporter := page.NewLittleContentReporter()
	if reporter.ErrorType != errors.ErrorLittleContent {
		t.Errorf("TestNoIssues: error type is not correct")
	}

	reportsIssue := reporter.Callback(pageReport, &html.Node{}, &http.Header{})

	if reportsIssue == true {
		t.Errorf("TestLittelContentNoIssues: reportsIssue should be false")
	}
}

// Test the LittleContent reporter with a pageReport that does
// have a little content issue. The reporter should report the issue.
func TestLittleContentIssues(t *testing.T) {
	pageReport := &models.PageReport{
		Crawled:    true,
		MediaType:  "text/html",
		StatusCode: 200,
		Words:      30,
	}

	reporter := page.NewLittleContentReporter()
	if reporter.ErrorType != errors.ErrorLittleContent {
		t.Errorf("TestNoIssues: error type is not correct")
	}

	reportsIssue := reporter.Callback(pageReport, &html.Node{}, &http.Header{})

	if reportsIssue == false {
		t.Errorf("TestLittleContentIssues: reportsIssue should be true")
	}
}

// Test NewIncorrectMediaType with URLs that have correct media types.
// It should not report any issue.
func TestIncorrectMediaTypeNoIssues(t *testing.T) {
	u := "https://example.com/no-issues"
	parsedURL, err := url.Parse(u)
	if err != nil {
		t.Errorf("error parsing URL: %v", err)
	}

	pageReport := &models.PageReport{
		MediaType: "text/html",
		URL:       u,
		ParsedURL: parsedURL,
	}

	reporter := page.NewIncorrectMediaType()
	if reporter.ErrorType != errors.ErrorIncorrectMediaType {
		t.Errorf("TestNoIssues: error type is not correct")
	}

	reportsIssue := reporter.Callback(pageReport, &html.Node{}, &http.Header{})
	if reportsIssue == true {
		t.Errorf("reportsIssue should be false")
	}

	// Test javascript extension.
	u = "https://example.com/script.js"
	parsedURL, err = url.Parse(u)
	if err != nil {
		t.Errorf("error parsing URL: %v", err)
	}

	pageReport = &models.PageReport{
		MediaType: "application/javascript",
		URL:       u,
		ParsedURL: parsedURL,
	}

	reportsIssue = reporter.Callback(pageReport, &html.Node{}, &http.Header{})
	if reportsIssue == true {
		t.Errorf("reportsIssue should be false")
	}
}

// Test NewIncorrectMediaType with URLs that have incorrect media types.
// It should report the issues.
func TestIncorrectMediaTypeIssues(t *testing.T) {
	pageReport := &models.PageReport{} // Test missing media type

	reporter := page.NewIncorrectMediaType()
	if reporter.ErrorType != errors.ErrorIncorrectMediaType {
		t.Errorf("TestNoIssues: error type is not correct")
	}

	reportsIssue := reporter.Callback(pageReport, &html.Node{}, &http.Header{})
	if reportsIssue == false {
		t.Errorf("reportsIssue should be true")
	}

	// Test media type that doesn't match the file extension.
	u := "https://example.com/issues.pdf"
	parsedURL, err := url.Parse(u)
	if err != nil {
		t.Errorf("error parsing URL: %v", err)
	}

	pageReport = &models.PageReport{
		MediaType: "text/html",
		URL:       u,
		ParsedURL: parsedURL,
	}

	reportsIssue = reporter.Callback(pageReport, &html.Node{}, &http.Header{})
	if reportsIssue == false {
		t.Errorf("reportsIssue should be true")
	}
}
