/*
 * SonarQube
 * Copyright (C) 2009-2024 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.xoo.rule;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.List;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.sonar.api.batch.fs.InputFile;
import org.sonar.api.batch.fs.internal.DefaultFileSystem;
import org.sonar.api.batch.fs.internal.DefaultInputFile;
import org.sonar.api.batch.fs.internal.TestInputFileBuilder;
import org.sonar.api.batch.rule.ActiveRule;
import org.sonar.api.batch.rule.ActiveRules;
import org.sonar.api.batch.sensor.internal.SensorContextTester;
import org.sonar.api.internal.apachecommons.io.IOUtils;
import org.sonar.api.rule.RuleKey;
import org.sonar.xoo.Xoo;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class OneVulnerabilityPerSecurityStandardSensorTest {
  @Rule
  public TemporaryFolder temp = new TemporaryFolder();

  private final ActiveRules activeRules = mock(ActiveRules.class);

  @Before
  public void before() {
    List<ActiveRule> rules = List.of(mockActiveRule("OneVulnerabilityPerLine_stig-ASD_V5R3:V12345"),
      mockActiveRule("OneVulnerabilityPerLine_stig-ASD_V5R3:V12346"));
    when(activeRules.findAll()).thenReturn(rules);
  }

  private ActiveRule mockActiveRule(String ruleKey) {
    ActiveRule activeRuleMock = mock(ActiveRule.class);
    when(activeRuleMock.ruleKey()).thenReturn(RuleKey.of(XooRulesDefinition.XOO_REPOSITORY, ruleKey));
    return activeRuleMock;
  }

  @Test
  public void execute_dataAndExecutionFlowsAreDetectedAndMessageIsFormatted() throws IOException {
    DefaultInputFile inputFile = newTestFile(IOUtils.toString(getClass().getResource("vulnerabilities.xoo"), StandardCharsets.UTF_8));

    DefaultFileSystem fs = new DefaultFileSystem(temp.newFolder());
    fs.add(inputFile);

    OneVulnerabilityPerSecurityStandardSensor sensor = new OneVulnerabilityPerSecurityStandardSensor(fs, activeRules);

    SensorContextTester sensorContextTester = SensorContextTester.create(fs.baseDir());
    sensorContextTester.setFileSystem(fs);
    sensor.execute(sensorContextTester);

    assertThat(sensorContextTester.allIssues()).hasSize(2);
  }

  private DefaultInputFile newTestFile(String content) {
    return new TestInputFileBuilder("foo", "src/vulnerabilities.xoo")
      .setLanguage(Xoo.KEY)
      .setType(InputFile.Type.MAIN)
      .setCharset(Charset.defaultCharset())
      .setContents(content)
      .build();
  }
}
