/*
 * SonarQube
 * Copyright (C) 2009-2024 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.xoo.rule;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Map;
import java.util.stream.Collectors;
import org.sonar.api.batch.fs.FileSystem;
import org.sonar.api.batch.fs.InputFile;
import org.sonar.api.batch.rule.ActiveRule;
import org.sonar.api.batch.rule.ActiveRules;
import org.sonar.api.batch.sensor.Sensor;
import org.sonar.api.batch.sensor.SensorContext;
import org.sonar.api.batch.sensor.SensorDescriptor;
import org.sonar.api.batch.sensor.issue.NewIssue;
import org.sonar.api.rule.RuleKey;
import org.sonar.xoo.Xoo;

/**
 * Raise a vulnerability issue per standard that are defined on the line.
 * Eg. if line contains: owaspAsvs-4.0:1.4.2 will raise an issue with the corresponding standard.
 */
public class OneVulnerabilityPerSecurityStandardSensor implements Sensor {

  public static final String RULE_KEY_PREFIX = "OneVulnerabilityPerLine_";

  private final FileSystem fs;
  private final ActiveRules activeRules;

  public OneVulnerabilityPerSecurityStandardSensor(FileSystem fs, ActiveRules activeRules) {
    this.fs = fs;
    this.activeRules = activeRules;
  }


  @Override
  public void describe(SensorDescriptor descriptor) {
    descriptor
      .onlyOnLanguage(Xoo.KEY)
      .createIssuesForRuleRepository(XooRulesDefinition.XOO_REPOSITORY);
  }

  @Override
  public void execute(SensorContext context) {
    doAnalyse(context, Xoo.KEY);
  }

  private void doAnalyse(SensorContext context, String languageKey) {
    String rulePrefix = languageKey + ":" + RULE_KEY_PREFIX;
    Map<String, RuleKey> rulesBySecurityStandard = activeRules.findAll().stream()
      .filter(r -> r.ruleKey().toString().startsWith(rulePrefix))
      .collect(Collectors.toMap(c -> c.ruleKey().toString().substring(rulePrefix.length()), ActiveRule::ruleKey));

    for (InputFile inputFile : fs.inputFiles(fs.predicates().hasLanguage(languageKey))) {
      for (String securityStandard : rulesBySecurityStandard.keySet()) {
        processFile(inputFile, context, rulesBySecurityStandard.get(securityStandard), securityStandard);
      }
    }
  }


  protected void processFile(InputFile inputFile, SensorContext context, RuleKey ruleKey, String securityStandard) {
    try {
      int[] lineCounter = {1};
      try (InputStreamReader isr = new InputStreamReader(inputFile.inputStream(), inputFile.charset());
           BufferedReader reader = new BufferedReader(isr)) {
        reader.lines().forEachOrdered(lineStr -> {

          if (lineStr.contains(securityStandard)) {
            NewIssue newIssue = context.newIssue();
            newIssue
              .forRule(ruleKey)
              .at(newIssue.newLocation()
                .on(inputFile)
                .at(inputFile.newRange(lineCounter[0], lineStr.indexOf(securityStandard), lineCounter[0], lineStr.indexOf(securityStandard) + securityStandard.length())))
              .save();
          }
          lineCounter[0]++;
        });
      }
    } catch (IOException e) {
      throw new IllegalStateException("Fail to process " + inputFile, e);
    }
  }

}
