import path from 'path';
import { ComponentFile, PackageDefinition } from '../../components';

/**
 * Generate component builder template
 * @param {(PackageDefinition | ComponentFile)[]} components components to include in component builder
 * @param {string} distPath destination path for component builder
 * @returns generated component builder template
 */
export const getComponentBuilderTemplate = (
  components: (PackageDefinition | ComponentFile)[],
  distPath: string
) => {
  const componentFiles = components.filter(
    (component) => (component as ComponentFile).componentName
  ) as ComponentFile[];
  const packages = components.filter(
    (component) => (component as PackageDefinition).components
  ) as PackageDefinition[];

  return `/* eslint-disable */
// Do not edit this file, it is auto-generated at build time!
// See scripts/generate-component-builder/index.js to modify the generation of this file.

import { ComponentBuilder } from '@sitecore-jss/sitecore-jss-react';
${packages
  .map((pkg) => {
    const list = pkg.components.map((c) => c.moduleName).join(', ');
    return `import { ${list} } from '${pkg.name}';\n`;
  })
  .join('')}
${componentFiles
  .map((component) => {
    const sourcePath = path.relative(path.dirname(distPath), component.path).replace(/\\/g, '/');
    return `import ${component.moduleName} from '${sourcePath}';`;
  })
  .join('\n')}

const components = new Map();
${packages
  .map((p) =>
    p.components.map(
      (component) => `components.set('${component.componentName}', ${component.moduleName});\n`
    )
  )
  .flat()
  .join('')}
${componentFiles
  .map((component) => `components.set('${component.componentName}', ${component.moduleName});`)
  .join('\n')}

const componentBuilder = new ComponentBuilder({ components });

export const componentFactory = componentBuilder.getComponentFactory();
`;
};
