const fs = require('fs');
const path = require('path');
const chokidar = require('chokidar');

/*
  COMPONENT FACTORY GENERATION
  Generates the /src/temp/componentFactory.js file which maps Vue components
  to JSS components.

  The component factory is a mapping between a string name and a Vue component instance.
  When the Sitecore Layout service returns a layout definition, it returns named components.
  This mapping is used to construct the Vue hierarchy for the layout.

  The default convention uses the parent folder name as the component name,
  but it is customizable in generateComponentFactory().

  NOTE: this script can run in two modes. The default mode, the component factory file is written once.
  But if `--watch` is a process argument, the component factory source folder will be watched,
  and the componentFactory.js rewritten on added or deleted files.
  This is used during `jss start` to pick up new or removed components at runtime.
*/

const componentFactoryPath = path.resolve('src/temp/componentFactory.js');
const componentRootPath = 'src/components';

const isWatch = process.argv.some((arg) => arg === '--watch');

if (isWatch) {
  watchComponentFactory();
} else {
  writeComponentFactory();
}

function watchComponentFactory() {
  console.log(`Watching for changes to component factory sources in ${componentRootPath}...`);

  chokidar
    .watch(componentRootPath, { ignoreInitial: true, awaitWriteFinish: true })
    .on('add', writeComponentFactory)
    .on('unlink', writeComponentFactory);
}

function writeComponentFactory() {
  const componentFactory = generateComponentFactory();

  console.log(`Writing component factory to ${componentFactoryPath}`);

  fs.writeFileSync(componentFactoryPath, componentFactory, { encoding: 'utf8' });
}

function generateComponentFactory() {
  // by convention, we expect to find Vue components
  // * under /src/components
  // * with a .vue extension to define a component file
  // If you'd like to use your own convention, encode it below.
  // NOTE: generating the component factory is also totally optional,
  // and it can be maintained manually if preferred.

  const imports = [];
  const registrations = [];
  /**
   * You can specify components which you want to import from external/internal packages
   * in format:
   *  {
   *    name: 'package name',
   *    components: [
   *      {
   *        componentName: 'component name', // component rendering name,
   *        moduleName: 'module name' // component name to import from the package
   *      }
   *    ]
   *  }
   */
  const packages = [];

  packages.forEach((p) => {
    const variables = p.components
      .map((c) => {
        registrations.push(`components.set('${c.componentName}', ${c.moduleName});`);

        return c.moduleName;
      })
      .join(', ');

    imports.push(`import { ${variables} } from '${p.name}'`);
  });

  const componentFiles = extractVueFiles(componentRootPath);
  componentFiles.forEach((componentFile) => {
    if (!fs.existsSync(componentFile)) return;

    console.debug(`Registering JSS component ${componentFile}`);
    const componentName = path.basename(componentFile, '.vue');
    const importVarName = componentName.replace(/[^\w]+/g, '');
    imports.push(
      `import ${importVarName} from '../components/${componentFile
        .replace(path.join(componentRootPath, '/'), '')
        .replace(/\\/g, '/')
        .replace('.vue', '')}';`
    );
    registrations.push(`components.set('${componentName}', ${importVarName});`);
  });

  return `/* eslint-disable */
// Do not edit this file, it is auto-generated at build time!
// See scripts/bootstrap.js to modify the generation of this file.
${imports.join('\n')}

const components = new Map();
${registrations.join('\n')}

export default function componentFactory(componentName) {
  return components.get(componentName);
};
`;
}

/**
 * Recursively iterates the given folderPath, creating a flat array of found .vue file paths.
 * For example, given the following folder structure and using `/components` as the root folderPath:
 * /components/component0.vue
 * /components/subfolder/component1.vue
 *
 * The output would be:
 * ['component0.vue', 'subfolder/component1.vue']
 */
function extractVueFiles(folderPath) {
  let results = [];
  fs.readdirSync(folderPath).forEach((pathName) => {
    const computedPath = path.join(folderPath, pathName);
    const stat = fs.statSync(computedPath);
    if (stat && stat.isDirectory()) {
      results = results.concat(extractVueFiles(computedPath));
    } else if (path.extname(computedPath).toLowerCase() === '.vue') {
      results.push(computedPath);
    }
  });
  return results;
}
