import 'dotenv/config';
import * as fs from 'fs';
import * as path from 'path';
import { constantCase } from 'constant-case';
import { jssConfigFactory } from './config';
import { JssConfig } from 'lib/config';

/**
 * Generate config
 * The object returned from this function will be made available by importing src/environments/environment.js.
 * This is executed prior to the build running, so it's a way to inject environment or build config-specific
 * settings as variables into the JSS app.
 * NOTE! Any configs returned here will be written into the client-side JS bundle. DO NOT PUT SECRETS HERE.
 */

const defaultConfigValue: JssConfig = {
  production: false,
  sitecoreApiKey: process.env[`${constantCase('sitecoreApiKey')}`],
  sitecoreApiHost: process.env[`${constantCase('sitecoreApiHost')}`],
  sitecoreSiteName: process.env[`${constantCase('sitecoreSiteName')}`],
  defaultLanguage: process.env[`${constantCase('defaultLanguage')}`],
  graphQLEndpoint: process.env[`${constantCase('graphQLEndpoint')}`],
  graphQLEndpointPath: process.env[`${constantCase('graphQLEndpointPath')}`],
  defaultServerRoute: '/',
};

async function main() {
  await generateConfig('src/environments/environment.js', defaultConfigValue, {
    production: false,
  });
  await generateConfig('src/environments/environment.prod.js', defaultConfigValue, {
    production: true,
  });
}

main();

/**
 * Generates the JSS config based on config plugins (under ./config/plugins)
 * and then writes the config to disk.
 * @param {string} outputPath the output path of the generated config.
 * @param {JssConfig} [defaultConfig] Default configuration.
 * @param {[key: string]:unknown} [configOverrides] configuration values that override the generated ones.
 */
export function generateConfig(
  outputPath: string,
  defaultConfig: JssConfig = defaultConfigValue,
  configOverrides?: { [key: string]: unknown }
) {
  // Handle undefined values
  defaultConfig = Object.keys(defaultConfig).reduce((acc, key) => {
    return {
      ...acc,
      [key]: defaultConfig[key] || '',
    };
  }, {});

  return jssConfigFactory
    .create(defaultConfig)
    .then((config) => {
      writeConfig(Object.assign(config, configOverrides), outputPath);
    })
    .catch((e) => {
      console.error('Error generating config');
      console.error(e);
      process.exit(1);
    });
}

/**
 * Writes the config object to disk with support for environment variables.
 * @param {JssConfig} config JSS configuration to write.
 * @param {string} outputPath the outputh path of the generated config.
 */
export function writeConfig(config: JssConfig, outputPath?: string) {
  if (!outputPath) {
    outputPath = 'src/environments/environment.js';
  }

  let configText = `/* eslint-disable */
  // Do not edit this file, it is auto-generated at build time!
  // See scripts/bootstrap.ts to modify the generation of this file.
  const config = {};\n`;

  // Set base configuration values, allowing override with environment variables
  Object.keys(config).forEach((prop) => {
    // Handle undefined values
    const value = config[prop] || '';
    configText += `config.${prop} = process.env.${constantCase(
      prop
    )} || "${value.toString().trim()}";\n`;
  });

  configText += `module.exports.environment = config;`;

  const configPath = path.resolve(outputPath);
  console.log(`Writing runtime config to ${configPath}`);
  fs.writeFileSync(configPath, configText, { encoding: 'utf8' });
}
