
/**
 * @depends DropdownBaseElement.js
 * @depends DataRendererLabelElement.js
 * @depends DataListElement.js
 */

//////////////////////////////////////////////////////////////
//////////////////DropdownElement/////////////////////////////

/**
 * @class DropdownElement
 * @inherits DropdownBaseElement
 * 
 * DropdownElement is a compound button that creates a pop-up drop-down list which the user
 * can select a value which is then displayed by the Dropdown. The values
 * in the list are generated by a supplied ListCollection and associated styling.
 * 
 * The Dropdown button itself contains a child button which is used to render
 * the divider line and arrow. Dropdown proxies its SkinState style to the arrow
 * button so the arrow button will change states along with the Dropdown itself.
 * See the default item renderer, DataRendererLabelElement and the default 
 * skin for the arrow button, DropdownArrowButtonSkinElement for additional styles.
 * 
 * @seealso DropdownArrowButtonSkinElement
 * 
 * 
 * @constructor DropdownElement 
 * Creates new DropdownElement instance.
 */
function DropdownElement()
{
	DropdownElement.base.prototype.constructor.call(this);

	this._listCollection = null; //Data collection
	
	this._selectedIndex = -1;
	this._selectedItem = null;
	
	this._popupContainer = null; //Same as base._popupElement;
	this._dataListPopup = null; 
	
	this._openDirection = null;
	this._openHeight = null;
	this._dropdownManagerMetrics = null;
	
	this._sampledTextWidth = null;
	

	/////////////////////
	
	var _self = this;
	
	//Private event listener, need an instance for each DropdownElement, proxy to prototype.
	this._onDropdownListCollectionChangedInstance = 
		function (collectionChangedEvent)
		{
			_self._onDropdownListCollectionChanged(collectionChangedEvent);
		};
		
	this._onDropdownDataListPopupChangedInstance = 
		function (event)
		{
			_self._onDropdownDataListPopupChanged(event);
		};
	
	this._onDropdownDataListPopupListItemClickedInstance = 
		function (event)
		{
			_self._onDropdownDataListPopupListItemClicked(event);
		};
		
	this._onDropdownDataListPopupLayoutCompleteInstance = 
		function (event)
		{
			_self._onDropdownDataListPopupLayoutComplete(event);
		};
}

//Inherit from ButtonElement
DropdownElement.prototype = Object.create(DropdownBaseElement.prototype);
DropdownElement.prototype.constructor = DropdownElement;
DropdownElement.base = DropdownBaseElement;


////////////Events///////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the drop down selection changes as a result of user input.
 * 
 * @event listitemclick ElementListItemClickEvent
 * Dispatched when a DataRenderer in the popup list is clicked. Includes associated collection item/index.
 */


/////////////Style Types/////////////////////////

DropdownElement._StyleTypes = Object.create(null);

/**
 * @style ItemLabelFunction Function
 * A function that returns a text string per a supplied collection item.
 * function (itemData) { return "" }
 */
DropdownElement._StyleTypes.ItemLabelFunction = 			StyleableBase.EStyleType.NORMAL; 		// function (itemData) { return "" }

/**
 * @style PopupDataListStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to apply to the pop up list element.
 */
DropdownElement._StyleTypes.PopupDataListStyle = 			StyleableBase.EStyleType.SUBSTYLE; 		// StyleDefinition

/**
 * @style MaxPopupHeight Number
 * Maximum height in pixels of the pop up list element.
 */
DropdownElement._StyleTypes.MaxPopupHeight = 				StyleableBase.EStyleType.NORMAL; 		// number

/**
 * @style PopupDataListClipTopOrBottom Number
 * 
 * Size in pixels to clip off the pop up list. Clips top when opening down, bottom when opening up. 
 * Defaults to 1 to collapse pop up list and dropdown default borders.
 */
DropdownElement._StyleTypes.PopupDataListClipTopOrBottom = 	StyleableBase.EStyleType.NORMAL; 		// number


////////////Default Styles////////////////////

//DataList Scrollbar style
DropdownElement.DataListScrollBarStyleDefault = new StyleDefinition();
DropdownElement.DataListScrollBarStyleDefault.setStyle("Padding", -1);			//Expand by 1px to share borders

DropdownElement.DataListItemUpSkinStyleDefault = new StyleDefinition();
DropdownElement.DataListItemUpSkinStyleDefault.setStyle("BackgroundFill", 		"#FFFFFF");

DropdownElement.DataListItemAltSkinStyleDefault = new StyleDefinition();
DropdownElement.DataListItemAltSkinStyleDefault.setStyle("BackgroundFill", 		"#F0F0F0");

//DataList ListItem style
DropdownElement.DataListItemStyleDefault = new StyleDefinition();
DropdownElement.DataListItemStyleDefault.setStyle("UpSkinStyle", 				DropdownElement.DataListItemUpSkinStyleDefault);
DropdownElement.DataListItemStyleDefault.setStyle("AltSkinStyle", 				DropdownElement.DataListItemAltSkinStyleDefault);

//DataList style
DropdownElement.DataListStyleDefault = new StyleDefinition();
DropdownElement.DataListStyleDefault.setStyle("ScrollBarStyle", 				DropdownElement.DataListScrollBarStyleDefault);
DropdownElement.DataListStyleDefault.setStyle("ListItemClass", 					DataRendererLabelElement);	//Same as DataList default (not needed)
DropdownElement.DataListStyleDefault.setStyle("ListItemStyle", 					DropdownElement.DataListItemStyleDefault);										
DropdownElement.DataListStyleDefault.setStyle("BorderType", 					"solid");
DropdownElement.DataListStyleDefault.setStyle("BorderThickness", 				1);
DropdownElement.DataListStyleDefault.setStyle("PaddingTop",						1);
DropdownElement.DataListStyleDefault.setStyle("PaddingBottom",					1);
DropdownElement.DataListStyleDefault.setStyle("PaddingLeft",					1);
DropdownElement.DataListStyleDefault.setStyle("PaddingRight",					1);
///////////////////////////////////

DropdownElement.StyleDefault = new StyleDefinition();

DropdownElement.StyleDefault.setStyle("PopupDataListStyle", 					DropdownElement.DataListStyleDefault); 			// StyleDefinition
DropdownElement.StyleDefault.setStyle("MaxPopupHeight", 						200); 											// number
DropdownElement.StyleDefault.setStyle("PopupDataListClipTopOrBottom", 			1); 											// number
DropdownElement.StyleDefault.setStyle("ItemLabelFunction", 						DataListElement.DefaultItemLabelFunction); 		// function (itemData) { return "" }


/////////Style Proxy Maps/////////////////////////////

//Proxy map for styles we want to pass to the DataList popup.
DropdownElement._PopupDataListProxyMap = Object.create(null);
DropdownElement._PopupDataListProxyMap.ItemLabelFunction = 				true;
DropdownElement._PopupDataListProxyMap._Arbitrary = 					true;


/////////////Public///////////////////////////////

/**
 * @function setSelectedIndex
 * Sets the selection collection index. Also updates selected item.
 * 
 * @param index int
 * Collection index to select.
 */
DropdownElement.prototype.setSelectedIndex = 
	function (index)
	{
		if (this._selectedIndex == index)
			return false;
		
		if (this._listCollection == null || index > this._listCollection.length -1)
			return false;
		
		if (index < -1)
			index = -1;
		
		if (this._dataListPopup != null)
			this._dataListPopup.setSelectedIndex(index);
		
		this._selectedIndex = index;
		this._selectedItem = this._listCollection.getItemAt(index);
		this._updateText();

		return true;
	};

/**
 * @function getSelectedIndex
 * Gets the selected collection index.
 * 
 * @returns int
 * Selected collection index or -1 if none selected.
 */	
DropdownElement.prototype.getSelectedIndex = 
	function ()
	{
		return this._selectedIndex;
	};
	
/**
 * @function setSelectedItem
 * Sets the collection item to select, also updates selected index.
 * 
 * @param item Object
 * Collection item to select.
 */	
DropdownElement.prototype.setSelectedItem = 
	function (item)
	{
		var index = this._listCollection.getItemIndex(item);
		this.setSelectedIndex(index);
	};
	
/**
 * @function getSelectedItem
 * Gets the selected collection item.
 * 
 * @returns Object
 * Selected collection item or null if none selected.
 */	
DropdownElement.prototype.getSelectedItem = 
	function ()
	{
		return this._selectedItem;
	};
	
/**
 * @function setListCollection
 * Sets the ListCollection to be used as the data-provider.
 * 
 * @param listCollection ListCollection
 * ListCollection to be used as the data-provider.
 */	
DropdownElement.prototype.setListCollection = 
	function (listCollection)
	{
		if (this._listCollection == listCollection)
			return;
	
		if (this._manager == null)
		{
			this._listCollection = listCollection;
		}
		else
		{
			if (this._listCollection != null)
				this._listCollection.removeEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance);
			
			this._listCollection = listCollection;
			
			if (this._listCollection != null)
				this._listCollection.addEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance);
		}
		
		//Fix selected index/item
		if (this._listCollection == null)
		{
			this._selectedIndex = -1;
			this._selectedItem = null;
		}
		else
		{
			if (this._selectedItem != null)
			{
				this._selectedIndex = this._listCollection.getItemIndex(this._selectedItem);
				
				if (this._selectedIndex == -1)
					this._selectedItem = null;
			}
		}
		
		this._updateText();
		this._sampledTextWidth = null;
		this._invalidateMeasure();
		
		if (this._dataListPopup != null)
			this._dataListPopup.setListCollection(listCollection);
	};	

	
/////////////Internal///////////////////////////////	

//@override
DropdownElement.prototype._createPopup = 
	function ()
	{
		this._popupContainer = new CanvasElement();
		this._popupContainer.setStyle("ClipContent", true);
		
		this._dataListPopup = new DataListElement();
		this._dataListPopup._owner = this; //Set owner - base sets its on the container
		this._popupContainer._addChild(this._dataListPopup);
		
		this._dataListPopup._setStyleProxy(new StyleProxy(this, DropdownElement._PopupDataListProxyMap));
		this._applySubStylesToElement("PopupDataListStyle", this._dataListPopup);
		
		this._dataListPopup.setListCollection(this._listCollection);
		this._dataListPopup.setSelectedIndex(this._selectedIndex);
		
		this._dataListPopup.addEventListener("changed", this._onDropdownDataListPopupChangedInstance);
		this._dataListPopup.addEventListener("listitemclick", this._onDropdownDataListPopupListItemClickedInstance);
		this._dataListPopup.addEventListener("layoutcomplete", this._onDropdownDataListPopupLayoutCompleteInstance);
		
		return this._popupContainer;
	};	
	
//@override
DropdownElement.prototype._addPopup = 
	function ()
	{
		if (DropdownElement.base.prototype._addPopup.call(this) == false)
			return false;
		
		this._openDirection = "down";
		this._openHeight = this.getStyle("MaxPopupHeight");
		this._dropdownManagerMetrics = this.getMetrics(this._manager);
		
		this._popupContainer.setStyle("Height", 0);
		this._onDropdownDataListPopupLayoutComplete(null);
		
		return true;
	};
	
//@override
DropdownElement.prototype._removePopup = 
	function ()
	{
		if (DropdownElement.base.prototype._removePopup.call(this) == false)
			return false;
		
		this._openDirection = null;
		this._openHeight = null;
		this._dropdownManagerMetrics = null;
		
		return true;
	};

//@override	
DropdownElement.prototype._getOpenedTweenValue = 
	function ()
	{
		return this._openHeight;
	};

	
//@override	
DropdownElement.prototype._updateTweenPosition = 
	function (value)
	{
		this._popupContainer.setStyle("Height", value);
		
		if (this._openDirection == "up")
		{
			this._popupContainer.setStyle("Y", this._dropdownManagerMetrics._y - value);
			this._dataListPopup._setActualPosition(0, 0);
		}
		else //if (this._openDirection == "down")
		{
			this._popupContainer.setStyle("Y", this._dropdownManagerMetrics._y + this._dropdownManagerMetrics._height);
			this._dataListPopup._setActualPosition(0, value - this._dataListPopup._height);
		}
	};

/**
 * @function _onDropdownDataListPopupLayoutComplete
 * Event handler for pop up list "layoutcomplete". 
 * Updates the pop up list height after content size is known and determines
 * if drop down opens up or down depending on available space.
 * 
 * @param event DispatcherEvent
 * DispatcherEvent to process.
 */		
DropdownElement.prototype._onDropdownDataListPopupLayoutComplete =
	function (event)
	{
		var maxHeight = this.getStyle("MaxPopupHeight");
		var height = null;
		
		if (this._dataListPopup.getStyle("ListDirection") == "horizontal")
			height = maxHeight;
		else
		{
			//Get actual Popup list height.
			var contentSize = this._dataListPopup._getContentSize();
			
			if (contentSize < maxHeight)
			{
				if (this._listCollection != null && this._dataListPopup._getNumRenderers() < this._listCollection.getLength())
					height = maxHeight;
				else
					height = contentSize;
			}
			else //contentSize >= maxHeight
				height = maxHeight;
		}
		
		//Determine open up/down and correct if not enough available space.
		var availableBottom = this._manager._height - (this._dropdownManagerMetrics._y + this._dropdownManagerMetrics._height);
		if (availableBottom >= height)
		{
			this._openDirection = "down";
			this._openHeight = height;
		}
		else //if (availableBottom < height)
		{
			var availableTop = this._dropdownManagerMetrics._y;
			if (availableTop >= height)
			{
				this._openDirection = "up";
				this._openHeight = height;
			}
			else //if (availableTop < height)
			{
				if (availableBottom >= availableTop)
				{
					this._openDirection = "down";
					this._openHeight = availableBottom;
				}
				else
				{
					this._openDirection = "up";
					this._openHeight = availableTop;
				}
			}
		}
		
		this._popupContainer.setStyle("X", this._dropdownManagerMetrics._x);
		this._popupContainer.setStyle("Width", this._dropdownManagerMetrics._width);
		this._dataListPopup._setActualSize(this._dropdownManagerMetrics._width, this._openHeight);

		this._openHeight -= this.getStyle("PopupDataListClipTopOrBottom");
		
		//Fix position if tween is not running
		if (this._getTweenRunning() == false)
			this._updateTweenPosition(this._openHeight);
	};
	
/**
 * @function _onDropdownDataListPopupChanged
 * Event handler for pop up list "changed" event. Updates selected item/index and re-dispatches "changed" event.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to process.
 */	
DropdownElement.prototype._onDropdownDataListPopupChanged = 
	function (elementEvent)
	{
		this.setSelectedIndex(this._dataListPopup.getSelectedIndex());
		
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};

/**
 * @function _onDropdownDataListPopupListItemClicked
 * Event handler for pop up list "listitemclick" event. 
 * Re-dispatches the list event and closes the Dropdown.
 * 
 * @param elementListItemClickEvent ElementListItemClickEvent
 * ElementListItemClickEvent to process.
 */		
DropdownElement.prototype._onDropdownDataListPopupListItemClicked = 
	function (elementListItemClickEvent)
	{
		//Just proxy the event from the popup list
		if (this.hasEventListener("listitemclick", null) == true)
			this.dispatchEvent(elementListItemClickEvent.clone());
		
		var shouldClose = true;
		if (this._dataListPopup.getStyle("Selectable") == false ||
			(elementListItemClickEvent.getItem().hasOwnProperty("selectable") == true &&
			elementListItemClickEvent.getItem().selectable == false))
		{
			shouldClose = false;
		}
		
		if (shouldClose == true)
		{
			this.close(true);
			
			if (this.hasEventListener("closed", null) == true)
				this.dispatchEvent(new DispatcherEvent("closed"));
		}
	};
	
//@override	
DropdownElement.prototype._updateText = 
	function ()
	{
		var text = null;
		var labelFunction = this.getStyle("ItemLabelFunction");
		
		if (this._selectedItem == null || labelFunction == null)
			text = this.getStyle("Text");
		else
			text = labelFunction(this._selectedItem);
		
		this._setLabelText(text);
	};	
	
/**
 * @function _onDropdownListCollectionChanged
 * Event handler for the ListCollection data-providers "collectionchanged" event. 
 * 
 * @param collectionChangedEvent CollectionChangedEvent
 * CollectionChangedEvent to process.
 */	
DropdownElement.prototype._onDropdownListCollectionChanged = 
	function (collectionChangedEvent)
	{
		//Room to optimize here
//		var type = collectionChangedEvent.getKind();
//		var index = collectionChangedEvent.getIndex();
	
		//Fix selected index/item 
		if (this._selectedItem != null)
		{
			this._selectedIndex = this._listCollection.getItemIndex(this._selectedItem);
			
			if (this._selectedIndex == -1)
				this._selectedItem = null;
		}
		
		this._updateText();
		this._sampledTextWidth = null;
		this._invalidateMeasure();
	};	
	
//@Override	
DropdownElement.prototype._onCanvasElementAdded = 
	function (addedRemovedEvent)
	{
		DropdownElement.base.prototype._onCanvasElementAdded.call(this, addedRemovedEvent);
	
		if (this._listCollection != null && this._listCollection.hasEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance) == false)
			this._listCollection.addEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance);
	};

//@Override	
DropdownElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		DropdownElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		if (this._listCollection != null && this._listCollection.hasEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance) == true)
			this._listCollection.removeEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance);
		
		this.close(false);
	};	

//@override
DropdownElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DropdownElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("ItemLabelFunction" in stylesMap)
		{
			this._sampledTextWidth = null;
			this._invalidateMeasure();
			this._updateText();
		}
		
		if ("PopupDataListStyle" in stylesMap && this._dataListPopup != null)
			this._applySubStylesToElement("PopupDataListStyle", this._dataListPopup);
		
		if ("TextStyle" in stylesMap ||
			"TextFont" in stylesMap ||
			"TextSize" in stylesMap ||
			"TextHorizontalAlign" in stylesMap ||
			"TextVerticalAlign" in stylesMap || 
			"TextLinePaddingTop" in stylesMap ||
			"TextLinePaddingBottom" in stylesMap ||
			"Text" in stylesMap)
		{
			this._sampledTextWidth = null;
			this._invalidateMeasure();
		}
	};
	
/**
 * @function _sampleTextWidths
 * Measures text width of first 10 ListCollection items for measurement.
 * 
 * @returns Number
 * Largest text width in pixels.
 */	
DropdownElement.prototype._sampleTextWidths = 
	function ()
	{
		var labelFont = this._getFontString();
		
		var text = this.getStyle("Text");
		if (text == null)
			text = "";
		
		var measuredTextWidth = CanvasElement._measureText(text, labelFont);
		
		//Sample the first 10 items.
		var labelFunction = this.getStyle("ItemLabelFunction");
		if (this._listCollection != null && labelFunction != null)
		{
			var textWidth = 0;
			for (var i = 0; i < 10; i++)
			{
				if (i == this._listCollection.getLength())
					break;
				
				textWidth = CanvasElement._measureText(labelFunction(this._listCollection.getItemAt(i)), labelFont);
				
				if (textWidth > measuredTextWidth)
					measuredTextWidth = textWidth;
			}
		}
		
		return measuredTextWidth;
	};
	
//@override
DropdownElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		if (this._sampledTextWidth == null)
			this._sampledTextWidth = this._sampleTextWidths();
		
		var textHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
		
		var measuredWidth = this._sampledTextWidth + padWidth + 20; //Add some extra space
		var measuredHeight = textHeight + padHeight;
		
		if (this._arrowButton != null)
		{
			var arrowWidth = this._arrowButton.getStyle("Width");
			var arrowHeight = this._arrowButton.getStyle("Height");
			
			if (arrowHeight != null && arrowHeight > measuredHeight)
				measuredHeight = arrowHeight;
			if (arrowWidth != null)
				measuredWidth += arrowWidth;
			else
				measuredWidth += Math.round(measuredHeight * .85);
		}

		this._setMeasuredSize(measuredWidth, measuredHeight);
	};	
	
//@override	
DropdownElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		DropdownElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		if (this._openDirection != null) //dropdown open
		{
			//Update the dropdown metrics
			this._dropdownManagerMetrics = this.getMetrics(this._manager);
			
			//Update the widths of the popup container and list. (Heights handled by tween / list layoutcomplete)
			//This is here so that when the Dropdown is using measured width, and the collection changes,
			//it may change the width of the dropdown button, so we need to make sure we keep the widths in sync.
			this._popupContainer.setStyle("Width", this._dropdownManagerMetrics._width);
			this._popupContainer.setStyle("X", this._dropdownManagerMetrics._x);
			
			this._dataListPopup._setActualSize(this._dropdownManagerMetrics._width, this._dataListPopup._height);
		}
	};
	
	