//MIT License
//
//Copyright (c) 2017-2019 SENSORMATIC ELECTRONICS LLC, NATHAN E NELSON
//
//Permission is hereby granted, free of charge, to any person obtaining a copy
//of this software and associated documentation files (the "Software"), to deal
//in the Software without restriction, including without limitation the rights
//to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//copies of the Software, and to permit persons to whom the Software is
//furnished to do so, subject to the following conditions:
//
//The above copyright notice and this permission notice shall be included in all
//copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//SOFTWARE.

///////////////////////////////////////////////////////
//////////////////Tween////////////////////////////////
	
/**
 * @class Tween
 * 
 * Tween is a helper class that is used to interpolate values
 * across a given time span and is essentially just a calculator. 
 * It can be used for nearly any type of animation and supports
 * easing for acceleration and deceleration. 
 * 
 * If you're unsure about easing, a hint is that SineInOut is a kind of magic salt 
 * you can sprinkle on just about any linear animation that usually makes everything 
 * look smoother and less jarring without being obvious.
 * 
 * @constructor Tween 
 * Creates new Tween instance.
 */
function Tween()
{
	/**
	 * @member startVal Number
	 * Beginning value at the start time of the tween.
	 */
	this.startVal = 0;
	
	/**
	 * @member endVal Number
	 * Ending value at the end time of the tween duration.
	 */
	this.endVal = 0;
	
	/**
	 * @member duration Number
	 * Duration in milliseconds the tween will run.
	 */
	this.duration = 0;
	
	/**
	 * @member startTime Number
	 * Time in milliseconds that the tween should start as returned by Date.now().
	 */
	this.startTime = null;
	
	/**
	 * @member easingFunction Function
	 * Easing function to use when calculating the tween value. This is used
	 * to create acceleration/deceleration. Setting this to null will result
	 * in a linear tween. This is a function that accepts a fraction
	 * between 0 and 1 and returns a fraction between 0 and 1. The result is used
	 * to calculate the value based on progress and start/end values. There are several
	 * standard easing functions built in as static functions of Tween that you can set to this.
	 */
	this.easingFunction = null;
}
	
//Tween is base object, no inheritance.
Tween.prototype.constructor = Tween;


/**
 * @function getProgress
 * Gets the current progress of the tween based on the start time and the current time.
 * 
 * @param time Number
 * The current time as returned by Date.now().
 * 
 * @returns Number
 * Fraction between 0 and 1.
 */
Tween.prototype.getProgress = 
	function (time)
	{
		if (time >= this.startTime + this.duration)
			return 1;
		else if (time <= this.startTime)
			return 0;
		
		return (time - this.startTime) / this.duration;
	};
	
/**
 * @function getValue
 * Gets the current value based on the supplied time.
 * 
 * @param time Number
 * The current time as returned by Date.now().
 * 
 * @returns Number
 * A number between the start and end values (inclusive).
 */	
Tween.prototype.getValue = 
	function (time)
	{
		var progress = this.getProgress(time);
		
		if (progress == 1)
			return this.endVal;
		else if (progress == 0)
			return this.startVal;
	
		if (this.easingFunction != null)
			progress = this.easingFunction(progress);
		
		var range = Math.abs(this.endVal - this.startVal);
		
		if (this.startVal < this.endVal)
			return this.startVal + (range * progress);
		
		return this.startVal - (range * progress);
	};

//////Static//////////////////
	
/**
 * @function easeInQuad
 * @static
 * Use for quadratic easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */	
Tween.easeInQuad = 
	function (fraction)
	{
		return fraction * fraction;
	};
	
/**
 * @function easeOutQuad
 * @static
 * Use for quadratic easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */		
Tween.easeOutQuad = 
	function (fraction)
	{
		return 1 - Tween.easeInQuad(1 - fraction);
	};

/**
 * @function easeInOutQuad
 * @static
 * Use for quadratic easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */		
Tween.easeInOutQuad = 
	function (fraction)
	{
		if (fraction < 0.5)
			return Tween.easeInQuad(fraction * 2.0) / 2.0;
		
		return 1 - (Tween.easeInQuad((1 - fraction) * 2.0) / 2.0);  
	};
	
/**
 * @function easeInCubic
 * @static
 * Use for cubic easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */		
Tween.easeInCubic = 
	function (fraction)
	{
		return Math.pow(fraction, 3);
	};
	
/**
 * @function easeOutCubic
 * @static
 * Use for cubic easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */		
Tween.easeOutCubic = 
	function (fraction)
	{
		return 1 - Tween.easeInCubic(1 - fraction);
	};

/**
 * @function easeInOutCubic
 * @static
 * Use for cubic easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */		
Tween.easeInOutCubic = 
	function (fraction)
	{
		if (fraction < 0.5)
			return Tween.easeInCubic(fraction * 2.0) / 2.0;
		
		return 1 - (Tween.easeInCubic((1 - fraction) * 2.0) / 2.0);  
	};	
	
/**
 * @function easeOutSine
 * @static
 * Use for sine easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */		
Tween.easeOutSine = 
	function (fraction)
	{
		return Math.sin(fraction * (Math.PI / 2.0));
	};

/**
 * @function easeInSine
 * @static
 * Use for sine easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */		
Tween.easeInSine = 
	function (fraction)
	{
		return 1 - Tween.easeOutSine(1 - fraction);
	};	
	
/**
 * @function easeInOutSine
 * @static
 * Use for sine easing.
 * 
 * @param fraction Number
 * A number between 0 and 1.
 * 
 * @returns Number
 * A number between 0 and 1.
 */		
Tween.easeInOutSine = 
	function (fraction)
	{
		if (fraction < 0.5)
			return Tween.easeInSine(fraction * 2.0) / 2.0;
		
		return 1 - (Tween.easeInSine((1 - fraction) * 2.0) / 2.0);  
	};
	
	


///////////////////////////////////////////////////////////////////////////	
///////////////////////StyleProxy////////////////////////////////////////		
	
/**
 * @class StyleProxy
 * 
 * Internal class used to wrap CanvasElements to proxy styles to other elements. 
 * This should only be used by component developers. When a proxy is assigned
 * to an element, the proxy is included in its style chain lookup after assigned
 * styles (instance, and styleDefinition) but before default styles.   
 * 
 * @constructor StyleProxy 
 * Creates new StyleProxy instance.
 * 
 * @param styleProxyElement CanvasElement
 * The element to proxy styles from.
 * 
 * @param styleProxyMap Object
 * A map of styleNames to proxy. This Object is walked for members so
 * should always be created using a null prototype: Object.create(null) and
 * members created for each styleName to proxy (set to true). 
 * 
 * MyProxyMap = Object.create(null);
 * MyProxyMap.StyleName1 = true;
 * MyProxyMap.StyleName2 = true;
 * 
 * MyProxyMap._Arbitrary = true; 
 * 
 * _Arbitrary is a special flag that indicates all styles that are not defined / unknown 
 * by the element will also be proxied.
 * 
 * For example, a Button will proxy several styles to its skins such as "BackgroundFill" by including
 * them in the proxy map it passes to its skins. Styles like "Visible" however, are omitted from the proxy
 * map. Also, the button sets the _Arbitrary flag so any styles the Button is not aware of and does not define itself, 
 * are automatically proxied to the skin, without having to be added to the proxy map. 
 * This is so that skins may have custom styles and still be blanket set by setting the Button style itself. 
 */
function StyleProxy(styleProxyElement, styleProxyMap)
{
	this._proxyElement = styleProxyElement;
	this._proxyMap = styleProxyMap;
}

//No Inheritance
StyleProxy.prototype.constructor = StyleProxy;

	


//////////////////////////////////////////////////////////////////////
/////////////////////DispatcherEvent//////////////////////////////////

/**
 * @class DispatcherEvent
 * Base class for all events.
 * 
 * @constructor DispatcherEvent 
 * Creates new DispatcherEvent instance
 * 
 * @param type String
 * String representing the event type
 * 
 */
function DispatcherEvent(type)
{
	this._type = type;
	
	this._target = null;
	this._canceled = false;
}

//DispatcherEvent is base object, no inheritance.
DispatcherEvent.prototype.constructor = DispatcherEvent;


/**
 * @function getType
 * Gets the event type
 * 
 * @returns String
 * String representing the event type
 */
DispatcherEvent.prototype.getType = 
	function ()
	{
		return this._type;
	};

	
/**
 * @function getTarget
 * Gets event target
 * 
 * @returns Object
 * Object that originally dispatched the event 
 */	
DispatcherEvent.prototype.getTarget = 
	function ()
	{
		return this._target;
	};

/**
 * @function cancelEvent
 * Prevents processing of any subsequent event handlers
 */
DispatcherEvent.prototype.cancelEvent = 
	function ()
	{
		this._canceled = true;
	};	
	
/**
 * @function getIsCanceled
 * Checks if the event has been canceled
 * 
 * @returns boolean
 * Returns true if the event has been canceled, otherwise false
 */	
DispatcherEvent.prototype.getIsCanceled = 
	function ()
	{
		return this._canceled;
	};
	
/**
 * @function clone
 * Duplicates an instance of an Event or Event subclass. 
 * The event dispatcher calls this when dispatching or re-dispatching events to multiple targets. 
 * When creating a custom event class, you should override this and call the base class's clone() 
 * then copy the new event properties to the cloned instance.
 * 
 * @returns DispatcherEvent
 * A new event object instance identical to the cloned instance.
 */	
DispatcherEvent.prototype.clone = 
	function ()
	{
		var clonedEvent = this._cloneInstance();
		
		clonedEvent._target = this._target;
		clonedEvent._canceled = this._canceled;
		
		return clonedEvent;
	};
	
/**
 * @function _cloneInstance
 * Calls and returns the constructor() of the appropriate event subclass when cloning an event. 
 * When creating a custom event class, you should override this and return the appropriate event subclass type.
 * 
 * @returns DispatcherEvent
 * A new event instance of the same type being cloned.
 */	
DispatcherEvent.prototype._cloneInstance = 
	function ()
	{
		return new DispatcherEvent(this._type);
	};	
	
	


/**
 * @depends DispatcherEvent.js
 */

//////////////////////////////////////////////////////////////////////
/////////////////////StyleChangedEvent////////////////////////////////	
	
/**
 * @class StyleChangedEvent
 * @inherits DispatcherEvent
 * 
 * Event that is dispatched when a style is changed of type "stylechanged". 
 * This is typically an internal event that the system uses to monitor 
 * style changes to Elements, StyleDefnitions, or any style-able objects.
 * 
 * 
 * @constructor StyleChangedEvent 
 * Creates new StyleChangedEvent instance.
 * 
 * @param styleName String
 * String representing style type that was updated
 */
function StyleChangedEvent(styleName)
{
	StyleChangedEvent.base.prototype.constructor.call(this, "stylechanged");
	
	this._styleName = styleName;
}	
	
//Inherit from DispatcherEvent
StyleChangedEvent.prototype = Object.create(DispatcherEvent.prototype);
StyleChangedEvent.prototype.constructor = StyleChangedEvent;
StyleChangedEvent.base = DispatcherEvent;

/**
 * @function getStyleName
 * Gets the style name of the style which has changed
 * 
 * @returns String
 * String representing the style that has changed
 */
StyleChangedEvent.prototype.getStyleName = 
	function ()
	{
		return this._styleName;
	};
	
//@override
StyleChangedEvent.prototype.clone =
	function ()
	{
		var clonedEvent = StyleChangedEvent.base.prototype.clone.call(this);
		
		//No additional property copies (handled by constructor)
		
		return clonedEvent;
	};
	
//@override
StyleChangedEvent.prototype._cloneInstance = 
	function ()
	{
		return new StyleChangedEvent(this._styleName, this._oldValue, this._newValue);
	};
	
	


/**
 * @depends DispatcherEvent.js
 */

//////////////////////////////////////////////////////////////////////
/////////////////////ElementEvent/////////////////////////////////////	

/**
 * @class ElementEvent
 * @inherits DispatcherEvent
 * 
 * Base class for CanvasElement UI events. ElementEvents support
 * capture and bubbling phase when dispatched from CanvasElement(s). A bubbling event
 * invokes capture listeners from the root parent to the target child element and then
 * bubbling (normal) listeners from the target element to the root parent. 
 * Bubbling events are used to detect events dispatched on child elements.
 * Capture events are typically not needed but sometimes useful if you wish to 
 * detect an event before the target has a chance to process it.
 * 
 * For Example, when a Button dispatches a ElementMouseEvent.
 * The event propagates from the root parent (CanvasManager) down the display chain
 * from child to child dispatching capture events to any parents with registered listeners. 
 * Once reaching the target element (Button) the event then propagates back up the display chain 
 * from parent to parent dispatching bubbling events.  
 * You may cancel the event at any time to stop the event flow.
 * 
 * 
 * @constructor ElementEvent 
 * Creates new ElementEvent instance.
 * 
 * @param type String
 * String representing the event type
 * 
 * @param bubbles boolean
 * True if the ElementEvent should be dispatch capture and bubbling events.
 */

function ElementEvent(type, bubbles)
{
	ElementEvent.base.prototype.constructor.call(this, type);
	
	this._currentTarget = null;
	this._bubbles = bubbles;
	this._phase = null;  // "capture" || "bubble"
	this._defaultPrevented = false;
}

//Inherit from DispatcherEvent
ElementEvent.prototype = Object.create(DispatcherEvent.prototype);
ElementEvent.prototype.constructor = ElementEvent;
ElementEvent.base = DispatcherEvent;

/**
 * @function getCurrentTarget
 * 
 * Gets the element that is currently dispatching the event. Note that is
 * is not always the same as getTarget() which returns the element that
 * originally dispatched the event. 
 * 
 * For Example, when a click listener is registered to an element, and a child of that
 * element dispatches a click (like a Button), the target will be the child (Button) and the 
 * current target will be the element that registered the click listener.
 * 
 * 
 * @returns CanvasElement
 * The element that is currently dispatching the event.
 */
ElementEvent.prototype.getCurrentTarget = 
	function ()
	{
		return this._currentTarget;
	};

/**
 * @function getPhase
 * 
 * Gets the current phase of the event. ("bubbling" or "capture")
 * 
 * @returns String
 * String representing the event's current phase when dispatched ("bubbling" or "capture")
 */
ElementEvent.prototype.getPhase = 
	function ()
	{
		return this._phase;
	};
	
/**
 * @function preventDefault
 * 
 * Prevents the event's typical action from being taken. This is also sometimes used to "consume"
 * the event so it is only processed once. Such as preventing a mousewheel event from scrolling multiple
 * parent/child views at once. A scroll-able child will call preventDefault() to "consume" the event
 * and prevent any parents from also scrolling.
 */	
ElementEvent.prototype.preventDefault = 
	function ()
	{
		this._defaultPrevented = true;
	};

/**
 * @function getDefaultPrevented
 * 
 * Gets the default prevented state of the event.
 * 
 * @returns boolean
 * Returns true if preventDefault() has been called, false otherwise.
 */	
ElementEvent.prototype.getDefaultPrevented = 
	function ()
	{
		return this._defaultPrevented;
	};
	
//@Override	
ElementEvent.prototype.clone = 
	function ()
	{
		var clonedEvent = ElementEvent.base.prototype.clone.call(this);
		
		clonedEvent._currentTarget = this._currentTarget;
		clonedEvent._phase = this._phase;
		clonedEvent._defaultPrevented = this._defaultPrevented;
		clonedEvent._bubbles = this._bubbles; //Need to set, some subclasses always pass true in constructor.
		
		return clonedEvent;
	};
	
//@Override	
ElementEvent.prototype._cloneInstance = 
	function ()
	{
		return new ElementEvent(this._type, this._bubbles);
	};


/**
 * @depends ElementEvent.js
 */

//////////////////////////////////////////////////////////////////////////	
/////////////////////ElementMouseEvent////////////////////////////////////

/**
 * @class ElementMouseEvent
 * @inherits ElementEvent
 * 
 * Event class used to represent mouse events of type "mousedown", "mouseup" or "click". 
 * Every "mousedown" event is always paired with a "mouseup" event. Note that the mouse is
 * not necessarily still over the same object when "mouseup" is dispatched. The user may have
 * pressed and then moved the mouse before releasing. A "click" event however, is only dispatched
 * if the mouse is still over the "mousedown" object when the mouse is released.
 * 
 * @constructor ElementMouseEvent 
 * Creates new ElementMouseEvent instance.
 * 
 * @param type String
 * String representing the event type ("mousedown", "mouseup", or "click")
 * 
 * @param x int
 * The X coordinate of the mouse relative to the object dispatching the mouse event.
 * 
 * @param y int
 * The Y coordinate of the mouse relative to the object dispatching the mouse event.
 */
function ElementMouseEvent(type, x, y)
{
	ElementMouseEvent.base.prototype.constructor.call(this, type, true);
	
	this._x = x; 
	this._y = y;
}

//Inherit from ElementEvent
ElementMouseEvent.prototype = Object.create(ElementEvent.prototype);
ElementMouseEvent.prototype.constructor = ElementMouseEvent;
ElementMouseEvent.base = ElementEvent;

/**
 * @function getX
 * 
 * Gets the X coordinate of the mouse relative to the object dispatching the mouse event. 
 * 
 * @returns int
 * The X coordinate of the mouse relative to the object dispatching the mouse event. 
 */
ElementMouseEvent.prototype.getX = 
	function()
	{
		return this._x;
	};
	
/**
 * @function getY
 * 
 * Gets the Y coordinate of the mouse relative to the object dispatching the mouse event. 
 * 
 * @returns int
 * The Y coordinate of the mouse relative to the object dispatching the mouse event. 
 */	
ElementMouseEvent.prototype.getY = 
	function()
	{
		return this._y;
	};
	
//@Override
ElementMouseEvent.prototype.clone =
	function ()
	{
		var clonedEvent = ElementMouseEvent.base.prototype.clone.call(this);
		
		//No additional property copies (handled by constructor)
		
		return clonedEvent;
	};
	
//@Override
ElementMouseEvent.prototype._cloneInstance = 
	function ()
	{
		return new ElementMouseEvent(this._type, this._x, this._y);
	};
	


/**
 * @depends ElementMouseEvent.js
 */

//////////////////////////////////////////////////////////////////////////
/////////////////////ElementMouseWheelEvent///////////////////////////////

/**
 * @class ElementMouseWheelEvent
 * @inherits ElementMouseEvent
 * 
 * Event class used to represent mouse wheel events of type "wheel". 
 * 
 * 
 * @constructor ElementMouseWheelEvent 
 * Creates new ElementMouseWheelEvent instance.
 * 
 * @param x int
 * The X coordinate of the mouse relative to the object dispatching the mouse event.
 * 
 * @param y int
 * The Y coordinate of the mouse relative to the object dispatching the mouse event.
 * 
 * @param deltaX int
 * The change of the X position of the the mouse wheel. (Currently -1, 0, or +1)
 * 
 * @param deltaY int
 * The change of the Y position of the the mouse wheel. (Currently -1, 0, or +1)
 */
function ElementMouseWheelEvent(x, y, deltaX, deltaY)
{
	ElementMouseWheelEvent.base.prototype.constructor.call(this, "wheel", x, y);

	this._deltaX = deltaX;
	this._deltaY = deltaY;
}

//Inherit from ElementMouseEvent
ElementMouseWheelEvent.prototype = Object.create(ElementMouseEvent.prototype);
ElementMouseWheelEvent.prototype.constructor = ElementMouseWheelEvent;
ElementMouseWheelEvent.base = ElementMouseEvent;	

/**
 * @function getDeltaX
 * 
 * Gets the change of the X position of the mouse wheel. The system normalizes this
 * across browsers to values -1, 0, or +1. 
 * 
 * @returns int
 * The change of the X position of the mouse wheel.
 */
ElementMouseWheelEvent.prototype.getDeltaX = 
	function()
	{
		return this._deltaX;
	};
	
/**
 * @function getDeltaY
 * 
 * Gets the change of the Y position of the mouse wheel. The system normalizes this
 * across browsers to values -1, 0, or +1. 
 * 
 * @returns int
 * The change of the Y position of the mouse wheel.
 */	
ElementMouseWheelEvent.prototype.getDeltaY = 
	function()
	{
		return this._deltaY;
	};
	
//@Override
ElementMouseWheelEvent.prototype.clone =
	function ()
	{
		var clonedEvent = ElementMouseWheelEvent.base.prototype.clone.call(this);
		
		//No additional property copies (handled by constructor)
		
		return clonedEvent;
	};
	
//@Override
ElementMouseWheelEvent.prototype._cloneInstance = 
	function ()
	{
		return new ElementMouseWheelEvent(this._x, this._y, this._deltaX, this._deltaY);
	};
	


/**
 * @depends ElementEvent.js
 */

//////////////////////////////////////////////////////////////////////////
/////////////////////ElementListItemClickEvent////////////////////////////

/**
 * @class ElementListItemClickEvent
 * @inherits ElementEvent
 * 
 * Event class dispatched when a DataRenderer is clicked of type "listitemclick". 
 * 
 * 
 * @constructor ElementListItemClickEvent 
 * Creates new ElementListItemClickEvent instance.
 * 
 * @param item Object
 * The collection item associated with the DataRenderer that was clicked.
 * 
 * @param index int
 * The collection index associated with the DataRenderer that was clicked.
 */
function ElementListItemClickEvent(item, index)
{
	ElementListItemClickEvent.base.prototype.constructor.call(this, "listitemclick", false);
	
	this._item = item;
	this._index = index;
}

//Inherit from ElementEvent
ElementListItemClickEvent.prototype = Object.create(ElementEvent.prototype);
ElementListItemClickEvent.prototype.constructor = ElementListItemClickEvent;
ElementListItemClickEvent.base = ElementEvent;	

/**
 * @function getItem
 * Gets the collection item associated with the DataRenderer that was clicked.
 * 
 * @returns Object
 * The collection item associated with the DataRenderer that was clicked.
 */
ElementListItemClickEvent.prototype.getItem = 
	function()
	{
		return this._item;
	};

/**
 * @function getIndex
 * Gets the collection index associated with the DataRenderer that was clicked.
 * 
 * @returns int
 * The collection index associated with the DataRenderer that was clicked.
 */	
ElementListItemClickEvent.prototype.getIndex = 
	function()
	{
		return this._index;
	};

//@Override
ElementListItemClickEvent.prototype.clone =
	function ()
	{
		var clonedEvent = ElementListItemClickEvent.base.prototype.clone.call(this);

		//No additional property copies (handled by constructor)

		return clonedEvent;
};

//@Override
ElementListItemClickEvent.prototype._cloneInstance = 
	function ()
	{
		return new ElementListItemClickEvent(this._item, this._index);
	};	


/**
 * @depends ElementEvent.js
 */

//////////////////////////////////////////////////////////////////////////
/////////////////////ElementKeyboardEvent/////////////////////////////////

/**
 * @class ElementKeyboardEvent
 * @inherits ElementEvent
 * 
 * Event class used to represent keyboard events of type "keydown" or "keyup". 
 * Note that unlike mouse events, every "keydown" is not necessarily paired with a "keyup".
 * When a key is held down, "keydown" events will repeatedly be dispatched until the key
 * is released which will then dispatch a "keyup" event.
 * 
 * @constructor ElementKeyboardEvent 
 * Creates new ElementKeyboardEvent instance.
 * 
 * @param type String
 * String representing the event type ("keydown" or "keyup")
 * 
 * @param key String
 * Printable representation of the key. If the key is not printable such as
 * Shift or Return this should be an emtpy string "".
 * 
 * @param keyCode int
 * The keycode of the key that caused the event.
 * 
 * @param ctrl boolean
 * True if the ctrl key is pressed, false otherwise.
 * 
 * @param alt boolean
 * True if the alt key is pressed, false otherwise.
 * 
 * @param shift boolean
 * True if the shift key is pressed, false otherwise.
 * 
 * @param meta boolean
 * True if the meta key (such as windows key) is pressed, false otherwise.
 */
function ElementKeyboardEvent(type, key, keyCode, ctrl, alt, shift, meta)
{
	ElementKeyboardEvent.base.prototype.constructor.call(this, type, true);
	
	//IE key names are different... normalize
	if (key == "Spacebar")
		key = " ";
	else if (key == "Left")
		key = "ArrowLeft";
	else if (key == "Right")
		key = "ArrowRight";
	else if (key == "Up")
		key = "ArrowUp";
	else if (key == "Down")
		key = "ArrowDown";
	else if (key == "Del")
		key = "Delete";
	
	this._key = key;
	this._keyCode = keyCode;
	
	this._ctrl = ctrl;
	this._alt = alt;
	this._shift = shift;
	this._meta = meta;
}

//Inherit from ElementEvent
ElementKeyboardEvent.prototype = Object.create(ElementEvent.prototype);
ElementKeyboardEvent.prototype.constructor = ElementKeyboardEvent;
ElementKeyboardEvent.base = ElementEvent;	

/**
 * @function getKey
 * 
 * Gets the printable version of the key which caused the event. 
 * 
 * @returns String
 * The printable version of the key which caused the event. Empty string "" if the
 * key is not printable.
 */
ElementKeyboardEvent.prototype.getKey = 
	function ()
	{
		return this._key;
	};
	
/**
 * @function getKeyCode
 * 
 * Gets the key code of the key which caused the event. 
 * 
 * @returns int
 * The keycode of the key which caused the event.
 */	
ElementKeyboardEvent.prototype.getKeyCode = 
	function ()
	{
		return this._keyCode;
	};	

/**
 * @function getCtrl
 * 
 * Gets the state of the ctrl key. 
 * 
 * @returns boolean
 * True if the ctrl key is pressed, otherwise false.
 */	
ElementKeyboardEvent.prototype.getCtrl = 
	function ()
	{
		return this._ctrl;
	};		

/**
 * @function getAlt
 * 
 * Gets the state of the alt key. 
 * 
 * @returns boolean
 * True if the alt key is pressed, otherwise false.
 */		
ElementKeyboardEvent.prototype.getAlt = 
	function ()
	{
		return this._alt;
	};

/**
 * @function getShift
 * 
 * Gets the state of the shift key. 
 * 
 * @returns boolean
 * True if the shift key is pressed, otherwise false.
 */	
ElementKeyboardEvent.prototype.getShift = 
	function ()
	{
		return this._shift;
	};	
	
/**
 * @function getMeta
 * 
 * Gets the state of the meta key (such as the windows key). 
 * 
 * @returns boolean
 * True if the meta key is pressed, otherwise false.
 */		
ElementKeyboardEvent.prototype.getMeta = 
	function ()
	{
		return this._meta;
	};		
	
//@Override
ElementKeyboardEvent.prototype.clone =
	function ()
	{
		var clonedEvent = ElementKeyboardEvent.base.prototype.clone.call(this);
		
		//No additional property copies (handled by constructor)
		
		return clonedEvent;
	};
	
//@Override
ElementKeyboardEvent.prototype._cloneInstance = 
	function ()
	{
		return new ElementKeyboardEvent(this._type, 
				this._key, this._keyCode, 
				this._ctrl, this._alt, this._shift, this._meta);
	};	
	


/**
 * @depends ElementListItemClickEvent.js
 */

//////////////////////////////////////////////////////////////////////////
/////////////////////ElementGridItemClickEvent////////////////////////////

/**
 * @class ElementGridItemClickEvent
 * @inherits ElementListItemClickEvent
 * 
 * Event class dispatched when a DataGrid DataRenderer is clicked of type "listitemclick". 
 * 
 * 
 * @constructor ElementGridItemClickEvent 
 * Creates new ElementGridItemClickEvent instance.
 * 
 * @param item Object
 * The collection item associated with the DataRenderer that was clicked.
 * 
 * @param index int
 * The collection index associated with the DataRenderer that was clicked.
 * 
 * @param columnIndex int
 * The column index associated with the DataRenderer that was clicked.
 */
function ElementGridItemClickEvent(item, index, columnIndex)
{
	ElementGridItemClickEvent.base.prototype.constructor.call(this, item, index);
	
	this._columnIndex = columnIndex;
}

//Inherit from ElementListItemClickEvent
ElementGridItemClickEvent.prototype = Object.create(ElementListItemClickEvent.prototype);
ElementGridItemClickEvent.prototype.constructor = ElementGridItemClickEvent;
ElementGridItemClickEvent.base = ElementListItemClickEvent;	

/**
 * @function getColumnIndex
 * Gets the column index that dispatched the event.
 * 
 * @returns int
 * Column index.
 */
ElementGridItemClickEvent.prototype.getColumnIndex = 
	function()
	{
		return this._columnIndex;
	};

//@Override
ElementGridItemClickEvent.prototype.clone =
	function ()
	{
		var clonedEvent = ElementGridItemClickEvent.base.prototype.clone.call(this);
		
		//No additional property copies (handled by constructor)
		
		return clonedEvent;
	};

//@Override
ElementGridItemClickEvent.prototype._cloneInstance = 
	function ()
	{
		return new ElementListItemClickEvent(this._item, this._index, this._columnIndex);
	};	


/**
 * @depends DispatcherEvent.js
 */

///////////////////////////////////////////////////////////////////
//////////////////CollectionChangedEvent///////////////////////////	
	
/**
 * @class CollectionChangedEvent
 * @inherits DispatcherEvent
 * 
 * Event that is dispatched when a data collection is changed of type "collectionchanged". 
 * This is typically an internal event that the data driven containers use to monitor 
 * changes to their data collections.
 * 
 * 
 * @constructor CollectionChangedEvent 
 * Creates new CollectionChangedEvent instance.
 * 
 * @param kind String
 * String representing type of change that occurred to the collection.
 * Allowable values are "add", "remove", "update", and "reset".
 * 
 * @param index int
 * Index position the change occurred (or -1 if kind is "reset").
 */
function CollectionChangedEvent(kind, index)
{
	CollectionChangedEvent.base.prototype.constructor.call(this, "collectionchanged");
	
	this._kind = kind;
	this._index = index;
}	
	
//Inherit from DispatcherEvent
CollectionChangedEvent.prototype = Object.create(DispatcherEvent.prototype);
CollectionChangedEvent.prototype.constructor = CollectionChangedEvent;
CollectionChangedEvent.base = DispatcherEvent;

/**
 * @function getKind
 * Gets the kind of the collection event. Possible types are "add", "remove", "update", and "reset".
 * 
 * @returns String
 * The kind of the collection event.
 */
CollectionChangedEvent.prototype.getKind = 
	function ()
	{
		return this._kind;
	};
	
/**
 * @function getIndex
 * Gets the index of the data collection that the change occurred.
 * 
 * @returns int
 * The index of the data collection that the change occurred.
 */	
CollectionChangedEvent.prototype.getIndex = 
	function ()
	{
		return this._index;
	};

//@Override
CollectionChangedEvent.prototype.clone =
	function ()
	{
		var clonedEvent = CollectionChangedEvent.base.prototype.clone.call(this);
		
		//No additional property copies (handled by constructor)
		
		return clonedEvent;
	};
	
//@Override
CollectionChangedEvent.prototype._cloneInstance = 
	function ()
	{
		return new CollectionChangedEvent(this._kind, this._index);
	};
	
	


/**
 * @depends DispatcherEvent.js
 */

//////////////////////////////////////////////////////////////////////
/////////////////////AddedRemovedEvent////////////////////////////////		
	
/**
 * @class AddedRemovedEvent
 * @inherits DispatcherEvent
 * 
 * Event that is dispatched when a CanvasElement is added or removed from
 * a CanvasManager and can be of type "added" or "removed". This is used to detect
 * when an Element is added or removed from the display and to create / destroy or 
 * associate / dissociate resources. For example, CanvasElement uses these events
 * to add and remove event listeners to its associated StyleDefinitions which are 
 * external resources and would cause memory leaks if not cleaned up. 
 * 
 * 
 * @constructor AddedRemovedEvent 
 * Creates new AddedRemovedEvent instance.
 * 
 * @param type String
 * String representing the event type ("added" or "removed")
 * 
 * @param manager CanvasManager
 * The CanvasManager instance that the element is being added or removed.
 */
function AddedRemovedEvent(type, manager)
{
	AddedRemovedEvent.base.prototype.constructor.call(this, type);
	
	this._manager = manager;
}	
	
//Inherit from DispatcherEvent
AddedRemovedEvent.prototype = Object.create(DispatcherEvent.prototype);
AddedRemovedEvent.prototype.constructor = AddedRemovedEvent;
AddedRemovedEvent.base = DispatcherEvent;

/**
 * @function getManager
 * Gets the CanvasManager instance that the Element has been added or removed. 
 * Note that when an element is removed, the Element is no longer associated with the CanvasManager
 * so you must use this method to get the appropriate CanvasManager reference.
 * 
 * @returns CanvasManager
 * The CanvasManager instance the element is now associated with when added, or no longer associated with when removed.
 */
AddedRemovedEvent.prototype.getManager = 
	function ()
	{
		return this._manager;
	};
	
//@Override
AddedRemovedEvent.prototype.clone =
	function ()
	{
		var clonedEvent = AddedRemovedEvent.base.prototype.clone.call(this);
		
		//No additional property copies (handled by constructor)
		
		return clonedEvent;
	};
	
//@Override
AddedRemovedEvent.prototype._cloneInstance = 
	function ()
	{
		return new AddedRemovedEvent(this._type, this._manager);
	};	
	
	


//////////////////////////////////////////////////////////////////////
/////////////////////EventDispatcher//////////////////////////////////
	
/**
 * @class EventDispatcher
 * Base class for all objects that dispatch events.
 * 
 * @constructor EventDispatcher 
 * Creates new EventDispatcher instance.
 */
function EventDispatcher()
{
	this._eventListeners = Object.create(null); //Map of arrays by event name.
}
	
//EventDispatcher is base object, no inheritance.
EventDispatcher.prototype.constructor = EventDispatcher;

///////////EventDispatcher Public Functions///////////////////////////

/**
 * @function addEventListener
 * Registers an event lister function to be called when an event occurs.
 * 
 * @param type String
 * String representing the event type.
 * 
 * @param callback Function
 * Function to be called when the event occurs.
 */
EventDispatcher.prototype.addEventListener = 
	function (type, callback)
	{
		if (this._eventListeners[type] == null)
			this._eventListeners[type] = [];
		
		this._eventListeners[type].push(callback);
	};

/**
 * @function removeEventListener
 * Removes a callback from the EventDispatcher
 * 
 * @param type String
 * String representing the event type.
 * 
 * @param callback Function
 * Function callback to be removed.
 * 
 * @returns boolean
 * Returns true if the callback was successfully removed, otherwise false
 * such as if the function callback was not previously registered.
 */	
EventDispatcher.prototype.removeEventListener = 
	function (type, callback)
	{
		if (!(type in this._eventListeners))
			return false;
	
		for (var i = 0; i < this._eventListeners[type].length; i++)
		{
			if (this._eventListeners[type][i] == callback)
			{
				this._eventListeners[type].splice(i, 1);
				return true;
			}
		}
		
		return false;
	};

/**
 * @function hasEventListener
 * Checks if an event listener has been registered with this EventDispatcher
 * 
 * @param type String
 * String representing the event type.
 * 
 * @param callback Function
 * Function callback to be called when the event occurs. This may be null to check
 * if the EventDispatcher has any events registered for the provided type.
 * 
 * @returns boolean
 * Returns true if the EventDispatcher has the provided callback registered for the 
 * provided type, or any callback for the provided type if the callback parameter is null.
 * Otherwise, returns false.
 */	
EventDispatcher.prototype.hasEventListener = 
	function (type, callback)
	{
		if (!(type in this._eventListeners))
			return false;
	
		if (callback == null)
		{
			if (this._eventListeners[type].length > 0)
				return true;
			
			return false;
		}
		
		
		for (var i = 0; i < this._eventListeners[type].length; i++)
		{
			if (this._eventListeners[type][i] == callback)
				return true;
		}
		
		return false;
	};
	
/**
 * @function dispatchEvent
 * Dispatches an event to be processed by registered event listeners. The Event's target is the
 * EventDispatcher which called dispatchEvent. The Event will be cloned prior to passing to callback functions
 * to ensure the callback cannot modify the Event data or properties. You can check if the event was canceled
 * by calling the Event's getIsCanceled after dispatching it. Re-dispatching the same event will re-set its canceled state to false.
 * 
 * @param event DispatcherEvent
 * The DispatcherEvent class or subclass to be dispatched. 
 */	
EventDispatcher.prototype.dispatchEvent = 
	function (event)
	{
		event._canceled = false;
	
		if (event._type in this._eventListeners && this._eventListeners[event._type].length > 0)
		{
			//Copy the list of event handlers, if event handlers add/remove other handlers or themselves, 
			//we dont want to miss an event, or inconsistently dispatch newly added events.
			var listeners = this._eventListeners[event._type].slice();
			
			//TODO: Sort by priority (no priority available yet).
			
			var cloneEvent = null;
			for (var i = 0; i < listeners.length; i++)
			{
				//Clone the event so the handler can't fudge our event data.
				cloneEvent = event.clone(); 
				cloneEvent._target = this;				

				listeners[i](cloneEvent);
				
				if (cloneEvent._canceled == true)
				{
					event._canceled = true;
					return;
				}
			}
		}
	};	
	
	
	


/**
 * @depends EventDispatcher.js
 */

///////////////////////////////////////////////////////////////////
///////////////////////ToggleButtonGroup///////////////////////////

/**
 * @class ToggleButtonGroup
 * @inherits EventDispatcher
 * 
 * Convenience helper class for grouping ToggleButtons or subclasses.
 * The ToggleButtonGroup can be assigned to set of toggle buttons
 * and will only allow a single ToggleButton to be selected at a time.
 * When a ToggleButton changes state, the ToggleButtonGroup will dispatch
 * a changed event. Use this for functionality like RadioButtons and Tabs.
 * 
 * @constructor ToggleButtonGroup 
 * Creates new ToggleButtonGroup instance.
 */
function ToggleButtonGroup()
{
	ToggleButtonGroup.base.prototype.constructor.call(this);
	
	this._selectedButton = null;
	
	this._toggleButtons = [];
	
	var _self = this;
	
	this._toggleButtonChangedInstance =
		function (event)
		{
			_self._toggleButtonChanged(event);
		};
}

//Inherit from EventDispatcher
ToggleButtonGroup.prototype = Object.create(EventDispatcher.prototype);
ToggleButtonGroup.prototype.constructor = ToggleButtonGroup;
ToggleButtonGroup.base = EventDispatcher;

////////////Events/////////////////////////////////////

/**
 * @event changed DispatcherEvent
 * Dispatched when the selected ToggleButton is changed due to user interaction.
 */

//////////////Public Functions/////////////////////////////////////////

/**
 * @function addButton
 * Adds a ToggleButton or subclass to be managed by ToggleButtonGroup.
 * 
 * @param toggleButton ToggleButtonElement
 * ToggleButton or subclass to be managed by ToggleButtonGroup.
 * 
 * @returns boolean
 * True when successfully added, false if is not a instance of ToggleButton or already added.
 */	
ToggleButtonGroup.prototype.addButton = 
	function (toggleButton)
	{
		if (toggleButton == null || 
			toggleButton instanceof ToggleButtonElement == false ||
			this._toggleButtons.indexOf(toggleButton) > -1)
			return false;
		
		this._toggleButtons.push(toggleButton);
		toggleButton.addEventListener("changed", this._toggleButtonChangedInstance);
		
		return true;
	};

/**
 * @function removeButton
 * Removes a ToggleButton or subclass currently being managed by ToggleButtonGroup
 * 
 * @param toggleButton ToggleButtonElement
 * ToggleButton or subclass to be removed from ToggleButtonGroup.
 * 
 * @returns boolean
 * True when successfully removed, false if the toggle button is not currently managed by ToggleButtonGroup.
 */		
ToggleButtonGroup.prototype.removeButton = 
	function (toggleButton)
	{
		var index = this._toggleButtons.indexOf(toggleButton);
	
		if (index == -1)
			return false;
		
		this._toggleButtons.splice(index, 1);
		toggleButton.removeEventListener("changed", this._toggleButtonChangedInstance);
	};	

/**
 * @function clearButtons
 * Removes all ToggleButtons currently managed by ToggleButtonGroup.
 */		
ToggleButtonGroup.prototype.clearButtons = 
	function ()
	{
		for (var i = 0; i < this._toggleButtons.length; i++)
			this._toggleButtons[i].removeEventListener("changed", this._toggleButtonChangedInstance);
		
		this._toggleButtons = [];
	};
	
/**
 * @function setSelectedButton
 * Sets the ToggleButton to be selected.
 * 
 * @param toggleButton ToggleButtonElement
 * ToggleButton or subclass to be selected. May be set to null.
 */	
ToggleButtonGroup.prototype.setSelectedButton = 
	function (toggleButton)
	{
		if (this._selectedButton == toggleButton)
			return;
			
		if (toggleButton == null || this._toggleButtons.indexOf(toggleButton) > -1)
		{
			this._selectedButton = toggleButton;
			
			if (this._selectedButton != null)
				this._selectedButton.setSelected(true);
			
			for (var i = 0; i < this._toggleButtons.length; i++)
			{
				if (this._toggleButtons[i] != toggleButton)
					this._toggleButtons[i].setSelected(false);
			}
		}
	};
	
/**
 * @function getSelectedButton
 * Gets the selected ToggleButton.
 * 
 * @returns ToggleButtonElement
 * ToggleButton or subclass currently selected. May be null.
 */		
ToggleButtonGroup.prototype.getSelectedButton = 
	function ()
	{
		return this._selectedButton;
	};
	
////////////////////Internal/////////////////////////
	
	
/**
 * @function _toggleButtonChanged
 * Event handler for managed ToggleButton's "changed" event. 
 * Updates toggle button selected states and dispatches "changed" event.
 * 
 * @param event ElementEvent
 * ElementEvent to be processed.
 */		
ToggleButtonGroup.prototype._toggleButtonChanged = 
	function (elementEvent)
	{
		var toggleButton = elementEvent.getTarget();
		
		if (toggleButton.getSelected() == true)
			this._selectedButton = toggleButton;
		else
			this._selectedButton = null;
		
		for (var i = 0; i < this._toggleButtons.length; i++)
		{
			if (this._toggleButtons[i] != toggleButton)
				this._toggleButtons[i].setSelected(false);
		}
		
		//Dispatch changed event.
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new DispatcherEvent("changed", false));
	};
	
	


/**
 * @depends EventDispatcher.js
 */

///////////////////////////////////////////////////////////////////
///////////////////////StyleDefinition/////////////////////////////	

/**
 * @class StyleDefinition
 * @inherits EventDispatcher
 * 
 * Stores a key value data set of style values by name.
 * 
 * 
 * @constructor StyleDefinition 
 * Creates new StyleDefinition instance.
 */
function StyleDefinition()
{
	StyleDefinition.base.prototype.constructor.call(this);

	this._styleMap = Object.create(null);
}

//Inherit from EventDispatcher
StyleDefinition.prototype = Object.create(EventDispatcher.prototype);
StyleDefinition.prototype.constructor = StyleDefinition;
StyleDefinition.base = EventDispatcher;

/**
 * @event stylechanged StyleChangedEvent
 * 
 * Dispatched when a style is added, cleared, or changed.
 */


/**
 * @function getStyle
 * Gets the stored style value for this object.
 * 
 * @param styleName String
 * String representing the style to return.
 * 
 * @returns Any
 * Returns the associated style value if found, otherwise undefined.
 */	
StyleDefinition.prototype.getStyle = 
	function (styleName)
	{
		if (styleName in this._styleMap)
			return this._styleMap[styleName];
		
		return undefined;
	};

/**
 * @function setStyle
 * Sets the stored style value for this object.
 * 
 * @param styleName String
 * String representing the style to set.
 * 
 * @param value Any
 * The value to store. This may be null or undefined. 
 * Note that a null style is different from an absent (undefined) style. A null style
 * will terminate a style chain lookup and return null value. An undefined style will cause
 * the system to look further up the style chain for a value. Passing undefined is the
 * same as calling clearStyle().
 */
StyleDefinition.prototype.setStyle = 
	function (styleName, value)
	{
		var oldStyle = undefined;
		if (styleName in this._styleMap)
			oldStyle = this._styleMap[styleName];
		
		//No change
		if (oldStyle === value)
			return;
	
		if (this.hasEventListener("stylechanged", null) == true)
		{
			oldStyle = this.getStyle(styleName);
			
			if (value === undefined)
				delete this._styleMap[styleName];
			else
				this._styleMap[styleName] = value;
			
			var newStyle = this.getStyle(styleName);
			
			//Strict equality required (undefined !== null)
			if (newStyle !== oldStyle)
				this.dispatchEvent(new StyleChangedEvent(styleName));
		}
		else
		{
			if (value === undefined)
				delete this._styleMap[styleName];
			else
				this._styleMap[styleName] = value;
		}
	};

/**
 * @function clearStyle
 * Clears style data from this object. This is the same 
 * passing undefined to setStyle().
 * 
 * @param styleName String
 * String representing the style to clear.
 */		
StyleDefinition.prototype.clearStyle = 
	function (styleName)
	{
		this.setStyle(styleName, undefined);
	};
	
	


/**
 * @depends StyleDefinition.js
 */

///////////////////////////////////////////////////////////////////////////	
/////////////////////////StyleData/////////////////////////////////////////

/**
 * @class StyleData
 * 
 * Storage class for style data.
 * 
 * 
 * @constructor StyleData 
 * Creates new StyleData instance.
 * 
 * @param styleName String
 * String representing style name for associated data.
 */

function StyleData(styleName)
{
	/**
	 * @member styleName string
	 * Read Only - Name of associated style
	 */
	this.styleName = styleName;
	
	/**
	 * @member value Any
	 * Read Only - Value of associated style 
	 */
	this.value = undefined;
	
	/**
	 * @member priority Array
	 * Read Only - Array of integers representing the position 
	 * in the style chain the style was found.
	 */
	this.priority = [];
}

//StyleData is base object, no inheritance.
StyleData.prototype.constructor = StyleData;

/**
 * @function equals
 * 
 * Checks if two instances of StyleData contain the same values.
 * 
 * @param equalToStyleData StyleData
 * StyleData instance to compare.
 * 
 * @returns boolean
 * Returns true when both instances contain the same values.
 */
StyleData.prototype.equals = 
	function (equalToStyleData)
	{
		if (equalToStyleData.styleName != this.styleName || 
			equalToStyleData.priority.length != this.priority.length ||
			equalToStyleData.value !== this.value)	//Strict equality required (undefined !== null)
		{
			return false;
		}
			
		for (var i = 0; i < this.priority.length; i++)
		{
			if (equalToStyleData.priority[i] != this.priority[i])
				return false;
		}
		
		return true;
	};
	
/**
 * @function comparePriority
 * Compares the style priority (position in style chain) of two StyleData instances.
 * 
 * @param compareToStyleData StyleData
 * StyleData instance to compare.
 * 
 * @returns int
 * Returns -1 if this instance is lower priority than compareToStyleData.
 * Returns +1 if this instance is higher priority than compareToStyleData.
 * Returns 0 if this instance is the same priority as compareToStyleData.
 */	
StyleData.prototype.comparePriority = 
	function (compareToStyleData)
	{
		var minLength = Math.min(this.priority.length, compareToStyleData.priority.length);
		
		for (var i = 0; i < minLength; i++)
		{
			if (this.priority[i] < compareToStyleData.priority[i])
				return +1;
			
			if (this.priority[i] > compareToStyleData.priority[i])
				return -1;
		}

		//Dont worry about different lengths... 
		//Anything with an equal priority will be the same length.
		return 0;
	};
	

	
/**
 * @function clone
 * Duplicates an instance of StyleData (deep copy). 
 * 
 * @returns StyleData
 * A new StyleData instance identical to the cloned instance.
 */	
StyleData.prototype.clone = 
function ()
{
	var cloned = new StyleData(this.styleName);
	cloned.value = this.value;
	cloned.priority = this.priority.slice();
	
	return cloned;
};	
	
	

///////////////////////////////////////////////////////////////////////////	
///////////////////////StyleableBase///////////////////////////////////////
	
/**
 * @class StyleableBase
 * @inherits StyleDefinition
 * 
 * Internal abstract base class for classes that define styles. 
 * 
 * StylableBase defines no styles itself, but allows subclasses to define them by adding  
 * and populating static Object _StyleTypes and StyleDefinition StyleDefault on the class.
 * See example.
 * 
 * There are three different types of styles, NORMAL, INHERITABLE, and SUBSTYLE.
 * 
 * StyleableBase.EStyleType.NORMAL: No special behavior.
 * 
 * StyleableBase.EStyleType.INHERITABLE: Only applicable for CanvasElements.
 * If no explicit style is set (instance, style definition, or proxy) look up the
 * parent chain for the first element supporting the style with inheritable.
 * If no style is found up the parent chain, use the element's default style.
 * 
 * StyleableBase.EStyleType.SUBSTYLE: Only applicable for CanvasElements.
 * Sub styles are a StyleDefinition or array of [StyleDefinition]s to be applied
 * as the style definition(s) of an element's sub component. For example, elements
 * that supports skins, will have several sub styles, one for each skin element. 
 * You can apply styles to the skin elements, by setting StyleDefinitions to the elements
 * "___SkinStyle" sub styles. This is not limited to skins, many elements have sub styles
 * for many sub components, such as built in scroll bars, buttons, or other sub components.
 * 
 * The system handles sub styles differently than other styles, as sub styles are [StyleDefinition]s
 * that can contain many individual styles. When implementing a sub style for a custom component, you 
 * should use CanvasElement's _applySubStylesToElement() function to apply the sub style [StyleDefinition]s 
 * to your sub component, such as within the _doStylesUpdated() function when the sub style has changed.
 * 
 * Subclasses can add new styles and override the style types or defaults of their base
 * classes by creating their own _StyleTypes and StyleDefault objects.
 * 
 * Example:
 * 
 * StylableBaseSubclass._StyleTypes = Object.create(null);
 * StylableBaseSubclass._StyleTypes.Visible = 				StyleableBase.EStyleType.NORMAL;		
 * StylableBaseSubclass._StyleTypes.BorderType = 			StyleableBase.EStyleType.NORMAL;		
 * StylableBaseSubclass._StyleTypes.SkinStyle = 			StyleableBase.EStyleType.NORMAL;		
 * StylableBaseSubclass._StyleTypes.TextStyle =				StyleableBase.EStyleType.INHERITABLE;			
 * StylableBaseSubclass._StyleTypes.TextFont =				StyleableBase.EStyleType.INHERITABLE;			
 * StylableBaseSubclass._StyleTypes.TextSize =				StyleableBase.EStyleType.INHERITABLE;			
 * 
 * StylableBaseSubclass.StyleDefault = new StyleDefinition();
 * StylableBaseSubclass.StyleDefault.setStyle("Visible", 				true);
 * StylableBaseSubclass.StyleDefault.setStyle("BorderType", 			"none");
 * StylableBaseSubclass.StyleDefault.setStyle("SkinStyle", 				null);
 * StylableBaseSubclass.StyleDefault.setStyle("TextStyle", 				"normal");
 * StylableBaseSubclass.StyleDefault.setStyle("TextFont", 				"Arial");
 * StylableBaseSubclass.StyleDefault.setStyle("TextSize", 				12);
 * 
 * 
 * @constructor StyleableBase 
 * Creates new StyleableBase instance.
 */
function StyleableBase()
{
	StyleableBase.base.prototype.constructor.call(this);
	
	//Map of StyleData by styleName, (allocation of StyleData can be expensive considering how often its queried) 
	this._stylesCache = Object.create(null); // {styleData:new StyleData(styleName), cacheInvalid:true};
}
	
//Inherit from StyleDefinition
StyleableBase.prototype = Object.create(StyleDefinition.prototype);
StyleableBase.prototype.constructor = StyleableBase;
StyleableBase.base = StyleDefinition;

//Priority enum
StyleableBase.EStylePriorities = 
	{
		INSTANCE:0,
		CLASS:1
	};

//StyleType enum
StyleableBase.EStyleType = 
	{
		NORMAL:1,
		INHERITABLE:2,
		SUBSTYLE:3
	};


//////////////Public//////////////////////

//@override	
StyleableBase.prototype.getStyle = 
	function (styleName)
	{
		return this.getStyleData(styleName).value;
	};	

//@override
StyleableBase.prototype.setStyle = 
	function (styleName, value)
	{
		var oldStyle = undefined;
		if (styleName in this._styleMap)
			oldStyle = this._styleMap[styleName];
		
		//No change
		if (oldStyle === value)
			return;
		
		if (value === undefined)
			delete this._styleMap[styleName];
		else
			this._styleMap[styleName] = value;
		
		//Get the cache for this style.
		var styleCache = this._stylesCache[styleName];
		
		//Create cache if doesnt exist, flag cache as invalid to cause a full lookup by getStyleData()
		if (styleCache == null)
		{
			styleCache = {styleData:new StyleData(styleName), cacheInvalid:true};
			this._stylesCache[styleName] = styleCache;
		}
		else
			styleCache.cacheInvalid = true;
		
		if (this.hasEventListener("stylechanged", null) == true)
			this.dispatchEvent(new StyleChangedEvent(styleName));
	};
	
/**
 * @function getStyleData
 * 
 * Gets the style data for the supplied style name, this includes
 * additional info than getStyle() such as the style priority. You should
 * not modify the returned StyleData.
 * 
 * @param styleName String
 * String representing style to return the associated StyleData.
 * 
 * @returns StyleData
 * Returns the associated StyleData
 */	
StyleableBase.prototype.getStyleData = 
	function (styleName)
	{
		//Create cache if does not exist.
		var styleCache = this._stylesCache[styleName];
		if (styleCache == null)
		{
			styleCache = {styleData:new StyleData(styleName), cacheInvalid:true};
			
			//We always only have a depth of 1, so we push this now, and just update it as necessary.
			styleCache.styleData.priority.push(StyleableBase.EStylePriorities.INSTANCE); 
			
			this._stylesCache[styleName] = styleCache;
		}
	
		if (styleCache.cacheInvalid == false)
			return styleCache.styleData;
		
		styleCache.cacheInvalid = false;
	
		styleCache.styleData.value = StyleableBase.base.prototype.getStyle.call(this, styleName);
		if (styleCache.styleData.value !== undefined)
		{
			styleCache.styleData.priority[0] = StyleableBase.EStylePriorities.INSTANCE;
			return styleCache.styleData;			
		}
		
		styleCache.styleData.value = this._getClassStyle(styleName);
		styleCache.styleData.priority[0] = StyleableBase.EStylePriorities.CLASS;
		return styleCache.styleData;
	};
	
///////////////Internal///////////////////	
	
//@private	
StyleableBase.prototype._getStyleType = 
	function (styleName)
	{
		this._flattenStyleTypes();
		
		if (styleName in this.constructor.__StyleTypesFlatMap)
			return this.constructor.__StyleTypesFlatMap[styleName];
		
		return null;
	};

//@private	
StyleableBase.prototype._flattenStyleTypes = 
	function ()
	{
		if (this.constructor.__StyleTypesFlatMap != null)
			return;
		
		this.constructor.__StyleTypesFlatMap = Object.create(null);
		this.constructor.__StyleTypesFlatArray = [];
		
		var thisClass = this.constructor;
		var thisProto = Object.getPrototypeOf(this);
		var styleName = null;
		
		while (true)
		{
			if ("_StyleTypes" in thisClass)
			{
				for (styleName in thisClass._StyleTypes)
				{
					if (styleName in this.constructor.__StyleTypesFlatMap)
						continue;
					
					this.constructor.__StyleTypesFlatMap[styleName] = thisClass._StyleTypes[styleName];
					this.constructor.__StyleTypesFlatArray.push({styleName:styleName, styleType:thisClass._StyleTypes[styleName]});
				}
			}
			
			thisProto = Object.getPrototypeOf(thisProto);
			if (thisProto == null || thisProto.hasOwnProperty("constructor") == false)
				break;
			
			thisClass = thisProto.constructor;
		}
	};
	
/**
 * @function _getClassStyle
 * 
 * Gets the default style value for the supplied style name specified on this
 * classes StyleDefault map. Subclasses override base class values.
 *  
 * @param styleName String
 * String representing the default style to return.
 * 
 * @returns Any
 * Returns the associated default style value or undefined if none specified.
 */	
StyleableBase.prototype._getClassStyle = 
	function (styleName)
	{
		this._flattenClassStyles();
		
		if (styleName in this.constructor.__StyleDefaultsFlatMap)
			return this.constructor.__StyleDefaultsFlatMap[styleName][this.constructor.__StyleDefaultsFlatMap[styleName].length - 1];
		
		return undefined;
	};	
	
/**
 * @function _getInstanceStyle
 * 
 * Gets the assigned style value for the supplied style name specified.
 *  
 * @param styleName String
 * String representing the style to return.
 * 
 * @returns Any
 * Returns the associated style value or undefined if none specified.
 */		
StyleableBase.prototype._getInstanceStyle = 
	function (styleName)
	{
		if (styleName in this._styleMap)
			return this._styleMap[styleName];
		
		return undefined;
	};
	
/**
 * @function _applySubStylesToElement
 * 
 * Convienence function for setting sub styles of sub components.
 * Applies appropriate sub styling from this Styleable to the 
 * supplied elements definition and default definition style lists.
 *  
 * @param styleName String
 * String representing the sub style to apply.
 * 
 * @param elementToApply CanvasElement
 * The sub component element to apply sub styles.
 */		
StyleableBase.prototype._applySubStylesToElement = 
	function (styleName, elementToApply)
	{
		//Set default style definitions (class styles)
		elementToApply._setStyleDefinitions(this._getClassStyleList(styleName), true);
		
		//Get instance style or array of styles
		var instanceStyle = this._getInstanceStyle(styleName);

		var instanceDefinitions = [];
		
		if (instanceStyle !== undefined)
			instanceDefinitions.push(instanceStyle);
		
		//Set style definitions
		elementToApply._setStyleDefinitions(instanceDefinitions, false);
	};	
	
/**
 * @function _getClassStyleList
 * 
 * Gets the default style values for the supplied style name specified on this
 * classes and base classes StyleDefault maps. This is used for sub styles as all
 * stub styles in the inheritance chain are applied to sub components.
 *  
 * @param styleName String
 * String representing the default style list to return.
 * 
 * @returns Array
 * Returns an array of all default styles on this class, and base classes
 * for the supplied styleName.
 */	
StyleableBase.prototype._getClassStyleList = 
	function (styleName)
	{
		this._flattenClassStyles();
	
		var i;
		var styleList = [];
		var styleFlatArray;
		
		if (styleName in this.constructor.__StyleDefaultsFlatMap)
		{
			styleFlatArray = this.constructor.__StyleDefaultsFlatMap[styleName];
			
			for (i = 0; i < styleFlatArray.length; i++)
				styleList.push(styleFlatArray[i]);
		}
	
		return styleList;
	};
	
//@private	
StyleableBase.prototype._flattenClassStyles = 
	function ()
	{
		if (this.constructor.__StyleDefaultsFlatMap != null)
			return;
		
		this.constructor.__StyleDefaultsFlatMap = Object.create(null);
		
		var thisClass = this.constructor;
		var thisProto = Object.getPrototypeOf(this);
		var styleName = null;
		
		while (true)
		{
			if ("StyleDefault" in thisClass)
			{
				for (styleName in thisClass.StyleDefault._styleMap)
				{
					if (this.constructor.__StyleDefaultsFlatMap[styleName] == null)
						this.constructor.__StyleDefaultsFlatMap[styleName] = [];
					
					this.constructor.__StyleDefaultsFlatMap[styleName].splice(0, 0, thisClass.StyleDefault._styleMap[styleName]);
				}
			}
			
			thisProto = Object.getPrototypeOf(thisProto);
			if (thisProto == null || thisProto.hasOwnProperty("constructor") == false)
				break;
			
			thisClass = thisProto.constructor;
		}
	};
	


/**
 * @depends StyleableBase.js
 */

////////////////////////////////////////////////////////
////////////////////ShapeBase///////////////////////////

/**
 * @class ShapeBase
 * @inherits StyleableBase
 * 
 * Abstract base class for drawing vector shape paths. 
 * This is used by CanvasElements when drawing their background shape
 * and can be assigned to CanvasElement's "BackgroundShape" style.
 * When sub-classing, add any necessary styles and implement the drawShape() function.
 * 
 * @constructor ShapeBase 
 * Creates new ShapeBase instance.
 */
function ShapeBase()
{
	ShapeBase.base.prototype.constructor.call(this);
}

//Inherit from StyleableBase
ShapeBase.prototype = Object.create(StyleableBase.prototype);
ShapeBase.prototype.constructor = ShapeBase;
ShapeBase.base = StyleableBase;

////////////Public//////////////////////

/**
 * @function drawShape
 * Used to draw a sub-path shape path to the supplied Canvas2DContext using the supplied metrics.
 * Override this to draw custom shapes. Do *not* call beginPath() as that will destroy previous 
 * sub-paths and *do not* do any filling or other context calls. Only draw and closePath() the sub-path.
 * 
 * @param ctx Canvas2DContext
 * The Canvas2DContext to draw the sub-path on.
 * 
 * @param metrics DrawMetrics
 * DrawMetrics object to use as the bounding box for the sub-path.
 */
ShapeBase.prototype.drawShape = 
	function (ctx, metrics)
	{
		//Stub for override.
	};
	
	


/**
 * @depends ShapeBase.js
 */

////////////////////////////////////////////////////////
/////////////RoundedRectangleShape//////////////////////	

/**
 * @class RoundedRectangleShape
 * @inherits ShapeBase
 * 
 * Draws rectangles and rounded rectangles.
 * 
 * @constructor RoundedRectangleShape 
 * Creates new RoundedRectangleShape instance.
 */
function RoundedRectangleShape()
{
	RoundedRectangleShape.base.prototype.constructor.call(this);
}

//Inherit from ShapeBase
RoundedRectangleShape.prototype = Object.create(ShapeBase.prototype);
RoundedRectangleShape.prototype.constructor = RoundedRectangleShape;
RoundedRectangleShape.base = ShapeBase;

/////////////Style Types///////////////////////////////

RoundedRectangleShape._StyleTypes = Object.create(null);

/**
 * @style CornerRadius Number
 * 
 * Radius of the rectangles corners in pixels. 
 * CornerRadius effects all corners of the rectangle.
 * This will override CornerRadiusPercent style.
 */
RoundedRectangleShape._StyleTypes.CornerRadius = 					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusPercent Number
 * 
 * Radius of the rectangles corners as a percent size of the minimum dimension (width/height). 
 * CornerRadiusPercent effects all corners of the rectangle.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusPercent = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusTopLeft Number
 * 
 * Radius size of the rectangles top left corner in pixels.  
 * This will override the CornerRadius, CornerRadiusPercent, & CornerRadiusTopLeftPercent styles.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusTopLeft = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusTopLeftPercent Number
 * 
 * Radius of the rectangles top left corner as a percent size of the minimum dimension (width/height).  
 * This will override the CornerRadius & CornerRadiusPercent styles.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusTopLeft = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusTopRight Number
 * 
 * Radius size of the rectangles top right corner in pixels.  
 * This will override the CornerRadius, CornerRadiusPercent, & CornerRadiusTopRightPercent styles.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusTopRight = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusTopRightPercent Number
 * 
 * Radius of the rectangles top right corner as a percent size of the minimum dimension (width/height).  
 * This will override the CornerRadius & CornerRadiusPercent styles.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusTopRightPercent = 	StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusBottomLeft Number
 * 
 * Radius size of the rectangles bottom left corner in pixels.  
 * This will override the CornerRadius, CornerRadiusPercent, & CornerRadiusBottomLeftPercent styles.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusBottomLeft = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusBottomLeftPercent Number
 * 
 * Radius of the rectangles bottom left corner as a percent size of the minimum dimension (width/height).  
 * This will override the CornerRadius & CornerRadiusPercent styles.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusBottomLeftPercent = 	StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusBottomRight Number
 * 
 * Radius size of the rectangles bottom right corner in pixels.  
 * This will override the CornerRadius, CornerRadiusPercent, & CornerRadiusBottomRightPercent styles.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusBottomRight = 		StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusBottomRightPercent Number
 * 
 * Radius of the rectangles bottom right corner as a percent size of the minimum dimension (width/height).  
 * This will override the CornerRadius & CornerRadiusPercent styles.
 */
RoundedRectangleShape._StyleTypes.CornerRadiusBottomRightPercent = 	StyleableBase.EStyleType.NORMAL;		// number || null


////////////Style Defaults////////////////////////////

RoundedRectangleShape.StyleDefault = new StyleDefinition();

RoundedRectangleShape.StyleDefault.setStyle("CornerRadius", 					null);
RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusPercent", 				null);

RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusTopLeft",				null);
RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusTopLeftPercent",		null);

RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusTopRight",				null);
RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusTopRightPercent",		null);

RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusBottomLeft",			null);
RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusBottomLeftPercent",	null);

RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusBottomRight",			null);
RoundedRectangleShape.StyleDefault.setStyle("CornerRadiusBottomRightPercent",	null);


////////////Public//////////////////////

//@Override
RoundedRectangleShape.prototype.drawShape = 
	function (ctx, metrics)
	{
		var x = metrics.getX();
		var y = metrics.getY();
		
		var width = metrics.getWidth();
		var height = metrics.getHeight();
		var size = Math.min(width, height);
		
		var c = this.getStyle("CornerRadius");
		var cTl = this.getStyle("CornerRadiusTopLeft");
		var cTr = this.getStyle("CornerRadiusTopRight");
		var cBl = this.getStyle("CornerRadiusBottomLeft");
		var cBr = this.getStyle("CornerRadiusBottomRight");
		
		if (c == null)
		{
			var cp = this.getStyle("CornerRadiusPercent");
			if (cp == null)
				c = 0;
			else
				c = size * (cp / 100);
		}
		
		if (cTl == null)
		{
			var cTlp = this.getStyle("CornerRadiusTopLeftPercent");
			if (cTlp == null)
				cTl = c;
			else
				cTl = size * (cTlp / 100);
		}
		
		if (cTr == null)
		{
			var cTrp = this.getStyle("CornerRadiusTopRightPercent");
			if (cTrp == null)
				cTr = c;
			else
				cTr = size * (cTrp / 100);
		}
		
		if (cBl == null)
		{
			var cBlp = this.getStyle("CornerRadiusBottomLeftPercent");
			if (cBlp == null)
				cBl = c;
			else
				cBl = size * (cBlp / 100);
		}
		
		if (cBr == null)
		{
			var cBrp = this.getStyle("CornerRadiusBottomRightPercent");
			if (cBrp == null)
				cBr = c;
			else
				cBr = size * (cBrp / 100);
		}
		
		ctx.moveTo(x, y + cTl);
		
		if (cTl > 0)
			ctx.arcTo(x, y, 
				x + cTl, y, 
				cTl);
		
		ctx.lineTo(x + width - cTr, y);
		
		if (cTr > 0)
			ctx.arcTo(x + width, y, 
				x + width, y + cTr, 
				cTr);
		
		ctx.lineTo(x + width, y + height - cBr);
		
		if (cBr > 0)
			ctx.arcTo(x + width, y + height, 
				x + width - cBr, y + height, 
				cBr);
		
		ctx.lineTo(x + cBl, y + height);
		
		if (cBl > 0)
			ctx.arcTo(x, y + height, 
				x, y + height - cBl, 
				cBl);
		
		ctx.closePath();
	};
	
	


/**
 * @depends ShapeBase.js
 */

////////////////////////////////////////////////////////
/////////////////EllipseShape///////////////////////////

/**
 * @class EllipseShape
 * @inherits ShapeBase
 * 
 * Draws an ellipse that fills the supplied metrics rectangle.
 * 
 * @constructor EllipseShape 
 * Creates new EllipseShape instance.
 */
function EllipseShape()
{
	EllipseShape.base.prototype.constructor.call(this);
}

//Inherit from ShapeBase
EllipseShape.prototype = Object.create(ShapeBase.prototype);
EllipseShape.prototype.constructor = EllipseShape;
EllipseShape.base = ShapeBase;

////////////Public//////////////////////

//@Override
EllipseShape.prototype.drawShape = 
	function (ctx, metrics)
	{
		var w = metrics.getWidth();
		var h = metrics.getHeight();
		
		var spline4Magic = 0.551784;
		var xOffset = (w / 2) * spline4Magic;
		var yOffset = (h / 2) * spline4Magic;
		
		var xStart = metrics.getX();
		var yStart = metrics.getY();
		var xMiddle = xStart + (w / 2);
		var yMiddle = yStart + (h / 2);
		var xEnd = xStart + w;
		var yEnd = yStart + h;
		
		ctx.moveTo(xStart, yMiddle);
		ctx.bezierCurveTo(xStart, yMiddle - yOffset, xMiddle - xOffset, yStart, xMiddle, yStart);
		ctx.bezierCurveTo(xMiddle + xOffset, yStart, xEnd, yMiddle - yOffset, xEnd, yMiddle);
		ctx.bezierCurveTo(xEnd, yMiddle + yOffset, xMiddle + xOffset, yEnd, xMiddle, yEnd);
		ctx.bezierCurveTo(xMiddle - xOffset, yEnd, xStart, yMiddle + yOffset, xStart, yMiddle);
		ctx.closePath();
	};
	
	


/**
 * @depends ShapeBase.js
 */

////////////////////////////////////////////////////////
/////////////////ArrowShape/////////////////////////////	

/**
 * @class ArrowShape
 * @inherits ShapeBase
 * 
 * Draws a variety of arrow-ish shapes such as triangles, rounded pointers,
 * and traditional arrows.
 * 
 * @constructor ArrowShape 
 * Creates new ArrowShape instance.
 */
function ArrowShape()
{
	ArrowShape.base.prototype.constructor.call(this);
}

//Inherit from ShapeBase
ArrowShape.prototype = Object.create(ShapeBase.prototype);
ArrowShape.prototype.constructor = ArrowShape;
ArrowShape.base = ShapeBase;

/////////////Style Types///////////////////////////////

ArrowShape._StyleTypes = Object.create(null);

/**
 * @style Direction String
 * 
 * Determines the direction that the arrow or triangle will point. Acceptable values are "up", "down", "left", and "right".
 * Other styles are named as such when the Arrow is pointed "up". Styles do not change with orientation.
 */
ArrowShape._StyleTypes.Direction = 						StyleableBase.EStyleType.NORMAL;		// "up" || "down" || "left" || "right"

/**
 * @style RectBaseWidth Number
 * 
 * The size in pixels used for the width of the rectangular base of the arrow. Setting this to zero creates a triangle.
 * It is preferrable to use RectBasePercentWidth so that the arrow can scale.
 */
ArrowShape._StyleTypes.RectBaseWidth = 					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style RectBaseHeight Number
 * 
 * The size in pixels used for the height of the rectangular base of the arrow. Setting this to zero creates a triangle.
 * It is preferrable to use RectBasePercentHeight so that the arrow can scale.
 */
ArrowShape._StyleTypes.RectBaseHeight = 				StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style RectBasePercentWidth Number
 * 
 * The percentage of available width to use for the width of the rectangular base of the arrow. 
 * Acceptable values are between 0 and 100. Setting this to zero will create a triangle.
 */
ArrowShape._StyleTypes.RectBasePercentWidth = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style RectBasePercentHeight Number
 * 
 * The percentage of available height to use for the height of the rectangular base of the arrow. 
 * Acceptable values are between 0 and 100. Setting this to zero will create a triangle.
 */
ArrowShape._StyleTypes.RectBasePercentHeight = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadius Number
 * 
 * Radius size in pixels for the rectangular base's corners. 
 * CornerRadius effects all corners of the rectangular base. 
 */
ArrowShape._StyleTypes.CornerRadius = 					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusTopLeft Number
 * 
 * Radius size in pixels for the rectangular base's top left corner. 
 * This will override the CornerRadius style unless it is null.
 */
ArrowShape._StyleTypes.CornerRadiusTopLeft = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusTopRight Number
 * 
 * Radius size in pixels for the rectangular base's top right corner. 
 * This will override the CornerRadius style unless it is null.
 */
ArrowShape._StyleTypes.CornerRadiusTopRight = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusBottomLeft Number
 * 
 * Radius size in pixels for the rectangular base's bottom left corner. 
 * This will override the CornerRadius style unless it is null. Rounding both bottom corners
 * will give the effect of a rounded pointer. 
 */
ArrowShape._StyleTypes.CornerRadiusBottomLeft = 		StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style CornerRadiusBottomRight Number
 * 
 * Radius size in pixels for the rectangular base's bottom right corner. 
 * This will override the CornerRadius style unless it is null. Rounding both bottom corners
 * will give the effect of a rounded pointer. 
 */
ArrowShape._StyleTypes.CornerRadiusBottomRight = 		StyleableBase.EStyleType.NORMAL;		// number || null


////////////Default Styles///////////////////////////

ArrowShape.StyleDefault = new StyleDefinition();

ArrowShape.StyleDefault.setStyle("Direction", 						"up");	// "up" || "down" || "left" || "right"

ArrowShape.StyleDefault.setStyle("RectBaseWidth", 					null); 	// number || null
ArrowShape.StyleDefault.setStyle("RectBaseHeight", 					null); 	// number || null
ArrowShape.StyleDefault.setStyle("RectBasePercentWidth", 			null);	// number || null
ArrowShape.StyleDefault.setStyle("RectBasePercentHeight", 			null); 	// number || null

ArrowShape.StyleDefault.setStyle("CornerRadius", 					0);		// number || null
ArrowShape.StyleDefault.setStyle("CornerRadiusTopLeft",				null);	// number || null
ArrowShape.StyleDefault.setStyle("CornerRadiusTopRight",			null);	// number || null
ArrowShape.StyleDefault.setStyle("CornerRadiusBottomLeft",			null);	// number || null
ArrowShape.StyleDefault.setStyle("CornerRadiusBottomRight",			null);	// number || null


////////////Public//////////////////////

ArrowShape.prototype.drawShape = 
	function (ctx, metrics)
	{
		var direction = this.getStyle("Direction");
		
		if (direction != "up" && direction != "down" && direction != "left" && direction != "right")
			return;
		
		var x = metrics.getX();
		var y = metrics.getY();
		var width = metrics.getWidth();
		var height = metrics.getHeight();
		
		var c = this.getStyle("CornerRadius");
		var cornerTl = this.getStyle("CornerRadiusTopLeft");
		var cornerTr = this.getStyle("CornerRadiusTopRight");
		var cornerBl = this.getStyle("CornerRadiusBottomLeft");
		var cornerBr = this.getStyle("CornerRadiusBottomRight");
		
		if (c == null)
			c = 0;
		if (cornerTl == null)
			cornerTl = c;
		if (cornerTr == null)
			cornerTr = c;
		if (cornerBl == null)
			cornerBl = c;
		if (cornerBr == null)
			cornerBr = c;
		
		var baseWidth = this.getStyle("RectBaseWidth");
		var baseHeight = this.getStyle("RectBaseHeight");
		
		if (baseWidth == null)
		{
			var rectWidthPercent = this.getStyle("RectBasePercentWidth");
			if (rectWidthPercent == null)
				baseWidth = 0;
			else
				baseWidth = Math.round(width * (rectWidthPercent / 100));
		}
		if (baseHeight == null)
		{
			var rectHeightPercent = this.getStyle("RectBasePercentHeight");
			if (rectHeightPercent == null)
				baseHeight = 0;
			else
				baseHeight = Math.round(height * (rectHeightPercent / 100));
		}
		
		if (baseWidth == 0 || baseHeight == 0)
		{
			baseWidth = 0;
			baseHeight = 0;
		}
		
		if (direction == "down")
		{
			ctx.moveTo(x + ((width - baseWidth) / 2), 
				y + cornerTl);
			
			if (cornerTl > 0)
				ctx.arcTo(x + ((width - baseWidth) / 2), y, 
					x + ((width - baseWidth) / 2) + cornerTl, y, 
					cornerTl);
			
			ctx.lineTo(x + ((width - baseWidth) / 2) + baseWidth - cornerTr, 
				y);
			
			if (cornerTr > 0)
				ctx.arcTo(x + ((width - baseWidth) / 2) + baseWidth, y, 
					x + ((width - baseWidth) / 2) + baseWidth, y + cornerTr, 
					cornerTr);
			
			ctx.lineTo(x + ((width - baseWidth) / 2) + baseWidth, y + baseHeight - cornerBr);
			
			if (cornerBr > 0)
				ctx.arcTo(x + ((width - baseWidth) / 2) + baseWidth, y + baseHeight,
					Math.min(x + ((width - baseWidth) / 2) + baseWidth + cornerBr, x + width), y + baseHeight,
					Math.min(cornerBr, (width - baseWidth) / 2));			
			
			ctx.lineTo(x + width, y + baseHeight);
			ctx.lineTo(x + (width / 2), y + height);
			ctx.lineTo(x, y + baseHeight);
			
			ctx.lineTo(Math.max(x + ((width - baseWidth) / 2) - cornerBl, x), y + baseHeight);
			
			if (cornerBl > 0)
				ctx.arcTo(x + ((width - baseWidth) / 2), y + baseHeight,
					x + ((width - baseWidth) / 2), y + baseHeight - cornerBl,
					Math.min(cornerBl, (width - baseWidth) / 2));
		}
		else if (direction == "left")
		{
			ctx.moveTo(x + width - cornerTr, 
					y +  ((height - baseHeight) / 2));
			
			if (cornerTr > 0)
				ctx.arcTo(x + width, y +  ((height - baseHeight) / 2), 
					x + width, y +  ((height - baseHeight) / 2) + cornerTr, 
					cornerTr);
			
			ctx.lineTo(x + width, 
				y + ((height - baseHeight) / 2) + baseHeight - cornerBr);
			
			if (cornerBr > 0)
				ctx.arcTo(x + width, y + ((height - baseHeight) / 2) + baseHeight, 
					x + width - cornerBr, y + ((height - baseHeight) / 2) + baseHeight, 
					cornerBr);
			
			ctx.lineTo(x + width - baseWidth + cornerBl, y + ((height - baseHeight) / 2) + baseHeight);
			
			if (cornerBl > 0)
				ctx.arcTo(x + width - baseWidth, y + ((height - baseHeight) / 2) + baseHeight,
					x + width - baseWidth, Math.min(y + ((height - baseHeight) / 2) + baseHeight + cornerBl, y + height),
					Math.min(cornerBl, (height - baseHeight) / 2));
			
			
			ctx.lineTo(x + width - baseWidth, y + height);
			ctx.lineTo(x, y + (height / 2));
			ctx.lineTo(x + width - baseWidth, y);
			
			ctx.lineTo(x + width - baseWidth, Math.max(y, y + ((height - baseHeight) / 2) - cornerTl));

			if (cornerTl > 0)
				ctx.arcTo(x + width - baseWidth, y + ((height - baseHeight) / 2),
					x + width - baseWidth + cornerTl, y + ((height - baseHeight) / 2),
					Math.min(cornerTl, (height - baseHeight) / 2));
		}
		else if (direction == "up")
		{
			ctx.moveTo(x + ((width - baseWidth) / 2) + baseWidth, 
				y + height - cornerBr);
			
			if (cornerBr > 0)
				ctx.arcTo(x + ((width - baseWidth) / 2) + baseWidth, y + height, 
					x + ((width - baseWidth) / 2) + baseWidth - cornerBr, y + height, 
					cornerBr);
			
			ctx.lineTo(x + ((width - baseWidth) / 2) + cornerBl, 
				y + height);
			
			if (cornerBl > 0)
				ctx.arcTo(x + ((width - baseWidth) / 2), y + height, 
					x + ((width - baseWidth) / 2), y + height - cornerBl, 
					cornerBl);
			
			ctx.lineTo(x + ((width - baseWidth) / 2), y + height - baseHeight + cornerTl);
			
			if (cornerTl > 0)
				ctx.arcTo(x + ((width - baseWidth) / 2), y + height - baseHeight,
					Math.max(x + ((width - baseWidth) / 2) - cornerTl, x), y + height - baseHeight,
					Math.min(cornerTl, (width - baseWidth) / 2));			
			
			
			ctx.lineTo(x, y + height - baseHeight);
			ctx.lineTo(x + (width / 2), y);
			ctx.lineTo(x + width, y + height - baseHeight);
			
			
			ctx.lineTo(Math.min(x + ((width - baseWidth) / 2) + baseWidth + cornerTr, x + width), y + height - baseHeight);
			
			if (cornerTr > 0)
				ctx.arcTo(x + ((width - baseWidth) / 2) + baseWidth, y + height - baseHeight,
					x + ((width - baseWidth) / 2) + baseWidth, y + height - baseHeight + cornerTr,
					Math.min(cornerTr, (width - baseWidth) / 2));
		}
		else if (direction == "right")
		{
			ctx.moveTo(x + cornerBl, 
				y +  ((height - baseHeight) / 2) + baseHeight);
			
			if (cornerBl > 0)
				ctx.arcTo(x, y +  ((height - baseHeight) / 2) + baseHeight, 
					x, y +  ((height - baseHeight) / 2) + baseHeight - cornerBl, 
					cornerBl);
			
			ctx.lineTo(x, 
				y + ((height - baseHeight) / 2) + cornerTl);
			
			if (cornerTl > 0)
				ctx.arcTo(x, y + ((height - baseHeight) / 2), 
					x + cornerTl, y + ((height - baseHeight) / 2), 
					cornerTl);
			
			ctx.lineTo(x + baseWidth - cornerTr, y + ((height - baseHeight) / 2));
			
			if (cornerTr > 0)
				ctx.arcTo(x + baseWidth, y + ((height - baseHeight) / 2),
					x + baseWidth, Math.max(y + ((height - baseHeight) / 2) - cornerTr, y),
					Math.min(cornerTr, (height - baseHeight) / 2));
			
			
			ctx.lineTo(x + baseWidth, y);
			ctx.lineTo(x + width, y + (height / 2));
			ctx.lineTo(x + baseWidth, y + height);
			
			ctx.lineTo(x + baseWidth, Math.min(y + height, y + ((height - baseHeight) / 2) + baseHeight + cornerBr));

			if (cornerBr > 0)
				ctx.arcTo(x + baseWidth, y + ((height - baseHeight) / 2) + baseHeight,
					x + baseWidth - cornerBr, y + ((height - baseHeight) / 2) + baseHeight,
					Math.min(cornerBr, (height - baseHeight) / 2));
		}
		
		ctx.closePath();
	};	
	
	


/**
 * @depends EventDispatcher.js
 */

///////////////////////////////////////////////////////////////////
///////////////////////ListCollection//////////////////////////////	

/**
 * @class ListCollection
 * @inherits EventDispatcher
 * 
 * ListCollection is a wrapper for an Array that dispatches "collectionchanged"
 * events when the collection is modified. This is used by data driven containers
 * like the DataList or DataGrid to track and respond to changes on their associated data.
 * 
 * 
 * @constructor ListCollection 
 * Creates new ListCollection instance.
 * 
 * @param sourceArray Array
 * Backing array to be used as the collection's source. This is used
 * if you have an existing array you want to give to the ListCollection
 * to manage. Setting this to null will cause the ListCollection to create 
 * its own internal array.
 */
function ListCollection(sourceArray)
{
	ListCollection.base.prototype.constructor.call(this);
	
	if (sourceArray != null)
		this._backingArray = sourceArray;
	else
		this._backingArray = [];
	
	this._collectionSort = null;	
	
	
}

//Inherit from EventDispatcher
ListCollection.prototype = Object.create(EventDispatcher.prototype);
ListCollection.prototype.constructor = ListCollection;
ListCollection.base = EventDispatcher;

/**
 * @event collectionchanged CollectionChangedEvent
 * 
 * Dispatched when the collection is modified. CollectionChangedEvents can be of kinds "add", "remove", "update", "reset",
 * and include the index which has been changed.
 */


////////////Public///////////////////////////////

/**
 * @function getLength
 * Gets the number of elements in the collection.
 * 
 * @returns int
 * The number of elements in the collection.
 */
ListCollection.prototype.getLength = 
	function ()
	{
		return this._backingArray.length;
	};

/**
 * @function setSourceArray
 * Sets the source array to be used for the collection and dispatches a "collectionchanged" "reset" event.
 * 
 * @param sourceArray Array
 * The source array to be used for the collection.
 */	
ListCollection.prototype.setSourceArray = 
	function (sourceArray)
	{
		this._backingArray = sourceArray;
		this.dispatchEvent(new CollectionChangedEvent("reset", -1));
	};
	
/**
 * @function getSourceArray
 * Gets the source array currently used for the collection.
 * 
 * @returns Array
 * The source array currently used for the collection.
 */		
ListCollection.prototype.getSourceArray = 
	function ()
	{
		return this._backingArray;
	};
	
/**
 * @function setCollectionSort
 * Sets the CollectionSort to be used when calling sort().
 * 
 * @param collectionSort CollectionSort
 * The CollectionSort to be used when calling sort(). (or null)
 */	
ListCollection.prototype.setCollectionSort = 
	function (collectionSort)
	{
		if (this._collectionSort == collectionSort)
			return; 
			
		if (!(collectionSort instanceof CollectionSort))
			return;
			
		this._collectionSort = collectionSort;
	};
	
/**
 * @function getCollectionSort
 * Gets the CollectionSort used when calling sort().
 * 
 * @returns CollectionSort
 * The CollectionSort to be used when calling sort().
 */	
ListCollection.prototype.getCollectionSort = 
	function ()
	{
		return this._collectionSort;
	};
	
/**
 * @function sort
 * Sorts the collection and dispatches a "collectionchanged" "reset" event.
 */	
ListCollection.prototype.sort = 
	function ()
	{
		if (this._backingArray == null)
			return;
	
		if (this._collectionSort == null)
			this._backingArray.sort();
		else
			this._collectionSort.sort(this._backingArray);
		
		this.dispatchEvent(new CollectionChangedEvent("reset", -1));
	};
	
/**
 * @function getItemIndex
 * Gets the collection index of the item.
 * 
 * @param item Object
 * The item to which to return the collection index.
 * 
 * @returns int
 * The collection index or -1 if the item is not in the collection.
 */	
ListCollection.prototype.getItemIndex = 
	function (item)
	{
		if (item == null)
			return -1;
		
		return this._backingArray.indexOf(item);
	};
	
/**
 * @function getItemAt
 * Gets an item in the collection at the supplied index.
 * 
 * @param index int
 * The index to which to return the collection item.
 * 
 * @returns Object
 * The collection item or null if the index is out of range.
 */		
ListCollection.prototype.getItemAt = 
	function (index)
	{
		if (index < 0 || index >= this._backingArray.length)
			return null;
	
		return this._backingArray[index];
	};
	
/**
 * @function addItem
 * Adds an item to the end of the collection and dispatches a "collectionchanged" "add" event.
 * 
 * @param item Object
 * The item to add to the collection
 * 
 * @returns Object
 * The item just added to the collection.
 */	
ListCollection.prototype.addItem = 
	function (item)
	{
		return this.addItemAt(item, this._backingArray.length);
	};
	
/**
 * @function addItemAt
 * Adds an item to the collection at the supplied index and dispatches a "collectionchanged" "add" event.
 * 
 * @param item Object
 * The item to add to the collection
 * 
 * @param index int
 * The index to insert the item.
 * 
 * @returns Object
 * The item just added to the collection, or null if the index was out of range.
 */		
ListCollection.prototype.addItemAt = 
	function (item, index)
	{
		if (index < 0 || index > this._backingArray.length || item == null)
			return null;
		
		this._backingArray.splice(index, 0, item);
		
		this.dispatchEvent(new CollectionChangedEvent("add", index));
		
		return item;
	};
	
/**
 * @function removeItem
 * Removes an item from the collection and dispatches a "collectionchanged" "remove" event.
 * 
 * @param item Object
 * The item to remove from the collection.
 * 
 * @returns Object
 * The item just removed from the collection, or null if the item was not in the collection.
 */			
ListCollection.prototype.removeItem = 
	function (item)
	{
		return this.removeItemAt(this._backingArray.indexOf(item));
	};
	
/**
 * @function removeItemAt
 * Removes an item from the collection at the supplied index and dispatches a "collectionchanged" "remove" event.
 * 
 * @param index int
 * The index to remove the item.
 * 
 * @returns Object
 * The item just removed to the collection, or null if the index was out of range.
 */		
ListCollection.prototype.removeItemAt = 
	function (index)
	{
		if (index < 0 || index >= this._backingArray.length)
			return null;
		
		var removed = this._backingArray.splice(index, 1)[0]; //Returns array of removed items.
		
		this.dispatchEvent(new CollectionChangedEvent("remove", index));
		
		return removed;
	};

/**
 * @function replaceItemAt
 * Replaces an item in the collection at the supplied index and dispatches a "collectionchanged" "update" event.
 * 
 * @param item Object
 * The new item to which replace the existing item.
 * 
 * @param index int
 * The index to replace.
 * 
 * @returns Object
 * The item just replaced, or null if the index was out of range.
 */
ListCollection.prototype.replaceItemAt = 
	function (item, index)
	{
		if (index < 0 || index >= this._backingArray.length)
			return null;
		
		var oldItem = this._backingArray[index];
		this._backingArray[index] = item;
		
		this.indexUpdated(index);
		
		return oldItem;
	};
	
/**
 * @function clear
 * Clears the collection and dispatches a "collectionchanged" "reset" event.
 */	
ListCollection.prototype.clear = 
	function ()
	{
		this._backingArray.length = 0;
		
		this.dispatchEvent(new CollectionChangedEvent("reset", -1));
	};
	
/**
 * @function indexUpdated
 * Dispatches a "collectionchanged" "update" event. When a data objects internal
 * data is changed, call this to update the Container such as a DataList or DataGrid.
 * 
 * @param index int
 * The index to dispatch the "update" event.
 */	
ListCollection.prototype.indexUpdated = 
	function (index)
	{
		this.dispatchEvent(new CollectionChangedEvent("update", index));
	};


/**
 * @depends StyleableBase.js
 */

////////////////////////////////////////////////////////
////////////////////FillBase///////////////////////////

/**
 * @class FillBase
 * @inherits StyleableBase
 * 
 * Abstract base class for filling element's background shape.
 * This can be assigned to CanvasElement's "BackgroundFill" style.
 * When sub-classing, add any necessary styles and implement the drawFill() function.
 * 
 * @constructor FillBase 
 * Creates new FillBase instance.
 */
function FillBase()
{
	FillBase.base.prototype.constructor.call(this);
}

//Inherit from StyleableBase
FillBase.prototype = Object.create(StyleableBase.prototype);
FillBase.prototype.constructor = FillBase;
FillBase.base = StyleableBase;

////////////Public//////////////////////

/**
 * @function drawFill
 * Abstract stub used to fill an elements background.
 * Override this, setup the Canvas2DContext's fill style via ctx.fillStyle and call ctx.fill().
 * The background shape path will have already been drawn by the elements ShapeBase class.
 * 
 * @param ctx Canvas2DContext
 * The Canvas2DContext to draw the fill on.
 * 
 * @param metrics DrawMetrics
 * DrawMetrics object to use as the bounding box for the fill.
 * 
 */
FillBase.prototype.drawFill = 
	function (ctx, metrics)
	{
		//Stub for override.
	};
	
	


/**
 * @depends FillBase.js
 */

////////////////////////////////////////////////////////
/////////////////SolidFill//////////////////////////////	

/**
 * @class SolidFill
 * @inherits FillBase
 * 
 * Fills an element a solid color. This class is automatically used when
 * an element's BackgroundFill style is set to a color string like "#FF0000".
 * 
 * @constructor SolidFill 
 * Creates new SolidFill instance.
 */
function SolidFill()
{
	SolidFill.base.prototype.constructor.call(this);
}

//Inherit from FillBase
SolidFill.prototype = Object.create(FillBase.prototype);
SolidFill.prototype.constructor = SolidFill;
SolidFill.base = FillBase;


/////////////Style Types///////////////////////////////

SolidFill._StyleTypes = Object.create(null);


/**
 * @style FillColor color
 * 
 * Color which the element should be filled.
 */
SolidFill._StyleTypes.FillColor = 							StyleableBase.EStyleType.NORMAL;		// "#FF0000"


////////////Default Styles///////////////////////////

SolidFill.StyleDefault = new StyleDefinition();

SolidFill.StyleDefault.setStyle("FillColor", 				"#FF0000");	// "#FF0000"


////////////Public//////////////////////

//@override
SolidFill.prototype.drawFill = 
	function (ctx, metrics)
	{
		ctx.fillStyle = this.getStyle("FillColor");
		ctx.fill();
	};	
	



/**
 * @depends FillBase.js
 */

////////////////////////////////////////////////////////
/////////////////LinearGradientFill/////////////////////	

/**
 * @class LinearGradientFill
 * @inherits FillBase
 * 
 * Fills a linear gradient across the element with supplied angle and color stops.
 * 
 * @constructor LinearGradientFill 
 * Creates new LinearGradientFill instance.
 */
function LinearGradientFill()
{
	LinearGradientFill.base.prototype.constructor.call(this);
}

//Inherit from ShapeBase
LinearGradientFill.prototype = Object.create(FillBase.prototype);
LinearGradientFill.prototype.constructor = LinearGradientFill;
LinearGradientFill.base = FillBase;


/////////////Style Types///////////////////////////////

LinearGradientFill._StyleTypes = Object.create(null);


/**
 * @style GradientDegrees Number
 * 
 * Angle 0-360 which the gradient should be drawn across the element.
 */
LinearGradientFill._StyleTypes.GradientDegrees = 						StyleableBase.EStyleType.NORMAL;		// 45

/**
 * @style GradientColorStops Array
 * 
 * Array of color stops to apply to the gradient.
 * Format like [[position, "color"], [position, "color"], ...]
 * Position is a number between 0 and 1 representing where on the gradient line
 * to place the color stop.
 */
LinearGradientFill._StyleTypes.GradientColorStops = 					StyleableBase.EStyleType.NORMAL;		// [[0, "#FFFFFF"], [1, "#000000]]

/**
 * @style GradientCoverage String
 * 
 * Determines the size of the gradient line based on the GradientDegrees.
 * This style has no effect on gradients at 90 degree intervals.
 * Available values are "inner" and "outer".
 * Inner gradient will draw a line across the center of the element at the specified degrees.
 * Outer gradient extends the line beyond the element's bounds so that the gradient is 
 * perpendicular to the outer most corners of the element. 
 */
LinearGradientFill._StyleTypes.GradientCoverage = 						StyleableBase.EStyleType.NORMAL;		// "inner" || "outer"


////////////Default Styles///////////////////////////

LinearGradientFill.StyleDefault = new StyleDefinition();

LinearGradientFill.StyleDefault.setStyle("GradientDegrees", 			0);	
LinearGradientFill.StyleDefault.setStyle("GradientColorStops", 			[[0, "#FF0000"], [1, "#000000"]]);
LinearGradientFill.StyleDefault.setStyle("GradientCoverage", 			"inner");


////////////Public//////////////////////

//@override
LinearGradientFill.prototype.drawFill = 
	function (ctx, metrics)
	{
		var degrees = CanvasElement.normalizeDegrees(this.getStyle("GradientDegrees"));
		var colorStops = this.getStyle("GradientColorStops");
		var coverage = this.getStyle("GradientCoverage");
		
		var pointsStart = this._calculateInnerOuterPoints(CanvasElement.normalizeDegrees(degrees), metrics);
		var pointsStop = this._calculateInnerOuterPoints(CanvasElement.normalizeDegrees(degrees + 180), metrics);
		
		var pointsIndex = 0;
		if (coverage == "outer")
			pointsIndex = 1;
		
		var pointStart = pointsStart[pointsIndex];
		var pointStop = pointsStop[pointsIndex];
		
		var gradient = ctx.createLinearGradient(pointStart.x, pointStart.y, pointStop.x, pointStop.y);
		
		if (Array.isArray(colorStops) == true)
		{
			for (var i = 0; i < colorStops.length; i++)
			{
				if (colorStops[i].length > 1)
				{
					try
					{
						gradient.addColorStop(colorStops[i][0], colorStops[i][1]);
					}
					catch (ex) 
					{ 
						//Swallow 
					}
				}
			}
		}
		
		ctx.fillStyle = gradient;
		ctx.fill();
	};	
	
	
///////////Internal////////////////////
	
//@private	
LinearGradientFill.prototype._calculateInnerOuterPoints =
	function (degrees, metrics)
	{
		degrees = CanvasElement.normalizeDegrees(degrees);
	
		var x = metrics.getX();
		var y = metrics.getY();
		var w = metrics.getWidth();
		var h = metrics.getHeight();

		var radians = CanvasElement.degreesToRadians(degrees);
		
		var xRadius = (w / 2);
		var yRadius = (h / 2);
		
		var x1 = CanvasElement.cot(radians) * yRadius;
		var y1 = Math.tan(radians) * xRadius;
		
		if (Math.abs(x1) > xRadius)
		{
			if (x1 < 0)
				x1 = xRadius * -1;
			else
				x1 = xRadius;
		}
		if (Math.abs(y1) > yRadius)
		{
			if (y1 < 0)
				y1 = yRadius * -1;
			else
				y1 = yRadius;
		}
		
		if (degrees > 90 && degrees <= 180)
		{
			y1 = y1 * -1;
		}
		else if (degrees > 180 && degrees <= 270)
		{
			x1 = x1 * -1;
			y1 = y1 * -1;
		}
		else if (degrees > 270)
		{
			x1 = x1 * -1;
		}	
		
		var edgeRadius = Math.sqrt((x1 * x1) + (y1 * y1));
		
		var finalRadius;
		if (Math.abs(y1) < yRadius)
			finalRadius = edgeRadius + Math.abs(((yRadius - Math.abs(y1)) * Math.sin(radians)));
		else
			finalRadius = edgeRadius + Math.abs(((xRadius - Math.abs(x1)) * Math.cos(radians)));
		
		var x2 = finalRadius * Math.cos(radians);
		var y2 = finalRadius * Math.sin(radians);
		
		return [{x:x1 + xRadius, y:y1 + yRadius}, 
				{x:x2 + xRadius, y:y2 + yRadius}];
	};
	
	


/////////////////////////////////////////////////////////////////////
/////////////////////DrawMetrics/////////////////////////////////////	

/**
 * @class DrawMetrics
 * 
 * Simple data structure to represent bounds. (X, Y, Width, Height). 
 * 
 * 
 * @constructor DrawMetrics 
 * Creates new DrawMetrics instance.
 */

//Supporting class used to indicate element bounds.
function DrawMetrics()
{
	this._x = 0;
	this._y = 0;
	this._width = 0;
	this._height = 0;
}	

//DrawMetrics is base object, no inheritance.
DrawMetrics.prototype.constructor = DrawMetrics;

/**
 * @function equals
 * 
 * Checks if two instances of DrawMetrics contain the same values.
 * 
 * @param drawMetrics DrawMetrics
 * DrawMetrics instance to compare.
 * 
 * @returns bool
 * True when both instances contain the same values.
 */
DrawMetrics.prototype.equals = 
	function(drawMetrics)
	{
		if (this._x == drawMetrics._x && 
			this._y == drawMetrics._y &&
			this._width == drawMetrics._width && 
			this._height == drawMetrics._height)
		{
			return true;
		}
		
		return false;
	};

/**
 * @function clone
 * Duplicates an instance of DrawMetrics. 
 * 
 * @returns DrawMetrics
 * A new DrawMetrics instance identical to the cloned instance.
 */		
DrawMetrics.prototype.clone = 
	function ()
	{
		var clonedMetrics = new DrawMetrics();
		
		clonedMetrics._x = this._x;
		clonedMetrics._y = this._y;
		clonedMetrics._width = this._width;
		clonedMetrics._height = this._height;
		
		return clonedMetrics;
	};
	
//@private (for now)	
DrawMetrics.prototype.copyFrom = 
	function (copyFromMetrics)
	{
		this._x = copyFromMetrics._x;
		this._y = copyFromMetrics._y;
		this._width = copyFromMetrics._width;
		this._height = copyFromMetrics._height;
	};
	
//@private (for now)
DrawMetrics.prototype.mergeExpand = 
	function (mergeWithDrawMetrics)
	{
		if (mergeWithDrawMetrics._x < this._x)
		{
			this._width += (this._x - mergeWithDrawMetrics._x);
			this._x = mergeWithDrawMetrics._x;
		}
		if (mergeWithDrawMetrics._y < this._y)
		{
			this._height += (this._y - mergeWithDrawMetrics._y);
			this._y = mergeWithDrawMetrics._y;
		}
		if (mergeWithDrawMetrics._x + mergeWithDrawMetrics._width > this._x + this._width)
			this._width += ((mergeWithDrawMetrics._x + mergeWithDrawMetrics._width) - (this._x + this._width));
		if (mergeWithDrawMetrics._y + mergeWithDrawMetrics._height > this._y + this._height)
			this._height += ((mergeWithDrawMetrics._y + mergeWithDrawMetrics._height) - (this._y + this._height));
	};
	
//@private (for now)	
DrawMetrics.prototype.mergeReduce = 
	function (mergeWithDrawMetrics)
	{
		if (this._x < mergeWithDrawMetrics._x)
		{
			this._width -= (mergeWithDrawMetrics._x - this._x);
			this._x = mergeWithDrawMetrics._x;
		}
		if (this._y < mergeWithDrawMetrics._y)
		{
			this._height -= (mergeWithDrawMetrics._y - this._y);
			this._y = mergeWithDrawMetrics._y;
		}
		if (this._x + this._width > mergeWithDrawMetrics._x + mergeWithDrawMetrics._width)
			this._width -= ((this._x + this._width) - (mergeWithDrawMetrics._x + mergeWithDrawMetrics._width));
		if (this._y + this._height > mergeWithDrawMetrics._y + mergeWithDrawMetrics._height)
			this._height -= ((this._y + this._height) - (mergeWithDrawMetrics._y + mergeWithDrawMetrics._height));	
	};
	
DrawMetrics.prototype.roundToPrecision = 
	function (precision)
	{
		this._x = CanvasElement.roundToPrecision(this._x, precision);
		this._y = CanvasElement.roundToPrecision(this._y, precision);
		this._width = CanvasElement.roundToPrecision(this._width, precision);
		this._height = CanvasElement.roundToPrecision(this._height, precision);
	};
	
//@private (for now)	
DrawMetrics.prototype.roundUp = 
	function ()
	{
		var x1 = this._x;
		var x2 = this._x + this._width;
		var y1 = this._y;
		var y2 = this._y + this._height;
		
		x1 = Math.floor(x1);
		x2 = Math.ceil(x2);
		y1 = Math.floor(y1);
		y2 = Math.ceil(y2);
		
		this._x = x1;
		this._y = y1;
		this._width = x2 - x1;
		this._height = y2 - y1;
	};
	
/**
 * @function getX
 * 
 * Gets the X value in pixels, this may be fractional. 
 * 
 * @returns Number
 * The X value.
 */
DrawMetrics.prototype.getX = 
	function()
	{
		return this._x;
	};
	
/**
 * @function getY
 * 
 * Gets the Y value in pixels, this may be fractional. 
 * 
 * @returns Number
 * The Y value.
 */	
DrawMetrics.prototype.getY = 
	function()
	{
		return this._y;
	};
	
/**
 * @function getWidth
 * 
 * Gets the Width value in pixels, this may be fractional. 
 * 
 * @returns Number
 * The Width value.
 */		
DrawMetrics.prototype.getWidth = 
	function()
	{
		return this._width;
	};
	
/**
 * @function getHeight
 * 
 * Gets the Height value in pixels, this may be fractional. 
 * 
 * @returns Number
 * The Height value.
 */		
DrawMetrics.prototype.getHeight = 
	function()
	{
		return this._height;
	};
	
	


///////////////////////////////////////////////////////////////////////
///////////////////////DataListData////////////////////////////////////

/**
 * @class DataListData
 * 
 * Internal data storage class passed to CanvasElements when they are used as
 * DataRenderers for a DataListElement or subclass.
 * 
 * 
 * @constructor DataListData 
 * Creates new DataListData instance.
 * 
 * @param parentList DataListElement
 * The parent DataListElement or subclass.
 * 
 * @param itemIndex int
 * The Collection item index.
 */
function DataListData(parentList, itemIndex)
{
	/**
	 * @member _parentList DataListElement
	 * Read Only - The parent DataListElement or subclass. 
	 */
	this._parentList = parentList;
	
	/**
	 * @member _itemIndex int
	 * Read Only - The Collection item index.
	 */
	this._itemIndex = itemIndex;
};	


///////////////////////////////////////////////////////////////////////
///////////////////////DataGridItemData////////////////////////////////

/**
 * @class DataGridItemData
 * 
 * Internal data storage class passed to CanvasElements when they are used as
 * DataRenderers for a DataGrid.
 * 
 * 
 * @constructor DataGridItemData 
 * Creates new DataGridItemData instance.
 * 
 * @param parentGrid DataGridElement
 * The parent DataListElement or subclass.
 * 
 * @param itemIndex int
 * The Collection item index.
 * 
 * @param columnIndex int
 * The column index associated with the DataGrid renderer.
 */
function DataGridItemData(parentGrid, itemIndex, columnIndex)
{
	/**
	 * @member _parentGrid DataGridElement
	 * Read Only - The parent DataGridElement or subclass. 
	 */
	this._parentGrid = parentGrid;
	
	/**
	 * @member _itemIndex int
	 * Read Only - The Collection item index.
	 */
	this._itemIndex = itemIndex;
	
	/**
	 * @member _columnIndex int
	 * Read Only - Column index associated with the DataGrid renderer.
	 */
	this._columnIndex = columnIndex;
};	




/**
 * @depends StyleableBase.js
 */

////////////////////////////////////////////////////////
//////////////CursorDefinition//////////////////////////	

/**
 * @class CursorDefinition
 * @inherits StyleableBase
 * 
 * CursorDefintion stores styles necessary to render/animate custom cursors.
 * This is used for CanvasElement's Cursor style (roll-over cursor) and can
 * also be added directly to CanvasManager. 
 *  
 * 
 * @constructor CursorDefinition 
 * Creates new CursorDefinition instance.
 */
function CursorDefinition()
{
	CursorDefinition.base.prototype.constructor.call(this);
	
	this._cursorElement = null;
	this._addedCount = 0;
}

//Inherit from StyleableBase
CursorDefinition.prototype = Object.create(StyleableBase.prototype);
CursorDefinition.prototype.constructor = CursorDefinition;
CursorDefinition.base = StyleableBase;

/////////////Style Types///////////////////////////////

CursorDefinition._StyleTypes = Object.create(null);

/**
 * @style CursorClass CanvasElement
 * 
 * The CanvasElement constructor or browser string type to use for the cursor.
 */
CursorDefinition._StyleTypes.CursorClass = 						StyleableBase.EStyleType.NORMAL;		// CanvasElement() constructor

/**
 * @style CursorStyle StyleDefinition
 * 
 * The StyleDefinition to apply to the cursor class. (Including Width and Height, unless you've implemented
 * the doMeasure() function into a custom CanvasElement subclass).
 */
CursorDefinition._StyleTypes.CursorStyle = 						StyleableBase.EStyleType.NORMAL;		// StyleDefinition

/**
 * @style CursorOffsetX Number
 * 
 * The X offset from the actual mouse position the cursor should be rendered.
 */
CursorDefinition._StyleTypes.CursorOffsetX = 					StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style CursorOffsetY Number
 * 
 * The Y offset from the actual mouse position the cursor should be rendered.
 */
CursorDefinition._StyleTypes.CursorOffsetY = 					StyleableBase.EStyleType.NORMAL;		// number


///////////Default Styles/////////////////////////////

CursorDefinition.StyleDefault = new StyleDefinition();

CursorDefinition.StyleDefault.setStyle("CursorClass", 							"default"); 	// "browsertype" || CanvasElement() constructor
CursorDefinition.StyleDefault.setStyle("CursorStyle", 							null); 			// StyleDefinition
CursorDefinition.StyleDefault.setStyle("CursorOffsetX", 						0); 			// number
CursorDefinition.StyleDefault.setStyle("CursorOffsetY", 						0); 			// number



/**
 * @depends StyleableBase.js
 */

///////////////////////////////////////////////////////////////////////////	
/////////////////GridContainerRowColumnDefinition//////////////////////////

/**
 * @class GridContainerRowColumnDefinition
 * @inherits StyleableBase
 * 
 * GridContainerRowColumnDefinition defines and stores styles used by 
 * the GridContainerElement to layout rows and columns.
 * 
 * @constructor GridContainerRowColumnDefinition 
 * Creates new GridContainerRowColumnDefinition instance.
 */

function GridContainerRowColumnDefinition()
{
	GridContainerRowColumnDefinition.base.prototype.constructor.call(this);
}

//Inherit from StyleableBase
GridContainerRowColumnDefinition.prototype = Object.create(StyleableBase.prototype);
GridContainerRowColumnDefinition.prototype.constructor = GridContainerRowColumnDefinition;
GridContainerRowColumnDefinition.base = StyleableBase;


/////////////Style Types///////////////////////////////

GridContainerRowColumnDefinition._StyleTypes = Object.create(null);

/**
 * @style Size Number
 * 
 * Size in pixels this row or column should consume. This style overrides all other sizing styles.
 */
GridContainerRowColumnDefinition._StyleTypes.Size = 						StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style PercentSize Number
 * 
 * The percentage of available size the row or column should consume relative to its parent GridContainer. 
 * Note that percentage width is calculated based on the available space left over *after* static sized rows 
 * and columns are considered. Percentages are allowed to add to more than 100 and will consume all of the 
 * available space. When percents add to more than 100 the elements will share the available space
 * per the ratio of percent vs total percent used.
 */
GridContainerRowColumnDefinition._StyleTypes.PercentSize = 					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style MinSize Number
 * 
 * Minimum number of pixels the row or column should consume.
 * This does not affect rows or columns that have the Size style set.
 */
GridContainerRowColumnDefinition._StyleTypes.MinSize = 						StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style MaxSize Number
 * 
 * Maximum number of pixels the row or column should consume.
 * This does not affect rows or columns that have the Size style set.
 */
GridContainerRowColumnDefinition._StyleTypes.MaxSize = 						StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style StretchPriority Number
 * 
 * When rows or columns are forced to stretch due to cells spanning multiple rows or columns,
 * rows or columns with the highest StretchPriority will stretch first, equal priorities will stretch evenly.
 * This does not affect rows or columns that have Size or PercentSize styles set.
 */
GridContainerRowColumnDefinition._StyleTypes.StretchPriority = 				StyleableBase.EStyleType.NORMAL;		// number || null


////////////Default Styles////////////////////////////

GridContainerRowColumnDefinition.StyleDefault = new StyleDefinition();

GridContainerRowColumnDefinition.StyleDefault.setStyle("Size", 						null);
GridContainerRowColumnDefinition.StyleDefault.setStyle("PercentSize", 				null);
GridContainerRowColumnDefinition.StyleDefault.setStyle("MinSize", 					null);
GridContainerRowColumnDefinition.StyleDefault.setStyle("MaxSize", 					null);
GridContainerRowColumnDefinition.StyleDefault.setStyle("StretchPriority", 			0);





///////////////////////////////////////////////////////////////////
///////////////////////CollectionSort//////////////////////////////	

/**
 * @class CollectionSort
 * 
 * CollectionSort is a helper class that stores a comparatorFunction
 * and a isDecending flag used to invert the sort.
 * 
 * 
 * @constructor CollectionSort 
 * Creates new CollectionSort instance.
 * 
 * @param comparatorFunction Function
 * The sort comparator function to use when sorting an array.
 * 
 * @param isDecending boolean
 * When true invert the sort.
 */
function CollectionSort(comparatorFunction, isDecending)
{
	this._comparatorFunction = comparatorFunction;
	this._isDecending = isDecending;
	
	var _self = this;
	
	//Private function to invert the comparator (decending sort). 
	//This gets passed to Array as function pointer so there's no point in using prototype.
	this._collectionSortDecendingComparator = 
		function (objA, objB)
		{
			return _self._comparatorFunction(objB, objA);
		};
}

//No inheritance (base object)
CollectionSort.prototype.constructor = CollectionSort;

/**
 * @function setComparatorFunction
 * Sets the comparator function to be used when sorting. Comparators accept 2 parameters and return -1, 0, or +1 
 * depending on the sort relation between the 2 parameters.
 * 
 * function (objA, objB) { return objA - objB; };
 * 
 *  @param comparatorFunction Function
 *  The function to be used as the comparator.
 */
CollectionSort.prototype.setComparatorFunction = 
	function (comparatorFunction)
	{
		this._comparatorFunction = comparatorFunction;
	};
	
/**
 * @function getComparatorFunction
 * Gets the comparator function used when sorting.
 * 
 * @returns Function
 * The comparator function used when sorting.
 */	
CollectionSort.prototype.getComparatorFunction = 
	function ()
	{
		return this._comparatorFunction;
	};

/**
 * @function sort
 * Sorts an array using the comparator function and isDecending flag.
 * 
 * @param array Array
 * Array to be sorted.
 */	
CollectionSort.prototype.sort = 
	function (array)
	{
		if (this._isDecending == true)
			array.sort(this._collectionSortDecendingComparator);
		else
			array.sort(this._comparatorFunction);
	};
	
/**
 * @function setIsDecending
 * Sets the isDecending flag. True to invert the sort.
 * 
 * @param isDecending bool
 * When true, invert the sort comparator function.
 */	
CollectionSort.prototype.setIsDecending = 
	function (isDecending)
	{
		this._isDecending = isDecending;
	};
	
/**
 * @function getIsDecending
 * Gets the state of the isDecending flag.
 * 
 * @returns boolean
 * The state of the isDecending flag.
 */	
CollectionSort.prototype.getIsDecending = 
	function ()
	{
		return this._isDecending;
	};
	
	


/**
 * @depends StyleableBase.js
 */

///////////////////////////////////////////////////////////////////////////	
///////////////////////CanvasElement///////////////////////////////////////	
	
/**
 * @class CanvasElement
 * @inherits StyleableBase
 * 
 * Base class for all Elements to be rendered to the Canvas by CanvasManager. 
 * CanvasElement supports all basic system functions for render-able objects such 
 * as the display chain hierarchy, user interactivity and events, 
 * style management, vector based rendering, etc. 
 * 
 * CanvasElement is the most basic type that can be added to CanvasManager and can be
 * used to automatically draw shapes or any custom rendering to the canvas.
 * 
 * 
 * @constructor CanvasElement 
 * Creates new CanvasElement instance.
 */

function CanvasElement()
{
	CanvasElement.base.prototype.constructor.call(this);
	
	//Proxy styles from a different element.
	this._styleProxy = null;
	
	//This is *not* class based defaults. Its a default version of _styleDefinition.
	//Used when the framework wants to apply a default definition that override class 
	//based default styles but *not* user applied styles.
	this._styleDefinitionDefaults = []; 
	
	//Assigned style definitions
	this._styleDefinitions = [];
	
	//Storage for the current background shape ShapeBase() per styling. We need to store a reference 
	//because we listen for style changed events and need to be able to remove the listener when
	//this is changed (via styles) or added/removed to display chain.
	this._backgroundShape = null;
	
	//Storage for the current background fill FillBase() per styling. We need to store a reference 
	//because we listen for style changed events and need to be able to remove the listener when
	//this is changed (via styles) or added/removed to display chain.
	this._backgroundFill = null;
	
	this._manager = null; //Canvas Manager reference
	this._displayDepth = 0; //Depth in display chain hierarchy
	
	//Event listeners for capture phase. (Only ElementEvent events support capture)
	this._captureListeners = Object.create(null);
	
	
	this._name = null;	//User defined identifier
	
	/**
	 * @member _x Number
	 * Read only - X position in pixels relative to this elements parent. This is not updated immediately, only
	 * after our parent has finished its layout phase.
	 */
	this._x = 0;
	
	/**
	 * @member _y Number
	 * Read only - Y position in pixels relative to this elements parent. This is not updated immediately, only
	 * after our parent has finished its layout phase will this be valid.
	 */
	this._y = 0;
	
	/**
	 * @member _width Number
	 * Read only - This element's actual width in pixels. This is not updated immediately, only
	 * after our parent has finished its layout phase will this be valid.
	 */
	this._width = 0;
	
	/**
	 * @member _height Number
	 * Read only - This element's actual height in pixels. This is not updated immediately, only
	 * after our parent has finished its layout phase will this be valid.
	 */
	this._height = 0;
	
	/**
	 * @member _measuredWidth Number
	 * Read only - This element's measured width in pixels. This is not updated immediately, only
	 * after this element has finished its measure phase will this be valid.
	 */
	this._measuredWidth = 0;
	
	/**
	 * @member _measuredHeight Number
	 * Read only - This element's measured height in pixels. This is not updated immediately, only
	 * after this element has finished its measure phase will this be valid.
	 */
	this._measuredHeight = 0;
	
	/**
	 * @member _mouseIsOver boolean
	 * Read only - true if the mouse is over this element, otherwise false.
	 */
	this._mouseIsOver = false;
	
	/**
	 * @member _mouseIsDown boolean
	 * Read only - true if the mouse is pressed on this element, otherwise false.
	 */
	this._mouseIsDown = false;
	
	/**
	 * @member _isFocused boolean
	 * Read only - true if this element currently has focus, otherwise false.
	 */
	this._isFocused = false;
	
	/**
	 * @member _rotateDegrees Number
	 * Read only - Degrees this element is rotated. This is not updated immediately, only
	 * after our parent has finished its layout phase will this be valid.
	 */
	this._rotateDegrees = 0;
	
	/**
	 * @member _rotateCenterX Number
	 * Read only - The X position relative to the element's parent this element is rotated around. This is not updated immediately, only
	 * after our parent has finished its layout phase will this be valid.
	 */
	this._rotateCenterX = 0;
	
	/**
	 * @member _rotateCenterY Number
	 * Read only - The Y position relative to the element's parent this element is rotated around. This is not updated immediately, only
	 * after our parent has finished its layout phase will this be valid.
	 */
	this._rotateCenterY = 0;
	
	/**
	 * @member _parent CanvasElement
	 * Read only - This elements parent element.
	 */
	this._parent = null; 	
	
	
	this._children = [];
	
	this._stylesInvalid = true;
	this._stylesInvalidMap = Object.create(null);	//Dirty map of changed styles for _doStylesChanged()
	this._stylesValidateNode = new CmLinkedNode();	//Reference to linked list iterator
	this._stylesValidateNode.data = this;
	
	this._measureInvalid = true;					//Dirty flag for _doMeasure()
	this._measureValidateNode = new CmLinkedNode();	//Reference to linked list iterator
	this._measureValidateNode.data = this;
	
	this._layoutInvalid = true;						//Dirty flag for _doLayout()
	this._layoutValidateNode = new CmLinkedNode(); 	//Reference to linked list iterator
	this._layoutValidateNode.data = this;
	
	this._renderInvalid = true;						//Dirty flag for _doRender()
	this._renderValidateNode = new CmLinkedNode();	//Reference to linked list iterator
	this._renderValidateNode.data = this;
	
	this._redrawRegionInvalid = true;						//Dirty flag for redraw region
	this._redrawRegionValidateNode = new CmLinkedNode();	//Reference to linked list iterator
	this._redrawRegionValidateNode.data = this;
	
	this._transformRegionValidateNode = new CmLinkedNode();
	this._transformRegionValidateNode.data = this;
	
	//Off screen canvas for rendering this element.
	this._graphicsCanvas = null;
	this._graphicsCtx = null;
	this._graphicsClear = true;					//Optimization, sometimes we may *have* a canvas, but its been cleared so no need to render.
	
	//Metrics used for redraw region relative to composite parents (and ourself if we're a composite layer).
	this._compositeMetrics = [];				//Array of {element:element, metrics:DrawMetrics, drawableMetrics:DrawMetrics}
	
	this._forceRegionUpdate = false;			//Flag set by validateRedrawRegion() when update required due to composite effect on composite parent.
	this._renderChanged = true;					//Dirty flag for redraw region set to true when _graphicsCanvas has been modified.
	
	/**
	 * @member _renderVisible boolean
	 * Read only - False if any element in the parent chain is not visible.
	 * This is not updated immediately, only after redraw regions have been updated.
	 * Useful for things like "enterframe" listeners to prevent processing when not necessary.
	 */
	this._renderVisible = false; 				//False if any element in the composite parent chain is not visible.	
	
	/////////Composite Rendering////////////////
	
	//Composite rendering is used for effects like shadow, alpha, and transformations which
	//require aggregating child renderings, then re-rendering with the desired effect.
	//When an element requires composite rendering, it and its children are rendered to _compositeCanvas,
	//then _compositeCanvas is rendered to the parent composite (or root canvas) and appropriate effects are applied.
	//These values are only populated when this element requires composite rendering.
	
	this._compositeRenderInvalid = false;
	this._compositeRenderValidateNode = new CmLinkedNode();
	this._compositeRenderValidateNode.data = this;
	
	this._compositeEffectChanged = true;
	
	//Pre-effect / transform. Utilizes re-draw regions when rendering.
	this._compositeVisibleMetrics = null;			//Visible area of the composite layer.																			
	this._redrawRegionMetrics = null;				//Region to redraw																								
	
	this._compositeCtx = null;						//Graphics context																								
	this._compositeCanvas = null;					//Off screen canvas for aggregate rendering of this + child renderings.											
	this._compositeCanvasMetrics = null;			//Metrics of the composite canvas. 																				
	
	//Post-effect / transform. 
	this._transformVisibleMetrics = null;			//Transformed _compositeVisibleMetrics
	this._transformDrawableMetrics = null;			//Transformed _compositeVisibleMetrics region after clipping is applied															
	////////////////////////////////////////////
	
	
	this._rollOverCursorInstance = null; 			//Reference to cursor list iterator for roll-over cursor.
	
	this._renderFocusRing = false;
	
	var _self = this;
	
	//Private event handlers, need instance for each element, proxy to prototype.
	this._onExternalStyleChangedInstance = 
		function (styleChangedEvent)
		{
			_self._onExternalStyleChanged(styleChangedEvent);
		};
	
	this._onBackgroundShapeStyleChangedInstance = 
		function (styleChangedEvent)
		{
			_self._onBackgroundShapeStyleChanged(styleChangedEvent);
		};
		
	this._onBackgroundFillStyleChangedInstance = 
		function (styleChangedEvent)
		{
			_self._onBackgroundFillStyleChanged(styleChangedEvent);
		};
		
	this._onCanvasElementAddedRemovedInstance = 
		function (addedRemovedEvent)
		{
			if (addedRemovedEvent.getType() == "added")
				_self._onCanvasElementAdded(addedRemovedEvent);
			else if (addedRemovedEvent.getType() == "removed")
				_self._onCanvasElementRemoved(addedRemovedEvent);
		};
		
	this._onCanvasElementCursorOverOutInstance = 
		function (elementEvent)
		{
			_self._updateRolloverCursorDefinition();
		};
		
		
	//Listen for added/removed to display chain. (Setup / Cleanup)	
	this.addEventListener("added", this._onCanvasElementAddedRemovedInstance);
	this.addEventListener("removed", this._onCanvasElementAddedRemovedInstance);
	
	
	//////////Dynamic Properties////////////////  //Added at runtime when required.
	
	///////DataRenderer/////////////
	
	/**
	 * @member _listData DataListData
	 * Read only - List data provided by parent DataList when acting as a DataRenderer
	 */
	//this._listData = any;
	
	/**
	 * @member _itemData Object
	 * Read only - Collection item associated with this DataRenderer
	 */
	//this._itemData = any;
	
	/**
	 * @member _listSelected Any
	 * Read only - DataRenderer selected state.
	 */
	//this._listSelected = any;
	
	
	///////Owner/////////////
	
	/**
	 * @member _owner CanvasElement
	 * Read only - This elements owner element. This is set by elements that pop up other elements.
	 * Such as DropdownBase sets this on the pop up element. 
	 */
	//this._owner = null; 
}

//Inherit from StyleableBase
CanvasElement.prototype = Object.create(StyleableBase.prototype);
CanvasElement.prototype.constructor = CanvasElement;
CanvasElement.base = StyleableBase;

//Style priority enum
CanvasElement.EStylePriorities = 
{
	INSTANCE:0,
	DEFINITION:1,
	PROXY:2,
	INHERITED:3,
	DEFAULT_DEFINITION:4,
	CLASS:5
};

////////////Events/////////////////////////////////////

/**
 * @event localechanged DispatcherEvent
 * @broadcast
 * Dispatched when CanvasManager's locale selection changes.
 * 
 * @event enterframe DispatcherEvent
 * @broadcast
 * Dispatched at the beginning of the frame before any life cycle processing begins.
 * 
 * @event exitframe DispatcherEvent
 * @broadcast
 * Dispatched at the end of the frame after life cycle processing ends.
 * 
 * @event mousemoveex ElementMouseEvent
 * @broadcast
 * Dispatched when the mouse moves anywhere, even outside of the browser window. Mouse coordinates are relative to CanvasManager.
 * 
 * @event resize DispatcherEvent
 * Dispatched when the element's size changes.
 * 
 * @event layoutcomplete DispatcherEvent
 * Typically an internal event. Dispatched when an element has completed its
 * layout phase. This is used when it is necessary to wait for an element to
 * finish its layout pass so things such as its PercentWidth calculation is complete.
 * This is very expensive and should only be used when absolutely necessary. Its usually
 * only needed when elements are not directly related via parent/child. 
 * For example, DropdownElement uses this to adjust the height of the dropdown popup
 * since we do not know how much height it will need until after it has finished layout.
 * 
 * @event measurecomplete DispatcherEvent
 * Typically an internal event. Dispatched when an element has completed its 
 * measure phase. This is used when it is necessary to wait for an element to
 * finish its measure pass so that its content size or _measuredSize calculation is complete.
 * This is very expensive and should only be used when absolutely necessary.  Its usually
 * only needed when elements are not directly related via parent/child.
 * For example, ViewportElement uses this to invoke a layout pass when its content child
 * changes _measuredSize. The Viewport uses an intermediate CanvasElement as a clipping container
 * for the content child, which does not measure children, so an event is needed to notify the Viewport.
 * 
 * @event keydown ElementKeyboardEvent
 * Dispatched when the element has focus and a key is pressed, repeatedly dispatched if the key is held down.
 * 
 * @event keyup ElementKeyboardEvent
 * Dispatched when the element has focus and a key is released.
 * 
 * @event mousedown ElementMouseEvent
 * Dispatched when the mouse is pressed over this element.
 * 
 * @event mouseup ElementMouseEvent
 * Dispatched when the mouse is released. Note that the mouse may not still be over the element.
 * 
 * @event click ElementMouseEvent
 * Dispatched when the mouse is pressed and released over the same element.
 * 
 * @event mousemove ElementMouseEvent
 * Dispatched when the mouse moves over this element.
 * 
 * @event wheel ElementMouseWheelEvent
 * Dispatched when the mouse wheel is rolled while over this element.
 * 
 * @event dragging ElementEvent
 * Dispatched when this element is moved due to it being dragged.
 * 
 * @event rollover ElementEvent
 * Dispatched when the mouse moves over this element.
 * 
 * @event rollout ElementEvent
 * Dispatched when the mouse moves outside of this element.
 * 
 * @event focusin ElementEvent
 * Dispatched when this element gains focus.
 * 
 * @event focusout ElementEvent
 * Dispatched when this element loses focus.
 * 
 * @event added AddedRemovedEvent
 * Dispatched when this element is added to the display hierarchy and becomes a descendant of CanvasManager. 
 * 
 * @event removed AddedRemovedEvent
 * Dispatched when this element is removed from the display hierarchy and is no longer a descendant of CanvasManager. 
 */


/////////////Style Types///////////////////////////////

CanvasElement._StyleTypes = Object.create(null);

//Rendering
/**
 * @style Visible boolean
 * 
 * When false the element will not be rendered.
 */
CanvasElement._StyleTypes.Visible = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style BorderType String
 * 
 * Determines the border type the CanvasElement should render. Allowable values are
 * "solid", "inset", "outset" or "none" / null. Note that borders are internal and drawn on the inside
 * of the elements bounding area.
 */
CanvasElement._StyleTypes.BorderType = 				StyleableBase.EStyleType.NORMAL;		// "none"/null || "solid" || "inset" || "outset"

/**
 * @style BorderColor String
 * 
 * Hex color value to be used when drawing the border. Format like "#FF0000" (red)
 */
CanvasElement._StyleTypes.BorderColor = 			StyleableBase.EStyleType.NORMAL;		// "#FF0000" or null

/**
 * @style BorderThickness Number
 * 
 * Thickness in pixels to be used when drawing the border. 
 */
CanvasElement._StyleTypes.BorderThickness = 		StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style BackgroundFill FillBase
 * 
 * Fill to use when filling the background. May be any FillBase subclass instance, 
 * "color" string, or null for transparent. 
 */
CanvasElement._StyleTypes.BackgroundFill = 			StyleableBase.EStyleType.NORMAL;		// FillBase() || "#FF0000" || null

/**
 * @style ShadowSize Number
 * 
 * Size in pixels that the drop shadow should be rendered. Note that the drop shadow may be rendered
 * outside the elements bounding area. This will cause the element to be composite rendered.
 */
CanvasElement._StyleTypes.ShadowSize = 				StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style ShadowOffsetX Number
 * 
 * X offset that the drop shadow will be rendered.
 */
CanvasElement._StyleTypes.ShadowOffsetX = 			StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style ShadowOffsetY Number
 * 
 * Y offset that the drop shadow will be rendered.
 */
CanvasElement._StyleTypes.ShadowOffsetY = 			StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style ShadowColor String
 * 
 * Hex color value to be used when drawing the drop shadow. This may be set to null and no
 * shadow will be rendered. Format like "#FF0000" (red)
 */
CanvasElement._StyleTypes.ShadowColor = 			StyleableBase.EStyleType.NORMAL;		// "#FF0000" or null

/**
 * @style Alpha Number
 * 
 * Alpha value to use when rendering this component. Allowable values are between 0 and 1 with
 * 0 being transparent and 1 being opaque. This causes the element to perform composite rendering
 * when a value between 1 and 0 is used.
 */
CanvasElement._StyleTypes.Alpha = 					StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style ClipContent boolean
 * 
 * Determines if out of bounds rendering is allowed. If true the element will clip all rendering
 * and children's rendering to the elements bounding box. This style is inheritable for container elements.
 */
CanvasElement._StyleTypes.ClipContent = 			StyleableBase.EStyleType.NORMAL;		// number (true || false)

/**
 * @style SkinState String
 * 
 * This is an internal style used to toggle an element's current skin for different states such
 * as normal, mouse-over, mouse-down, etc. Its also commonly used by skin classes to identify their skin state.
 */
CanvasElement._StyleTypes.SkinState = 				StyleableBase.EStyleType.NORMAL;		// "state"

/**
 * @style BackgroundShape ShapeBase
 * 
 * Shape to be used when rendering the elements background. May be any ShapeBase subclass instance.
 */
CanvasElement._StyleTypes.BackgroundShape = 		StyleableBase.EStyleType.NORMAL;		// ShapeBase()

/**
 * @style FocusColor String
 * @inheritable
 * 
 * Hex color value to be used when drawing the elements focus indicator. Format "#FF0000" (Red). 
 * The focus indicator is only rendered when the element gains focus due to a tab stop.
 */
CanvasElement._StyleTypes.FocusColor = 				StyleableBase.EStyleType.INHERITABLE;			// color ("#000000")

/**
 * @style FocusThickness Number
 * @inheritable
 * 
 * Size in pixels that the focus ring should be rendered. Note that the focus ring is rendered
 * outside the elements bounding area.
 */
CanvasElement._StyleTypes.FocusThickness =			StyleableBase.EStyleType.INHERITABLE;			// number


//Layout
/**
 * @style Padding Number
 * 
 * Size in pixels that inner content should be spaced from the outer bounds of the element. 
 * Padding effects all sides of the element. Padding may be negative under certain circumstances like
 * expanding an inner child to allow border collapsing with its parent.
 */
CanvasElement._StyleTypes.Padding = 				StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style PaddingTop Number
 * 
 * Size in pixels that inner content should be spaced from the upper bounds of the element. 
 * This will override the Padding style.
 */
CanvasElement._StyleTypes.PaddingTop = 				StyleableBase.EStyleType.NORMAL;		// number or null

/**
 * @style PaddingBottom Number
 * 
 * Size in pixels that inner content should be spaced from the lower bounds of the element. 
 * This will override the Padding style.
 */
CanvasElement._StyleTypes.PaddingBottom = 			StyleableBase.EStyleType.NORMAL;		// number or null

/**
 * @style PaddingLeft Number
 * 
 * Size in pixels that inner content should be spaced from the left most bounds of the element. 
 * This will override the Padding style.
 */
CanvasElement._StyleTypes.PaddingLeft = 			StyleableBase.EStyleType.NORMAL;		// number or null

/**
 * @style PaddingRight Number
 * 
 * Size in pixels that inner content should be spaced from the right most bounds of the element. 
 * This will override the Padding style.
 */
CanvasElement._StyleTypes.PaddingRight = 			StyleableBase.EStyleType.NORMAL;		// number or null


//Functional

/**
 * @style Enabled boolean
 * 
 * When false disables user interaction with the element.
 */
CanvasElement._StyleTypes.Enabled = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style MouseEnabled boolean
 * 
 * When false disables mouse events for the element.
 */
CanvasElement._StyleTypes.MouseEnabled = 			StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style Draggable boolean
 * 
 * When true allows the element to be dragged by the user. This does not work for containers
 * that do not allow absolute positioning such as a ListContainer.
 */
CanvasElement._StyleTypes.Draggable = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style Cursor CursorDefinition
 * 
 * Specifies the cursor to be displayed when the mouse is over the element. A custom CursorDefinition
 * may be used or a browser type String ("text", "none", etc) may be used.
 */
CanvasElement._StyleTypes.Cursor = 					StyleableBase.EStyleType.NORMAL;		// CursorDefinition()

/**
 * @style TabStop int
 * 
 * Determines if an element can be focused using tab stops. -1 indicates the element cannot
 * take focus, 0 is default and the element will be focused in the order it appears in the display chain.
 * Numbers higher than 0 indicate a specific order to be used (not yet implemented).
 */
CanvasElement._StyleTypes.TabStop = 				StyleableBase.EStyleType.NORMAL;		// number


//Container Placement
/**
 * @style X Number
 * 
 * The X position the element should be rendered relative to its parent container. This only
 * works if the element is a child of an AnchorContainer.
 */
CanvasElement._StyleTypes.X =						StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style Y Number
 * 
 * The Y position the element should be rendered relative to its parent container. This only
 * works if the element is a child of an AnchorContainer.
 */
CanvasElement._StyleTypes.Y =						StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style Width Number
 * 
 * The Width the element should be rendered relative to its parent container. This only
 * works if the element is a child of a Container element.
 */
CanvasElement._StyleTypes.Width =					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style Height Number
 * 
 * The Height the element should be rendered relative to its parent container. This only
 * works if the element is a child of a Container element.
 */
CanvasElement._StyleTypes.Height =					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style Top Number
 * 
 * The distance the element should be positioned from the upper bounds of the parent container. 
 * This only works if the element is a child of an AnchorContainer. 
 */
CanvasElement._StyleTypes.Top =						StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style Left Number
 * 
 * The distance the element should be positioned from the left most bounds of the parent container. 
 * This only works if the element is a child of an AnchorContainer. 
 */
CanvasElement._StyleTypes.Left =					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style Bottom Number
 * 
 * The distance the element should be positioned from the lower bounds of the parent container. 
 * This only works if the element is a child of an AnchorContainer. 
 */
CanvasElement._StyleTypes.Bottom =					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style Right Number
 * 
 * The distance the element should be positioned from the right most bounds of the parent container. 
 * This only works if the element is a child of an AnchorContainer. 
 */
CanvasElement._StyleTypes.Right =					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style PercentWidth Number
 * 
 * The percentage of available width the element should consume relative to its parent container. This only
 * works if the element is a child of a Container element. Note that percentage width is calculated
 * based on the available space left over *after* static sized elements considered. Percentages
 * are allowed to add to more than 100 and will consume all of the available space. For containers
 * like ListContainers, when percents add to more than 100 the elements will share the available space
 * per the ratio of percent vs total percent used so it is perfectly reasonable to set 3 elements all
 * to 100 and allow them to split the real-estate by 3.
 */
CanvasElement._StyleTypes.PercentWidth =			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style PercentHeight Number
 * 
 * The percentage of available height the element should consume relative to its parent container. This only
 * works if the element is a child of a Container element. Note that percentage height is calculated
 * based on the available space left over *after* static sized elements considered. Percentages
 * are allowed to add to more than 100 and will consume all of the available space. For containers
 * like ListContainers, when percents add to more than 100 the elements will share the available space
 * per the ratio of percent vs total percent used so it is perfectly reasonable to set 3 elements all
 * to 100 and allow them to split the real-estate by 3.
 */
CanvasElement._StyleTypes.PercentHeight =			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style MinWidth Number
 * 
 * The minimum width in pixels the element should consume. This only
 * works if the element is a child of a Container element.
 */
CanvasElement._StyleTypes.MinWidth =				StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style MinHeight Number
 * 
 * The minimum height in pixels the element should consume. This only
 * works if the element is a child of a Container element.
 */
CanvasElement._StyleTypes.MinHeight =				StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style MaxWidth Number
 * 
 * The maximum width in pixels the element should consume. This only
 * works if the element is a child of a Container element.
 */
CanvasElement._StyleTypes.MaxWidth =				StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style MaxHeight Number
 * 
 * The maximum height in pixels the element should consume. This only
 * works if the element is a child of a Container element.
 */
CanvasElement._StyleTypes.MaxHeight =				StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style HorizontalCenter Number
 * 
 * The distance in pixels from the horizontal center of the parent the element should be positioned.
 * Negative numbers indicate left of center, positive right of center. 
 * This only works if the element is a child of an AnchorContainer. 
 */
CanvasElement._StyleTypes.HorizontalCenter =		StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style VerticalCenter Number
 * 
 * The distance in pixels from the vertical center of the parent the element should be positioned.
 * Negative numbers indicate left of center, positive right of center. 
 * This only works if the element is a child of an AnchorContainer. 
 */
CanvasElement._StyleTypes.VerticalCenter =			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style RotateDegrees Number
 * 
 * The number of degrees the element should be rotated (clockwise). When no RotateCenterX or
 * RotateCenterY is set, the element is rotated via its center point and rotated objects are
 * still positioned relative to their parent's coordinate plane after the transform has occurred.
 */
CanvasElement._StyleTypes.RotateDegrees = 			StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style RotateCenterX Number
 * 
 * The X position of the parent container the element should be rotated around. 
 * This only works if the element is a child of an AnchorContainer. 
 */
CanvasElement._StyleTypes.RotateCenterX = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style RotateCenterY Number
 * 
 * The Y position of the parent container the element should be rotated around. 
 * This only works if the element is a child of an AnchorContainer. 
 */
CanvasElement._StyleTypes.RotateCenterY = 			StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style IncludeInLayout boolean
 * 
 * When false, the element is no longer considered in the parent container's layout. 
 * Typically this is used in conjunction with Visible, however sometimes you may want to
 * hide an element, but still have it consume container space.
 */
CanvasElement._StyleTypes.IncludeInLayout = 		StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style IncludeInMeasure boolean
 * 
 * When false, the element is no longer considered in the parent container's measurement. 
 * This is useful if you do not want containers to respect this child's measured size and/or
 * do not care if content is clipped, such as when using MinSize, MaxSize, and percent sizing together.
 * This also prevents the element's _doMeasure() function from running and forces a 0x0 measured size.
 * However, this will not prevent the "measurecomplete" event after this element finishes its measure phase.
 */
CanvasElement._StyleTypes.IncludeInMeasure = 		StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style CompositeLayer boolean
 * 
 * When true, this element renders itself and all children to a single layer and is treated
 * by its parent as a single element when rendering.  This is necessary and automatically enabled
 * for styles like alpha where the component and all its children must be pre-rendered, and then 
 * re-rendered with the appropriate effect.  
 * 
 * This is very expensive but can also be very beneficial when used appropriately.  
 * For example, if you have an application with a scrolling or constantly changing background
 * thereby always causing a full screen redraw, its beneficial to make the layer on top of the
 * background a composite layer.  This effectively buffers the layer. Only the delta changes
 * will be drawn to the composite. Otherwise the entire display chain would have to be re-drawn 
 * when the background moves. This is memory intensive as it effectively duplicates the rendering
 * area. Composite elements/children changing will update their region of their composite parent, 
 * then that region of the composite parent needs to be copied up to the grandparent, 
 * resulting in an additional buffer copy.
 */
CanvasElement._StyleTypes.CompositeLayer = 					StyleableBase.EStyleType.NORMAL;		//true || false

//Text
/**
 * @style TextStyle String
 * @inheritable
 * 
 * Determines the style to render text. Available values are "normal", "bold", "italic", and "bold italic".
 */
CanvasElement._StyleTypes.TextStyle =						StyleableBase.EStyleType.INHERITABLE;		// "normal" || "bold" || "italic" || "bold italic"


/**
 * @style TextDecoration String
 * @inheritable
 * 
 * Determines the text decoration used.  Available values are "none" or "underline".
 */
CanvasElement._StyleTypes.TextDecoration =					StyleableBase.EStyleType.INHERITABLE;		// "none" || null || "underline"

/**
 * @style TextFont String
 * @inheritable
 * 
 * Determines the font family to use when rendering text such as "Arial".
 */
CanvasElement._StyleTypes.TextFont =						StyleableBase.EStyleType.INHERITABLE;		// "Arial"

/**
 * @style TextSize int
 * @inheritable
 * 
 * Determines the size in pixels to render text.
 */
CanvasElement._StyleTypes.TextSize =						StyleableBase.EStyleType.INHERITABLE;		// number

/**
 * @style TextHorizontalAlign String
 * @inheritable
 * 
 * Determines alignment when rendering text. Available values are "left", "center", and "right".
 */
CanvasElement._StyleTypes.TextHorizontalAlign =						StyleableBase.EStyleType.INHERITABLE;		// "left" || "center" || "right"

/**
 * @style TextVerticalAlign String
 * @inheritable
 * 
 * Determines the baseline when rendering text. Available values are "top", "middle", or "bottom".
 */
CanvasElement._StyleTypes.TextVerticalAlign =					StyleableBase.EStyleType.INHERITABLE;  	// "top" || "middle" || "bottom"

/**
 * @style TextLinePaddingTop Number
 * @inheritable
 * 
 * Padding to apply to the top of each line of text. This also impacts the size of the highlight background.
 * This is useful when using strange fonts that exceed their typical vertical bounds.
 */
CanvasElement._StyleTypes.TextLinePaddingTop = 				StyleableBase.EStyleType.INHERITABLE;		// number

/**
 * @style TextLinePaddingBottom Number
 * @inheritable
 * 
 * Padding to apply to the bottom of each line of text. This also impacts the size of the highlight background.
 * This is useful when using strange fonts that exceed their typical vertical bounds.
 */
CanvasElement._StyleTypes.TextLinePaddingBottom = 			StyleableBase.EStyleType.INHERITABLE;		// number

/**
 * @style TextLineSpacing Number
 * @inheritable
 * 
 * Vertical line spacing in pixels.
 */
CanvasElement._StyleTypes.TextLineSpacing = 				StyleableBase.EStyleType.INHERITABLE;		// number

/**
 * @style TextColor String
 * @inheritable
 * 
 * Hex color value to be used when drawing text. Format like "#FF0000" (red)
 */
CanvasElement._StyleTypes.TextColor =						StyleableBase.EStyleType.INHERITABLE;		// "#000000"

/**
 * @style TextFillType String
 * @inheritable
 * 
 * Determines the fill type when rendering text. Available values are "fill" and "stroke".
 * Stroke draws a border around characters, while fill completely fills them.
 */
CanvasElement._StyleTypes.TextFillType =					StyleableBase.EStyleType.INHERITABLE;		// "fill" || "stroke"

/**
 * @style TextHighlightedColor String
 * @inheritable
 * 
 * Hex color value to be used when drawing highlighted text. Format like "#FF0000" (red)
 */
CanvasElement._StyleTypes.TextHighlightedColor = 			StyleableBase.EStyleType.INHERITABLE;		// color "#000000"

/**
 * @style TextHighlightedColor String
 * @inheritable
 * 
 * Hex color value to be used when drawing highlighted text background. Format like "#FF0000" (red)
 */
CanvasElement._StyleTypes.TextHighlightedBackgroundColor = 	StyleableBase.EStyleType.INHERITABLE;		// color "#000000"

/**
 * @style TextCaretColor String
 * @inheritable
 * 
 * Hex color value to be used when drawing blinking text caret. "#FF0000" (red)
 */
CanvasElement._StyleTypes.TextCaretColor = 					StyleableBase.EStyleType.INHERITABLE;		// color "#000000"

/**
 * @style PasswordMaskCharacter String
 * @inheritable
 * 
 * Character to use when masking a password field
 */
CanvasElement._StyleTypes.PasswordMaskCharacter = 			StyleableBase.EStyleType.INHERITABLE;		// "●"



/////////////Default Styles///////////////////////////////

CanvasElement.StyleDefault = new StyleDefinition();
//CanvasElement specific styles.
CanvasElement.StyleDefault.setStyle("Visible", 							true);
CanvasElement.StyleDefault.setStyle("BorderType", 						"none");
CanvasElement.StyleDefault.setStyle("BorderColor", 						"#000000");
CanvasElement.StyleDefault.setStyle("BorderThickness", 					1);
CanvasElement.StyleDefault.setStyle("BackgroundFill", 					null); 
CanvasElement.StyleDefault.setStyle("ShadowSize", 						0);
CanvasElement.StyleDefault.setStyle("ShadowOffsetX",					0);
CanvasElement.StyleDefault.setStyle("ShadowOffsetY",					0);
CanvasElement.StyleDefault.setStyle("ShadowColor",						"#000000");
CanvasElement.StyleDefault.setStyle("Alpha", 							1);
CanvasElement.StyleDefault.setStyle("ClipContent",						false);
CanvasElement.StyleDefault.setStyle("SkinState", 						"");
CanvasElement.StyleDefault.setStyle("BackgroundShape", 					null); 		//ShapeBase
CanvasElement.StyleDefault.setStyle("FocusColor", 						"#3333FF");	// color ("#000000")
CanvasElement.StyleDefault.setStyle("FocusThickness", 					1);			// number

CanvasElement.StyleDefault.setStyle("Padding", 							0); 		//Not necessary, just for completeness
CanvasElement.StyleDefault.setStyle("PaddingTop", 						0);
CanvasElement.StyleDefault.setStyle("PaddingBottom",					0);
CanvasElement.StyleDefault.setStyle("PaddingLeft", 						0);
CanvasElement.StyleDefault.setStyle("PaddingRight", 					0);

CanvasElement.StyleDefault.setStyle("Enabled", 							true);
CanvasElement.StyleDefault.setStyle("MouseEnabled", 					true);
CanvasElement.StyleDefault.setStyle("Draggable", 						false);
CanvasElement.StyleDefault.setStyle("Cursor", 							null);		// "browsertype" || CursorDefinition
CanvasElement.StyleDefault.setStyle("TabStop", 							-1);		// number

CanvasElement.StyleDefault.setStyle("X", 								null);
CanvasElement.StyleDefault.setStyle("Y", 								null);
CanvasElement.StyleDefault.setStyle("Width", 							null);
CanvasElement.StyleDefault.setStyle("Height", 							null);
CanvasElement.StyleDefault.setStyle("Top", 								null);
CanvasElement.StyleDefault.setStyle("Left", 							null);
CanvasElement.StyleDefault.setStyle("Bottom", 							null);
CanvasElement.StyleDefault.setStyle("Right", 							null);
CanvasElement.StyleDefault.setStyle("PercentWidth", 					null);
CanvasElement.StyleDefault.setStyle("PercentHeight", 					null);
CanvasElement.StyleDefault.setStyle("MinWidth", 						5);		
CanvasElement.StyleDefault.setStyle("MinHeight", 						5);
CanvasElement.StyleDefault.setStyle("MaxWidth", 						null);
CanvasElement.StyleDefault.setStyle("MaxHeight", 						null);
CanvasElement.StyleDefault.setStyle("HorizontalCenter", 				null);
CanvasElement.StyleDefault.setStyle("VerticalCenter", 					null);
CanvasElement.StyleDefault.setStyle("RotateDegrees", 					0);
CanvasElement.StyleDefault.setStyle("RotateCenterX", 					null);
CanvasElement.StyleDefault.setStyle("RotateCenterY", 					null);
CanvasElement.StyleDefault.setStyle("IncludeInLayout", 					true);
CanvasElement.StyleDefault.setStyle("IncludeInMeasure", 				true);
CanvasElement.StyleDefault.setStyle("CompositeLayer",					false);

CanvasElement.StyleDefault.setStyle("TextStyle", 						"normal");
CanvasElement.StyleDefault.setStyle("TextDecoration", 					null);
CanvasElement.StyleDefault.setStyle("TextFont", 						"Arial");
CanvasElement.StyleDefault.setStyle("TextSize", 						12);
CanvasElement.StyleDefault.setStyle("TextHorizontalAlign",				"left");
CanvasElement.StyleDefault.setStyle("TextVerticalAlign", 				"middle");
CanvasElement.StyleDefault.setStyle("TextLinePaddingTop", 				2);
CanvasElement.StyleDefault.setStyle("TextLinePaddingBottom", 			1);
CanvasElement.StyleDefault.setStyle("TextLineSpacing", 					0);
CanvasElement.StyleDefault.setStyle("TextColor", 						"#000000");
CanvasElement.StyleDefault.setStyle("TextFillType", 					"fill");
CanvasElement.StyleDefault.setStyle("TextHighlightedColor", 			"#000000");
CanvasElement.StyleDefault.setStyle("TextHighlightedBackgroundColor", 	"#CCCCCC");
CanvasElement.StyleDefault.setStyle("TextCaretColor", 					"#000000");
CanvasElement.StyleDefault.setStyle("PasswordMaskCharacter", 			"●");


///////////CanvasElement Public Functions///////////////////////////////

/**
 * @function addStyleDefinition
 * Adds a style definition to the end element's definition list. Styles in this definition
 * will override styles in previously added definitions (lower index). Instance styles, set 
 * using setStyle() will override all definition styles.
 * Adding style definitions to elements already attached to the display chain is expensive, 
 * for better performance add definitions before attaching the element via addElement()
 * 
 * @param styleDefinition StyleDefinition
 * The StyleDefinition to add and associate with the element.
 * 
 * @returns StyleDefinition
 * The style definition just added.
 */
CanvasElement.prototype.addStyleDefinition = 
	function (styleDefinition)
	{
		return this.addStyleDefinitionAt(styleDefinition, this._styleDefinitions.length);
	};
	
/**
 * @function addStyleDefinitionAt
 * Inserts a style definition to this elements definition list at the specified index.
 * Definitions with higher indexes (added later) are higher priority. Instance styles, set 
 * using setStyle() will override all definition styles. 
 * Adding style definitions to elements already attached to the display chain is expensive, 
 * for better performance add definitions before attaching the element via addElement()
 * 
 * @param styleDefinition StyleDefinition
 * StyleDefinition to be added to this elements definition list.
 * 
 * @param index int
 * The index to insert the style definition within the elements definition list.
 * 
 * @returns StyleDefinition
 * Returns StyleDefinition just added when successfull, null if the StyleDefinition could not
 * be added due to the index being out of range or other error.
 */	
CanvasElement.prototype.addStyleDefinitionAt = 
	function (styleDefinition, index)
	{
		return this._addStyleDefinitionAt(styleDefinition, index, false);
	};
	
	
/**
 * @function getStyleDefinitionIndex
 * Returns the index of the supplied style definition or -1 if the style definition
 * has not been added.
 * 
 * @param styleDefinition StyleDefinition
 * StyleDefinition to return associated index.
 * 
 * @returns int
 * Index of the StyleDefinition supplied via the styleDefinition parameter, or -1 if does not exist.
 */
CanvasElement.prototype.getStyleDefinitionIndex = 
	function (styleDefinition)
	{
		return this._styleDefinitions.indexOf(styleDefinition);
	};
	
/**
 * @function removeStyleDefinition
 * Removes the supplied style definition from the element's style chain.
 * 
 * @param styleDefinition StyleDefinition
 * The StyleDefinition to remove from the element.
 * 
 * @returns StyleDefinition
 * The style definition just removed, or null if the supplied style 
 * definition is not associated with this element.
 */	
CanvasElement.prototype.removeStyleDefinition = 
	function (styleDefinition)
	{
		var index = this._styleDefinitions.indexOf(styleDefinition);
		if (index == -1)
			return null;
	
		return this.removeStyleDefinitionAt(index);
	};
	
/**
 * @function removeStyleDefinitionAt
 * Removes the style definition from the elements definition list at the supplied index.
 * 
 * @param index int
 * Index to be removed.
 * 
 * @returns StyleDefinition
 * Returns the StyleDefinition just removed if successfull, null if the definition could
 * not be removed due it it not being in this elements definition list, or index out of range.
 */		
CanvasElement.prototype.removeStyleDefinitionAt = 
	function (index)
	{
		return this._removeStyleDefinitionAt(index, false);
	};
	
/**
 * @function clearStyleDefinitions
 * Removes all style definitions from the element. This is more efficient than
 * removing definitions one at a time.
 */		
CanvasElement.prototype.clearStyleDefinitions = 
	function ()
	{
		return this._setStyleDefinitions([], false);
	};
	
/**
 * @function setStyleDefinitions
 * Replaces the elements current style definition list. This is more effecient than removing or 
 * adding style definitions one at a time.
 * 
 * @param styleDefinitions StyleDefinition
 * May be a StyleDefinition, or an Array of StyleDefinition, including nested Arrays.
 * Note that nested arrays will be flattened into a single array.
 */
CanvasElement.prototype.setStyleDefinitions = 
	function (styleDefinitions)
	{
		return this._setStyleDefinitions(styleDefinitions, false);
	};
	

	
/**
 * @function getNumStyleDefinitions
 * Gets the number of style definitions associated with this element.
 * 
 * @returns int
 * The number of style definitions.
 */		
CanvasElement.prototype.getNumStyleDefinitions = 
	function ()
	{
		return this._getNumStyleDefinitions(false);
	};
	
/**
 * @function _getNumStyleDefinitions
 * Gets the number of style definitions or default definitions associated with this element.
 * 
 * @param isDefault bool
 * When true, returns the number of default definitions.
 * 
 * @returns int
 * The number of style definitions.
 */			
CanvasElement.prototype._getNumStyleDefinitions = 
	function (isDefault)
	{
		if (isDefault == true)
			return this._styleDefinitionDefaults.length;

		return this._styleDefinitions.length;
	};
	
/**
 * @function getStyleDefinitionAt
 * Gets the style definition at the supplied zero base index.
 * 
 * @param index int
 * Index of the style definition to return;
 * 
 * @returns StyleDefinition
 * The style defenition at the supplied index, or null if index is out of range. 
 */		
CanvasElement.prototype.getStyleDefinitionAt = 
	function (index)
	{
		return this._getStyleDefinitionAt(index, false);
	};
	
/**
 * @function getStyle
 * @override
 * 
 * Gets the style value for this element. When retrieving a style, CanvasElements look
 * through their associated style chain, at each step if undefined is returned, they look
 * at the next step until a non-undefined value is found.
 * 
 * 1) Instance - Styles set directly to the element via setStyle()
 * 2) StyleDefinitions - Styles associated via its assigned StyleDefinitions
 * 3) StyleProxy - If proxy element is assigned, move to proxy element and repeat steps 1-3
 * 4) Inheritable - If style is inheritable, move up to parent element and repeat steps 1-4
 * 5) Default styles
 * 
 * @seealso StyleProxy
 * @seealso StyleableBase
 * 
 * @param styleName String
 * String representing the style value to be returned.
 * 
 * @returns Any
 * Returns the associated style value if found, otherwise undefined.
 * 
 */
CanvasElement.prototype.getStyle = 
	function (styleName)
	{
		return CanvasElement.base.prototype.getStyle.call(this, styleName);
	};	
	
//@override
CanvasElement.prototype.getStyleData = 
	function (styleName)
	{
		//Create cache if does not exist.
		var styleCache = this._stylesCache[styleName];
		if (styleCache == null)
		{
			styleCache = {styleData:new StyleData(styleName), cacheInvalid:true};
			this._stylesCache[styleName] = styleCache;
		}
		
		//Check cache
		if (styleCache.cacheInvalid == false)
			return styleCache.styleData;
		
		styleCache.cacheInvalid = false;
		var styleData = styleCache.styleData;
		
		//Reset the cache data.
		styleData.value = undefined;
		
		//Check instance
		if (styleName in this._styleMap)
			styleData.value = this._styleMap[styleName];
		
		if (styleData.value !== undefined)
		{
			styleData.priority.length = 1;
			styleData.priority[0] = CanvasElement.EStylePriorities.INSTANCE;
			
			return styleData;
		}
		
		//Counters (priority depth)
		var ctr = 0;
		
		//Check definitions
		for (ctr = this._styleDefinitions.length - 1; ctr >= 0; ctr--)
		{
			styleData.value = this._styleDefinitions[ctr].getStyle(styleName);
			
			if (styleData.value !== undefined)
			{
				styleData.priority.length = 2;
				styleData.priority[0] = CanvasElement.EStylePriorities.DEFINITION;
				styleData.priority[1] = (this._styleDefinitions.length - 1) - ctr; //StyleDefinition depth
				
				return styleData;
			}
		}
		
		var thisStyleType = this._getStyleType(styleName);
		
		var styleType = null;
		var proxy = null;
		var ctr2 = 0;
		
		//Proxy / Inheritable not allowed for sub styles.
		if (thisStyleType != StyleableBase.EStyleType.SUBSTYLE)
		{
			//Check proxy
			proxy =	this._styleProxy;
			while (proxy != null)
			{
				styleType = proxy._proxyElement._getStyleType(styleName);
				
				//Proxy not allowed for sub styles.
				if (styleType == StyleableBase.EStyleType.SUBSTYLE)
					break;
				
				if ((styleType != null && styleName in proxy._proxyMap == false) ||		//Defined & not in proxy map
					(styleType == null && "_Arbitrary" in proxy._proxyMap == false)) 	//Not defined and no _Arbitrary flag
					break;
				
				//Check proxy instance
				if (styleName in proxy._proxyElement._styleMap)
					styleData.value = proxy._proxyElement._styleMap[styleName];
				
				if (styleData.value !== undefined)
				{
					styleData.priority.length = 3;
					styleData.priority[0] = CanvasElement.EStylePriorities.PROXY;		
					styleData.priority[1] = ctr;	//Proxy depth (chained proxies)
					styleData.priority[2] = CanvasElement.EStylePriorities.INSTANCE;	
					
					return styleData;
				}
				
				//Check proxy definitions
				for (ctr2 = proxy._proxyElement._styleDefinitions.length - 1; ctr2 >= 0; ctr2--)
				{
					styleData.value = proxy._proxyElement._styleDefinitions[ctr2].getStyle(styleName);
					
					if (styleData.value !== undefined)
					{
						styleData.priority.length = 4;
						styleData.priority[0] = CanvasElement.EStylePriorities.PROXY;
						styleData.priority[1] = ctr;	//Proxy depth (chained proxies)
						styleData.priority[2] = CanvasElement.EStylePriorities.DEFINITION;	
						styleData.priority[3] = (proxy._proxyElement._styleDefinitions.length - 1) - ctr2; //definition depth	
						
						return styleData;
					}
				}
				
				ctr++;
				proxy = proxy._proxyElement._styleProxy;
			}
			
			//Check inherited
			proxy = null;
			styleType = thisStyleType;
			var parent = this;
			
			ctr = 0;
			ctr2 = 0;
			var ctr3 = 0;
			
			while (styleType == StyleableBase.EStyleType.INHERITABLE)
			{
				parent = parent._parent;
				
				if (parent == null)
					break;
				
				//Check parent instance
				if (styleName in parent._styleMap)
					styleData.value = parent._styleMap[styleName];
				
				if (styleData.value !== undefined)
				{
					styleData.priority.length = 3;	
					styleData.priority[0] = CanvasElement.EStylePriorities.INHERITED;	
					styleData.priority[1] = ctr;	//Parent depth
					styleData.priority[2] = CanvasElement.EStylePriorities.INSTANCE;
									
					return styleData;
				}
				
				//Check style definitions
				for (ctr2 = parent._styleDefinitions.length - 1; ctr2 >= 0; ctr2--)
				{
					styleData.value = parent._styleDefinitions[ctr2].getStyle(styleName);
					
					if (styleData.value !== undefined)
					{
						styleData.priority.length = 4;
						styleData.priority[0] = CanvasElement.EStylePriorities.INHERITED;	
						styleData.priority[1] = ctr;	//Parent depth
						styleData.priority[2] = CanvasElement.EStylePriorities.DEFINITION;
						styleData.priority[3] = (parent._styleDefinitions.length - 1) - ctr2; //Definition depth	
						
						return styleData;
					}
				}
				
				//Check parent proxy
				proxy = parent._styleProxy;
				ctr2 = 0;
				while (proxy != null)
				{
					styleType = proxy._proxyElement._getStyleType(styleName);
					
					//Proxy not allowed for sub styles.
					if (styleType == StyleableBase.EStyleType.SUBSTYLE)
						break;
					
					if ((styleType != null && styleName in proxy._proxyMap == false) ||		//Defined & not in proxy map
						(styleType == null && "_Arbitrary" in proxy._proxyMap == false)) 	//Not defined and no _Arbitrary flag
						break;
					
					//Check proxy instance
					if (styleName in proxy._proxyElement._styleMap)
						styleData.value = proxy._proxyElement._styleMap[styleName];
					
					if (styleData.value !== undefined)
					{
						styleData.priority.length = 5;
						styleData.priority[0] = CanvasElement.EStylePriorities.INHERITED;		
						styleData.priority[1] = ctr;	//Parent depth
						styleData.priority[2] = CanvasElement.EStylePriorities.PROXY;		
						styleData.priority[3] = ctr2;	//Proxy depth (chained proxies)
						styleData.priority[4] = CanvasElement.EStylePriorities.INSTANCE;		
						
						return styleData;
					}
					
					//Check proxy definition
					for (ctr3 = proxy._proxyElement._styleDefinitions.length - 1; ctr3 >= 0; ctr3--)
					{
						styleData.value = proxy._proxyElement._styleDefinitions[ctr3].getStyle(styleName);
						
						if (styleData.value !== undefined)
						{
							styleData.priority.length = 6;
							styleData.priority[0] = CanvasElement.EStylePriorities.INHERITED;	
							styleData.priority[1] = ctr;	//Parent depth
							styleData.priority[2] = CanvasElement.EStylePriorities.PROXY;	
							styleData.priority[3] = ctr2;	//Proxy depth (chained proxies)
							styleData.priority[4] = CanvasElement.EStylePriorities.DEFINITION;
							styleData.priority[5] = (parent._styleDefinitions.length - 1) - ctr3; //Definition depth	
														
							return styleData;
						}
					}
	
					ctr2++;
					proxy = proxy._proxyElement._styleProxy;
				}
				
				ctr++;
				styleType = parent._getStyleType(styleName);
			}
		}
		
		//Check default definitions
		for (ctr = this._styleDefinitionDefaults.length - 1; ctr >= 0; ctr--)
		{
			styleData.value = this._styleDefinitionDefaults[ctr].getStyle(styleName);
			
			if (styleData.value !== undefined)
			{
				styleData.priority.length = 2;
				styleData.priority[0] = CanvasElement.EStylePriorities.DEFAULT_DEFINITION;
				styleData.priority[1] = (this._styleDefinitionDefaults.length - 1) - ctr; //StyleDefinition depth
				
				return styleData;
			}
		}
		
		//Check class
		styleData.priority.length = 1;
		styleData.value = this._getClassStyle(styleName);
		styleData.priority[0] = CanvasElement.EStylePriorities.CLASS;
		
		return styleData;		
	};
	
//@override	
CanvasElement.prototype.setStyle = 
	function (styleName, value)
	{
		var oldValue = undefined;
		if (styleName in this._styleMap)
			oldValue = this._styleMap[styleName];

		//No change
		if (oldValue === value)
			return;
		
		if (value === undefined)
			delete this._styleMap[styleName];
		else
			this._styleMap[styleName] = value;
		
		//Spoof a style changed event and pass it to _onExternalStyleChanged for normal handling
		this._onExternalStyleChanged(new StyleChangedEvent(styleName));
	};			

/**
 * @function getManager
 * Gets the CanvasManager currently associated with this element.
 * 
 * @returns CanvasManager
 * The CanvasManager currently associated with this element.
 */	
CanvasElement.prototype.getManager = 
	function ()
	{
		return this._manager;
	};

/**
 * @function setName
 * Sets an arbitrary name to this element. The system does not use this value,
 * it is for use by implementors if a way to differentiate elements is needed.
 * 
 * @param name String
 * A String to use as the element's name.
 */	
CanvasElement.prototype.setName = 
	function (name)
	{
		if (this._name == name)
			return false;
		
		this._name = name;
		return true;
	};
	
/**
 * @function getName
 * Gets the name associated with this element.
 * 
 * @returns String
 * The name associated with this element.
 */		
CanvasElement.prototype.getName = 
	function ()
	{
		return this._name;
	};
	
/**
 * @function getMouseIsDown
 * Gets the state of the mouse for this element.
 * 
 * @returns boolean
 * Returns true if the mouse is currently pressed, false otherwise.
 */		
CanvasElement.prototype.getMouseIsDown = 
	function()
	{
		return this._mouseIsDown;
	};	

/**
 * @function getParent
 * Gets this element's parent element.
 * 
 * @returns CanvasElement
 * This element's parent element.
 */		
CanvasElement.prototype.getParent = 
	function ()
	{
		return this._parent;
	};
	
/**
 * @function rotatePoint
 * Rotates a point point on this element's parent relative to this element's rotation transformation.
 * This is used to transform a point from the parent's coordinate plane to a child's coordinate plane or vice versa.
 * Typically you should use translatePointFrom() or translatePointTo() rather than rotatePoint().
 * 
 * @param point Object
 * Point object {x:0, y:0};
 * 
 * @param reverse boolean
 * When true, rotates a point on the parent's plane, to the childs plane. 
 * When false, rotates a point on the childs plane, to the parents plane.
 */	
CanvasElement.prototype.rotatePoint = 
	function (point, reverse)
	{
		if (this._rotateDegrees == 0)
			return;
		
		var radius = 
			Math.sqrt(
					(Math.abs(point.x - this._rotateCenterX) * Math.abs(point.x - this._rotateCenterX)) +
					(Math.abs(point.y - this._rotateCenterY) * Math.abs(point.y - this._rotateCenterY))
					);
		
		var degrees;
		if (reverse == false)
			degrees = 360 - this._rotateDegrees + CanvasElement.radiansToDegrees(Math.atan2(point.x - this._rotateCenterX, point.y - this._rotateCenterY));
		else
			degrees = 360 + this._rotateDegrees + CanvasElement.radiansToDegrees(Math.atan2(point.x - this._rotateCenterX, point.y - this._rotateCenterY));
			
		point.x = Math.sin(CanvasElement.degreesToRadians(degrees)) * radius + this._rotateCenterX;
		point.y = Math.cos(CanvasElement.degreesToRadians(degrees)) * radius + this._rotateCenterY;
	};

/**
 * @function translatePointFrom
 * Translates a point from an element to this element regardless of this element's transformation,
 * depth, or position in the display hierarchy. For example, you can call this to translate a point on
 * the canvas to the relative point on this element.
 * 
 * @param point Object
 * Point - object containing {x:0, y:0}.
 * 
 * @param relativeFromElement CanvasElement
 * The element that the supplied point is relative too.
 */	
CanvasElement.prototype.translatePointFrom = 
	function (point, relativeFromElement)
	{
		return relativeFromElement.translatePointTo(point, this);
	};

/**
 * @function translatePointTo
 * Translates a point from this element to another element regardless of this element's transformation,
 * depth, or position in the display hierarchy. For example, you can call this to translate a point on
 * this element to a point on the canvas.
 * 
 * @param point Object
 * Point - object containing {x:0, y:0}.
 * 
 * @param relativeToElement CanvasElement
 * The element to translate this element's point too.
 */		
CanvasElement.prototype.translatePointTo = 
	function (point, relativeToElement)
	{
		if (relativeToElement == null || relativeToElement == this)
			return false;
		
		if (this._manager == null || this._manager != relativeToElement._manager)
			return false;
		
		//Build up both parent chains so we can find common parent//
		////////////////////////////////////////////////////////////
		
		var commonParent = null;
		
		//We are a child of relativeElement
		var thisChain = [];
		thisChain.push(this);
		while (commonParent == null && thisChain[thisChain.length - 1]._parent != null)
		{
			if (thisChain[thisChain.length - 1]._parent == relativeToElement)
				commonParent = relativeToElement;
			else
				thisChain.push(thisChain[thisChain.length - 1]._parent);
		}
		
		//Relative element is a child of us.
		var relativeChain = [];
		if (commonParent == null)
		{
			relativeChain.push(relativeToElement);
			while (commonParent == null && relativeChain[thisChain.length - 1]._parent != null)
			{
				if (relativeChain[relativeChain.length - 1]._parent == this)
					commonParent = this;
				else
					relativeChain.push(relativeChain[relativeChain.length - 1]._parent);
			}
		}
		
		//Someone is doing something weird and we're not in each others direct chains so we have to translate up AND down.
		if (commonParent == null)
		{
			//We know we have the same canvas manager, so just keep popping both arrays till we find something different.
			while (thisChain[thisChain.length - 1] == relativeChain[relativeChain.length - 1])
			{
				commonParent = thisChain[thisChain.length - 1];
				
				thisChain.pop();
				relativeChain.pop();
			}
		}
		
		//Translate up to common parent.
		var currentParent = this;
		while (currentParent != null && currentParent != commonParent)
		{
			point.x += currentParent._x;
			point.y += currentParent._y;
			
			currentParent.rotatePoint(point, false);
			
			currentParent = currentParent._parent;
		}
		
		//Translate down to relativeElement
		for (var i = relativeChain.length - 1; i >= 0; i--)
		{
			//Rotate the point backwards so we can translate the point to the element's rotated plane.
			relativeChain[i].rotatePoint(point, true);
			
			//Adjust the mouse point to within this element rather than its position in parent.
			point.x -= relativeChain[i]._x;
			point.y -= relativeChain[i]._y;
		}
		
		return true;
	};
	
/**
 * @function translateMetricsFrom
 * Translates a DrawMetrics from another element's to this element regardless of this element's transformation,
 * depth, or position in the display hierarchy. 
 * 
 * @param metrics DrawMetrics
 * Metrics to transform from the relative to this element.
 * 
 * @param relativeFromElement CanvasElement
 * The element to translate the supplied metrics too. If relativeToElement equals
 * null or this, will return metrics the same as the supplied metrics.
 * 
 * @returns DrawMetrics
 * Translated DrawMetrics relative to the supplied element.
 */	
CanvasElement.prototype.translateMetricsFrom = 	
	function (metrics, relativeFromElement)
	{
		return relativeFromElement.translateMetricsTo(metrics, this);
	};
	
/**
 * @function translateMetricsTo
 * Translates a DrawMetrics from this element's to another element regardless of this element's transformation,
 * depth, or position in the display hierarchy. 
 * 
 * @param metrics DrawMetrics
 * Metrics to transform to from this element to the supplied relative element.
 * 
 * @param relativeToElement CanvasElement
 * The element to translate the supplied metrics too. If relativeToElement equals
 * null or this, will return metrics the same as the supplied metrics.
 * 
 * @returns DrawMetrics
 * Translated DrawMetrics relative to the supplied element.
 */	
CanvasElement.prototype.translateMetricsTo = 
	function (metrics, relativeToElement)
	{
		var translatedMetrics = new DrawMetrics();
		if (relativeToElement == null || relativeToElement == this)
		{
			translatedMetrics._x = metrics._x;
			translatedMetrics._y = metrics._y;
			translatedMetrics._width = metrics._width;
			translatedMetrics._height = metrics._height;
			
			return translatedMetrics;
		}

		if (this._manager == null || this._manager != relativeToElement._manager)
			return null;
		
		//Build up both parent chains so we can find common parent.
		var commonParent = null;
		
		//We are a child of relativeElement
		var thisChain = [];
		thisChain.push(this);
		while (commonParent == null && thisChain[thisChain.length - 1]._parent != null)
		{
			if (thisChain[thisChain.length - 1]._parent == relativeToElement)
				commonParent = relativeToElement;
			else
				thisChain.push(thisChain[thisChain.length - 1]._parent);
		}
		
		//Relative element is a child of us.
		var relativeChain = [];
		if (commonParent == null)
		{
			relativeChain.push(relativeToElement);
			while (commonParent == null && relativeChain[thisChain.length - 1]._parent != null)
			{
				if (relativeChain[relativeChain.length - 1]._parent == this)
					commonParent = this;
				else
					relativeChain.push(relativeChain[relativeChain.length - 1]._parent);
			}
		}
		
		//Someone is doing something weird and we're not in each others direct chains so we have to translate up AND down.
		if (commonParent == null)
		{
			//We know we have the same canvas manager, so just keep popping both arrays till we find something different.
			while (thisChain[thisChain.length - 1] == relativeChain[relativeChain.length - 1])
			{
				commonParent = thisChain[thisChain.length - 1];
				
				thisChain.pop();
				relativeChain.pop();
			}
		}
		
		var pointTl = {x:metrics._x, y:metrics._y};
		var pointTr = {x:metrics._x + metrics._width, y:metrics._y};
		var pointBr = {x:metrics._x + metrics._width, y:metrics._y + metrics._height};
		var pointBl = {x:metrics._x, y:metrics._y + metrics._height};
		
		//Translate up to common parent.
		var currentParent = this;
		while (currentParent != null && currentParent != commonParent)
		{
			pointTl.x += currentParent._x;
			pointTl.y += currentParent._y;
			
			pointTr.x += currentParent._x;
			pointTr.y += currentParent._y;
			
			pointBr.x += currentParent._x;
			pointBr.y += currentParent._y;
			
			pointBl.x += currentParent._x;
			pointBl.y += currentParent._y;
			
			currentParent.rotatePoint(pointTl, false);
			currentParent.rotatePoint(pointTr, false);
			currentParent.rotatePoint(pointBl, false);
			currentParent.rotatePoint(pointBr, false);
			
			currentParent = currentParent._parent;
		}
		
		//Translate down to relativeElement
		for (var i = relativeChain.length - 1; i >= 0; i--) 
		{
			//Rotate the point backwards so we can translate the point to the element's rotated plane.
			relativeChain[i].rotatePoint(pointTl, true);
			relativeChain[i].rotatePoint(pointTr, true);
			relativeChain[i].rotatePoint(pointBl, true);
			relativeChain[i].rotatePoint(pointBr, true);
			
			//Adjust the mouse point to within this element rather than its position in parent.
			pointTl.x -= relativeChain[i]._x;
			pointTl.y -= relativeChain[i]._y;
			
			pointTr.x -= relativeChain[i]._x;
			pointTr.y -= relativeChain[i]._y;
			
			pointBr.x -= relativeChain[i]._x;
			pointBr.y -= relativeChain[i]._y;
			
			pointBl.x -= relativeChain[i]._x;
			pointBl.y -= relativeChain[i]._y;
		}
		
		var minX = Math.min(pointTl.x, pointTr.x, pointBr.x, pointBl.x);
		var maxX = Math.max(pointTl.x, pointTr.x, pointBr.x, pointBl.x);
		var minY = Math.min(pointTl.y, pointTr.y, pointBr.y, pointBl.y);
		var maxY = Math.max(pointTl.y, pointTr.y, pointBr.y, pointBl.y);
		
		translatedMetrics._x = minX;
		translatedMetrics._y = minY;
		translatedMetrics._width = maxX - minX;
		translatedMetrics._height = maxY - minY;
		
		return translatedMetrics;
	};	
	
/**
 * @function getMetrics
 * Gets a DrawMetrics object containing the elements bounding box information
 * x, y, width, height, relative to the supplied element regardless of this element's transformation,
 * depth, or position in the display hierarchy. For example, you can call this get the elements width and height,
 * or to get this element's bounding box relative to the canvas or any other element.
 * 
 * @param relativeToElement CanvasElement
 * The element to translate this elements bounding box too. If relativeToElement equals
 * null or this, will return metrics relative to this element: {x:0, y:0, width:thisWidth, height:thisHeight}.
 * 
 * @returns DrawMetrics
 * DrawMetrics of this element relative to the supplied element.
 */	
CanvasElement.prototype.getMetrics = 
	function (relativeToElement)
	{
		if (relativeToElement == null)
			relativeToElement = this;
	
		if (this._manager == null || this._manager != relativeToElement._manager)
			return null;
	
		var metrics = new DrawMetrics();
		metrics._x = 0;
		metrics._y = 0;
		metrics._width = this._width;
		metrics._height = this._height;
		
		if (relativeToElement == this)
			return metrics;
		
		return this.translateMetricsTo(metrics, relativeToElement);
	};

//@Override
CanvasElement.prototype.addEventListener = 	
	function (type, callback)
	{
		CanvasElement.base.prototype.addEventListener.call(this, type, callback);
	
		//Broadcast events (dispatched only by manager)
		if ((type == "enterframe" || 
			type == "exitframe" ||
			type == "localechanged" || 
			type == "mousemoveex") &&
			this._manager != null)
		{
			this._manager._broadcastDispatcher.addEventListener(type, callback);
		}
		
		return true;
	};

//@Override	
CanvasElement.prototype.removeEventListener = 
	function (type, callback)
	{
		if (CanvasElement.base.prototype.removeEventListener.call(this, type, callback) == true)
		{
			//Broadcast events (dispatched only by manager)
			if ((type == "enterframe" || 
				type == "exitframe" ||
				type == "localechanged" || 
				type == "mousemoveex") &&
				this._manager != null)
			{
				this._manager._broadcastDispatcher.removeEventListener(type, callback);
			}
			
			return true;
		}
		
		return false;
	};	

	
////////////Capture Phase Event Listeners///////////////////////	
	
/**
 * @function addCaptureListener
 * Registers an event listener function to be called during capture phase.
 * 
 * @seealso ElementEvent
 * 
 * 
 * @param type String
 * String representing the event type.
 * 
 * @param callback Function
 * Function to be called when the event occurs.
 */	
CanvasElement.prototype.addCaptureListener = 
	function (type, callback)
	{
		if (this._captureListeners[type] == null)
			this._captureListeners[type] = [];
		
		this._captureListeners[type].push(callback);
		
		return true;
	};

/**
 * @function removeCaptureListener
 * Removes a capture event listener.
 * 
 * @seealso ElementEvent
 * 
 * 
 * @param type String
 * String representing the event type.
 * 
 * @param callback Function
 * Function callback to be removed.
 * 
 * @returns boolean
 * Returns true if the callback was successfully removed, otherwise false
 * such as if the function callback was not previously registered.
 */		
CanvasElement.prototype.removeCaptureListener = 
	function (type, callback)
	{
		if (!(type in this._captureListeners))
			return false;
	
		for (var i = 0; i < this._captureListeners[type].length; i++)
		{
			if (this._captureListeners[type][i] == callback)
			{
				this._captureListeners[type].splice(i, 1);
				return true;
			}
		}
		
		return false;
	};

/**
 * @function hasCaptureListener
 * Checks if an event capture listener has been registered with this CanvasElement
 * 
 * @seealso ElementEvent
 * 
 * 
 * @param type String
 * String representing the event type.
 * 
 * @param callback Function
 * Function callback to be called when the event occurs. This may be null to check
 * if the CanvasElement has any capture events registered for the provided type.
 * 
 * @returns boolean
 * Returns true if the CanvasElement has the provided capture callback registered for the 
 * provided type, or any capture callback for the provided type if the callback parameter is null.
 * Otherwise, returns false.
 */		
CanvasElement.prototype.hasCaptureListener = 
	function (type, callback)
	{
		if (!(type in this._captureListeners))
			return false;
	
		if (callback == null && this._captureListeners[type].length > 0)
			return true;
		
		for (var i = 0; i < this._captureListeners[type].length; i++)
		{
			if (this._captureListeners[type][i] == callback)
				return true;
		}
		
		return false;
	};	
	
/////////////CanvasElement Public Static Functions//////////////////

/**
 * @function adjustColorLight
 * @static
 * Adjusts supplied color brightness.
 * 
 * @param color String
 * Hex color value be adjusted. Format like "#FF0000" (red)
 * 
 * @param percent Number
 * Value between -1 and +1. -1 will return white. +1 will return black.
 * 
 * @returns String
 * Adjusted Hex color value.
 */
//Looks complicated... not really. Its using a percentage of the distance between black(neg) or white(pos) on all 3 channels independently.
CanvasElement.adjustColorLight = 
	function (color, percent) 
	{   
	    var f = parseInt(color.slice(1), 16);
	    var t = percent < 0 ? 0 : 255;
	    var p = percent < 0 ? percent * -1 : percent;
	    var R = f >> 16;
	    var G = f >> 8 & 0x00FF;
	    var B = f & 0x0000FF;
	    
	    return "#"+(0x1000000+(Math.round((t-R)*p)+R)*0x10000+(Math.round((t-G)*p)+G)*0x100+(Math.round((t-B)*p)+B)).toString(16).slice(1);
	};
	
/**
 * @function cot
 * @static
 * Calculates the cotangent of supplied radians.
 * 
 * @param radians Number
 * Radians to calculate cotangent
 * 
 * @returns Number
 * Resulting cotangent from radians
 */
CanvasElement.cot = 
	function (radians)
	{
		return 1 / Math.tan(radians);
	};
	
/**
 * @function radiansToDegrees
 * @static
 * Calculates radians to degrees.
 * 
 * @param radians Number
 * Radians to be calculated to degrees.
 * 
 * @returns Number
 * Resulting degrees from supplied radians.
 */	
CanvasElement.radiansToDegrees = 
	function (radians)
	{
		return radians * (180 / Math.PI);
	};
	
/**
 * @function degreesToRadians
 * @static
 * Calculates degrees to radians.
 * 
 * @param degrees Number
 * Degrees to be calculated to degrees.
 * 
 * @returns Number
 * Resulting radians from supplied degrees.
 */		
CanvasElement.degreesToRadians = 
	function (degrees)
	{
		return degrees * (Math.PI / 180);
	};

/**
 * @function normalizeDegrees
 * @static
 * Adjusts degrees less than 0 or greater than 360 to corresponding degrees between 0 and 360. 
 * This is useful when rotating an element by increments.
 * 
 * @param value Number
 * Degrees to normalize.
 * 
 * @returns Number
 * Degrees between 0 and 360.
 */	
CanvasElement.normalizeDegrees = 
	function (value)
	{
		while (value >= 360)
			value = value - 360;
		while (value < 0)
			value = value + 360;
		
		return value;
	};	

/**
 * @function roundToPrecision
 * @static
 * Rounds a number to specified precision (decimal points).
 * 
 * @param value Number
 * Number to round.
 * 
 * @param precision int
 * Number of decimal points.
 * 
 * @returns Number
 * Rounded value.
 */	
CanvasElement.roundToPrecision = 
	function (value, precision)
	{
		if (precision == 0)
			return Math.round(value);
		
		var multiplier = Math.pow(10, precision);
		
		value = value * multiplier;
		value = Math.round(value);
		return value / multiplier;
	};
	
/////////////CanvasElement Internal Static Functions//////////////////	
	
CanvasElement._browserType = "";	
	
//Map of maps for character widths by font size/style. Some browsers render canvas text by pixel rather 
//than character width. For example, an uppercase "T" with a lowercase "e" next to it ("Te"), 
//the "e" will actually render overlapping the "T" since the "e" is not tall enough to collide with the top of the "T". 
//This doesnt work for word processing, we need to be able to identify each character so we measure and 
//store character widths here, and render all text on a character by character basis for consistency.
CanvasElement._characterWidthMap = Object.create(null); 

CanvasElement._characterFillBitmapMap = Object.create(null);
CanvasElement._characterStrokeBitmapMap = Object.create(null);

CanvasElement._measureCharBitmap = null;
CanvasElement._measureCharContext = null;
(	function () 
	{
		CanvasElement._measureCharBitmap = document.createElement("canvas");
		CanvasElement._measureCharBitmap.width = 1;
		CanvasElement._measureCharBitmap.height = 1;
		
		CanvasElement._measureCharContext = CanvasElement._measureCharBitmap.getContext("2d");
	}
)();

/**
 * @function _measureText
 * @static
 * Measures text on a character by character basis. Unfortunately, browsers will give
 * different widths for strings of text, than text measured character by character. It appears
 * text width changes depending on which characters are next to which characters. This behavior
 * cannot be used for text that requires highlighting or editing. This function records the character
 * width per font in a map, and then uses that map to measure text widths. This surprisingly turns out to be 
 * just as fast as measuring full text via the canvas context (since canvas sucks so bad at text rendering).
 * 
 * @param text String
 * The text string to measure.
 * 
 * @param fontString String
 * Font styling to use when measuring. Use _getFontString()
 * 
 * @returns Number
 * Width of the text as measured via characters.
 */
CanvasElement._measureText = 
	function (text, fontString)
	{
		var charMap = CanvasElement._characterWidthMap[fontString];
		if (charMap == null)
		{
			charMap = Object.create(null);
			CanvasElement._characterWidthMap[fontString] = charMap;
		}
		
		var result = 0;
		var charWidth = 0;
		var fontSet = false;
		
		for (var i = 0; i < text.length; i++)
		{
			charWidth = charMap[text[i]];
			if (charWidth == null)
			{
				if (fontSet == false) 
				{
					CanvasElement._measureCharContext.font = fontString;
					fontSet = true;
				}
				
				charWidth = Math.ceil(CanvasElement._measureCharContext.measureText(text[i]).width);
				charMap[text[i]] = charWidth;
			}
			
			result += charWidth;
		}
		
		return result;
	};

/**
 * @function _fillText
 * @static
 * Renders text on a character by character basis. Unfortunately, browsers will give
 * different widths for strings of text, than text measured character by character. It appears
 * text width changes depending on which characters are next to which characters. This behavior
 * cannot be used for text that requires highlighting or editing. This function records character
 * bitmaps per font in a map, and then uses that map to render characters. This surprisingly turns out to be 
 * just as fast as rendering full text via the canvas context (since canvas sucks so bad at text rendering).
 * 
 * @param ctx Canvas2DContext
 * The canvas context to render the text. 
 * 
 * @param text String
 * The text string to render.
 * 
 * @param x Number
 * The X coordinate to render the text.
 * 
 * @param y Number
 * The Y coordinate to render the text.
 * 
 * @param fontString String
 * Font styling to use when measuring. Use _getFontString()
 * 
 * @param color String
 * Hex color value to be used to render the text. Format like "#FF0000" (red).
 * 
 * @param baseline String
 * Text Y position relative to Y coordinate. ("top", "middle", or "bottom")
 */	
CanvasElement._fillText = 
	function (ctx, text, x, y, fontString, color, baseline)
	{
		if (CanvasElement._browserType == "Firefox" || CanvasElement._browserType == "Chrome")
			y += 1;
	
		var bitmapMap = CanvasElement._characterFillBitmapMap[fontString];
		if (bitmapMap == null)
		{
			bitmapMap = Object.create(null);
			CanvasElement._characterFillBitmapMap[fontString] = bitmapMap;
		}
		
		var charWidth = 0;
		for (var i = 0; i < text.length; i++)
		{
			charWidth = CanvasElement._measureText(text[i], fontString);
			if (charWidth <= 0)
				continue;
			
			var bitmapAndContext = bitmapMap[text[i]];
			
			if (bitmapAndContext == null)
			{
				bitmapAndContext = {canvas:null, context:null, fontSize:0};
				bitmapMap[text[i]] = bitmapAndContext;
				
				bitmapAndContext.canvas = document.createElement("canvas");
				
				var fontSplit = fontString.split(" ");
				var fontSize = 0;
				for (var i2 = 0; i2 < fontSplit.length; i2++)
				{
					if (fontSplit[i2].length >= 3)
					{
						var pxString = fontSplit[i2].substr(fontSplit[i2].length - 2, 2);
						if (pxString == "px")
						{
							fontSize = Number(fontSplit[i2].substr(0, fontSplit[i2].length - 2));
							break;
						}
					}
				}
				
				bitmapAndContext.fontSize = fontSize;
				
				bitmapAndContext.canvas.height = fontSize + 4;
				bitmapAndContext.canvas.width = charWidth;
				
				bitmapAndContext.context = bitmapAndContext.canvas.getContext("2d");
				bitmapAndContext.context.font = fontString;
				bitmapAndContext.context.textBaseline = "middle";
				bitmapAndContext.context.textAlign = "left";
				bitmapAndContext.context.strokeStyle = "#000000";
				bitmapAndContext.context.fillStyle = "#000000";
				bitmapAndContext.context.fillText(text[i], 0, bitmapAndContext.canvas.height / 2);
				
				bitmapAndContext.context.globalCompositeOperation = "source-atop";
			}
			
			if (bitmapAndContext.context.fillStyle != color) 
			{
				bitmapAndContext.context.fillStyle = color;
				
				bitmapAndContext.context.beginPath();
				bitmapAndContext.context.moveTo(0, 0);
				bitmapAndContext.context.lineTo(bitmapAndContext.canvas.width, 0);
				bitmapAndContext.context.lineTo(bitmapAndContext.canvas.width, bitmapAndContext.canvas.height);
				bitmapAndContext.context.lineTo(0, bitmapAndContext.canvas.height);
				bitmapAndContext.context.closePath();
				
				bitmapAndContext.context.fill();
			}
			
			if (baseline == "top")
				ctx.drawImage(bitmapAndContext.canvas, x, Math.round(y - ((bitmapAndContext.canvas.height - bitmapAndContext.fontSize) / 2)));
			else if (baseline == "bottom")
				ctx.drawImage(bitmapAndContext.canvas, x, Math.round(y - (((bitmapAndContext.canvas.height - bitmapAndContext.fontSize) / 2) + bitmapAndContext.fontSize)));
			else //	"middle"
				ctx.drawImage(bitmapAndContext.canvas, x, Math.round(y - (bitmapAndContext.canvas.height / 2)));
			
			if (text.length == 1)
				return;
			
			x += charWidth;
		}
	};	
	
/**
 * @function _strokeText
 * @static
 * Renders text on a character by character basis. Unfortunately, browsers will give
 * different widths for strings of text, than text measured character by character. It appears
 * text width changes depending on which characters are next to which characters. This behavior
 * cannot be used for text that requires highlighting or editing. This function records character
 * bitmaps per font in a map, and then uses that map to render characters. This surprisingly turns out to be 
 * just as fast as rendering full text via the canvas context (since canvas sucks so bad at text rendering).
 * 
 * @param ctx Canvas2DContext
 * The canvas context to render the text. 
 * 
 * @param text String
 * The text string to render.
 * 
 * @param x Number
 * The X coordinate to render the text (Upper left).
 * 
 * @param y Number
 * The Y coordinate to render the text (Uppder left).
 * 
 * @param fontString String
 * Font styling to use when measuring. Use _getFontString()
 * 
 * @param color String
 * Hex color value to be used to render the text. Format like "#FF0000" (red).
 * 
 * @param baseline String
 * Text Y position relative to Y coordinate. ("top", "middle", or "bottom")
 */	
CanvasElement._strokeText = 
	function (ctx, text, x, y, fontString, color, baseline)
	{
		//Firefox weirdly renders text higher than normal
		if (CanvasElement._browserType == "Firefox")
			y += 2;
	
		var bitmapMap = CanvasElement._characterStrokeBitmapMap[fontString];
		if (bitmapMap == null)
		{
			bitmapMap = Object.create(null);
			CanvasElement._characterStrokeBitmapMap[fontString] = bitmapMap;
		}
		
		var charWidth = 0;
		for (var i = 0; i < text.length; i++)
		{
			charWidth = CanvasElement._measureText(text[i], fontString);
			if (charWidth <= 0)
				continue;
			
			var bitmapAndContext = bitmapMap[text[i]];
			
			if (bitmapAndContext == null)
			{
				bitmapAndContext = {canvas:null, context:null};
				bitmapMap[text[i]] = bitmapAndContext;
				
				bitmapAndContext.canvas = document.createElement("canvas");
				
				var fontSplit = fontString.split(" ");
				var fontSize = 0;
				for (var i2 = 0; i2 < fontSplit.length; i2++)
				{
					if (fontSplit[i2].length >= 3)
					{
						var pxString = fontSplit[i2].substr(fontSplit[i2].length - 2, 2);
						if (pxString == "px")
						{
							fontSize = Number(fontSplit[i2].substr(0, fontSplit[i2].length - 2));
							break;
						}
					}
						
				}
				
				bitmapAndContext.fontSize = fontSize;
				
				bitmapAndContext.canvas.height = fontSize + 4;
				bitmapAndContext.canvas.width = charWidth;
				
				bitmapAndContext.context = bitmapAndContext.canvas.getContext("2d");
				bitmapAndContext.context.font = fontString;
				bitmapAndContext.context.textBaseline = "middle";
				bitmapAndContext.context.textAlign = "left";
				bitmapAndContext.context.strokeStyle = "#000000";
				bitmapAndContext.context.fillStyle = "#000000";
				bitmapAndContext.context.strokeText(text[i], 0, bitmapAndContext.canvas.height / 2);
				
				bitmapAndContext.context.globalCompositeOperation = "source-atop";
			}
			
			if (bitmapAndContext.context.fillStyle != color) 
			{
				bitmapAndContext.context.fillStyle = color;
				
				bitmapAndContext.context.beginPath();
				bitmapAndContext.context.moveTo(0, 0);
				bitmapAndContext.context.lineTo(bitmapAndContext.canvas.width, 0);
				bitmapAndContext.context.lineTo(bitmapAndContext.canvas.width, bitmapAndContext.canvas.height);
				bitmapAndContext.context.lineTo(0, bitmapAndContext.canvas.height);
				bitmapAndContext.context.closePath();
				
				bitmapAndContext.context.fill();
			}
			
			if (baseline == "top")
				ctx.drawImage(bitmapAndContext.canvas, x, Math.round(y - ((bitmapAndContext.canvas.height - bitmapAndContext.fontSize) / 2)));
			else if (baseline == "bottom")
				ctx.drawImage(bitmapAndContext.canvas, x, Math.round(y - (((bitmapAndContext.canvas.height - bitmapAndContext.fontSize) / 2) + bitmapAndContext.fontSize)));
			else //	"middle"
				ctx.drawImage(bitmapAndContext.canvas, x, Math.round(y - (bitmapAndContext.canvas.height / 2)));
			
			if (text.length == 1)
				return;
			
			x += charWidth;
		}
	};		
	
/**
 * @function _calculateMinMaxPercentSizes
 * @static
 * Used to calculate size in pixels that percent sized elements should consume given
 * a supplied size in pixels. Populates .actualSize field on objects in supplied 
 * percentSizedObjects array. This function automatically rounds all sizes to the
 * nearest pixel to prevent anti-aliasing and fuzzy lines.
 * 
 * @param percentSizedObjects Array
 * Array of objects containing size data: {minSize:Number, maxSize:Number, percentSize:Number}
 * 
 * @param size Number
 * Available size in pixels.
 */
CanvasElement._calculateMinMaxPercentSizes = 
	function (percentSizedObjects, size)
	{
		if (percentSizedObjects.length == 0)
			return;
	
		var percentObjects = percentSizedObjects.slice();
		var availableSize = size;
		var totalPercentUsed = 0;
		var i;
		
		//Fix values, record total percent used.
		for (i = 0; i < percentObjects.length; i++)
		{
			if (percentObjects[i].minSize == null)
				percentObjects[i].minSize = 0;
			if (percentObjects[i].maxSize == null)
				percentObjects[i].maxSize = Number.MAX_VALUE;
			if (percentObjects[i].percentSize == null)
				percentObjects[i].percentSize = 100;
			
			totalPercentUsed += percentObjects[i].percentSize;
		}
		
		//Size all percent sized elements.
		var done = false;
		while (done == false)
		{
			var size = 0;
			done = true;
			
			for (i = 0; i < percentObjects.length; i++)
			{
				size = availableSize * (percentObjects[i].percentSize / totalPercentUsed);
				if (size > percentObjects[i].maxSize)
				{
					percentObjects[i].actualSize = percentObjects[i].maxSize;
					totalPercentUsed -= percentObjects[i].percentSize;
					availableSize -= percentObjects[i].maxSize;
					
					percentObjects.splice(i, 1);
					done = false;
					break;
				}
				else if (size < percentObjects[i].minSize)
				{
					percentObjects[i].actualSize = percentObjects[i].minSize;
					totalPercentUsed -= percentObjects[i].percentSize;
					availableSize -= percentObjects[i].minSize;
					
					percentObjects.splice(i, 1);
					done = false;
					break;
				}
				else
					percentObjects[i].actualSize = Math.floor(size);
			}
		}
		
		for (i = 0; i < percentObjects.length; i++)
			availableSize -= percentObjects[i].actualSize;
		
		//Distribute excess pixels (rounding error)
		while (availableSize >= 1 && percentObjects.length > 0)
		{
			for (i = 0; i < percentObjects.length; i++)
			{
				while (percentObjects[i].actualSize + 1 > percentObjects[i].maxSize)
				{
					percentObjects.splice(i, 1);
					if (i == percentObjects.length)
						break;
				}
				
				if (i == percentObjects.length)
					break;
				
				percentObjects[i].actualSize++;
				availableSize--;
				
				if (availableSize <= 0)
					break;
			}
		}
	};
	
///////////////CanvasElement Internal Functions////////////////////////////////////

//@private	
CanvasElement.prototype._onBackgroundShapeStyleChanged = 
	function (styleChangedEvent)
	{
		this._invalidateRender();
	};
	
//@private
CanvasElement.prototype._onBackgroundFillStyleChanged = 
	function (styleChangedEvent)
	{
		this._invalidateRender();
	};
	
/**
 * @function _addStyleDefinitionAt
 * Inserts a style definition to this elements style definition or default definition lists.
 * Adding style definitions to elements already attached to the display chain is expensive, 
 * for better performance add definitions before attaching the element via addElement().
 * Default definitions are used when styling sub components with sub styles. 
 * 
 * @param styleDefinition StyleDefinition
 * StyleDefinition to be added to this elements definition list.
 * 
 * @param index int
 * The index to insert the style definition within the elements definition list.
 * 
 * @param isDefault bool
 * When true, inserts the definition into the element's default definition list.
 * 
 * @returns StyleDefinition
 * Returns StyleDefinition just added when successfull, null if the StyleDefinition could not
 * be added due to the index being out of range or other error.
 */		
CanvasElement.prototype._addStyleDefinitionAt = 
	function (styleDefinition, index, isDefault)
	{
		if (!(styleDefinition instanceof StyleDefinition))
			return null;
	
		//Get the appropriate style definition storage array.
		var styleDefArray;
		if (isDefault == true)
			styleDefArray = this._styleDefinitionDefaults;
		else
			styleDefArray = this._styleDefinitions;
		
		if (index < 0 || index > styleDefArray)
			return null;
		
		styleDefArray.splice(index, 0, styleDefinition);
		
		if (this._manager != null) //Attached to display chain
		{
			styleDefinition.addEventListener("stylechanged", this._onExternalStyleChangedInstance);
			
			//_onExternalStyleChanged() is expensive! We use the map to make sure we only do each style once.
			var styleNamesMap = Object.create(null);
			var styleName = null;
			
			//We're shifting the priority of all existing style definitions with a lower index (previously added) 
			//when we add a new one, so we need to invoke a style change on all associated styles.
			
			//Record relevant style names
			for (var i = index; i >= 0; i--)
			{
				for (styleName in styleDefArray[i]._styleMap)
					styleNamesMap[styleName] = true;
			}
			
			//Spoof style changed events for normal handling.
			for (styleName in styleNamesMap)
				this._onExternalStyleChanged(new StyleChangedEvent(styleName));
		}
		
		return styleDefinition;
	};	

/**
 * @function _removeStyleDefinitionAt
 * Removes a style definition from this elements style definition or default definitions at the supplied index.
 * Default definitions are used when styling sub components with sub styles. 
 * 
 * @param index int
 * Index to be removed.
 * 
 * @param isDefault bool
 * When true, removes the definition from the element's default definition list.
 * 
 * @returns StyleDefinition
 * Returns the StyleDefinition just removed if successfull, null if the definition could
 * not be removed due it it not being in this elements definition list, or index out of range.
 */			
CanvasElement.prototype._removeStyleDefinitionAt = 
	function (index, isDefault)
	{
		//Get the appropriate style definition storage array.
		var styleDefArray;
		if (isDefault == true)
			styleDefArray = this._styleDefinitionDefaults;
		else
			styleDefArray = this._styleDefinitions;
	
		if (index < 0 || index > styleDefArray - 1)
			return null;
		
		var styleDefinition = null;
		
		if (this._manager != null) //Attached to display chain
		{
			//_onExternalStyleChanged() is expensive! We use the map to make sure we only do each style once.
			var styleNamesMap = Object.create(null);
			var styleName = null;
			
			//We're shifting the priority of all existing style definitions with a lower index (previously added) 
			//when we add a new one, so we need to invoke a style change on all associated styles.
			
			//Record relevant styles
			for (var i = index; i >= 0; i--)
			{
				for (styleName in styleDefArray[i]._styleMap)
					styleNamesMap[styleName] = true;
			}
			
			//Remove definition
			styleDefinition = styleDefArray.splice(index, 1)[0]; //Returns array of removed items.
			styleDefinition.removeEventListener("stylechanged", this._onExternalStyleChangedInstance);
			
			//Spoof style changed event for relevant styles.
			for (styleName in styleNamesMap)
				this._onExternalStyleChanged(new StyleChangedEvent(styleName));
		}
		else //Not attached, just remove the definition
			styleDefinition = styleDefArray.splice(index, 1)[0]; //Returns array of removed items.
		
		return styleDefinition;
	};	

/**
 * @function _setStyleDefinitions
 * Replaces the elements current style definition or default definition lists. 
 * This is more effecient than removing or adding style definitions one at a time.
 * Default definitions are used when styling sub components with sub styles. 
 * 
 * @param styleDefinitions StyleDefinition
 * May be a StyleDefinition, or an Array of StyleDefinition
 * 
 * @param isDefault bool
 * When true, replaces the default definition list.
 * 
 * @param styleNamesChangedMap Object
 * Optional - A empty map object - Object.create(null) to populate style changes
 * due to swapping definitions while this element is attached to the display chain. 
 * When specified (not null), does not automatically invoke style changed events.
 */	
CanvasElement.prototype._setStyleDefinitions = 
	function (styleDefinitions, isDefault, styleNamesChangedMap)
	{
		var i;
		
		if (styleDefinitions == null)
			styleDefinitions = [];
		
		if (Array.isArray(styleDefinitions) == false)
			styleDefinitions = [styleDefinitions];

		//Flatten nested arrays
		var stack = styleDefinitions.slice();
		var next = null;
		var flattened = [];
		while (stack.length > 0)
		{
			next = stack.pop();
			if (Array.isArray(next) == true)
			{
				for (i = 0; i < next.length; i++)
					stack.push(next[i]);
			}
			else
				flattened.push(next);
		}
		
		flattened.reverse();
		styleDefinitions = flattened;
		
		//Remove any null definitions
		for (i = styleDefinitions.length - 1; i >= 0; i--)
		{
			if (styleDefinitions[i] == null)
				styleDefinitions.splice(i, 1);
		}
		
		//Get the appropriate style definition storage array.
		var styleDefArray;
		if (isDefault == true)
			styleDefArray = this._styleDefinitionDefaults;
		else
			styleDefArray = this._styleDefinitions;
		
		if (this._manager != null) //Attached to display chain
		{
			var oldIndex = styleDefArray.length - 1;
			var newIndex = styleDefinitions.length - 1;
			
			var styleName = null;
			var changed = false;
			
			var styleNamesMap = styleNamesChangedMap;
			if (styleNamesMap == null)
				styleNamesMap = Object.create(null);
			
			//Compare styles from the ends of the arrays
			while (oldIndex >= 0 || newIndex >= 0)
			{
				//Detect change
				if (changed == false && (oldIndex < 0 || newIndex < 0 || styleDefinitions[newIndex] != styleDefArray[oldIndex]))
					changed = true;
				
				//Change detected
				if (changed == true)
				{
					if (oldIndex >= 0)
					{
						//Record removed style names
						for (styleName in styleDefArray[oldIndex]._styleMap)
							styleNamesMap[styleName] = true;
					}
					if (newIndex >= 0)
					{
						//Record removed style names
						for (styleName in styleDefinitions[newIndex]._styleMap)
							styleNamesMap[styleName] = true;
					}
				}
				
				oldIndex--;
				newIndex--;
			}
			
			//Bail no changes
			if (changed == false)
				return;
			
			//Clear the definition list
			while (styleDefArray.length > 0)
			{
				styleDefArray[styleDefArray.length - 1].removeEventListener("stylechanged", this._onExternalStyleChangedInstance);
				styleDefArray.splice(styleDefArray.length - 1, 1);
			}
			
			//Add the new definitions.
			for (i = 0; i < styleDefinitions.length; i++)
			{
				styleDefinitions[i].addEventListener("stylechanged", this._onExternalStyleChangedInstance);
				styleDefArray.push(styleDefinitions[i]);
			}
			
			//Spoof style changed events for normal style changed handling.
			if (styleNamesChangedMap == null)
			{
				for (styleName in styleNamesMap)
					this._onExternalStyleChanged(new StyleChangedEvent(styleName));
			}
		}
		else //Not attached to display chain, just swap the definitions
		{
			//Clear the definition list
			styleDefArray.splice(0, styleDefArray.length);
			
			//Add the new definitions.
			for (i = 0; i < styleDefinitions.length; i++)
				styleDefArray.push(styleDefinitions[i]);
		}
	};	

/**
 * @function _getStyleDefinitionAt
 * Gets the style definition or default definition at the supplied zero base index.
 * 
 * @param index int
 * Index of the style definition to return;
 * 
 * @param isDefault bool
 * When true, returns the default definition at the supplied index.
 * 
 * @returns StyleDefinition
 * The style defenition at the supplied index, or null if index is out of range. 
 */		
CanvasElement.prototype._getStyleDefinitionAt = 
	function (index, isDefault)
	{
		//Get the appropriate style definition storage array.
		var styleDefArray;
		if (isDefault == true)
			styleDefArray = this._styleDefinitionDefaults;
		else
			styleDefArray = this._styleDefinitions;
	
		if (index < 0 || index >= styleDefArray.length)
			return null;
		
		return styleDefArray[index];
	};		
	
//@private	
CanvasElement.prototype._onExternalStyleChanged = 
	function (styleChangedEvent)
	{
		//Not attached to display chain, bail.
		if (this._manager == null)
			return;
		
		var isProxy = false;
		var isParent = false;
		var validStyle = false;
		var styleName = styleChangedEvent.getStyleName();	
		var styleType = this._getStyleType(styleName);
		
		if (this._styleProxy != null && styleChangedEvent.getTarget() == this._styleProxy._proxyElement)
			isProxy = true;
		if (this._parent != null && styleChangedEvent.getTarget() == this._parent)
			isParent = true;
		
		if (isProxy == true || isParent == true)
		{
			//If coming from proxy, we cannot be a substyle, and style name must be in proxy map, or _Arbitrary specified and not in proxy map.
			//If coming from parent, style must be inheritable.
			if ((isProxy == true && styleType != StyleableBase.EStyleType.SUBSTYLE && 
				(styleName in this._styleProxy._proxyMap == true || ("_Arbitrary" in this._styleProxy._proxyMap == true && this._styleProxy._proxyElement._getStyleType(styleName) == null))) ||
				(isParent == true && styleType == StyleableBase.EStyleType.INHERITABLE))
			{
				validStyle = true;
			}
			else
				validStyle = false;
		}
		else
			validStyle = true;
		
		//Style we dont care about, bail.
		if (validStyle == false)
			return;
		
		//Get the cache for this style.
		var styleCache = this._stylesCache[styleName];
		
		//Create cache if doesnt exist.
		if (styleCache == null)
		{
			styleCache = {styleData:new StyleData(styleName), cacheInvalid:true};
			this._stylesCache[styleName] = styleCache;
		}
		
		//Check if the StyleData changed (if we're not a sub style)
		if (styleType != StyleableBase.EStyleType.SUBSTYLE)
		{
			var oldStyleData = null;
			var newStyleData = null;
			
			//Cache valid, copy it for later compare.
			if (styleCache.cacheInvalid == false)
				oldStyleData = styleCache.styleData.clone();
			
			//Invalidate the cache
			styleCache.cacheInvalid = true;
			
			//Get updated data.
			newStyleData = this.getStyleData(styleName);
			
			//No change, bail.
			if (oldStyleData != null && oldStyleData.equals(newStyleData) == true)
				return;
		}
		else //Sub styles always invalidate (the entire chain is used)
		{
			//Invalidate the cache
			styleCache.cacheInvalid = true;
			
			//Update data
			this.getStyleData(styleName);
		}
		
		if (styleType != null)
			this._invalidateStyle(styleName);
		
		//Re-dispatch from ourself.
		this.dispatchEvent(styleChangedEvent); 
	};

/**
 * @function _getStyleList
 * 
 * Gets the style values for the supplied style name in this elements
 * style definition list and instance styles for the supplied style name. 
 * This is used for sub styles as all stub styles in the list are applied to sub components.
 * This list should be supplied to the sub components style definition list.
 *  
 * @param styleName String
 * String representing the style list to return.
 * 
 * @returns Array
 * Returns an array of all styles in this elements definition list and instance styles
 * for the associated style name. 
 */	
CanvasElement.prototype._getStyleList = 
	function (styleName)
	{
		var styleList = [];
		var styleValue = null;
		var i;
		
		//Add definitions
		for (var i = 0; i < this._styleDefinitions.length; i++)
		{
			styleValue = this._styleDefinitions[i].getStyle(styleName);
			
			if (styleValue !== undefined)
				styleList.push(styleValue);
		}
		
		//Add instance
		if (styleName in this._styleMap)
			styleList.push(this._styleMap[styleName]);
		
		return styleList;
	};
	
/**
 * @function _getDefaultStyleList
 * 
 * Gets the style values for the supplied style name in this elements
 * default style definition list and class list styles for the supplied style name. 
 * This is used for sub styles as all stub styles in the list are applied to sub components.
 * This list should be supplied to the sub components default style definition list.
 *  
 * @param styleName String
 * String representing the default style list to return.
 * 
 * @returns Array
 * Returns an array of all styles in this elements default definition and class list styles 
 * for the associated style name. 
 */	
CanvasElement.prototype._getDefaultStyleList = 
	function (styleName)
	{
		var i;
		var styleValue = null;
		
		//Get class list
		var styleList = this._getClassStyleList(styleName);
		
		//Check default definitions
		for (var i = 0; i < this._styleDefinitionDefaults.length; i++)
		{
			styleValue = this._styleDefinitionDefaults[i].getStyle(styleName);
			
			if (styleValue !== undefined)
				styleList.push(styleValue);
		}
		
		return styleList;
	};
	
/**
 * @function _applySubStylesToElement
 * @override
 * 
 * Convienence function for setting sub styles of sub components.
 * Applies appropriate sub styling from this element to the 
 * supplied elements definition and default definition style lists.
 * You should call this on sub components from within the _doStylesUpdated()
 * function when the associated sub style changes.
 *  
 * @param styleName String
 * String representing the sub style to apply.
 * 
 * @param elementToApply CanvasElement
 * The sub component element to apply sub styles.
 */		
CanvasElement.prototype._applySubStylesToElement = 
	function (styleName, elementToApply)
	{
		var changedStyleName = null;
		var styleNamesChangedMap = Object.create(null);
	
		elementToApply._setStyleDefinitions(this._getStyleList(styleName), false, styleNamesChangedMap);
		elementToApply._setStyleDefinitions(this._getDefaultStyleList(styleName), true, styleNamesChangedMap);
		
		//Spoof style changed events for normal style changed handling.
		for (changedStyleName in styleNamesChangedMap)
			elementToApply._onExternalStyleChanged(new StyleChangedEvent(changedStyleName));
	};
	
/**
 * @function _setStyleProxy
 * 
 * Sets the element which is to proxy styles to this element. See getStyle() and StyleProxy.
 * This should be set prior to added this element to the display hierarchy via addElement() or _addChild().
 * 
 * @param styleProxy StyleProxy
 * The StyleProxy element wrapper to use to proxy styles from the proxy element to this element.
 * 
 * @seealso StyleProxy
 */	
CanvasElement.prototype._setStyleProxy = 
	function (styleProxy)
	{
		this._styleProxy = styleProxy;
	};
	
/**
 * @function _onCanvasElementAdded
 * Invoked when the element is added to the canvas. Every CanvasElement already adds its own
 * "added" event listener so overriding this is identical but more efficient than adding your own "added" event listener.
 * You should *always* call the base class function.
 * 
 * @param addedRemovedEvent AddedRemovedEvent
 * The AddedRemovedEvent to process.
 */	
CanvasElement.prototype._onCanvasElementAdded = 
	function (addedRemovedEvent)
	{
		/////////Added to the Display Chain/////////////
	
		var i;
	
		for (i = 0; i < this._styleDefinitions.length; i++)
		{
			if (this._styleDefinitions[i].hasEventListener("stylechanged", this._onExternalStyleChangedInstance) == false)
				this._styleDefinitions[i].addEventListener("stylechanged", this._onExternalStyleChangedInstance);
		}

		//If proxy is our parent, we dont want duplicate listeners.
		if (this._styleProxy != null && this._styleProxy._proxyElement.hasEventListener("stylechanged", this._onExternalStyleChangedInstance) == false)
			this._styleProxy._proxyElement.addEventListener("stylechanged", this._onExternalStyleChangedInstance);
		
		if (this._backgroundShape != null && this._backgroundShape.hasEventListener("stylechanged", this._onBackgroundShapeStyleChangedInstance) == false)
			this._backgroundShape.addEventListener("stylechanged", this._onBackgroundShapeStyleChangedInstance);
		
		if (this._backgroundFill != null && this._backgroundFill.hasEventListener("stylechanged", this._onBackgroundFillStyleChangedInstance) == false)
			this._backgroundFill.addEventListener("stylechanged", this._onBackgroundFillStyleChangedInstance);
		
		for (i = 0; i < this._styleDefinitionDefaults.length; i++)
		{
			if (this._styleDefinitionDefaults[i].hasEventListener("stylechanged", this._onExternalStyleChangedInstance) == false)
				this._styleDefinitionDefaults[i].addEventListener("stylechanged", this._onExternalStyleChangedInstance);
		}
		
		//Add broadcast events to manager//
		if ("enterframe" in this._eventListeners && this._eventListeners["enterframe"].length > 0)
		{
			for (i = 0; i < this._eventListeners["enterframe"].length; i++)
				addedRemovedEvent.getManager()._broadcastDispatcher.addEventListener("enterframe", this._eventListeners["enterframe"][i]);
		}
		if ("exitframe" in this._eventListeners && this._eventListeners["exitframe"].length > 0)
		{
			for (i = 0; i < this._eventListeners["exitframe"].length; i++)
				addedRemovedEvent.getManager()._broadcastDispatcher.addEventListener("exitframe", this._eventListeners["exitframe"][i]);
		}
		if ("localechanged" in this._eventListeners && this._eventListeners["localechanged"].length > 0)
		{
			for (i = 0; i < this._eventListeners["localechanged"].length; i++)
				addedRemovedEvent.getManager()._broadcastDispatcher.addEventListener("localechanged", this._eventListeners["localechanged"][i]);
		}
		if ("mousemoveex" in this._eventListeners && this._eventListeners["mousemoveex"].length > 0)
		{
			for (i = 0; i < this._eventListeners["mousemoveex"].length; i++)
				addedRemovedEvent.getManager()._broadcastDispatcher.addEventListener("mousemoveex", this._eventListeners["mousemoveex"][i]);
		}
		
		//Invalidate redraw and composite render
		this._invalidateRedrawRegion();
		this._invalidateCompositeRender();
		
		///////////Invalidate All Styles////////////////
		
		//Invalidate all cache
		for (var prop in this._stylesCache)
			this._stylesCache[prop].cacheInvalid = true;
		
		//Invalidate *all* styles, don't need to propagate, display propagates when attaching.
		this._flattenStyleTypes();
		for (i = 0; i < this.constructor.__StyleTypesFlatArray.length; i++)
			this._invalidateStyle(this.constructor.__StyleTypesFlatArray[i].styleName);
		
		//Always dispatch when added.
		if (this.hasEventListener("localechanged", null) == true)
			this.dispatchEvent(new DispatcherEvent("localechanged"));
	};

/**
 * @function _onCanvasElementRemoved
 * Invoked when the element is removed to the canvas. Every CanvasElement already adds its own
 * "removed" event listener so overriding this is identical but more efficient than adding your own "removed" event listener.
 * You should *always* call the base class function.
 * 
 * @param addedRemovedEvent AddedRemovedEvent
 * The AddedRemovedEvent to process.
 */		
CanvasElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		///////Removed from display chain///////////////////
	
		var i = 0;
	
		for (i = 0; i < this._styleDefinitions.length; i++)
		{
			if (this._styleDefinitions[i].hasEventListener("stylechanged", this._onExternalStyleChangedInstance) == true)
				this._styleDefinitions[i].removeEventListener("stylechanged", this._onExternalStyleChangedInstance);
		}
		
		if (this._styleProxy != null && this._styleProxy._proxyElement.hasEventListener("stylechanged", this._onExternalStyleChangedInstance) == true)
			this._styleProxy._proxyElement.removeEventListener("stylechanged", this._onExternalStyleChangedInstance);
		
		if (this._backgroundShape != null && this._backgroundShape.hasEventListener("stylechanged", this._onBackgroundShapeStyleChangedInstance) == true)
			this._backgroundShape.removeEventListener("stylechanged", this._onBackgroundShapeStyleChangedInstance);
		
		if (this._backgroundFill != null && this._backgroundFill.hasEventListener("stylechanged", this._onBackgroundFillStyleChangedInstance) == true)
			this._backgroundFill.removeEventListener("stylechanged", this._onBackgroundFillStyleChangedInstance);
		
		for (i = 0; i < this._styleDefinitionDefaults.length; i++)
		{
			if (this._styleDefinitionDefaults[i].hasEventListener("stylechanged", this._onExternalStyleChangedInstance) == true)
				this._styleDefinitionDefaults[i].removeEventListener("stylechanged", this._onExternalStyleChangedInstance);
		}
		
		if (this._rollOverCursorInstance != null)
		{
			addedRemovedEvent.getManager().removeCursor(this._rollOverCursorInstance);
			this._rollOverCursorInstance = null;
		}
		
		//Update the redraw region of any composite parents still attached to the display chain.
		for (i = 0; i < this._compositeMetrics.length; i++)
		{
			if (this._compositeMetrics[i].element._manager != null)
				this._compositeMetrics[i].element._updateRedrawRegion(this._compositeMetrics[i].drawableMetrics);
		}
		
		//Reset cycle flags
		this._stylesInvalid = true;
		this._measureInvalid = true;
		this._layoutInvalid = true;
		this._renderInvalid = true;
		this._redrawRegionInvalid = true;
		
		//Nuke graphics canvas
		this._graphicsCanvas = null;
		this._graphicsCtx = null;
		this._graphicsClear = true;					
		
		//Nuke composite canvas
		this._compositeCtx = null;																														
		this._compositeCanvas = null;																
		this._compositeCanvasMetrics = null;			 
		
		//Reset redraw flags
		this._renderChanged = true;					
		this._renderVisible = false; 	
		this._forceRegionUpdate = false;
		this._compositeEffectChanged = true;
		
		//Nuke composite data
		this._compositeMetrics.length = 0;
		this._compositeVisibleMetrics = null;																						
		this._redrawRegionMetrics = null;																												
		this._transformVisibleMetrics = null;			
		this._transformDrawableMetrics = null;				
		
		//Remove broadcast events from manager//
		if ("enterframe" in this._eventListeners && this._eventListeners["enterframe"].length > 0)
		{
			for (i = 0; i < this._eventListeners["enterframe"].length; i++)
				addedRemovedEvent.getManager()._broadcastDispatcher.removeEventListener("enterframe", this._eventListeners["enterframe"][i]);
		}
		if ("exitframe" in this._eventListeners && this._eventListeners["exitframe"].length > 0)
		{
			for (i = 0; i < this._eventListeners["exitframe"].length; i++)
				addedRemovedEvent.getManager()._broadcastDispatcher.removeEventListener("exitframe", this._eventListeners["exitframe"][i]);
		}
		if ("localechanged" in this._eventListeners && this._eventListeners["localechanged"].length > 0)
		{
			for (i = 0; i < this._eventListeners["localechanged"].length; i++)
				addedRemovedEvent.getManager()._broadcastDispatcher.removeEventListener("localechanged", this._eventListeners["localechanged"][i]);
		}
		if ("mousemoveex" in this._eventListeners && this._eventListeners["mousemoveex"].length > 0)
		{
			for (i = 0; i < this._eventListeners["mousemoveex"].length; i++)
				addedRemovedEvent.getManager()._broadcastDispatcher.removeEventListener("mousemoveex", this._eventListeners["mousemoveex"][i]);
		}
	};	
	
/**
 * @function _getFontString
 * Gets a font string that can be applied to the canvas's Context2D via the element's text styles.
 * This is just a helper to gather and format the styles for the canvas context.
 * 
 * @returns String
 * String to be applied to the canvas contex's font. "bold 14px Arial".
 */	
CanvasElement.prototype._getFontString = 
	function ()
	{
		return this.getStyle("TextStyle") + " " + this.getStyle("TextSize") + "px " + this.getStyle("TextFont");
	};		
	
//@Override
CanvasElement.prototype.dispatchEvent = 
	function (dispatchEvent)
	{
		if (!(dispatchEvent instanceof ElementEvent))
		{
			CanvasElement.base.prototype.dispatchEvent.call(this, dispatchEvent);
			return;
		}
	
		dispatchEvent._canceled = false;
		dispatchEvent._defaultPrevented = false;
		
		//We're transforming the event as we bubble. We shouldn't change the instance given to the dispatcher. 
		var event = dispatchEvent.clone();
	
		event._target = this;
		
		//Clone the event when calling the handlers so they cannot fudge the event data.
		var handlerEvent = null;
		
		if (event._bubbles == true)
		{
			var currentElement = this;
			var currentMousePoint = {x:0, y:0};
			if (event instanceof ElementMouseEvent)
			{
				currentMousePoint.x = event._x;
				currentMousePoint.y = event._y;
			}
			
			//Get parent chain.
			var parentChain = [];
			while (currentElement != null)
			{
				parentChain.push({element:currentElement, 
								x:currentMousePoint.x, 
								y:currentMousePoint.y});
				
				//Adjust mouse point for parent.
				if (event instanceof ElementMouseEvent)
				{
					currentMousePoint.x += currentElement._x;
					currentMousePoint.y += currentElement._y;
					
					currentElement.rotatePoint(currentMousePoint, false);
				}
				
				currentElement = currentElement._parent;
			}
			
			//Dispatch Capture Events.
			event._phase = "capture";
			for (var i = parentChain.length -1; i >= 0; i--)
			{
				currentElement = parentChain[i].element;
				
				if (event._type in currentElement._captureListeners && currentElement._captureListeners[event._type].length > 0)
				{
					event._currentTarget = currentElement;
					if (event instanceof ElementMouseEvent)
					{
						event._x = parentChain[i].x;
						event._y = parentChain[i].y;
					}
					
					//Copy the list of event handlers, if event handlers add/remove other handlers or themselves, 
					//we dont want to miss an event, or inconsistently dispatch newly added events.
					var listeners = currentElement._captureListeners[event._type].slice();
					
					//TODO: Sort by priority (no priority available yet).
					
					for (var i2 = 0; i2 < listeners.length; i2++)
					{
						handlerEvent = event.clone(); //Clone the event so the handler can't fudge our event data.
						listeners[i2](handlerEvent);
						
						if (handlerEvent._defaultPrevented == true)
						{
							dispatchEvent._defaultPrevented = true;
							event._defaultPrevented = true;
						}
						
						if (handlerEvent._canceled == true)
						{
							dispatchEvent._canceled = true;
							return;
						}
					}
				}
			}
			
			//Dispatch Bubble Events.
			event._phase = "bubble";
			for (var i = 0; i < parentChain.length; i++)
			{
				currentElement = parentChain[i].element;
				
				if (event._type in currentElement._eventListeners && currentElement._eventListeners[event._type].length > 0)
				{
					event._currentTarget = currentElement;
					if (event instanceof ElementMouseEvent)
					{
						event._x = parentChain[i].x;
						event._y = parentChain[i].y;
					}
					
					//Copy the list of event handlers, if event handlers add/remove other handlers or themselves, 
					//we dont want to miss an event, or inconsistently dispatch newly added events.
					var listeners = currentElement._eventListeners[event._type].slice();
					
					//TODO: Sort by priority (no priority available yet).
					
					for (var i2 = 0; i2 < listeners.length; i2++)
					{
						handlerEvent = event.clone(); //Clone the event so the handler can't fudge our event data.
						listeners[i2](handlerEvent);
						
						if (handlerEvent._defaultPrevented == true)
						{
							dispatchEvent._defaultPrevented = true;
							event._defaultPrevented = true;
						}
						
						if (handlerEvent._canceled == true)
						{
							dispatchEvent._canceled = true;
							return;
						}
					}
				}
			}
			
		}
		else //Dispatch only target events.
		{ 
			event._currentTarget = this;

			event._phase = "capture";
			if (event._type in this._captureListeners && this._captureListeners[event._type].length > 0)
			{
				//Copy the list of event handlers, if event handlers add/remove other handlers or themselves, 
				//we dont want to miss an event, or inconsistently dispatch newly added events.
				var listeners = this._captureListeners[event._type].slice();
				
				//TODO: Sort by priority (no priority available yet).
				
				for (var i2 = 0; i2 < listeners.length; i2++)
				{
					handlerEvent = event.clone(); //Clone the event so the handler can't fudge our event data.
					listeners[i2](handlerEvent);
					
					if (handlerEvent._defaultPrevented == true)
					{
						dispatchEvent._defaultPrevented = true;
						event._defaultPrevented = true;
					}
					
					if (handlerEvent._canceled == true)
					{
						dispatchEvent._canceled = true;
						return;
					}
				}
			}
			
			event._phase = "bubble";
			if (event._type in this._eventListeners && this._eventListeners[event._type].length > 0)
			{
				//Copy the list of event handlers, if event handlers add/remove other handlers or themselves, 
				//we dont want to miss an event, or inconsistently dispatch newly added events.
				var listeners = this._eventListeners[event._type].slice();
				
				//TODO: Sort by priority (no priority available yet).
				
				for (var i2 = 0; i2 < listeners.length; i2++)
				{
					handlerEvent = event.clone(); //Clone the event so the handler can't fudge our event data.
					listeners[i2](handlerEvent);
					
					if (handlerEvent._defaultPrevented == true)
					{
						dispatchEvent._defaultPrevented = true;
						event._defaultPrevented = true;
					}
					
					if (handlerEvent._canceled == true)
					{
						dispatchEvent._canceled = true;
						return;
					}
				}
			}
		}
	};

/**
 * @function _addChild
 * Adds a child element to the end of this element's child list.
 * 
 * @param element CanvasElement
 * CanvasElement to be added as a child of this element.
 * 
 * @returns CanvasElement
 * Returns the element just added.
 */	
CanvasElement.prototype._addChild = 
	function (element)
	{
		return this._addChildAt(element, this._children.length);
	};

/**
 * @function _addChildAt
 * Inserts a child element to this elements child list at the specified index.
 * 
 * @param element CanvasElement
 * CanvasElement to be added as a child of this element.
 * 
 * @param index int
 * The index position to insert the child in the elements child list.
 * 
 * @returns CanvasElement
 * Returns the element just added when successfull, null if the element could not
 * be added due to the index being out of range.
 */		
CanvasElement.prototype._addChildAt = 
	function (element, index)
	{
		if (!(element instanceof CanvasElement))
			return null;
		
		if (index < 0 || index > this._children.length)
			return null;
		
		//Elements may only have 1 parent.
		if (element._parent != null)
			element._parent._removeChild(element);
		
		element._parent = this;
		this._children.splice(index, 0, element);
		this.addEventListener("stylechanged", element._onExternalStyleChangedInstance);
		
		element._propagateChildData();
		
		this._invalidateMeasure();
		this._invalidateLayout();
		
		if (this._manager != null)
		{
			this._manager._rollOverInvalid = true;
			this._manager._processAddRemoveDisplayChainQueue();
		}
		
		return element;
	};
	
/**
 * @function _removeChild
 * Removes a child element from this elements child list.
 * 
 * @param element CanvasElement
 * Child to be removed.
 * 
 * @returns CanvasElement
 * Returns the element just removed if successfull, null if the
 * element could not be removed due to it not being a child of this element.
 */	
CanvasElement.prototype._removeChild = 
	function (element)
	{
		var childIndex = this._children.indexOf(element);
		if (childIndex == -1)
			return null;
	
		return this._removeChildAt(childIndex);
	};

/**
 * @function _removeChildAt
 * Removes a child element at specified index.
 * 
 * @param index int
 * Index to be removed.
 * 
 * @returns CanvasElement
 * Returns the element just removed if successfull, null if the element could
 * not be removed due it it not being a child of this element, or index out of range.
 */		
CanvasElement.prototype._removeChildAt = 
	function (index)
	{
		if (index < 0 || index > this._children.length - 1)
			return null;
		
		var element = this._children.splice(index, 1)[0]; //Returns array of removed items.
		
		//We removed an element that is in mouse-down state. 
		//Change the mouseup target to the parent of this element.
		if (element._mouseIsDown == true)
			element._manager._mouseDownElement = element._parent;
		
		if (element._mouseIsOver == true)
			element._manager._rollOverElement = element._parent;
		
		element._parent = null;
		this.removeEventListener("stylechanged", element._onExternalStyleChangedInstance);
		
		element._propagateChildData();
		
		this._invalidateMeasure();
		this._invalidateLayout();
		
		if (this._manager != null)
		{
			this._manager._rollOverInvalid = true;
			this._manager._processAddRemoveDisplayChainQueue();
		}
		
		return element;
	};	

/**
 * @function _getChildAt
 * Gets the child element at the supplied index.
 * 
 * @param index int
 * Index of child element to return;
 * 
 * @returns CanvasElement
 * The element at the supplied index, or null if index is out of range. 
 */	
CanvasElement.prototype._getChildAt = 
	function (index)
	{
		if (index < 0 || index > this._children.length - 1)
			return null;
		
		return this._children[index];
	};
	
/**
 * @function _getChildIndex
 * Returns the index of the supplied child element.
 * 
 * @param element CanvasElement
 * Child element to return the index.
 * 
 * @returns int
 * Returns the child index or -1 if the element is not
 * a child of this element.
 */	
CanvasElement.prototype._getChildIndex = 
	function (element)
	{
		return this._children.indexOf(element);
	};
	
/**
 * @function _setChildIndex
 * Changes a child element's index. 
 * 
 * @param element CanvasElement
 * Child element to change index.
 * 
 * @param index int
 * New index of the child element.
 * 
 * @returns boolean
 * Returns true if the child's index is successfully changed, false if the element
 * is not a child of this element or the index is out of range.
 */	
CanvasElement.prototype._setChildIndex = 
	function (element, index)
	{
		if (index < 0 || index > this._children.length - 1)
			return false;
		
		var currentIndex = this._getChildIndex(element);
		if (currentIndex == -1 || currentIndex == index)
			return false;
		
		this._children.splice(index, 0, this._children.splice(currentIndex, 1)[0]);
		
		this._invalidateMeasure();
		this._invalidateLayout();
		
		return true;
	};
	
/**
 * @function _getNumChildren
 * Gets this elements number of children.
 * 
 * @returns int
 * The number of child elements.
 */	
CanvasElement.prototype._getNumChildren = 
	function ()
	{
		return this._children.length;
	};
	
//@private	
CanvasElement.prototype._propagateChildData = 
	function ()
	{
		var isManager = (this instanceof CanvasManager);
	
		if ((isManager == false && (this._parent == null || this._parent._displayDepth == 0)) || 
			isManager == true && this._manager != null)
		{//Removed from display chain
			
			//Purge manager data.
			if (this._manager != null)
			{
				if (this._stylesInvalid == true)
					this._manager._updateStylesQueue.removeNode(this._stylesValidateNode, this._displayDepth);
				
				if (this._measureInvalid == true)
					this._manager._updateMeasureQueue.removeNode(this._measureValidateNode, this._displayDepth);
				
				if (this._layoutInvalid == true)
					this._manager._updateLayoutQueue.removeNode(this._layoutValidateNode, this._displayDepth);
				
				if (this._renderInvalid == true)
					this._manager._updateRenderQueue.removeNode(this._renderValidateNode, this._displayDepth);
				
				if (this._redrawRegionInvalid == true)
					this._manager._updateRedrawRegionQueue.removeNode(this._redrawRegionValidateNode, this._displayDepth);
				
				if (this._compositeRenderInvalid == true)
					this._manager._compositeRenderQueue.removeNode(this._compositeRenderValidateNode, this._displayDepth);
				
				if (this == this._manager._draggingElement)
					this._manager._clearDraggingElement();
				
				if (this == this._manager._focusElement)
					this._manager._focusElement = null;
				
				this._manager._pushAddRemoveDisplayChainQueue(this, "removed");
			}
			
			this._renderFocusRing = false;
			this._isFocused = false;
			this._mouseIsOver = false;
			this._mouseIsDown = false;
			this._displayDepth = 0;
			this._manager = null;
		}
		else
		{//Added to display chain
			
			if (isManager == true)
			{
				this._displayDepth = 1;
				this._manager = this;
			}
			else
			{
				this._displayDepth = this._parent._displayDepth + 1;
				this._manager = this._parent._manager;
			}
			
			//Add manager data.
			if (this._manager != null)
			{
				if (this._stylesInvalid == true)
					this._manager._updateStylesQueue.addNode(this._stylesValidateNode, this._displayDepth);
				
				if (this._measureInvalid == true)
					this._manager._updateMeasureQueue.addNode(this._measureValidateNode, this._displayDepth);
				
				if (this._layoutInvalid == true)
					this._manager._updateLayoutQueue.addNode(this._layoutValidateNode, this._displayDepth);
				
				if (this._renderInvalid == true)
					this._manager._updateRenderQueue.addNode(this._renderValidateNode, this._displayDepth);
				
				if (this._redrawRegionInvalid == true)
					this._manager._updateRedrawRegionQueue.addNode(this._redrawRegionValidateNode, this._displayDepth);
				
				if (this._compositeRenderInvalid == true)
					this._manager._compositeRenderQueue.addNode(this._compositeRenderValidateNode, this._displayDepth);
				
				this._manager._pushAddRemoveDisplayChainQueue(this, "added");
			}
		}
		
		for (var i = 0; i < this._children.length; i++)
			this._children[i]._propagateChildData();
	};	

/**
 * @function _setRelativePosition
 * Sets the elements position relative to a supplied element regardless of this element's transformation,
 * depth, or position in the display hierarchy or relation to the supplied relativeToElement. This should typically
 * only be called during the parent element's layout phase. Setting relativeToElement to null has an identical
 * effect to calling _setActualPosition(). This is used by some containers to position the element relative
 * to a parent's coordinate plane rather than the child's transformed plane.
 * 
 * @param x Number
 * The relative X position to move this element's position.
 * 
 * @param y Number
 * The relative Y position to move this element's position.
 * 
 * @param relativeToElement CanvasElement
 * The CanvasElement to move this element relative too.
 */	
CanvasElement.prototype._setRelativePosition = 
	function (x, y, relativeToElement)
	{
		if (relativeToElement == null || relativeToElement == this)
		{
			if (this._x == x && this._y == y)
				return;
			
			this._x = x;
			this._y = y;
			
			if (this._manager != null)
				this._manager._rollOverInvalid = true;
			
			this._invalidateRedrawRegion();
		}
		
		if (this._manager == null || this._manager != relativeToElement._manager)
			return;
		
		//Use relative parent metrics. We want to shift this elements entire plane if its
		//transformed (rotated), we dont want to slide the element around on its transformed plane.
		var parentMetrics = this.getMetrics(this._parent);
		
		//Get the move-to position within our parent element.
		var newPosition = {x:x, y:y};
		relativeToElement.translatePointTo(newPosition, this._parent);
		
		//We haven't moved.
		if (newPosition.x == parentMetrics.getX() && newPosition.y == parentMetrics.getY())
			return;
		
		//Get the delta in its position.
		var deltaX = newPosition.x - parentMetrics.getX();
		var deltaY = newPosition.y - parentMetrics.getY();
		
		this._x = this._x + deltaX;
		this._y = this._y + deltaY;
		
		if (this._rotateDegrees != 0)
		{
			this._rotateCenterX += deltaX;
			this._rotateCenterY += deltaY;
		}
		
		if (this._manager != null)
			this._manager._rollOverInvalid = true;
		
		this._invalidateRedrawRegion();
	};	
		
/**
 * @function _setActualPosition
 * Sets the elements position within its parent. Note that if the element is transformed or rotated,
 * this sets the elements position within its transformed plane. If you wish to position a transformed
 * element relative to its parents coordinate plane, use _setRelativePosition(). This should typically
 * only be called from within the parents layout phase.
 * 
 * @param x int
 * The X position to move the element.
 * 
 * @param y int
 * The Y position to move the element.
 */	
CanvasElement.prototype._setActualPosition = 
	function (x, y)
	{
		x = Math.round(x);
		y = Math.round(y);
		
		if (this._x == x && this._y == y)
			return;
		
		this._x = x;
		this._y = y;
		
		if (this._manager != null)
			this._manager._rollOverInvalid = true;
		
		this._invalidateRedrawRegion();
	};	
	
/**
 * @function _setActualSize
 * Sets this element's size in pixels prior to any transformation or rotation. 
 * This should typically only be called from within the parents layout phase.
 * 
 * @param width Number
 * The width in pixels to size this element.
 * 
 * @param height Number
 * The height in pixels to size this element.
 */	
CanvasElement.prototype._setActualSize = 
	function (width, height)
	{
		//if (typeof width !== "number" || typeof height !== "number" || isNaN(width) || isNaN(height))
		//	throw "Invalid Size";
	
		//TODO: This is BAD!!!  This is effectively a fix for components that arent rounding / drawing on
		//		on even pixel lines causing anti-aliasing fuzz. This *needs* to be removed, and offending components fixed.
		width = Math.round(width);
		height = Math.round(height);
		
		if (this._width == width && this._height == height)
			return false;
		
		this._width = width;
		this._height = height;
		
		this._invalidateLayout();
		this._invalidateRender();
		this._invalidateRedrawRegion();
		
		if (this.hasEventListener("resize", null) == true)
			this.dispatchEvent(new DispatcherEvent("resize"), false);
		
		if (this._manager != null)
			this._manager._rollOverInvalid = true;
		
		return true;
	};
	
/**
 * @function _setActualRotation
 * Sets this elements rotation degrees and rotation point relative to its parent. This should typically
 * only be called from within the parent's layout phase.
 * 
 * @param degrees Number
 * Degrees to rotate the element (clockwise).
 * 
 * @param centerX Number
 * The X position relative to the elements parent to rotate around.
 * 
 * @param centerY Number
 * The Y position relative to the elements parent to rotate around.
 */	
CanvasElement.prototype._setActualRotation = 
	function (degrees, centerX, centerY)
	{
		if (centerX == null || centerY == null)
		{
			centerX = 0;
			centerY = 0;
		}
	
		if (this._rotateDegrees != degrees || this._rotateCenterX != centerX || this._rotateCenterY != centerY)
		{
			this._invalidateRedrawRegion();
			
			if (this._rotateDegrees != degrees)
			{
				this._compositeEffectChanged = true;
				this._invalidateCompositeRender();
			}
		}
		
		this._rotateDegrees = degrees;
		this._rotateCenterX = centerX;
		this._rotateCenterY = centerY;
	};
	
//@private
CanvasElement.prototype._setMeasuredSize = 
	function (width, height)
	{
		if (this._measuredWidth == width && this._measuredHeight == height)
			return;
		
		this._measuredWidth = width;
		this._measuredHeight = height;
		
		if (this._parent != null)
		{
			this._parent._invalidateMeasure();
			this._parent._invalidateLayout();
		}
	};
	
//@private	
CanvasElement.prototype._setRenderFocusRing = 
	function (shouldRender)
	{
		if (this._renderFocusRing == shouldRender)
			return;
		
		this._renderFocusRing = shouldRender;
		this._invalidateRender();
	};

//@private	
CanvasElement.prototype._createMetrics = 
	function ()
	{
		return new DrawMetrics();
	};
	
	
/**
 * @function _getGraphicsCtx
 * Returns the canvas context used when rendering this element. This should typically
 * only be called from within the element's _doRender() phase, and only if you intend
 * to actually draw. Calling this will impact the canvas redraw regions.
 *  
 * @returns Canvas2DContext
 * Canvas context used when rendering this element.
 */	
CanvasElement.prototype._getGraphicsCtx = 
	function ()
	{
		if (this._graphicsCanvas == null)
		{
			this._graphicsCanvas = document.createElement("canvas");
			this._graphicsCtx = this._graphicsCanvas.getContext("2d");
			
			this._graphicsCanvas.width = this._width;
			this._graphicsCanvas.height = this._height;
		}
		
		this._renderChanged = true;
		this._graphicsClear = false;
		
		this._invalidateRedrawRegion();
		
		return this._graphicsCtx;
	};
	
	
/**
 * @function _getPaddingSize
 * Helper function that returns the elements total padding width and height per its applied styles.
 * 
 * @returns Object
 * Returns an object containing 
 * {width:paddingWidth, height:paddingHeight,
 * paddingBottom:paddingBottom, paddingTop:paddingTop,
 * paddingLeft:paddingLeft, paddingRight:paddingRight}.
 */	
CanvasElement.prototype._getPaddingSize = 
	function ()
	{
		var paddingData = this.getStyleData("Padding");
		var paddingTopData = this.getStyleData("PaddingTop");
		var paddingBottomData = this.getStyleData("PaddingBottom");
		var paddingLeftData = this.getStyleData("PaddingLeft");
		var paddingRightData = this.getStyleData("PaddingRight");
		
		var paddingTop = paddingTopData.value;
		if (paddingData.comparePriority(paddingTopData) > 0) //Use Padding if higher priority
			paddingTop = paddingData.value;
		
		var paddingBottom = paddingBottomData.value;
		if (paddingData.comparePriority(paddingBottomData) > 0) //Use Padding if higher priority
			paddingBottom = paddingData.value;
		
		var paddingLeft = paddingLeftData.value;
		if (paddingData.comparePriority(paddingLeftData) > 0) //Use Padding if higher priority
			paddingLeft = paddingData.value;
		
		var paddingRight = paddingRightData.value;
		if (paddingData.comparePriority(paddingRightData) > 0) //Use Padding if higher priority
			paddingRight = paddingData.value;
		
		return { width: paddingLeft + paddingRight, 
				height: paddingTop + paddingBottom, 
				paddingBottom:paddingBottom, 
				paddingTop:paddingTop, 
				paddingLeft:paddingLeft, 
				paddingRight:paddingRight};
	};

/**
 * @function _getBorderThickness
 * Helper function that returns the elements border thickness per its applied styles.
 * 
 * @returns Number
 * The elements border thickness.
 */	
CanvasElement.prototype._getBorderThickness = 
	function ()
	{
		var borderThickness = 0;
		var borderType = this.getStyle("BorderType");
		var borderColor = this.getStyle("BorderColor");
		if ((borderType == "solid" || borderType == "inset" || borderType == "outset") && borderColor != null)
		{
			borderThickness = this.getStyle("BorderThickness");
			if (borderThickness < 0)
				borderThickness = 0;
		}
		
		return borderThickness;
	};

/**
 * @function _getStyledOrMeasuredWidth
 * Helper function that returns this elements styled width, or measured width if no style is set. Typically
 * called from within a parent containers layout phase.
 * 
 * @returns Number
 * The elements width.
 */	
CanvasElement.prototype._getStyledOrMeasuredWidth = 
	function ()
	{
		var widthData = this.getStyleData("Width");
		var percentWidthData = this.getStyleData("PercentWidth");
		
		var width = null;
		
		//Use width if it is equal or higher priority than percent width
		if (widthData.comparePriority(percentWidthData) >= 0)
			width = widthData.value;
		
		if (width == null)
		{
			var maxWidth = this.getStyle("MaxWidth");
			var minWidth = this.getStyle("MinWidth");
			
			if (minWidth == null)
				minWidth = 0;
			if (maxWidth == null)
				maxWidth = Number.MAX_VALUE;
			
			width = this._measuredWidth;
			width = Math.min(width, maxWidth);
			width = Math.max(width, minWidth);
		}
		
		return width;
	};

/**
 * @function _getStyledOrMeasuredHeight
 * Helper function that returns this elements styled height, or measured height if no style is set. Typically
 * called from within a parent containers layout phase.
 * 
 * @returns Number
 * The elements height.
 */		
CanvasElement.prototype._getStyledOrMeasuredHeight = 
	function ()
	{
		var heightData = this.getStyleData("Height");
		var percentHeightData = this.getStyleData("PercentHeight");
		
		var height = null;
		
		//Use height if is equal or higher priority that percent height
		if (heightData.comparePriority(percentHeightData) >= 0)
			height = heightData.value;
	
		if (height == null)
		{
			var maxHeight = this.getStyle("MaxHeight");
			var minHeight = this.getStyle("MinHeight");			
			
			if (minHeight == null)
				minHeight = 0;
			if (maxHeight == null)
				maxHeight = Number.MAX_VALUE;	
			
			height = this._measuredHeight;
			height = Math.min(height, maxHeight);
			height = Math.max(height, minHeight);
		}
		
		return height;
	};
	
/**
 * @function _getBorderMetrics
 * Helper function that returns a DrawMetrics object whose bounding area is inside this elements border.
 * 
 * @returns DrawMetrics
 * Returns DrawMetrics that define a bounding area inside this elements border.
 */	
CanvasElement.prototype._getBorderMetrics = 
	function ()
	{
		var metrics = this._createMetrics();
		
		var borderThickness = this._getBorderThickness();
		
		metrics._x = borderThickness / 2;
		metrics._y = borderThickness / 2;
		
		metrics._width = this._width - borderThickness;
		metrics._height = this._height - borderThickness;
		
		return metrics;
	};
	
/**
 * @function _getPaddingMetrics
 * Helper function that returns a DrawMetrics object whose bounding area is inside this elements padding area.
 * 
 * @returns DrawMetrics
 * Returns DrawMetrics that define a bounding area inside this elements padding area.
 */		
CanvasElement.prototype._getPaddingMetrics = 
	function()
	{
		var metrics = this._createMetrics();
		var paddingSize = this._getPaddingSize();
		
		metrics._x = paddingSize.paddingLeft;
		metrics._y = paddingSize.paddingTop;
		metrics._width = this._width -  paddingSize.paddingLeft - paddingSize.paddingRight;
		metrics._height = this._height - paddingSize.paddingTop - paddingSize.paddingBottom;
		
		return metrics;
	};

/**
 * @function _drawBackgroundShape
 * Used to draw the path to the Canvas2DContext that is to be used to render the focus ring,
 * fill the background, and draw the border. You should never need to explicitly call this. 
 * The system calls this during render phase.
 * Typically you should use the BackgroundShape style
 * for this, but may override it under more complex scenarios.
 * 
 * @param ctx Canvas2DContext
 * Canvas2DContext to draw the background shape path.
 * 
 * @param borderMetrics DrawMetrics
 * The DrawMetrics containing x,y,width,height used to draw the background. These
 * metrics should be the same as the ones used to stroke a solid border.
 */	
CanvasElement.prototype._drawBackgroundShape = 
	function (ctx, borderMetrics)
	{
		if (this._backgroundShape == null)
		{
			//Full rectangle
			var x = borderMetrics.getX();
			var y = borderMetrics.getY();
			var w = borderMetrics.getWidth();
			var h = borderMetrics.getHeight();
			
			ctx.moveTo(x, y);
			ctx.lineTo(x + w, y);
			ctx.lineTo(x + w, y + h);
			ctx.lineTo(x, y + h);
			ctx.closePath();
		}
		else
		{
			this._backgroundShape.drawShape(ctx, borderMetrics);
		}
	};

/**
 * @function _drawFocusRing
 * Used to draw the focus ring when a tab-able element gains focus due to a tab stop per the elements styles.
 * You should never need to explicitly call this. The system calls this during
 * the render phase. You may override if you wish to draw a more complex focus indicator.
 * Focus ring is drawn *outside* the elements bounding box. Focus ring is rendered
 * before the background and border. 
 * 
 * @param ctx Canvas2DContext
 * Canvas2DContext to render the focus ring.
 * 
 * @param borderMetrics DrawMetrics
 * The DrawMetrics containing x,y,width,height used to draw the background. These
 * metrics should be the same as the ones used to stroke a solid border.
 */		
CanvasElement.prototype._drawFocusRing = 
	function (ctx, borderMetrics)
	{
		var focusRingThickness = this.getStyle("FocusThickness");
		if (focusRingThickness <= 0)
			return;
		
		var focusRingColor = this.getStyle("FocusColor");
		if (focusRingColor == null)
			return;		
		
		var metrics = this.getMetrics(this);
		
		var x = metrics.getX() - focusRingThickness;
		var y = metrics.getY() - focusRingThickness;
		var w = metrics.getWidth() + (focusRingThickness * 2);
		var h = metrics.getHeight() + (focusRingThickness * 2);
		
		ctx.beginPath();
		
		//Draw anticlockwise
		ctx.moveTo(x, y);
		ctx.lineTo(x, y + h);
		ctx.lineTo(x + w, y + h);
		ctx.lineTo(x + w, y);
		ctx.closePath();
		
		//Draws clockwise (shape inside shape)
		this._drawBackgroundShape(ctx, borderMetrics);
		
		//Clip the *inside* of the background shape
		ctx.clip();

		//Draw border
		ctx.beginPath();
		this._drawBackgroundShape(ctx, borderMetrics);
		
		ctx.strokeStyle = focusRingColor;

		//Increase thickness
		ctx.lineWidth = this._getBorderThickness() + (focusRingThickness * 2);
		ctx.stroke();
	};
	
/**
 * @function _fillBackground
 * Used to fill the elements background shape according to the element's BackgroundShape and BackgroundFill styles.
 * You should never need to explicitly call this. The system calls this during
 * the render phase. You may override if you need to do a more complex background fill. The background fill
 * is rendered after the focus ring and before the border. 
 * 
 * @param ctx Canvas2DContext
 * Canvas2DContext to fill the background shape.
 * 
 * @param borderMetrics DrawMetrics
 * The DrawMetrics containing x,y,width,height used to draw the background. These
 * metrics should be the same as the ones used to stroke a solid border.
 */	
CanvasElement.prototype._fillBackground = 
	function (borderMetrics)
	{
		if (this._backgroundFill == null)
			return;
	
		var ctx = this._getGraphicsCtx();
		
		ctx.beginPath();
		this._drawBackgroundShape(ctx, borderMetrics);
		
		this._backgroundFill.drawFill(ctx, borderMetrics);
	};

//@private	
CanvasElement.prototype._drawSolidBorder = 
	function (ctx, borderMetrics)
	{
		var borderColor = this.getStyle("BorderColor");
		if (borderColor == null)
			return;
		
		var borderThickness = this.getStyle("BorderThickness");
		if (borderThickness < 0)
			return;
	
		ctx.beginPath();
		this._drawBackgroundShape(ctx, borderMetrics);
		ctx.strokeStyle = borderColor;

		ctx.lineWidth = borderThickness;
		ctx.stroke();
	};
	
/**
 * @function _drawBorder
 * Used to render the elements border according to the element's style settings.
 * You should never need to explicitly call this. The system calls this during
 * the render phase. You may override if you need to do a more complex border. The border
 * is rendered last, on top of the focus ring and background fill.
 * 
 * @param ctx Canvas2DContext
 * Canvas2DContext to render the border.
 * 
 * @param borderMetrics DrawMetrics
 * The DrawMetrics containing x,y,width,height used to draw the background. These
 * metrics should be the same as the ones used to stroke a solid border.
 */		
CanvasElement.prototype._drawBorder = 
	function (borderMetrics)
	{
		var borderType = this.getStyle("BorderType");
	
		if (borderType != "solid" && borderType != "inset" && borderType != "outset")
			return;			
	
		var ctx = this._getGraphicsCtx();
		
		if (borderType == "solid")
		{
			this._drawSolidBorder(ctx, borderMetrics);
		}
		else //inset || outset
		{
			var borderColor = this.getStyle("BorderColor");
			var borderThickness = this.getStyle("BorderThickness");
			
			if (borderColor == null || borderThickness <= 0)
				return;
			
			var x = 0;
			var y = 0;
			var w = this._width;
			var h = this._height;
			
			var lighterColor = CanvasElement.adjustColorLight(borderColor, .3);
			var darkerColor = CanvasElement.adjustColorLight(borderColor, .3 * -1);
			
			var tlColor = borderType == "inset" ? darkerColor : lighterColor;
			var brColor = borderType == "inset" ? lighterColor : darkerColor;
			
			ctx.beginPath();
			ctx.moveTo(x, y + h);
			ctx.lineTo(x, y);
			ctx.lineTo(x + w, y);
			
			ctx.lineTo(x + w - borderThickness, y + borderThickness);
			ctx.lineTo(x + borderThickness, y + borderThickness);
			ctx.lineTo(x + borderThickness, y + h - borderThickness);
			ctx.closePath();
			
			ctx.fillStyle = tlColor;
			ctx.fill();
			
			ctx.beginPath();
			ctx.moveTo(x, y + h);
			ctx.lineTo(x + w, y + h);
			ctx.lineTo(x + w, y);
			
			ctx.lineTo(x + w - borderThickness, y + borderThickness);
			ctx.lineTo(x + w - borderThickness, y + h - borderThickness);
			ctx.lineTo(x + borderThickness, y + h - borderThickness);
			ctx.closePath();
			
			ctx.fillStyle = brColor;
			ctx.fill();
			
			ctx.lineWidth = 1;
			ctx.globalAlpha= .15;
			ctx.strokeStyle = "#000000";
			
			ctx.beginPath();
			ctx.moveTo(x, y);
			ctx.lineTo(x + borderThickness, y + borderThickness);
			ctx.stroke();
			
			ctx.beginPath();
			ctx.moveTo(x + w, y + h);
			ctx.lineTo(x + w - borderThickness, y + h - borderThickness);
			ctx.stroke();
		}
	};

//@private	
CanvasElement.prototype._validateStyles = 
	function ()
	{
		this._stylesInvalid = false;
	
		//Reset and record the current set of invalid styles
		var stylesInvalidMap = Object.create(null);
		
		for (var prop in this._stylesInvalidMap)
		{
			if (this._stylesInvalidMap[prop] == true)
			{
				stylesInvalidMap[prop] = true; //Record
				this._stylesInvalidMap[prop] = false; //Reset
			}
		}
		
		this._doStylesUpdated(stylesInvalidMap);
	};
	
//@private	
CanvasElement.prototype._validateMeasure = 
	function ()
	{
		this._measureInvalid = false;
	
		if (this.getStyle("IncludeInMeasure") == true)
		{
			var paddingSize = this._getPaddingSize();
			this._doMeasure(paddingSize.width, paddingSize.height);
		}
		else
			this._setMeasuredSize(0, 0);
		
		if (this.hasEventListener("measurecomplete", null) == true)
			this.dispatchEvent(new DispatcherEvent("measurecomplete"));
	};
	
//@private	
CanvasElement.prototype._validateLayout = 
	function ()
	{
		this._layoutInvalid = false;
		this._doLayout(this._getPaddingMetrics());
		
		if (this._layoutInvalid == false && this.hasEventListener("layoutcomplete", null) == true)
			this.dispatchEvent(new DispatcherEvent("layoutcomplete"));
	};	
	
//@private
CanvasElement.prototype._validateRender = 
	function ()
	{
		this._renderInvalid = false;
		
		if (this._graphicsCanvas != null)
		{
			this._graphicsCanvas.width = this._width;
			this._graphicsCanvas.height = this._height;
			this._graphicsCtx.clearRect(0, 0, this._graphicsCanvas.width, this._graphicsCanvas.height);
		}
		
		if (this._graphicsClear == false)
		{
			this._renderChanged = true;
			this._graphicsClear = true;
			
			this._invalidateRedrawRegion();
		}
		
		this._doRender();
	};

//@private
CanvasElement.prototype._getCompositeMetrics = 
	function (compositeParent)
	{
		for (var i = 0; i < this._compositeMetrics.length; i++)
		{
			if (this._compositeMetrics[i].element == compositeParent)
				return this._compositeMetrics[i];
		}
		
		return null;
	};
	
//@private	
CanvasElement.prototype._validateRedrawRegion = 
	function (cachePool)
	{
		this._redrawRegionInvalid = false;
	
		var i;
		
		var newCompositeMetrics = cachePool.compositeMetrics;
		var newCompositeMetricsLength = 0;
		
		var oldVisible = this._renderVisible;
		var forceRegionUpdate = this._forceRegionUpdate; 
		
		//Get new visibility
		var newVisible = true;
		if ((this._parent != null && this._parent._renderVisible == false) || 
			this.getStyle("Visible") == false || 
			this.getStyle("Alpha") <= 0)
		{
			newVisible = false;
		}
		
		//Wipe out the composite metrics (rebuild as we recurse children if we're a composite layer)
		this._compositeVisibleMetrics = null;
		this._transformVisibleMetrics = null;
		this._transformDrawableMetrics = null;
		this._forceRegionUpdate = false;
		
		//Composite effect on this element changed, we *must* update the region on ourself and all of our children.
		if (this._compositeEffectChanged == true)
			forceRegionUpdate = true;
		
		if ((this._renderChanged == true || this._graphicsClear == false) &&
			(oldVisible == true || newVisible == true))
		{
			var parent = this;

			//Transformed via points up parent chain
			var rawMetrics = cachePool.rawMetrics; 
			rawMetrics._x = 0;
			rawMetrics._y = 0;
			rawMetrics._width = this._width;
			rawMetrics._height = this._height;
			
			//Transformed via metrics up parent chain (recalculated each layer, expands, and clips)
			var drawableMetrics = cachePool.drawableMetrics; 
			drawableMetrics.copyFrom(rawMetrics);
			
			//Used for transforming the raw metrics up the parent chain.
			var pointRawTl = cachePool.pointRawTl;
			var pointRawTr = cachePool.pointRawTr;
			var pointRawBr = cachePool.pointRawBr;
			var pointRawBl = cachePool.pointRawBl;
			
			pointRawTl.x = rawMetrics._x;
			pointRawTl.y = rawMetrics._y;
			pointRawTr.x = rawMetrics._x + rawMetrics._width;
			pointRawTr.y = rawMetrics._y;
			pointRawBr.x = rawMetrics._x + rawMetrics._width;
			pointRawBr.y = rawMetrics._y + rawMetrics._height;
			pointRawBl.x = rawMetrics._x;
			pointRawBl.y = rawMetrics._y + rawMetrics._height;
			
			var pointDrawableTl = cachePool.pointDrawableTl;
			var pointDrawableTr = cachePool.pointDrawableTr;
			var pointDrawableBr = cachePool.pointDrawableBr;
			var pointDrawableBl = cachePool.pointDrawableBl;
			
			pointDrawableTl.x = 0;
			pointDrawableTl.y = 0;
			pointDrawableTr.x = 0;
			pointDrawableTr.y = 0;
			pointDrawableBr.x = 0;
			pointDrawableBr.y = 0;
			pointDrawableBl.x = 0;
			pointDrawableBl.y = 0;
			
			var minX = null;
			var maxX = null;
			var minY = null;
			var maxY = null;
			
			//Cached storage of previous metrics per composite parent.
			var oldMetrics = null;	//{element:element, metrics:DrawMetrics, drawableMetrics:DrawMetrics}
			
			var clipMetrics = cachePool.clipMetrics;
			var shadowMetrics = cachePool.shadowMetrics;
			
			var shadowSize = 0;
			
			var drawableMetricsChanged = false;
			var rawMetricsChanged = false;
			
			//Walk up the parent chain invalidating the redraw region and updating _compositeVisibleMetrics
			while (parent != null)
			{
				//Apply clipping to drawable metrics (Always clip root manager)
				if (drawableMetrics != null && (parent == this || parent.getStyle("ClipContent") == true))
				{
					//Clip metrics relative to current element
					clipMetrics._x = 0;
					clipMetrics._y = 0;
					clipMetrics._width = parent._width;
					clipMetrics._height = parent._height;
					
					//Reduce drawable metrics via clipping metrics.
					drawableMetrics.mergeReduce(clipMetrics);
					
					//Kill metrics if completely clipped
					if (drawableMetrics._width <= 0 || drawableMetrics._height <= 0)
						drawableMetrics = null;
				}
				
				//Update redraw region, _compositeVisibleMetrics, and record new stored composite metrics.
				if (parent._isCompositeElement() == true)
				{
					oldMetrics = this._getCompositeMetrics(parent);
					
					if (drawableMetrics != null && newVisible == true && this._graphicsClear == false)
					{
						//newCompositeMetrics.push({element:parent, metrics:rawMetrics.clone(), drawableMetrics:drawableMetrics.clone()});
						if (newCompositeMetrics[newCompositeMetricsLength] == null)
							newCompositeMetrics[newCompositeMetricsLength] = {element:parent, metrics:rawMetrics.clone(), drawableMetrics:drawableMetrics.clone()};
						else
						{
							newCompositeMetrics[newCompositeMetricsLength].element = parent;
							newCompositeMetrics[newCompositeMetricsLength].metrics.copyFrom(rawMetrics);
							newCompositeMetrics[newCompositeMetricsLength].drawableMetrics.copyFrom(drawableMetrics);
						}
						
						newCompositeMetricsLength++;
						
						//Update composite parents visible metrics
						if (parent._compositeVisibleMetrics == null)
							parent._compositeVisibleMetrics = drawableMetrics.clone();
						else
							parent._compositeVisibleMetrics.mergeExpand(drawableMetrics);
					}
					
					drawableMetricsChanged = true;
					if ((oldMetrics == null && drawableMetrics == null) ||
						(oldMetrics != null && drawableMetrics != null && oldMetrics.drawableMetrics.equals(drawableMetrics) == true))
					{
						drawableMetricsChanged = false;
					}
					
					rawMetricsChanged = true;
					if (oldMetrics != null && oldMetrics.metrics.equals(rawMetrics) == true)
					{
						rawMetricsChanged = false;
					}
					
					//Update the composite element's redraw region
					if (forceRegionUpdate == true || 		//Composite effect changed	
						this._renderChanged == true ||		//Render changed
						oldVisible != newVisible ||			//Visible changed
						drawableMetricsChanged == true ||	//Drawable region changed (clipping)
						rawMetricsChanged == true)			//Position changed
					{
						//If was visible, redraw old metrics
						if (oldVisible == true && oldMetrics != null)
							parent._updateRedrawRegion(oldMetrics.drawableMetrics);
						
						//Redraw new metrics
						if (newVisible == true)
							parent._updateRedrawRegion(drawableMetrics);
					}
				}				
				
				//Drawable metrics will be null if we've been completely clipped. No reason to do any more translation.
				if (drawableMetrics != null)
				{	//Fix current metrics so that we're now relative to our parent.
					
					//Update position
					
					//Drawable metrics////
					drawableMetrics._x += parent._x;
					drawableMetrics._y += parent._y;
					
					shadowSize = parent.getStyle("ShadowSize");
					
					//Expand metrics for shadow
					if (shadowSize > 0 && parent.getStyle("ShadowColor") != null)
					{
						//Copy drawable metrics
						shadowMetrics.copyFrom(drawableMetrics);
						
						//Create shadow position metrics
						shadowMetrics._width += (shadowSize * 2);
						shadowMetrics._height += (shadowSize * 2);
						shadowMetrics._x -= shadowSize;
						shadowMetrics._y -= shadowSize;
						shadowMetrics._x += parent.getStyle("ShadowOffsetX");
						shadowMetrics._y += parent.getStyle("ShadowOffsetY");
						
						//Merge the shadow metrics with the drawable metrics
						drawableMetrics.mergeExpand(shadowMetrics);
						
						//Handle transform
						if (CanvasElement.normalizeDegrees(parent._rotateDegrees) != 0)
						{
							//Transform drawable metrics/////////////
							pointDrawableTl.x = drawableMetrics._x;
							pointDrawableTl.y = drawableMetrics._y;
							
							pointDrawableTr.x = drawableMetrics._x + drawableMetrics._width;
							pointDrawableTr.y = drawableMetrics._y;
							
							pointDrawableBr.x = drawableMetrics._x + drawableMetrics._width;
							pointDrawableBr.y = drawableMetrics._y + drawableMetrics._height;
							
							pointDrawableBl.x = drawableMetrics._x;
							pointDrawableBl.y = drawableMetrics._y + drawableMetrics._height;
							
							parent.rotatePoint(pointDrawableTl, false);
							parent.rotatePoint(pointDrawableTr, false);
							parent.rotatePoint(pointDrawableBl, false);
							parent.rotatePoint(pointDrawableBr, false);
							
							minX = Math.min(pointDrawableTl.x, pointDrawableTr.x, pointDrawableBr.x, pointDrawableBl.x);
							maxX = Math.max(pointDrawableTl.x, pointDrawableTr.x, pointDrawableBr.x, pointDrawableBl.x);
							minY = Math.min(pointDrawableTl.y, pointDrawableTr.y, pointDrawableBr.y, pointDrawableBl.y);
							maxY = Math.max(pointDrawableTl.y, pointDrawableTr.y, pointDrawableBr.y, pointDrawableBl.y);
							
							drawableMetrics._x = minX;
							drawableMetrics._y = minY;
							drawableMetrics._width = maxX - minX;
							drawableMetrics._height = maxY - minY;
						}
						
						//Use drawable metrics as the new raw metrics (shadow uses context of parent applying shadow)
						rawMetrics.copyFrom(drawableMetrics);
						
						pointRawTl.x = rawMetrics._x;
						pointRawTl.y = rawMetrics._y;
						
						pointRawTr.x = rawMetrics._x + rawMetrics._width;
						pointRawTr.y = rawMetrics._y;
						
						pointRawBr.x = rawMetrics._x + rawMetrics._width;
						pointRawBr.y = rawMetrics._y + rawMetrics._height;
						
						pointRawBl.x = rawMetrics._x;
						pointRawBl.y = rawMetrics._y + rawMetrics._height;
					}
					else
					{
						//Raw metrics
						pointRawTl.x += parent._x;
						pointRawTl.y += parent._y;
						
						pointRawTr.x += parent._x;
						pointRawTr.y += parent._y;
						
						pointRawBr.x += parent._x;
						pointRawBr.y += parent._y;
						
						pointRawBl.x += parent._x;
						pointRawBl.y += parent._y;
						
						//Handle transform
						if (CanvasElement.normalizeDegrees(parent._rotateDegrees) != 0)
						{
							//Rotate raw metrics points
							parent.rotatePoint(pointRawTl, false);
							parent.rotatePoint(pointRawTr, false);
							parent.rotatePoint(pointRawBl, false);
							parent.rotatePoint(pointRawBr, false);
							
							//Transform drawable metrics/////////////
							pointDrawableTl.x = drawableMetrics._x;
							pointDrawableTl.y = drawableMetrics._y;
							
							pointDrawableTr.x = drawableMetrics._x + drawableMetrics._width;
							pointDrawableTr.y = drawableMetrics._y;
							
							pointDrawableBr.x = drawableMetrics._x + drawableMetrics._width;
							pointDrawableBr.y = drawableMetrics._y + drawableMetrics._height;
							
							pointDrawableBl.x = drawableMetrics._x;
							pointDrawableBl.y = drawableMetrics._y + drawableMetrics._height;
							
							parent.rotatePoint(pointDrawableTl, false);
							parent.rotatePoint(pointDrawableTr, false);
							parent.rotatePoint(pointDrawableBl, false);
							parent.rotatePoint(pointDrawableBr, false);
							
							minX = Math.min(pointDrawableTl.x, pointDrawableTr.x, pointDrawableBr.x, pointDrawableBl.x);
							maxX = Math.max(pointDrawableTl.x, pointDrawableTr.x, pointDrawableBr.x, pointDrawableBl.x);
							minY = Math.min(pointDrawableTl.y, pointDrawableTr.y, pointDrawableBr.y, pointDrawableBl.y);
							maxY = Math.max(pointDrawableTl.y, pointDrawableTr.y, pointDrawableBr.y, pointDrawableBl.y);
							
							drawableMetrics._x = minX;
							drawableMetrics._y = minY;
							drawableMetrics._width = maxX - minX;
							drawableMetrics._height = maxY - minY;
							/////////////////////////////////////
						}
						
						//Update transformed raw metrics
						minX = Math.min(pointRawTl.x, pointRawTr.x, pointRawBr.x, pointRawBl.x);
						maxX = Math.max(pointRawTl.x, pointRawTr.x, pointRawBr.x, pointRawBl.x);
						minY = Math.min(pointRawTl.y, pointRawTr.y, pointRawBr.y, pointRawBl.y);
						maxY = Math.max(pointRawTl.y, pointRawTr.y, pointRawBr.y, pointRawBl.y);
						
						rawMetrics._x = minX;
						rawMetrics._y = minY;
						rawMetrics._width = maxX - minX;
						rawMetrics._height = maxY - minY;
					}
					
					//Reduce the precision (random rounding errors at *very* high decimal points)
					rawMetrics.roundToPrecision(3);
					
					//Reduce drawable metrics via raw metrics.
					drawableMetrics.mergeReduce(rawMetrics);
					
					//Reduce the precision (random rounding errors at *very* high decimal points)
					drawableMetrics.roundToPrecision(3);
				}
				
				parent = parent._parent;
			}
		}
		
		this._compositeMetrics.length = newCompositeMetricsLength;
		for (i = 0; i < newCompositeMetricsLength; i++)
		{
			if (this._compositeMetrics[i] == null)
			{
				this._compositeMetrics[i] = {element:newCompositeMetrics[i].element, 
											 metrics:newCompositeMetrics[i].metrics.clone(),
											 drawableMetrics:newCompositeMetrics[i].drawableMetrics.clone()};
			}
			else
			{
				this._compositeMetrics[i].element = newCompositeMetrics[i].element;
				this._compositeMetrics[i].metrics.copyFrom(newCompositeMetrics[i].metrics);
				this._compositeMetrics[i].drawableMetrics.copyFrom(newCompositeMetrics[i].drawableMetrics);
			}
		}
		
		this._renderVisible = newVisible;
		this._renderChanged = false;
		
		//Recurse children if we were or are visible.
		if (oldVisible == true || newVisible == true)
		{
			for (i = 0; i < this._children.length; i++)
			{
				if (forceRegionUpdate == true)
					this._children[i]._forceRegionUpdate = true;
					
				this._children[i]._invalidateRedrawRegion();
			}
			
			if (this._isCompositeElement() == true)
				this._invalidateTransformRegion();
		}
	};
		
//@private	
CanvasElement.prototype._validateTransformRegion = 
	function()
	{
		//No transform of root manager, or invisible layers.
		if (this == this._manager || this._compositeVisibleMetrics == null)
			return;
		
		var pointTl = {x:0, y:0};
		var pointTr = {x:0, y:0};
		var pointBr = {x:0, y:0};
		var pointBl = {x:0, y:0};
		
		var minX = null;
		var maxX = null;
		var minY = null;
		var maxY = null;
		
		var shadowSize = 0;
		
		var parent = this;
		this._transformVisibleMetrics = this._compositeVisibleMetrics.clone();
		this._transformDrawableMetrics = this._compositeVisibleMetrics.clone();
		
		var clipMetrics = new DrawMetrics();
		var done = false;
		
		while (true)
		{
			if (this._transformDrawableMetrics != null && parent.getStyle("ClipContent") == true)
			{
				//Clip metrics relative to current element
				clipMetrics._x = 0;
				clipMetrics._y = 0;
				clipMetrics._width = parent._width;
				clipMetrics._height = parent._height;
				
				//Reduce drawable metrics via clipping metrics.
				this._transformDrawableMetrics.mergeReduce(clipMetrics);
				
				//Kill metrics if completely clipped
				if (this._transformDrawableMetrics._width <= 0 || this._transformDrawableMetrics._height <= 0)
				{
					this._transformDrawableMetrics = null;
					this._transformVisibleMetrics = null;
					
					return;
				}
			}
			
			if (done == true)
				break;
			
			this._transformVisibleMetrics._x += parent._x;
			this._transformVisibleMetrics._y += parent._y;
			
			this._transformDrawableMetrics._x += parent._x;
			this._transformDrawableMetrics._y += parent._y;

			shadowSize = parent.getStyle("ShadowSize");
			
			//Expand metrics for shadow
			if (shadowSize > 0 && parent.getStyle("ShadowColor") != null)
			{
				//Copy drawable metrics
				var shadowMetrics = this._transformDrawableMetrics.clone();
				
				//Create shadow position metrics
				shadowMetrics._width += (shadowSize * 2);
				shadowMetrics._height += (shadowSize * 2);
				shadowMetrics._x -= shadowSize;
				shadowMetrics._y -= shadowSize;
				shadowMetrics._x += parent.getStyle("ShadowOffsetX");
				shadowMetrics._y += parent.getStyle("ShadowOffsetY");
				
				//Merge the shadow metrics with the drawable metrics
				this._transformDrawableMetrics.mergeExpand(shadowMetrics);
			}
			
			if (CanvasElement.normalizeDegrees(parent._rotateDegrees) != 0)
			{
				//Transform visible
				pointTl.x = this._transformVisibleMetrics._x;
				pointTl.y = this._transformVisibleMetrics._y;
				
				pointTr.x = this._transformVisibleMetrics._x + this._transformVisibleMetrics._width;
				pointTr.y = this._transformVisibleMetrics._y;

				pointBr.x = this._transformVisibleMetrics._x + this._transformVisibleMetrics._width;
				pointBr.y = this._transformVisibleMetrics._y + this._transformVisibleMetrics._height;
				
				pointBl.x = this._transformVisibleMetrics._x;
				pointBl.y = this._transformVisibleMetrics._y + this._transformVisibleMetrics._height;
				
				parent.rotatePoint(pointTl, false);
				parent.rotatePoint(pointTr, false);
				parent.rotatePoint(pointBl, false);
				parent.rotatePoint(pointBr, false);
				
				minX = Math.min(pointTl.x, pointTr.x, pointBr.x, pointBl.x);
				maxX = Math.max(pointTl.x, pointTr.x, pointBr.x, pointBl.x);
				minY = Math.min(pointTl.y, pointTr.y, pointBr.y, pointBl.y);
				maxY = Math.max(pointTl.y, pointTr.y, pointBr.y, pointBl.y);
				
				this._transformVisibleMetrics._x = minX;
				this._transformVisibleMetrics._y = minY;
				this._transformVisibleMetrics._width = maxX - minX;
				this._transformVisibleMetrics._height = maxY - minY;
				
				this._transformVisibleMetrics.roundToPrecision(3);
				
				//Transform Drawable
				pointTl.x = this._transformDrawableMetrics._x;
				pointTl.y = this._transformDrawableMetrics._y;
				
				pointTr.x = this._transformDrawableMetrics._x + this._transformDrawableMetrics._width;
				pointTr.y = this._transformDrawableMetrics._y;
				
				pointBr.x = this._transformDrawableMetrics._x + this._transformDrawableMetrics._width;
				pointBr.y = this._transformDrawableMetrics._y + this._transformDrawableMetrics._height;
				
				pointBl.x = this._transformDrawableMetrics._x;
				pointBl.y = this._transformDrawableMetrics._y + this._transformDrawableMetrics._height;
				
				parent.rotatePoint(pointTl, false);
				parent.rotatePoint(pointTr, false);
				parent.rotatePoint(pointBl, false);
				parent.rotatePoint(pointBr, false);
				
				minX = Math.min(pointTl.x, pointTr.x, pointBr.x, pointBl.x);
				maxX = Math.max(pointTl.x, pointTr.x, pointBr.x, pointBl.x);
				minY = Math.min(pointTl.y, pointTr.y, pointBr.y, pointBl.y);
				maxY = Math.max(pointTl.y, pointTr.y, pointBr.y, pointBl.y);
				
				this._transformDrawableMetrics._x = minX;
				this._transformDrawableMetrics._y = minY;
				this._transformDrawableMetrics._width = maxX - minX;
				this._transformDrawableMetrics._height = maxY - minY;
				
				this._transformDrawableMetrics.roundToPrecision(3);
			}
			
			parent = parent._parent;
			
			if (parent._isCompositeElement() == true)
				done = true;
		}
	};	
	
//@private
CanvasElement.prototype._validateCompositeRender = 
	function ()
	{
		if (this._isCompositeElement() == true)
		{
			this._updateCompositeCanvas();
			
			if (this._redrawRegionMetrics != null && this._compositeCanvasMetrics != null)
			{
				//Add a 1 pixel buffer to the redraw region. 
				//This accounts for rounding errors considering redraw regions are calculated per element
				//and composite layers calculated are rendered as an aggregate.
				this._redrawRegionMetrics._x -= 1;
				this._redrawRegionMetrics._y -= 1;
				this._redrawRegionMetrics._width += 2;
				this._redrawRegionMetrics._height += 2;
				this._redrawRegionMetrics.roundUp();
				
				//Composite canvas may have shrunk to smaller than the redraw region, adjust the redraw region
				this._redrawRegionMetrics.mergeReduce(this._compositeCanvasMetrics);
				
				if (this._redrawRegionMetrics._width > 0 && this._redrawRegionMetrics._height > 0)
				{
					this._compositeCtx.clearRect(this._redrawRegionMetrics._x, this._redrawRegionMetrics._y, this._redrawRegionMetrics._width, this._redrawRegionMetrics._height);
					this._renderRedrawRegion(this);
				}
			}
		}
		else //No longer a composite element, nuke the compositing canvases
		{
			this._compositeCanvas = null;				
			this._compositeCtx = null;				
			this._compositeCanvasMetrics = null;		
		}
		
		this._compositeRenderInvalid = false;
		this._compositeEffectChanged = false;
		this._redrawRegionMetrics = null;
	};	
	
//@private	
CanvasElement.prototype._updateCompositeCanvas = 
	function ()
	{
		if (this._compositeVisibleMetrics == null)
		{
			this._compositeCanvas = null;
			this._compositeCtx = null;
			this._compositeCanvasMetrics = null;
		}
		else 
		{
			var newMetrics = this._compositeVisibleMetrics.clone();
			newMetrics.roundUp();
			
			if (this._compositeCanvas == null || this._compositeCanvasMetrics.equals(newMetrics) == false)
			{
				if (this._compositeCanvas == null)
				{
					this._compositeCanvas = document.createElement("canvas");
					this._compositeCtx = this._compositeCanvas.getContext("2d");
				}
				
				if (this._compositeCanvasMetrics == null)
					this._compositeCanvasMetrics = newMetrics;
				else
					this._compositeCanvasMetrics.copyFrom(newMetrics);
				
				this._compositeCanvas.width = this._compositeCanvasMetrics._width;
				this._compositeCanvas.height = this._compositeCanvasMetrics._height;
				
				//Translate x / y
				this._compositeCtx.setTransform(1, 0, 0, 1, this._compositeCanvasMetrics._x * -1, this._compositeCanvasMetrics._y * -1);
				
				//Expand the redraw region to this whole canvas.
				if (this._redrawRegionMetrics == null)
					this._redrawRegionMetrics = this._compositeCanvasMetrics.clone();
				else
					this._redrawRegionMetrics.mergeExpand(this._compositeCanvasMetrics);
			}
		}
	};
	
//@private	
CanvasElement.prototype._isCompositeElement = 
	function ()
	{
		if (this == this._manager)
			return true;
		
		var alpha = this.getStyle("Alpha");
		if (alpha > 0 && alpha < 1)
			return true;
		
		var rotateDegrees = CanvasElement.normalizeDegrees(this._rotateDegrees);
		if (rotateDegrees != 0)
			return true;
		
		if (this.getStyle("ShadowSize") > 0 && this.getStyle("ShadowColor") != null)
			return true;
		
		if (this.getStyle("CompositeLayer") == true)
			return true;
		
		return false;
	};
	
//@private	
CanvasElement.prototype._updateRedrawRegion = 
	function (changedMetrics)
	{
		if (changedMetrics == null || changedMetrics._width <= 0 || changedMetrics._height <= 0)
			return;
	
		if (this._redrawRegionMetrics == null)
			this._redrawRegionMetrics = changedMetrics.clone();
		else
			this._redrawRegionMetrics.mergeExpand(changedMetrics);
		
		this._invalidateCompositeRender();
	};	
	
//@private - This is here to handle a firefox bug. FF ignores clipping if shadow is applied prior to any other
//			 rendering. We draw one pixel outside the canvas bounds to make sure the clipping region gets applied.
CanvasElement.prototype._primeShadow = 
	function ()
	{
		this._compositeCtx.beginPath();
		this._compositeCtx.moveTo(this._compositeCanvasMetrics._x - 1, this._compositeCanvasMetrics._y - 1);
		this._compositeCtx.moveTo(this._compositeCanvasMetrics._x, this._compositeCanvasMetrics._y - 1);
		this._compositeCtx.moveTo(this._compositeCanvasMetrics._x, this._compositeCanvasMetrics._y);
		this._compositeCtx.moveTo(this._compositeCanvasMetrics._x - 1, this._compositeCanvasMetrics._y);
		this._compositeCtx.closePath();
		
		this._compositeCtx.fillStyle = "#000000";
		this._compositeCtx.fill();
	};
	
//@private	
CanvasElement.prototype._renderRedrawRegion = 
	function (element)
	{
		if (element._renderVisible == false)
			return;
	
		var isCompositeChildElement = false;
		if (this != element && element._isCompositeElement() == true)
			isCompositeChildElement = true;
		
		var elementGraphics = null;
		var drawableMetrics = null;
		var compositeMetrics = null;
		
		if (isCompositeChildElement == true && element._transformDrawableMetrics != null)
		{
			compositeMetrics = element._transformVisibleMetrics;
			drawableMetrics = element._transformDrawableMetrics;
			elementGraphics = element._compositeCanvas;
			
			this._compositeCtx.globalAlpha = element.getStyle("Alpha");
		}
		else if (isCompositeChildElement == false && element._compositeMetrics.length > 0 && element._graphicsClear == false)
		{
			compositeMetrics = element._compositeMetrics[0].metrics;
			drawableMetrics = element._compositeMetrics[0].drawableMetrics;
			elementGraphics = element._graphicsCanvas;
			
			this._compositeCtx.globalAlpha = 1;
		}
		
		if (elementGraphics != null &&
			drawableMetrics._width > 0 && drawableMetrics._height > 0 &&
			drawableMetrics._x < this._redrawRegionMetrics._x + this._redrawRegionMetrics._width && 
			drawableMetrics._x + drawableMetrics._width > this._redrawRegionMetrics._x && 
			drawableMetrics._y < this._redrawRegionMetrics._y + this._redrawRegionMetrics._height && 
			drawableMetrics._y + drawableMetrics._height > this._redrawRegionMetrics._y)
		{
			if (isCompositeChildElement == true && CanvasElement.normalizeDegrees(element._rotateDegrees) != 0)
			{
				var clipMetrics = drawableMetrics.clone();
				clipMetrics.mergeReduce(this._redrawRegionMetrics);
				
				this._compositeCtx.save();
				
				//Clip the region we need to re-draw
				this._compositeCtx.beginPath();
				this._compositeCtx.moveTo(clipMetrics._x, clipMetrics._y);
				this._compositeCtx.lineTo(clipMetrics._x + clipMetrics._width, clipMetrics._y);
				this._compositeCtx.lineTo(clipMetrics._x + clipMetrics._width, clipMetrics._y + clipMetrics._height);
				this._compositeCtx.lineTo(clipMetrics._x, clipMetrics._y + clipMetrics._height);
				this._compositeCtx.closePath();
				this._compositeCtx.clip();
				
				//Translate canvas to actual position
				var parent = element._parent;
				while (parent._isCompositeElement() == false)
				{
					this._compositeCtx.translate(parent._x, parent._y);
					parent = parent._parent;
				} 
				
				//Do rotation
				this._compositeCtx.translate(element._rotateCenterX, element._rotateCenterY);
				this._compositeCtx.rotate(CanvasElement.degreesToRadians(element._rotateDegrees));
				this._compositeCtx.translate(-element._rotateCenterX, -element._rotateCenterY);
			
				//Account for composite canvas translation.
				this._compositeCtx.translate(element._compositeCanvasMetrics._x, element._compositeCanvasMetrics._y);
				
				//Handle shadow
				if (element.getStyle("ShadowSize") > 0 && element.getStyle("ShadowColor") != null)
				{
					//Handle firefox bug.
					this._primeShadow();
					
					this._compositeCtx.shadowBlur = element.getStyle("ShadowSize");
					this._compositeCtx.shadowColor = element.getStyle("ShadowColor");
					
					//We need to rotate the shadow to match the element's rotation
					var shadowOffsetX = element.getStyle("ShadowOffsetX");
					var shadowOffsetY = element.getStyle("ShadowOffsetY");
					var radius = Math.sqrt((shadowOffsetX * shadowOffsetX) + (shadowOffsetY * shadowOffsetY));
					var degrees = 360 - element._rotateDegrees + CanvasElement.radiansToDegrees(Math.atan2(shadowOffsetX, shadowOffsetY));
					
					this._compositeCtx.shadowOffsetX = Math.sin(CanvasElement.degreesToRadians(degrees)) * radius;
					this._compositeCtx.shadowOffsetY = Math.cos(CanvasElement.degreesToRadians(degrees)) * radius;
				}
				
				this._compositeCtx.drawImage(
					elementGraphics, 
					0, 0, element._compositeCanvasMetrics._width, element._compositeCanvasMetrics._height, 
					element._x, element._y, element._compositeCanvasMetrics._width, element._compositeCanvasMetrics._height);
				
				this._compositeCtx.restore();
			}
			else if (isCompositeChildElement == true && element.getStyle("ShadowSize") > 0 && element.getStyle("ShadowColor") != null)
			{
				this._compositeCtx.save();
				
				var clipMetrics = drawableMetrics.clone();
				clipMetrics.mergeReduce(this._redrawRegionMetrics);
				
				//Clip the region we need to re-draw
				this._compositeCtx.beginPath();
				this._compositeCtx.moveTo(clipMetrics._x, clipMetrics._y);
				this._compositeCtx.lineTo(clipMetrics._x + clipMetrics._width, clipMetrics._y);
				this._compositeCtx.lineTo(clipMetrics._x + clipMetrics._width, clipMetrics._y + clipMetrics._height);
				this._compositeCtx.lineTo(clipMetrics._x, clipMetrics._y + clipMetrics._height);
				this._compositeCtx.closePath();
				this._compositeCtx.clip();
				
				//Handle firefox bug.
				this._primeShadow();
				
				this._compositeCtx.shadowBlur = element.getStyle("ShadowSize");
				this._compositeCtx.shadowColor = element.getStyle("ShadowColor");
				this._compositeCtx.shadowOffsetX = element.getStyle("ShadowOffsetX");
				this._compositeCtx.shadowOffsetY = element.getStyle("ShadowOffsetY");
				
				//Account for canvas edge buffer (visible - canvas)
				var destX = compositeMetrics._x - (element._compositeVisibleMetrics._x - element._compositeCanvasMetrics._x);
				var destY = compositeMetrics._y - (element._compositeVisibleMetrics._y - element._compositeCanvasMetrics._y);
				
				this._compositeCtx.drawImage(
						elementGraphics, 
						0, 0, element._compositeCanvasMetrics._width, element._compositeCanvasMetrics._height, 
						destX, destY, element._compositeCanvasMetrics._width, element._compositeCanvasMetrics._height);
				
				this._compositeCtx.restore();
			}
			else
			{
				var sourceX = 0;
				var sourceY = 0;
				
				var destX = 0;
				var destY = 0;

				var width = 0;
				var height = 0;
				
				if (drawableMetrics._x >= this._redrawRegionMetrics._x)
				{
					sourceX = drawableMetrics._x - compositeMetrics._x;
					destX = drawableMetrics._x;
					
					width = Math.min(drawableMetrics._width,  this._redrawRegionMetrics._width - (drawableMetrics._x - this._redrawRegionMetrics._x));
				}
				else
				{
					sourceX = this._redrawRegionMetrics._x - compositeMetrics._x;
					destX = this._redrawRegionMetrics._x;
					
					width = Math.min(this._redrawRegionMetrics._width,  drawableMetrics._width - (this._redrawRegionMetrics._x - drawableMetrics._x));
				}
				
				if (drawableMetrics._y >= this._redrawRegionMetrics._y)
				{
					sourceY = drawableMetrics._y - compositeMetrics._y;
					destY = drawableMetrics._y;
					
					height = Math.min(drawableMetrics._height,  this._redrawRegionMetrics._height - (drawableMetrics._y - this._redrawRegionMetrics._y));
				}
				else
				{
					sourceY = this._redrawRegionMetrics._y - compositeMetrics._y;
					destY = this._redrawRegionMetrics._y;
					
					height = Math.min(this._redrawRegionMetrics._height,  drawableMetrics._height - (this._redrawRegionMetrics._y - drawableMetrics._y));
				}
				
				this._compositeCtx.drawImage(
					elementGraphics, 
					sourceX, sourceY, width, height, 
					destX, destY, width, height);
			}
		}
		
		if (isCompositeChildElement == false)
		{
			for (var i = 0; i < element._children.length; i++)
				this._renderRedrawRegion(element._children[i]);
		}
	};			
	
/**
 * @function _invalidateStyle
 * Invalidates the supplied style and causes the system to invoke the _doStylesChanged() function.
 * This should never concievably be called and is exclusively handled by the system. 
 * This is the starting point for the element lifecycle. If you're program is dependent on 
 * _doStylesChanged() when no styles have actually changed, you might have a design issue. 
 * Do not override this function.
 * 
 * @param styleName String
 * String representing the style to be invalidated.
 */	
CanvasElement.prototype._invalidateStyle = 
	function (styleName)
	{
		if (this._stylesInvalid == false)
		{
			this._stylesInvalid = true;
			
			if (this._manager != null)
				this._manager._updateStylesQueue.addNode(this._stylesValidateNode, this._displayDepth);
		}
		
		this._stylesInvalidMap[styleName] = true;
	};	
	
/**
 * @function _invalidateMeasure
 * Invalidates the element's measured sizes and causes the system to invoke 
 * doMeasure() on the next measure phase. The system calls this automatically for all
 * existing components, this is only necessary for custom component development. 
 * This should only be called when a change is made
 * to the element that will impact its measurement. Such as property changes or from within the elements doStylesUpdated() 
 * when a style change impacts the elements measurement (such as Padding).  On rare occasions
 * you may need to re-invalidate measurement from within the doMeasure() function (such as wrapping text).
 * This will cause a 2nd measurement pass for that element which is valid (but expensive).
 * Do not override this function.
 */	
CanvasElement.prototype._invalidateMeasure = 
	function()
	{
		//Only invalidate once.
		if (this._measureInvalid == false)
		{
			this._measureInvalid = true;
			
			if (this._manager != null)
				this._manager._updateMeasureQueue.addNode(this._measureValidateNode, this._displayDepth);
		}
	};
	
/**
 * @function _invalidateLayout
 * Invalidates the elements child layout and causes the system to invoke doLayout() on
 * the next layout phase. The system calls this automatically for all existing components,
 * this is only necessary for custom component development.
 * This should only be called when a change is made to the element that will impact its layout.
 * Such as property changes or from within the elements doStylesUpdated() when a style change impacts
 * the element's layout (such as Padding). On rare occasions you may need to re-invalidate layout
 * from within the doLayout() function. An example is when a DataList adds a new DataRenderer it
 * re-invalidates layout to allow the renderer to measure before continuing layout.
 * Do not override this function.
 */	
CanvasElement.prototype._invalidateLayout = 
	function()
	{
		//Only invalidate once.
		if (this._layoutInvalid == false)
		{
			this._layoutInvalid = true;
			
			if (this._manager != null)
				this._manager._updateLayoutQueue.addNode(this._layoutValidateNode, this._displayDepth);
		}
	};	
	
/**
 * @function _invalidateRender
 * Invalidates the elements child render and causes the system to invoke doRender() on
 * the next render phase. The system calls this automatically for all existing components,
 * this is only necessary for custom component development.
 * This should only be called when a change is made to the element that will impact its rendering.
 * Such as property changes or from within the elements doStylesUpdated() when a style change impacts
 * the element's rendering (such as BackgroundFill). 
 * Do not override this function.
 */	
CanvasElement.prototype._invalidateRender =
	function ()
	{
		//Only invalidate once.
		if (this._renderInvalid == false)
		{
			this._renderInvalid = true;
			
			if (this._manager != null)
				this._manager._updateRenderQueue.addNode(this._renderValidateNode, this._displayDepth);
		}
	};	
	
//@private	
CanvasElement.prototype._invalidateRedrawRegion = 
	function ()
	{
		if (this._redrawRegionInvalid == false)
		{
			this._redrawRegionInvalid = true;
			
			if (this._manager != null)
				this._manager._updateRedrawRegionQueue.addNode(this._redrawRegionValidateNode, this._displayDepth);
		}
	};	

//@private - only called from within validateRedrawRegion()	
CanvasElement.prototype._invalidateTransformRegion = 
	function ()
	{
		this._manager._updateTransformRegionQueue.addNode(this._transformRegionValidateNode, this._displayDepth);
	};
	
//@private
CanvasElement.prototype._invalidateCompositeRender = 
	function ()
	{
		if (this._compositeRenderInvalid == false)
		{
			this._compositeRenderInvalid = true;
			
			if (this._manager != null)
				this._manager._compositeRenderQueue.addNode(this._compositeRenderValidateNode, this._displayDepth);
		}
	};
	
/**
 * @function _doStylesUpdated
 * Lifecycle method for custom component development. Never call this function directly. The system
 * calls this function in response to style changes or elements being added to the display hierarchy.
 * Override this function to handle style changes to the element. Style changes may impact other styles,
 * event listeners, layout, measurement, rendering, etc. You should call appropriate _invalidate() method per style.
 * Note that style handling should not be dependent on the *current* state of a style. This function
 * should be able to run repeatedly with the same values. An example of when this may happen is if
 * an element is temporarily removed from the display hierarchy then added back. Whenever an element is added
 * all of its styles are invalidated. This is necessary as when an element is *not* part of the display chain
 * style changes are not tracked. So when an element is added there is no way to know which styles may have changed
 * hence all of them are invalidated. 
 * 
 * @param stylesMap Object
 * An Object map containing all the style names that have been updated. Map contains entries {styleName:true}.
 * Elements should typically check this map before assuming a style has changed for performance reasons. 
 * You should always call the base method. If you wish to override the style handling behavior of a specific style, 
 * you can simply delete it from the map before passing it to the base function.
 */	
CanvasElement.prototype._doStylesUpdated = 
	function (stylesMap)
	{
		if (this._parent != null && 
			(this._parent._measureInvalid == false || this._parent._layoutInvalid == false))
		{
			if ("X" in stylesMap ||
				"Y" in stylesMap ||
				"Width" in stylesMap ||
				"Height" in stylesMap ||
				"PercentWidth" in stylesMap ||
				"PercentHeight" in stylesMap ||
				"MinWidth" in stylesMap ||
				"MinHeight" in stylesMap ||
				"MaxWidth" in stylesMap ||
				"MaxHeight" in stylesMap ||
				"Top" in stylesMap ||
				"Left" in stylesMap ||
				"Bottom" in stylesMap ||
				"Right" in stylesMap ||
				"HorizontalCenter" in stylesMap ||
				"VerticalCenter" in stylesMap ||
				"IncludeInLayout" in stylesMap ||
				"RotateDegrees" in stylesMap ||
				"RotateCenterX" in stylesMap ||
				"RotateCenterY" in stylesMap)
			{
				this._parent._invalidateMeasure();
				this._parent._invalidateLayout();
			}
		}
		
		if (this._measureInvalid == false || this._layoutInvalid == false)
		{
			if ("Padding" in stylesMap ||
				"PaddingTop" in stylesMap ||
				"PaddingBottom" in stylesMap ||
				"PaddingLeft" in stylesMap ||
				"PaddingRight" in stylesMap)
			{
				this._invalidateMeasure();
				this._invalidateLayout();
			}
			else if ("IncludeInMeasure" in stylesMap)
				this._invalidateMeasure();
		}
		
		if ("BorderThickness" in stylesMap || 
			"BorderType" in stylesMap || 
			"BorderColor" in stylesMap || 
			(this._renderFocusRing == true && ("FocusColor" in stylesMap || "FocusThickness" in stylesMap)))
		{
			this._invalidateRender();
		}
		
		if ("Visible" in stylesMap ||
			"ClipContent" in stylesMap)
		{
			this._invalidateRedrawRegion();
		}
		
		if ("Alpha" in stylesMap ||
			"ShadowSize" in stylesMap)
		{
			this._compositeEffectChanged = true;
			this._invalidateRedrawRegion();
			this._invalidateCompositeRender();
		}
		
		if ("BackgroundShape" in stylesMap)
		{
			var bgShape = this.getStyle("BackgroundShape");
			
			//Only handle if changed, attached/detached handles initial add/remove listener.
			if (bgShape != this._backgroundShape)
			{
				if (this._backgroundShape != null)
					this._backgroundShape.removeEventListener("stylechanged", this._onBackgroundShapeStyleChangedInstance);
				
				this._backgroundShape = bgShape;
				
				if (this._backgroundShape != null)
					this._backgroundShape.addEventListener("stylechanged", this._onBackgroundShapeStyleChangedInstance);
				
				this._invalidateRender();
			}
		}
		
		if ("BackgroundFill" in stylesMap)
		{
			var bgFill = this.getStyle("BackgroundFill"); //FillBase or color string
			
			//Only handle if changed, attached/detached handles initial add/remove listener.
			if (bgFill != this._backgroundFill)
			{
				//Check if new fill is solid (SolidFill or color string)
				var isSolidFillOrColor = false;
				if ((bgFill != null && bgFill instanceof FillBase == false) || bgFill instanceof SolidFill) //We're a color or a SolidFill class
					isSolidFillOrColor = true;
				
				if (this._backgroundFill instanceof SolidFill == true && isSolidFillOrColor == true) //Existing and new are both SolidFill
				{
					if (bgFill instanceof SolidFill == true) //Swap the solid fill classes
					{
						this._backgroundFill.removeEventListener("stylechanged", this._onBackgroundFillStyleChangedInstance);
						this._backgroundFill = bgFill;
						this._backgroundFill.addEventListener("stylechanged", this._onBackgroundFillStyleChangedInstance);
						
						this._invalidateRender();
					}
					else //Set the color to the current SolidFill
						this._backgroundFill.setStyle("FillColor", bgFill); //Will invalidate render if fill color changed
				}
				else //Definately different fill classes
				{
					//Purge the old background fill
					if (this._backgroundFill != null)
						this._backgroundFill.removeEventListener("stylechanged", this._onBackgroundFillStyleChangedInstance);
					
					this._backgroundFill = null;
					
					if (bgFill != null)
					{
						if (bgFill instanceof FillBase == false) //color
						{
							//Create new solid fill
							this._backgroundFill = new SolidFill();
							this._backgroundFill.setStyle("FillColor", bgFill);
						}
						else //Fill class
							this._backgroundFill = bgFill;
						
						this._backgroundFill.addEventListener("stylechanged", this._onBackgroundFillStyleChangedInstance);
					}
					
					this._invalidateRender();
				}
			}
		}
		
		if ("Cursor" in stylesMap)
		{
			var cursorDef = this.getStyle("Cursor");
			
			if (cursorDef == null && this.hasEventListener("rollover", this._onCanvasElementCursorOverOutInstance) == true)
			{
				this.removeEventListener("rollover", this._onCanvasElementCursorOverOutInstance);
				this.removeEventListener("rollout", this._onCanvasElementCursorOverOutInstance);
			}
			else if (cursorDef != null && this.hasEventListener("rollover", this._onCanvasElementCursorOverOutInstance) == false)
			{
				this.addEventListener("rollover", this._onCanvasElementCursorOverOutInstance);
				this.addEventListener("rollout", this._onCanvasElementCursorOverOutInstance);
			}
			
			this._updateRolloverCursorDefinition();
		}
		
		//Kill focus ring if disabled.
		if ("Enabled" in stylesMap && this.getStyle("Enabled") == false)
			this._setRenderFocusRing(false);
		
		if ("Visible" in stylesMap || "MouseEnabled" in stylesMap)
			this._manager._rollOverInvalid = true;
	};

//@private	
CanvasElement.prototype._updateRolloverCursorDefinition = 
	function ()
	{
		var cursorDef = this.getStyle("Cursor");
		
		if (this._rollOverCursorInstance != null && 
			(cursorDef == null || this._mouseIsOver == false))
		{
			this._manager.removeCursor(this._rollOverCursorInstance);
			this._rollOverCursorInstance = null;
		}
		else if (this._mouseIsOver == true && cursorDef != null)
		{
			if (this._rollOverCursorInstance == null)
				this._rollOverCursorInstance = this._manager.addCursor(cursorDef, 0);
			else if (this._rollOverCursorInstance.data != cursorDef)
			{
				this._manager.removeCursor(this._rollOverCursorInstance);
				this._rollOverCursorInstance = this._manager.addCursor(cursorDef, 0);
			}
		}
	};

/**
 * @function _doMeasure
 * Lifecycle method for custom component development. Never call this function directly. The system
 * calls this function in response to changes that effect measurement or elements being added to the display hierarchy.
 * Override this function to calculate the measured size of the element based on its styling, children, etc. 
 * You should set the elements measured size by calling _setMeasuredSize() from within this function.
 * 
 * @param padWidth Number
 * Simply a convienence as padding typically effects measurement (but not always) depending on the component.
 * Use any supporting functions such as _getBorderThickness that are needed to measure the element.
 * 
 * @param padHeight Number
 * Simply a convienence as padding typically effects measurement (but not always) depending on the component.
 * Use any supporting functions such as _getBorderThickness that are needed to measure the element.
 */		
CanvasElement.prototype._doMeasure = 
	function (padWidth, padHeight)
	{
		//Stub for override.
	
		this._setMeasuredSize(padWidth, padHeight);
	};
	
/**
 * @function _doLayout
 * Lifecycle method for custom component development. Never call this function directly. The system
 * calls this function in response to changes that effect layout or elements being added to the display hierarchy.
 * Override this function to layout its children. You should call layout functions such as _setActualPosition() and
 * _setActualSize() on child elements from within this function. Sometimes you may need to add/remove additional elements
 * during layout such as when a Datagrid or Viewport adds/removes scroll bars. Keep in mind that adding/removing elements
 * automatically re-invalidates layout, so its best to bail out and wait for the next layout pass for best performance
 * under this scenario.
 * 
 * @param paddingMetrics DrawMetrics
 * Contains metrics to use after considering the element's padding. Simply a convienence as padding typically 
 * effects layout (but not always) depending on the component. Use any supporting functions such as _getBorderMetrics()
 * that are needed to layout the element.
 */		
CanvasElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		//Stub for override.
	};	
	
/**
 * @function _doRender
 * Lifecycle method for custom component development. Never call this function directly. The system
 * calls this function in response to changes that effect rendering or elements being added to the display hierarchy.
 * Override this function to render the element. Keep in mind that any child elements are rendered on top of this
 * element. This function is typically not needed unless rendering a very primitive object such as a custom skin or 
 * a display object that cannot be handled via a BackgroundShape class and/or _fillBackground() function.
 * Most advanced objects should use skins and other means to render themselves.
 * 
 * CanvasElement draws its background and border via supplied shape and gradient settings.
 */		
CanvasElement.prototype._doRender =
	function()
	{
		var borderMetrics = this._getBorderMetrics();
		
		if (this._renderFocusRing == true)
		{
			//ctx.save();
			//this._drawFocusRing(borderMetrics);
			//ctx.restore();
		}
		
		this._fillBackground(borderMetrics);
		
		this._drawBorder(borderMetrics);
	};
	
	
//////////DataRenderer Dynamic Properties/////////////////
	
/**
 * @function _setListData
 * This function is called when the element is being used as a DataRenderer for containers
 * like DataList and DataGrid. Override this function to make any changes to the DataRenderer
 * per the supplied DataListData and itemData objects. Update any styles, states, add/remove children, call any
 * necessary _invalidate() functions, etc.
 * 
 * @param listData DataListData
 * A DataListData or subclass object passed by the parent DataList or subclass with data necessary
 * to update the DataRender like the parent DataList reference and row/column index being rendered.
 * 
 * @param itemData Object
 * The data Object (such as row data) supplied by the implementor's ListCollection to render the row/column DataRenderer.
 */	
CanvasElement.prototype._setListData = 
	function (listData, itemData)
	{
		this._listData = listData;
		this._itemData = itemData;
	};

/**
 * @function _setListSelected
 * This function is called when the element is being used as a DataRenderer for containers
 * like DataList and DataGrid to change the DataRenderer's selection state. 
 * Override this function to make any changes to the DataRenderer per the selection state.
 * Update any styles, states, add/remove children, call any necessary _invalidate() functions, etc.
 * 
 * @param selectedData Any
 * Selected data, differs per container.
 */	
CanvasElement.prototype._setListSelected = 
	function (selectedData)
	{
		this._listSelected = selectedData;
	};	
	
	


/**
 * @depends CanvasElement.js
 */

/////////////////////////////////////////////////////////
/////////////////TextFieldElement////////////////////////

/**
 * @class TextFieldElement
 * @inherits CanvasElement
 * 
 * Internal class used for consistently rendering text used by controls like TextElement, TextInput, and TextArea.
 * You typically should not use this class directly it is designed to be wrapped by a higher level control. 
 * This class allows text to be selected and edited, it renders a text position caret, watches
 * focus/mouse/keyboard events, maintains position of individual characters, and allows copy/cut/paste.
 * 
 * TextField also normalizes text width. The canvas natively will give
 * different widths for strings than when measuring and adding character widths 
 * which will not work for highlighting or editing. 
 * 
 * 
 * @constructor TextFieldElement 
 * Creates new TextFieldElement instance.
 */
function TextFieldElement()
{
	TextFieldElement.base.prototype.constructor.call(this);
	
	//Element used as the blinky text caret when focused.
	this._textCaret = null;
	
	this._textHighlightStartIndex = 0;
	this._caretIndex = 0;
	this._caretEnabled = false;
	this._caretBlinkTime = 0;
	this._caretBlinkVisible = false;
	this._dragScrollTime = 0;
	this._layoutShouldScroll = false;
	
	this._text = "";
	
	this._shiftPressed = false;
	this._charMetrics = null; 	// array of {x, w}
	this._spaceSpans = null; 	// array of {start, end, type} _charMetric positions of spaces for wrapping text.
	
	//Used to detect double / triple click for word / all highlight
	this._lastMouseDownTime1 = 0;
	this._lastMouseDownTime2 = 0;
	this._mouseDownCount = 0; //Change behavior of drag highlight for double / triple
	this._blockHighlightStartRange = null; //For double / triple click + drag we need to know the initial word / line highlight range
	
	//Container clipping
	this._textClipContainer = new CanvasElement();
	this._textClipContainer.setStyle("ClipContent", true);
	this._textClipContainer.setStyle("MouseEnabled", false);
	
		//Container for lines of text, we slide this container around to scroll the block of text when needed.
		this._textLinesContainer = new CanvasElement();
		this._textLinesContainer._addChild(new TextFieldLineElement()); //Always need at least one line
		
	this._textClipContainer._addChild(this._textLinesContainer);	
	this._addChild(this._textClipContainer);
	
	var _self = this;
	
	//Private event handlers, need different instance for each TextField. Proxy to prototype.
	this._onTextFieldFocusEventInstance = 
		function (event)
		{
			if (event.getType() == "focusin")
				_self._onTextFieldFocusIn(event);
			else
				_self._onTextFieldFocusOut(event);
		};
	
	this._onTextFieldKeyInstance = 
		function (keyboardEvent)
		{
			if (keyboardEvent.getType() == "keydown")
				_self._onTextFieldKeyDown(keyboardEvent);
			else if (keyboardEvent.getType() == "keyup")
				_self._onTextFieldKeyUp(keyboardEvent);
		};
		
	this._onTextFieldMouseEventInstance =
		function (mouseEvent)
		{
			if (mouseEvent.getType() == "mousedown")
				_self._onTextFieldMouseDown(mouseEvent);
			else if (mouseEvent.getType() == "mouseup")
				_self._onTextFieldMouseUp(mouseEvent);
		};
	
	this._onTextFieldEnterFrameInstance =
		function (event)
		{
			_self._onTextFieldEnterFrame(event);
		};
	
	this._onTextFieldCopyPasteInstance = 
		function (event)
		{
			window.removeEventListener(event.type, _self._onTextFieldCopyPasteInstance);
		
			try
			{
				if (event.clipboardData)
				{
					if (event.type == "copy")
						_self._onTextFieldCopy(event.clipboardData);
					else if (event.type == "paste")
						_self._onTextFieldPaste(event.clipboardData);
					else
						_self._onTextFieldCut(event.clipboardData);
				}
				
				if (event.preventDefault)
					event.preventDefault();
				
				return false;
			}
			catch (ex)
			{
				//Swallow
			}
		};
		
	this._createCharMetrics();
}

//Inherit from CanvasElement
TextFieldElement.prototype = Object.create(CanvasElement.prototype);
TextFieldElement.prototype.constructor = TextFieldElement;
TextFieldElement.base = CanvasElement;	


/////////////Events///////////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the text is changed due to user interaction.
 */


/////////////Style Types///////////////////////////////

TextFieldElement._StyleTypes = Object.create(null);

/**
 * @style Selectable boolean
 * 
 * When true, the text can be highlighted and copied.
 */
TextFieldElement._StyleTypes.Selectable = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style MaxChars int
 * 
 * The maximum number of characters allowed for this TextField. When 0 unlimited characters are allowed.
 */
TextFieldElement._StyleTypes.MaxChars = 				StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style Multiline boolean
 * 
 * When true, newline characters are respected and text will be rendered on multiple lines if necessary.
 */
TextFieldElement._StyleTypes.Multiline = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style WordWrap boolean
 * 
 * When true, text will wrap when width is constrained and will be rendered on multiple lines if necessary. 
 */
TextFieldElement._StyleTypes.WordWrap = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style MaskCharacter String
 * 
 * When not null, all characters are replaced with the MaskCharacter. 
 */
TextFieldElement._StyleTypes.MaskCharacter = 			StyleableBase.EStyleType.NORMAL;		// true || false


////////////Default Styles////////////////////////////

TextFieldElement.StyleDefault = new StyleDefinition();

TextFieldElement.StyleDefault.setStyle("Selectable", 					false);
TextFieldElement.StyleDefault.setStyle("MaxChars", 						0);
TextFieldElement.StyleDefault.setStyle("Multiline", 					false);
TextFieldElement.StyleDefault.setStyle("WordWrap", 						false);
TextFieldElement.StyleDefault.setStyle("MaskCharacter", 				null);

TextFieldElement.StyleDefault.setStyle("Enabled", 						false);
TextFieldElement.StyleDefault.setStyle("TabStop",						0);
TextFieldElement.StyleDefault.setStyle("Cursor", 						"text");			

TextFieldElement.StyleDefault.setStyle("BorderType", 					"none");
TextFieldElement.StyleDefault.setStyle("PaddingTop", 					1);
TextFieldElement.StyleDefault.setStyle("PaddingBottom",					1);
TextFieldElement.StyleDefault.setStyle("PaddingLeft", 					3);
TextFieldElement.StyleDefault.setStyle("PaddingRight", 					2);
TextFieldElement.StyleDefault.setStyle("BackgroundFill",				null);


////////Public///////////////////////

/**
 * @function setText
 * Sets the text string to be rendered.
 * 
 * @param text String
 * Text string to be rendered
 */
TextFieldElement.prototype.setText = 
	function (text)
	{
		if (text == null)
			text = "";
	
		//Make sure we have an actual string
		if (typeof text !== "string")
		{
			try
			{
				text = text.toString();
			}
			catch (ex)
			{
				text = "";
			}
		}
		
		var maxChars = this.getStyle("MaxChars");
		
		if (maxChars > 0 && text.length > maxChars)
			text = text.substring(0, maxChars);
		
		if (text != this._text)
		{
			this._text = text;
			
			this._createCharMetrics();
			
			this.setSelection(0, 0);
			
			//Reset scroll position
			this._textLinesContainer._setActualPosition(0, 0);
			
			this._invalidateMeasure();
			this._invalidateLayout();
		}
	};

/**
 * @function getText
 * Gets the current text string.
 * 
 * @returns String
 * Current text string.
 */	
TextFieldElement.prototype.getText = 
	function ()
	{
		return this._text;
	};

/**
 * @function setSelection
 * Sets the text selection or text caret position. When startIndex and endIndex are the same
 * it places the text caret at that position, when different, it selects / highlights that range of characters.
 * 
 * @param startIndex int
 * Character index to begin the selection.
 * 
 * @param endIndex int
 * Character index to end the selection.
 */	
TextFieldElement.prototype.setSelection = 
	function (startIndex, endIndex)
	{
		if (startIndex < 0)
			startIndex = 0;
		if (startIndex > this._text.length)
			startIndex = this._text.length;
		
		if (endIndex < 0)
			endIndex = 0;
		if (endIndex > this._text.length)
			endIndex = this._text.length;
		
		if (startIndex == this._textHighlightStartIndex && endIndex == this._caretIndex)
			return;
		
		this._textHighlightStartIndex = startIndex;
		this._caretIndex = endIndex;
		
		if (this._caretEnabled == true && startIndex == endIndex)
		{
			this._caretBlinkVisible = true;
			this._caretBlinkTime = Date.now() + 800;
		}
		
		this._updateCaretVisibility();
		this._invalidateLayout();
	};
	
/**
 * @function getSelection
 * Gets the current text selection or text caret position.
 * 
 * @returns Object
 * Object containing the start and end selection indexes. {startIndex, endIndex}
 */	
TextFieldElement.prototype.getSelection = 
	function ()
	{
		return {startIndex:this._textHighlightStartIndex, endIndex:this._caretIndex};
	};	
	
////////Internal/////////////////////	

/**
 * @function _createTextCaret
 * Generates a CanvasElement to be used as the text caret.
 * 
 * @returns CanvasElement
 * New CanvasElement instance to be used as the text caret.
 */
TextFieldElement.prototype._createTextCaret = 
	function ()
	{
		var textCaret = new CanvasElement();
		textCaret.setStyle("MouseEnabled", false);
		textCaret.setStyle("BackgroundFill", this.getStyle("TextCaretColor"));
		
		return textCaret;
	};
	
//@private
TextFieldElement.prototype._updateCaretVisibility = 
	function ()
	{
		if (this._caretEnabled == true &&
			this._caretBlinkVisible == true && 
			this._caretIndex > -1 && this._caretIndex <= this._text.length) //Dont think this line is necessary
		{
			if (this._textCaret == null)
			{
				this._textCaret = this._createTextCaret();
				this._addChild(this._textCaret);
			}
			
			this._textCaret.setStyle("Visible", true);
		}
		else if (this._textCaret != null)
			this._textCaret.setStyle("Visible", false);
	};
	
//@private - only active when caret is enabled or dragging highlight selection is scrolling.
TextFieldElement.prototype._onTextFieldEnterFrame = 
	function (event)
	{
		var i;
		var currentTime = Date.now();
		
		if (currentTime > this._caretBlinkTime && 
			this._caretEnabled == true &&
			this._caretIndex > -1 && this._caretIndex <= this._text.length) //Dont think this line is necessary
		{	
			if (this._caretBlinkVisible == true)
			{//Shutting off caret
				
				if (this._caretBlinkTime + 400 < currentTime)
					this._caretBlinkTime = currentTime + 400;
				else
					this._caretBlinkTime += 400; 
			}
			else
			{//Turning on caret
				
				if (this._caretBlinkTime + 800 < currentTime)
					this._caretBlinkTime = currentTime + 800;
				else
					this._caretBlinkTime += 800; 
			}
			
			this._caretBlinkVisible = !(this._caretBlinkVisible);
			this._updateCaretVisibility();
		}
		
		//Handle drag highlight / scroll
		if (this._mouseIsDown == true)
		{
			//Get mouse position
			var mousePos = {x:this._manager._mouseX, y:this._manager._mouseY};
			this.translatePointFrom(mousePos, this._manager);
			
			//Get caret index from mouse
			var caretIndex = this._getCaretIndexFromMouse(mousePos.x, mousePos.y);
			var newStartIndex = this._textHighlightStartIndex;
			
			if (this._mouseDownCount > 1) //Handle double (word) and triple (line) click + drag
			{
				var blockRange;
				if (this._mouseDownCount == 2)
					blockRange = this._getWordRangeFromCharIndex(caretIndex);
				else // if (this._mouseDownCount == 3)
					blockRange = this._getLineRangeFromCharIndex(caretIndex);
				
				if (blockRange.start < this._blockHighlightStartRange.start)
				{
					newStartIndex = this._blockHighlightStartRange.end;
					caretIndex = blockRange.start;
				}
				else
				{
					newStartIndex = this._blockHighlightStartRange.start;
					caretIndex = blockRange.end;
				}
			}
			
			//Find the line
			var textFieldLine = this._textLinesContainer._getChildAt(0);
			if (caretIndex > textFieldLine._charMetricsEndIndex)
			{
				for (i = 1; i < this._textLinesContainer._getNumChildren(); i++)
				{
					textFieldLine = this._textLinesContainer._getChildAt(i);
					if (caretIndex >= textFieldLine._charMetricsStartIndex && caretIndex <= textFieldLine._charMetricsEndIndex)
						break;
				}
			}
			
			//Get caret position
			var caretX = this._textLinesContainer._x + textFieldLine._x + (this._charMetrics[caretIndex].x - this._charMetrics[textFieldLine._charMetricsStartIndex].x);
			var caretW = 1;
			var caretY = this._textLinesContainer._y + textFieldLine._y;
			var caretH = textFieldLine._height;
			
			var caretXWithinBounds = false;
			var caretYWithinBounds = false;
			
			if (caretX >= 0 && caretX + caretW <= this._textClipContainer._width)
				caretXWithinBounds = true;
			
			if (caretY + caretH > 0 && caretY < this._textClipContainer._height)
				caretYWithinBounds = true;
			
			//Set selection and scroll immediately
			if ((caretXWithinBounds == true && caretYWithinBounds == true) || 
				(caretYWithinBounds == true && (caretIndex == textFieldLine._charMetricsStartIndex || caretIndex == textFieldLine._charMetricsEndIndex)))
			{
				this.setSelection(newStartIndex, caretIndex);
				
				this._layoutShouldScroll = true;
				this._invalidateLayout();
			}
			else if (currentTime > this._dragScrollTime) //Wait for time expiration
			{
				this._dragScrollTime = currentTime + 220;
				
				this.setSelection(newStartIndex, caretIndex);
				
				this._layoutShouldScroll = true;
				this._invalidateLayout();
			}
		}
	};
	
//@private	
TextFieldElement.prototype._enableCaret = 
	function ()
	{
		if (this._caretEnabled == true)
			return;
	
		this._caretEnabled = true;
		
		this._updateEnterFrameListener();
		this._updateCaretVisibility();
	};
	
//@private	
TextFieldElement.prototype._disableCaret = 
	function ()
	{
		if (this._caretEnabled == false)
			return;
	
		this._caretEnabled = false;
		
		this._updateEnterFrameListener();
		this._updateCaretVisibility();
	};

//@private - Only active if TextField is Enabled or Selectable.
TextFieldElement.prototype._onTextFieldFocusIn = 
	function (elementEvent)
	{
		//Tab focus (mouse would have already set caret)
		if (this._caretEnabled == true || this.getStyle("Enabled") == false)
			return;
	
		this._enableCaret();
		this.setSelection(0, this._text.length);
	};
	
//@private - Only active if TextField is Enabled or Selectable.	
TextFieldElement.prototype._onTextFieldFocusOut = 
	function (event)
	{
		this._disableCaret();
		this.setSelection(0, 0);
	};

/**
 * @function _getLineIndexFromCharIndex
 * Gets the line index from the supplied character index.
 * 
 * @param charIndex int
 * Character index to get the line index from.
 * 
 * @returns int
 * Corresponding line index.
 */	
TextFieldElement.prototype._getLineIndexFromCharIndex = 
	function (charIndex)
	{
		var textFieldLineIndex = 0;	
		var textFieldLine = this._textLinesContainer._getChildAt(0);
		for (var i = 1; i < this._textLinesContainer._getNumChildren(); i++)
		{
			if (textFieldLine._charMetricsEndIndex >= charIndex)
				break;
			
			textFieldLine = this._textLinesContainer._getChildAt(i);
			textFieldLineIndex = i;
		}
		
		return textFieldLineIndex;
	};
	
/**
 * @function _getCaretIndexFromMouse
 * Gets the position to place the text caret based on the position of the mouse.
 * 
 * @param mouseX Number
 * Current X position of the mouse.
 * 
 * @param mouseY Number
 * Current Y position of the mouse.
 * 
 * @returns int
 * Corresponding caret character index.
 */	
TextFieldElement.prototype._getCaretIndexFromMouse = 
	function (mouseX, mouseY)
	{
		if (this._charMetrics == null || this._charMetrics.length == 0 || this._textLinesContainer._getNumChildren() == 0)
			return 0;
	
		var i;
		
		var x = this._textClipContainer._x + this._textLinesContainer._x;
		var y = this._textClipContainer._y + this._textLinesContainer._y;
		mouseX += 2; //Text cursor is slightly offset.
		
		//Find the line 
		//TODO: Account for gap due to line spacing
		var textFieldLine = this._textLinesContainer._getChildAt(0);
		for (i = 1; i < this._textLinesContainer._getNumChildren(); i++)
		{
			if (mouseY < y + textFieldLine._y + textFieldLine._height)
				break;
			
			textFieldLine = this._textLinesContainer._getChildAt(i);
		}
		
		var charX = 0;
		var charW = 0;
		
		//Find the position
		var newCaretIndex = 0;
		for (i = textFieldLine._charMetricsStartIndex; i <= textFieldLine._charMetricsEndIndex; i++)
		{
			charX = (this._charMetrics[i].x - this._charMetrics[textFieldLine._charMetricsStartIndex].x) + x + textFieldLine._x;
			charW = this._charMetrics[i].width;
			
			newCaretIndex = i;
			
			if (mouseX <= charX + (charW / 2))
				break;
		}
		
		return newCaretIndex;
	};

/**
 * @function _getWordRangeFromCaretIndex
 * Returns the exclusive range of character indexes for the current word the character is included in.
 * If the character is a space, a block of spaces will be returned.
 * 
 * @param charIndex int
 * Character index to return word range for.
 */	
TextFieldElement.prototype._getWordRangeFromCharIndex = 
	function (charIndex)
	{
		if (charIndex >= this._text.length)
			return {start:charIndex, end:charIndex};
	
		if (this._text[charIndex] == '\n')
			return {start:charIndex, end:charIndex};
			
		var wordStart = charIndex;
		var wordEnd = charIndex;
		
		var isSpace = false;
		if (this._text[wordStart] == ' ')
			isSpace = true;
		
		if (isSpace == true)
		{
			while (wordStart > 0 && (this._text[wordStart - 1] == ' ' || this._text[wordStart - 1] == '\n'))
				wordStart--;
			
			while (wordEnd < this._text.length - 1 && (this._text[wordEnd + 1] == ' ' || this._text[wordEnd + 1] == '\n'))
				wordEnd++;
		}
		else
		{
			while (wordStart > 0 && (this._text[wordStart - 1] != ' ' && this._text[wordStart - 1] != '\n'))
				wordStart--;
			
			while (wordEnd < this._text.length - 1 && (this._text[wordEnd + 1] != ' ' && this._text[wordEnd + 1] != '\n'))
				wordEnd++;
		}
		
		return {start:wordStart, end:wordEnd + 1};
	};
	
/**
 * @function _getLineRangeFromCharIndex
 * Returns the exclusive range of character indexes for the current line the character is included in.
 * 
 * @param charIndex int
 * Character index to return line range for.
 */		
TextFieldElement.prototype._getLineRangeFromCharIndex = 
	function (charIndex)
	{
		if (charIndex == this._text.length)
			return {start:charIndex, end:charIndex};
	
		if (this._text[charIndex] == '\n')
			return {start:charIndex, end:charIndex};
			
		var lineStart = charIndex;
		var lineEnd = charIndex;
		
		while (lineStart > 0 && this._text[lineStart - 1] != '\n')
			lineStart--;
		
		while (lineEnd < this._text.length - 1 && this._text[lineEnd + 1] != '\n')
			lineEnd++;
		
		return {start:lineStart, end:lineEnd + 1};
	};
	
/**
 * @function _getVerticalScrollParameters
 * Returns parameters representing the vertical scroll page, view, and line sizes.
 * 
 * @returns Object
 * An object containing both width and height: {page:100, view:100, line:14}
 */	
TextFieldElement.prototype._getVerticalScrollParameters = 
	function ()
	{
		var params = {page:0, view:0, line:14, value:0};
		
		params.page = this._textLinesContainer._height;
		params.view = this._textClipContainer._height;
	
		var textSize = this.getStyle("TextSize");
		var lineSpacing = this.getStyle("TextLineSpacing");
		var linePaddingTop = this.getStyle("TextLinePaddingTop");
		var linePaddingBottom = this.getStyle("TextLinePaddingBottom");
		
		params.line = textSize + linePaddingTop + linePaddingBottom + lineSpacing;
		
		params.value = this._textLinesContainer._y * -1;
		
		return params;
	};
	
/**
 * @function _setVerticalScrollValue
 * Sets the vertical scroll position.
 * 
 * @param value int
 * Y position to scroll too
 */		
TextFieldElement.prototype._setVerticalScrollValue = 
	function (value)
	{
		value = Math.round(value);
		
		this._textLinesContainer._setActualPosition(this._textLinesContainer._x, value * -1);
	};	
	
/**
 * @function _getHorizontalScrollParameters
 * Returns parameters representing the vertical scroll page, view, line, and value sizes.
 * 
 * @returns Object
 * An object containing both width and height: {page:100, view:100, line:14, value:0}
 */	
TextFieldElement.prototype._getHorizontalScrollParameters = 
	function ()
	{
		var params = {page:0, view:0, line:14, value:0};
		
		params.page = this._textLinesContainer._width;
		params.view = this._textClipContainer._width;
	
		var textSize = this.getStyle("TextSize");
		var lineSpacing = this.getStyle("TextLineSpacing");
		var linePaddingTop = this.getStyle("TextLinePaddingTop");
		var linePaddingBottom = this.getStyle("TextLinePaddingBottom");
		
		params.line = textSize + linePaddingTop + linePaddingBottom + lineSpacing;
		
		params.value = this._textLinesContainer._x * -1;
		
		return params;
	};	
	
/**
 * @function _setHorizontalScrollValue
 * Sets the horizontal scroll position.
 * 
 * @param value int
 * X position to scroll too
 */		
TextFieldElement.prototype._setHorizontalScrollValue = 
	function (value)
	{
		value = Math.round(value);
		
		this._textLinesContainer._setActualPosition(value * -1, this._textLinesContainer._y);
	};
	
//@private - Only active if TextField is Enabled or Selectable.		
TextFieldElement.prototype._onTextFieldMouseDown = 
	function (mouseEvent)
	{
		var caretIndex = this._getCaretIndexFromMouse(mouseEvent.getX(), mouseEvent.getY());
		
		if (this.getStyle("Enabled") == true)
			this._enableCaret();
		
		if (this._shiftPressed == true)
			this.setSelection(this._textHighlightStartIndex, caretIndex);
		else
		{
			var currentTime = Date.now();
			var isDoubleClick = false;
			var isTripleClick = false;
			
			if (currentTime - this._lastMouseDownTime1 < 700)
				this._mouseDownCount = 3;
			else if (currentTime - this._lastMouseDownTime2 < 350)
				this._mouseDownCount = 2;
			else
				this._mouseDownCount = 1;
			
			if (this._mouseDownCount == 3)
			{
				var lineRange = this._getLineRangeFromCharIndex(caretIndex);
				this._blockHighlightStartRange = lineRange;
				this.setSelection(lineRange.start, lineRange.end);
			}
			else if (this._mouseDownCount == 2)
			{
				var wordRange = this._getWordRangeFromCharIndex(caretIndex);
				this._blockHighlightStartRange = wordRange;
				this.setSelection(wordRange.start, wordRange.end);
			}
			else
				this.setSelection(caretIndex, caretIndex);
			
			this._lastMouseDownTime1 = this._lastMouseDownTime2;
			this._lastMouseDownTime2 = currentTime;
		}
		
		this._layoutShouldScroll = true;
		this._invalidateLayout();
		
		this._dragScrollTime = 0;
		
		this._updateEnterFrameListener();
	};
	
//@private - Only active if TextField is Enabled or Selectable.		
TextFieldElement.prototype._onTextFieldMouseUp = 
	function (mouseEvent)
	{
		this._mouseDownCount = 0;
		this._blockHighlighStartRange = null;
		this._updateEnterFrameListener();
	};	
	
//@private	
TextFieldElement.prototype._updateCharMetricsAndSpaceSpans = 
	function ()
	{
		if (this._charMetrics == null || this._charMetrics.length == 0)
			return;
		
		if (this._spaceSpans == null)
			this._spaceSpans = [];
		
		var currentPos = 0;
		var currentSpaceSpan = null;	
		var spaceSpanIndex = 0;
		
		for (var i = 0; i < this._charMetrics.length; i++)
		{
			//Update metrics position
			this._charMetrics[i].x = currentPos;
			currentPos += this._charMetrics[i].width;
			
			//Update space spans (re-use existing array items to save memory/GC)
			if (this._text.length > i)
			{
				if (this._text[i] == " ")
				{
					if (currentSpaceSpan == null)
					{
						currentSpaceSpan = this._spaceSpans[spaceSpanIndex];
						if (currentSpaceSpan == null)
						{
							this._spaceSpans[spaceSpanIndex] = {start:i, end:i, type:"space"};
							currentSpaceSpan = this._spaceSpans[spaceSpanIndex];
						}
						else
						{
							currentSpaceSpan.start = i;
							currentSpaceSpan.end = i;
							currentSpaceSpan.type = "space";
						}
						
						spaceSpanIndex++;
					}
					else
						currentSpaceSpan.end = i;
				}
				else 
				{
					if (this._text[i] == '\n')
					{
						currentSpaceSpan = this._spaceSpans[spaceSpanIndex];
						if (currentSpaceSpan == null)
							this._spaceSpans[spaceSpanIndex] = {start:i, end:i, type:"nline"};
						else
						{
							currentSpaceSpan.start = i;
							currentSpaceSpan.end = i;
							currentSpaceSpan.type = "nline";
						}	
						
						spaceSpanIndex++;
					}
					
					currentSpaceSpan = null;
				}
			}
		}
		
		//Purge excess
		if (this._spaceSpans.length > spaceSpanIndex)
			this._spaceSpans.splice(spaceSpanIndex, this._spaceSpans.length - spaceSpanIndex);
	};

//@private	
TextFieldElement.prototype._deleteHighlightChars = 
	function ()
	{
		if (this._textHighlightStartIndex == this._caretIndex)
			return;
	
		var highlightBegin = Math.min(this._caretIndex, this._textHighlightStartIndex);
		var highlightEnd = Math.max(this._caretIndex, this._textHighlightStartIndex);
	
		//Update string
		var strLeft = this._text.substring(0, highlightBegin);
		var strRight = this._text.substring(highlightEnd);
		this._text = strLeft + strRight;
		
		//Fix char metrics
		this._charMetrics.splice(highlightBegin, highlightEnd - highlightBegin);
		
		//Move caret
		this.setSelection(highlightBegin, highlightBegin);
	};	
	
/**
 * @function _onTextFieldKeyUp
 * Event handler for "keyup" event. Only active when TextField is enabled and focused.
 * Handles editing and cursor navigation / selection.
 * 
 * @param keyboardEvent ElementKeyboardEvent
 * ElementKeyboardEvent to process.
 */		
TextFieldElement.prototype._onTextFieldKeyUp = 
	function (keyboardEvent)
	{
		if (keyboardEvent.getDefaultPrevented() == true)
			return;
		
		if (keyboardEvent.getKey() == "Shift")
		{
			this._shiftPressed = false;
		}
	};
	
/**
 * @function _onTextFieldKeyDown
 * Event handler for "keydown" event. Only active when TextField is enabled and focused.
 * Handles editing and cursor navigation / selection.
 * 
 * @param keyboardEvent ElementKeyboardEvent
 * ElementKeyboardEvent to process.
 */	
TextFieldElement.prototype._onTextFieldKeyDown = 
	function (keyboardEvent)
	{
		if (keyboardEvent.getDefaultPrevented() == true)
			return;
	
		var enabled = this.getStyle("Enabled");
		var multiline = this.getStyle("Multiline");
		var maxChars = this.getStyle("MaxChars");
		
		var keyString = keyboardEvent.getKey();
		var dispatchChanged = false;
		var i;
		var textFieldLineIndex;
		var textFieldLine;
		
		if (keyString == "c" && keyboardEvent.getCtrl() == true)
		{
			if (this._textHighlightStartIndex == this._caretIndex)
			{
				keyboardEvent.preventDefault();
				return;
			}
			
			//IE
			if (window.clipboardData)
			{
				this._onTextFieldCopy(window.clipboardData);
				keyboardEvent.preventDefault();
			} 
			else //FF, Chrome, Webkit (Allow keyboard event to invoke the copy / paste listener)
			{
				window.addEventListener("copy", this._onTextFieldCopyPasteInstance);
				this._invalidateLayout(); //Purges the listener if something upstream cancels the keyboard event.
			}
			
			return;
		}
		else if (keyString == "Shift")
		{
			this._shiftPressed = true;
		}
		else if (keyString == "a" && keyboardEvent.getCtrl() == true)
		{
			this.setSelection(0, this._text.length);
		}
		else if (keyString == "ArrowLeft")
		{
			if (keyboardEvent.getShift() == true && 
				(this._textHighlightStartIndex != this._caretIndex || enabled == true))
			{
				this.setSelection(this._textHighlightStartIndex, this._caretIndex - 1);
			}
			else if (enabled == true)
			{
				if (this._textHighlightStartIndex != this._caretIndex)
					this.setSelection(Math.min(this._caretIndex, this._textHighlightStartIndex), Math.min(this._caretIndex, this._textHighlightStartIndex));
				else
					this.setSelection(this._caretIndex - 1, this._caretIndex - 1);
			}
		}
		else if (keyString == "ArrowRight")
		{
			if (keyboardEvent.getShift() == true && 
				(this._textHighlightStartIndex != this._caretIndex || enabled == true))
			{
				this.setSelection(this._textHighlightStartIndex, this._caretIndex + 1);
			}
			else if (enabled == true)
			{
				if (this._textHighlightStartIndex != this._caretIndex)
					this.setSelection(Math.max(this._caretIndex, this._textHighlightStartIndex), Math.max(this._caretIndex, this._textHighlightStartIndex));
				else
					this.setSelection(this._caretIndex + 1, this._caretIndex + 1);
			}
		}
		else if (keyString == "ArrowDown" || keyString == "ArrowUp")
		{
			//Find the current line
			textFieldLineIndex = this._getLineIndexFromCharIndex(this._caretIndex);
			textFieldLine = this._textLinesContainer._getChildAt(textFieldLineIndex);
			
			//Get the new caret index
			var newIndex = -1;
			
			if (keyString == "ArrowUp" && textFieldLineIndex == 0)
				newIndex = 0;
			else if (keyString == "ArrowDown" && textFieldLineIndex == this._textLinesContainer._getNumChildren() - 1)
				newIndex = this._charMetrics.length -1;
			else
			{
				var charX = this._textLinesContainer._x + textFieldLine._x + (this._charMetrics[this._caretIndex].x - this._charMetrics[textFieldLine._charMetricsStartIndex].x);
				
				var moveToTextFieldLine = null;
				
				if (keyString == "ArrowUp")
					moveToTextFieldLine = this._textLinesContainer._getChildAt(textFieldLineIndex - 1);
				else
					moveToTextFieldLine = this._textLinesContainer._getChildAt(textFieldLineIndex + 1);
				
				var moveToCharX;
				var moveToCharW;
				
				for (i = moveToTextFieldLine._charMetricsStartIndex; i <= moveToTextFieldLine._charMetricsEndIndex; i++)
				{
					moveToCharX = this._textLinesContainer._x + moveToTextFieldLine._x + (this._charMetrics[i].x - this._charMetrics[moveToTextFieldLine._charMetricsStartIndex].x);
					moveToCharW = this._charMetrics[i].width;
					
					newIndex = i;
					
					if (charX <= moveToCharX + (moveToCharW / 2))
						break;
				}
			}	
				
			if (keyboardEvent.getShift() == true && (this._textHighlightStartIndex != this._caretIndex || enabled == true))
			{
				this.setSelection(this._textHighlightStartIndex, newIndex);
			}
			else if (enabled == true)
			{
				this.setSelection(newIndex, newIndex);
			}
		}
		else if (keyString == "End")
		{
			if (keyboardEvent.getCtrl() == true)
			{
				if (keyboardEvent.getShift() == true && 
					(this._textHighlightStartIndex != this._caretIndex || enabled == true))
				{
					this.setSelection(this._textHighlightStartIndex, this._text.length);
				}
				else if (enabled == true)
					this.setSelection(this._text.length, this._text.length);
			}
			else
			{
				//Find the current line
				textFieldLineIndex = this._getLineIndexFromCharIndex(this._caretIndex);
				textFieldLine = this._textLinesContainer._getChildAt(textFieldLineIndex);
				
				if (keyboardEvent.getShift() == true && 
					(this._textHighlightStartIndex != this._caretIndex || enabled == true))
				{
					this.setSelection(this._textHighlightStartIndex, textFieldLine._charMetricsEndIndex);
				}
				else if (enabled == true)
					this.setSelection(textFieldLine._charMetricsEndIndex, textFieldLine._charMetricsEndIndex);
			}
		}
		else if (keyString == "Home")
		{
			if (keyboardEvent.getCtrl() == true)
			{
				if (keyboardEvent.getShift() == true && 
					(this._textHighlightStartIndex != this._caretIndex || enabled == true))
				{
					this.setSelection(this._textHighlightStartIndex, 0);
				}
				else if (enabled == true)
					this.setSelection(0, 0);
			}
			else
			{
				//Find the current line
				textFieldLineIndex = this._getLineIndexFromCharIndex(this._caretIndex);
				textFieldLine = this._textLinesContainer._getChildAt(textFieldLineIndex);
				
				if (keyboardEvent.getShift() == true && 
					(this._textHighlightStartIndex != this._caretIndex || enabled == true))
				{
					this.setSelection(this._textHighlightStartIndex, textFieldLine._charMetricsStartIndex);
				}
				else if (enabled == true)
					this.setSelection(textFieldLine._charMetricsStartIndex, textFieldLine._charMetricsStartIndex);
			}
		}
		else if (enabled == false) 
		{
			return;
		}
		else if (keyString == "Backspace")
		{
			if (this._textHighlightStartIndex != this._caretIndex)
				this._deleteHighlightChars();
			else
			{
				if (this._text.length == 0 || this._caretIndex == 0)
				{
					keyboardEvent.preventDefault();
					return;
				}
				
				//Update string
				var strLeft = this._text.substring(0, this._caretIndex - 1);
				var strRight = this._text.substring(this._caretIndex);
				this._text = strLeft + strRight;
				
				//Fix char metrics
				this._charMetrics.splice(this._caretIndex - 1, 1);
				
				//Move caret
				this.setSelection(this._caretIndex - 1, this._caretIndex - 1);
			}
			
			dispatchChanged = true;
		}
		else if (keyString == "Delete")
		{
			if (this._textHighlightStartIndex != this._caretIndex)
				this._deleteHighlightChars();
			else
			{
				if (this._text.length == 0 || this._caretIndex == this._text.length)
				{
					keyboardEvent.preventDefault();
					return;
				}
	
				//Update string
				var strLeft = this._text.substring(0, this._caretIndex);
				var strRight = this._text.substring(this._caretIndex + 1);
				this._text = strLeft + strRight;
				
				//Fix char metrics
				this._charMetrics.splice(this._caretIndex, 1);
			}
			
			dispatchChanged = true;
		}
		else if (keyString == "v" && keyboardEvent.getCtrl() == true)
		{
			//IE
			if (window.clipboardData)
			{
				this._onTextFieldPaste(window.clipboardData);
				keyboardEvent.preventDefault();
			} 
			else //FF, Chrome, Webkit (Allow keyboard event to invoke the copy / paste listener)
			{
				window.addEventListener("paste", this._onTextFieldCopyPasteInstance);
				this._invalidateLayout(); //Purges the listener if something upstream cancels the keyboard event.
			}
			
			return;
		}
		else if (keyString == "x" && keyboardEvent.getCtrl() == true)
		{
			if (this._textHighlightStartIndex == this._caretIndex)
			{
				keyboardEvent.preventDefault();
				return;
			}
			
			//IE
			if (window.clipboardData)
			{
				this._onTextFieldCut(window.clipboardData);
				keyboardEvent.preventDefault();
			} 
			else //FF, Chrome, Webkit (Allow keyboard event to invoke the copy / paste listener)
			{
				window.addEventListener("cut", this._onTextFieldCopyPasteInstance);
				this._invalidateLayout(); //Purges the listener if something upstream cancels the keyboard event.
			}
			
			return;
		}
		else if (keyString == "Enter" && multiline == true)
		{
			this._deleteHighlightChars();
			
			if (maxChars > 0 && maxChars <= this._text.length)
			{
				keyboardEvent.preventDefault();
				return;
			}
			
			//Update string
			var strLeft = this._text.substring(0, this._caretIndex);
			var strRight = this._text.substring(this._caretIndex);
			this._text = strLeft + "\n" + strRight;
			
			//Fix char metrics
			this._charMetrics.splice(this._caretIndex, 0, {x:0, width:CanvasElement._measureText("\n", this._getFontString())});
			
			//Move caret
			this.setSelection(this._caretIndex + 1, this._caretIndex + 1);
			
			dispatchChanged = true;
		}
		else if (keyString.length == 1)
		{
			this._deleteHighlightChars();
			
			if (maxChars > 0 && maxChars <= this._text.length)
			{
				keyboardEvent.preventDefault();
				return;
			}
			
			//Measure new char
			var maskCharacter = this.getStyle("MaskCharacter");
			
			var printCharacter = keyString;
			if (maskCharacter != null)
				printCharacter = maskCharacter;
			
			var newCharMetrics = {x:0, width:CanvasElement._measureText(printCharacter, this._getFontString())};
			
			//Update string
			var strLeft = this._text.substring(0, this._caretIndex);
			var strRight = this._text.substring(this._caretIndex);
			this._text = strLeft + keyString + strRight;
			
			//Fix char metrics
			this._charMetrics.splice(this._caretIndex, 0, newCharMetrics);
			
			//Move caret
			this.setSelection(this._caretIndex + 1, this._caretIndex + 1);
			
			dispatchChanged = true;
		}
		else
			return;
		
		if (dispatchChanged == true)
		{
			this._updateCharMetricsAndSpaceSpans();
			this.dispatchEvent(new ElementEvent("changed", false));
		}
		
		this._layoutShouldScroll = true;
		this._invalidateLayout();
		
		keyboardEvent.preventDefault();
	};

/**
 * @function _onTextFieldCopy
 * Event handler for native browser "copy" event. Copies selected text to clipboard.
 * 
 * @param clipboardData BrowserClipboard
 * The browser clipboard object to copy text too.
 */	
TextFieldElement.prototype._onTextFieldCopy = 
	function (clipboardData)
	{
		var highlightBegin = Math.min(this._caretIndex, this._textHighlightStartIndex);
		var highlightEnd = Math.max(this._caretIndex, this._textHighlightStartIndex);
		
		var copyText = this._text.substring(highlightBegin, highlightEnd);
		
		clipboardData.setData("Text", copyText);
	};
	
/**
 * @function _onTextFieldCopy
 * Event handler for native browser "paste" event. Pastes clipboard text into TextField.
 * 
 * @param clipboardData BrowserClipboard
 * The browser clipboard object to copy text from.
 */		
TextFieldElement.prototype._onTextFieldPaste = 
	function (clipboardData)
	{
		var pasteString = clipboardData.getData("Text");
		
		if (pasteString == null || pasteString.length == 0)
			return;
		
		var maxChars = this.getStyle("MaxChars");
		if (maxChars > 0 && this._text.length >= maxChars && this._caretIndex >= maxChars)
			return;
		
		this._deleteHighlightChars();
		
		var maskCharacter = this.getStyle("MaskCharacter");
		
		//Update string
		var strLeft = this._text.substring(0, this._caretIndex);
		var strRight = this._text.substring(this._caretIndex);
		this._text = strLeft + pasteString + strRight;
		
		//Measure new chars
		var fontString = this._getFontString();
		for (var i = 0; i < pasteString.length; i++)
		{
			var printCharacter = pasteString[i];
			if (maskCharacter != null)
				printCharacter = maskCharacter;
			
			this._charMetrics.splice(this._caretIndex + i, 0, 
					{x:0, width:CanvasElement._measureText(printCharacter, fontString)});
		}

		//Fix char metrics
		this._updateCharMetricsAndSpaceSpans();
		
		//Move caret
		this.setSelection(this._caretIndex + pasteString.length, this._caretIndex + pasteString.length);
		
		//Truncate if exceeding max characters
		if (maxChars > 0 && this._text.length > maxChars)
		{
			this._text = this._text.subString(0, maxChars);
			this._charMetrics.splice(0, this._text.length - maxChars);
			this.setSelection(this._text.length, this._text.length);
		}
		
		this._layoutShouldScroll = true;
		this._invalidateLayout();
		
		this.dispatchEvent(new ElementEvent("changed", false));
	};

/**
 * @function _onTextFieldCut
 * Event handler for native browser "cut" event. Copies selected text to clipboard and deletes from TextField.
 * 
 * @param clipboardData BrowserClipboard
 * The browser clipboard object to copy text too.
 */		
TextFieldElement.prototype._onTextFieldCut = 
	function (clipboardData)
	{
		var highlightBegin = Math.min(this._caretIndex, this._textHighlightStartIndex);
		var highlightEnd = Math.max(this._caretIndex, this._textHighlightStartIndex);
		
		var copyText = this._text.substring(highlightBegin, highlightEnd);
		
		clipboardData.setData("Text", copyText);
		
		this._deleteHighlightChars();
		this._updateCharMetricsAndSpaceSpans();
		
		this._layoutShouldScroll = true;
		this._invalidateLayout();
		
		this.dispatchEvent(new ElementEvent("changed", false));
	};
	
//@Override	
TextFieldElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		TextFieldElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		this._disableCaret();
		
		this._shiftPressed = false;
		this._layoutShouldScroll = false;
		
		if (this.hasEventListener("enterframe", this._onTextFieldEnterFrameInstance) == true)
			this.removeEventListener("enterframe", this._onTextFieldEnterFrameInstance);
	};		
	
//@private	
TextFieldElement.prototype._updateEnterFrameListener = 
	function ()
	{
		if (this._caretEnabled == true || (this._mouseIsDown == true && this.getStyle("Selectable") == true))
		{
			if (this.hasEventListener("enterframe", this._onTextFieldEnterFrameInstance) == false)
				this.addEventListener("enterframe", this._onTextFieldEnterFrameInstance);
		}
		else
		{
			if (this.hasEventListener("enterframe", this._onTextFieldEnterFrameInstance) == true)
				this.removeEventListener("enterframe", this._onTextFieldEnterFrameInstance);
		}
	};
	
/**
 * @function _updateEventListeners
 * Adds removes mouse, keyboard, and focus event listeners based on Enabled and Selectable styles.
 * Called in response to style changes.
 */	
TextFieldElement.prototype._updateEventListeners = 
	function ()
	{
		var enabled = this.getStyle("Enabled");
		var selectable = this.getStyle("Selectable");
		
		if (selectable == true || enabled == true)
		{
			if (this.hasEventListener("keydown", this._onTextFieldKeyInstance) == false)
				this.addEventListener("keydown", this._onTextFieldKeyInstance);
			
			if (this.hasEventListener("keyup", this._onTextFieldKeyInstance) == false)
				this.addEventListener("keyup", this._onTextFieldKeyInstance);
			
			if (this.hasEventListener("mousedown", this._onTextFieldMouseEventInstance) == false)
				this.addEventListener("mousedown", this._onTextFieldMouseEventInstance);
			
			if (this.hasEventListener("mouseup", this._onTextFieldMouseEventInstance) == false)
				this.addEventListener("mouseup", this._onTextFieldMouseEventInstance);
			
			if (this.hasEventListener("focusin", this._onTextFieldFocusEventInstance) == false)
				this.addEventListener("focusin", this._onTextFieldFocusEventInstance);
			
			if (this.hasEventListener("focusout", this._onTextFieldFocusEventInstance) == false)
				this.addEventListener("focusout", this._onTextFieldFocusEventInstance);
		}
		else
		{
			if (this.hasEventListener("keydown", this._onTextFieldKeyInstance) == true)
				this.removeEventListener("keydown", this._onTextFieldKeyInstance);
			
			if (this.hasEventListener("keyup", this._onTextFieldKeyInstance) == true)
				this.removeEventListener("keyup", this._onTextFieldKeyInstance);
			
			if (this.hasEventListener("mousedown", this._onTextFieldMouseEventInstance) == true)
				this.removeEventListener("mousedown", this._onTextFieldMouseEventInstance);
			
			if (this.hasEventListener("mouseup", this._onTextFieldMouseEventInstance) == true)
				this.removeEventListener("mouseup", this._onTextFieldMouseEventInstance);
			
			if (this.hasEventListener("focusin", this._onTextFieldFocusEventInstance) == true)
				this.removeEventListener("focusin", this._onTextFieldFocusEventInstance);
			
			if (this.hasEventListener("focusout", this._onTextFieldFocusEventInstance) == true)
				this.removeEventListener("focusout", this._onTextFieldFocusEventInstance);
		}
		
		if (enabled == false)
			this._disableCaret();
	};

	
//@Override
TextFieldElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		TextFieldElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		////Update line renderers////
		if ("TextStyle" in stylesMap ||
			"TextFont" in stylesMap ||
			"TextSize" in stylesMap ||
			"TextColor" in stylesMap ||
			"TextFillType" in stylesMap || 
			"TextDecoration" in stylesMap ||
			"MaskCharacter" in stylesMap)
		{
			for (var i = 0; i < this._textLinesContainer._getNumChildren(); i++)
				this._textLinesContainer._getChildAt(i)._invalidateRender();
		}
		else if ("TextHighlightedColor" in stylesMap ||
				"TextHighlightedBackgroundColor" in stylesMap)
		{
			for (var i = 0; i < this._textLinesContainer._getNumChildren(); i++)
			{
				//Only re-render if in fact we have a highlighted selection.
				if (this._textLinesContainer._getChildAt(i)._highlightMinIndex != this._textLinesContainer._getChildAt(i)._highlightMaxIndex)
					this._textLinesContainer._getChildAt(i)._invalidateRender();
			}
		}
		
		//Update ourself
		if ("TextStyle" in stylesMap ||
			"TextFont" in stylesMap ||
			"TextSize" in stylesMap ||
			"MaskCharacter" in stylesMap)
		{
			this._createCharMetrics();
			
			this._invalidateMeasure();
			this._invalidateLayout();
		}
		else if ("Multiline" in stylesMap ||
			"WordWrap" in stylesMap ||
			"TextLinePaddingTop" in stylesMap ||
			"TextLinePaddingBottom" in stylesMap)
		{
			this._invalidateMeasure();
			this._invalidateLayout();
		}
		else if ("TextHorizontalAlign" in stylesMap ||
			"TextVerticalAlign" in stylesMap || 
			"TextLineSpacing" in stylesMap)
		{
			this._invalidateLayout();
		}
		
		if ("MaxChars" in stylesMap)
			this.setText(this._text); //Will trim if needed.
		
		if ("TextCaretColor" in stylesMap && this._textCaret != null)
			this._textCaret.setStyle("BackgroundFill", this.getStyle("TextCaretColor"));
		
		if ("Enabled" in stylesMap || "Selectable" in stylesMap)
			this._updateEventListeners();
		
		this._updateEnterFrameListener();
		this._updateCaretVisibility();
	};	
	
//@private	
TextFieldElement.prototype._createCharMetrics = 
	function ()
	{
		var currentX = 0;
		var currentWidth = 0;
		
		this._charMetrics = [];
		this._spaceSpans = [];
		
		var currentSpaceSpan = null;
		
		if (this._text.length > 0)
		{
			var fontString = this._getFontString();	
			
			var maskCharacter = this.getStyle("MaskCharacter");
			var maskCharacterWidth = 0;
			
			if (maskCharacter != null)
				maskCharacterWidth = CanvasElement._measureText(maskCharacter, fontString);
			
			for (var i = 0; i < this._text.length; i++)
			{
				currentWidth = CanvasElement._measureText(this._text[i], fontString);
				
				if (maskCharacter != null && currentWidth > 0)
					currentWidth = maskCharacterWidth;				
				
				this._charMetrics.push(
					{
						x:		currentX,
						width: 	currentWidth
					});
				
				if (this._text[i] == " ")
				{
					if (currentSpaceSpan == null)
						currentSpaceSpan = {start:i, end:i, type:"space"};
					else
						currentSpaceSpan.end = i;
				}
				else if (currentSpaceSpan != null)
				{
					this._spaceSpans.push(currentSpaceSpan);
					currentSpaceSpan = null;
				}
				
				if (this._text[i] == '\n')
					this._spaceSpans.push({start:i, end:i, type:"nline"});
				
				currentX += currentWidth;
			}
		}
		
		if (currentSpaceSpan != null)
			this._spaceSpans.push(currentSpaceSpan);
		
		//Dummy for caret at end of string
		this._charMetrics.push( { x:currentX, width:0 }); 
		this._invalidateLayout();
	};
	
//@Override
TextFieldElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var linePadTop = this.getStyle("TextLinePaddingTop");
		var linePadBottom = this.getStyle("TextLinePaddingBottom");
		var textSize = this.getStyle("TextSize");

		var textWidth = this._charMetrics[this._text.length].x;
		var textHeight = textSize + linePadTop + linePadBottom;		
		
		//If using word wrap, height is dependent on actual width so layout
		//must run and do the actual measurment...
		if (this.getStyle("WordWrap") == true)
		{	
			//We need the parent to know it can contract us.
			textWidth = this.getStyle("MinWidth") - padWidth; //padWidth added back at end
			
			this._invalidateLayout();
		}
		else if (this.getStyle("Multiline") == true)
		{
			var widestLineSize = -1;
			var lineStartIndex = 0;
			var numLines = 1;
			for (var i = 0; i < this._spaceSpans.length; i++)
			{
				//Only care about newline characters
				if (this._spaceSpans[i].type != "nline")
					continue;
				
				if (this._charMetrics[this._spaceSpans[i].start].x - this._charMetrics[lineStartIndex].x > widestLineSize)
					widestLineSize = this._charMetrics[this._spaceSpans[i].start].x - this._charMetrics[lineStartIndex].x;
				
				lineStartIndex = this._spaceSpans[i].start + 1;
				numLines++;
			}
			
			if (numLines > 1)
			{
				//Measure last line
				if (lineStartIndex < this._charMetrics.length - 1)
				{
					if (this._charMetrics[this._charMetrics.length - 1].x - this._charMetrics[lineStartIndex].x > widestLineSize)
						widestLineSize = this._charMetrics[this._charMetrics.length - 1].x - this._charMetrics[lineStartIndex].x;
				}
					
				textWidth = widestLineSize;
					
				textHeight = textHeight * numLines;
				textHeight = textHeight + (this.getStyle("TextLineSpacing") * (numLines - 1));
			}
		}
		
		//Always add 1 for text caret 
		//TODO: This should be the text caret's width only when editable
		this._setMeasuredSize(1 + textWidth + padWidth, textHeight + padHeight);
	};	
	
//@Override	
TextFieldElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		TextFieldElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		//Adjust text x position per scroll / align.
		var availableWidth = w - 1; // account for caret width.
		
		//Size / Position the line container.
		this._textClipContainer._setActualPosition(x, y);
		this._textClipContainer._setActualSize(availableWidth, h);
		
		var isMultiline = this.getStyle("Multiline");
		var isWordWrap = this.getStyle("WordWrap");
		var textHorizontalAlign = this.getStyle("TextHorizontalAlign");
		var textVerticalAlign = this.getStyle("TextVerticalAlign");
		var textSize = this.getStyle("TextSize");
		var lineSpacing = this.getStyle("TextLineSpacing");
		var linePaddingTop = this.getStyle("TextLinePaddingTop");
		var linePaddingBottom = this.getStyle("TextLinePaddingBottom");
		var lineHeight = textSize + linePaddingTop + linePaddingBottom;
		
		var spaceSpanIndex = 0;
		var lineStartCharIndex = 0;
		var lineEndCharIndex = 0;
		
		var newLineData = null;
		var lines = [];
		
		var textHeight = 0;
		var textWidth = 0;
		
		var caretLineIndex = 0;
		var newlineFound = false;
		
		//Calculate lines of text based on multiline, wordwrap, spaces, and newlines.
		while (lineStartCharIndex < this._charMetrics.length)
		{
			newLineData = {charMetricsStartIndex:-1, charMetricsEndIndex:-1};
			
			if (isMultiline == false && isWordWrap == false)
			{
				newLineData.charMetricsStartIndex = 0; 
				newLineData.charMetricsEndIndex = this._charMetrics.length - 1;
				caretLineIndex = 0;
				lineStartCharIndex = this._charMetrics.length;
			}
			else
			{
				newLineData.charMetricsStartIndex = lineStartCharIndex;
				newlineFound = false;
				
				for (var i = spaceSpanIndex; i < this._spaceSpans.length; i++)
				{
					//Ignore spaces if wordwrap is off
					if (this._spaceSpans[i].type == "space" && isWordWrap == false)
					{
						spaceSpanIndex++;
						continue;
					}
					
					if (textHorizontalAlign == "left")
						lineEndCharIndex = this._spaceSpans[i].end;
					else
						lineEndCharIndex = this._spaceSpans[i].start;
					
					if (this._charMetrics[lineEndCharIndex].x - this._charMetrics[newLineData.charMetricsStartIndex].x <= availableWidth ||
						newLineData.charMetricsEndIndex == -1)
					{
						newLineData.charMetricsEndIndex = lineEndCharIndex;
						
						spaceSpanIndex++;
						lineStartCharIndex = lineEndCharIndex + 1;
						
						//Handle newline as space if multiline is off
						if (this._spaceSpans[i].type == "nline" && isMultiline == true)
						{
							newlineFound = true;
							break;
						}
					}
					else
						break;
				}
				
				//Last line, no more spaces for breaks.
				if (newLineData.charMetricsEndIndex == -1 || 
					(this._charMetrics[this._charMetrics.length - 1].x - this._charMetrics[newLineData.charMetricsStartIndex].x <= availableWidth && newlineFound == false))
				{
					newLineData.charMetricsEndIndex = this._charMetrics.length - 1;
					lineStartCharIndex = this._charMetrics.length;
				}
			}
			
			//Record max line width
			if (textWidth < this._charMetrics[newLineData.charMetricsEndIndex].x - this._charMetrics[newLineData.charMetricsStartIndex].x)
				textWidth = this._charMetrics[newLineData.charMetricsEndIndex].x - this._charMetrics[newLineData.charMetricsStartIndex].x;
			
			lines.push(newLineData);
		}
		
		textHeight = (lines.length * lineHeight) + ((lines.length - 1) * lineSpacing); 
		
		//Update the measured size now that we know the height. (May cause another layout pass if causes parent to change our size)
		if (isWordWrap == true)
			this._setMeasuredSize(this._measuredWidth, textHeight + this._getPaddingSize().height);
			
		//Size the lines container to the size of lines of text
		this._textLinesContainer._setActualSize(textWidth, textHeight);
		
		//Update text lines
		
		//Purge excess
		while (this._textLinesContainer._getNumChildren() > lines.length)
			this._textLinesContainer._removeChildAt(this._textLinesContainer._getNumChildren() - 1);
		
		//Update Add
		var textFieldLine = null;
		var lineWidth = 0;
		var lineXPosition;
		var lineYPosition = 0;
		for (var i = 0; i < lines.length; i++)
		{
			if (i < this._textLinesContainer._getNumChildren()) //Update line
				textFieldLine = this._textLinesContainer._getChildAt(i);
			else //Line added
			{
				textFieldLine = new TextFieldLineElement();
				this._textLinesContainer._addChild(textFieldLine);
			}
			
			//Update line
			textFieldLine.setParentLineMetrics(this, lines[i].charMetricsStartIndex, lines[i].charMetricsEndIndex);
			textFieldLine.setParentSelection(this._textHighlightStartIndex, this._caretIndex);
			
			textFieldLine.setStyle("PaddingTop", linePaddingTop);
			textFieldLine.setStyle("PaddingBottom", linePaddingBottom);
			
			lineWidth = textFieldLine.getLineWidth();
			textFieldLine._setActualSize(lineWidth, lineHeight);
			
			//Handle horizontal alignment
			if (textHorizontalAlign == "right")
				lineXPosition = this._textLinesContainer._width - lineWidth;
			else if (textHorizontalAlign == "center")
				lineXPosition = Math.round((this._textLinesContainer._width / 2) - (lineWidth / 2));
			else // "left"
				lineXPosition = 0;
			
			textFieldLine._setActualPosition(lineXPosition, lineYPosition);
			lineYPosition += (lineHeight + lineSpacing);
		}
		
		//Position the lines container
		var lineContainerX = this._textLinesContainer._x;
		var lineContainerY = this._textLinesContainer._y;
		
		//Handle vertical alignment
		if (this._textLinesContainer._height <= this._textClipContainer._height)
		{
			if (textVerticalAlign == "bottom")
				lineContainerY = this._textClipContainer._height - this._textLinesContainer._height;
			else if (textVerticalAlign == "middle")
				lineContainerY = Math.round((this._textClipContainer._height / 2) - (this._textLinesContainer._height / 2));
			else
				lineContainerY = 0;
		}
		else //Fix resize
		{
			if (this._textLinesContainer._y > 0)
				lineContainerY = 0;
			else if (this._textLinesContainer._y + this._textLinesContainer._height < this._textClipContainer._height)
				lineContainerY = this._textClipContainer._height - this._textLinesContainer._height;
		}
		
		//Handle horizontal alignment
		if (this._textLinesContainer._width <= this._textClipContainer._width)
		{
			if (textHorizontalAlign == "right")
				lineContainerX = this._textClipContainer._width - this._textLinesContainer._width;
			else if (textHorizontalAlign == "center")
				lineContainerX = Math.round((this._textClipContainer._width / 2) - (this._textLinesContainer._width / 2));
			else
				lineContainerX = 0;
		}
		else //Fix resize
		{
			if (this._textLinesContainer._x > 0)
				lineContainerX = 0;
			else if (this._textLinesContainer._x + this._textLinesContainer._width < this._textClipContainer._width)
				lineContainerX = this._textClipContainer._width - this._textLinesContainer._width;
		}
			
		//Scroll if caret out of bounds - we only scroll if caret moved out of bounds due to user input
		if (this._layoutShouldScroll == true)
		{
			this._layoutShouldScroll = false;
			
			//Only need to scroll if text size is larger than clip region
			if (this._textLinesContainer._height > this._textClipContainer._height ||
				this._textLinesContainer._width > this._textClipContainer._width)
			{
				//Find the line the caret is on
				textFieldLine = this._textLinesContainer._getChildAt(this._getLineIndexFromCharIndex(this._caretIndex));
			
				//Get the carets actual position
				var caretX = lineContainerX + textFieldLine._x + (this._charMetrics[this._caretIndex].x - this._charMetrics[textFieldLine._charMetricsStartIndex].x);
				var caretY = lineContainerY + textFieldLine._y;
				
				//Scroll the text lines container if caret out of bounds
				if (caretX < 0)
					lineContainerX = lineContainerX + (caretX * -1) + 35; //Arbitrary 35 pixel X over-scroll (this should probably be a style)
				else if (caretX + 1 > this._textClipContainer._width) 
					lineContainerX = lineContainerX - ((caretX + 1) - this._textClipContainer._width) - 35;
	
				if (caretY < 0)
					lineContainerY = lineContainerY + (caretY * -1);
				else if (caretY + textFieldLine._height > this._textClipContainer._height)
					lineContainerY = lineContainerY - ((caretY + textFieldLine._height) - this._textClipContainer._height);
				
				//Fix over-scroll
				if (this._textLinesContainer._height > this._textClipContainer._height)
				{
					if (lineContainerY > 0)
						lineContainerY = 0;
					else if (lineContainerY + this._textLinesContainer._height < this._textClipContainer._height)
						lineContainerY = this._textClipContainer._height - this._textLinesContainer._height;
				}
				
				if (this._textLinesContainer._width > this._textClipContainer._width)
				{
					if (lineContainerX > 0)
						lineContainerX = 0;
					else if (lineContainerX + this._textLinesContainer._width < this._textClipContainer._width)
						lineContainerX = this._textClipContainer._width - this._textLinesContainer._width;
				}
			}
		}
		
		//Set the position of the text lines container
		this._textLinesContainer._setActualPosition(lineContainerX, lineContainerY);
		
		//Handle caret placement
		if (this._textCaret != null)
		{
			if (this._caretIndex < 0 || this._caretIndex > this._text.length)
				this._textCaret._setActualSize(0, 0);
			else
			{
				//Find the line the caret is on.
				textFieldLine = this._textLinesContainer._getChildAt(this._getLineIndexFromCharIndex(this._caretIndex));

				//Get the text area metrics		
				var visibleMetrics = new DrawMetrics();
				visibleMetrics._x = x;
				visibleMetrics._y = y;
				visibleMetrics._width = w;
				visibleMetrics._height = h;
				
				//Get the caret metrics
				var caretMetrics = new DrawMetrics();
				caretMetrics._x = x + this._textLinesContainer._x + textFieldLine._x + (this._charMetrics[this._caretIndex].x - this._charMetrics[textFieldLine._charMetricsStartIndex].x);
				caretMetrics._y = y + this._textLinesContainer._y + textFieldLine._y;
				caretMetrics._width = 1;
				caretMetrics._height = textFieldLine._height;
				
				//Clip caret metrics to text area
				caretMetrics.mergeReduce(visibleMetrics);
				
				//Size and position caret
				this._textCaret._setActualSize(Math.max(0, caretMetrics._width), Math.max(0, caretMetrics._height));
				this._textCaret._setActualPosition(caretMetrics._x, caretMetrics._y);
			}
		}
		
		//If we added a global listener, but a parent canceled the keyboard event, we need to purge these.
		window.removeEventListener("copy", this._onTextFieldCopyPasteInstance);
		window.removeEventListener("paste", this._onTextFieldCopyPasteInstance);
		window.removeEventListener("cut", this._onTextFieldCopyPasteInstance);
	};		
	
	
///////Internal class for rendering lines of text/////////////////

//This class is only used for rendering lines. 
//No measure() or layout() needed (handled by parent TextField).
function TextFieldLineElement()
{
	TextFieldLineElement.base.prototype.constructor.call(this);
	
	this._text = "";
	
	this._highlightMinIndex = 0;
	this._highlightMaxIndex = 0;
	
	this._parentTextField = null;
	this._charMetricsStartIndex = -1;
	this._charMetricsEndIndex = -1;	//Non-inclusive
}
	
//Inherit from CanvasElement
TextFieldLineElement.prototype = Object.create(CanvasElement.prototype);
TextFieldLineElement.prototype.constructor = TextFieldLineElement;
TextFieldLineElement.base = CanvasElement;	

//Optimize - turn off inheriting for rendering styles. We'll pull styles from parent so
//we can utilize the parents cache rather than each line having to lookup and cache styles.
//Parent also responsible for invalidating our render when styles changes.
TextFieldLineElement._StyleTypes = Object.create(null);
TextFieldLineElement._StyleTypes.TextStyle =						StyleableBase.EStyleType.NORMAL;		
TextFieldLineElement._StyleTypes.TextFont =							StyleableBase.EStyleType.NORMAL;		
TextFieldLineElement._StyleTypes.TextSize =							StyleableBase.EStyleType.NORMAL;		
TextFieldLineElement._StyleTypes.TextColor =						StyleableBase.EStyleType.NORMAL;			
TextFieldLineElement._StyleTypes.TextFillType =						StyleableBase.EStyleType.NORMAL;			
TextFieldLineElement._StyleTypes.TextHighlightedColor = 			StyleableBase.EStyleType.NORMAL;			
TextFieldLineElement._StyleTypes.TextHighlightedBackgroundColor = 	StyleableBase.EStyleType.NORMAL;	
TextFieldLineElement._StyleTypes.TextDecoration =					StyleableBase.EStyleType.NORMAL;

TextFieldLineElement.prototype.setParentLineMetrics = 
	function (parentTextField, charStartIndex, charEndIndex)
	{
		this._parentTextField = parentTextField;
		this._charMetricsStartIndex = charStartIndex;
		this._charMetricsEndIndex = charEndIndex;
		
		var newText = parentTextField._text.substring(charStartIndex, charEndIndex);
		if (newText != this._text)
		{
			this._text = newText;
			this._invalidateRender();
		}
	};

TextFieldLineElement.prototype.setParentSelection = 
	function (startIndex, endIndex)
	{
		var minIndex = Math.min(startIndex, endIndex);
		var maxIndex = Math.max(startIndex, endIndex);
		
		if (minIndex < this._charMetricsStartIndex)
			minIndex = this._charMetricsStartIndex;
		if (maxIndex > this._charMetricsEndIndex)
			maxIndex = this._charMetricsEndIndex;
		
		//Highlight is outside of bounds, nuke it.
		if (minIndex > maxIndex || minIndex == maxIndex)
		{
			minIndex = 0;
			maxIndex = 0;
		}
		
		if (this._highlightMinIndex == minIndex && this._highlightMaxIndex == maxIndex)
			return;
		
		this._highlightMinIndex = minIndex;
		this._highlightMaxIndex = maxIndex;
		
		this._invalidateRender();
	};

TextFieldLineElement.prototype.getLineWidth = 
	function ()
	{
		if (this._charMetricsStartIndex > -1 && this._charMetricsEndIndex > -1)
			return this._parentTextField._charMetrics[this._charMetricsEndIndex].x - this._parentTextField._charMetrics[this._charMetricsStartIndex].x;
		
		return 0;
	};	
	
//@override
TextFieldLineElement.prototype._doRender =
	function()
	{
		TextFieldLineElement.base.prototype._doRender.call(this);
		
		if (this._text.length == 0)
			return;
		
		var paddingMetrics = this._getPaddingMetrics();
		var ctx = this._getGraphicsCtx();
		
		//Get styles
		var textFillType = this._parentTextField.getStyle("TextFillType");
		var textColor = this._parentTextField.getStyle("TextColor");
		var highlightTextColor = this._parentTextField.getStyle("TextHighlightedColor");
		var backgroundHighlightTextColor = this._parentTextField.getStyle("TextHighlightedBackgroundColor");
		var fontString = this._parentTextField._getFontString();
		var textDecoration = this._parentTextField.getStyle("TextDecoration");	
		var maskCharacter = this._parentTextField.getStyle("MaskCharacter");
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY(); // + (paddingMetrics.getHeight() / 2); 
		var w = paddingMetrics.getWidth();
		
		for (var i = 0; i < this._text.length; i++)
		{
			var charWidth = this._parentTextField._charMetrics[i + this._charMetricsStartIndex].width;
			
			var printChar = this._text[i];
			if (maskCharacter != null)
				printChar = maskCharacter;
			
			if (this._highlightMinIndex <= i + this._charMetricsStartIndex && this._highlightMaxIndex > i + this._charMetricsStartIndex)
			{
				ctx.fillStyle = backgroundHighlightTextColor;
				
				ctx.beginPath();
				ctx.moveTo(x, 0);
				ctx.lineTo(x + charWidth, 0);
				ctx.lineTo(x + charWidth, this._height);
				ctx.lineTo(x, this._height);
				ctx.closePath();
				ctx.fill();
				
				if (textFillType == "stroke")
					CanvasElement._strokeText(ctx, printChar, x, y, fontString, highlightTextColor, "top");
				else
					CanvasElement._fillText(ctx, printChar, x, y, fontString, highlightTextColor, "top");
			}
			else
			{
				if (textFillType == "stroke")
					CanvasElement._strokeText(ctx, printChar, x, y, fontString, textColor, "top");
				else
					CanvasElement._fillText(ctx, printChar, x, y, fontString, textColor, "top");
			}
			
			x += charWidth;
		}
		
		if (textDecoration == "underline")
		{
			y = this._height - .5;
			
			ctx.beginPath();
			ctx.moveTo(x, y);
			ctx.lineTo(x + w, y);
			ctx.lineWidth = 1;
			
			if (this._highlightMinIndex == this._highlightMaxIndex)
				ctx.strokeStyle = textColor;
			else 
				ctx.strokeStyle = highlightTextColor;
			
			ctx.stroke();
		}
	};		
	
	
	


/**
 * @depends CanvasElement.js
 */

/////////////////////////////////////////////////////////
/////////////////TextElement/////////////////////////////	
	
/**
 * @class TextElement
 * @inherits CanvasElement
 * 
 * Renders mutli-line style-able select-able text. 
 * TextElement respects newline characters and will
 * wrap text when width is constrained. If only a single
 * line of text is needed, LabelElement is more efficient.
 * 
 * @constructor TextElement 
 * Creates new TextElement instance.
 */
function TextElement()
{
	TextElement.base.prototype.constructor.call(this);
	
	this._textField = new TextFieldElement();
	this._textField.setStyle("Cursor", null);
	this._textField.setStyle("TabStop", -1);
	this._addChild(this._textField);
}

//Inherit from CanvasElement
TextElement.prototype = Object.create(CanvasElement.prototype);
TextElement.prototype.constructor = TextElement;
TextElement.base = CanvasElement;

/////////////Style Types///////////////////////////////

TextElement._StyleTypes = Object.create(null);

/**
 * @style Text String
 * Text to be rendered by the TextElement.
 */
TextElement._StyleTypes.Text = 					StyleableBase.EStyleType.NORMAL;		// "any string" || null

/**
 * @style Selectable boolean
 * When true, text can be highlighted and copied.
 */
TextElement._StyleTypes.Selectable = 			StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style Multiline boolean
 * When true, newline characters are respected and text will be rendered on multiple lines if necessary.
 */
TextElement._StyleTypes.Multiline = 			StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style WordWrap boolean
 * When true, text will wrap when width is constrained and will be rendered on multiple lines if necessary. 
 */
TextElement._StyleTypes.WordWrap = 				StyleableBase.EStyleType.NORMAL;		// true || false


////////////Default Styles////////////////////////////

TextElement.StyleDefault = new StyleDefinition();

//Override base class styles
TextElement.StyleDefault.setStyle("PaddingTop", 					2);
TextElement.StyleDefault.setStyle("PaddingBottom", 					2);
TextElement.StyleDefault.setStyle("PaddingLeft", 					2);
TextElement.StyleDefault.setStyle("PaddingRight", 					2);
TextElement.StyleDefault.setStyle("TextHorizontalAlign", 			"left");
TextElement.StyleDefault.setStyle("TextVerticalAlign", 				"top");
TextElement.StyleDefault.setStyle("Cursor", 						"text");

//TextElement specific styles
TextElement.StyleDefault.setStyle("Text", 							null);
TextElement.StyleDefault.setStyle("Selectable", 					true);
TextElement.StyleDefault.setStyle("Multiline", 						true);
TextElement.StyleDefault.setStyle("WordWrap", 						true);



/////////////Internal Functions///////////////////

//@Override
TextElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		TextElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		if ("Text" in stylesMap)
			this._textField.setText(this.getStyle("Text"));
		
		if ("Selectable" in stylesMap)
			this._textField.setStyle("Selectable", this.getStyle("Selectable"));
		
		if ("Multiline" in stylesMap)
			this._textField.setStyle("Multiline", this.getStyle("Multiline"));
		
		if ("WordWrap" in stylesMap)
			this._textField.setStyle("WordWrap", this.getStyle("WordWrap"));
		
		//Force the textField to use our defaults rather than inherited.
		if ("TextHorizontalAlign" in stylesMap)
			this._textField.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
		if ("TextVerticalAlign" in stylesMap)
			this._textField.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
		
		//Proxy padding to TextField for proper mouse handling
		if ("Padding" in stylesMap ||
			"PaddingTop" in stylesMap ||
			"PaddingBottom" in stylesMap ||
			"PaddingLeft" in stylesMap ||
			"PaddingRight" in stylesMap)
		{
			var paddingSize = this._getPaddingSize();
			
			this._textField.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textField.setStyle("PaddingBottom", paddingSize.paddingBottom);
			this._textField.setStyle("PaddingLeft", paddingSize.paddingLeft);
			this._textField.setStyle("PaddingRight", paddingSize.paddingRight);
		}
	};
	
//@Override
TextElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		//Ignore padding, proxied to TextField
		this._setMeasuredSize(this._textField._measuredWidth, this._textField._measuredHeight);
	};	

//@Override	
TextElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		TextElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		//Ignore padding, proxied to TextField for mouse handling.
		this._textField._setActualPosition(0, 0);
		this._textField._setActualSize(this._width, this._height);
	};



/**
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////////////////////
/////////////////////ScrollButtonSkinElement///////////////////////////

/**
 * @class ScrollButtonSkinElement
 * @inherits CanvasElement
 * 
 * Default skin class for the ScrollButton.
 * 
 * 
 * @constructor ScrollButtonSkinElement 
 * Creates new ScrollButtonSkinElement instance.
 */
function ScrollButtonSkinElement()
{
	ScrollButtonSkinElement.base.prototype.constructor.call(this);
}

//Inherit from CanvasElement
ScrollButtonSkinElement.prototype = Object.create(CanvasElement.prototype);
ScrollButtonSkinElement.prototype.constructor = ScrollButtonSkinElement;
ScrollButtonSkinElement.base = CanvasElement;		
	
//////Style Types//////////////////////
ScrollButtonSkinElement._StyleTypes = Object.create(null);

/**
 * @style ArrowColor String
 * 
 * Hex color value to be used for the arrow icon. Format like "#FF0000" (red).
 */
ScrollButtonSkinElement._StyleTypes.ArrowColor =					StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style ArrowDirection String
 * 
 * Determines the arrow direction. Allowable values are "up", "down", "left", "right". 
 * Note that ScrollBar sets this style directly to the parent button depending on the scroll bar orientation.
 */
ScrollButtonSkinElement._StyleTypes.ArrowDirection =						StyleableBase.EStyleType.NORMAL;	//"up" || "down" || "left" || "right"


////////Default Styles//////////////////

ScrollButtonSkinElement.StyleDefault = new StyleDefinition();

ScrollButtonSkinElement.StyleDefault.setStyle("ArrowColor", 						"#000000");
ScrollButtonSkinElement.StyleDefault.setStyle("ArrowDirection", 					"up");


/////////Internal Functions////////////////////////

//@Override
ScrollButtonSkinElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ScrollButtonSkinElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("ArrowColor" in stylesMap || 
			"ArrowDirection" in stylesMap)
		{
			this._invalidateRender();
		}
	};

//@Override
ScrollButtonSkinElement.prototype._doRender = 
	function()
	{
		ScrollButtonSkinElement.base.prototype._doRender.call(this);
		
		var arrowDirection = this.getStyle("ArrowDirection");
		var arrowColor = this.getStyle("ArrowColor");
		
		if (arrowColor == null || arrowDirection == null)
			return;
		
		var ctx = this._getGraphicsCtx();
		
		var borderThickness = this._getBorderThickness();
		
		var x = borderThickness;
		var y = borderThickness;
		var width = this._width - (borderThickness * 2);
		var height = this._height - (borderThickness * 2);
		
		ctx.beginPath();
		
		if (arrowDirection == "up")
		{
			ctx.moveTo(x + (width / 2), y + (height * .35));
			ctx.lineTo(x + (width * .80), y + (height * .65));
			ctx.lineTo(x + (width * .20), y + (height * .65));
		}
		else if (arrowDirection == "down")
		{
			ctx.moveTo(x + (width / 2), y + (height * .65));
			ctx.lineTo(x + (width * .80), y + (height * .35));
			ctx.lineTo(x + (width * .20), y + (height * .35));
		}
		else if (arrowDirection == "left")
		{
			ctx.moveTo(x + (width * .35), y + (height / 2));
			ctx.lineTo(x + (width * .65), y + (height * .20));
			ctx.lineTo(x + (width * .65), y + (height * .80));
		}
		else if (arrowDirection == "right")
		{
			ctx.moveTo(x + (width * .65), y + (height / 2));
			ctx.lineTo(x + (width * .35), y + (height * .20));
			ctx.lineTo(x + (width * .35), y + (height * .80));
		}
		
		ctx.closePath();
		
		ctx.fillStyle = arrowColor;
		ctx.fill();
		
	};		
	
	
	


/**
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////////////////////
/////////////////////RadioButtonSkinElement////////////////////////////

/**
 * @class RadioButtonSkinElement
 * @inherits CanvasElement
 * 
 * Default skin class for the RadioButtonElement.  
 * Renders an inner selected indicator using the BackgroundShape style.
 * 
 * 
 * @constructor RadioButtonSkinElement 
 * Creates new RadioButtonSkinElement instance.
 */
function RadioButtonSkinElement()
{
	RadioButtonSkinElement.base.prototype.constructor.call(this);
}

//Inherit from CanvasElement
RadioButtonSkinElement.prototype = Object.create(CanvasElement.prototype);
RadioButtonSkinElement.prototype.constructor = RadioButtonSkinElement;
RadioButtonSkinElement.base = CanvasElement;		
	
//////Style Types//////////////////////
RadioButtonSkinElement._StyleTypes = Object.create(null);

/**
 * @style CheckColor String
 * 
 * Hex color value to be used for the check icon. Format like "#FF0000" (red).
 */
RadioButtonSkinElement._StyleTypes.CheckColor =						StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style CheckSize Number
 * 
 * Value between 0 and 1 used to determine the size that the "selected" indicator 
 * should be rendered relative to this element's size.
 */
RadioButtonSkinElement._StyleTypes.CheckSize = 						StyleableBase.EStyleType.NORMAL;


////////Default Styles////////////////
RadioButtonSkinElement.StyleDefault = new StyleDefinition();

//RadioButtonSkinElement specific styles
RadioButtonSkinElement.StyleDefault.setStyle("CheckColor", 			"#000000");
RadioButtonSkinElement.StyleDefault.setStyle("CheckSize", 			.35);


/////////Protected Functions////////////////////////

//@Override
RadioButtonSkinElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		RadioButtonSkinElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("SkinState" in stylesMap || "CheckColor" in stylesMap || "CheckSize" in stylesMap)
			this._invalidateRender();
	};

//@Override
RadioButtonSkinElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		this._setMeasuredSize(14, 14);
	};

//@Override
RadioButtonSkinElement.prototype._doRender = 
	function()
	{
		RadioButtonSkinElement.base.prototype._doRender.call(this);
		
		var currentState = this.getStyle("SkinState");
		
		//Draw indicator.
		if (currentState.indexOf("selected") == 0)
		{
			var ctx = this._getGraphicsCtx();
			
			var checkSize = this.getStyle("CheckSize");
			
			var indicatorMetrics = new DrawMetrics();
			indicatorMetrics._width = this._width * checkSize;
			indicatorMetrics._height = this._height * checkSize;
			indicatorMetrics._x = (this._width - indicatorMetrics._width) / 2;
			indicatorMetrics._y = (this._height - indicatorMetrics._height) / 2;
			
			if (indicatorMetrics._width <= 0 || indicatorMetrics._height <= 0)
				return;
			
			ctx.beginPath();
			this._drawBackgroundShape(ctx, indicatorMetrics);
			
			ctx.fillStyle = this.getStyle("CheckColor");
			ctx.fill();
		}
	};		
	
	


/**
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////////////////////
/////////////////DropdownArrowButtonSkinElement////////////////////////

/**
 * @class DropdownArrowButtonSkinElement
 * @inherits CanvasElement
 * 
 * Default skin class for Arrow button in the DropdownElement.
 * Renders the divider line and an arrow.
 *  
 * 
 * @constructor DropdownArrowButtonSkinElement 
 * Creates new DropdownArrowButtonSkinElement instance.
 */
function DropdownArrowButtonSkinElement()
{
	DropdownArrowButtonSkinElement.base.prototype.constructor.call(this);
}

//Inherit from CanvasElement
DropdownArrowButtonSkinElement.prototype = Object.create(CanvasElement.prototype);
DropdownArrowButtonSkinElement.prototype.constructor = DropdownArrowButtonSkinElement;
DropdownArrowButtonSkinElement.base = CanvasElement;		
	
//////Style Types//////////////////////
DropdownArrowButtonSkinElement._StyleTypes = Object.create(null);

/**
 * @style ArrowColor String
 * 
 * Hex color value to be used for the arrow. Format like "#FF0000" (red).
 */
DropdownArrowButtonSkinElement._StyleTypes.ArrowColor =				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style LineColor String
 * 
 * Hex color value to be used for the divider line. Format like "#FF0000" (red).
 */
DropdownArrowButtonSkinElement._StyleTypes.LineColor =				StyleableBase.EStyleType.NORMAL;		//"#000000"


//////Default Styles///////////////////

DropdownArrowButtonSkinElement.StyleDefault = new StyleDefinition();

DropdownArrowButtonSkinElement.StyleDefault.setStyle("ArrowColor", 				"#000000"); 		
DropdownArrowButtonSkinElement.StyleDefault.setStyle("LineColor", 				"#000000");


/////////Internal Functions////////////////////////

//@Override
DropdownArrowButtonSkinElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DropdownArrowButtonSkinElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("ArrowColor" in stylesMap || 
			"LineColor" in stylesMap)
		{
			this._invalidateRender();
		}
	};

//@Override
DropdownArrowButtonSkinElement.prototype._doRender = 
	function()
	{
		DropdownArrowButtonSkinElement.base.prototype._doRender.call(this);
		
		var ctx = this._getGraphicsCtx();
		var paddingMetrics = this._getPaddingMetrics();
		
		var lineColor = this.getStyle("LineColor");
		var arrowColor = this.getStyle("ArrowColor");
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var width = paddingMetrics.getWidth();
		var height = paddingMetrics.getHeight();
		
		if (arrowColor != null)
		{
			ctx.beginPath();
			
			ctx.moveTo(x + (width / 2), y + (height * .60));
			ctx.lineTo(x + (width * .70), y + (height * .40));
			ctx.lineTo(x + (width * .30), y + (height * .40));
			
			ctx.closePath();
			
			ctx.fillStyle = arrowColor;
			ctx.fill();
		}

		if (lineColor != null)
		{
			var lineHeight = height * .65;
			
			ctx.beginPath();
	
			ctx.moveTo(x, y + (height / 2) - (lineHeight / 2));
			ctx.lineTo(x, y + (height / 2) + (lineHeight / 2));
			ctx.lineTo(x + 1, y + (height / 2) + (lineHeight / 2));
			ctx.lineTo(x + 1, y + (height / 2) - (lineHeight / 2));
			
			ctx.closePath();
			
			ctx.fillStyle = lineColor;
			ctx.fill();
		}
	};	
	
	


/**
 * @depends CanvasElement.js
 */

//////////////////////////////////////////////////////////////////
//////DataGridHeaderColumnDividerSkinElement//////////////////////		
	
/**
 * @class DataGridHeaderColumnDividerSkinElement
 * @inherits CanvasElement
 * 
 * Default skin class for the draggable DataGrid column dividers.
 * Renders a line, and drag arrows when mouse is over.
 * 
 * 
 * @constructor DataGridHeaderColumnDividerSkinElement 
 * Creates new DataGridHeaderColumnDividerSkinElement instance.
 */
function DataGridHeaderColumnDividerSkinElement()
{
	DataGridHeaderColumnDividerSkinElement.base.prototype.constructor.call(this);
}

//Inherit from CanvasElement
DataGridHeaderColumnDividerSkinElement.prototype = Object.create(CanvasElement.prototype);
DataGridHeaderColumnDividerSkinElement.prototype.constructor = DataGridHeaderColumnDividerSkinElement;
DataGridHeaderColumnDividerSkinElement.base = CanvasElement;		
	
//////Style Types//////////////////////
DataGridHeaderColumnDividerSkinElement._StyleTypes = Object.create(null);

/**
 * @style DividerLineColor String
 * 
 * Hex color value to be used for the divider line. Format like "#FF0000" (red).
 */
DataGridHeaderColumnDividerSkinElement._StyleTypes.DividerLineColor =			StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style DividerArrowColor String
 * 
 * Hex color value to be used for the arrows. Format like "#FF0000" (red).
 */
DataGridHeaderColumnDividerSkinElement._StyleTypes.DividerArrowColor =			StyleableBase.EStyleType.NORMAL;		//"up" || "down" || "left" || "right"


////////Default Styles////////////////

DataGridHeaderColumnDividerSkinElement.StyleDefault = new StyleDefinition();

DataGridHeaderColumnDividerSkinElement.StyleDefault.setStyle("DividerLineColor", 		"#777777");
DataGridHeaderColumnDividerSkinElement.StyleDefault.setStyle("DividerArrowColor", 		"#444444");



//@Override
DataGridHeaderColumnDividerSkinElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DataGridHeaderColumnDividerSkinElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("DividerLineColor" in stylesMap ||
			"DividerArrowColor" in stylesMap)
		{
			this._invalidateRender();
		}
	};

//@Override
DataGridHeaderColumnDividerSkinElement.prototype._doRender = 
	function()
	{
		DataGridHeaderColumnDividerSkinElement.base.prototype._doRender.call(this);
		
		var ctx = this._getGraphicsCtx();
		
		var lineColor = this.getStyle("DividerLineColor");
		var arrowColor = this.getStyle("DividerArrowColor");
		var currentState = this.getStyle("SkinState");
		
		var x = 0;
		var y = 0;
		var w = this._width;
		var h = this._height;
		
		ctx.beginPath();

		ctx.moveTo(x + (w / 2) - .5, y);
		ctx.lineTo(x + (w / 2) - .5, y + h);
		ctx.lineTo(x + (w / 2) + .5, y + h);
		ctx.lineTo(x + (w / 2) + .5, y);
		
		ctx.closePath();
		
		ctx.fillStyle = lineColor;
		ctx.fill();
		
		////////////////////////////
		
		if (currentState == "over" || currentState == "down")
		{
			var arrowHeight = h / 2;
			
			ctx.fillStyle = arrowColor;
			
			ctx.beginPath();
			
			ctx.moveTo(x + (w / 2) - .5 - 1, (h / 2) - (arrowHeight / 2));
			ctx.lineTo(x + (w / 2) - .5 - 1, (h / 2) + (arrowHeight / 2));
			ctx.lineTo(x, y + (h / 2));
			
			ctx.closePath();
			ctx.fill();
			
			ctx.beginPath();
			
			ctx.moveTo(x + (w / 2) + .5 + 1, (h / 2) - (arrowHeight / 2));
			ctx.lineTo(x + (w / 2) + .5 + 1, (h / 2) + (arrowHeight / 2));
			ctx.lineTo(x + w, y + (h / 2));
			
			ctx.closePath();
			ctx.fill();
		}
	};
	
	


/**
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////////CheckboxSkinElement/////////////////////////////

/**
 * @class CheckboxSkinElement
 * @inherits CanvasElement
 * 
 * Default skin class for the CheckboxElement.
 * 
 * 
 * @constructor CheckboxSkinElement 
 * Creates new CheckboxSkinElement instance.
 */
function CheckboxSkinElement()
{
	CheckboxSkinElement.base.prototype.constructor.call(this);
}

//Inherit from CanvasElement
CheckboxSkinElement.prototype = Object.create(CanvasElement.prototype);
CheckboxSkinElement.prototype.constructor = CheckboxSkinElement;
CheckboxSkinElement.base = CanvasElement;		
	
//////Style Types//////////////////////
CheckboxSkinElement._StyleTypes = Object.create(null);

/**
 * @style CheckColor String
 * 
 * Hex color value to be used for the check icon. Format like "#FF0000" (red).
 */
CheckboxSkinElement._StyleTypes.CheckColor =				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style CheckSize Number
 * 
 * Value between 0 and 1 used to determine the size that the "selected" indicator 
 * should be rendered relative to this element's size.
 */
CheckboxSkinElement._StyleTypes.CheckSize = 				StyleableBase.EStyleType.NORMAL;


////////Default Styles////////////////

CheckboxSkinElement.StyleDefault = new StyleDefinition();

//CheckboxSkinElement specific styles
CheckboxSkinElement.StyleDefault.setStyle("CheckColor", 						"#000000");
CheckboxSkinElement.StyleDefault.setStyle("CheckSize", 							.80);




/////////Protected Functions////////////////////////

//@Override
CheckboxSkinElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		CheckboxSkinElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("SkinState" in stylesMap || "CheckColor" in stylesMap || "CheckSize" in stylesMap)
			this._invalidateRender();
	};

//@Override
CheckboxSkinElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		this._setMeasuredSize(14, 14);
	};

//@Override
CheckboxSkinElement.prototype._doRender = 
	function()
	{
		CheckboxSkinElement.base.prototype._doRender.call(this);
		
		var currentState = this.getStyle("SkinState");
		
		//Draw check or dash.
		if (currentState.indexOf("selected") == 0 || 
			currentState.indexOf("half") == 0)
		{
			var ctx = this._getGraphicsCtx();
			
			var borderThickness = this._getBorderThickness();
			var checkColor = this.getStyle("CheckColor");
			
			var checkSize = this.getStyle("CheckSize");
			var width = this._width * checkSize;
			var height = this._height * checkSize;
			
			var x = 0 + ((this._width - width) / 2);
			var y = 0 + ((this._height - height) / 2);
			
			if (currentState.indexOf("selected") == 0) //Draw check
			{
				ctx.beginPath();
				
				ctx.moveTo(x + (width * .10), 
							y + (height * .60));
				
				ctx.lineTo(x + (width * .40),
							y + height * .90);
				
				ctx.lineTo(x + (width * .90),
							y + (height * .26));
				
				ctx.lineTo(x + (width * .78),
							y + (height * .10));
				
				ctx.lineTo(x + (width * .38),
							y + height * .65);
				
				ctx.lineTo(x + (width * .20),
							y + height * .45);
				
				ctx.closePath();
			}
			else //Half selected - Draw dash
			{
				ctx.beginPath();
				
				ctx.moveTo(x + (width * .12), 
							y + (height * .42));
				
				ctx.lineTo(x + (width * .12),
							y + height * .58);
				
				ctx.lineTo(x + (width * .88),
							y + (height * .58));
				
				ctx.lineTo(x + (width * .88),
							y + (height * .42));
				
				ctx.closePath();
			}
			
			ctx.fillStyle = checkColor;
			ctx.fill();
		}
	};		
	
	


/**
 * @depends CanvasElement.js
 */

//////////////////////////////////////////////////////////////////////
///////////////////////SkinnableElement///////////////////////////////	
	
/**
 * @class SkinnableElement
 * @inherits CanvasElement
 * 
 * Abstract base class for skin-able components. Allows changing states, stores a list
 * of skins per state and toggles skin visibility per the current state. 
 * Any states may be used. As an example, ButtonElement uses "up", "over", "down", and "disabled" states.
 * Override appropriate functions to return skin classes and style definitions per the element's states. 
 * SkinnableElement does not render itself, its skins do. It proxies all rendering 
 * related styles to its skins (such as BackgroundFill).
 * 
 * @seealso StyleProxy
 * 
 * 
 * @constructor SkinnableElement 
 * Creates new SkinnableElement instance.
 */
function SkinnableElement()
{
	SkinnableElement.base.prototype.constructor.call(this);
	
	this._skins = Object.create(null);
	this._currentSkin = null;
	
	/**
	 * @member _currentSkinState String
	 * Read only - String representing the current state.
	 */
	this._currentSkinState = "";
	
}

//Inherit from CanvasElement
SkinnableElement.prototype = Object.create(CanvasElement.prototype);
SkinnableElement.prototype.constructor = SkinnableElement;
SkinnableElement.base = CanvasElement;


//Proxy map for styles we want to pass to skins.
SkinnableElement._SkinProxyMap = Object.create(null);

//Proxy styles that affect rendering.
SkinnableElement._SkinProxyMap.BorderType = 				true;
SkinnableElement._SkinProxyMap.BorderColor = 				true;
SkinnableElement._SkinProxyMap.BorderThickness = 			true;
SkinnableElement._SkinProxyMap.BackgroundFill = 			true;
SkinnableElement._SkinProxyMap.BackgroundShape = 			true;

//Proxy styles that are not defined by the element.
SkinnableElement._SkinProxyMap._Arbitrary =					true;


//////////////////Internal Functions///////////////////
/**
 * @function _getSkinClass
 * Gets the skin class to use per the provided state. 
 * Override this to return different skin classes for different states.
 * 
 * @param state String
 * The state for which to return a skin class.
 * 
 * @returns Function
 * Return the constructor of the appropriate skin class.
 */
SkinnableElement.prototype._getSkinClass = 
	function (state)
	{
		return null;
	};

/**
 * @function _getSubStyleNameForSkinState
 * Gets the style name of the sub style to be applied to the skin
 * for the associated state. Override this to return the associated
 * sub style name for the supplied state.
 * 
 * @param state String
 * The state for which to return a sub style name.
 * 
 * @returns string
 * Sub style name to apply to the skin of the associated state.
 */	
SkinnableElement.prototype._getSubStyleNameForSkinState = 
	function (state)
	{
		return null;
	};
	
/**
 * @function _getSkinStyleProxyMap
 * Gets the Style proxy map to pass to skins. Override this if you need to pass additional styles
 * to custom skins. You should include all the styles provided in the default SkinnableElement style map.
 * 
 * @returns Object
 * Return a style proxy map to be applied to this element to all skins. Formatted as:
 * 
 * MyProxyMap = Object.create(null);
 * MyProxyMap.StyleName1 = true;
 * MyProxyMap.StyleName2 = true;
 * 
 * @seealso StyleProxy
 */	
SkinnableElement.prototype._getSkinStyleProxyMap = 
	function ()
	{
		return SkinnableElement._SkinProxyMap;
	};
	
/**
 * @function _updateSkinStyleDefinitions
 * Updates the StyleDefinition for the skin of the provided state. Subclasses should call
 * this within their _doStylesUpdated() when skin style definitions change.
 * 
 * @param state String
 * The state for which to update the StyleDefinition.
 */	
SkinnableElement.prototype._updateSkinStyleDefinitions = 
	function (state)
	{
		var skinElement = this._skins[state];
		
		//Skin instance not yet created.
		if (skinElement == null)
			return;
	
		var subStyleName = this._getSubStyleNameForSkinState(state);
		
		if (subStyleName != null)
			this._applySubStylesToElement(subStyleName, skinElement);
	};
	
/**
 * @function _updateSkinClass
 * Updates the skin class for the skin of the provided state. Subclasses should call
 * this within their _doStylesUpdated() when skin style class changes.
 * 
 * @param state String
 * The state for which to update the skin class.
 */		
SkinnableElement.prototype._updateSkinClass = 
	function (state)
	{
		//If the skin hasnt been created bail. State change will create later.
		if (this._skins[state] == null && state != this._currentSkinState)
			return;
		
		var newSkinClass = this._getSkinClass(state);
		var currentSkinClass = null;
		
		if (this._skins[state] != null)
			currentSkinClass = this._skins[state].constructor;
		
		//Skin class has not changed.
		if (newSkinClass == currentSkinClass)
			return;
		
		//Nuke the old skin
		if (this._skins[state] != null)
		{
			this._removeChild(this._skins[state]);
			this._skins[state] = null;
		}
		
		//Only create the new skin if its active, otherwise state change will create later.
		if (this._currentSkinState == state)
		{
			//Create new and adjust visibility if needed
			var newSkin = this._createSkin(state);
			this._currentSkin = newSkin;
			
			if (newSkin != null)
				newSkin.setStyle("Visible", true);
		}
	};	
	
//@private	
SkinnableElement.prototype._createSkin = 
	function (state)
	{
		var skinClass = this._getSkinClass(state);
		if (skinClass == null)
		{
			this._skins[state] = null;
			return null;
		}
	
		var newSkin = new (skinClass)();
		
		this._skins[state] = newSkin;
		
		newSkin._setStyleProxy(new StyleProxy(this, this._getSkinStyleProxyMap()));
		
		this._updateSkinStyleDefinitions(state);
		
		newSkin.setStyle("MouseEnabled", false);
		newSkin.setStyle("SkinState", state);
		
		this._addChildAt(newSkin, 0);
		
		return newSkin;
	};
	
/**
 * @function _changeState
 * Called when the element changes skin state. Do not call this function directly.
 * You may override this if you need to make additional changes to your component
 * when the skin state changes (such as updating a label color).
 * 
 * @param state String
 * The skin state that this element is changing too.
 * 
 * @returns boolean
 * Returns true if the element state has actually changed, false if it is the same state. 
 * Subclasses can check what this base function returns before making additional changes for better performance.
 */	
SkinnableElement.prototype._changeState = 
	function (state)
	{
		if (this._currentSkinState == state || state == "" || state == null)
			return false;
	
		this._currentSkinState = state;
		
		var foundSkin = false;
		for (var skinState in this._skins)
		{
			//Ignore null skins.
			if (this._skins[skinState] == null)
				continue;
			
			if (skinState == state)
			{
				this._currentSkin = this._skins[skinState];
				this._skins[skinState].setStyle("Visible", true);
				foundSkin = true;
			}
			else 
				this._skins[skinState].setStyle("Visible", false);
		}
		
		//Attempt to create the skin (this may be null anyway)
		if (foundSkin == false)
		{
			this._currentSkin = this._createSkin(state);
			
			if (this._currentSkin != null)
				this._currentSkin.setStyle("Visible", true);
		}
		
		return true;
	};

//@override
SkinnableElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		SkinnableElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("SkinState" in stylesMap)
			this._changeState(this.getStyle("SkinState"));
	};	
	
//@override	
SkinnableElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		SkinnableElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		//We have to size all skins, not just the current skin. Otherwise if multiple skins 
		//are created within 1 cycle (multiple immediate state changes) we could end up with 
		//a skin that never gets sized unless we invalidate layout on every skin change.
		for (var prop in this._skins)
		{
			if (this._skins[prop] == null)
				continue;
			
			this._skins[prop]._setActualSize(this._width, this._height);
			this._skins[prop]._setActualPosition(0, 0);
		}
	};	

//@Override
SkinnableElement.prototype._doRender = 
	function ()
	{
		//Do nothing, don't call base. SkinnableElement does not render itself, its skins do.
	
		//TODO: Use the active skin metrics & shape to render the focus ring.
		//if (this._renderFocusRing == true)
		//	this._drawFocusRing(ctx, this._getBorderMetrics());
	};	
	
	


/**
 * @depends SkinnableElement.js
 */

/////////////////////////////////////////////////////////
///////////////TextInputElement//////////////////////////	
	
/**
 * @class TextInputElement
 * @inherits SkinnableElement
 * 
 * TextInput is an edit-able single line text box.
 * 
 * 
 * @constructor TextInputElement 
 * Creates new TextInputElement instance.
 */
function TextInputElement()
{
	TextInputElement.base.prototype.constructor.call(this);
	
	this._textField = new TextFieldElement();
	this._textField.setStyle("Selectable", true);
	this._textField.setStyle("TabStop", -1);
	this._addChild(this._textField);
	
	var _self = this;
	
	//Private event handlers, need different instance for each TextInput. Proxy to prototype.
	this._onTextInputFocusEventInstance = 
		function (event)
		{
			if (event.getType() == "focusin")
				_self._onTextInputFocusIn(event);
			else
				_self._onTextInputFocusOut(event);
		};
	
	this._onTextInputKeyUpDownInstance = 
		function (keyboardEvent)
		{
			if (keyboardEvent.getType() == "keydown")
				_self._onTextInputKeyDown(keyboardEvent);
			else // if (keyboardEvent.getType() == "keyup")
				_self._onTextInputKeyUp(keyboardEvent);
		};
		
	this._onTextInputTextFieldChangedInstance = 
		function (event)
		{
			_self._onTextInputTextFieldChanged(event);
		};
		
	this.addEventListener("focusin", this._onTextInputFocusEventInstance);
	this.addEventListener("focusout", this._onTextInputFocusEventInstance);	
}

//Inherit from SkinnableElement
TextInputElement.prototype = Object.create(SkinnableElement.prototype);
TextInputElement.prototype.constructor = TextInputElement;
TextInputElement.base = SkinnableElement;

/////////////Events////////////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the text is modified as a result of user input.
 */


/////////////Style Types///////////////////////////////

TextInputElement._StyleTypes = Object.create(null);

/**
 * @style MaxChars int
 * 
 * Maximum number of characters allowed.
 */
TextInputElement._StyleTypes.MaxChars = 								StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style SkinClass CanvasElement
 * 
 * The CanvasElement constructor type to apply to all skin states. 
 * Specific states such as UpSkinClass will override SkinClass.
 */
TextInputElement._StyleTypes.SkinClass =								StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style UpSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the skin when in the "up" state. 
 * This will override SkinClass.
 */
TextInputElement._StyleTypes.UpSkinClass = 								StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style UpSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "up" state skin element.
 */
TextInputElement._StyleTypes.UpSkinStyle = 								StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style UpTextColor String
 * 
 * Hex color value to be used for the button TextInput is in the "up" state. Format like "#FF0000" (red).
 * This will override the TextColor style.
 */
TextInputElement._StyleTypes.UpTextColor = 								StyleableBase.EStyleType.NORMAL;		// color "#000000"

/**
 * @style UpTextHighlightedColor String
 * 
 * Hex color value to be used for highlighted text when the TextInput is in the "up" state. Format like "#FF0000" (red).
 * This will override the TextHighlightedColor style.
 */
TextInputElement._StyleTypes.UpTextHighlightedColor = 					StyleableBase.EStyleType.NORMAL;		// color "#FFFFFF"

/**
 * @style UpTextHighlightedBackgroundColor String
 * 
 * Hex color value to be used for highlighted text background when the TextInput is in the "up" state. Format like "#FF0000" (red).
 * This will override the TextHighlightedBackgroundColor style.
 */
TextInputElement._StyleTypes.UpTextHighlightedBackgroundColor = 		StyleableBase.EStyleType.NORMAL;			// color "#000000"

/**
 * @style DisabledSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the TextInput is in the "disabled" state.
 * When this is null, the base SkinClass style will be used.
 */
TextInputElement._StyleTypes.DisabledSkinClass = 						StyleableBase.EStyleType.NORMAL;		// Element constructor()

/**
 * @style DisabledSkinStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to apply to the "disabled" state skin element.
 * When this is null, the base SkinTyle will be used.
 */
TextInputElement._StyleTypes.DisabledSkinStyle = 						StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style DisabledTextColor String
 * 
 * Hex color value to be used for the button TextInput is in the "disabled" state. Format like "#FF0000" (red).
 * This will override the TextColor style.
 */
TextInputElement._StyleTypes.DisabledTextColor = 						StyleableBase.EStyleType.NORMAL;		// color "#000000"

/**
 * @style DisabledTextHighlightedColor String
 * 
 * Hex color value to be used for highlighted text when the TextInput is in the "disabled" state. Format like "#FF0000" (red).
 * When this is null, the base TextHighlightedColor style will be used.
 */
TextInputElement._StyleTypes.DisabledTextHighlightedColor = 			StyleableBase.EStyleType.NORMAL;		// color "#FFFFFF"

/**
 * @style DisabledTextHighlightedBackgroundColor String
 * 
 * Hex color value to be used for highlighted text background when the TextInput is in the "disabled" state. Format like "#FF0000" (red).
 * When this is null, the base TextHighlightedBackgroundColor style will be used.
 */
TextInputElement._StyleTypes.DisabledTextHighlightedBackgroundColor = 	StyleableBase.EStyleType.NORMAL;		// color "#000000"

/**
 * @style DisplayAsPassword boolean
 * 
 * When true, text will be masked using the PasswordMaskCharacter style.
 */
TextInputElement._StyleTypes.DisplayAsPassword = 						StyleableBase.EStyleType.NORMAL;		// false



/////////////Default Styles///////////////////////////

TextInputElement.StyleDefault = new StyleDefinition();

TextInputElement.StyleDefault.setStyle("TextHorizontalAlign", 						"left");
TextInputElement.StyleDefault.setStyle("TextVerticalAlign", 						"middle");

TextInputElement.StyleDefault.setStyle("MaxChars", 									0);
TextInputElement.StyleDefault.setStyle("Enabled", 									true);

TextInputElement.StyleDefault.setStyle("UpTextColor", 								"#000000");
TextInputElement.StyleDefault.setStyle("DisabledTextColor", 						"#888888");

TextInputElement.StyleDefault.setStyle("DisplayAsPassword", 						false);

TextInputElement.StyleDefault.setStyle("PaddingTop",								3);
TextInputElement.StyleDefault.setStyle("PaddingBottom",								3);
TextInputElement.StyleDefault.setStyle("PaddingLeft",								3);
TextInputElement.StyleDefault.setStyle("PaddingRight",								3);

TextInputElement.StyleDefault.setStyle("TabStop", 									0);

TextInputElement.StyleDefault.setStyle("SkinClass", 								CanvasElement);
TextInputElement.StyleDefault.setStyle("UpSkinClass", 								CanvasElement);
TextInputElement.StyleDefault.setStyle("DisabledSkinClass", 						CanvasElement);

/////Skin styles//
TextInputElement.DisabledSkinStyleDefault = new StyleDefinition();

TextInputElement.DisabledSkinStyleDefault.setStyle("BorderType", 					"inset");
TextInputElement.DisabledSkinStyleDefault.setStyle("BorderThickness", 				1);
TextInputElement.DisabledSkinStyleDefault.setStyle("BorderColor", 					"#999999");
TextInputElement.DisabledSkinStyleDefault.setStyle("BackgroundFill", 				"#ECECEC");

TextInputElement.UpSkinStyleDefault = new StyleDefinition();

TextInputElement.UpSkinStyleDefault.setStyle("BorderType", 							"inset");
TextInputElement.UpSkinStyleDefault.setStyle("BorderThickness", 					1);
TextInputElement.UpSkinStyleDefault.setStyle("BorderColor", 						"#606060");

TextInputElement.UpSkinStyleDefault.setStyle("BackgroundFill", 						"#F5F5F5");

//Apply skin defaults
TextInputElement.StyleDefault.setStyle("UpSkinStyle", 								TextInputElement.UpSkinStyleDefault);
TextInputElement.StyleDefault.setStyle("DisabledSkinStyle", 						TextInputElement.DisabledSkinStyleDefault);



////////Public///////////////////////

/**
 * @function setText
 * Sets the text to be displayed.
 * 
 * @param text String
 * Text to be displayed.
 */
TextInputElement.prototype.setText = 
	function (text)
	{
		this._textField.setText(text);
	};

/**
 * @function getText
 * Gets the text currently displayed.
 * 
 * @returns String
 * Text currently displayed.
 */	
TextInputElement.prototype.getText = 
	function ()
	{
		return this._textField.getText();
	};


////////Internal/////////////////////

/**
 * @function _onTextInputTextFieldChanged
 * Event handler for the internal TextField "changed" event. Only active when TextInput is Enabled.
 * Dispatches a "changed" event from this TextInput element.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to be processed.
 */	
TextInputElement.prototype._onTextInputTextFieldChanged = 
	function (elementEvent)
	{
		//Pass on the changed event
	
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};
	
/**
 * @function _onTextInputKeyDown
 * Event handler for "keydown" event. Only active when TextInput is enabled. 
 * Proxies keyboard event to internal TextField.
 * 
 * @param keyboardEvent ElementKeyboardEvent
 * ElementKeyboardEvent to process.
 */	
TextInputElement.prototype._onTextInputKeyDown = 
	function (keyboardEvent)
	{
		if (keyboardEvent.getDefaultPrevented() == true)
			return;
		
		var clonedEvent = keyboardEvent.clone();
		clonedEvent._bubbles = false; //Dont bubble.
		
		//Dispatch non-bubbling keyboard event to our text field.
		this._textField.dispatchEvent(clonedEvent);
		
		if (clonedEvent.getIsCanceled() == true)
			keyboardEvent.cancelEvent();
			
		if (clonedEvent.getDefaultPrevented() == true)
			keyboardEvent.preventDefault();
	};

/**
 * @function _onTextInputKeyUp
 * Event handler for "keyup" event. Only active when TextInput is enabled. 
 * Proxies keyboard event to internal TextField.
 * 
 * @param keyboardEvent ElementKeyboardEvent
 * ElementKeyboardEvent to process.
 */	
TextInputElement.prototype._onTextInputKeyUp = 
	function (keyboardEvent)
	{
		if (keyboardEvent.getDefaultPrevented() == true)
			return;
		
		var clonedEvent = keyboardEvent.clone();
		clonedEvent._bubbles = false; //Dont bubble.
		
		//Dispatch non-bubbling keyboard event to our text field.
		this._textField.dispatchEvent(clonedEvent);
		
		if (clonedEvent.getIsCanceled() == true)
			keyboardEvent.cancelEvent();
			
		if (clonedEvent.getDefaultPrevented() == true)
			keyboardEvent.preventDefault();
	};	
	
/**
 * @function _onTextInputFocusIn
 * Event handler for "focusin" event. 
 * Proxies focus event to internal TextField.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to process.
 */		
TextInputElement.prototype._onTextInputFocusIn = 
	function (elementEvent)
	{
		//This only works because TextField doesnt look at _isFocused (manages caret state with different flag)
		this._textField.dispatchEvent(elementEvent.clone()); 
	};

/**
 * @function _onTextInputFocusOut
 * Event handler for "focusout" event. 
 * Proxies focus event to internal TextField.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to process.
 */		
TextInputElement.prototype._onTextInputFocusOut = 
	function (elementEvent)
	{
		//This only works because TextField doesnt look at _isFocused (manages caret state with different flag)
		this._textField.dispatchEvent(elementEvent.clone());
	};
	
//@Override
TextInputElement.prototype._getSkinClass = 
	function (state)
	{
		var stateSkinClass = null;
		
		if (state == "up")
			stateSkinClass = this.getStyleData("UpSkinClass");
		else if (state == "disabled")
			stateSkinClass = this.getStyleData("DisabledSkinClass");
		
		var skinClass = this.getStyleData("SkinClass");
		
		//Shouldnt have null stateSkinClass
		if (stateSkinClass == null || skinClass.comparePriority(stateSkinClass) > 0) //Use skinClass if higher priority
			return skinClass.value;
		
		return stateSkinClass.value;
	};
	
//@override	
TextInputElement.prototype._getSubStyleNameForSkinState = 
	function (state)
	{
		if (state == "up")
			return "UpSkinStyle";
		if (state == "disabled")
			return "DisabledSkinStyle";
		
		return TextInputElement.base.prototype._getSubStyleNameForSkinState.call(this, state);
	};			
	
/**
 * @function _updateState
 * Updates the current SkinState in response to style changes.
 */	
TextInputElement.prototype._updateState = 
	function ()
	{
		var newState = "up";

		if (this.getStyle("Enabled") == false)
			newState = "disabled";
		
		this.setStyle("SkinState", newState);
	};	
	
//@Override
TextInputElement.prototype._changeState = 
	function (state)
	{
		TextInputElement.base.prototype._changeState.call(this, state);
	
		this._updateTextColors();
	};
	
/**
 * @function _updateTextColors
 * Updates the text colors based on the current state. Called when state changes and when added to display hierarchy.
 */	
TextInputElement.prototype._updateTextColors = 
	function ()
	{
		this._textField.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._textField.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._textField.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
	};
	
/**
 * @function _getTextColor
 * Gets the text color for the supplied state based on text styles.
 * 
 * @param state String
 * The skin state to return the text color.
 * 
 * @returns String
 * Hex color value.
 */	
TextInputElement.prototype._getTextColor = 
	function (state)
	{
		var stateTextColor = null;
		
		if (state == "up")
			stateTextColor = this.getStyleData("UpTextColor");
		else if (state == "disabled")
			stateTextColor = this.getStyleData("DisabledTextColor");
	
		var textColor = this.getStyleData("TextColor");
		
		//Shouldnt have null stateTextColor
		if (stateTextColor == null || textColor.comparePriority(stateTextColor) > 0) //Use textColor if higher priority
			return textColor.value;
		
		return stateTextColor.value;
	};
	
/**
 * @function _getTextHighlightedColor
 * Gets the highlighted text color for the supplied state based on text styles.
 * 
 * @param state String
 * The skin state to return the highlighted text color.
 * 
 * @returns String
 * Hex color value.
 */		
TextInputElement.prototype._getTextHighlightedColor = 
	function (state)
	{
		var stateTextColor = null;
		
		if (state == "up")
			stateTextColor = this.getStyleData("UpTextHighlightedColor");
		else if (state == "disabled")
			stateTextColor = this.getStyleData("DisabledTextHighlightedColor");
	
		var textColor = this.getStyleData("TextHighlightedColor");
		
		//Shouldnt have null stateTextColor
		if (stateTextColor == null || textColor.comparePriority(stateTextColor) > 0) //Use textColor if higher priority
			return textColor.value;
		
		return stateTextColor.value;
	};
	
/**
 * @function _getTextHighlightedBackgroundColor
 * Gets the highlighted text background color for the supplied state based on text styles.
 * 
 * @param state String
 * The skin state to return the highlighted text background color.
 * 
 * @returns String
 * Hex color value.
 */		
TextInputElement.prototype._getTextHighlightedBackgroundColor = 
	function (state)
	{
		var stateTextColor = null;
		
		if (state == "up")
			stateTextColor = this.getStyleData("UpTextHighlightedBackgroundColor");
		else if (state == "disabled")
			stateTextColor = this.getStyleData("DisabledTextHighlightedBackgroundColor");
	
		var textColor = this.getStyleData("TextHighlightedBackgroundColor");
		
		//Shouldnt have null stateTextColor
		if (stateTextColor == null || textColor.comparePriority(stateTextColor) > 0) //Use textColor if higher priority
			return textColor.value;
		
		return stateTextColor.value;
	};
	
/**
 * @function _updateSkinStyles
 * Updates skin related styles. Called by _doStylesUpdated()
 * 
 * @param stylesMap Object
 * Map of styles that have been changed
 */	
TextInputElement.prototype._updateSkinStyles = 
	function (stylesMap)
	{
		////Update skin classes and sub styles.
		if ("SkinClass" in stylesMap || "UpSkinClass" in stylesMap)
			this._updateSkinClass("up");
		if ("UpSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("up");
		
		if ("SkinClass" in stylesMap || "DisabledSkinClass" in stylesMap)
			this._updateSkinClass("disabled");
		if ("DisabledSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("disabled");
	
		this._updateState();
		this._updateTextColors();
	};
	
//@Override
TextInputElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		TextInputElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		//Force the textField to use our defaults rather than inherited.
		if ("TextHorizontalAlign" in stylesMap)
			this._textField.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
		if ("TextVerticalAlign" in stylesMap)
			this._textField.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
		
		if ("MaxChars" in stylesMap)
			this._textField.setStyle("MaxChars", this.getStyle("MaxChars"));
		
		if ("Enabled" in stylesMap)
		{
			var enabled = this.getStyle("Enabled");
			this._textField.setStyle("Enabled", enabled);
			
			if (enabled == true)
			{
				if (this.hasEventListener("keydown", this._onTextInputKeyUpDownInstance) == false)
					this.addEventListener("keydown", this._onTextInputKeyUpDownInstance);
				
				if (this.hasEventListener("keyup", this._onTextInputKeyUpDownInstance) == false)
					this.addEventListener("keyup", this._onTextInputKeyUpDownInstance);
				
				if (this._textField.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == false)
					this._textField.addEventListener("changed", this._onTextInputTextFieldChangedInstance);					
			}
			else
			{
				if (this.hasEventListener("keydown", this._onTextInputKeyUpDownInstance) == true)
					this.removeEventListener("keydown", this._onTextInputKeyUpDownInstance);
				
				if (this.hasEventListener("keyup", this._onTextInputKeyUpDownInstance) == true)
					this.removeEventListener("keyup", this._onTextInputKeyUpDownInstance);
				
				if (this._textField.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == true)
					this._textField.removeEventListener("changed", this._onTextInputTextFieldChangedInstance);
			}
		}
		
		if ("TextLinePaddingTop" in stylesMap || 
			"TextLinePaddingBottom" in stylesMap)
		{
			this._invalidateMeasure();
		}
		
		if ("Padding" in stylesMap ||
			"PaddingTop" in stylesMap ||
			"PaddingBottom" in stylesMap ||
			"PaddingLeft" in stylesMap ||
			"PaddingRight" in stylesMap)
		{
			var paddingSize = this._getPaddingSize();
			
			this._textField.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textField.setStyle("PaddingBottom", paddingSize.paddingBottom);
			this._textField.setStyle("PaddingLeft", paddingSize.paddingLeft);
			this._textField.setStyle("PaddingRight", paddingSize.paddingRight);
			
			this._invalidateMeasure();
		}
		
		if ("DisplayAsPassword" in stylesMap ||
			"PasswordMaskCharacter" in stylesMap)
		{
			var maskCharacter = null;
			if (this.getStyle("DisplayAsPassword") == true)
				maskCharacter = this.getStyle("PasswordMaskCharacter")
			
			this._textField.setStyle("MaskCharacter", maskCharacter);
		}
		
		this._updateSkinStyles(stylesMap);
	};
	
//@Override
TextInputElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var measuredHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
		var measuredWidth = measuredHeight * 10;
		
		measuredWidth += padWidth;
		measuredHeight += padHeight;
	
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};
	
//@Override	
TextInputElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		TextInputElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		//Ignore padding, proxied to TextField for proper mouse handling.		
		this._textField._setActualPosition(0, 0);
		this._textField._setActualSize(this._width, this._height);
	};
	
	


/**
 * @depends TextInputElement.js
 */

/////////////////////////////////////////////////////////
/////////////TimeInputElement////////////////////////////
	
/**
 * @class TimeInputElement
 * @inherits TextInputElement
 * 
 * TimeInputElement is an editable time field.
 * Note that TimeInput supports both 12 and 24 hour time, but does not supply AM/PM.
 * 
 * @constructor TimeInputElement 
 * Creates new TimeInputElement instance.
 */
function TimeInputElement()
{
	TimeInputElement.base.prototype.constructor.call(this);
	
	//Steal the text field from base TextInput - re-use as hour field
	this._textFieldHour = this._textField;
	this._removeChild(this._textField);
	
		//Use list container to layout text fields
		this._listContainer = new ListContainerElement();
		this._listContainer.setStyle("LayoutDirection", "horizontal");
		
			this._textFieldHour.setStyle("Selectable", true);
			this._textFieldHour.setStyle("PercentHeight", 100);
			this._textFieldHour.setStyle("PercentWidth", 100);
			this._textFieldHour.setStyle("MaxChars", 2);
			this._textFieldHour.setStyle("TabStop", -1);
			
			this._labelColon1 = new LabelElement();
			this._labelColon1.setStyle("PercentHeight", 100);
			this._labelColon1.setStyle("TextHorizontalAlign", "center");
			this._labelColon1.setStyle("Text", ":");
			this._labelColon1.setStyle("PaddingLeft", 0);
			this._labelColon1.setStyle("PaddingRight", 0);
			this._labelColon1.setStyle("TextStyle", "bold");
			
			this._textFieldMinute = new TextFieldElement();
			this._textFieldMinute.setStyle("Selectable", true);
			this._textFieldMinute.setStyle("PercentHeight", 100);
			this._textFieldMinute.setStyle("PercentWidth", 100);
			this._textFieldMinute.setStyle("MaxChars", 2);
			this._textFieldMinute.setStyle("TabStop", -1);
			
			this._labelColon2 = new LabelElement();
			this._labelColon2.setStyle("PercentHeight", 100);
			this._labelColon2.setStyle("TextHorizontalAlign", "center");
			this._labelColon2.setStyle("Text", ":");
			this._labelColon2.setStyle("PaddingLeft", 0);
			this._labelColon2.setStyle("PaddingRight", 0);
			this._labelColon2.setStyle("TextStyle", "bold");
			
			this._textFieldSecond = new TextFieldElement();
			this._textFieldSecond.setStyle("Selectable", true);
			this._textFieldSecond.setStyle("PercentHeight", 100);
			this._textFieldSecond.setStyle("PercentWidth", 100);
			this._textFieldSecond.setStyle("MaxChars", 2);
			this._textFieldSecond.setStyle("TabStop", -1);
			
		this._listContainer.addElement(this._textFieldHour);
		this._listContainer.addElement(this._labelColon1);
		this._listContainer.addElement(this._textFieldMinute);
		this._listContainer.addElement(this._labelColon2);
		this._listContainer.addElement(this._textFieldSecond);
		
	this._addChild(this._listContainer);
	
	////////////////////////
	
	var _self = this;
	
	//Private event handlers, need different instance for each TimeInput. Proxy to prototype.
	this._onTimeInputTextFieldMouseDownInstance = 
		function (mouseEvent)
		{
			_self._onTimeInputTextFieldMouseDown(mouseEvent);
		};	
		
	this._onTimeInputTextFieldFocusOutInstance = 
		function (event)
		{
			_self._onTimeInputTextFieldFocusOut(event);
		};
		
	this._onTimeInputEnterFrameInstance = 
		function (event)
		{
			_self._onTimeInputEnterFrame(event);
		};
		
	this._textFieldHour.addEventListener("focusout", this._onTimeInputTextFieldFocusOutInstance);
	this._textFieldMinute.addEventListener("focusout", this._onTimeInputTextFieldFocusOutInstance);
	this._textFieldSecond.addEventListener("focusout", this._onTimeInputTextFieldFocusOutInstance);
	
	this._textFieldHour.addEventListener("mousedown", this._onTimeInputTextFieldMouseDownInstance);
	this._textFieldMinute.addEventListener("mousedown", this._onTimeInputTextFieldMouseDownInstance);
	this._textFieldSecond.addEventListener("mousedown", this._onTimeInputTextFieldMouseDownInstance);
	
	this.addEventListener("enterframe", this._onTimeInputEnterFrameInstance);
	
	/////
	
	//Currently focused text field
	this._textFieldFocused = null;
	this._clockBase = Date.now();
	
	var time = new Date();
	this.setHours(time.getHours());
	this.setMinutes(time.getMinutes());
	this.setSeconds(time.getSeconds());
}

//Inherit from SkinnableElement
TimeInputElement.prototype = Object.create(TextInputElement.prototype);
TimeInputElement.prototype.constructor = TimeInputElement;
TimeInputElement.base = TextInputElement;

/////////////Events////////////////////////////////////

/**
 * @event hourswrapped DispatcherEvent
 * 
 * Dispatched when hours are wrapped due to clock change.
 * This is useful to detect date or AM/PM changes due to the clock updating.
 */

/////////////Style Types///////////////////////////////

TimeInputElement._StyleTypes = Object.create(null);

/**
 * @style Is24HourTime boolean
 * 
 * Defaults to true, valid hours are 0-23.
 * When false, valid hours are 1-12. (AM / PM not supplied by this control)
 */
TimeInputElement._StyleTypes.Is24HourTime =								StyleableBase.EStyleType.NORMAL;		//Element constructor()


/////////////Default Styles///////////////////////////

TimeInputElement.StyleDefault = new StyleDefinition();

TimeInputElement.StyleDefault.setStyle("Is24HourTime", 								true);
TimeInputElement.StyleDefault.setStyle("TextHorizontalAlign", 						"center");
TimeInputElement.StyleDefault.setStyle("TextVerticalAlign", 						"middle");

TimeInputElement.StyleDefault.setStyle("PaddingLeft",								5);
TimeInputElement.StyleDefault.setStyle("PaddingRight",								5);


////////Public///////////////////////

/**
 * @function setText
 * @override
 * Sets time to be displayed. 
 * 
 * @param text String
 * Time to be displayed. Expected format is "hour:minute:second".
 */
TimeInputElement.prototype.setText = 
	function (text)
	{
		var hour = 0; 
		var minute = 0;
		var second = 0;
		var n;
		
		var timeArray = text.split(":");
		for (var i = 0; i < timeArray.length; i++)
		{
			if (i == 3)
				return;
			
			n = Number(timeArray[i])
			if (isNaN(n) == true)
				n = 0;
			
			if (i == 0)
				hour = n;
			else if (i == 1)
				minute = n;
			else 
				second = n;
		}
		
		this.setHours(hour);
		this.setMinutes(minute);
		this.setSeconds(second);		
	};

/**
 * @function getText
 * @override
 * Gets the time string currently displayed.
 * 
 * @returns String
 * Time currently displayed, formatted as "HH:MM:SS"
 */	
TimeInputElement.prototype.getText = 
	function ()
	{
		var hour = this._textFieldHour.getText();
		var minute = this._textFieldMinute.getText();
		var second = this._textFieldSecond.getText();
		
		while (hour.length < 2)
			hour = "0" + hour;
		
		while (minute.length < 2)
			minute = "0" + minute;
		
		while (second.length < 2)
			second = "0" + second;
		
		return hour + ":" + minute + ":" + second;
	};	
	
/**
 * @function setHours
 * Sets the hours to be displayed.
 * Range is 0-23 when "Is24HourTime" style is true, otherwise 1-12
 * Will wrap hours when out of range.
 * 
 * @param hour int
 * Hour to be displayed.
 */
TimeInputElement.prototype.setHours = 
	function (hour)
	{
		this._setHoursInternal(hour, false);
	};

/**
 * @function getHour
 * Gets the hours currently displayed. 
 * 
 * @returns int
 * Hour currently displayed.
 */	
TimeInputElement.prototype.getHours = 
	function ()
	{
		return Number(this._textFieldHour.getText());
	};

/**
 * @function setMinutes
 * Sets the minutes to be displayed. Range is 0-59.
 * Will wrap minutes and update hours when out of range.
 * 
 * @param minute int
 * Minute to be displayed.
 */
TimeInputElement.prototype.setMinutes = 
	function (minute)
	{
		this._setMinutesInternal(minute, false)
	};

/**
 * @function getMinutes
 * Gets the minutes currently displayed. 
 * 
 * @returns int
 * Minute currently displayed.
 */	
TimeInputElement.prototype.getMinutes = 
	function ()
	{
		return Number(this._textFieldMinute.getText());
	};	
	
/**
 * @function setSeconds
 * Sets the seconds to be displayed. Range is 0-59.
 * Will wrap seconds and update minutes when out of range.
 * 
 * @param second int
 * Seconds to be displayed.
 */
TimeInputElement.prototype.setSeconds = 
	function (second)
	{
		this._setSecondsInternal(second, false);
	};

/**
 * @function getSeconds
 * Gets the seconds currently displayed. 
 * 
 * @returns int
 * Seconds currently displayed.
 */	
TimeInputElement.prototype.getSeconds = 
	function ()
	{
		return Number(this._textFieldSecond.getText());
	};		
	

////////Internal/////////////////////

/**
 * @function _setHoursInternal
 * Sets the hours to be displayed, ignored if set by clock when user has hour field focused.
 * Range is 0-23 when "Is24HourTime" style is true, otherwise 1-12
 * 
 * @param hour int
 * Hours to be displayed.
 * 
 * @param isClock boolean
 * True when this is called via clock change.
 */	
TimeInputElement.prototype._setHoursInternal = 
	function (hour, isClock)
	{
		//Dont update if its the clock and field is focused by user
		if (isClock == true && this._textFieldFocused == this._textFieldHour)
			return;
	
		var wrapCount = 0;
		
		if (hour == null)
			hour = 0;
		
		var h = Number(hour);
		if (isNaN(h) == true)
			h = 0;
		
		h = Math.round(h);
		
		var is24Hour = this.getStyle("Is24HourTime");
		
		//Wrap the hour
		if (is24Hour == true)
		{
			while (h < 0)
			{
				h += 24;
			}
			while (h > 23)
			{
				h -= 24;
				wrapCount++;
			}
		}
		else
		{
			while (h < 0)
			{
				h += 12;
			}
			while (h > 11) //Wrap on 12
			{
				h -= 12;
				wrapCount++;
			}
			
			if (h == 0)
				h = 12;
		}
		
		var textHour = h.toString();
		while (textHour.length < 2)
			textHour = "0" + textHour;
		
		this._textFieldHour.setText(textHour);
		
		if (isClock == false)
			this._clockBase = Date.now();
		
		if (wrapCount > 0)
		{
			//Dont need to worry about wrap direction, clock always wraps forward
			if (isClock == true && this.hasEventListener("hourswrapped", null) == true)
			{
				var wrapEvent = new DispatcherEvent("hourswrapped");
			
				//This is unlikely to ever be > 1 since its a clock update, unless the control
				//has been removed and re-added after a *very* long time.
				for (var i = 0; i < wrapCount; i++)
					this.dispatchEvent(wrapEvent);
			}
		}
	};
	
/**
 * @function _setMinutesInternal
 * Sets the minutes to be displayed, ignored if set by clock when user has minutes field focused.
 * 
 * @param minute int
 * Minutes to be displayed.
 * 
 * @param isClock boolean
 * True when this is called via clock change.
 */	
TimeInputElement.prototype._setMinutesInternal = 
	function (minute, isClock)
	{
		//Dont update if its the clock and field is focused by user
		if (isClock == true && this._textFieldFocused == this._textFieldMinute)
			return;
	
		var wrapped = false;
		
		if (minute == null)
			minute = 0;
		
		var m = Number(minute);
		if (isNaN(m) == true)
			m = 0;
		
		m = Math.round(m);
		
		var currentHour = this.getHours();
		
		//Wrap minutes
		while (m > 59)
		{
			m = m - 60;
			currentHour++;
			wrapped = true;
		}
		
		while (m < 0)
		{
			m = m + 60
			currentHour--;
			wrapped = true;
		}
		
		if (wrapped == true)
			this._setHoursInternal(currentHour, isClock);
		
		var textMinute = m.toString();
		while (textMinute.length < 2)
			textMinute = "0" + textMinute;
		
		this._textFieldMinute.setText(textMinute);
		
		if (isClock == false)
			this._clockBase = Date.now();
	};
	
/**
 * @function _setSecondsInternal
 * Sets the seconds to be displayed, ignored if set by clock when user has seconds field focused.
 * 
 * @param second int
 * Seconds to be displayed.
 * 
 * @param isClock boolean
 * True when this is called via clock change.
 */		
TimeInputElement.prototype._setSecondsInternal = 
	function (second, isClock)
	{
		//Dont update if its the clock and field is focused by user	
		if (isClock == true && this._textFieldFocused == this._textFieldSecond)
			return;
	
		var wrapped = false;
		
		if (second == null)
			second = 0;
		
		var s = Number(second);
		if (isNaN(s) == true)
			m = 0;
		
		s = Math.round(s);
		
		var currentMinute = this.getMinutes();
		
		//Wrap seconds
		while (s > 59)
		{
			s = s - 60;
			currentMinute++;
			wrapped = true;
		}
		
		while (s < 0)
		{
			s = s + 60
			currentMinute--;
			wrapped = true;
		}
		
		if (wrapped == true)
			this._setMinutesInternal(currentMinute, isClock);
		
		var textSecond = s.toString();
		while (textSecond.length < 2)
			textSecond = "0" + textSecond;
		
		this._textFieldSecond.setText(textSecond);
		
		if (isClock == false)
			this._clockBase = Date.now();
	};
	
/**
 * @function _onTimeInputEnterFrame
 * Event handler for "enterframe" event.  Updates the time displayed via clock.
 * 
 * @param event DispatcherEvent
 * DispatcherEvent to be processed.
 */	
TimeInputElement.prototype._onTimeInputEnterFrame = 
	function (event)
	{
		if (this._renderVisible == false)
			return;
	
		var time = Date.now();
		var delta = time - this._clockBase;
		var deltaSeconds = Math.floor(delta / 1000);
		
		if (deltaSeconds > 0)
		{
			this._setSecondsInternal(this.getSeconds() + deltaSeconds, true);
			this._clockBase += deltaSeconds * 1000;
		}
	};
	
//@override
TimeInputElement.prototype._onTextInputKeyDown = 
	function (keyboardEvent)
	{
		if (keyboardEvent.getDefaultPrevented() == true)
			return;
		
		var key = keyboardEvent.getKey();
		
		if (key.length == 1 && 
			key != "0" && key != "1" &&
			key != "2" && key != "3" &&
			key != "4" && key != "5" &&
			key != "6" && key != "7" &&
			key != "8" && key != "9" &&
			key != ":")
		{
			return;
		}
		
		if (key == "Tab" || key == ":") //Move focus
		{
			var shiftPressed = false;
			
			if (key == "Tab")
				shiftPressed = keyboardEvent.getShift();
			
			if (shiftPressed == false)
			{
				if (this._textFieldFocused == this._textFieldSecond)
					return;
				else	
				{
					//Prevent normal tab stop handling
					keyboardEvent.preventDefault();
					
					this._textFieldFocused.dispatchEvent(new ElementEvent("focusout", false));
					
					if (this._textFieldFocused == this._textFieldHour)
						this._textFieldFocused = this._textFieldMinute;
					else //if (this._textFieldFocused == this._textFieldMinute)
						this._textFieldFocused = this._textFieldSecond;
					
					this._textFieldFocused.dispatchEvent(new ElementEvent("focusin", false));
				}
			}
			else //if (shiftPressed == true)
			{
				if (this._textFieldFocused == this._textFieldHour)
					return;
				else
				{
					//Prevent normal tab stop handling
					keyboardEvent.preventDefault();
					
					this._textFieldFocused.dispatchEvent(new ElementEvent("focusout", false));
					
					if (this._textFieldFocused == this._textFieldSecond)
						this._textFieldFocused = this._textFieldMinute;
					else //if (this._textFieldFocused == this._textFieldMinute)
						this._textFieldFocused = this._textFieldHour;
					
					this._textFieldFocused.dispatchEvent(new ElementEvent("focusin", false));
				}
			}
		}
		else
		{
			var clonedEvent = keyboardEvent.clone();
			clonedEvent._bubbles = false; //Dont bubble.
			
			//Dispatch non-bubbling keyboard event to our text field.
			this._textFieldFocused.dispatchEvent(clonedEvent);
			
			if (clonedEvent.getIsCanceled() == true)
				keyboardEvent.cancelEvent();
				
			if (clonedEvent.getDefaultPrevented() == true)
				keyboardEvent.preventDefault();
		}
	};

//@override
TimeInputElement.prototype._onTextInputKeyUp = 
	function (keyboardEvent)
	{
		if (keyboardEvent.getDefaultPrevented() == true)
			return;
		
		var key = keyboardEvent.getKey();
		
		if (key.length == 1 && 
			key != "0" && key != "1" &&
			key != "2" && key != "3" &&
			key != "4" && key != "5" &&
			key != "6" && key != "7" &&
			key != "8" && key != "9")
		{
			return;
		}
		
		var clonedEvent = keyboardEvent.clone();
		clonedEvent._bubbles = false; //Dont bubble.
		
		//Dispatch non-bubbling keyboard event to our text field.
		this._textFieldFocused.dispatchEvent(clonedEvent);
		
		if (clonedEvent.getIsCanceled() == true)
			keyboardEvent.cancelEvent();
			
		if (clonedEvent.getDefaultPrevented() == true)
			keyboardEvent.preventDefault();
	};	
	
//@override	
TimeInputElement.prototype._onTextInputFocusIn = 
	function (elementEvent)
	{
		//Mousedown already focused
		if (this._textFieldFocused != null)
			return;
	
		//This only works because TextField doesnt look at _isFocused (manages caret state with different flag)
		this._textFieldHour.dispatchEvent(elementEvent.clone()); 
		this._textFieldFocused = this._textFieldHour;
	};

//@override	
TimeInputElement.prototype._onTextInputFocusOut = 
	function (elementEvent)
	{
		//This only works because TextField doesnt look at _isFocused (manages caret state with different flag)
		this._textFieldFocused.dispatchEvent(elementEvent.clone());
		this._textFieldFocused = null; 
	};

/**
 * @function _onTimeInputTextFieldMouseDown
 * Event handler for the internal TextField's "mousedown" event.
 * 
 * @param mouseEvent ElementMouseEvent
 * ElementMouseEvent to process.
 */	
TimeInputElement.prototype._onTimeInputTextFieldMouseDown = 
	function (mouseEvent)
	{
		if (mouseEvent.getTarget() != this._textFieldFocused && this._textFieldFocused != null)
			this._textFieldFocused.dispatchEvent(new ElementEvent("focusout", false));
		
		this._textFieldFocused = mouseEvent.getTarget();
	};	
	
/**
 * @function _onTimeInputTextFieldFocusOut
 * Event handler for the internal TextField's "focusout" event.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to process.
 */		
TimeInputElement.prototype._onTimeInputTextFieldFocusOut = 
	function (event)
	{
		var textField = event.getTarget();
		var value = Number(textField.getText());
		
		var changed = false;
		
		if (textField == this._textFieldHour)
		{
			var is24Hour = this.getStyle("Is24HourTime");
			
			if (is24Hour == true)
			{
				if (value < 0 || value > 23)
					changed = true;
			}
			else
			{
				if (value < 1 || value > 12)
					changed = true;
			}
			
			if (changed == true)
				this._setHoursInternal(value);
		}
		else if (textField == this._textFieldMinute && (value < 0 || value > 59))
		{
			changed = true;	
			this._setMinutesInternal(value);
		}
		else if (textField == this._textFieldSecond && (value < 0 || value > 59))
		{
			changed = true;
			this._setSecondsInternal(value);
		}
		
		//Dispatch a changed event if the focus out caused values to change (wrapped)
		if (changed == true && this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};
	
//@override
TimeInputElement.prototype._updateTextColors = 
	function ()
	{
		this._textFieldHour.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._textFieldHour.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._textFieldHour.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._labelColon1.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._labelColon1.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._labelColon1.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._textFieldMinute.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._textFieldMinute.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._textFieldMinute.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._labelColon2.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._labelColon2.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._labelColon2.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._textFieldSecond.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._textFieldSecond.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._textFieldSecond.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
	};
	
//@override
TimeInputElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		//TimeInputElement.base.base - skip TextInput._doStylesUpdated()
		TimeInputElement.base.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		//Force the textField to use our defaults rather than inherited.
		if ("TextHorizontalAlign" in stylesMap)
		{
			this._textFieldHour.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
			this._textFieldMinute.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
			this._textFieldSecond.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
		}
		
		if ("TextVerticalAlign" in stylesMap)
		{
			this._textFieldHour.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			this._textFieldMinute.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			this._textFieldSecond.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			
			this._labelColon1.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			this._labelColon2.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
		}
		
		if ("Enabled" in stylesMap)
		{
			var enabled = this.getStyle("Enabled");
			
			this._textFieldHour.setStyle("Enabled", enabled);
			this._textFieldMinute.setStyle("Enabled", enabled);
			this._textFieldSecond.setStyle("Enabled", enabled);
			
			if (enabled == true)
			{
				if (this.hasEventListener("keydown", this._onTextInputKeyUpDownInstance) == false)
					this.addEventListener("keydown", this._onTextInputKeyUpDownInstance);
				
				if (this.hasEventListener("keyup", this._onTextInputKeyUpDownInstance) == false)
					this.addEventListener("keyup", this._onTextInputKeyUpDownInstance);
				
				if (this._textFieldHour.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == false)
					this._textFieldHour.addEventListener("changed", this._onTextInputTextFieldChangedInstance);		
				
				if (this._textFieldMinute.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == false)
					this._textFieldMinute.addEventListener("changed", this._onTextInputTextFieldChangedInstance);	
				
				if (this._textFieldSecond.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == false)
					this._textFieldSecond.addEventListener("changed", this._onTextInputTextFieldChangedInstance);	
			}
			else
			{
				if (this.hasEventListener("keydown", this._onTextInputKeyUpDownInstance) == true)
					this.removeEventListener("keydown", this._onTextInputKeyUpDownInstance);
				
				if (this.hasEventListener("keyup", this._onTextInputKeyUpDownInstance) == true)
					this.removeEventListener("keyup", this._onTextInputKeyUpDownInstance);
				
				if (this._textFieldHour.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == true)
					this._textFieldHour.removeEventListener("changed", this._onTextInputTextFieldChangedInstance);
				
				if (this._textFieldMinute.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == true)
					this._textFieldMinute.removeEventListener("changed", this._onTextInputTextFieldChangedInstance);
				
				if (this._textFieldSecond.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == true)
					this._textFieldSecond.removeEventListener("changed", this._onTextInputTextFieldChangedInstance);
			}
		}
		
		if ("TextLinePaddingTop" in stylesMap || 
			"TextLinePaddingBottom" in stylesMap)
		{
			this._invalidateMeasure();
		}
		
		if ("Padding" in stylesMap ||
			"PaddingTop" in stylesMap ||
			"PaddingBottom" in stylesMap ||
			"PaddingLeft" in stylesMap ||
			"PaddingRight" in stylesMap)
		{
			var paddingSize = this._getPaddingSize();
			
			this._textFieldHour.setStyle("PaddingLeft", paddingSize.paddingLeft);
			this._textFieldHour.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textFieldHour.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._labelColon1.setStyle("PaddingTop", paddingSize.paddingTop);
			this._labelColon1.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._textFieldMinute.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textFieldMinute.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._labelColon2.setStyle("PaddingTop", paddingSize.paddingTop);
			this._labelColon2.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._textFieldSecond.setStyle("PaddingRight", paddingSize.paddingRight);
			this._textFieldSecond.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textFieldSecond.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._invalidateMeasure();
		}
		
		//Will adjust 24 -> 12 hour
		if ("Is24HourTime" in stylesMap)
			this.setHours(this.getHours());
		
		this._updateSkinStyles(stylesMap);
	};
	
//@override
TimeInputElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var fontString = this._getFontString();
	
		var measuredHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
		
		var measuredWidth = (CanvasElement._measureText("00", fontString) + 6) * 3;
		measuredWidth += this._labelColon1._getStyledOrMeasuredWidth() * 2;
		
		measuredWidth += padWidth;
		measuredHeight += padHeight;
	
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};
	
//@override	
TimeInputElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		//TimeInputElement.base.base - Skip TextInput._doLayout()
		TimeInputElement.base.base.prototype._doLayout.call(this, paddingMetrics);
		
		//Ignore padding, proxied to TextFields for proper mouse handling.		
		this._listContainer._setActualPosition(0, 0);
		this._listContainer._setActualSize(this._width, this._height);
	};
	
	
	


/**
 * @depends TextInputElement.js
 */

/////////////////////////////////////////////////////////
///////////////TextAreaElement///////////////////////////	
	
/**
 * @class TextAreaElement
 * @inherits TextInputElement
 * 
 * TextAreaElement is a skin-able multi-line editable text box
 * which supports two skin states, "up" and "disabled".
 * Scroll bars will optionally be added when the text becomes
 * larger than the TextArea's visible region.
 * 
 * @constructor TextAreaElement 
 * Creates new TextAreaElement instance.
 */
function TextAreaElement() //extends TextInputElement
{
	TextAreaElement.base.prototype.constructor.call(this);
	
	this._horizontalScrollBar = null;
	this._verticalScrollBar = null;
	
	var _self = this;
	
	this._onTextAreaTextFieldLayoutCompleteInstance = 
		function (event)
		{
			_self._onTextAreaTextFieldLayoutComplete(event);
		};
	
	this._onTextAreaMouseWheelEventInstance = 
		function (elementMouseWheelEvent)
		{
			_self._onTextAreaMouseWheelEvent(elementMouseWheelEvent);
		};
		
	this._onTextAreaScrollBarChangeInstance =
		function (elementEvent)
		{
			_self._onTextAreaScrollBarChange(elementEvent);
		};	
		
	this.addEventListener("wheel", this._onTextAreaMouseWheelEventInstance);	
	this._textField.addEventListener("layoutcomplete", this._onTextAreaTextFieldLayoutCompleteInstance);
}

//Inherit from TextInputElement
TextAreaElement.prototype = Object.create(TextInputElement.prototype);
TextAreaElement.prototype.constructor = TextAreaElement;
TextAreaElement.base = TextInputElement;


/////////////Style Types///////////////////////////////

TextAreaElement._StyleTypes = Object.create(null);

/**
 * @style Selectable boolean
 * When true, text can be highlighted and copied.
 */
TextAreaElement._StyleTypes.Selectable = 						StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style Multiline boolean
 * When true, newline characters are respected and text will be rendered on multiple lines if necessary.
 */
TextAreaElement._StyleTypes.Multiline = 						StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style WordWrap boolean
 * When true, text will wrap when width is constrained and will be rendered on multiple lines if necessary. 
 */
TextAreaElement._StyleTypes.WordWrap = 							StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style MeasureContentWidth boolean
 * When true, the TextArea's measured width will use its text measured width. 
 * Use this when you want the TextArea to expand its width when possible rather than scroll, 
 * causing scrolling to happen on a parent viewport.
 */
TextAreaElement._StyleTypes.MeasureContentWidth = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style MeasureContentHeight boolean
 * When true, the TextArea's measured height will use its text measured height.
 * Use this when you want the TextArea to expand when its height possible rather than scroll, 
 * causing scrolling to happen on a parent viewport.
 */
TextAreaElement._StyleTypes.MeasureContentHeight = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style HorizontalScrollBarDisplay String
 * Determines the behavior of the horizontal scroll bar. Allowable values are "on", "off", or "auto".
 */
TextAreaElement._StyleTypes.HorizontalScrollBarDisplay = 		StyleableBase.EStyleType.NORMAL;		// "on" || "off" || "auto"

/**
 * @style HorizontalScrollBarPlacement String
 * Determines the position of the horizontal scroll bar. Allowable values are "top" or "bottom".
 */
TextAreaElement._StyleTypes.HorizontalScrollBarPlacement = 		StyleableBase.EStyleType.NORMAL;		// "top" || "bottom"

/**
 * @style VerticalScrollBarDisplay String
 * Determines the behavior of the vertical scroll bar. Allowable values are "on", "off", or "auto".
 */
TextAreaElement._StyleTypes.VerticalScrollBarDisplay = 			StyleableBase.EStyleType.NORMAL;		// "on" || "off" || "auto"

/**
 * @style VerticalScrollBarPlacement String
 * Determines the position of the vertical scroll bar. Allowable values are "left" or "right".
 */
TextAreaElement._StyleTypes.VerticalScrollBarPlacement = 		StyleableBase.EStyleType.NORMAL;		// "left" || "right"

//ScrollBar styles.
/**
 * @style HorizontalScrollBarStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to be applied to the horizontal scroll bar.
 */
TextAreaElement._StyleTypes.HorizontalScrollBarStyle = 			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style VerticalScrollBarStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to be applied to the vertical scroll bar.
 */
TextAreaElement._StyleTypes.VerticalScrollBarStyle = 			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition


////////////Default Styles////////////////////////////

TextAreaElement.StyleDefault = new StyleDefinition();

TextAreaElement.StyleDefault.setStyle("MinWidth", 									10);
TextAreaElement.StyleDefault.setStyle("MinHeight", 									10);

TextAreaElement.StyleDefault.setStyle("TextHorizontalAlign", 						"left");
TextAreaElement.StyleDefault.setStyle("TextVerticalAlign", 							"top");

TextAreaElement.StyleDefault.setStyle("Selectable", 								true);
TextAreaElement.StyleDefault.setStyle("Multiline", 									true);
TextAreaElement.StyleDefault.setStyle("WordWrap", 									true);

TextAreaElement.StyleDefault.setStyle("Cursor", 									null);

TextAreaElement.StyleDefault.setStyle("HorizontalScrollBarDisplay", 				"auto");
TextAreaElement.StyleDefault.setStyle("HorizontalScrollBarPlacement", 				"bottom");

TextAreaElement.StyleDefault.setStyle("VerticalScrollBarDisplay", 					"auto");
TextAreaElement.StyleDefault.setStyle("VerticalScrollBarPlacement", 				"right");

TextAreaElement.StyleDefault.setStyle("HorizontalScrollBarStyle", 					null);
TextAreaElement.StyleDefault.setStyle("VerticalScrollBarStyle", 					null);

TextAreaElement.StyleDefault.setStyle("MeasureContentWidth", 						false);
TextAreaElement.StyleDefault.setStyle("MeasureContentHeight", 						false);


/////////////Internal Functions///////////////////

/**
 * @function _onTextAreaTextFieldLayoutComplete
 * Event handler for the TextField "layoutcomplete" event. 
 * Invalidates measure and layout to adjust scroll bars.
 * 
 * @param event DispatcherEvent
 * The DispatcherEvent to process.
 */	
TextAreaElement.prototype._onTextAreaTextFieldLayoutComplete = 
	function (event)
	{
		//Our layout is dependent on the TextField layout for determining
		//scroll bars and actual content size. TextField itself requires
		//multiple layout passes due to the word wrap. If not careful
		//we can end up with an infinite loop because of this dependency.
	
		this._invalidateMeasure();
		this._invalidateLayout();
	};

/**
 * @function _onTextAreaScrollBarChange
 * Event handler for the scroll bar "changed" event. 
 * Updates the child elements position within the Viewport.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
TextAreaElement.prototype._onTextAreaScrollBarChange = 
	function (elementEvent)
	{
		if (elementEvent.getTarget() == this._horizontalScrollBar)
			this._textField._setHorizontalScrollValue(this._horizontalScrollBar.getScrollValue());
		else // if (elementEvent.getTarget() == this._verticalScrollBar)
			this._textField._setVerticalScrollValue(this._verticalScrollBar.getScrollValue());
	};

/**
 * @function _onTextAreaMouseWheelEvent
 * Event handler for the Viewport's "wheel" event. Starts the scroll bar tween.
 * 
 * @param elementMouseWheelEvent ElementMouseWheelEvent
 * The ElementMouseWheelEvent to process.
 */		
TextAreaElement.prototype._onTextAreaMouseWheelEvent = 
	function (elementMouseWheelEvent)
	{
		if (elementMouseWheelEvent.getDefaultPrevented() == true)
			return;
	
		var consumeEvent = false;
		
		var scrollPageSize = null;
		var scrollViewSize = null;
		var scrollLineSize = null;
		var scrollValue = null;
		var maxScrollValue = null;
		
		var deltaX = elementMouseWheelEvent.getDeltaX();
		var deltaY = elementMouseWheelEvent.getDeltaY();
		
		if (deltaX != 0 && this._horizontalScrollBar != null)
		{
			scrollPageSize = this._horizontalScrollBar.getScrollPageSize();
			scrollViewSize = this._horizontalScrollBar.getScrollViewSize();
			scrollLineSize = this._horizontalScrollBar.getScrollLineSize();
			
			maxScrollValue = scrollPageSize - scrollViewSize;
			if (maxScrollValue > 0)
			{
				scrollValue = this._horizontalScrollBar.getTweenToValue();
				if (scrollValue == null)
					scrollValue = this._horizontalScrollBar.getScrollValue();
				
				if (deltaX < 0 && scrollValue > 0)
				{
					this._horizontalScrollBar.startScrollTween(Math.max(scrollValue + (deltaX * (scrollLineSize * 3)), 0));
					consumeEvent = true;
				}
				else if (deltaX > 0 && scrollValue < maxScrollValue)
				{
					this._horizontalScrollBar.startScrollTween(Math.min(scrollValue + (deltaX * (scrollLineSize * 3)), maxScrollValue));
					consumeEvent = true;
				}
			}
		}
		
		if (deltaY != 0 && this._verticalScrollBar != null)
		{
			scrollPageSize = this._verticalScrollBar.getScrollPageSize();
			scrollViewSize = this._verticalScrollBar.getScrollViewSize();
			scrollLineSize = this._verticalScrollBar.getScrollLineSize();
			
			maxScrollValue = scrollPageSize - scrollViewSize;
			if (maxScrollValue > 0)
			{
				scrollValue = this._verticalScrollBar.getTweenToValue();
				if (scrollValue == null)
					scrollValue = this._verticalScrollBar.getScrollValue();
				
				if (deltaY < 0 && scrollValue > 0)
				{
					this._verticalScrollBar.startScrollTween(Math.max(scrollValue + (deltaY * (scrollLineSize * 3)), 0));
					consumeEvent = true;
				}
				else if (deltaY > 0 && scrollValue < maxScrollValue)
				{
					this._verticalScrollBar.startScrollTween(Math.min(scrollValue + (deltaY * (scrollLineSize * 3)), maxScrollValue));
					consumeEvent = true;
				}
			}
		}
		
		//We've consumed the wheel event, don't want parents double scrolling.
		if (consumeEvent == true)
		{
			elementMouseWheelEvent.preventDefault();
			this._invalidateLayout();
		}
	};

//@override
TextAreaElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		TextAreaElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		if ("Selectable" in stylesMap)
			this._textField.setStyle("Selectable", this.getStyle("Selectable"));
		
		if ("Multiline" in stylesMap)
			this._textField.setStyle("Multiline", this.getStyle("Multiline"));
		
		if ("WordWrap" in stylesMap)
			this._textField.setStyle("WordWrap", this.getStyle("WordWrap"));
		
		if ("HorizontalScrollBarDisplay" in stylesMap ||
			"VerticalScrollBarDisplay" in stylesMap)
		{
			this._invalidateLayout();
			this._invalidateMeasure();
		}
		else 
		{	
			if ("HorizontalScrollBarPlacement" in stylesMap ||
				"VerticalScrollBarPlacement" in stylesMap)
			{
				this._invalidateLayout();
			}
			
			if ("MeasureContentWidth" in stylesMap || 
				"MeasureContentHeight" in stylesMap)
			{
				this._invalidateMeasure();
			}
		}
		
		if ("HorizontalScrollBarStyle" in stylesMap && this._horizontalScrollBar != null)
			this._applySubStylesToElement("HorizontalScrollBarStyle", this._horizontalScrollBar);
		if ("VerticalScrollBarStyle" in stylesMap && this._verticalScrollBar != null)
			this._applySubStylesToElement("VerticalScrollBarStyle", this._verticalScrollBar);
	};
	
//@override
TextAreaElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var w = 0;
		var h = 0;
		
		if (this.getStyle("MeasureContentWidth") == true)
			w += this._textField._getStyledOrMeasuredWidth();
		
		if (this.getStyle("MeasureContentHeight") == true)
			h += this._textField._getStyledOrMeasuredHeight();
		
		if (this._verticalScrollBar != null)
			w += this._verticalScrollBar._getStyledOrMeasuredWidth();

		if (this._horizontalScrollBar != null)
			h += this._horizontalScrollBar._getStyledOrMeasuredHeight();
		
		//Padding included by textField
		this._setMeasuredSize(w, h);
	};	

//@override
TextAreaElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		//base.base, skipping TextInput _doLayout()
		TextAreaElement.base.base.prototype._doLayout.call(this, paddingMetrics);
		
		var hDisplay = this.getStyle("HorizontalScrollBarDisplay");
		var vDisplay = this.getStyle("VerticalScrollBarDisplay");
		
		var horizontalBarPlacement = this.getStyle("HorizontalScrollBarPlacement");
		var verticalBarPlacement = this.getStyle("VerticalScrollBarPlacement");
		
		var vScrollParams = this._textField._getVerticalScrollParameters();
		var hScrollParams = this._textField._getHorizontalScrollParameters();
		
		var needsHScroll = false;
		var needsVScroll = false;
		
		//We need the scroll bar.
		if (hDisplay == "on" || (hDisplay == "auto" && hScrollParams.page > hScrollParams.view))
			needsHScroll = true;
			
		if (vDisplay == "on" || (vDisplay == "auto" && vScrollParams.page > vScrollParams.view))
			needsVScroll = true;
		
		//Destroy
		if (needsHScroll == false)
		{
			if (this._horizontalScrollBar != null)
			{
				this._removeChild(this._horizontalScrollBar);
				this._horizontalScrollBar = null;
			}
		}
		else //Create
		{
			if (this._horizontalScrollBar == null)
			{
				this._horizontalScrollBar = new ScrollBarElement();
				this._applySubStylesToElement("HorizontalScrollBarStyle", this._horizontalScrollBar);
				this._horizontalScrollBar.setStyle("LayoutDirection", "horizontal");
				this._horizontalScrollBar.addEventListener("changed", this._onTextAreaScrollBarChangeInstance);
				
				this._addChild(this._horizontalScrollBar);
			}
		}
		
		//Destroy
		if (needsVScroll == false)
		{
			if (this._verticalScrollBar != null)
			{
				this._removeChild(this._verticalScrollBar);
				this._verticalScrollBar = null;
			}
		}
		else //Create
		{
			if (this._verticalScrollBar == null)
			{
				this._verticalScrollBar = new ScrollBarElement();
				this._applySubStylesToElement("VerticalScrollBarStyle", this._verticalScrollBar);
				this._verticalScrollBar.setStyle("LayoutDirection", "vertical");
				this._verticalScrollBar.addEventListener("changed", this._onTextAreaScrollBarChangeInstance);

				this._addChild(this._verticalScrollBar);
			}
		}
		
		var verticalBarWidth = 0;
		var horizontalBarHeight = 0;
		
		//Setup vertical scroll bar
		if (this._verticalScrollBar != null)
		{
			this._verticalScrollBar.setScrollPageSize(vScrollParams.page);
			this._verticalScrollBar.setScrollViewSize(vScrollParams.view);
			this._verticalScrollBar.setScrollLineSize(vScrollParams.line);
			this._verticalScrollBar.setScrollValue(vScrollParams.value);
			
			verticalBarWidth = this._verticalScrollBar._getStyledOrMeasuredWidth();
		}
		
		//Setup horizontal scroll bar
		if (this._horizontalScrollBar != null)
		{
			this._horizontalScrollBar.setScrollPageSize(hScrollParams.page);
			this._horizontalScrollBar.setScrollViewSize(hScrollParams.view);
			this._horizontalScrollBar.setScrollLineSize(hScrollParams.line);
			this._horizontalScrollBar.setScrollValue(hScrollParams.value);
			
			horizontalBarHeight = this._horizontalScrollBar._getStyledOrMeasuredHeight();
		}			

		//Size and position vertical scroll bar
		if (this._verticalScrollBar != null)
		{
			this._verticalScrollBar._setActualSize(verticalBarWidth, this._height - horizontalBarHeight);
			
			if (verticalBarPlacement == "left")
			{
				if (horizontalBarPlacement == "top")
					this._verticalScrollBar._setActualPosition(0, horizontalBarHeight);
				else
					this._verticalScrollBar._setActualPosition(0, 0);
			}
			else //if (verticalBarPlacement == "right")
			{
				if (horizontalBarPlacement == "top")
					this._verticalScrollBar._setActualPosition(this._width - verticalBarWidth, horizontalBarHeight);
				else
					this._verticalScrollBar._setActualPosition(this._width - verticalBarWidth, 0);
			}				
		}
		
		//Size and position horizontal scroll bar
		if (this._horizontalScrollBar != null)
		{
			this._horizontalScrollBar._setActualSize(this._width - verticalBarWidth, horizontalBarHeight);
			
			if (horizontalBarPlacement == "top")
			{
				if (verticalBarPlacement == "left")
					this._horizontalScrollBar._setActualPosition(verticalBarWidth, 0);
				else
					this._horizontalScrollBar._setActualPosition(0, 0);
			}
			else //if (horizontalBarPlacement == "bottom")
			{
				if (verticalBarPlacement == "left")
					this._horizontalScrollBar._setActualPosition(verticalBarWidth, this._height - horizontalBarHeight);
				else
					this._horizontalScrollBar._setActualPosition(0, this._height - horizontalBarHeight);
			}	
		}
		
		//Size and position the TextField
		this._textField._setActualSize(this._width - verticalBarWidth, this._height - horizontalBarHeight);
		
		var tfX = 0;
		if (verticalBarPlacement == "left")
			tfX = verticalBarWidth;
		
		var tfY = 0;
		if (horizontalBarPlacement == "top")
			tfY = horizontalBarHeight;
		
		this._textField._setActualPosition(tfX, tfY);
	};



/**
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////
/////////////ProgressElement///////////////////////////
	
/**
 * @class ProgressElement
 * 
 * ProgressElement is a CanvasElement that adds a second FillBase style
 * called "ProgressFill". The progress fill is drawn on top of the background fill
 * and clipped according to the current progress value. This can be used to make
 * progress bar type elements of any shape or styling.
 * 
 * @constructor ProgressElement 
 * Creates new ProgressElement instance.
 */
function ProgressElement() //extends CanvasElement
{
	ProgressElement.base.prototype.constructor.call(this);
	
	this._progressValue = 0;
	
	//Storage for the current progress fill FillBase() per styling. We need to store a reference 
	//because we listen for style changed events and need to be able to remove the listener when
	//this is changed (via styles) or added/removed to display chain.
	this._progressFill = null;
	
	
	var _self = this;
	
	this._onProgressFillStyleChangedInstance = 
		function (styleChangedEvent)
		{
			_self._onProgressFillStyleChanged(styleChangedEvent);
		};
}

//Inherit from CanvasElement
ProgressElement.prototype = Object.create(CanvasElement.prototype);
ProgressElement.prototype.constructor = ProgressElement;
ProgressElement.base = CanvasElement;


/////////////Style Types///////////////////////////////

ProgressElement._StyleTypes = Object.create(null);


/**
 * @style ProgressFill FillBase
 * 
 * Fill to use when filling the progress indicator. May be any FillBase subclass instance or "color" string. 
 */
ProgressElement._StyleTypes.ProgressFill = 								StyleableBase.EStyleType.NORMAL;		// FillBase() || "#FF0000" || null

/**
 * @style ProgressFillStart String
 * 
 * Determines the start position of the progress fill indicator, available values are "top", "bottom", "left" or "right". 
 */
ProgressElement._StyleTypes.ProgressFillStart = 						StyleableBase.EStyleType.NORMAL;		// "top" || "bottom" || "left" || "right"


////////////Default Styles///////////////////////////

ProgressElement.StyleDefault = new StyleDefinition();

ProgressElement.StyleDefault.setStyle("BackgroundFill", 				"#CCCCCC");
ProgressElement.StyleDefault.setStyle("ProgressFill", 					"#00FF00");	
ProgressElement.StyleDefault.setStyle("ProgressFillStart", 				"left");



///////Public////////

/**
 * @function setProgressValue
 * Sets the progress value, this determines how much of the progress bar will be filled.
 * 
 * @param value Number
 * Value between 0 and 1 (inclusive)
 */	
ProgressElement.prototype.setProgressValue = 
	function (value)
	{
		if (value < 0)
			value = 0;
		if (value > 1)
			value = 1;
		
		this._progressValue = value;
		
		this._invalidateRender();
	};

/**
 * @function getProgressValue
 * Gets the progress value, this determines how much of the progress bar will be filled.
 * 
 * @returns Number
 * Value between 0 and 1 (inclusive)
 */		
ProgressElement.prototype.getProgressValue = 
	function ()
	{
		return this._progressValue;
	};

////////Internal////////	
	
ProgressElement.prototype._onProgressFillStyleChanged = 
	function (styleChangedEvent)
	{
		this._invalidateRender();
	};
	
//@override
ProgressElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ProgressElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		//Copied from CanvasElement
		if ("ProgressFill" in stylesMap)
		{
			var bgFill = this.getStyle("ProgressFill"); //FillBase or color string
			
			//Only handle if changed, attached/detached handles initial add/remove listener.
			if (bgFill != this._progressFill)
			{
				//Check if new fill is solid (SolidFill or color string)
				var isSolidFillOrColor = false;
				if ((bgFill != null && bgFill instanceof FillBase == false) || bgFill instanceof SolidFill) //We're a color or a SolidFill class
					isSolidFillOrColor = true;
				
				if (this._progressFill instanceof SolidFill == true && isSolidFillOrColor == true) //Existing and new are both SolidFill
				{
					if (bgFill instanceof SolidFill == true) //Swap the solid fill classes
					{
						this._progressFill.removeEventListener("stylechanged", this._onProgressFillStyleChangedInstance);
						this._progressFill = bgFill;
						this._progressFill.addEventListener("stylechanged", this._onProgressFillStyleChangedInstance);
						
						this._invalidateRender();
					}
					else //Set the color to the current SolidFill
						this._progressFill.setStyle("FillColor", bgFill); //Will invalidate render if fill color changed
				}
				else //Definately different fill classes
				{
					//Purge the old background fill
					if (this._progressFill != null)
						this._progressFill.removeEventListener("stylechanged", this._onProgressFillStyleChangedInstance);
					
					this._progressFill = null;
					
					if (bgFill != null)
					{
						if (bgFill instanceof FillBase == false) //color
						{
							//Create new solid fill
							this._progressFill = new SolidFill();
							this._progressFill.setStyle("FillColor", bgFill);
						}
						else //Fill class
							this._progressFill = bgFill;
						
						this._progressFill.addEventListener("stylechanged", this._onProgressFillStyleChangedInstance);
					}
					
					this._invalidateRender();
				}
			}
		}
		
		if ("ProgressFillStart" in stylesMap)
			this._invalidateRender();
	};	

//@override	
ProgressElement.prototype._onCanvasElementAdded = 
	function (addedRemovedEvent)
	{
		ProgressElement.base.prototype._onCanvasElementAdded.call(this, addedRemovedEvent);
		
		if (this._progressFill != null && this._progressFill.hasEventListener("stylechanged", this._onProgressFillStyleChangedInstance) == false)
			this._progressFill.addEventListener("stylechanged", this._onProgressFillStyleChangedInstance);
	};
	
//@override	
ProgressElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		ProgressElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		if (this._progressFill != null && this._progressFill.hasEventListener("stylechanged", this._onProgressFillStyleChangedInstance) == true)
			this._progressFill.removeEventListener("stylechanged", this._onProgressFillStyleChangedInstance);
	};		
	
//override	
ProgressElement.prototype._fillBackground = 
	function (borderMetrics)
	{
		ProgressElement.base.prototype._fillBackground.call(this, borderMetrics);
	
		if (this._progressFill == null || this._progressValue == 0)
			return;
		
		var ctx = this._getGraphicsCtx();
		var fillStart = this.getStyle("ProgressFillStart");
		
		var fillSize;
		if (fillStart == "top" || fillStart == "bottom")
			fillSize = Math.round(borderMetrics._height * this._progressValue);
		else
			fillSize = Math.round(borderMetrics._width * this._progressValue);
		
		//Clip the region we need to re-draw
		if (this._progressValue < 1)
		{
			if (this._progressValue > 0)
			{
				ctx.save();
				
				ctx.beginPath();
				
				if (fillStart == "left")
				{
					ctx.moveTo(borderMetrics._x, borderMetrics._y);
					ctx.lineTo(borderMetrics._x + fillSize, borderMetrics._y);
					ctx.lineTo(borderMetrics._x + fillSize, borderMetrics._y + borderMetrics._height);
					ctx.lineTo(borderMetrics._x, borderMetrics._y + borderMetrics._height);
					ctx.closePath();
				}
				else if (fillStart == "bottom")
				{
					ctx.moveTo(borderMetrics._x, borderMetrics._y + borderMetrics._height - fillSize);
					ctx.lineTo(borderMetrics._x + borderMetrics._width, borderMetrics._y + borderMetrics._height - fillSize);
					ctx.lineTo(borderMetrics._x + borderMetrics._width, borderMetrics._y + borderMetrics._height);
					ctx.lineTo(borderMetrics._x, borderMetrics._y + borderMetrics._height);
					ctx.closePath();
				}
				else if (fillStart == "top")
				{
					ctx.moveTo(borderMetrics._x, borderMetrics._y);
					ctx.lineTo(borderMetrics._x + borderMetrics._width, borderMetrics._y);
					ctx.lineTo(borderMetrics._x + borderMetrics._width, borderMetrics._y + fillSize);
					ctx.lineTo(borderMetrics._x, borderMetrics._y + fillSize);
					ctx.closePath();
				}
				else //if (fillStart == "right")
				{
					ctx.moveTo(borderMetrics._x + borderMetrics._width - fillSize, borderMetrics._y);
					ctx.lineTo(borderMetrics._x + borderMetrics._width, borderMetrics._y);
					ctx.lineTo(borderMetrics._x + borderMetrics._width, borderMetrics._y + borderMetrics._height);
					ctx.lineTo(borderMetrics._x + borderMetrics._width - fillSize, borderMetrics._y + borderMetrics._height);
					ctx.closePath();
				}
		
				ctx.clip();
				
				ctx.beginPath();
				this._drawBackgroundShape(ctx, borderMetrics);
				this._progressFill.drawFill(ctx, borderMetrics);
				
				ctx.restore();
			}
		}
		else
		{
			ctx.beginPath();
			this._drawBackgroundShape(ctx, borderMetrics);
			this._progressFill.drawFill(ctx, borderMetrics);
		}
	};
	
	


/**
 * @depends CanvasElement.js
 */

/////////////////////////////////////////////////////////
/////////////////LabelElement////////////////////////////	
	
/**
 * @class LabelElement
 * @inherits CanvasElement
 * 
 * Basic label for rendering single line style-able text. 
 * Can be styled to automatically truncate text to fit the available 
 * space with a supplied string like ellipses "...".
 * 
 * 
 * @constructor LabelElement 
 * Creates new LabelElement instance.
 */
function LabelElement()
{
	LabelElement.base.prototype.constructor.call(this);
	
	this._textWidth = null;
	this._textHeight = null;
	this._truncateStringWidth = null;
}

//Inherit from CanvasElement
LabelElement.prototype = Object.create(CanvasElement.prototype);
LabelElement.prototype.constructor = LabelElement;
LabelElement.base = CanvasElement;

/////////////Style Types///////////////////////////////

LabelElement._StyleTypes = Object.create(null);

/**
 * @style Text String
 * Text to be rendered by the Label.
 */
LabelElement._StyleTypes.Text = 				StyleableBase.EStyleType.NORMAL;		// "any string" || null

/**
 * @style TruncateToFit String
 * String to use when truncating a label that does not fit the available area. Defaults to "...".
 */
LabelElement._StyleTypes.TruncateToFit = 		StyleableBase.EStyleType.NORMAL;		// null || "string" ("...")


////////////Default Styles////////////////////////////

LabelElement.StyleDefault = new StyleDefinition();

//Override base class styles
LabelElement.StyleDefault.setStyle("PaddingTop",					2);
LabelElement.StyleDefault.setStyle("PaddingBottom",					2);
LabelElement.StyleDefault.setStyle("PaddingLeft",					2);
LabelElement.StyleDefault.setStyle("PaddingRight",					2);

//LabelElement specific styles
LabelElement.StyleDefault.setStyle("Text", 							null);
LabelElement.StyleDefault.setStyle("TruncateToFit", 				"...");


/////////////LabelElement Internal Functions///////////////////

//@Override
LabelElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		LabelElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		if ("TextStyle" in stylesMap ||
			"TextFont" in stylesMap ||
			"TextSize" in stylesMap ||
			"Text" in stylesMap ||
			"TextLinePaddingTop" in stylesMap || 
			"TextLinePaddingBottom" in stylesMap)
		{
			this._textWidth = null;
			this._textHeight = null;
			
			this._invalidateMeasure();
			this._invalidateRender();
		}
		
		if ("TruncateToFit" in stylesMap)
		{
			this._truncateStringWidth = null;
			
			this._invalidateRender();
		}
		
		if ("TextHorizontalAlign" in stylesMap ||
			"TextVerticalAlign" in stylesMap ||
			"TextColor" in stylesMap ||
			"TextFillType" in stylesMap ||
			"TextDecoration" in stylesMap)
		{
			this._invalidateRender();
		}
	};	
	
//@Override
LabelElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		if (this._textWidth == null || this._textHeight == null)
		{
			var measureText = this.getStyle("Text");
			if (measureText == null)
				measureText = "";
		
			this._textHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
			this._textWidth = CanvasElement._measureText(measureText, this._getFontString());
		}
		
		this._setMeasuredSize(this._textWidth + padWidth, this._textHeight + padHeight);
	};	

//@override
LabelElement.prototype._doRender = 
	function ()
	{
		LabelElement.base.prototype._doRender.call(this);
		
		var text = this.getStyle("Text");
		if (text == null || text.length == 0)
			return;
		
		var ctx = this._getGraphicsCtx();
		var paddingMetrics = this._getPaddingMetrics();
		
		//For convienence
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		var fontString = this._getFontString();
		var totalWidth =  this._textWidth;
		
		//Truncate if necessary
		if (totalWidth > w)
		{
			var truncateString = this.getStyle("TruncateToFit");
			
			//Get number of truncate chars
			var numTruncateChars = 0;
			if (truncateString != null)
				numTruncateChars = truncateString.length;
			
			//Get truncate chars width
			if (this._truncateStringWidth == null)
			{
				if (truncateString == null)
					this._truncateStringWidth = 0;
				else
					this._truncateStringWidth = CanvasElement._measureText(truncateString, fontString);
			}
			
			var charWidth = 0;
			var numTextChars = text.length;
			totalWidth = this._textWidth + this._truncateStringWidth;
			
			//Remove text characters until we fit or run out.
			while (numTextChars > 0 && totalWidth > w)
			{
				charWidth = CanvasElement._measureText(text[numTextChars - 1], fontString);
				
				numTextChars--;
				totalWidth -= charWidth;
			}
			
			//Remove truncate characters until we fit or run out
			while (numTruncateChars > 0 && totalWidth > w)
			{
				charWidth = CanvasElement._measureText(truncateString[numTruncateChars - 1], fontString);
				
				numTruncateChars--;
				totalWidth -= charWidth;
			}
			
			text = text.substring(0, numTextChars) + truncateString.substring(0, numTruncateChars);
		}
		
		var linePaddingTop = this.getStyle("TextLinePaddingTop");
		var linePaddingBottom = this.getStyle("TextLinePaddingBottom");
		
		var textBaseline = this.getStyle("TextVerticalAlign");
		var textAlign = this.getStyle("TextHorizontalAlign");
		var textFillType = this.getStyle("TextFillType");
		var textColor = this.getStyle("TextColor");
		var textDecoration = this.getStyle("TextDecoration");
		
		//Get x position
		var textXPosition;
		if (textAlign == "left")
			textXPosition = x;
		else if (textAlign == "right")
			textXPosition = x + w - totalWidth;
		else //center
			textXPosition = Math.round(x + (w / 2) - (totalWidth / 2));
		
		//Get y position
		var textYPosition;
		if (textBaseline == "top")
			textYPosition = y + linePaddingTop;
		else if (textBaseline == "bottom")
			textYPosition = y + h - linePaddingBottom;
		else //middle
			textYPosition = Math.round(y + (h / 2) + (linePaddingTop / 2) - (linePaddingBottom / 2));
		
		//Render text
		if (textFillType == "stroke")
			CanvasElement._strokeText(ctx, text, textXPosition, textYPosition, fontString, textColor, textBaseline);
		else
			CanvasElement._fillText(ctx, text, textXPosition, textYPosition, fontString, textColor, textBaseline);
		
		if (textDecoration == "underline")
		{
			y = paddingMetrics.getY() + linePaddingTop + this.getStyle("TextSize") + linePaddingBottom - .5;
			
			ctx.beginPath();
			ctx.moveTo(x, y);
			ctx.lineTo(x + w, y);
			ctx.lineWidth = 1;
			ctx.strokeStyle = textColor;
			
			ctx.stroke();
		}
	};
	
	
	


/**
 * @depends TextInputElement.js
 */

/////////////////////////////////////////////////////////
///////////////IpInputElement////////////////////////////
	
/**
 * @class IpInputElement
 * @inherits TextInputElement
 * 
 * IpInputElement is an edit-able IPv4 address field.
 * 
 * @constructor IpInputElement 
 * Creates new IpInputElement instance.
 */
function IpInputElement()
{
	IpInputElement.base.prototype.constructor.call(this);
	
	//Steal the text field from base TextInput - re-use as ip1 field
	this._textFieldIp1 = this._textField;
	this._removeChild(this._textField);
	
		//Use list container to layout text fields
		this._listContainer = new ListContainerElement();
		this._listContainer.setStyle("LayoutDirection", "horizontal");
		
			this._textFieldIp1.setStyle("PercentHeight", 100);
			this._textFieldIp1.setStyle("PercentWidth", 100);
			this._textFieldIp1.setStyle("MaxChars", 3);
			this._textFieldIp1.setStyle("TabStop", -1);
			
			this._labelFieldDot1 = new LabelElement();
			this._labelFieldDot1.setStyle("PercentHeight", 100);
			this._labelFieldDot1.setStyle("TextHorizontalAlign", "center");
			this._labelFieldDot1.setStyle("Text", ".");
			this._labelFieldDot1.setStyle("PaddingLeft", 0);
			this._labelFieldDot1.setStyle("PaddingRight", 0);
			this._labelFieldDot1.setStyle("TextStyle", "bold");
			
			this._textFieldIp2 = new TextFieldElement();
			this._textFieldIp2.setStyle("PercentHeight", 100);
			this._textFieldIp2.setStyle("PercentWidth", 100);
			this._textFieldIp2.setStyle("MaxChars", 3);
			this._textFieldIp2.setStyle("TabStop", -1);
			
			this._labelFieldDot2 = new LabelElement();
			this._labelFieldDot2.setStyle("PercentHeight", 100);
			this._labelFieldDot2.setStyle("TextHorizontalAlign", "center");
			this._labelFieldDot2.setStyle("Text", ".");
			this._labelFieldDot2.setStyle("PaddingLeft", 0);
			this._labelFieldDot2.setStyle("PaddingRight", 0);
			this._labelFieldDot2.setStyle("TextStyle", "bold");
			
			this._textFieldIp3 = new TextFieldElement();
			this._textFieldIp3.setStyle("PercentHeight", 100);
			this._textFieldIp3.setStyle("PercentWidth", 100);
			this._textFieldIp3.setStyle("MaxChars", 3);
			this._textFieldIp3.setStyle("TabStop", -1);
			
			this._labelFieldDot3 = new LabelElement();
			this._labelFieldDot3.setStyle("PercentHeight", 100);
			this._labelFieldDot3.setStyle("TextHorizontalAlign", "center");
			this._labelFieldDot3.setStyle("Text", ".");
			this._labelFieldDot3.setStyle("PaddingLeft", 0);
			this._labelFieldDot3.setStyle("PaddingRight", 0);
			this._labelFieldDot3.setStyle("TextStyle", "bold");
			
			this._textFieldIp4 = new TextFieldElement();
			this._textFieldIp4.setStyle("PercentHeight", 100);
			this._textFieldIp4.setStyle("PercentWidth", 100);
			this._textFieldIp4.setStyle("MaxChars", 3);
			this._textFieldIp4.setStyle("TabStop", -1);
			
		this._listContainer.addElement(this._textFieldIp1);
		this._listContainer.addElement(this._labelFieldDot1);
		this._listContainer.addElement(this._textFieldIp2);
		this._listContainer.addElement(this._labelFieldDot2);
		this._listContainer.addElement(this._textFieldIp3);
		this._listContainer.addElement(this._labelFieldDot3);
		this._listContainer.addElement(this._textFieldIp4);
		
	this._addChild(this._listContainer);

	//////////////////////////
	
	var _self = this;
		
	this._onIpInputTextFieldMouseDownInstance = 
		function (mouseEvent)
		{
			_self._onIpInputTextFieldMouseDown(mouseEvent);
		};	
		
	this._onIpInputTextFieldFocusOutInstance = 
		function (event)
		{
			_self._onIpInputTextFieldFocusOut(event);
		};
		
	this._textFieldIp1.addEventListener("focusout", this._onIpInputTextFieldFocusOutInstance);
	this._textFieldIp2.addEventListener("focusout", this._onIpInputTextFieldFocusOutInstance);
	this._textFieldIp3.addEventListener("focusout", this._onIpInputTextFieldFocusOutInstance);
	this._textFieldIp4.addEventListener("focusout", this._onIpInputTextFieldFocusOutInstance);
	
	this._textFieldIp1.addEventListener("mousedown", this._onIpInputTextFieldMouseDownInstance);
	this._textFieldIp2.addEventListener("mousedown", this._onIpInputTextFieldMouseDownInstance);
	this._textFieldIp3.addEventListener("mousedown", this._onIpInputTextFieldMouseDownInstance);
	this._textFieldIp4.addEventListener("mousedown", this._onIpInputTextFieldMouseDownInstance);
	
	////////////////////
	
	//Currently focused IP field
	this._textFieldFocused = null;
}

//Inherit from SkinnableElement
IpInputElement.prototype = Object.create(TextInputElement.prototype);
IpInputElement.prototype.constructor = IpInputElement;
IpInputElement.base = TextInputElement;

/////////////Events////////////////////////////////////


/////////////Style Types///////////////////////////////


/////////////Default Styles///////////////////////////

IpInputElement.StyleDefault = new StyleDefinition();

IpInputElement.StyleDefault.setStyle("TextHorizontalAlign", 						"center");
IpInputElement.StyleDefault.setStyle("TextVerticalAlign", 							"middle");

IpInputElement.StyleDefault.setStyle("PaddingLeft",									5);
IpInputElement.StyleDefault.setStyle("PaddingRight",								5);


////////Public///////////////////////

/**
 * @function setText
 * @override
 * Sets the IP to be displayed.
 * 
 * @param text String
 * IP to be displayed. Formatted as IPv4 address: "192.168.1.1"
 */
IpInputElement.prototype.setText = 
	function (text)
	{
		if (text == null || text == "")
		{
			this._textFieldIp1.setText("");
			this._textFieldIp2.setText("");
			this._textFieldIp3.setText("");
			this._textFieldIp4.setText("");
			
			return;
		}
		
		var i;
		var i2;
		var n;
		var ipArray = text.split(".");
		
		for (i = 0; i < ipArray.length; i++)
		{
			if (i == 4)
				return;
			
			n = Number(ipArray[i]);
			if (isNaN(n) == true)
				n = 0;
			
			if (n > 255)
				n = 255;
			
			if (i == 0)
				this._textFieldIp1.setText(n.toString());
			else if (i == 1)
				this._textFieldIp2.setText(n.toString());
			else if (i == 2)
				this._textFieldIp3.setText(n.toString());
			else //if (i == 3)
				this._textFieldIp4.setText(n.toString());
		}
		
	};

/**
 * @function getText
 * @override 
 * Gets the IP currently displayed. 
 * 
 * @returns String
 * IP currently displayed formatted as IPv4 address: "192.168.1.1".
 * When all fields are empty an empty string "" will be returned.
 * When some but not all fields are empty null will be returned (invalid IP).
 */	
IpInputElement.prototype.getText = 
	function ()
	{
		if (this._textFieldIp1.getText().length == 0 &&
			this._textFieldIp2.getText().length == 0 &&
			this._textFieldIp3.getText().length == 0 &&
			this._textFieldIp4.getText().length == 0)
		{
			return "";
		}
		
		if (this._textFieldIp1.getText().length == 0 ||
			this._textFieldIp2.getText().length == 0 ||
			this._textFieldIp3.getText().length == 0 ||
			this._textFieldIp4.getText().length == 0)
		{
			return null;
		}
	
		return this._textFieldIp1.getText() + "." +
				this._textFieldIp2.getText() + "." +
				this._textFieldIp3.getText() + "." +
				this._textFieldIp4.getText();
	};


////////Internal/////////////////////

//@override
IpInputElement.prototype._onTextInputKeyDown = 
	function (keyboardEvent)
	{
		if (keyboardEvent.getDefaultPrevented() == true)
			return;
		
		var key = keyboardEvent.getKey();
		
		if (key.length == 1 && 
			key != "0" && key != "1" &&
			key != "2" && key != "3" &&
			key != "4" && key != "5" &&
			key != "6" && key != "7" &&
			key != "8" && key != "9" &&
			key != ".")
		{
			return;
		}
		
		if (key == "Tab" || key == ".") //Move focus
		{
			var shiftPressed = false;
			
			if (key == "Tab")
				shiftPressed = keyboardEvent.getShift();
			
			if (shiftPressed == false)
			{
				if (this._textFieldFocused == this._textFieldIp4)
					return;
				else	
				{
					//Prevent normal tab stop handling
					keyboardEvent.preventDefault();
					
					this._textFieldFocused.dispatchEvent(new ElementEvent("focusout", false));
					
					if (this._textFieldFocused == this._textFieldIp1)
						this._textFieldFocused = this._textFieldIp2;
					else if (this._textFieldFocused == this._textFieldIp2)
						this._textFieldFocused = this._textFieldIp3;
					else //if (this._textFieldFocused == this._textFieldIp3)
						this._textFieldFocused = this._textFieldIp4;
					
					this._textFieldFocused.dispatchEvent(new ElementEvent("focusin", false));
				}
			}
			else //if (shiftPressed == true)
			{
				if (this._textFieldFocused == this._textFieldIp1)
					return;
				else
				{
					//Prevent normal tab stop handling
					keyboardEvent.preventDefault();
					
					this._textFieldFocused.dispatchEvent(new ElementEvent("focusout", false));
					
					if (this._textFieldFocused == this._textFieldIp4)
						this._textFieldFocused = this._textFieldIp3;
					else if (this._textFieldFocused == this._textFieldIp3)
						this._textFieldFocused = this._textFieldIp2;
					else //if (this._textFieldFocused == this._textFieldIp2)
						this._textFieldFocused = this._textFieldIp1;
					
					this._textFieldFocused.dispatchEvent(new ElementEvent("focusin", false));
				}
			}
		}
		else
		{
			var clonedEvent = keyboardEvent.clone();
			clonedEvent._bubbles = false; //Dont bubble.
			
			//Dispatch non-bubbling keyboard event to our text field.
			this._textFieldFocused.dispatchEvent(clonedEvent);
			
			if (clonedEvent.getIsCanceled() == true)
				keyboardEvent.cancelEvent();
				
			if (clonedEvent.getDefaultPrevented() == true)
				keyboardEvent.preventDefault();
		}
	};

//@override
IpInputElement.prototype._onIpInputKeyUp = 
	function (keyboardEvent)
	{
		if (keyboardEvent.getDefaultPrevented() == true)
			return;
		
		var key = keyboardEvent.getKey();
		
		if (key.length == 1 && 
			key != "0" && key != "1" &&
			key != "2" && key != "3" &&
			key != "4" && key != "5" &&
			key != "6" && key != "7" &&
			key != "8" && key != "9")
		{
			return;
		}
		
		var clonedEvent = keyboardEvent.clone();
		clonedEvent._bubbles = false; //Dont bubble.
		
		//Dispatch non-bubbling keyboard event to our text field.
		this._textFieldFocused.dispatchEvent(clonedEvent);
		
		if (clonedEvent.getIsCanceled() == true)
			keyboardEvent.cancelEvent();
			
		if (clonedEvent.getDefaultPrevented() == true)
			keyboardEvent.preventDefault();
	};	
	
//@override	
IpInputElement.prototype._onTextInputFocusIn = 
	function (elementEvent)
	{
		//Mouse down already focused
		if (this._textFieldFocused != null)
			return;
	
		//This only works because TextField doesnt look at _isFocused (manages caret state with different flag)
		this._textFieldIp1.dispatchEvent(elementEvent.clone()); 
		this._textFieldFocused = this._textFieldIp1;
	};

//@override	
IpInputElement.prototype._onTextInputFocusOut = 
	function (elementEvent)
	{
		//This only works because TextField doesnt look at _isFocused (manages caret state with different flag)
		this._textFieldFocused.dispatchEvent(elementEvent.clone());
		this._textFieldFocused = null; 
	};

/**
 * @function _onIpInputTextFieldMouseDown
 * Event handler for the internal TextField's "mousedown" event.
 * 
 * @param mouseEvent ElementMouseEvent
 * ElementMouseEvent to process.
 */	
IpInputElement.prototype._onIpInputTextFieldMouseDown = 
	function (mouseEvent)
	{
		if (mouseEvent.getTarget() != this._textFieldFocused && this._textFieldFocused != null)
			this._textFieldFocused.dispatchEvent(new ElementEvent("focusout", false));
		
		this._textFieldFocused = mouseEvent.getTarget();
	};	
	
/**
 * @function _onIpInputTextFieldMouseDown
 * Event handler for the internal TextField's "focusout" event.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to process.
 */		
IpInputElement.prototype._onIpInputTextFieldFocusOut = 
	function (event)
	{
		var textField = event.getTarget();
		var text = textField.getText();
		
		if (text.length == 0)
			return;
		
		var changed = false;
		
		//Trim leading zeros
		while (text.length > 1 && text.charAt(0) == "0")
		{
			text = text.slice(1);
			changed = true;
		}
		
		//Fix invalid number
		if (Number(textField.getText()) > 255)
		{
			text = "255";
			changed = true;
		}
			
		if (changed == true)
		{
			textField.setText(text);
		
			if (this.hasEventListener("changed", null) == true)
				this.dispatchEvent(new ElementEvent("changed", false));
		}
	};
	
//@override
IpInputElement.prototype._updateTextColors = 
	function ()
	{
		this._textFieldIp1.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._textFieldIp1.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._textFieldIp1.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._labelFieldDot1.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._labelFieldDot1.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._labelFieldDot1.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._textFieldIp2.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._textFieldIp2.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._textFieldIp2.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._labelFieldDot2.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._labelFieldDot2.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._labelFieldDot2.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._textFieldIp3.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._textFieldIp3.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._textFieldIp3.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._labelFieldDot3.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._labelFieldDot3.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._labelFieldDot3.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
		
		this._textFieldIp4.setStyle("TextColor", this._getTextColor(this._currentSkinState));
		this._textFieldIp4.setStyle("TextHighlightedColor", this._getTextHighlightedColor(this._currentSkinState));
		this._textFieldIp4.setStyle("TextHighlightedBackgroundColor", this._getTextHighlightedBackgroundColor(this._currentSkinState));
	};
	
//@override
IpInputElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		//IpInputElement.base.base - skip TextInput._doStylesUpdated()
		IpInputElement.base.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		//Force the textField to use our defaults rather than inherited.
		if ("TextHorizontalAlign" in stylesMap)
		{
			this._textFieldIp1.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
			this._textFieldIp2.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
			this._textFieldIp3.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
			this._textFieldIp4.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
		}
		
		if ("TextVerticalAlign" in stylesMap)
		{
			this._textFieldIp1.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			this._textFieldIp2.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			this._textFieldIp3.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			this._textFieldIp4.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			
			this._labelFieldDot1.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			this._labelFieldDot2.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
			this._labelFieldDot3.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
		}
		
		if ("Enabled" in stylesMap)
		{
			var enabled = this.getStyle("Enabled");
			
			this._textFieldIp1.setStyle("Enabled", enabled);
			this._textFieldIp2.setStyle("Enabled", enabled);
			this._textFieldIp3.setStyle("Enabled", enabled);
			this._textFieldIp4.setStyle("Enabled", enabled);
			
			if (enabled == true)
			{
				if (this.hasEventListener("keydown", this._onTextInputKeyUpDownInstance) == false)
					this.addEventListener("keydown", this._onTextInputKeyUpDownInstance);
				
				if (this.hasEventListener("keyup", this._onTextInputKeyUpDownInstance) == false)
					this.addEventListener("keyup", this._onTextInputKeyUpDownInstance);
				
				if (this._textFieldIp1.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == false)
					this._textFieldIp1.addEventListener("changed", this._onTextInputTextFieldChangedInstance);		
				
				if (this._textFieldIp2.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == false)
					this._textFieldIp2.addEventListener("changed", this._onTextInputTextFieldChangedInstance);	
				
				if (this._textFieldIp3.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == false)
					this._textFieldIp3.addEventListener("changed", this._onTextInputTextFieldChangedInstance);	
				
				if (this._textFieldIp4.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == false)
					this._textFieldIp4.addEventListener("changed", this._onTextInputTextFieldChangedInstance);	
			}
			else
			{
				if (this.hasEventListener("keydown", this._onTextInputKeyUpDownInstance) == true)
					this.removeEventListener("keydown", this._onTextInputKeyUpDownInstance);
				
				if (this.hasEventListener("keyup", this._onTextInputKeyUpDownInstance) == true)
					this.removeEventListener("keyup", this._onTextInputKeyUpDownInstance);
				
				if (this._textFieldIp1.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == true)
					this._textFieldIp1.removeEventListener("changed", this._onTextInputTextFieldChangedInstance);
				
				if (this._textFieldIp2.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == true)
					this._textFieldIp2.removeEventListener("changed", this._onTextInputTextFieldChangedInstance);
				
				if (this._textFieldIp3.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == true)
					this._textFieldIp3.removeEventListener("changed", this._onTextInputTextFieldChangedInstance);
				
				if (this._textFieldIp4.hasEventListener("changed", this._onTextInputTextFieldChangedInstance) == true)
					this._textFieldIp4.removeEventListener("changed", this._onTextInputTextFieldChangedInstance);
			}
		}
		
		if ("TextLinePaddingTop" in stylesMap || 
			"TextLinePaddingBottom" in stylesMap)
		{
			this._invalidateMeasure();
		}
		
		if ("Padding" in stylesMap ||
			"PaddingTop" in stylesMap ||
			"PaddingBottom" in stylesMap ||
			"PaddingLeft" in stylesMap ||
			"PaddingRight" in stylesMap)
		{
			var paddingSize = this._getPaddingSize();
			
			this._textFieldIp1.setStyle("PaddingLeft", paddingSize.paddingLeft);
			this._textFieldIp1.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textFieldIp1.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._textFieldIp2.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textFieldIp2.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._textFieldIp3.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textFieldIp3.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._textFieldIp4.setStyle("PaddingRight", paddingSize.paddingRight);
			this._textFieldIp4.setStyle("PaddingTop", paddingSize.paddingTop);
			this._textFieldIp4.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._labelFieldDot1.setStyle("PaddingTop", paddingSize.paddingTop);
			this._labelFieldDot1.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._labelFieldDot2.setStyle("PaddingTop", paddingSize.paddingTop);
			this._labelFieldDot2.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._labelFieldDot3.setStyle("PaddingTop", paddingSize.paddingTop);
			this._labelFieldDot3.setStyle("PaddingBottom", paddingSize.paddingBottom);
			
			this._invalidateMeasure();
		}
		
		this._updateSkinStyles(stylesMap);
	};
	
//@override
IpInputElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var fontString = this._getFontString();
	
		var measuredHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
		
		var measuredWidth = (CanvasElement._measureText("000", fontString) + 6) * 4;
		measuredWidth += this._labelFieldDot1._getStyledOrMeasuredWidth() * 3;
		
		measuredWidth += padWidth;
		measuredHeight += padHeight;
	
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};
	
//@override	
IpInputElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		//IpInputElement.base.base - skip TextInputElement._doLayout()
		IpInputElement.base.base.prototype._doLayout.call(this, paddingMetrics);
		
		//Ignore padding, proxied to TextFields for proper mouse handling.		
		this._listContainer._setActualPosition(0, 0);
		this._listContainer._setActualSize(this._width, this._height);
	};
	
	


/**
 * @depends CanvasElement.js
 */

////////////////////////////////////////////////////
//////////////////ImageElement//////////////////////

/**
 * @class ImageElement
 * @inherits CanvasElement
 * 
 * ImageElement renders an image (imagine that). 
 * Images can be loaded via "url" or pre-loaded via DOM image reference and can be stretched, tiled, or aligned. 
 * 
 * @constructor ImageElement 
 * Creates new ImageElement instance.
 */
function ImageElement()
{
	ImageElement.base.prototype.constructor.call(this);
	
	this._imageLoader = null;
	
	this._imageSource = null;
	
	/**
	 * @member _imageLoadComplete boolean
	 * Read Only - True if the image has completed loading, otherwise false.
	 */
	this._imageLoadComplete = false;
	
	var _self = this;
	
	//Private event handler, need different instance for each element.
	this._onImageElementLoadCompleteInstance = 
		function (event)
		{
			_self._imageLoader.removeEventListener("load", _self._onImageElementLoadCompleteInstance);
			_self._imageLoadComplete = true;
			
			_self._invalidateMeasure();
			_self._invalidateLayout();
			_self._invalidateRender();
		};
}

//Inherit from CanvasElement
ImageElement.prototype = Object.create(CanvasElement.prototype);
ImageElement.prototype.constructor = ImageElement;
ImageElement.base = CanvasElement;


/////////////Style Types///////////////////////////////

ImageElement._StyleTypes = Object.create(null);

/**
 * @style ImageSource String
 * Image to render. This may be a String URL, or a reference to a DOM image or canvas.
 */
ImageElement._StyleTypes.ImageSource = 					StyleableBase.EStyleType.NORMAL;		// null || image || "url" 

/**
 * @style ImageSourceClipX Number
 * X position in pixels of the clip area for the source image.
 */
ImageElement._StyleTypes.ImageSourceClipX = 			StyleableBase.EStyleType.NORMAL;		// null || number

/**
 * @style ImageSourceClipY Number
 * Y position in pixels of the clip area for the source image.
 */
ImageElement._StyleTypes.ImageSourceClipY = 			StyleableBase.EStyleType.NORMAL;		// null || number

/**
 * @style ImageSourceClipWidth Number
 * Width in pixels of the clip area for the source image.
 */
ImageElement._StyleTypes.ImageSourceClipWidth = 		StyleableBase.EStyleType.NORMAL;		// null || number

/**
 * @style ImageSourceClipHeight Number
 * Height in pixels of the clip area for the source image.
 */
ImageElement._StyleTypes.ImageSourceClipHeight = 		StyleableBase.EStyleType.NORMAL;		// null || number

/**
 * @style ImageScaleType String
 * Determines how the image should be stretched or scaled. Allowable values are "none", "fit", "stretch", "tile", or "tilefit".
 * "none" will not modify the original image, 
 * "fit" stretches the image but maintains the aspect ratio to fit the available area's minimum width/height constraint,
 * "stretch" stretches the image to fit the entire available area disregarding aspect ratio,
 * "tile" will not modify the original image but repeat it in both directions to fill the available area.
 * "tilefit" stretches the image but maintains the aspect ratio, any remaining space in the areas maximum width/height constraint is tiled with the stretched image.
 */
ImageElement._StyleTypes.ImageScaleType = 				StyleableBase.EStyleType.NORMAL;		// "none" || "fit" || "stretch" || "tile" || "tilefit"

/**
 * @style ImageVerticalAlign String
 * Aligns the image vertically when using ImageScaleType "none" or "fit" and not all of the available space is consumed.
 * Allowable values are "top", "middle", "bottom".
 */
ImageElement._StyleTypes.ImageVerticalAlign = 			StyleableBase.EStyleType.NORMAL;		// "top" || "middle" || "bottom"

/**
 * @style ImageHorizontalAlign String
 * Aligns the image horizontally when using ImageScaleType "none" or "fit" and not all of the available space is consumed.
 * Allowable values are "left", "center", "right".
 */
ImageElement._StyleTypes.ImageHorizontalAlign = 		StyleableBase.EStyleType.NORMAL;		// "left" || "center" || "right"


////////////Default Styles////////////////////////////

ImageElement.StyleDefault = new StyleDefinition();

ImageElement.StyleDefault.setStyle("ImageSource", 						null);			// null || image || "url"
ImageElement.StyleDefault.setStyle("ImageSourceClipX", 					null);			// null || number
ImageElement.StyleDefault.setStyle("ImageSourceClipY", 					null);			// null || number
ImageElement.StyleDefault.setStyle("ImageSourceClipWidth", 				null);			// null || number
ImageElement.StyleDefault.setStyle("ImageSourceClipHeight", 			null);			// null || number

ImageElement.StyleDefault.setStyle("ImageScaleType", 					"stretch");		// "none" || "fit" || "stretch" || "tile" || "tilefit"
ImageElement.StyleDefault.setStyle("ImageHorizontalAlign", 				"left");		// "left" || "center" || "right"
ImageElement.StyleDefault.setStyle("ImageVerticalAlign", 				"top");			// "top" || "middle" || "bottom"


/////////////ImageElement Protected Functions///////////////////

//@Override
ImageElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ImageElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		if ("ImageSource" in stylesMap)
		{
			var newSource = this.getStyle("ImageSource");
			if (this._imageSource != newSource)
			{
				//Clean up old loader
				if (this._imageLoader != null && this._imageLoadComplete == false)
					this._imageLoader.removeEventListener("load", this._onImageElementLoadCompleteInstance);
				
				this._imageLoader = null;
				this._imageLoadComplete = false;
				
				//May be img, string, TODO: Canvas / Video
				this._imageSource = newSource;
				
				if (this._imageSource instanceof HTMLImageElement)
				{
					this._imageLoader = this._imageSource;
					this._imageLoadComplete = this._imageSource.complete;
					
					if (this._imageLoadComplete == false)
						this._imageLoader.addEventListener("load", this._onImageElementLoadCompleteInstance, false);
				}
				else
				{
					this._imageLoader = new Image();
					this._imageLoader.addEventListener("load", this._onImageElementLoadCompleteInstance, false);
					this._imageLoader.src = this._imageSource;
				}
				
				this._invalidateMeasure();
				this._invalidateRender();
			}
		}
		else
		{
			if ("ImageSourceClipX" in stylesMap ||
				"ImageSourceClipY" in stylesMap ||
				"ImageSourceClipWidth" in stylesMap ||
				"ImageSourceClipHeight" in stylesMap || 
				"ImageScaleType" in stylesMap)
			{
				this._invalidateMeasure();
				this._invalidateRender();
			}
			else if ("ImageVerticalAlign" in stylesMap ||
					"ImageHorizontalAlign" in stylesMap)
			{
				this._invalidateRender();
			}
		}
	};
	
//@Override
ImageElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var measuredWidth = padWidth;
		var measuredHeight = padHeight;
		
		var clipX = this.getStyle("ImageSourceClipX");
		var clipY = this.getStyle("ImageSourceClipY");
		
		var clipWidth = this.getStyle("ImageSourceClipWidth");
		var clipHeight = this.getStyle("ImageSourceClipHeight");
		
		if (clipX == null)
			clipX = 0;
		
		if (clipY == null)
			clipY = 0;
		
		if (clipWidth != null)
			measuredWidth += clipWidth;
		else if (this._imageLoadComplete == true)
			measuredWidth += (this._imageLoader.naturalWidth - clipX);

		if (clipHeight != null)
			measuredHeight += clipHeight;
		else if (this._imageLoadComplete == true)
			measuredHeight += (this._imageLoader.naturalHeight - clipY);
		
		if (this.getStyle("ImageScaleType") == "fit")
			this._invalidateLayout();
		
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};	
	
//@override	
ImageElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		ImageElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		//When type "fit", we use layout to adjust measured size. This allows the implementor
		//to set only width OR height, and the image's measured size will scale appropriately.
		if (this.getStyle("ImageScaleType") != "fit" || this._imageLoadComplete == false)
			return;
	
		//Only adjust measured sizes if width OR height doesnt match measured, bail if both match, or miss match.
		if ((this._width == this._measuredWidth && this._height == this._measuredHeight) ||
			(this._width != this._measuredWidth && this._height != this._measuredHeight))
			return;
		
		var imgW = paddingMetrics.getWidth();
		var imgH = paddingMetrics.getHeight();
		
		//Zero size, bail
		if (imgW <= 0 || imgH <= 0)
			return;

		var clipX = this.getStyle("ImageSourceClipX");
		var clipY = this.getStyle("ImageSourceClipY");
		var clipW = this.getStyle("ImageSourceClipWidth");
		var clipH = this.getStyle("ImageSourceClipHeight");
		
		if (clipX == null)
			clipX = 0;
		if (clipY == null)
			clipY = 0;
		
		if (clipW == null)
			clipW = this._imageLoader.naturalWidth - clipX;
		if (clipH == null)
			clipH = this._imageLoader.naturalHeight - clipY;
		
		if (clipW <= 0 || clipH <= 0)
			return;
		
		var paddingSize = this._getPaddingSize();
		var padW = paddingSize.width;
		var padH = paddingSize.height;
		
		var imageRatio = clipW / clipH;
		
		var measuredWidth;
		var measuredHeight;
		
		//Size to our height
		if (this._height != this._measuredHeight) //Height must be explicitly set
		{
			measuredWidth = (imgH * imageRatio) + padW;
			measuredHeight = this._height;
		}
		else //Size to our width
		{
			measuredWidth = this._width;
			measuredHeight = (w / imageRatio) + padH;
		}
		
		measuredWidth = Math.round(measuredWidth);
		measuredHeight = Math.round(measuredHeight);
		
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};
	
//@Override
ImageElement.prototype._doRender =
	function()
	{
		ImageElement.base.prototype._doRender.call(this);
		
		if (this._imageLoadComplete == false)
			return;
		
		var paddingMetrics = this._getPaddingMetrics();
		var ctx = this._getGraphicsCtx();
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		if (w <= 0 || h <= 0)
			return;
		
		var clipX = this.getStyle("ImageSourceClipX");
		var clipY = this.getStyle("ImageSourceClipY");
		var clipW = this.getStyle("ImageSourceClipWidth");
		var clipH = this.getStyle("ImageSourceClipHeight");
		
		var scaleType = this.getStyle("ImageScaleType");
		
		if (clipX == null)
			clipX = 0;
		if (clipY == null)
			clipY = 0;
		
		if (clipW == null)
			clipW = this._imageLoader.naturalWidth - clipX;
		if (clipH == null)
			clipH = this._imageLoader.naturalHeight - clipY;
		
		if (clipW <= 0 || clipH <= 0)
			return;
		
		if (scaleType == "stretch")
		{
			ctx.drawImage(
				this._imageLoader, 
				clipX, clipY, clipW, clipH, 
				x, y, w, h);
		}
		else if (scaleType == "tile")
		{
			var currentX = x;
			var currentY = y;
			
			var drawWidth = clipW;
			var drawHeight = clipH;
			
			while (true)
			{
				drawWidth = Math.min(clipW, x + w - currentX);
				drawHeight = Math.min(clipH, y + h - currentY);
				
				ctx.drawImage(
					this._imageLoader, 
					clipX, clipY, drawWidth, drawHeight, 
					currentX, currentY, drawWidth, drawHeight);
				
				currentX += drawWidth;
				if (currentX >= x + w)
				{
					currentX = x;
					currentY += drawHeight;
					
					if (currentY >= y + h)
						break;
				}
			}
		}
		else if (scaleType == "tilefit")
		{
			var thisRatio = w / h;
			var imageRatio = clipW / clipH;
			
			var drawWidth = clipW;
			var drawHeight = clipH;
			
			//Size to our height
			if (thisRatio > imageRatio)
			{
				var currentX = x;
				
				drawHeight = h;
				drawWidth = h * imageRatio;
				
				while (true)
				{
					if (currentX + drawWidth > x + w)
					{
						var availableWidth = x + w - currentX;
						var widthRatio = availableWidth / drawWidth;
						
						ctx.drawImage(
							this._imageLoader, 
							clipX, clipY, clipW * widthRatio, clipH, 
							currentX, y, drawWidth * widthRatio, drawHeight);
						
						break;
					}
					else
					{
						ctx.drawImage(
							this._imageLoader, 
							clipX, clipY, clipW, clipH, 
							currentX, y, drawWidth, drawHeight);
						
						currentX += drawWidth;
						if (currentX == x + w)
								break;
					}
				}
			}
			else //Size to our width
			{
				var currentY = y;
				
				drawWidth = w;
				drawHeight = w / imageRatio;
				
				while (true)
				{
					if (currentY + drawHeight > y + h)
					{
						var availableHeight = y + h - currentY;
						var heightRatio = availableHeight / drawHeight;
						
						ctx.drawImage(
							this._imageLoader, 
							clipX, clipY, clipW, clipH * heightRatio, 
							x, currentY, drawWidth, drawHeight * heightRatio);
						
						break;
					}
					else
					{
						ctx.drawImage(
							this._imageLoader, 
							clipX, clipY, clipW, clipH, 
							x, currentY, drawWidth, drawHeight);
						
						currentY += drawHeight;
						if (currentY == y + h)
							break;
					}
				}
			}
		}
		else if (scaleType == "fit" || scaleType == "none" || scaleType == null)
		{
			var verticalAlign = this.getStyle("ImageVerticalAlign");
			var horizontalAlign = this.getStyle("ImageHorizontalAlign");
			
			var drawWidth = clipW;
			var drawHeight = clipH;
			
			if (scaleType == "fit")
			{
				var thisRatio = w / h;
				var imageRatio = clipW / clipH;
				
				//Size to our height
				if (thisRatio > imageRatio)
				{
					drawHeight = h;
					drawWidth = Math.round(h * imageRatio);
				}
				else //Size to our width
				{
					drawWidth = w;
					drawHeight = Math.round(w / imageRatio);
				}
			}
			else //scaleType == "none"
			{
				//Reduce image clipping area to render size.
				if (w < clipW)
				{
					if (horizontalAlign == "right")
						clipX += (clipW - w);
					else if (horizontalAlign == "center")
						clipX += ((clipW - w) / 2);
					
					clipW = w;
					drawWidth = w;
				}
				if (h < clipH)
				{
					if (verticalAlign == "bottom")
						clipY += (clipH - h);
					else if (verticalAlign == "middle")
						clipY += ((clipH - h) / 2);
					
					clipH = h;
					drawHeight = h;
				}
			}
			
			var drawX = x;
			var drawY = y;
			
			if (horizontalAlign == "right")
				drawX += w - drawWidth;
			else if (horizontalAlign == "center")
				drawX += ((w - drawWidth) / 2);
			
			if (verticalAlign == "bottom")
				drawY += h - drawHeight;
			else if (verticalAlign == "middle")
				drawY += ((h - drawHeight) / 2);
			
			ctx.drawImage(
				this._imageLoader, 
				clipX, clipY, clipW, clipH, 
				drawX, drawY, drawWidth, drawHeight);
		}
		
	};
	
	
	


/**
 * @depends CanvasElement.js
 * @depends ArrowShape.js
 */

//////////////////////////////////////////////////////////////
///////////////DatePickerElement/////////////////////////////

/**
 * @class DatePickerElement
 * @inherits CanvasElement
 * 
 * DatePickerElement is a class that displays a calendar used to select dates.  
 * 
 * @constructor DatePickerElement 
 * Creates new DatePickerElement instance.
 */

function DatePickerElement() //extends CanvasElement
{
	DatePickerElement.base.prototype.constructor.call(this);
	
	//Setup static style for days grid (all instances use same style for all rows/columns)
	if (DatePickerElement._GridDaysRowColumnStyle == null)
	{
		DatePickerElement._GridDaysRowColumnStyle = new GridContainerRowColumnDefinition();
		DatePickerElement._GridDaysRowColumnStyle.setStyle("PercentSize", 100);
	}
	
	/////////////
	
	//Use list container to control layout
	this._rootListContainer = new ListContainerElement();
	this._rootListContainer.setStyle("LayoutDirection", "vertical");
	this._rootListContainer.setStyle("LayoutHorizontalAlign", "center");
	
		this._listContainerYearMonthSelection = new ListContainerElement();
		this._listContainerYearMonthSelection.setStyle("PercentWidth", 100);
		this._listContainerYearMonthSelection.setStyle("LayoutDirection", "horizontal");
		this._listContainerYearMonthSelection.setStyle("LayoutHorizontalAlign", "center");
		this._listContainerYearMonthSelection.setStyle("LayoutVerticalAlign", "middle");
		
			this._buttonMonthDecrement = new ButtonElement();
			
			this._anchorContainerLabelMonths = new AnchorContainerElement();
			
				this._labelMonth1 = new LabelElement();
				this._labelMonth1.setStyle("Visible", false);
				this._labelMonth1.setStyle("PercentWidth", 100);
				
				this._labelMonth2 = new LabelElement();
				this._labelMonth2.setStyle("Visible", false);
				this._labelMonth2.setStyle("PercentWidth", 100);
				
				this._labelMonth3 = new LabelElement();
				this._labelMonth3.setStyle("Visible", false);
				this._labelMonth3.setStyle("PercentWidth", 100);
				
				this._labelMonth4 = new LabelElement();
				this._labelMonth4.setStyle("Visible", false);
				this._labelMonth4.setStyle("PercentWidth", 100);
				
				this._labelMonth5 = new LabelElement();
				this._labelMonth5.setStyle("Visible", false);
				this._labelMonth5.setStyle("PercentWidth", 100);
				
				this._labelMonth6 = new LabelElement();
				this._labelMonth6.setStyle("Visible", false);
				this._labelMonth6.setStyle("PercentWidth", 100);
				
				this._labelMonth7 = new LabelElement();
				this._labelMonth7.setStyle("Visible", false);
				this._labelMonth7.setStyle("PercentWidth", 100);
				
				this._labelMonth8 = new LabelElement();
				this._labelMonth8.setStyle("Visible", false);
				this._labelMonth8.setStyle("PercentWidth", 100);
				
				this._labelMonth9 = new LabelElement();
				this._labelMonth9.setStyle("Visible", false);
				this._labelMonth9.setStyle("PercentWidth", 100);
				
				this._labelMonth10 = new LabelElement();
				this._labelMonth10.setStyle("Visible", false);
				this._labelMonth10.setStyle("PercentWidth", 100);
				
				this._labelMonth11 = new LabelElement();
				this._labelMonth11.setStyle("Visible", false);
				this._labelMonth11.setStyle("PercentWidth", 100);
				
				this._labelMonth12 = new LabelElement();
				this._labelMonth12.setStyle("Visible", false);
				this._labelMonth12.setStyle("PercentWidth", 100);
				
			this._anchorContainerLabelMonths.addElement(this._labelMonth1);
			this._anchorContainerLabelMonths.addElement(this._labelMonth2);
			this._anchorContainerLabelMonths.addElement(this._labelMonth3);
			this._anchorContainerLabelMonths.addElement(this._labelMonth4);
			this._anchorContainerLabelMonths.addElement(this._labelMonth5);
			this._anchorContainerLabelMonths.addElement(this._labelMonth6);
			this._anchorContainerLabelMonths.addElement(this._labelMonth7);
			this._anchorContainerLabelMonths.addElement(this._labelMonth8);
			this._anchorContainerLabelMonths.addElement(this._labelMonth9);
			this._anchorContainerLabelMonths.addElement(this._labelMonth10);
			this._anchorContainerLabelMonths.addElement(this._labelMonth11);
			this._anchorContainerLabelMonths.addElement(this._labelMonth12);
				
			this._buttonMonthIncrement = new ButtonElement();
			
			this._spacerYearMonth = new CanvasElement();
			this._spacerYearMonth.setStyle("PercentWidth", 100);
			this._spacerYearMonth.setStyle("MinWidth", 20);
			
			this._buttonYearDecrement = new ButtonElement();
			this._labelYear = new LabelElement();
			this._buttonYearIncrement = new ButtonElement();

		this._listContainerYearMonthSelection.addElement(this._buttonMonthDecrement);
		this._listContainerYearMonthSelection.addElement(this._anchorContainerLabelMonths);
		this._listContainerYearMonthSelection.addElement(this._buttonMonthIncrement);	
		this._listContainerYearMonthSelection.addElement(this._spacerYearMonth);	
		this._listContainerYearMonthSelection.addElement(this._buttonYearDecrement);
		this._listContainerYearMonthSelection.addElement(this._labelYear);
		this._listContainerYearMonthSelection.addElement(this._buttonYearIncrement);
		
		this._gridDaysContainer = new GridContainerElement();
		this._gridDaysContainer.setStyle("PercentWidth", 100);
		this._gridDaysContainer.setStyle("PercentHeight", 100);
		
			//All rows and columns set to 100% except days label row
			this._gridDaysContainer.setRowDefinition(DatePickerElement._GridDaysRowColumnStyle, 1);
			this._gridDaysContainer.setRowDefinition(DatePickerElement._GridDaysRowColumnStyle, 2);
			this._gridDaysContainer.setRowDefinition(DatePickerElement._GridDaysRowColumnStyle, 3);
			this._gridDaysContainer.setRowDefinition(DatePickerElement._GridDaysRowColumnStyle, 4);
			this._gridDaysContainer.setRowDefinition(DatePickerElement._GridDaysRowColumnStyle, 5);
			this._gridDaysContainer.setRowDefinition(DatePickerElement._GridDaysRowColumnStyle, 6);
			
			this._gridDaysContainer.setColumnDefinition(DatePickerElement._GridDaysRowColumnStyle, 0);
			this._gridDaysContainer.setColumnDefinition(DatePickerElement._GridDaysRowColumnStyle, 1);
			this._gridDaysContainer.setColumnDefinition(DatePickerElement._GridDaysRowColumnStyle, 2);
			this._gridDaysContainer.setColumnDefinition(DatePickerElement._GridDaysRowColumnStyle, 3);
			this._gridDaysContainer.setColumnDefinition(DatePickerElement._GridDaysRowColumnStyle, 4);
			this._gridDaysContainer.setColumnDefinition(DatePickerElement._GridDaysRowColumnStyle, 5);
			this._gridDaysContainer.setColumnDefinition(DatePickerElement._GridDaysRowColumnStyle, 6);
			
			this._labelDay1 = new LabelElement();
			this._labelDay2 = new LabelElement();
			this._labelDay3 = new LabelElement();
			this._labelDay4 = new LabelElement();
			this._labelDay5 = new LabelElement();
			this._labelDay6 = new LabelElement();
			this._labelDay7 = new LabelElement();
			
			//Toggle buttons added dynamically
			
		this._gridDaysContainer.setCellElement(this._labelDay1, 0, 0);
		this._gridDaysContainer.setCellElement(this._labelDay2, 0, 1);
		this._gridDaysContainer.setCellElement(this._labelDay3, 0, 2);
		this._gridDaysContainer.setCellElement(this._labelDay4, 0, 3);
		this._gridDaysContainer.setCellElement(this._labelDay5, 0, 4);
		this._gridDaysContainer.setCellElement(this._labelDay6, 0, 5);
		this._gridDaysContainer.setCellElement(this._labelDay7, 0, 6);

	this._rootListContainer.addElement(this._listContainerYearMonthSelection);
	this._rootListContainer.addElement(this._gridDaysContainer);

	this._addChild(this._rootListContainer);


	///////Event Handlers///////////
	
	//Private event handlers, different instance needed for each ColorPicker, proxy to prototype
	
	var _self = this;
	
	this._onButtonYearDecrementClickInstance = 
		function (mouseEvent)
		{
			_self._buttonYearDecrementClick(mouseEvent);
		};
		
	this._onButtonYearIncrementClickInstance = 
		function (mouseEvent)
		{
			_self._buttonYearIncrementClick(mouseEvent);
		};
		
	this._onButtonMonthDecrementClickInstance = 
		function (mouseEvent)
		{
			_self._buttonMonthDecrementClick(mouseEvent);
		};		
		
	this._onButtonMonthIncrementClickInstance = 
		function (mouseEvent)
		{
			_self._buttonMonthIncrementClick(mouseEvent);
		};
	
	this._buttonDayChangedInstance = 
		function (event)
		{
			_self._buttonDayChanged(event);
		};
		
	this._listContainerYearMonthSelectionLayoutCompleteInstance = 
		function (event)
		{
			_self._listContainerYearMonthSelectionLayoutComplete(event);
		};
		
	//////////////////
		
	//Populate days grid buttons
	var buttonDay = null;
	for (var week = 1; week < 7; week++)
	{
		for (var day = 0; day < 7; day++)
		{
			buttonDay = new ToggleButtonElement();
			buttonDay.setStyle("Enabled", false);
			buttonDay.setStyle("AllowDeselect", false);
			buttonDay.addEventListener("changed", this._buttonDayChangedInstance);
			
			this._gridDaysContainer.setCellElement(buttonDay, week, day);
		}
	}
	
	this._buttonYearDecrement.addEventListener("click", this._onButtonYearDecrementClickInstance);
	this._buttonYearIncrement.addEventListener("click", this._onButtonYearIncrementClickInstance);
	this._buttonMonthDecrement.addEventListener("click", this._onButtonMonthDecrementClickInstance);
	this._buttonMonthIncrement.addEventListener("click", this._onButtonMonthIncrementClickInstance);
	
	this._listContainerYearMonthSelection.addEventListener("layoutcomplete", this._listContainerYearMonthSelectionLayoutCompleteInstance);
	
	//////////////////
	
	var date = new Date();
	this._displayedYear = date.getFullYear();
	this._displayedMonth = date.getMonth();
	this._selectedDate = null;
	
	this.setSelectedDate(null);
}

//Inherit from CanvasElement
DatePickerElement.prototype = Object.create(CanvasElement.prototype);
DatePickerElement.prototype.constructor = DatePickerElement;
DatePickerElement.base = CanvasElement;


////////////Static//////////////////////

DatePickerElement._GridDaysRowColumnStyle = null;


////////////Events/////////////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the DatePicker selection state changes as a result of user interaction.
 */

/////////////Style Types///////////////////////////////

DatePickerElement._StyleTypes = Object.create(null);

/**
 * @style AllowDeselect boolean
 * When false, the day ToggleButtons cannot be de-selected by the user and the "selectedOver" and "selectedDown" states are not used.
 */
DatePickerElement._StyleTypes.AllowDeselect = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style LabelYearStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to apply to the year label element.
 */
DatePickerElement._StyleTypes.LabelYearStyle = 				StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style ButtonYearIncrementStyle StyleDefinition
 * StyleDefinition or [StyleDefinition] array to be applied to the year increment Button.
 */
DatePickerElement._StyleTypes.ButtonYearIncrementStyle = 	StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style ButtonYearDecrementStyle StyleDefinition
 * StyleDefinition or [StyleDefinition] array to be applied to the year decrement Button.
 */
DatePickerElement._StyleTypes.ButtonYearDecrementStyle = 	StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style LabelMonthStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the month label element.
 */
DatePickerElement._StyleTypes.LabelMonthStyle = 			StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style ButtonMonthIncrementStyle StyleDefinition
 * StyleDefinition or [StyleDefinition] array to be applied to the month increment Button.
 */
DatePickerElement._StyleTypes.ButtonMonthIncrementStyle = 	StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style ButtonMonthDecrementStyle StyleDefinition
 * StyleDefinition or [StyleDefinition] array to be applied to the month decrement Button.
 */
DatePickerElement._StyleTypes.ButtonMonthDecrementStyle = 	StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style LabelDayStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the day label elements.
 */
DatePickerElement._StyleTypes.LabelDayStyle = 				StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style ButtonDaysStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the day ToggleButton elements.
 */
DatePickerElement._StyleTypes.ToggleButtonDaysStyle = 		StyleableBase.EStyleType.SUBSTYLE;			//StyleDefinition

/**
 * @style GridDaysVerticalLayoutGap Number
 * Space in pixels between the day grid rows.
 */
DatePickerElement._StyleTypes.GridDaysVerticalLayoutGap = 	StyleableBase.EStyleType.NORMAL;		// Number

/**
 * @style GridDaysVerticalLayoutGap Number
 * Space in pixels between the day grid columns.
 */
DatePickerElement._StyleTypes.GridDaysHorizontalLayoutGap = StyleableBase.EStyleType.NORMAL;		// Number

/**
 * @style LayoutGap Number
 * Space in pixels between the month / year selection and the day selection grid.
 */
DatePickerElement._StyleTypes.LayoutGap = 					StyleableBase.EStyleType.NORMAL;		// Number

/**
 * @style Month1String String
 * String to use for month 1's name.
 */
DatePickerElement._StyleTypes.Month1String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month2String String
 * String to use for month 2's name.
 */
DatePickerElement._StyleTypes.Month2String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month3String String
 * String to use for month 3's name.
 */
DatePickerElement._StyleTypes.Month3String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month4String String
 * String to use for month 4's name.
 */
DatePickerElement._StyleTypes.Month4String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month5String String
 * String to use for month 5's name.
 */
DatePickerElement._StyleTypes.Month5String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month6String String
 * String to use for month 6's name.
 */
DatePickerElement._StyleTypes.Month6String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month7String String
 * String to use for month 7's name.
 */
DatePickerElement._StyleTypes.Month7String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month8String String
 * String to use for month 8's name.
 */
DatePickerElement._StyleTypes.Month8String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month9String String
 * String to use for month 9's name.
 */
DatePickerElement._StyleTypes.Month9String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month10String String
 * String to use for month 10's name.
 */
DatePickerElement._StyleTypes.Month10String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month11String String
 * String to use for month 11's name.
 */
DatePickerElement._StyleTypes.Month11String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Month12String String
 * String to use for month 12's name.
 */
DatePickerElement._StyleTypes.Month12String = 				StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Day1String String
 * String to use for day 1's name.
 */
DatePickerElement._StyleTypes.Day1String = 					StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Day2String String
 * String to use for day 2's name.
 */
DatePickerElement._StyleTypes.Day2String = 					StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Day3String String
 * String to use for day 3's name.
 */
DatePickerElement._StyleTypes.Day3String = 					StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Day4String String
 * String to use for day 4's name.
 */
DatePickerElement._StyleTypes.Day4String = 					StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Day5String String
 * String to use for day 5's name.
 */
DatePickerElement._StyleTypes.Day5String = 					StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Day6String String
 * String to use for day 6's name.
 */
DatePickerElement._StyleTypes.Day6String = 					StyleableBase.EStyleType.NORMAL;		// String

/**
 * @style Day7String String
 * String to use for day 7's name.
 */
DatePickerElement._StyleTypes.Day7String = 					StyleableBase.EStyleType.NORMAL;		// String


////////////Default Styles/////////////////////////////

DatePickerElement.StyleDefault = new StyleDefinition();

////Default Sub Styles/////

DatePickerElement.LabelYearMonthStyleDefault = new StyleDefinition();
DatePickerElement.LabelYearMonthStyleDefault.setStyle("TextHorizontalAlign", 	"center");
DatePickerElement.LabelYearMonthStyleDefault.setStyle("PaddingLeft", 			8);
DatePickerElement.LabelYearMonthStyleDefault.setStyle("PaddingRight", 			8);

DatePickerElement.ArrowShapeLeft = new ArrowShape();
DatePickerElement.ArrowShapeLeft.setStyle("Direction", "left");

DatePickerElement.ArrowShapeRight = new ArrowShape();
DatePickerElement.ArrowShapeRight.setStyle("Direction", "right");

DatePickerElement.ButtonYearMonthDecSkinStyleDefault = new StyleDefinition();
DatePickerElement.ButtonYearMonthDecSkinStyleDefault.setStyle("BackgroundShape", DatePickerElement.ArrowShapeLeft);

DatePickerElement.ButtonYearMonthIncSkinStyleDefault = new StyleDefinition();
DatePickerElement.ButtonYearMonthIncSkinStyleDefault.setStyle("BackgroundShape", DatePickerElement.ArrowShapeRight);

DatePickerElement.ButtonYearMonthDecStyleDefault = new StyleDefinition();
DatePickerElement.ButtonYearMonthDecStyleDefault.setStyle("PercentHeight", 		75);
DatePickerElement.ButtonYearMonthDecStyleDefault.setStyle("UpSkinStyle", 		DatePickerElement.ButtonYearMonthDecSkinStyleDefault);
DatePickerElement.ButtonYearMonthDecStyleDefault.setStyle("OverSkinStyle", 		DatePickerElement.ButtonYearMonthDecSkinStyleDefault);
DatePickerElement.ButtonYearMonthDecStyleDefault.setStyle("DownSkinStyle", 		DatePickerElement.ButtonYearMonthDecSkinStyleDefault);
DatePickerElement.ButtonYearMonthDecStyleDefault.setStyle("DisabledSkinStyle", 	DatePickerElement.ButtonYearMonthDecSkinStyleDefault);

DatePickerElement.ButtonYearMonthIncStyleDefault = new StyleDefinition();
DatePickerElement.ButtonYearMonthIncStyleDefault.setStyle("PercentHeight", 		75);
DatePickerElement.ButtonYearMonthIncStyleDefault.setStyle("UpSkinStyle", 		DatePickerElement.ButtonYearMonthIncSkinStyleDefault);
DatePickerElement.ButtonYearMonthIncStyleDefault.setStyle("OverSkinStyle", 		DatePickerElement.ButtonYearMonthIncSkinStyleDefault);
DatePickerElement.ButtonYearMonthIncStyleDefault.setStyle("DownSkinStyle", 		DatePickerElement.ButtonYearMonthIncSkinStyleDefault);
DatePickerElement.ButtonYearMonthIncStyleDefault.setStyle("DisabledSkinStyle", 	DatePickerElement.ButtonYearMonthIncSkinStyleDefault);

DatePickerElement.LabelDayStyleDefault = new StyleDefinition();
DatePickerElement.LabelDayStyleDefault.setStyle("TextHorizontalAlign",			"center");

DatePickerElement.ToggleButtonDaysSkinStyleDefault = new StyleDefinition();
DatePickerElement.ToggleButtonDaysSkinStyleDefault.setStyle("BorderType",		null);		

DatePickerElement.ToggleButtonDaysStyleDefault = new StyleDefinition();
DatePickerElement.ToggleButtonDaysStyleDefault.setStyle("UpSkinStyle", 			DatePickerElement.ToggleButtonDaysSkinStyleDefault);
DatePickerElement.ToggleButtonDaysStyleDefault.setStyle("OverSkinStyle", 		DatePickerElement.ToggleButtonDaysSkinStyleDefault);
DatePickerElement.ToggleButtonDaysStyleDefault.setStyle("DownSkinStyle", 		DatePickerElement.ToggleButtonDaysSkinStyleDefault);
DatePickerElement.ToggleButtonDaysStyleDefault.setStyle("DisabledSkinStyle", 	DatePickerElement.ToggleButtonDaysSkinStyleDefault);
DatePickerElement.ToggleButtonDaysStyleDefault.setStyle("PaddingTop", 			5);
DatePickerElement.ToggleButtonDaysStyleDefault.setStyle("PaddingBottom", 		5);
DatePickerElement.ToggleButtonDaysStyleDefault.setStyle("PaddingLeft", 			5);
DatePickerElement.ToggleButtonDaysStyleDefault.setStyle("PaddingRight", 		5);


////Root Styles////

DatePickerElement.StyleDefault.setStyle("AllowDeselect",						false);	
DatePickerElement.StyleDefault.setStyle("LabelYearStyle", 						DatePickerElement.LabelYearMonthStyleDefault);
DatePickerElement.StyleDefault.setStyle("ButtonYearDecrementStyle", 			DatePickerElement.ButtonYearMonthDecStyleDefault);
DatePickerElement.StyleDefault.setStyle("ButtonYearIncrementStyle", 			DatePickerElement.ButtonYearMonthIncStyleDefault);
DatePickerElement.StyleDefault.setStyle("LabelMonthStyle", 						DatePickerElement.LabelYearMonthStyleDefault);
DatePickerElement.StyleDefault.setStyle("ButtonMonthDecrementStyle", 			DatePickerElement.ButtonYearMonthDecStyleDefault);
DatePickerElement.StyleDefault.setStyle("ButtonMonthIncrementStyle", 			DatePickerElement.ButtonYearMonthIncStyleDefault);
DatePickerElement.StyleDefault.setStyle("LabelDayStyle", 						DatePickerElement.LabelDayStyleDefault);
DatePickerElement.StyleDefault.setStyle("ToggleButtonDaysStyle", 				DatePickerElement.ToggleButtonDaysStyleDefault);
DatePickerElement.StyleDefault.setStyle("LayoutGap",							8);
DatePickerElement.StyleDefault.setStyle("GridDaysVerticalLayoutGap",			1);
DatePickerElement.StyleDefault.setStyle("GridDaysHorizontalLayoutGap",			1);
DatePickerElement.StyleDefault.setStyle("PaddingTop", 							5);
DatePickerElement.StyleDefault.setStyle("PaddingBottom", 						5);
DatePickerElement.StyleDefault.setStyle("PaddingLeft", 							5);
DatePickerElement.StyleDefault.setStyle("PaddingRight", 						5);
DatePickerElement.StyleDefault.setStyle("BorderType", 							"solid");
DatePickerElement.StyleDefault.setStyle("BackgroundFill", 						"#FFFFFF");

DatePickerElement.StyleDefault.setStyle("Month1String", 						"Jan");
DatePickerElement.StyleDefault.setStyle("Month2String", 						"Feb");
DatePickerElement.StyleDefault.setStyle("Month3String", 						"Mar");
DatePickerElement.StyleDefault.setStyle("Month4String", 						"Apr");
DatePickerElement.StyleDefault.setStyle("Month5String", 						"May");
DatePickerElement.StyleDefault.setStyle("Month6String", 						"Jun");
DatePickerElement.StyleDefault.setStyle("Month7String", 						"Jul");
DatePickerElement.StyleDefault.setStyle("Month8String", 						"Aug");
DatePickerElement.StyleDefault.setStyle("Month9String", 						"Sep");
DatePickerElement.StyleDefault.setStyle("Month10String", 						"Oct");
DatePickerElement.StyleDefault.setStyle("Month11String", 						"Nov");
DatePickerElement.StyleDefault.setStyle("Month12String", 						"Dec");

DatePickerElement.StyleDefault.setStyle("Day1String", 							"S");
DatePickerElement.StyleDefault.setStyle("Day2String", 							"M");
DatePickerElement.StyleDefault.setStyle("Day3String", 							"T");
DatePickerElement.StyleDefault.setStyle("Day4String", 							"W");
DatePickerElement.StyleDefault.setStyle("Day5String", 							"T");
DatePickerElement.StyleDefault.setStyle("Day6String", 							"F");
DatePickerElement.StyleDefault.setStyle("Day7String", 							"S");



////////////Public////////////////
	
/**
 * @function getSelectedDate
 * Gets the selected date of the DatePickerElement.
 * 
 * @returns Date
 * Currently selected date or null if none selected.
 */
DatePickerElement.prototype.getSelectedDate = 
	function ()
	{
		return this._selectedDate;
	};

/**
 * @function setSelectedDate
 * Sets the selected date of the DatePickerElement.
 * 
 * @param date Date
 * Date to set as the selected date or null for no selection.
 */	
DatePickerElement.prototype.setSelectedDate = 
	function (date)
	{
		if (date != null && date instanceof Date == false)
			date = new Date();
	
		this._selectedDate = date;
		
		this._updateCalendar();		
	};
	
/**
 * @function getDisplayedYear
 * Gets the 4 digit year currently displayed
 * 
 * @returns int
 * 4 digit year currently displayed.
 */	
DatePickerElement.prototype.getDisplayedYear = 
	function ()
	{
		return this._displayedYear;
	};
	
/**
 * @function setSelectedDate
 * Sets the 4 digit currently displayed.
 * 
 * @param year int
 * 4 digit year to display.
 */	
DatePickerElement.prototype.setDisplayedYear = 
	function (year)
	{
		this._displayedYear = year;
		this._updateCalendar();		
	};
	
/**
 * @function getDisplayedMonth
 * Gets the 2 digit month currently displayed (0-11)
 * 
 * @returns int
 * 2 digit month currently displayed (0-11).
 */	
DatePickerElement.prototype.getDisplayedYear = 
	function ()
	{
		return this._displayedMonth;
	};
	
/**
 * @function setDisplayedMonth
 * Sets the 2 digit month currently displayed.
 * 
 * @param month int
 * 2 digit month currently displayed (0-11).
 * Out of range months will be wrapped and years will be adjusted.
 */	
DatePickerElement.prototype.setDisplayedMonth = 
	function (month)
	{
		while (month < 0)
		{
			month += 12;
			this._displayedYear--;
		}
		
		while (month > 11)
		{
			month -= 12;
			this._displayedYear++;
		}
	
		this._displayedMonth = month;
		this._updateCalendar();		
	};	
	
	
////////////Internal//////////////
	
/**
 * @function _buttonYearDecrementClick
 * Event handler for the decrement year Button click event
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
DatePickerElement.prototype._buttonYearDecrementClick = 
	function (elementMouseEvent)
	{
		this._displayedYear--;
		this._updateCalendar();
	};

/**
 * @function _buttonYearIncrementClick
 * Event handler for the increment year Button click event
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
DatePickerElement.prototype._buttonYearIncrementClick = 
	function (elementMouseEvent)
	{
		this._displayedYear++;
		this._updateCalendar();
	};	
	
/**
 * @function _buttonMonthDecrementClick
 * Event handler for the decrement month Button click event
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */			
DatePickerElement.prototype._buttonMonthDecrementClick = 
	function (elementMouseEvent)
	{
		this._displayedMonth--;
		if (this._displayedMonth == -1)
		{
			this._displayedYear--;
			this._displayedMonth = 11;
		}
		
		this._updateCalendar();
	};

/**
 * @function _buttonMonthIncrementClick
 * Event handler for the increment month Button click event
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
DatePickerElement.prototype._buttonMonthIncrementClick = 
	function (elementMouseEvent)
	{
		this._displayedMonth++;
		if (this._displayedMonth == 12)
		{
			this._displayedYear++;
			this._displayedMonth = 0;
		}
		
		this._updateCalendar();
	};		
	
/**
 * @function _listContainerYearMonthSelectionLayoutComplete
 * Event handler for the year / month selection list containers layoutcomplete event.
 * Used to adjust the measured sizes of the year / month increment & decrement buttons.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */			
DatePickerElement.prototype._listContainerYearMonthSelectionLayoutComplete = 
	function (elementEvent)
	{
		this._buttonYearDecrement._setMeasuredSize(Math.round(this._buttonYearDecrement._height * .8), this._buttonYearDecrement._height);
		this._buttonYearIncrement._setMeasuredSize(Math.round(this._buttonYearIncrement._height * .8), this._buttonYearIncrement._height);

		this._buttonMonthDecrement._setMeasuredSize(Math.round(this._buttonMonthDecrement._height * .8), this._buttonMonthDecrement._height);
		this._buttonMonthIncrement._setMeasuredSize(Math.round(this._buttonMonthIncrement._height * .8), this._buttonMonthIncrement._height);
	};
	
/**
 * @function _buttonDayChanged
 * Event handler for the day ToggleButton's changed event
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */			
DatePickerElement.prototype._buttonDayChanged = 
	function (elementEvent)
	{
		var day = Number(elementEvent.getTarget().getStyle("Text"));
		
		if (elementEvent.getTarget().getSelected() == false)
			this._selectedDate = null;
		else
		{
			this._selectedDate = new Date();
			this._selectedDate.setFullYear(this._displayedYear);
			this._selectedDate.setMonth(this._displayedMonth);
			this._selectedDate.setDate(day);
		}
		
		this._updateCalendar();
		
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};
	
/**
 * @function _updateCalendar
 * Updates the calendar when the displayed month, year, or selected date changes.
 */			
DatePickerElement.prototype._updateCalendar = 
	function ()
	{
		this._labelYear.setStyle("Text", this._displayedYear.toString());
		
		for (var i = 0; i < 12; i++)
		{
			if (this._displayedMonth == i)
				this["_labelMonth" + (i + 1).toString()].setStyle("Visible", true);
			else
				this["_labelMonth" + (i + 1).toString()].setStyle("Visible", false);
		}
		
		var date = new Date();
		date.setFullYear(this._displayedYear);
		date.setMonth(this._displayedMonth);
		date.setDate(0);
		date.setDate(date.getDate() - date.getDay());
		
		var toggleButton = null;
		for (var week = 1; week < 7; week++)
		{
			for (var day = 0; day < 7; day++)
			{
				toggleButton = this._gridDaysContainer.getCellElement(week, day);
				toggleButton.setStyle("Text", date.getDate().toString());
				
				if (date.getMonth() == this._displayedMonth)
				{
					toggleButton.setStyle("Enabled", this.getStyle("Enabled"));
					
					if (this._selectedDate != null &&
						date.getFullYear() == this._selectedDate.getFullYear() && 
						date.getMonth() == this._selectedDate.getMonth() &&
						date.getDate() == this._selectedDate.getDate())
					{
						toggleButton.setSelected(true);
					}
					else
						toggleButton.setSelected(false);
				}
				else
				{
					toggleButton.setStyle("Enabled", false);
					toggleButton.setSelected(false);
				}
					
				date.setDate(date.getDate() + 1);
			}
		}
	};

//@override
DatePickerElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DatePickerElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("LabelYearStyle" in stylesMap)
			this._applySubStylesToElement("LabelYearStyle", this._labelYear);
		
		if ("ButtonYearDecrementStyle" in stylesMap)
			this._applySubStylesToElement("ButtonYearDecrementStyle", this._buttonYearDecrement);
		
		if ("ButtonYearIncrementStyle" in stylesMap)
			this._applySubStylesToElement("ButtonYearIncrementStyle", this._buttonYearIncrement);
		
		if ("LabelMonthStyle" in stylesMap)
		{
			for (var i = 0; i < 12; i++)
				this._applySubStylesToElement("LabelMonthStyle", this["_labelMonth" + (i + 1).toString() ]);
		}
		
		if ("ButtonMonthDecrementStyle" in stylesMap)
			this._applySubStylesToElement("ButtonMonthDecrementStyle", this._buttonMonthDecrement);
		
		if ("ButtonMonthIncrementStyle" in stylesMap)
			this._applySubStylesToElement("ButtonMonthIncrementStyle", this._buttonMonthIncrement);
		
		if ("LabelDayStyle" in stylesMap)
		{
			for (var day = 0; day < 7; day++)
				this._applySubStylesToElement("LabelDayStyle", this._gridDaysContainer.getCellElement(0, day));
		}
		
		if ("ToggleButtonDaysStyle" in stylesMap)
		{
			for (var week = 1; week < 7; week++)
			{
				for (var day = 0; day < 7; day++)
					this._applySubStylesToElement("ToggleButtonDaysStyle", this._gridDaysContainer.getCellElement(week, day));
			}
		}
		
		//Pass "AllowDeselect" to day ToggleButtons
		if ("AllowDeselect" in stylesMap)
		{
			for (var week = 1; week < 7; week++)
			{
				for (var day = 0; day < 7; day++)
					this._gridDaysContainer.getCellElement(week, day).setStyle("AllowDeselect", this.getStyle("AllowDeselect"));
			}
		}
		
		if ("Enabled" in stylesMap)
			this._updateCalendar();
		
		if ("LayoutGap" in stylesMap)
			this._rootListContainer.setStyle("LayoutGap", this.getStyle("LayoutGap"));
		
		if ("GridDaysVerticalLayoutGap" in stylesMap)
			this._gridDaysContainer.setStyle("LayoutVerticalGap", this.getStyle("GridDaysVerticalLayoutGap"));
		
		if ("GridDaysHorizontalLayoutGap" in stylesMap)
			this._gridDaysContainer.setStyle("LayoutHorizontalGap", this.getStyle("GridDaysHorizontalLayoutGap"));
		
		//Update day strings
		if ("Day1String" in stylesMap)
			this._labelDay1.setStyle("Text", this.getStyle("Day1String"));
		if ("Day2String" in stylesMap)
			this._labelDay2.setStyle("Text", this.getStyle("Day2String"));
		if ("Day3String" in stylesMap)
			this._labelDay3.setStyle("Text", this.getStyle("Day3String"));
		if ("Day4String" in stylesMap)
			this._labelDay4.setStyle("Text", this.getStyle("Day4String"));
		if ("Day5String" in stylesMap)
			this._labelDay5.setStyle("Text", this.getStyle("Day5String"));
		if ("Day6String" in stylesMap)
			this._labelDay6.setStyle("Text", this.getStyle("Day6String"));
		if ("Day7String" in stylesMap)
			this._labelDay7.setStyle("Text", this.getStyle("Day7String"));
		
		//Update month string
		if ("Month1String" in stylesMap)
			this._labelMonth1.setStyle("Text", this.getStyle("Month1String"));
		if ("Month2String" in stylesMap)
			this._labelMonth2.setStyle("Text", this.getStyle("Month2String"));
		if ("Month3String" in stylesMap)
			this._labelMonth3.setStyle("Text", this.getStyle("Month3String"));
		if ("Month4String" in stylesMap)
			this._labelMonth4.setStyle("Text", this.getStyle("Month4String"));
		if ("Month5String" in stylesMap)
			this._labelMonth5.setStyle("Text", this.getStyle("Month5String"));
		if ("Month6String" in stylesMap)
			this._labelMonth6.setStyle("Text", this.getStyle("Month6String"));
		if ("Month7String" in stylesMap)
			this._labelMonth7.setStyle("Text", this.getStyle("Month7String"));
		if ("Month8String" in stylesMap)
			this._labelMonth8.setStyle("Text", this.getStyle("Month8String"));
		if ("Month9String" in stylesMap)
			this._labelMonth9.setStyle("Text", this.getStyle("Month9String"));
		if ("Month10String" in stylesMap)
			this._labelMonth10.setStyle("Text", this.getStyle("Month10String"));
		if ("Month11String" in stylesMap)
			this._labelMonth11.setStyle("Text", this.getStyle("Month11String"));
		if ("Month12String" in stylesMap)
			this._labelMonth12.setStyle("Text", this.getStyle("Month12String"));
	};	

//@override
DatePickerElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		//Root list container measures for us, so just add padding
		this._setMeasuredSize(padWidth + this._rootListContainer._measuredWidth, 
							padHeight + this._rootListContainer._measuredHeight);
	};
	
//@override	
DatePickerElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		DatePickerElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		//Place root list container and consider padding.
		this._rootListContainer._setActualSize(w, h);
		this._rootListContainer._setActualPosition(x, y);
	};	
	
	

	


/**
 * @depends SkinnableElement.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////////DataRendererBaseElement/////////////////////////
	
/**
 * @class DataRendererBaseElement
 * @inherits SkinnableElement
 * 
 * Abstract base class for DataList item rendering. Any CanvasElement
 * can be a data renderer. This class is just here for convenience as it
 * implements some commonly used functionality if you wish to subclass it. 
 * 
 * Adds skin states and styles for "up", "alt", "over", "disabled", and "selected" states. 
 * DataRendererBaseElement automatically checks for properties "selectable"
 * and "enabled" on supplied itemData and adjusts styles and skin states.
 *  
 * 
 * @constructor DataRendererBaseElement 
 * Creates new DataRendererBaseElement instance.
 */
function DataRendererBaseElement()
{
	DataRendererBaseElement.base.prototype.constructor.call(this);
	
	var _self = this;
	
	//Private event handler, proxy to prototype.
	this._onDataRendererBaseEventInstance = 
		function (elementEvent)
		{
			if (elementEvent.getType() == "rollover")
				_self._onDataRendererRollover(elementEvent);
			else if (elementEvent.getType() == "rollout")
				_self._onDataRendererRollout(elementEvent);
		};
		
	this._onDataRendererBaseClickInstance = 
		function (event)
		{
			_self._onDataRendererBaseClick(event);
		};
		
	this.addEventListener("rollover", this._onDataRendererBaseEventInstance);
	this.addEventListener("rollout", this._onDataRendererBaseEventInstance);
	this.addEventListener("click", this._onDataRendererBaseClickInstance);
}
	
//Inherit from SkinnableElement
DataRendererBaseElement.prototype = Object.create(SkinnableElement.prototype);
DataRendererBaseElement.prototype.constructor = DataRendererBaseElement;
DataRendererBaseElement.base = SkinnableElement;


/////////////Style Types////////////////////////////////////////////

DataRendererBaseElement._StyleTypes = Object.create(null);

/**
 * @style SkinClass CanvasElement
 * 
 * The CanvasElement constructor type to apply to all skin states. 
 * Specific states such as UpSkinClass will override SkinClass.
 */
DataRendererBaseElement._StyleTypes.SkinClass =					StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style UpSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the data renderer skin when in the "up" state. 
 * This will override SkinClass.
 */
DataRendererBaseElement._StyleTypes.UpSkinClass = 				StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style UpSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "up" state skin element.
 */
DataRendererBaseElement._StyleTypes.UpSkinStyle = 				StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style AltSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the data renderer skin when in the "alt" state. 
 * This is used to create different styles for alternating rows. 
 * This will override SkinClass.
 */
DataRendererBaseElement._StyleTypes.AltSkinClass = 				StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style AltSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "alt" state skin element.
 */
DataRendererBaseElement._StyleTypes.AltSkinStyle = 				StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style OverSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the data renderer skin when in the "over" state. 
 * This will override SkinClass.
 */
DataRendererBaseElement._StyleTypes.OverSkinClass = 			StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style OverSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "over" state skin element.
 */
DataRendererBaseElement._StyleTypes.OverSkinStyle = 			StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style SelectedSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the data renderer skin when in the "selected" state. 
 * This will override SkinClass.
 */
DataRendererBaseElement._StyleTypes.SelectedSkinClass = 		StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style SelectedSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "selected" state skin element.
 */
DataRendererBaseElement._StyleTypes.SelectedSkinStyle = 		StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style DisabledSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the data renderer skin when in the "disabled" state. 
 * This will override SkinClass.
 */
DataRendererBaseElement._StyleTypes.DisabledSkinClass = 		StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style DisabledSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "disabled" state skin element.
 */
DataRendererBaseElement._StyleTypes.DisabledSkinStyle = 		StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

//Proxied from DataList
/**
 * @style Selectable boolean
 * 
 * When false, prevents "over" and "selected" states. Proxied from parent DataList.
 */
DataRendererBaseElement._StyleTypes.Selectable = 				StyleableBase.EStyleType.NORMAL;		// true || false


///////////Default Styles///////////////////////////////////////////

DataRendererBaseElement.StyleDefault = new StyleDefinition();

//Skin Defaults////////////////////////////
DataRendererBaseElement.OverSkinStyleDefault = new StyleDefinition();

DataRendererBaseElement.OverSkinStyleDefault.setStyle("BackgroundFill", 				"#E0E0E0");

DataRendererBaseElement.SelectedSkinStyleDefault = new StyleDefinition();

DataRendererBaseElement.SelectedSkinStyleDefault.setStyle("BackgroundFill", 			"#CDCDCD");
//////////////////////////////////////////

DataRendererBaseElement.StyleDefault.setStyle("Selectable", 			true);												//Proxied from List - may be overridden in _setListData()

DataRendererBaseElement.StyleDefault.setStyle("SkinClass", 				CanvasElement);										// Element constructor()
DataRendererBaseElement.StyleDefault.setStyle("UpSkinClass", 			CanvasElement);										// Element constructor()
DataRendererBaseElement.StyleDefault.setStyle("AltSkinClass", 			CanvasElement);										// Element constructor()
DataRendererBaseElement.StyleDefault.setStyle("OverSkinClass", 			CanvasElement);										// Element constructor()
DataRendererBaseElement.StyleDefault.setStyle("SelectedSkinClass", 		CanvasElement);										// Element constructor()
DataRendererBaseElement.StyleDefault.setStyle("DisabledSkinClass", 		CanvasElement);										// Element constructor()

DataRendererBaseElement.StyleDefault.setStyle("UpSkinStyle", 			null);												// StyleDefinition
DataRendererBaseElement.StyleDefault.setStyle("AltSkinStyle", 			null);												// StyleDefinition
DataRendererBaseElement.StyleDefault.setStyle("OverSkinStyle", 			DataRendererBaseElement.OverSkinStyleDefault);		// StyleDefinition
DataRendererBaseElement.StyleDefault.setStyle("SelectedSkinStyle", 		DataRendererBaseElement.SelectedSkinStyleDefault);	// StyleDefinition
DataRendererBaseElement.StyleDefault.setStyle("DisabledSkinStyle", 		null);												// StyleDefinition


/////////////Internal///////////////////////////

/**
 * @function _onDataRendererBaseClick
 * Event handler for "click" event. Cancels the event if the DataRendererBase is disabled.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */			
DataRendererBaseElement.prototype._onDataRendererBaseClick = 
	function (elementMouseEvent)
	{
		//Implementor will not expect a click event when DataRendererBase is disabled. 
		if (this.getStyle("Enabled") == false)
			elementMouseEvent.cancelEvent();
	};

/**
 * @function _updateState
 * Updates the SkinState style in response to mouse and ListData changes.
 */
DataRendererBaseElement.prototype._updateState = 
	function ()
	{
		var newState = "";
	
		if (this._listSelected == true)
			newState = "selected";
		else if (this.getStyle("Enabled") == false)
			newState = "disabled";
		else if (this._mouseIsOver == true && this.getStyle("Selectable") == true)
			newState = "over";
		else // "up"
		{
			if (this._listData == null || this._listData._itemIndex % 2 == 0)
				newState = "up";
			else
				newState = "alt";
		}
		
		this.setStyle("SkinState", newState);
	};

//@Override
DataRendererBaseElement.prototype._getSkinClass = 
	function (state)
	{
		var stateSkinClass = null;
		
		if (state == "up")
			stateSkinClass = this.getStyleData("UpSkinClass");
		else if (state == "alt")
			stateSkinClass = this.getStyleData("AltSkinClass");
		else if (state == "over")
			stateSkinClass = this.getStyleData("OverSkinClass");
		else if (state == "selected")
			stateSkinClass = this.getStyleData("SelectedSkinClass");
		else if (state == "disabled")
			stateSkinClass = this.getStyleData("DisabledSkinClass");
		
		var skinClass = this.getStyleData("SkinClass");
		
		//Shouldnt have null stateSkinClass
		if (stateSkinClass == null || skinClass.comparePriority(stateSkinClass) > 0) //Use skinClass if higher priority
			return skinClass.value;
		
		return stateSkinClass.value;
	};
		
//@override	
DataRendererBaseElement.prototype._getSubStyleNameForSkinState = 
	function (state)
	{
		if (state == "up")
			return "UpSkinStyle";
		if (state == "alt")
			return "AltSkinStyle";
		if (state == "over")
			return "OverSkinStyle";
		if (state == "selected")
			return "SelectedSkinStyle";
		if (state == "disabled")
			return "DisabledSkinStyle";
		
		return DataRendererBaseElement.base.prototype._getSubStyleNameForSkinState.call(this, state);
	};	
	
/**
 * @function _onDataRendererRollover
 * Event handler for "rollover" event. Updates the skin state.
 * Overriding this is more efficient than adding an additional "rollover" event listener.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
DataRendererBaseElement.prototype._onDataRendererRollover = 
	function (elementEvent)
	{
		this._updateState();
	};
	
/**
 * @function _onDataRendererRollout
 * Event handler for "rollout" event. Updates the skin state.
 * Overriding this is more efficient than adding an additional "rollout" event listener.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
DataRendererBaseElement.prototype._onDataRendererRollout = 
	function (elementEvent)
	{
		this._updateState();
	};

//@Override	
DataRendererBaseElement.prototype._setListSelected = 
	function (selected)
	{
		DataRendererBaseElement.base.prototype._setListSelected.call(this, selected);
		
		this._updateState();
	};
	
//@Override	
DataRendererBaseElement.prototype._setListData = 
	function (listData, itemData)
	{
		DataRendererBaseElement.base.prototype._setListData.call(this, listData, itemData);
		
		if (itemData.hasOwnProperty("enabled") == true)
			this.setStyle("Enabled", itemData.enabled);
		else
			this.clearStyle("Enabled");
			
		if (itemData.hasOwnProperty("selectable") == true)
			this.setStyle("Selectable", itemData.selectable);
		else
			this.clearStyle("Selectable");
		
		this._updateState();
	};

//@Override
DataRendererBaseElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DataRendererBaseElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		
		////Update skin classes and sub styles.
		if ("SkinClass" in stylesMap || "UpSkinClass" in stylesMap)
			this._updateSkinClass("up");
		if ("UpSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("up");
		
		if ("SkinClass" in stylesMap || "AltSkinClass" in stylesMap)
			this._updateSkinClass("alt");
		if ("AltSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("alt");
		
		if ("SkinClass" in stylesMap || "OverSkinClass" in stylesMap)
			this._updateSkinClass("over");
		if ("OverSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("over");
		
		if ("SkinClass" in stylesMap || "SelectedSkinClass" in stylesMap)
			this._updateSkinClass("selected");
		if ("SelectedSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("selected");		
		
		if ("SkinClass" in stylesMap || "DisabledSkinClass" in stylesMap)
			this._updateSkinClass("disabled");
		if ("DisabledSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("disabled");
		
		if ("Selectable" in stylesMap || "Enabled" in stylesMap)
			this._updateState();
	};
	
	


/**
 * @depends DataRendererBaseElement.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////////DataRendererLabelElement////////////////////////

/**
 * @class DataRendererLabelElement
 * @inherits DataRendererBaseElement
 * 
 * DataList DataRenderer for a basic label.
 * Adds text color styles for DataRenderer states and 
 * sets label text when the parent DataList sets our list data.
 * 
 * @constructor DataRendererLabelElement 
 * Creates new DataRendererLabelElement instance.
 */
function DataRendererLabelElement()
{
	DataRendererLabelElement.base.prototype.constructor.call(this);
	
	this._labelElement = new LabelElement();
	this._labelElement.setStyle("Padding", 0); //Wipe out default padding (no doubly padding, only this elements padding is necessary)
	
	this._addChild(this._labelElement);
}
	
//Inherit from DataRendererBaseElement
DataRendererLabelElement.prototype = Object.create(DataRendererBaseElement.prototype);
DataRendererLabelElement.prototype.constructor = DataRendererLabelElement;
DataRendererLabelElement.base = DataRendererBaseElement;


/////////////Style Types/////////////////////////

DataRendererLabelElement._StyleTypes = Object.create(null);

/**
 * @style UpTextColor String
 * 
 * Hex color value to be used for the label when in the "up" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataRendererLabelElement._StyleTypes.UpTextColor = 				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style AltTextColor String
 * 
 * Hex color value to be used for the label when in the "alt" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataRendererLabelElement._StyleTypes.AltTextColor = 			StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style OverTextColor String
 * 
 * Hex color value to be used for the label when in the "over" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataRendererLabelElement._StyleTypes.OverTextColor = 			StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style SelectedTextColor String
 * 
 * Hex color value to be used for the label when in the "selected" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataRendererLabelElement._StyleTypes.SelectedTextColor = 		StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style DisabledTextColor String
 * 
 * Hex color value to be used for the label when in the "disabled" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataRendererLabelElement._StyleTypes.DisabledTextColor = 		StyleableBase.EStyleType.NORMAL;		//"#000000"


////////////Default Styles///////////////////////

DataRendererLabelElement.StyleDefault = new StyleDefinition();

DataRendererLabelElement.StyleDefault.setStyle("PaddingTop", 				4);
DataRendererLabelElement.StyleDefault.setStyle("PaddingBottom", 			4);
DataRendererLabelElement.StyleDefault.setStyle("PaddingLeft", 				4);
DataRendererLabelElement.StyleDefault.setStyle("PaddingRight", 				4);

DataRendererLabelElement.StyleDefault.setStyle("UpTextColor", 				"#000000");
DataRendererLabelElement.StyleDefault.setStyle("AltTextColor", 				"#000000");
DataRendererLabelElement.StyleDefault.setStyle("OverTextColor", 			"#000000");
DataRendererLabelElement.StyleDefault.setStyle("SelectedTextColor", 		"#000000");
DataRendererLabelElement.StyleDefault.setStyle("DisabledTextColor", 		"#888888");


////////////Internal/////////////////////////////

//@Override
DataRendererLabelElement.prototype._changeState = 
	function (state)
	{
		DataRendererLabelElement.base.prototype._changeState.call(this, state);
		
		this._updateLabelTextColor();
	};
	
/**
 * @function _getTextColor
 * Gets the text color style for the supplied state.
 * 
 * @param state String
 * The current state.
 * 
 * @returns String
 * Text color style for the supplied state.
 */	
DataRendererLabelElement.prototype._getTextColor = 
	function (state)
	{
		var stateTextColor = null;
		
		if (state == "up")
			stateTextColor = this.getStyleData("UpTextColor");
		else if (state == "alt")
			stateTextColor = this.getStyleData("AltTextColor");
		else if (state == "over")
			stateTextColor = this.getStyleData("OverTextColor");
		else if (state == "selected")
			stateTextColor = this.getStyleData("SelectedTextColor");
		else if (state == "disabled")
			stateTextColor = this.getStyleData("DisabledTextColor");
	
		var textColor = this.getStyleData("TextColor");
		
		//Shouldnt have null stateTextColor
		if (stateTextColor == null || textColor.comparePriority(stateTextColor) > 0) //Use textColor if higher priority
			return textColor.value;
		
		return stateTextColor.value;
	};

/**
 * @function _updateLabelTextColor
 * Updates the text color for the current state.
 */	
DataRendererLabelElement.prototype._updateLabelTextColor = 
	function ()
	{
		var color = this._getTextColor(this._currentSkinState);
		if (color != null)
			this._labelElement.setStyle("TextColor", color);
	};
	
/**
 * @function _updateLabelTextColor
 * Updates the label text base on the list data and ItemLabelFunction.
 */		
DataRendererLabelElement.prototype._updateLabelText = 
	function ()
	{
		if (this._itemData == null)
			this._labelElement.setStyle("Text", "");
		else
		{
			var labelFunction = this._listData._parentList.getStyle("ItemLabelFunction");
			this._labelElement.setStyle("Text", labelFunction(this._itemData));
		}
	};
	
//@Override
DataRendererLabelElement.prototype._setListData = 
	function (listData, itemData)
	{
		DataRendererLabelElement.base.prototype._setListData.call(this, listData, itemData);
		
		this._updateLabelText();
	};

//@Override
DataRendererLabelElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DataRendererLabelElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		this._updateLabelTextColor();
	};

//@Override
DataRendererLabelElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		this._setMeasuredSize(this._labelElement._getStyledOrMeasuredWidth() + padWidth, 
							this._labelElement._getStyledOrMeasuredHeight() + padHeight);
	};

//@Override	
DataRendererLabelElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		DataRendererLabelElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		this._labelElement._setActualPosition(paddingMetrics.getX(), paddingMetrics.getY());
		this._labelElement._setActualSize(paddingMetrics.getWidth(), paddingMetrics.getHeight());
	};
	
	


/**
 * @depends CanvasElement.js
 * @depends DataRendererLabelElement.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////////DataListElement/////////////////////////////////		
	
/**
 * @class DataListElement
 * @inherits CanvasElement
 * 
 * DataListElement is a data-driven container that renders items in a 
 * horizontal or vertical orientation via a supplied ListCollection and a supplied DataRenderer class.
 * A DataRenderer is any CanvasElement that implements _setListData() and _setListSelected()
 * and is used to render the corresponding ListCollection item. A scroll bar will be added
 * if the collection size exceeds the available area. DataListElement only renders visible 
 * DataRenderers so collection size does not impact performance.
 * 
 * DataRendereBaseElement is an abstract base class that implements common
 * features and can be sub-classed if desired.
 * 
 * The default DataRenderer is the DataRendererLabelElement which renders
 * a text label per the LabelFunction style. 
 * 
 * @seealso DataRendererBaseElement
 * @seealso DataRendererLabelElement
 * 
 * 
 * @constructor DataListElement 
 * Creates new DataListElement instance.
 */
function DataListElement()
{
	DataListElement.base.prototype.constructor.call(this);
	
	this._listCollection = null; //Data collection
	
	this._contentSize = 0;
	
	this._scrollBar = null;
	this._scrollIndex = 0;
	
	this._selectedIndex = -1;
	this._selectedItem = null;
	
	this._contentPane = new CanvasElement();
	this._contentPane.setStyle("ClipContent", true);
	this._addChild(this._contentPane);
	
	var _self = this;
	
	//Private event listener, need an instance for each DataList, proxy to prototype.
	this._onDataListCollectionChangedInstance = 
		function (collectionChangedEvent)
		{
			_self._onDataListCollectionChanged(collectionChangedEvent);
		};
	this._onDataListScrollBarChangedInstance = 
		function (elementEvent)
		{
			_self._onDataListScrollBarChanged(elementEvent);
		};
	this._onDataListMouseWheelEventInstance = 
		function (elementMouseWheelEvent)
		{
			_self._onDataListMouseWheelEvent(elementMouseWheelEvent);
		};
	this._onDataListRendererClickInstance = 
		function (elementMouseEvent)
		{
			_self._onDataListRendererClick(elementMouseEvent);
		};
	this._onContentPaneMeasureCompleteInstance = 
		function (event)
		{
			_self._onContentPaneMeasureComplete(event);
		};	
		
	this.addEventListener("wheel", this._onDataListMouseWheelEventInstance);	
	this._contentPane.addEventListener("measurecomplete", this._onContentPaneMeasureCompleteInstance);
}

//Inherit from SkinnableElement
DataListElement.prototype = Object.create(CanvasElement.prototype);
DataListElement.prototype.constructor = DataListElement;
DataListElement.base = CanvasElement;

/**
 * @function DefaultItemLabelFunction
 * @static
 * Default ItemLabelFunction function. Looks for typeof String or "label" property of supplied itemData.
 * 
 * @param itemData Object
 * Collection item to extract label text.
 * 
 * @returns String
 * Label text.
 */
DataListElement.DefaultItemLabelFunction = 
	function (itemData)
	{
		if (itemData == null)
			return "";
		
		if (typeof itemData === "string" || itemData instanceof String)
			return itemData;
	
		if ("label" in itemData)
			return itemData["label"];
		
		return itemData.toString();
	};


/////////////Events///////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the selected item/index changes as a result of user interaction.
 * 
 * @event listitemclick ElementListItemClickEvent
 * Dispatched when a DataRenderer is clicked. Includes associated collection item/index.
 */



/////////////Style Types////////////////////////////////////////////

DataListElement._StyleTypes = Object.create(null);

/**
 * @style LayoutDirection String
 * 
 * Determines the layout direction of this DataList. Allowable values are "horizontal" or "vertical".
 */
DataListElement._StyleTypes.LayoutDirection = 					StyleableBase.EStyleType.NORMAL;		// "horizontal" || "vertical

/**
 * @style LayoutGap Number
 * 
 * Space in pixels to leave between child elements. 
 * (Not yet implemented)
 */
DataListElement._StyleTypes.LayoutGap = 						StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style LayoutVerticalAlign String
 * 
 * Child vertical alignment to be used when children do not fill all available space. Allowable values are "top", "bottom", or "middle". 
 * (Only partially implemented, depending on LayoutDirection)
 */
DataListElement._StyleTypes.LayoutVerticalAlign = 				StyleableBase.EStyleType.NORMAL;		// "top" || "bottom" || "middle" 

/**
 * @style LayoutHorizontalAlign String
 * 
 * Child horizontal alignment to be used when children do not fill all available space. Allowable values are "left", "right", or "center". 
 * (Only partially implemented, depending on LayoutDirection)
 */
DataListElement._StyleTypes.LayoutHorizontalAlign = 			StyleableBase.EStyleType.NORMAL;		//"left" || "right" || "center"

/**
 * @style Selectable boolean
 * 
 * When true, list items can be selected and the DataList will dispatch "changed" events.
 */
DataListElement._StyleTypes.Selectable = 						StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style ScrollBarDisplay String
 * 
 * Determines the behavior of the scroll bar. Allowable values are "on", "off", and "auto".
 */
DataListElement._StyleTypes.ScrollBarDisplay = 					StyleableBase.EStyleType.NORMAL;		// "on" || "off" || "auto"

/**
 * @style ScrollBarPlacement String
 * 
 * Determines the placement of the scroll bar. 
 * Allowable values are "top" or "bottom" for horizontal layout and "left" or "right" for vertical layout.
 */
DataListElement._StyleTypes.ScrollBarPlacement = 				StyleableBase.EStyleType.NORMAL;		// "top" || "bottom" / "left || "right"

/**
 * @style ScrollBarStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to be applied to the scroll bar.
 */
DataListElement._StyleTypes.ScrollBarStyle = 					StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

//Returns the string to use for the label per provided data.
/**
 * @style ItemLabelFunction Function
 * 
 * A function that returns a text string per a supplied collection item.
 * function (itemData) { return "" }
 */
DataListElement._StyleTypes.ItemLabelFunction = 				StyleableBase.EStyleType.NORMAL; 		// function (itemData) { return "" }

/**
 * @style ListItemClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the DataRenderer.
 */
DataListElement._StyleTypes.ListItemClass = 					StyleableBase.EStyleType.NORMAL;		//DataRendererBaseElement constructor()

/**
 * @style ListItemStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to be applied to the DataRenderer.
 */
DataListElement._StyleTypes.ListItemStyle = 					StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition


///////////Default Styles////////////////////////////////////

DataListElement.StyleDefault = new StyleDefinition();

DataListElement.StyleDefault.setStyle("LayoutDirection", 			"vertical");								// "horizontal" || "vertical
DataListElement.StyleDefault.setStyle("LayoutVerticalAlign", 		"top");										//"top" || "middle" || "bottom"
DataListElement.StyleDefault.setStyle("LayoutHorizontalAlign", 		"left");									//"left" || "center" || "right"
DataListElement.StyleDefault.setStyle("LayoutGap", 					0);											//number

DataListElement.StyleDefault.setStyle("ItemLabelFunction", 			DataListElement.DefaultItemLabelFunction);	// function (data) { return "" }

DataListElement.StyleDefault.setStyle("ListItemClass", 				DataRendererLabelElement); 					// Element constructor()
DataListElement.StyleDefault.setStyle("ListItemStyle", 				null);										// StyleDefinition

DataListElement.StyleDefault.setStyle("Selectable", 				true);										// true || false

DataListElement.StyleDefault.setStyle("ScrollBarDisplay", 			"auto");									// "on" || "off" || "auto"
DataListElement.StyleDefault.setStyle("ScrollBarPlacement", 		"right");									// "top" || "bottom" / "left || "right"
DataListElement.StyleDefault.setStyle("ScrollBarStyle", 			null);										// StyleDefinition


/////////DataRenderer Proxy Map/////////////////////////////

//Proxy map for styles we want to pass to the DataRenderer.
DataListElement._DataRendererProxyMap = Object.create(null);

DataListElement._DataRendererProxyMap.Selectable = 				true;
DataListElement._DataRendererProxyMap._Arbitrary = 				true;


/////////////Public///////////////////////////////

/**
 * @function setSelectedIndex
 * Sets the selected collection index/item.
 * 
 * @param index int
 * The collection index to be selected.
 * 
 * @returns bool
 * Returns true if the selection changed.
 */	
DataListElement.prototype.setSelectedIndex = 
	function (index)
	{
		if (this._selectedIndex == index)
			return false;
		
		if (index > this._listCollection.length -1)
			return false;
		
		if (index < -1)
			index = -1;
		
		var oldIndex = this._selectedIndex;
		
		this._selectedIndex = index;
		this._selectedItem = this._listCollection.getItemAt(index);
		
		//Update renderer data.
		if (this._contentPane._children.length > 0)
		{
			var firstIndex = this._contentPane._children[0]._listData._itemIndex;
			var lastIndex = this._contentPane._children[this._contentPane._children.length - 1]._listData._itemIndex;
			
			if (index != null && index >= firstIndex && index <= lastIndex)
				this._contentPane._children[index - firstIndex]._setListSelected(true);
			if (oldIndex != null && oldIndex >= firstIndex && oldIndex <= lastIndex)
				this._contentPane._children[oldIndex - firstIndex]._setListSelected(false);
		}
		
		return true;
	};

/**
 * @function getSelectedIndex
 * Gets the selected collection index. 
 * 
 * @returns int
 * The selected collection index or -1 if none selected.
 */		
DataListElement.prototype.getSelectedIndex = 
	function ()
	{
		return this._selectedIndex;
	};
	
/**
 * @function setSelectedItem
 * Sets the selected collection item/index.
 * 
 * @param item Object
 * The collection item to be selected.
 */	
DataListElement.prototype.setSelectedItem = 
	function (item)
	{
		var index = this._listCollection.getItemIndex(item);
		this.setSelectedIndex(index);
	};
	
/**
 * @function getSelectedItem
 * Gets the selected collection item. 
 * 
 * @returns Object
 * The selected collection item or null if none selected.
 */		
DataListElement.prototype.getSelectedItem = 
	function ()
	{
		return this._selectedItem;
	};
	
/**
 * @function setScrollIndex
 * Sets the collection item index to scroll too. 
 * 
 * @param scrollIndex int
 * Collection item index.
 */	
DataListElement.prototype.setScrollIndex = 
	function (scrollIndex)
	{
		scrollIndex = CanvasElement.roundToPrecision(scrollIndex, 3);
	
		this._invalidateLayout();
		
		if (scrollIndex >= this._listCollection.getLength())
			scrollIndex = this._listCollection.getLength() - 1;
		if (scrollIndex < 0)
			scrollIndex = 0;		
		
		this._scrollIndex = scrollIndex;
		
		var itemIndex = Math.floor(scrollIndex);
		var currentIndex = this._contentPane._children[0]._listData._itemIndex;
		
		var renderer = null;
		var rendererUpdatedCount = 0;
		
		if (itemIndex < currentIndex)
		{
			while (rendererUpdatedCount < this._contentPane._children.length)
			{
				//If current renderer item index matches expected, we're done.
				if (this._contentPane._children[rendererUpdatedCount]._listData._itemIndex == itemIndex + rendererUpdatedCount)
					break;
				
				//Last renderer
				renderer = this._contentPane._children[this._contentPane._children.length - 1];
				
				//Move to top
				this._contentPane._setChildIndex(renderer, rendererUpdatedCount);
				this._updateRendererData(renderer, itemIndex + rendererUpdatedCount);
				
				rendererUpdatedCount++;
			}
		}
		else if (itemIndex > currentIndex)
		{
			while (rendererUpdatedCount < this._contentPane._children.length)
			{
				//If current renderer item index matches expected, we're done.
				if (this._contentPane._children[0]._listData._itemIndex == itemIndex)
					break;
				
				//First renderer
				renderer = this._contentPane._children[0];
				
				//Make sure we have data available
				if (itemIndex + (this._contentPane._children.length - 1) - rendererUpdatedCount < this._listCollection.getLength())
				{
					//Move to bottom
					this._contentPane._setChildIndex(renderer, (this._contentPane._children.length - 1) - rendererUpdatedCount);
					this._updateRendererData(renderer, itemIndex + (this._contentPane._children.length - 1) - rendererUpdatedCount);
					rendererUpdatedCount++;
				}
				else //No data available, just remove the element
				{
					this._contentPane._removeChildAt(0);
				}
			}
		}
	};

/**
 * @function setListCollection
 * Sets the DataLists's associated ListCollection to generate DataRenderers.
 * 
 * @param listCollection ListCollection
 * The ListCollection to be used as the data-provider.
 */	
DataListElement.prototype.setListCollection = 
	function (listCollection)
	{
		if (this._listCollection == listCollection)
			return;
	
		if (this._manager == null)
		{
			this._listCollection = listCollection;
		}
		else
		{
			if (this._listCollection != null)
				this._listCollection.removeEventListener("collectionchanged", this._onDataListCollectionChangedInstance);
			
			this._listCollection = listCollection;
			
			if (this._listCollection != null)
				this._listCollection.addEventListener("collectionchanged", this._onDataListCollectionChangedInstance);
		}
		
		//Fix selected index/item
		if (this._listCollection == null)
		{
			this._selectedIndex = -1;
			this._selectedItem = null;
		}
		else
		{
			if (this._selectedItem != null)
			{
				this._selectedIndex = this._listCollection.getItemIndex(this._selectedItem);
				
				if (this._selectedIndex == -1)
					this._selectedItem = null;
			}
		}
		
		this._resetRenderersListData();
		this._invalidateLayout();
	};
	
/**
 * @function getListCollection
 * Gets the DataLists's associated ListCollection. 
 * 
 * @returns ListCollection
 * The associated ListCollection or null if none assigned.
 */		
DataListElement.prototype.getListCollection = 
	function ()
	{
		return this._listCollection;
	};	
	
///////////Internal//////////////////////////////
	
/**
 * @function _getContentSize
 * Gets the content size of the DataList. This is only accurate after the DataList
 * has finished its layout phase. Currently only used by the Dropdown to fix the
 * vertical height of the drop down pop-up list when there are too few items.
 * 
 * @returns Number
 * Content size in pixels of the DataListElement. Only valid after layout phase completed.
 */	
//Helpers function (currently only used by dropdown) ///
DataListElement.prototype._getContentSize = 
	function ()
	{
		var paddingSize = this._getPaddingSize();
	
		if (this.getStyle("LayoutDirection") == "vertical")
			return this._contentSize + paddingSize.height;
		else //if (this.getStyle("LayoutDirection") == "horizontal")
			return this._contentSize + paddingSize.width;
	};

//@private
DataListElement.prototype._onContentPaneMeasureComplete = 
	function (event)
	{
		this._invalidateMeasure();
		this._invalidateLayout();
	};
	
/**
 * @function _getNumRenderers
 * Gets the number of DataRenderers that are currently being rendered.
 * 
 * @returns int
 * the number of DataRenderers that are currently being rendered.
 */	
DataListElement.prototype._getNumRenderers = 
	function ()
	{
		return this._contentPane._children.length;
	};	
	
/**
 * @function _onDataListMouseWheelEvent
 * Event handler for the DataList "wheel" event. Starts the scroll bar tween.
 * 
 * @param elementMouseWheelEvent ElementMouseWheelEvent
 * The ElementMouseWheelEvent to process.
 */		
DataListElement.prototype._onDataListMouseWheelEvent = 
	function (elementMouseWheelEvent)
	{
		if (elementMouseWheelEvent.getDefaultPrevented() == true)
			return;
	
		//No renderers or event prevented.
		if (this._contentPane._children.length == 0 || this._listCollection.getLength() == 0)
			return;
	
		var delta = 0;
		var listDirection = this.getStyle("LayoutDirection");
		
		var minScrolled = false;
		var maxScrolled = false;
		
		var firstRenderer = this._contentPane._children[0];
		var lastRenderer = this._contentPane._children[this._contentPane._children.length - 1];
		
		if (listDirection == "horizontal")
		{
			delta = elementMouseWheelEvent.getDeltaX();
			
			if (delta == 0)
				return;
			
			if (firstRenderer._listData._itemIndex == 0 && 
				firstRenderer._x >= 0)
			{
				minScrolled = true;
			}
			
			if (minScrolled == true && delta < 0)
				return;
			
			if (lastRenderer._listData._itemIndex == this._listCollection.getLength() - 1 && 
				lastRenderer._x <= this._contentPane._width - lastRenderer._width)
			{
				maxScrolled = true;
			}
			
			if (maxScrolled == true && delta > 0)
				return;
		}
		else //if (listDirection == "vertical")
		{
			delta = elementMouseWheelEvent.getDeltaY();
			
			if (delta == 0)
				return;
			
			if (firstRenderer._listData._itemIndex == 0 && 
				firstRenderer._y >= 0)
			{
				minScrolled = true;
			}
			
			if (minScrolled == true && delta < 0)
				return;
			
			if (lastRenderer._listData._itemIndex == this._listCollection.getLength() - 1 && 
				lastRenderer._y <= this._contentPane._height - lastRenderer._height)
			{
				maxScrolled = true;
			}
			
			if (maxScrolled == true && delta > 0)
				return;
		}
		
		if (this._scrollBar != null)
		{
			var tweeningTo = this._scrollBar.getTweenToValue();
			if (tweeningTo == null)
				tweeningTo = this._scrollIndex;
			
			this._scrollBar.startScrollTween(tweeningTo + delta);
		}
		else
			this.setScrollIndex(this._scrollIndex + delta);
		
		//We've consumed the wheel event, don't want parents double scrolling.
		elementMouseWheelEvent.preventDefault();
	};	

/**
 * @function _onDataListScrollBarChanged
 * Event handler for the scroll bar "changed" event. Updates DataRenderers.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
DataListElement.prototype._onDataListScrollBarChanged = 
	function (elementEvent)
	{
		//Handle rounding errors
		var scrollValue = CanvasElement.roundToPrecision(this._scrollBar.getScrollValue(), 3);
		var scrollPageSize = CanvasElement.roundToPrecision(this._scrollBar.getScrollPageSize(), 3);
		var scrollViewSize = CanvasElement.roundToPrecision(this._scrollBar.getScrollViewSize(), 3);
		
		//Fix for issue where last renderer is larger than first, resulting in exponential adjustments 
		//due to view size shrinking / scroll range increasing at the same time as scroll. We check if the
		//scroll bar hit the end and increment the Lists scroll index rather than taking the scroll bar value.
		if (scrollValue == scrollPageSize - scrollViewSize && scrollValue - this._scrollIndex < 1)
			scrollValue = this._scrollIndex + 1;
	
		this.setScrollIndex(scrollValue);
	};
	
/**
 * @function _onDataListCollectionChanged
 * Event handler for the ListCollection "collectionchanged" event. Updates DataRenderers.
 * 
 * @param collectionChangedEvent CollectionChangedEvent
 * The CollectionChangedEvent to process.
 */		
DataListElement.prototype._onDataListCollectionChanged = 
	function (collectionChangedEvent)
	{
		var type = collectionChangedEvent.getKind();
		var index = collectionChangedEvent.getIndex();
	
		//Always invalidate layout (we need to adjust the scroll bar)
		this._invalidateLayout();
		
		if (this._contentPane._children.length == 0)
			return;
	
		//Reset all renderers (collection was cleared, or swapped)
		if (type == "reset")
		{
			//Fix selected index/item
			if (this._selectedItem != null)
			{
				this._selectedIndex = this._listCollection.getItemIndex(this._selectedItem);
				
				if (this._selectedIndex == -1)
					this._selectedItem = null;
			}
			
			this._resetRenderersListData();
		}
		else
		{
			var firstIndex = 0;
			var lastIndex = 0;
			
			if (this._contentPane._children.length > 0)
			{
				firstIndex = this._contentPane._children[0]._listData._itemIndex;
				lastIndex = this._contentPane._children[this._contentPane._children.length - 1]._listData._itemIndex;
			}
			
			if (this._selectedIndex == index && type == "remove") //We removed the selected item.
			{
				this._selectedIndex = -1;
				this._selectedItem = null;
			}
			
			if (index <= lastIndex && (type == "add" || type == "remove"))
			{
				//Adjust selected index
				if (index <= this._selectedIndex)
				{
					if (type == "add")
						this._selectedIndex++;
					else // if (type == "remove)
						this._selectedIndex--;
				}
				
				if (index < firstIndex)
				{
					var newIndex;
					var indexAdjust;
					
					//Fix scroll/item indexes (we added/removed an item on top thats out of the view)
					//Dont invalidate, only the index has changed, not the data, we dont want renderers shuffling around.
					if (type == "add")
						indexAdjust = 1;
					else // "remove"
						indexAdjust = -1;
					
					this._scrollIndex = this._scrollIndex + indexAdjust;
					
					//Adjust all indexes
					for (var i = 0; i < this._contentPane._children.length; i++)
					{
						newIndex = this._contentPane._children[i]._listData._itemIndex + indexAdjust;
						this._updateRendererData(this._contentPane._children[i], newIndex);
					}
				}
				else //Visible renderers changed
				{
					if (type == "add")
					{
						var newRenderer = this._createRenderer(index);
						
						//Push in a new renderer, layout will deal with purging excess if necessary
						this._contentPane._addChildAt(newRenderer, index - firstIndex);
						index++;
					}
					else // if (type == "remove")
					{
						//Pop the removed renderer, layout will deal with adding more if necessary
						this._contentPane._removeChildAt(index - firstIndex);
					}
					
					//Adjust downstream indexes.
					for (var i = index - firstIndex; i < this._contentPane._children.length; i++)
					{
						this._updateRendererData(this._contentPane._children[i], index);
						index++;
					}
				}
			}
			else if (type == "update" && index >= firstIndex && index <= lastIndex)
				this._updateRendererData(this._contentPane._children[index - firstIndex], index);
		}
	};
	
//@Override	
DataListElement.prototype._onCanvasElementAdded = 
	function (addedRemovedEvent)
	{
		DataListElement.base.prototype._onCanvasElementAdded.call(this, addedRemovedEvent);
	
		if (this._listCollection != null)
			this._listCollection.addEventListener("collectionchanged", this._onDataListCollectionChangedInstance);
	};

//@Override	
DataListElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		DataListElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		if (this._listCollection != null)
			this._listCollection.removeEventListener("collectionchanged", this._onDataListCollectionChangedInstance);
	};	

/**
 * @function _invalidateListRenderersLayout
 * Calls _invalidateLayout() on all DataRenderers.
 */	
DataListElement.prototype._invalidateListRenderersLayout = 
	function ()
	{
		for (var i = 0; i < this._contentPane._children.length; i++)
			this._contentPane._children[i]._invalidateLayout();
	};
	
/**
 * @function _invalidateListRenderersMeasure
 * Calls _invalidateMeasure() on all DataRenderers.
 */		
DataListElement.prototype._invalidateListRenderersMeasure = 
	function ()
	{
		for (var i = 0; i < this._contentPane._children.length; i++)
			this._contentPane._children[i]._invalidateMeasure();
	};
	
/**
 * @function _resetRenderersListData
 * Updates list data on all renderers, such as when collection is changed.
 */		
DataListElement.prototype._resetRenderersListData = 
	function ()
	{
		var itemIndex = Math.floor(this._scrollIndex);

		for (var i = 0; i < this._contentPane._children.length; i++)
		{
			if (this._listCollection.getLength() > itemIndex + i)
			{
				//Update list data
				renderer = this._contentPane._children[i];
				this._updateRendererData(renderer, itemIndex + i);
			}
			else
			{
				//No more data, purge the rest of the renderers.
				while (this._contentPane._children.length > i)
					this._contentPane._removeChildAt(i);
				
				break;
			}
		}
	};
	
//@Override
DataListElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DataListElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("ListItemClass" in stylesMap)
		{
			//Check if class changed
			if (this._contentPane._children.length > 0 && 
				this._contentPane._children[0].constructor != this.getStyle("ListItemClass"))
			{
				//Purge all renderers
				while (this._contentPane._children.length > 0)
					this._contentPane._removeChildAt(0);
				
				this._invalidateLayout();
			}
		}
		
		if ("ListItemStyle" in stylesMap)
		{
			for (var i = 0; i < this._contentPane._children.length; i++)
				this._applySubStylesToElement("ListItemStyle", this._contentPane._children[i]);
			
			this._invalidateLayout();
		}
		
		if ("LayoutDirection" in stylesMap)
		{
			this._invalidateMeasure();
			this._invalidateLayout();
		}
		else if ("ScrollBarPlacement" in stylesMap || 
				"ScrollBarDisplay" in stylesMap ||  
				"LayoutGap" in stylesMap ||
				"LayoutHorizontalAlign" in stylesMap ||
				"LayoutVerticalAlign" in stylesMap)
		{
			this._invalidateLayout();
		}
		
		if ("ScrollBarStyle" in stylesMap && this._scrollBar != null)
			this._applySubStylesToElement("ScrollBarStyle", this._scrollBar);
		
		if ("ItemLabelFunction" in stylesMap)
			this._resetRenderersListData();
	};

/**
 * @function _onDataListRendererClick
 * Event handler for the DataRenderer "click" event. Updates selected index/item and dispatches "listitemclick" and "changed" events.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
DataListElement.prototype._onDataListRendererClick = 
	function (elementMouseEvent)
	{
		var itemIndex = elementMouseEvent.getCurrentTarget()._listData._itemIndex;
		var itemData = elementMouseEvent.getCurrentTarget()._itemData;
		
		var dispatchChanged = false;
		
		var selectable = elementMouseEvent.getCurrentTarget().getStyle("Selectable");
		
		//Update selected index
		if (this.getStyle("Selectable") == true && 
			(selectable === undefined || selectable == true))
		{
			if (this.setSelectedIndex(itemIndex) == true)
				dispatchChanged = true;
		}
		
		this.dispatchEvent(new ElementListItemClickEvent(itemData, itemIndex));
		
		//Only dispatch changed if selected index changed
		if (dispatchChanged == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};
	
/**
 * @function _createRenderer
 * Generates a DataRenderer based on the ListItemClass style.
 * 
 * @param itemIndex int
 * Collection index associated with the DataRenderer.
 * 
 * @returns CanvasElement
 * The new DataRenderer instance.
 */	
DataListElement.prototype._createRenderer = 
	function (itemIndex)
	{
		var newRenderer = new (this.getStyle("ListItemClass"))();
		newRenderer._setStyleProxy(new StyleProxy(this, DataListElement._DataRendererProxyMap));
		
		this._applySubStylesToElement("ListItemStyle", newRenderer);
		this._updateRendererData(newRenderer, itemIndex);
		
		newRenderer.addEventListener("click", this._onDataListRendererClickInstance);
		
		return newRenderer;
	};

/**
 * @function _updateRendererData
 * Updates the DataRenderer list data and selected state.
 * 
 * @param renderer CanvasElement
 * DataRenderer to update.
 * 
 * @param itemIndex int
 * Collection index to associate with the DataRenderer.
 */	
DataListElement.prototype._updateRendererData = 
	function (renderer, itemIndex)
	{
		var listData = null;
		
		//Optimize, dont create new data if already exists
		if (renderer._listData != null)
		{
			listData = renderer._listData;
			listData._parentList = this;
			listData._itemIndex = itemIndex;
		}
		else
			listData = new DataListData(this, itemIndex);
	
		//Always call the function even if data has not changed, this indicates to the
		//renderer to inspect its parent related data and it may make changes even if
		//this data is the same. An example is changes to a DataGrid's columns.
		renderer._setListData(
			listData,
			this._listCollection.getItemAt(itemIndex));
		
		if (this._selectedIndex == itemIndex)
			renderer._setListSelected(true);
		else
			renderer._setListSelected(false);
	};
	
//@override
DataListElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		//TODO: Sample text widths if label function is set.
		this._setMeasuredSize(padWidth + 16, padHeight + 16);
	};	
	
//@override	
DataListElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		DataListElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		var availableSize = h;
		var listItem = null;
		var i;
		
		var listDirection = this.getStyle("LayoutDirection");
		var itemIndex = Math.floor(this._scrollIndex);
		
		var collectionLength = 0;
		if (this._listCollection != null)
			collectionLength = this._listCollection.getLength();
		
		if (collectionLength == 0)
		{
			itemIndex = 0;
			this._scrollIndex = itemIndex;
		}
		else if (itemIndex > collectionLength -1)
		{
			itemIndex = collectionLength -1;
			this._scrollIndex = itemIndex;
		}
		
		var clipFirst = 0;
		var clipLast = 0;
		
		this._contentSize = 0;
		var itemSize = 0;
		
		//Measure existing content & clipping amounts.
		for (i = 0; i < this._contentPane._children.length; i++)
		{
			listItem = this._contentPane._children[i];
			
			if (listDirection == "horizontal")
				itemSize = listItem._getStyledOrMeasuredWidth();
			else // if (listDirection == "vertical")
				itemSize = listItem._getStyledOrMeasuredHeight();
				
			this._contentSize += itemSize;
			
			if (listItem._listData._itemIndex <= itemIndex)
			{
				if (listItem._listData._itemIndex < itemIndex)
					clipFirst += itemSize;
				else
					clipFirst += itemSize * (this._scrollIndex - itemIndex);
			}
			
			if (this._contentSize - clipFirst >= availableSize)
			{
				clipLast = (this._contentSize - clipFirst - availableSize);
				
				//Purge Excess renderers.
				while (this._contentPane._children.length - 1 > i)
					this._contentPane._removeChildAt(this._contentPane._children.length - 1);
			}
		}
		
		//Adjust scroll index due to new renderer added on top.
		//Happens when we're max scrolled and DataList size increases.
		if (this._contentPane._children.length > 0 && 
			this._contentPane._children[0]._listData._itemIndex < itemIndex)
		{
			clipFirst += clipLast;
			clipLast = 0; //No clipping last item (scrolled to bottom)
			
			itemIndex = this._contentPane._children[0]._listData._itemIndex;
			
			//Fix scroll index
			if (listDirection == "horizontal")
			{
				itemSize = this._contentPane._children[0]._getStyledOrMeasuredWidth();
				
				if (itemSize != 0)
					this._scrollIndex = itemIndex + (clipFirst / itemSize);
				else
					this._scrollIndex = itemIndex;
			}
			else // if (listDirection == "vertical")
			{
				itemSize = this._contentPane._children[0]._getStyledOrMeasuredHeight();
				
				if (itemSize != 0)
					this._scrollIndex = itemIndex + (clipFirst / itemSize);
				else
					this._scrollIndex = itemIndex;
			}
			
			//Handle rounding errors
			this._scrollIndex = CanvasElement.roundToPrecision(this._scrollIndex, 3);
		}
		
		//Extra space - need another renderer or scroll shift
		if (this._contentSize - clipFirst - clipLast < availableSize)
		{
			if (itemIndex + this._contentPane._children.length < collectionLength)
			{//Create a new renderer and put it on bottom.
				
				var newRenderer = this._createRenderer(itemIndex + this._contentPane._children.length);
				this._contentPane._addChild(newRenderer);
				
				//Wait for the new renderer to measure.
				this._invalidateLayout();
				return;
			}
			else
			{//Add before (or shift up scroll position)
				
				var excessSize = availableSize - (this._contentSize - clipFirst - clipLast);
				
				if (clipFirst >= excessSize) 
				{//We have enough clipping to cover the gap, un-clip and adjust scroll index
					
					clipFirst -= excessSize;
					
					if (listDirection == "horizontal")
					{
						itemSize = this._contentPane._children[0]._getStyledOrMeasuredWidth();
						
						if (itemSize != 0)
							this._scrollIndex = itemIndex + (clipFirst / itemSize);
						else
							this._scrollIndex = itemIndex;
					}
					else // if (listDirection == "vertical")
					{
						itemSize = this._contentPane._children[0]._getStyledOrMeasuredHeight();
						
						if (itemSize != 0)
							this._scrollIndex = itemIndex + (clipFirst / this._contentPane._children[0]._getStyledOrMeasuredHeight());
						else
							this._scrollIndex = itemIndex;
					}
					
					//Handle rounding errors
					this._scrollIndex = CanvasElement.roundToPrecision(this._scrollIndex, 3);
				}
				else if (clipFirst > 0 && collectionLength == this._contentPane._children.length)
				{//We dont have enough clipping, but we're out of data (cannot make new renderer)
					
					clipFirst = 0;
					this._scrollIndex = 0;
				}
				else if (collectionLength > this._contentPane._children.length)
				{//Create a new renderer and put it on top
					
					var newRenderer = this._createRenderer(itemIndex - 1);
					this._contentPane._addChildAt(newRenderer, 0);
					
					//Wait for the new renderer to measure.
					//Re-invalidate ourself, (content pane doesnt measure so wont do it for us).
					this._invalidateLayout();
					return;
				}
			}
		}
		
		var needsScrollBar = false;
		var scrollDisplay = this.getStyle("ScrollBarDisplay");
		
		if (scrollDisplay == "on" || 
			(scrollDisplay == "auto" && availableSize > 0 && (this._contentSize > availableSize || this._contentPane._children.length < collectionLength)))
		{
			needsScrollBar = true;
		}
		
		//Create ScrollBar
		if (needsScrollBar == true && this._scrollBar == null)
		{
			this._scrollBar = new ScrollBarElement();
			
			this._applySubStylesToElement("ScrollBarStyle", this._scrollBar);
			
			this._scrollBar.setScrollLineSize(1);
			
			this._scrollBar.addEventListener("changed", this._onDataListScrollBarChangedInstance);
			this._addChild(this._scrollBar);
			
			//Wait for measure.
			return;
		}
		
		//Destroy ScrollBar
		if (needsScrollBar == false && this._scrollBar != null)
		{			
			this._removeChild(this._scrollBar);
			this._scrollBar = null;
			
			//Wait for measure
			return;
		}
		
		//Size / Position the scroll bar and content pane.
		if (this._scrollBar != null)
		{
			this._scrollBar.setStyle("LayoutDirection", listDirection);
			
			var scrollBarPlacement = this.getStyle("ScrollBarPlacement");
			
			if (listDirection == "horizontal")
			{
				this._scrollBar._setActualSize(w, this._scrollBar._getStyledOrMeasuredHeight());
				this._contentPane._setActualSize(w, h - this._scrollBar._height);
				
				if (scrollBarPlacement == "top" || scrollBarPlacement == "left")
				{
					this._contentPane._setActualPosition(x, y + this._scrollBar._height);
					this._scrollBar._setActualPosition(x, y);
				}
				else //if (scrollBarPlacement == "bottom" || scrollBarPlacement == "right")
				{
					this._contentPane._setActualPosition(x, y);
					this._scrollBar._setActualPosition(x, y + this._contentPane._height);
				}
			}
			else // if (listDirection == "vertical")
			{
				this._scrollBar._setActualSize(this._scrollBar._getStyledOrMeasuredWidth(), h);
				this._contentPane._setActualSize(w - this._scrollBar._width, h);
				
				if (scrollBarPlacement == "top" || scrollBarPlacement == "left")
				{
					this._scrollBar._setActualPosition(x, y);
					this._contentPane._setActualPosition(x + this._scrollBar._width, y);
				}
				else //if (scrollBarPlacement == "bottom" || scrollBarPlacement == "right")
				{
					this._scrollBar._setActualPosition(x + this._contentPane._width, y);
					this._contentPane._setActualPosition(x, y);
				}
			}
		}
		else
		{
			this._contentPane._setActualPosition(x, y);
			this._contentPane._setActualSize(w, h);
		}

		//Layout content pane children.
		var currentPosition = clipFirst * -1;
		if (this._contentSize < availableSize)
		{
			var listAlign = null;
			if (listDirection == "horizontal")
				listAlign = this.getStyle("LayoutHorizontalAlign");
			else //if (listDirection == "vertical")
				listAlign = this.getStyle("LayoutVerticalAlign");
			
			if (listAlign == "top" || listAlign == "left")
				currentPosition = 0;
			else if (listAlign == "center" || listAlign == "middle")
				currentPosition = (availableSize / 2) - (this._contentSize / 2);
			else //if (listAlign == "bottom" || listAlign == "right")
				currentPosition = availableSize - this._contentSize;
		}
		
		for (i = 0; i < this._contentPane._children.length; i++)
		{
			listItem = this._contentPane._children[i];
			
			if (listDirection == "horizontal")
			{
				listItem._setActualSize(listItem._getStyledOrMeasuredWidth(), this._contentPane._height);
				listItem._setActualPosition(currentPosition, 0);
				
				currentPosition += listItem._width;
			}
			else // if (listDirection == "vertical")
			{
				listItem._setActualSize(this._contentPane._width, listItem._getStyledOrMeasuredHeight());
				listItem._setActualPosition(0, currentPosition);
				
				currentPosition += listItem._height;
			}
		}
		
		//Adjust scroll bar parameters.
		if (this._scrollBar != null)
		{
			var viewSize = this._contentPane._children.length;
			
			if (this._contentPane._children.length)
			{
				if (listDirection == "horizontal")
				{
					if (this._contentPane._children[0]._width != 0)
						viewSize -= clipFirst / this._contentPane._children[0]._width;
					
					if (this._contentPane._children[this._contentPane._children.length - 1]._width != 0)
						viewSize -= clipLast / this._contentPane._children[this._contentPane._children.length - 1]._width;
				}
				else // if (listDirection == "vertical")
				{
					if (this._contentPane._children[0]._height != 0)
						viewSize -= clipFirst / this._contentPane._children[0]._height;
					
					if (this._contentPane._children[this._contentPane._children.length - 1]._height != 0)
						viewSize -= clipLast / this._contentPane._children[this._contentPane._children.length - 1]._height;
				}
			}
			
			this._scrollBar.setScrollPageSize(collectionLength);
			this._scrollBar.setScrollViewSize(viewSize);
			
			if (CanvasElement.roundToPrecision(this._scrollBar.getScrollValue(), 3) != this._scrollIndex)
			{
				this._scrollBar.endScrollTween();
				this._scrollBar.setScrollValue(this._scrollIndex);
			}
		}
	};
	
	


/**
 * @depends DataRendererBaseElement.js
 */

///////////////////////////////////////////////////////////////////////
////////////////////DataGridItemRendererBase////////////////////////

/**
 * @class DataGridItemRendererBase
 * @inherits DataRendererBaseElement
 * 
 * Abstract base class for DataGrid row item rendering. Any CanvasElement
 * can be a data renderer. This class is just here for convenience as it
 * implements some commonly used functionality if you wish to subclass it. 
 * 
 * Adds skin states and styles for "up", "alt", "over", and "selected" states. 
 * 
 * @constructor DataGridItemRendererBase 
 * Creates new DataGridItemRendererBase instance.
 */
function DataGridItemRendererBase()
{
	DataGridItemRendererBase.base.prototype.constructor.call(this);
}
	
//Inherit from DataRendererBaseElement
DataGridItemRendererBase.prototype = Object.create(DataRendererBaseElement.prototype);
DataGridItemRendererBase.prototype.constructor = DataGridItemRendererBase;
DataGridItemRendererBase.base = DataRendererBaseElement;

DataGridItemRendererBase.UpSkinStyleDefault = new StyleDefinition();
DataGridItemRendererBase.UpSkinStyleDefault.setStyle("BackgroundFill", 				"#FFFFFF");

DataGridItemRendererBase.AltSkinStyleDefault = new StyleDefinition();
DataGridItemRendererBase.AltSkinStyleDefault.setStyle("BackgroundFill", 			"#F0F0F0");

DataGridItemRendererBase.StyleDefault = new StyleDefinition();
DataGridItemRendererBase.StyleDefault.setStyle("UpSkinStyle", DataGridItemRendererBase.UpSkinStyleDefault);
DataGridItemRendererBase.StyleDefault.setStyle("AltSkinStyle", DataGridItemRendererBase.AltSkinStyleDefault);


/////////////Internal///////////////////////////

//@override
DataGridItemRendererBase.prototype._updateState = 
	function ()
	{
		DataGridItemRendererBase.base.prototype._updateState.call(this);
		
		if (this._listSelected != null && this._listSelected.selected == true)
			newState = "selected";
		else if (this._listSelected != null && this._listSelected.highlight == true)
			newState = "over";
		else
		{
			if (this._listData == null || this._listData._itemIndex % 2 == 0)
				newState = "up";
			else
				newState = "alt";
		}
	
		this.setStyle("SkinState", newState);
	};






/**
 * @depends DataGridItemRendererBase.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////DataGridLabelItemRenderer/////////////////////////	
	
/**
 * @class DataGridLabelItemRenderer
 * @inherits DataGridItemRendererBase
 * 
 * DataGrid ItemRenderer for a basic label. Updates label text via 
 * DataGridColumnDefiniton RowItemLabelFunction.
 * 
 * @constructor DataGridLabelItemRenderer 
 * Creates new DataGridLabelItemRenderer instance.
 */
function DataGridLabelItemRenderer()
{
	DataGridLabelItemRenderer.base.prototype.constructor.call(this);
	
	this._labelElement = new LabelElement();
	this._labelElement.setStyle("Padding", 0); //Wipe out default padding (no doubly padding, only this elements padding is necessary)
	
	this._addChild(this._labelElement);
}

//Inherit from LabelElement
DataGridLabelItemRenderer.prototype = Object.create(DataGridItemRendererBase.prototype);
DataGridLabelItemRenderer.prototype.constructor = DataGridLabelItemRenderer;
DataGridLabelItemRenderer.base = DataGridItemRendererBase;


/////////////Style Types/////////////////////////

DataGridLabelItemRenderer._StyleTypes = Object.create(null);

/**
 * @style UpTextColor String
 * 
 * Hex color value to be used for the label when in the "up" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataGridLabelItemRenderer._StyleTypes.UpTextColor = 				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style AltTextColor String
 * 
 * Hex color value to be used for the label when in the "alt" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataGridLabelItemRenderer._StyleTypes.AltTextColor = 				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style OverTextColor String
 * 
 * Hex color value to be used for the label when in the "over" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataGridLabelItemRenderer._StyleTypes.OverTextColor = 				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style SelectedTextColor String
 * 
 * Hex color value to be used for the label when in the "selected" state. Format like "#FF0000" (red).
 * This will override the TextColor style of equal priority.
 */
DataGridLabelItemRenderer._StyleTypes.SelectedTextColor = 			StyleableBase.EStyleType.NORMAL;		//"#000000"


////////////Default Styles///////////////////////

DataGridLabelItemRenderer.StyleDefault = new StyleDefinition();

DataGridLabelItemRenderer.StyleDefault.setStyle("PaddingTop", 				4);
DataGridLabelItemRenderer.StyleDefault.setStyle("PaddingBottom", 			4);
DataGridLabelItemRenderer.StyleDefault.setStyle("PaddingLeft", 				4);
DataGridLabelItemRenderer.StyleDefault.setStyle("PaddingRight", 			4);

DataGridLabelItemRenderer.StyleDefault.setStyle("UpTextColor", 				"#000000");
DataGridLabelItemRenderer.StyleDefault.setStyle("AltTextColor", 			"#000000");
DataGridLabelItemRenderer.StyleDefault.setStyle("OverTextColor", 			"#000000");
DataGridLabelItemRenderer.StyleDefault.setStyle("SelectedTextColor", 		"#000000");


////////////Internal/////////////////////////////

//@Override
DataGridLabelItemRenderer.prototype._changeState = 
	function (state)
	{
		DataGridLabelItemRenderer.base.prototype._changeState.call(this, state);
		
		this._updateLabelTextColor();
	};
	
/**
 * @function _getTextColor
 * Gets the text color style for the supplied state.
 * 
 * @param state String
 * The current state.
 * 
 * @returns String
 * Text color style for the supplied state.
 */	
DataGridLabelItemRenderer.prototype._getTextColor = 
	function (state)
	{
		var stateTextColor = null;
		
		if (state == "up")
			stateTextColor = this.getStyleData("UpTextColor");
		else if (state == "alt")
			stateTextColor = this.getStyleData("AltTextColor");
		else if (state == "over")
			stateTextColor = this.getStyleData("OverTextColor");
		else if (state == "selected")
			stateTextColor = this.getStyleData("SelectedTextColor");
	
		var textColor = this.getStyleData("TextColor");
		
		//Shouldnt have null stateTextColor
		if (stateTextColor == null || textColor.comparePriority(stateTextColor) > 0) //Use textColor if higher priority
			return textColor.value;
		
		return stateTextColor.value;
	};

/**
 * @function _updateLabelTextColor
 * Updates the text color for the current state.
 */	
DataGridLabelItemRenderer.prototype._updateLabelTextColor = 
	function ()
	{
		var color = this._getTextColor(this._currentSkinState);
		if (color != null)
			this._labelElement.setStyle("TextColor", color);
	};
	
/**
 * @function _updateLabelTextColor
 * Updates the label text base on the DataGrid data and column RowItemLabelFunction.
 */		
DataGridLabelItemRenderer.prototype._updateLabelText = 
	function ()
	{
		if (this._itemData == null || this._listData == null)
			this._labelElement.setStyle("Text", "");
		else
		{
			var parentGrid = this._listData._parentGrid;
			var columnDefinition = parentGrid._gridColumns[this._listData._columnIndex];
			var labelFunction = columnDefinition.getStyle("RowItemLabelFunction");
			
			this._labelElement.setStyle("Text", labelFunction(this._itemData, this._listData._columnIndex));
		}
	};
	
//@override
DataGridLabelItemRenderer.prototype._setListData = 
	function (listData, itemData)
	{
		DataGridLabelItemRenderer.base.prototype._setListData.call(this, listData, itemData);
		
		this._updateLabelText();
	};

//@override
DataGridLabelItemRenderer.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DataGridLabelItemRenderer.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		this._updateLabelTextColor();
	};

//@override
DataGridLabelItemRenderer.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		this._setMeasuredSize(this._labelElement._getStyledOrMeasuredWidth() + padWidth, 
							this._labelElement._getStyledOrMeasuredHeight() + padHeight);
	};

//@override	
DataGridLabelItemRenderer.prototype._doLayout = 
	function (paddingMetrics)
	{
		DataGridLabelItemRenderer.base.prototype._doLayout.call(this, paddingMetrics);
		
		this._labelElement._setActualPosition(paddingMetrics.getX(), paddingMetrics.getY());
		this._labelElement._setActualSize(paddingMetrics.getWidth(), paddingMetrics.getHeight());
	};
	
	


/**
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////////DataGridDataRenderer////////////////////////////

/**
 * @class DataGridDataRenderer
 * @inherits CanvasElement
 * 
 * Default DataGrid ListItemClass used to render DataGrid rows. Renders
 * column items per the parent DataGrid's column definitions. 
 * 
 * @constructor DataGridDataRenderer 
 * Creates new DataGridDataRenderer instance.
 */

//Used to render the DataGrid rows. 
function DataGridDataRenderer()
{
	DataGridDataRenderer.base.prototype.constructor.call(this);
	
	//Use a containing element for the renderers so we dont interfere with our skins.
	this._itemRenderersContainer = new CanvasElement();
	this._addChild(this._itemRenderersContainer);
	
	var _self = this;
	
	this._onItemRenderersContainerMeasureCompleteInstance = 
		function (event)
		{
			_self.__onItemRenderersContainerMeasureComplete(event);
		};
	
	this._itemRenderersContainer.addEventListener("measurecomplete", this._onItemRenderersContainerMeasureCompleteInstance);
}
	
//Inherit from CanvasElement
DataGridDataRenderer.prototype = Object.create(CanvasElement.prototype);
DataGridDataRenderer.prototype.constructor = DataGridDataRenderer;
DataGridDataRenderer.base = CanvasElement;

//@private
DataGridDataRenderer.prototype.__onItemRenderersContainerMeasureComplete =
	function (event)
	{
		this._invalidateMeasure();
		this._invalidateLayout();
	};

//@Override
DataGridDataRenderer.prototype._setListData = 
	function (listData, itemData)
	{
		DataGridDataRenderer.base.prototype._setListData.call(this, listData, itemData);
		
		var renderer = null;
		for (var i = 0; i < listData._parentList._gridColumns.length; i++)
		{
			renderer = this._itemRenderersContainer._getChildAt(i);
			
			if (renderer == null)
			{
				renderer = listData._parentList._createRowItemRenderer(listData._itemIndex, i);
				this._itemRenderersContainer._addChildAt(renderer, i);
			}
			else
			{
				columnDef = listData._parentList._gridColumns[i];
				
				if (renderer.constructor != columnDef.getStyle("RowItemClass"))
				{ //Renderer Class changed
					
					this._itemRenderersContainer._removeChildAt(i);
					renderer = listData._parentList._createRowItemRenderer(listData._itemIndex, i);
					this._itemRenderersContainer._addChildAt(renderer, i);
				}
				else
				{ //Update DataGridData
					
					listData._parentList._updateRowItemRendererData(renderer, listData._itemIndex, i);
				}
			}
		}
		
		//Purge excess renderers.
		while (this._itemRenderersContainer._children.length > this._listData._parentList._gridColumns.length)
			this._itemRenderersContainer._removeChildAt(this._itemRenderersContainer._children.length - 1);
	};
	
//@override	
DataGridDataRenderer.prototype._setListSelected = 
	function (selectedData)
	{
		DataGridDataRenderer.base.prototype._setListSelected.call(this, selectedData);
		
		var columnData;
		var columnSelectionType;
		var columnSelectable;
		var columnHighlightable;
		var itemRenderer;
		var itemRendererSelectedData;
		
		var overColumn = null;
		if (selectedData.columnOverIndex >= 0 && selectedData.columnOverIndex < this._listData._parentList._gridColumns.length)
			overColumn = this._listData._parentList._gridColumns[selectedData.columnOverIndex];
		
		var overColumnSelectionType = null;
		if (overColumn != null)
			overColumnSelectionType = overColumn.getStyle("SelectionType");
		
		for (var i = 0; i < this._itemRenderersContainer._children.length; i++)
		{
			columnData = this._listData._parentList._gridColumns[i];
			columnSelectionType = columnData.getStyle("SelectionType");
			columnSelectable = columnData.getStyle("Selectable");
			columnHighlightable = columnData.getStyle("Highlightable");			
			itemRenderer = this._itemRenderersContainer._getChildAt(i);

			//Optimize, use existing data if available
			itemRendererSelectedData = itemRenderer._listSelected;
			if (itemRendererSelectedData == null)
				itemRendererSelectedData = {highlight:false, selected:false};
			else
			{
				itemRendererSelectedData.highlight = false;
				itemRendererSelectedData.selected = false;
			}
			
			if (columnSelectable == true)
			{
				if ((columnSelectionType == "row" && selectedData.rowIndex == this._listData._itemIndex && selectedData.columnIndex == -1) ||
					(columnSelectionType == "column" && i == selectedData.columnIndex) ||
					(columnSelectionType == "cell" && i == selectedData.columnIndex && this._listData._itemIndex == selectedData.rowIndex))
				{
					itemRendererSelectedData.selected = true;
				}
			}
			
			if (columnHighlightable == true)
			{
				if ((columnSelectionType == "row" && selectedData.rowOverIndex == this._listData._itemIndex && overColumnSelectionType == "row") ||
					(columnSelectionType == "column" && i == selectedData.columnOverIndex) ||
					(columnSelectionType == "cell" && i == selectedData.columnOverIndex && this._listData._itemIndex == selectedData.rowOverIndex))
				{
					itemRendererSelectedData.highlight = true;
				}
			}
			
			itemRenderer._setListSelected(itemRendererSelectedData);
		}
	};
	
//@Override
DataGridDataRenderer.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var measuredWidth = 0;
		var measuredHeight = 0;
		
		var childSize = 0;
		
		for (var i = 0; i < this._itemRenderersContainer._children.length; i++)
		{
			childSize = this._itemRenderersContainer._children[i]._getStyledOrMeasuredHeight();
			
			if (measuredHeight < childSize)
				measuredHeight = childSize;
			
			measuredWidth += this._itemRenderersContainer._children[i]._getStyledOrMeasuredWidth();
		}
	
		measuredWidth += padWidth;
		measuredHeight += padHeight;
		
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};
	
//@Override	
DataGridDataRenderer.prototype._doLayout = 
	function (paddingMetrics)
	{
		DataGridDataRenderer.base.prototype._doLayout.call(this, paddingMetrics);
		
		if (this._listData == null)
			return;
		
		this._itemRenderersContainer._setActualPosition(paddingMetrics.getX(), paddingMetrics.getY());
		this._itemRenderersContainer._setActualSize(paddingMetrics.getWidth(), paddingMetrics.getHeight());
		
		var parentGrid = this._listData._parentList;
		var rowItemRenderer = null;
		var currentPosition = 0;
		var columnSize = 0;
		
		var paddingSize = this._getPaddingSize();
		
		for (var i = 0; i < parentGrid._columnSizes.length; i++)
		{
			rowItemRenderer = this._itemRenderersContainer._children[i];
			columnSize = parentGrid._columnSizes[i];
			
			if (i == 0)
				columnSize -= paddingSize.paddingLeft;
			else if (i == parentGrid._columnSizes.length - 1) //Consume the rest available.
				columnSize = this._itemRenderersContainer._width - currentPosition;
			
			rowItemRenderer._setActualPosition(currentPosition, 0);
			rowItemRenderer._setActualSize(columnSize, this._itemRenderersContainer._height);
			
			currentPosition += columnSize;
		}
	};	
	
	


/**
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////////ViewportElement/////////////////////////////////	
	
/**
 * @class ViewportElement
 * @inherits CanvasElement
 * 
 * Viewport is a container that only supports one child element (usually another container).
 * When the child's content size is too large for the view area, the Viewport will optionally 
 * pop up scroll bars, otherwise the child element will assume the size of the ViewportElement.
 * 
 * This class needs more work. More styles are needed for controlling tween behavior and allowing
 * scrolling even if scroll bars are disabled.
 *  
 * 
 * @constructor ViewportElement 
 * Creates new ViewportElement instance.
 */
function ViewportElement()
{
	ViewportElement.base.prototype.constructor.call(this);
	
	this._viewElement = null;
	
	this._horizontalScrollBar = null;
	this._verticalScrollBar = null;
	
	this._viewPortContainer = new CanvasElement();
	this._viewPortContainer.setStyle("ClipContent", true);
	this._addChild(this._viewPortContainer);
	
	var _self = this;
	
	//Private event handler, need different instance for each Viewport, proxy to prototype.
	this._onViewportScrollBarChangeInstance =
		function (elementEvent)
		{
			_self._onViewportScrollBarChange(elementEvent);
		};
		
	this._onViewportMouseWheelEventInstance = 
		function (elementMouseWheelEvent)
		{
			_self._onViewportMouseWheelEvent(elementMouseWheelEvent);
		};
		
	this._onViewElementMeasureCompleteInstance = 
		function (event)
		{
			_self._onViewElementMeasureComplete(event);
		};
		
		
	this.addEventListener("wheel", this._onViewportMouseWheelEventInstance);
}

//Inherit from CanvasElement
ViewportElement.prototype = Object.create(CanvasElement.prototype);
ViewportElement.prototype.constructor = ViewportElement;
ViewportElement.base = CanvasElement;

/////////////Style Types////////////////////////////////////////////

ViewportElement._StyleTypes = Object.create(null);

/**
 * @style MeasureContentWidth boolean
 * When true, the viewport's measured width will use its content element's measured width. 
 * Use this when you want the viewport to expand its width when possible rather than scroll, 
 * causing scrolling to happen on a parent viewport.
 */
ViewportElement._StyleTypes.MeasureContentWidth = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style MeasureContentHeight boolean
 * When true, the viewport's measured height will use its content element's measured height.
 * Use this when you want the viewport to expand when its height possible rather than scroll, 
 * causing scrolling to happen on a parent viewport.
 */
ViewportElement._StyleTypes.MeasureContentHeight = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style HorizontalScrollBarDisplay String
 * Determines the behavior of the horizontal scroll bar. Allowable values are "on", "off", or "auto".
 */
ViewportElement._StyleTypes.HorizontalScrollBarDisplay = 		StyleableBase.EStyleType.NORMAL;		// "on" || "off" || "auto"

/**
 * @style HorizontalScrollBarPlacement String
 * Determines the position of the horizontal scroll bar. Allowable values are "top" or "bottom".
 */
ViewportElement._StyleTypes.HorizontalScrollBarPlacement = 		StyleableBase.EStyleType.NORMAL;		// "top" || "bottom"

/**
 * @style VerticalScrollBarDisplay String
 * Determines the behavior of the vertical scroll bar. Allowable values are "on", "off", or "auto".
 */
ViewportElement._StyleTypes.VerticalScrollBarDisplay = 			StyleableBase.EStyleType.NORMAL;		// "on" || "off" || "auto"

/**
 * @style VerticalScrollBarPlacement String
 * Determines the position of the vertical scroll bar. Allowable values are "left" or "right".
 */
ViewportElement._StyleTypes.VerticalScrollBarPlacement = 		StyleableBase.EStyleType.NORMAL;		// "left" || "right"

//ScrollBar styles.
/**
 * @style HorizontalScrollBarStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to be applied to the horizontal scroll bar.
 */
ViewportElement._StyleTypes.HorizontalScrollBarStyle = 			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style VerticalScrollBarStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to be applied to the vertical scroll bar.
 */
ViewportElement._StyleTypes.VerticalScrollBarStyle = 			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition


////////////Default Styles///////////////////////////////////////

ViewportElement.StyleDefault = new StyleDefinition();

ViewportElement.StyleDefault.setStyle("HorizontalScrollBarDisplay", 					"auto");
ViewportElement.StyleDefault.setStyle("HorizontalScrollBarPlacement", 					"bottom");

ViewportElement.StyleDefault.setStyle("VerticalScrollBarDisplay", 						"auto");
ViewportElement.StyleDefault.setStyle("VerticalScrollBarPlacement", 					"right");

ViewportElement.StyleDefault.setStyle("HorizontalScrollBarStyle", 						null);
ViewportElement.StyleDefault.setStyle("VerticalScrollBarStyle", 						null);

ViewportElement.StyleDefault.setStyle("MeasureContentWidth", 							false);
ViewportElement.StyleDefault.setStyle("MeasureContentHeight", 							false);



/////////////Public///////////////////////////////

/**
 * @function setElement
 * Sets the child element of the Viewport.
 * 
 * @param element CanvasElement
 * The child element of the Viewport (or null).
 */
ViewportElement.prototype.setElement = 
	function (element)
	{
		if (this._viewElement != null)
		{
			this._viewElement.removeEventListener("measurecomplete", this._onViewElementMeasureCompleteInstance);
			this._viewPortContainer._removeChild(this._viewElement);
		}
		
		this._viewElement = element;
		
		if (this._viewElement != null)
		{
			this._viewElement.addEventListener("measurecomplete", this._onViewElementMeasureCompleteInstance);
			this._viewPortContainer._addChild(this._viewElement);
		}
		
		this._invalidateMeasure();
		this._invalidateLayout();
	};


////////////Internal//////////////////////////////
	
/**
 * @function _onViewportScrollBarChange
 * Event handler for the scroll bar "changed" event. Updates the child elements position within the Viewport.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
ViewportElement.prototype._onViewportScrollBarChange = 
	function (elementEvent)
	{
		this._invalidateLayout();
	};

ViewportElement.prototype._onViewElementMeasureComplete = 
	function (event)
	{
		this._invalidateLayout();
	};
	
/**
 * @function _onViewportMouseWheelEvent
 * Event handler for the Viewport's "wheel" event. Starts the scroll bar tween.
 * 
 * @param elementMouseWheelEvent ElementMouseWheelEvent
 * The ElementMouseWheelEvent to process.
 */		
ViewportElement.prototype._onViewportMouseWheelEvent = 
	function (elementMouseWheelEvent)
	{
		if (elementMouseWheelEvent.getDefaultPrevented() == true)
			return;
	
		var consumeEvent = false;
		
		var scrollPageSize = null;
		var scrollViewSize = null;
		var scrollLineSize = null;
		var scrollValue = null;
		var maxScrollValue = null;
		
		var deltaX = elementMouseWheelEvent.getDeltaX();
		var deltaY = elementMouseWheelEvent.getDeltaY();
		
		if (deltaX != 0 && this._horizontalScrollBar != null)
		{
			scrollPageSize = this._horizontalScrollBar.getScrollPageSize();
			scrollViewSize = this._horizontalScrollBar.getScrollViewSize();
			scrollLineSize = this._horizontalScrollBar.getScrollLineSize();
			
			maxScrollValue = scrollPageSize - scrollViewSize;
			if (maxScrollValue > 0)
			{
				scrollValue = this._horizontalScrollBar.getTweenToValue();
				if (scrollValue == null)
					scrollValue = this._horizontalScrollBar.getScrollValue();
				
				if (deltaX < 0 && scrollValue > 0)
				{
					this._horizontalScrollBar.startScrollTween(Math.max(scrollValue + (deltaX * (scrollLineSize * 3)), 0));
					consumeEvent = true;
				}
				else if (deltaX > 0 && scrollValue < maxScrollValue)
				{
					this._horizontalScrollBar.startScrollTween(Math.min(scrollValue + (deltaX * (scrollLineSize * 3)), maxScrollValue));
					consumeEvent = true;
				}
			}
		}
		
		if (deltaY != 0 && this._verticalScrollBar != null)
		{
			scrollPageSize = this._verticalScrollBar.getScrollPageSize();
			scrollViewSize = this._verticalScrollBar.getScrollViewSize();
			scrollLineSize = this._verticalScrollBar.getScrollLineSize();
			
			maxScrollValue = scrollPageSize - scrollViewSize;
			if (maxScrollValue > 0)
			{
				scrollValue = this._verticalScrollBar.getTweenToValue();
				if (scrollValue == null)
					scrollValue = this._verticalScrollBar.getScrollValue();
				
				if (deltaY < 0 && scrollValue > 0)
				{
					this._verticalScrollBar.startScrollTween(Math.max(scrollValue + (deltaY * (scrollLineSize * 3)), 0));
					consumeEvent = true;
				}
				else if (deltaY > 0 && scrollValue < maxScrollValue)
				{
					this._verticalScrollBar.startScrollTween(Math.min(scrollValue + (deltaY * (scrollLineSize * 3)), maxScrollValue));
					consumeEvent = true;
				}
			}
		}
		
		//We've consumed the wheel event, don't want parents double scrolling.
		if (consumeEvent == true)
		{
			elementMouseWheelEvent.preventDefault();
			this._invalidateLayout();
		}
	};
	
//@Override
ViewportElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ViewportElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("HorizontalScrollBarDisplay" in stylesMap ||
			"VerticalScrollBarDisplay" in stylesMap)
		{
			this._invalidateLayout();
			this._invalidateMeasure();
		}
		else 
		{	
			if ("HorizontalScrollBarPlacement" in stylesMap ||
				"VerticalScrollBarPlacement" in stylesMap)
			{
				this._invalidateLayout();
			}
			
			if ("MeasureContentWidth" in stylesMap || 
				"MeasureContentHeight" in stylesMap)
			{
				this._invalidateMeasure();
			}
		}
		
		if ("HorizontalScrollBarStyle" in stylesMap && this._horizontalScrollBar != null)
			this._applySubStylesToElement("HorizontalScrollBarStyle", this._horizontalScrollBar);
		if ("VerticalScrollBarStyle" in stylesMap && this._verticalScrollBar != null)
			this._applySubStylesToElement("VerticalScrollBarStyle", this._verticalScrollBar);
	};

//@Override
ViewportElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var vBarWidth = 0;
		var vBarHeight = 0;
		
		var hBarWidth = 0;
		var hBarHeight = 0;
		
		var w = 0;
		var h = 0;
		
		if (this._viewElement != null)
		{
			if (this.getStyle("MeasureContentWidth") == true)
				w = this._viewElement._getStyledOrMeasuredWidth();
			
			if (this.getStyle("MeasureContentHeight") == true)
				h = this._viewElement._getStyledOrMeasuredHeight();
		}
		
		if (this._verticalScrollBar != null)
		{
			vBarWidth = this._verticalScrollBar._getStyledOrMeasuredWidth();
			vBarHeight = this._verticalScrollBar._getStyledOrMeasuredHeight();
		}
		if (this._horizontalScrollBar != null)
		{
			hBarWidth = this._horizontalScrollBar._getStyledOrMeasuredWidth();
			hBarHeight = this._horizontalScrollBar._getStyledOrMeasuredHeight();
		}
		
		if (w == 0)
			w = hBarWidth;
		if (h == 0)
			h = vBarHeight;
		
		w += vBarWidth;
		h += hBarHeight;
		
		this._setMeasuredSize(w + padWidth, h + padHeight);
	};
	
//@override	
ViewportElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		ViewportElement.base.prototype._doLayout.call(this, paddingMetrics);
	
		var hDisplay = this.getStyle("HorizontalScrollBarDisplay");
		var vDisplay = this.getStyle("VerticalScrollBarDisplay");
		
		var contentWidth = 0;
		var contentHeight = 0;
		if (this._viewElement != null)
		{
			contentWidth = this._viewElement._getStyledOrMeasuredWidth();
			contentHeight = this._viewElement._getStyledOrMeasuredHeight();
		}
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		var scrollBarsChanged = false;
		var needsHScroll = false;
		var needsVScroll = false;
		
		//We need the scroll bar.
		if (hDisplay == "on" || (hDisplay == "auto" && contentWidth > w))
			needsHScroll = true;
			
		if (vDisplay == "on" || (vDisplay == "auto" && contentHeight > h))
			needsVScroll = true;
		
		//2nd pass, we need the *other* scroll bar because the first took some of our content area.
		if (needsHScroll == true && needsVScroll == false && vDisplay == "auto" && this._horizontalScrollBar != null)
		{
			if (contentHeight > h - this._horizontalScrollBar._getStyledOrMeasuredHeight())
				needsVScroll = true;
		}

		if (needsVScroll == true && needsHScroll == false && hDisplay == "auto" && this._verticalScrollBar != null)
		{
			if (contentWidth > w - this._verticalScrollBar._getStyledOrMeasuredWidth())
				needsHScroll = true;
		}
		
		//Destroy
		if (needsHScroll == false)
		{
			if (this._horizontalScrollBar != null)
			{
				this._removeChild(this._horizontalScrollBar);
				this._horizontalScrollBar = null;
				scrollBarsChanged = true;
			}
		}
		else //Create
		{
			if (this._horizontalScrollBar == null)
			{
				this._horizontalScrollBar = new ScrollBarElement();
				this._applySubStylesToElement("HorizontalScrollBarStyle", this._horizontalScrollBar);

				this._horizontalScrollBar.setStyle("LayoutDirection", "horizontal");
				this._horizontalScrollBar.setScrollLineSize(25);
				
				this._horizontalScrollBar.addEventListener("changed", this._onViewportScrollBarChangeInstance);
				this._addChild(this._horizontalScrollBar);
				scrollBarsChanged = true;
			}
		}
		
		//Destroy
		if (needsVScroll == false)
		{
			if (this._verticalScrollBar != null)
			{
				this._removeChild(this._verticalScrollBar);
				this._verticalScrollBar = null;
				scrollBarsChanged = true;
			}
		}
		else //Create
		{
			if (this._verticalScrollBar == null)
			{
				this._verticalScrollBar = new ScrollBarElement();
				this._applySubStylesToElement("VerticalScrollBarStyle", this._verticalScrollBar);
				
				this._verticalScrollBar.setStyle("LayoutDirection", "vertical");
				this._verticalScrollBar.setScrollLineSize(25);
				
				this._verticalScrollBar.addEventListener("changed", this._onViewportScrollBarChangeInstance);
				this._addChild(this._verticalScrollBar);
				scrollBarsChanged = true;
			}
		}
		
		//Wait for next pass, adding / removing bars changes content size, need bars to measure.
		if (scrollBarsChanged == true)
			return;
		
		var horizontalBarHeight = 0;
		var verticalBarWidth = 0;
		
		var horizontalScrollValue = 0;
		var verticalScrollValue = 0;
		
		if (this._horizontalScrollBar != null)
		{
			horizontalScrollValue = this._horizontalScrollBar.getScrollValue();
			horizontalBarHeight = this._horizontalScrollBar._getStyledOrMeasuredHeight();
			h -= horizontalBarHeight;
		}
		
		if (this._verticalScrollBar != null)
		{
			verticalScrollValue = this._verticalScrollBar.getScrollValue();
			verticalBarWidth = this._verticalScrollBar._getStyledOrMeasuredWidth();
			w -= verticalBarWidth;
		}
		
		//Fix scroll values (size reduction forces us to scroll up)
		horizontalScrollValue = Math.min(horizontalScrollValue, contentWidth - w);
		horizontalScrollValue = Math.max(horizontalScrollValue, 0);
		
		verticalScrollValue = Math.min(verticalScrollValue, contentHeight - h);
		verticalScrollValue = Math.max(verticalScrollValue, 0);
		
		var horizontalBarPlacement = this.getStyle("HorizontalScrollBarPlacement");
		var verticalBarPlacement = this.getStyle("VerticalScrollBarPlacement");
		
		if (this._horizontalScrollBar != null)
		{
			this._horizontalScrollBar.setScrollPageSize(contentWidth);
			this._horizontalScrollBar.setScrollViewSize(w);
			this._horizontalScrollBar.setScrollValue(horizontalScrollValue);
			
			this._horizontalScrollBar._setActualSize(this._width - verticalBarWidth, horizontalBarHeight);
			
			if (horizontalBarPlacement == "top")
			{
				if (verticalBarPlacement == "left")
					this._horizontalScrollBar._setActualPosition(verticalBarWidth, 0);
				else
					this._horizontalScrollBar._setActualPosition(0, 0);
			}
			else
			{
				if (verticalBarPlacement == "left")
					this._horizontalScrollBar._setActualPosition(verticalBarWidth, this._height - horizontalBarHeight);
				else
					this._horizontalScrollBar._setActualPosition(0, this._height - horizontalBarHeight);
			}
		}
		
		if (this._verticalScrollBar != null)
		{
			this._verticalScrollBar.setScrollPageSize(contentHeight);
			this._verticalScrollBar.setScrollViewSize(h);
			this._verticalScrollBar.setScrollValue(verticalScrollValue);
			
			this._verticalScrollBar._setActualSize(verticalBarWidth, this._height - horizontalBarHeight);
			
			if (verticalBarPlacement == "left")
			{
				if (horizontalBarPlacement == "top")
					this._verticalScrollBar._setActualPosition(0, horizontalBarHeight);
				else
					this._verticalScrollBar._setActualPosition(0, 0);
			}
			else
			{
				if (horizontalBarPlacement == "top")
					this._verticalScrollBar._setActualPosition(this._width - verticalBarWidth, horizontalBarHeight);
				else
					this._verticalScrollBar._setActualPosition(this._width - verticalBarWidth, 0);
			}
		}
		
		if (horizontalBarPlacement == "top")
			y += horizontalBarHeight;
		if (verticalBarPlacement == "left")
			x += verticalBarWidth;		
		
		this._viewPortContainer._setActualSize(w, h);
		this._viewPortContainer._setActualPosition(x, y);
		
		if (this._viewElement != null)
		{
			this._viewElement._setActualSize(Math.max(w, contentWidth), Math.max(h, contentHeight));
			this._viewElement._setActualPosition(horizontalScrollValue * -1, verticalScrollValue * -1);
		}
	};
	
	


/**
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////////////////////////	
///////////////////////ContainerBaseElement////////////////////////////////

/**
 * @class ContainerBaseElement
 * @inherits CanvasElement
 * 
 * Abstract base class for Container elements. Wraps internal child modification functions
 * such as _addChild() and _removeChild() with public functions such as addElement() and removeElement() 
 * for proper index management when using skins and overlays in conjunction with content children. 
 * 
 * Container children are not all considered equal. Content children added via the addElement() and removeElement()
 * functions maintain their own indexes and are placed in between raw children, such as skins, which render
 * underneath and overlay children which render above (elements intended to always be on top of content children).
 * 
 * Raw children added via _addChild() or _addChildAt() will be indexed before content children.
 * Content children added via addElement() or addElementAt() will be indexed after raw children and before overlay children.
 * Overlay children added via _addOverlayChild() will be index last, after content children.
 * All 3 lists maintain their own indexes.
 * 
 * @constructor ContainerBaseElement 
 * Creates new ContainerBaseElement instance.
 */

function ContainerBaseElement()
{
	ContainerBaseElement.base.prototype.constructor.call(this);
	
	//Storage for user added elements.
	this._elements = [];
	
	//Children that come after user added elements
	this._overlayChildren = [];
}	
	
//Inherit from CanvasElement
ContainerBaseElement.prototype = Object.create(CanvasElement.prototype);
ContainerBaseElement.prototype.constructor = ContainerBaseElement;
ContainerBaseElement.base = CanvasElement;
	

/////////////Style Types///////////////////////////////

ContainerBaseElement._StyleTypes = Object.create(null);

/**
 * @style ClipContent boolean
 * @inheritable
 * 
 * Determines if out of bounds rendering is allowed. If true the element will clip all rendering
 * and children's rendering to the elements bounding box. This style is inheritable for container elements.
 */
ContainerBaseElement._StyleTypes.ClipContent = 			StyleableBase.EStyleType.INHERITABLE;		// number (true || false)


/////////////Default Styles///////////////////////////////

ContainerBaseElement.StyleDefault = new StyleDefinition();
ContainerBaseElement.StyleDefault.setStyle("ClipContent",						true);


////////////ContainerBaseElement Public Functions//////////////////////////

//Expose child modification functions.

/**
 * @function addElement
 * Adds a content child element to the end of this element's content child list.
 * 
 * @param element CanvasElement
 * CanvasElement to be added as a content child of this element.
 * 
 * @returns CanvasElement
 * Returns the element just added.
 */	
ContainerBaseElement.prototype.addElement = 
	function (element)
	{
		return this.addElementAt(element, this._elements.length);
	};

/**
 * @function addElementAt
 * Inserts a content child element to this element's content child list at the specified index.
 * 
 * @param element CanvasElement
 * CanvasElement to be added as a content child of this element.
 * 
 * @param index int
 * Child index to insert the element.
 * 
 * @returns CanvasElement
 * Returns the element just added when successful, null if the element could not
 * be added due to the index being out of range.
 */		
ContainerBaseElement.prototype.addElementAt = 
	function (element, index)
	{
		if (!(element instanceof CanvasElement))
			return null;
	
		if (index < 0 || index > this._elements.length)
			return null;
		
		var childIndex = this._children.length - this._overlayChildren.length - this._elements.length + index;
		
		this._elements.splice(index, 0, element);
		ContainerBaseElement.base.prototype._addChildAt.call(this, element, childIndex);
		
		return element;
	};
	
/**
 * @function removeElement
 * Removes a content child element from this element's content children list.
 * 
 * @param element CanvasElement
 * Content child to be removed.
 * 
 * @returns CanvasElement
 * Returns the CanvasElement just removed if successful, null if the
 * element could not be removed due to it not being a content child of this element.
 */		
ContainerBaseElement.prototype.removeElement = 
	function (element)
	{
		var index = this._elements.indexOf(element);
		return this.removeElementAt(index);
	};
	
/**
 * @function removeElementAt
 * Removes a content child element at specified index.
 * 
 * @param index int
 * Content index to be removed.
 * 
 * @returns CanvasElement
 * Returns the CanvasElement just removed if successful, null if the element could
 * not be removed due it it not being a child of this element, or index out of range.
 */			
ContainerBaseElement.prototype.removeElementAt = 
	function (index)
	{
		if (index < 0 || index >= this._elements.length)
			return null;
	
		var childIndex = this._children.length - this._overlayChildren.length - this._elements.length + index;

		this._elements.splice(index, 1);
		return ContainerBaseElement.base.prototype._removeChildAt.call(this, childIndex);
	};

/**
 * @function getElementAt
 * Gets the content child element at the supplied index.
 * 
 * @param index int
 * Content index of child element to return;
 * 
 * @returns CanvasElement
 * The CanvasElement at the supplied index, or null if index is out of range. 
 */		
ContainerBaseElement.prototype.getElementAt = 
	function (index)
	{
		if (index < 0 || index >= this._elements.length)
			return null;
		
		return this._elements[index];
	};
	
/**
 * @function getElementIndex
 * Returns the index of the supplied content child element.
 * 
 * @param element CanvasElement
 * Content child element to return the index.
 * 
 * @returns int
 * Returns the child index or -1 if the element is not
 * a content child of this element.
 */		
ContainerBaseElement.prototype.getElementIndex = 
	function (element)
	{
		return this._elements.indexOf(element);
	};

/**
 * @function setElementIndex
 * Changes a content child element's index. 
 * 
 * @param element CanvasElement
 * Content child element to change index.
 * 
 * @param index int
 * New content index of the content child element.
 * 
 * @returns boolean
 * Returns true if the child's index is successfully changed, false if the element
 * is not a content child of this element or the index is out of range.
 */		
ContainerBaseElement.prototype.setElementIndex = 
	function (element, index)
	{
		if (index < 0 || index >= this._elements.length)
			return false;
		
		var currentIndex = this._elements.indexOf(element);
		if (currentIndex == -1 || currentIndex == index)
			return false;
		
		var childIndex = this._children.length - this._overlayChildren.length - this._elements.length + index;
		
		this._elements.splice(index, 0, this._elements.splice(currentIndex, 1)[0]);
		ContainerBaseElement.base.prototype._setChildIndex.call(this, element, childIndex);
		
		return true;
	};
	
/**
 * @function getNumElements
 * Gets this elements number of content children.
 * 
 * @returns int
 * The number of content child elements.
 */		
ContainerBaseElement.prototype.getNumElements = 
	function ()
	{
		return this._elements.length;
	};

/**
 * @function _addOverlayChild
 * Adds an overlay child element to the end of this element's overlay child list.
 * 
 * @param element CanvasElement
 * Element to be added as an overlay child of this element.
 * 
 * @returns CanvasElement
 * Returns the element just added.
 */		
ContainerBaseElement.prototype._addOverlayChild = 
	function (element)
	{
		return this._addOverlayChildAt(element, this._overlayChildren.length);
	};
	
/**
 * @function _addOverlayChildAt
 * Inserts an overlay child element to this elements overlay child list at the specified index.
 * 
 * @param element CanvasElement
 * Element to be added as an overlay child of this element.
 * 
 * @returns CanvasElement
 * Returns the element just added when successful, null if the element could not
 * be added due to the index being out of range.
 */			
ContainerBaseElement.prototype._addOverlayChildAt = 
	function (element, index)
	{
		if (!(element instanceof CanvasElement))
			return null;
	
		if (index < 0 || index > this._overlayChildren.length)
			return null;
		
		var childIndex = this._children.length - this._overlayChildren.length + index;
		
		this._overlayChildren.splice(index, 0, element);
		ContainerBaseElement.base.prototype._addChildAt.call(this, element, childIndex);
		
		return element;
	};	

/**
 * @function _removeOverlayChild
 * Removes an overlay child element from this elements overlay child list.
 * 
 * @param element CanvasElement
 * Overlay child to be removed.
 * 
 * @returns CanvasElement
 * Returns the element just removed if successful, null if the
 * element could not be removed due to it not being an overlay child of this element.
 */		
ContainerBaseElement.prototype._removeOverlayChild = 
	function (element)
	{
		var index = this._overlayChildren.indexOf(element);
		return this._removeOverlayChildAt(index);
	};

/**
 * @function _removeOverlayChildAt
 * Removes an overlay child element at specified index.
 * 
 * @param index int
 * Overlay index to be removed.
 * 
 * @returns CanvasElement
 * Returns the element just removed if successful, null if the element could
 * not be removed due it it not being an overlay child of this element, or index out of range.
 */			
ContainerBaseElement.prototype._removeOverlayChildAt = 
	function (index)
	{
		if (index < 0 || index >= this._overlayChildren.length)
			return null;
		
		var childIndex = this._children.length - this._overlayChildren.length + index;

		this._overlayChildren.splice(index, 1);
		return ContainerBaseElement.base.prototype._removeChildAt.call(this, childIndex);
	};	

/**
 * @function _getOverlayChildAt
 * Gets the overlay child element at the supplied index.
 * 
 * @param index int
 * Overlay index of child element to return;
 * 
 * @returns CanvasElement
 * The element at the supplied overlay index, or null if index is out of range. 
 */		
ContainerBaseElement.prototype._getOverlayChildAt = 
	function (index)
	{
		if (index < 0 || index >= this._overlayChildren.length)
			return null;
		
		return this._overlayChildren[index];
	};	
	
/**
 * @function _getOverlayChildIndex
 * Returns the overlay index of the supplied child element.
 * 
 * @param element CanvasElement
 * Child element to return the overlay index.
 * 
 * @returns int
 * Returns the child's overlay index or -1 if the element is not
 * an overlay child of this element.
 */		
ContainerBaseElement.prototype._getOverlayChildIndex = 
	function (element)
	{
		return this._overlayChildren.indexOf(element);
	};	
	
/**
 * @function _setOverlayChildIndex
 * Changes an overlay child element's overlay index. 
 * 
 * @param element CanvasElement
 * Overlay child element to change index.
 * 
 * @param index int
 * New overlay index of the child element.
 * 
 * @returns boolean
 * Returns true if the child's index is successfully changed, false if the element
 * is not an overlay child of this element or the index is out of range.
 */		
ContainerBaseElement.prototype._setOverlayChildIndex = 
	function (element, index)
	{
		if (index < 0 || index >= this._overlayChildren.length)
			return false;
		
		var currentIndex = this._overlayChildren.indexOf(element);
		if (currentIndex < 0 || currentIndex == index)
			return false;
		
		var childIndex = this._children.length - this._overlayChildren.length + index;
		
		this._overlayChildren.splice(index, 0, this._overlayChildren.splice(currentIndex, 1)[0]);
		ContainerBaseElement.base.prototype._setChildIndex.call(this, element, childIndex);
		
		return true;
	}; 	
	
/**
 * @function _getNumOverlayChildren
 * Gets this elements number of overlay children.
 * 
 * @returns int
 * The number of overlay child elements.
 */		
ContainerBaseElement.prototype._getNumOverlayChildren = 
	function ()
	{
		return this._overlayChildren.length;
	};
	
//Override - Add the child before elements & overlay
ContainerBaseElement.prototype._addChild = 
	function (element)
	{
		var index = this._children.length - this._elements.length - this._overlayChildren.length;
		return ContainerBaseElement.base.prototype._addChildAt.call(this, element, index);
	};
	
//Override - Dont allow insertion into elements or overlay range
ContainerBaseElement.prototype._addChildAt = 
	function (element, index)
	{
		var maxIndex = this._children.length - this._elements.length - this._overlayChildren.length;
		
		if (index < 0 || index > maxIndex)
			return null;
		
		return ContainerBaseElement.base.prototype._addChildAt.call(this, element, index);
	};

//Override - Remove from element or overlay if necessary
ContainerBaseElement.prototype._removeChildAt = 
	function (index)
	{
		if (index < 0 || index >= this._children.length)
			return null;
		
		var element = this._children[index];
		
		var subIndex = this._elements.indexOf(element);
		if (subIndex >= 0)
			return this.removeElementAt(subIndex);
		
		subIndex = this._overlayChildren.indexOf(element);
		if (subIndex >= 0)
			return this._removeOverlayChildAt(subIndex);
		
		return ContainerBaseElement.base.prototype._removeChildAt.call(this, index);
	};

//@Override	- Dont allow swapping in or out of element & overlay ranges.
ContainerBaseElement.prototype._setChildIndex = 
	function (element, index)
	{
		var maxIndex = this._children.length - this._elements.length - this._overlayChildren.length;
	
		if (index < 0 || index >= maxIndex)
			return false;
		
		var currentIndex = this._getChildIndex(element);
		if (currentIndex < 0 || currentIndex >= maxIndex || currentIndex == index)
			return false;
		
		return ContainerBaseElement.base.prototype._setChildIndex.call(this, element, index);
	};
	
	


/**
 * @depends ContainerBaseElement.js
 */

///////////////////////////////////////////////////////////////////////////	
///////////////////////ListContainerElement////////////////////////////////

/**
 * @class ListContainerElement
 * @inherits ContainerBaseElement
 * 
 * The ListContainer can be used to lay out children in a vertical or horizontal fashion.
 * This container uses children's styles Width, Height, PercentWidth, PercentHeight, MinWidth,
 * MaxWidth, MinHeight, and MaxHeight.
 * 
 * Nesting containers is the best way to quickly and simply build complex layouts.
 * 
 * Width, and Height will override PercentWidth and PercentHeight styles when the same priority. 
 * Higher priority styles always override lower priority styles. For example, setting "PercentWidth" 
 * via setStyle() will override a "Width" style set via StyleDefinition. 
 * See CanvasElement getStyle() for priority chain info. 
 * 
 * @constructor ListContainerElement 
 * Creates new ListContainerElement instance.
 */
function ListContainerElement()
{
	ListContainerElement.base.prototype.constructor.call(this);
}

//Inherit from ContainerBaseElement
ListContainerElement.prototype = Object.create(ContainerBaseElement.prototype);
ListContainerElement.prototype.constructor = ListContainerElement;
ListContainerElement.base = ContainerBaseElement;	
	
/////////////Style Types///////////////////////////////

ListContainerElement._StyleTypes = Object.create(null);

/**
 * @style LayoutDirection String
 * 
 * Determines the layout direction of this ListContainer. Allowable values are "horizontal" or "vertical".
 */
ListContainerElement._StyleTypes.LayoutDirection = 			StyleableBase.EStyleType.NORMAL;		// "horizontal" || "vertical"

/**
 * @style LayoutGap Number
 * 
 * Space in pixels to leave between child elements.
 */
ListContainerElement._StyleTypes.LayoutGap = 				StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style LayoutVerticalAlign String
 * 
 * Child vertical alignment to be used when children do not fill all available space. Allowable values are "top", "bottom", or "middle". 
 */
ListContainerElement._StyleTypes.LayoutVerticalAlign = 		StyleableBase.EStyleType.NORMAL;		// "top" || "bottom" || "middle" 

/**
 * @style LayoutHorizontalAlign String
 * 
 * Child horizontal alignment to be used when children do not fill all available space. Allowable values are "left", "right", or "center". 
 */
ListContainerElement._StyleTypes.LayoutHorizontalAlign = 	StyleableBase.EStyleType.NORMAL;		//"left" || "right" || "center"


////////////Default Styles////////////////////////////

ListContainerElement.StyleDefault = new StyleDefinition();

//ListContainerElement specific styles
ListContainerElement.StyleDefault.setStyle("LayoutDirection", 			"vertical");
ListContainerElement.StyleDefault.setStyle("LayoutGap", 				0);
ListContainerElement.StyleDefault.setStyle("LayoutVerticalAlign", 		"top");
ListContainerElement.StyleDefault.setStyle("LayoutHorizontalAlign", 	"left");


//////////////ListContainerElement Protected Functions//////////////

//@Override
ListContainerElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ListContainerElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("LayoutDirection" in stylesMap ||
			"LayoutGap" in stylesMap)
		{
			this._invalidateMeasure();
			this._invalidateLayout();
		}
		else if ("LayoutVerticalAlign" in stylesMap || "LayoutHorizontalAlign" in stylesMap)
			this._invalidateLayout();
	};

//@Override
ListContainerElement.prototype._doMeasure = 
	function (padWidth, padHeight)
	{
		var measuredWidth = 0;
		var measuredHeight = 0;
		
		var layoutGap = this.getStyle("LayoutGap");
		var layoutDirection = this.getStyle("LayoutDirection");
		
		var child = null;
		
		var width = null;
		var height = null;
		var rotateDegrees = null;
		
		var tempWidth;
		var tempHeight;
		var tempRotateDegrees;		
		
		var insertGap = false;
		
		//TODO: Measure is incorrect when using percent sized elements that total < 100%
		//		Need to account for unused space.
		
		for (var i = 0; i < this._elements.length; i++)
		{
			child = this._elements[i];
			if (child.getStyle("IncludeInLayout") == false)
				continue;
			
			rotateDegrees = child.getStyle("RotateDegrees");
			
			width = child._getStyledOrMeasuredWidth();
			height = child._getStyledOrMeasuredHeight();
			
			if (rotateDegrees != 0)
			{
				//Record child's current w/h & rotation
				tempWidth = child._width;
				tempHeight = child._height;
				tempRotateDegrees = child._rotateDegrees;
				
				//TODO: Update getMetrics() so we can pass child values.
				//Spoof the rotation position/size so we can get parent metrics.
				child._width = width;
				child._height = height;
				child._rotateDegrees = rotateDegrees;
				
				//Get parent metrics for spoof position
				rotatedMetrics = child.getMetrics(this);
				
				//Put back current values
				child._width = tempWidth;
				child._height = tempHeight;
				child._rotateDegrees = tempRotateDegrees;
				
				width = Math.ceil(rotatedMetrics.getWidth());
				height = Math.ceil(rotatedMetrics.getHeight());
			}
		
			if (layoutDirection == "horizontal")
			{
				//Increment width
				measuredWidth += width;
				
				//Use maximum child height
				if (height > measuredHeight)
					measuredHeight = height;
			}
			else //if (layoutDirection == "vertical")
			{
				//Increment height
				measuredHeight += height;
				
				//Use maximum child height
				if (width > measuredWidth)
					measuredWidth = width;
			}
			
			if (insertGap == true)
			{
				if (layoutDirection == "horizontal")
					measuredWidth += layoutGap;
				else //if (layoutDirection == "vertical")
					measuredHeight += layoutGap;
			}
			else
				insertGap = true;
		}
		
		measuredWidth += padWidth;
		measuredHeight += padHeight;
		
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};

//@Override
ListContainerElement.prototype._doLayout =
	function(paddingMetrics)
	{
		ListContainerElement.base.prototype._doLayout.call(this, paddingMetrics);
	
		var layoutGap = this.getStyle("LayoutGap");
		var layoutDirection = this.getStyle("LayoutDirection");
		var layoutVerticalAlign = this.getStyle("LayoutVerticalAlign");
		var layoutHorizontalAlign = this.getStyle("LayoutHorizontalAlign");
	
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		var i;
		
		var child = null;
		var childSizeData = [];
		
		var totalPercentUsed = 0;
		var numRenderables = 0;
		
		var widthData = null;
		var heightData = null;
		var percentWidthData = null;
		var percentHeightData = null;
		
		//Record element sizing data.
		for (i = 0; i < this._elements.length; i++)
		{
			child = this._elements[i];
			if (child.getStyle("IncludeInLayout") == false)
				continue;
			
			numRenderables++;
			
			var sizeData = {
				element:child,
				width:null, 
				height:null, 
				pWidth:null, 
				pHeight:null, 
				maxWidth:null, 
				maxHeight:null, 
				minWidth:null, 
				minHeight:null,
				rotateDegrees:null};
			
			sizeData.rotateDegrees = child.getStyle("RotateDegrees");
			
			widthData = child.getStyleData("Width");
			percentWidthData = child.getStyleData("PercentWidth");
			
			//Use width if equal or higher priority than percent width
			if (widthData.comparePriority(percentWidthData) >= 0)
				sizeData.width = widthData.value;
			else
			{
				//Percent sizing not supported on transformed elements.
				if (sizeData.rotateDegrees == 0)
					sizeData.pWidth = percentWidthData.value;
				
				if (sizeData.pWidth != null && layoutDirection == "horizontal")
					totalPercentUsed += sizeData.pWidth;
			}
			
			heightData = child.getStyleData("Height");
			percentHeightData = child.getStyleData("PercentHeight");
			
			//Use height if equal or higher priority than percent height
			if (heightData.comparePriority(percentHeightData) >= 0)
				sizeData.height = heightData.value;
			else
			{
				//Percent sizing not supported on transformed elements.
				if (sizeData.rotateDegrees == 0)
					sizeData.pHeight = percentHeightData.value;
				
				if (sizeData.pHeight != null && layoutDirection == "vertical")
					totalPercentUsed += sizeData.pHeight;				
			}
			
			sizeData.minHeight = child.getStyle("MinHeight");
			sizeData.maxHeight = child.getStyle("MaxHeight");
			
			sizeData.minWidth = child.getStyle("MinWidth");
			sizeData.maxWidth = child.getStyle("MaxWidth");
			
			if (sizeData.minWidth == null)
				sizeData.minWidth = 0;
			if (sizeData.minHeight == null)
				sizeData.minHeight = 0;
			if (sizeData.maxWidth == null)
				sizeData.maxWidth = Number.MAX_VALUE;
			if (sizeData.maxHeight == null)
				sizeData.maxHeight = Number.MAX_VALUE;
			
			childSizeData.push(sizeData);
		}
		
		var totalGap = 0;
		if (numRenderables > 1)
			totalGap = (numRenderables - 1) * layoutGap;
		
		//Available space for children in layout axis.
		var availableSize = 0;
		if (layoutDirection == "horizontal")
			availableSize = w - totalGap;
		else
			availableSize = h - totalGap;
		
		////////////Calculate element sizes//////////////////
		
		var rotatedMetrics = null;
		var percentSizedElements = [];
		
		//Size all explicitly sized elements, record percent sized, and adjust available size for percent elements.
		for (i = 0; i < childSizeData.length; i++)
		{
			child = childSizeData[i];
			
			//Percent sized elements cannot be rotated
			child.element._setActualRotation(child.rotateDegrees, 0, 0);
			
			if (layoutDirection == "horizontal" && childSizeData[i].width == null && childSizeData[i].pWidth != null)
			{
				child.percentSize = child.pWidth;
				child.minSize = Math.max(child.minWidth, child.element._measuredWidth);
				child.maxSize = child.maxWidth;
				percentSizedElements.push(child);
				
				if (child.height == null)
				{
					if (child.pHeight != null)
						child.height = Math.round(h * (child.pHeight / 100));
					else
						child.height = child.element._measuredHeight;
					
					child.height = Math.min(child.maxHeight, child.height);
					child.height = Math.max(child.minHeight, child.height);
				}
			}
			else if (layoutDirection == "vertical" && childSizeData[i].height == null && childSizeData[i].pHeight != null)
			{
				child.percentSize = child.pHeight;
				child.minSize = Math.max(child.minHeight, child.element._measuredHeight);
				child.maxSize = child.maxHeight;
				percentSizedElements.push(child);
				
				if (child.width == null)
				{
					if (child.pWidth != null)
						child.width = Math.round(w * (child.pWidth / 100));
					else
						child.width = child.element._measuredWidth;
					
					child.width = Math.min(child.maxWidth, child.width);
					child.width = Math.max(child.minWidth, child.width);
				}
			}
			else
			{
				if (child.width == null)
				{
					if (child.pWidth != null)
						child.width = Math.round(w * (child.pWidth / 100));
					else
						child.width = child.element._measuredWidth;
					
					child.width = Math.min(child.maxWidth, child.width);
					child.width = Math.max(child.minWidth, child.width);
				}
				
				if (child.height == null)
				{
					if (child.pHeight != null)
						child.height = Math.round(h * (child.pHeight / 100));
					else
						child.height = child.element._measuredHeight;
					
					child.height = Math.min(child.maxHeight, child.height);
					child.height = Math.max(child.minHeight, child.height);
				}
				
				child.element._setActualSize(child.width, child.height);
				
				//Update the sizing to reflect size after rotation transform (for layout).
				if (child.rotateDegrees != 0)
				{
					rotatedMetrics = child.element.getMetrics(this);
					
					child.width = Math.ceil(rotatedMetrics.getWidth());
					child.height = Math.ceil(rotatedMetrics.getHeight());
				}
				
				if (layoutDirection == "horizontal")
					availableSize -= child.width;
				else // "vertical"
					availableSize -= child.height;
			}
		}
		
		//We're not using all the space, shrink us.
		if (totalPercentUsed < 100)
			availableSize = Math.round(availableSize * (totalPercentUsed / 100));
		
		//Calculate percent sized elements actual size.
		CanvasElement._calculateMinMaxPercentSizes(percentSizedElements, availableSize);
			
		//Size the percent sized elements.
		for (i = 0; i < percentSizedElements.length; i++)
		{
			child = percentSizedElements[i];
			
			if (layoutDirection == "horizontal")
				child.width = child.actualSize;
			else // "vertical"
				child.height = child.actualSize;
			
			child.element._setActualSize(child.width, child.height);
		}
			
		//Get total content size (gap + elements).
		var totalContentSize = totalGap;
		for (i = 0; i < childSizeData.length; i++)
		{
			if (layoutDirection == "horizontal")
				totalContentSize += childSizeData[i].width;
			else // "vertical"
				totalContentSize += childSizeData[i].height;
		}
		
		var actualX = x;
		var actualY = y;
		
		//Adjust starting position.
		if (layoutDirection == "horizontal" && totalContentSize != w)
		{
			if (layoutHorizontalAlign == "center")
				actualX += Math.round((w / 2) - (totalContentSize / 2));
			else if (layoutHorizontalAlign == "right")
				actualX += (w - totalContentSize);
		}
		else if (layoutDirection == "vertical" && totalContentSize != h)
		{
			if (layoutVerticalAlign == "middle")
				actualY += Math.round((h / 2) - (totalContentSize / 2));
			else if (layoutVerticalAlign == "bottom")
				actualY += (h - totalContentSize);
		}

		//Place elements.
		var insertGap = false;
		for (i = 0; i < childSizeData.length; i++)
		{
			child = childSizeData[i];
			
			if (layoutDirection == "horizontal")
			{
				if (insertGap == true)
					actualX += layoutGap;
				else
					insertGap = true;
				
				if (layoutVerticalAlign == "top")
					actualY = y;
				else if (layoutVerticalAlign == "bottom")
					actualY = y + h - child.height;
				else //middle
					actualY = Math.round(y + (h / 2) - (child.height / 2));
				
				if (child.rotateDegees == 0)
					child.element._setActualPosition(actualX, actualY);
				else
					child.element._setRelativePosition(actualX, actualY, this);
				
				actualX += child.width;
			}
			else // "vertical"
			{
				if (insertGap == true)
					actualY += layoutGap;
				else
					insertGap = true;
				
				if (layoutHorizontalAlign == "left")
					actualX = x;
				else if (layoutHorizontalAlign == "right")
					actualX = x + w - child.width;
				else //center
					actualX = Math.round(x + (w / 2) - (child.width / 2));
				
				if (child.rotateDegrees == 0)
					child.element._setActualPosition(actualX, actualY);
				else
					child.element._setRelativePosition(actualX, actualY, this);				
				
				actualY += child.height;
			}
		}
	};
	
	


/**
 * @depends ListContainerElement.js
 * @depends ScrollButtonSkinElement.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////////ScrollBarElement/////////////////////////////////

/**
 * @class ScrollBarElement
 * @inherits ListContainerElement
 * 
 * ScrollBarElement renders a skin-able scroll bar that can be
 * oriented horizontally or vertically and assigns a default
 * skin to the scroll buttons.
 * 
 * See the default skin ScrollButtonSkinElement for additional skin styles.
 * 
 * @seealso ScrollButtonSkinElement
 * 
 * 
 * @constructor ScrollBarElement 
 * Creates new ScrollBarElement instance.
 */
function ScrollBarElement()
{
	ScrollBarElement.base.prototype.constructor.call(this);
	
	this._buttonIncrement = null;
	this._buttonDecrement = null;
	this._buttonTrack = null;
	this._buttonTab = null;
	
	this._trackAndTabContainer = new CanvasElement();
	this._trackAndTabContainer.setStyle("ClipContent", false);
	this._trackAndTabContainer.setStyle("PercentWidth", 100);
	this._trackAndTabContainer.setStyle("PercentHeight", 100);
	this._trackAndTabContainer.setStyle("MinWidth", 0);	//We dont want base measuring this container
	this._trackAndTabContainer.setStyle("MinHeight", 0); //We dont want base measuring this container
	
	this.addElement(this._trackAndTabContainer);
	
	this._scrollPageSize = 0;
	this._scrollViewSize = 0;
	this._scrollLineSize = 1;
	
	this._scrollValue = 0;
	
	this._scrollTween = null;
	
	var _self = this;
	
	//Private event handlers, need different instance for each ScrollBar, proxy to prototype.
	this._onScrollButtonClickInstance = 
		function (elementMouseEvent)
		{
			_self._onScrollButtonClick(elementMouseEvent);
		};
	this._onScrollTabDragInstance = 
		function (elementEvent)
		{
			_self._onScrollTabDrag(elementEvent);
		};
	this._onScrollBarEnterFrameInstance = 
		function (event)
		{
			_self._onScrollBarEnterFrame(event);
		};
	this._onTrackAndTabContainerMeasureCompleteInstance = 
		function (event)
		{
			_self._onTrackAndTabContainerMeasureComplete(event);
		};
		
	this._trackAndTabContainer.addEventListener("measurecomplete", this._onTrackAndTabContainerMeasureCompleteInstance);
}

//Inherit from ListContainerElement
ScrollBarElement.prototype = Object.create(ListContainerElement.prototype);
ScrollBarElement.prototype.constructor = ScrollBarElement;
ScrollBarElement.base = ListContainerElement;

/////////////Events////////////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the scroll position changes as a result of user interaction or tween.
 */


/////////////Style Types///////////////////////////////

ScrollBarElement._StyleTypes = Object.create(null);

/**
 * @style ScrollTweenDuration Number
 * Time in milliseconds the scroll tween animation should run.
 */
ScrollBarElement._StyleTypes.ScrollTweenDuration =			StyleableBase.EStyleType.NORMAL;		// number (milliseconds)

//ScrollButton / Button styles.
/**
 * @style ButtonIncrementStyle StyleDefinition
 * StyleDefinition or [StyleDefinition] array to be applied to the Scroll increment Button.
 */
ScrollBarElement._StyleTypes.ButtonIncrementStyle = 	StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style ButtonDecrementStyle StyleDefinition
 * StyleDefinition or [StyleDefinition] array to be applied to the Scroll decrement Button.
 */
ScrollBarElement._StyleTypes.ButtonDecrementStyle = 	StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style ButtonTrackStyle StyleDefinition
 * StyleDefinition or [StyleDefinition] array to be applied to the scroll bar track Button.
 */
ScrollBarElement._StyleTypes.ButtonTrackStyle = 			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style ButtonTabStyle StyleDefinition
 * StyleDefinition or [StyleDefinition] array to be applied to the scroll bar tab (draggable) Button.
 */
ScrollBarElement._StyleTypes.ButtonTabStyle = 				StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition


////////////Default Styles////////////////////////////

//////TRACK

//up/over/down skins of track
ScrollBarElement.TrackSkinStyleDefault = new StyleDefinition();
ScrollBarElement.TrackSkinStyleDefault.setStyle("BorderType", 						"solid");
ScrollBarElement.TrackSkinStyleDefault.setStyle("BorderThickness", 					1);
ScrollBarElement.TrackSkinStyleDefault.setStyle("BorderColor", 						"#333333");
ScrollBarElement.TrackSkinStyleDefault.setStyle("BackgroundFill", 					"#D9D9D9");

//disabled skin of track
ScrollBarElement.DisabledTrackSkinStyleDefault = new StyleDefinition();
ScrollBarElement.DisabledTrackSkinStyleDefault.setStyle("BorderType", 				"solid");
ScrollBarElement.DisabledTrackSkinStyleDefault.setStyle("BorderThickness", 			1);
ScrollBarElement.DisabledTrackSkinStyleDefault.setStyle("BorderColor", 				"#999999");
ScrollBarElement.DisabledTrackSkinStyleDefault.setStyle("BackgroundFill", 			"#ECECEC");

//track button
ScrollBarElement.ButtonTrackStyleDefault = new StyleDefinition();
ScrollBarElement.ButtonTrackStyleDefault.setStyle("BorderType", 					"none");
ScrollBarElement.ButtonTrackStyleDefault.setStyle("MinWidth", 						15);
ScrollBarElement.ButtonTrackStyleDefault.setStyle("MinHeight", 						15);
ScrollBarElement.ButtonTrackStyleDefault.setStyle("UpSkinStyle", 					ScrollBarElement.TrackSkinStyleDefault);  
ScrollBarElement.ButtonTrackStyleDefault.setStyle("OverSkinStyle", 					ScrollBarElement.TrackSkinStyleDefault); 
ScrollBarElement.ButtonTrackStyleDefault.setStyle("DownSkinStyle", 					ScrollBarElement.TrackSkinStyleDefault);
ScrollBarElement.ButtonTrackStyleDefault.setStyle("DisabledSkinStyle", 				ScrollBarElement.DisabledTrackSkinStyleDefault); 

////Dynamically added based on LayoutDirection

//track button
ScrollBarElement.VButtonTrackStyleDefault = new StyleDefinition();
ScrollBarElement.VButtonTrackStyleDefault.setStyle("PercentWidth", 					100);

//track button
ScrollBarElement.HButtonTrackStyleDefault = new StyleDefinition();
ScrollBarElement.HButtonTrackStyleDefault.setStyle("PercentHeight", 				100);


//////ARROWS

//disabled skin of arrow buttons
ScrollBarElement.DisabledButtonScrollArrowSkinStyleDefault = new StyleDefinition();
ScrollBarElement.DisabledButtonScrollArrowSkinStyleDefault.setStyle("ArrowColor", 			"#777777");

//up / over / down skin of arrow buttons
ScrollBarElement.ButtonScrollArrowSkinStyleDefault = new StyleDefinition();
ScrollBarElement.ButtonScrollArrowSkinStyleDefault.setStyle("ArrowColor", 					"#000000");

//arrow buttons common
ScrollBarElement.ButtonScrollArrowStyleDefault = new StyleDefinition();
ScrollBarElement.ButtonScrollArrowStyleDefault.setStyle("SkinClass", 						ScrollButtonSkinElement);	
ScrollBarElement.ButtonScrollArrowStyleDefault.setStyle("MinWidth", 						15);
ScrollBarElement.ButtonScrollArrowStyleDefault.setStyle("MinHeight", 						15);
ScrollBarElement.ButtonScrollArrowStyleDefault.setStyle("UpSkinStyle", 						ScrollBarElement.ButtonScrollArrowSkinStyleDefault);
ScrollBarElement.ButtonScrollArrowStyleDefault.setStyle("OverSkinStyle", 					ScrollBarElement.ButtonScrollArrowSkinStyleDefault);
ScrollBarElement.ButtonScrollArrowStyleDefault.setStyle("DownSkinStyle", 					ScrollBarElement.ButtonScrollArrowSkinStyleDefault);
ScrollBarElement.ButtonScrollArrowStyleDefault.setStyle("DisabledSkinStyle", 				ScrollBarElement.DisabledButtonScrollArrowSkinStyleDefault);

////Dynamically added based on LayoutDirection

//arrow button (vertical increment)
ScrollBarElement.VButtonScrollArrowIncSkinStyleDefault = new StyleDefinition();
ScrollBarElement.VButtonScrollArrowIncSkinStyleDefault.setStyle("ArrowDirection", 			"down");

ScrollBarElement.VButtonScrollArrowIncStyleDefault = new StyleDefinition();
ScrollBarElement.VButtonScrollArrowIncStyleDefault.setStyle("PercentWidth", 				100);
ScrollBarElement.VButtonScrollArrowIncStyleDefault.setStyle("UpSkinStyle", 					ScrollBarElement.VButtonScrollArrowIncSkinStyleDefault);
ScrollBarElement.VButtonScrollArrowIncStyleDefault.setStyle("OverSkinStyle", 				ScrollBarElement.VButtonScrollArrowIncSkinStyleDefault);
ScrollBarElement.VButtonScrollArrowIncStyleDefault.setStyle("DownSkinStyle", 				ScrollBarElement.VButtonScrollArrowIncSkinStyleDefault);
ScrollBarElement.VButtonScrollArrowIncStyleDefault.setStyle("DisabledSkinStyle", 			ScrollBarElement.VButtonScrollArrowIncSkinStyleDefault);

//arrow button (vertical decrement)
ScrollBarElement.VButtonScrollArrowDecSkinStyleDefault = new StyleDefinition();
ScrollBarElement.VButtonScrollArrowDecSkinStyleDefault.setStyle("ArrowDirection", 			"up");

ScrollBarElement.VButtonScrollArrowDecStyleDefault = new StyleDefinition();
ScrollBarElement.VButtonScrollArrowDecStyleDefault.setStyle("PercentWidth", 				100);
ScrollBarElement.VButtonScrollArrowDecStyleDefault.setStyle("UpSkinStyle", 					ScrollBarElement.VButtonScrollArrowDecSkinStyleDefault);
ScrollBarElement.VButtonScrollArrowDecStyleDefault.setStyle("OverSkinStyle", 				ScrollBarElement.VButtonScrollArrowDecSkinStyleDefault);
ScrollBarElement.VButtonScrollArrowDecStyleDefault.setStyle("DownSkinStyle", 				ScrollBarElement.VButtonScrollArrowDecSkinStyleDefault);
ScrollBarElement.VButtonScrollArrowDecStyleDefault.setStyle("DisabledSkinStyle", 			ScrollBarElement.VButtonScrollArrowDecSkinStyleDefault);

//arrow button (horizontal increment)
ScrollBarElement.HButtonScrollArrowIncSkinStyleDefault = new StyleDefinition();
ScrollBarElement.HButtonScrollArrowIncSkinStyleDefault.setStyle("ArrowDirection", 			"right");

ScrollBarElement.HButtonScrollArrowIncStyleDefault = new StyleDefinition();
ScrollBarElement.HButtonScrollArrowIncStyleDefault.setStyle("PercentHeight", 				100);
ScrollBarElement.HButtonScrollArrowIncStyleDefault.setStyle("UpSkinStyle", 					ScrollBarElement.HButtonScrollArrowIncSkinStyleDefault);
ScrollBarElement.HButtonScrollArrowIncStyleDefault.setStyle("OverSkinStyle", 				ScrollBarElement.HButtonScrollArrowIncSkinStyleDefault);
ScrollBarElement.HButtonScrollArrowIncStyleDefault.setStyle("DownSkinStyle", 				ScrollBarElement.HButtonScrollArrowIncSkinStyleDefault);
ScrollBarElement.HButtonScrollArrowIncStyleDefault.setStyle("DisabledSkinStyle", 			ScrollBarElement.HButtonScrollArrowIncSkinStyleDefault);

//arrow button (horizontal decrement)
ScrollBarElement.HButtonScrollArrowDecSkinStyleDefault = new StyleDefinition();
ScrollBarElement.HButtonScrollArrowDecSkinStyleDefault.setStyle("ArrowDirection", 			"left");

ScrollBarElement.HButtonScrollArrowDecStyleDefault = new StyleDefinition();
ScrollBarElement.HButtonScrollArrowDecStyleDefault.setStyle("PercentHeight", 				100);
ScrollBarElement.HButtonScrollArrowDecStyleDefault.setStyle("UpSkinStyle", 					ScrollBarElement.HButtonScrollArrowDecSkinStyleDefault);
ScrollBarElement.HButtonScrollArrowDecStyleDefault.setStyle("OverSkinStyle", 				ScrollBarElement.HButtonScrollArrowDecSkinStyleDefault);
ScrollBarElement.HButtonScrollArrowDecStyleDefault.setStyle("DownSkinStyle", 				ScrollBarElement.HButtonScrollArrowDecSkinStyleDefault);
ScrollBarElement.HButtonScrollArrowDecStyleDefault.setStyle("DisabledSkinStyle", 			ScrollBarElement.HButtonScrollArrowDecSkinStyleDefault);


//////TAB

//Applied dynamically based on LayoutDirection (vertical)
ScrollBarElement.VButtonTabStyleDefault = new StyleDefinition();
ScrollBarElement.VButtonTabStyleDefault.setStyle("MinWidth", 						15);
ScrollBarElement.VButtonTabStyleDefault.setStyle("MinHeight", 						30);
ScrollBarElement.VButtonTabStyleDefault.setStyle("PercentWidth", 					100);

//Applied dynamically based on LayoutDirection (horizontal)
ScrollBarElement.HButtonTabStyleDefault = new StyleDefinition();
ScrollBarElement.HButtonTabStyleDefault.setStyle("MinWidth", 						30);
ScrollBarElement.HButtonTabStyleDefault.setStyle("MinHeight", 						15);
ScrollBarElement.HButtonTabStyleDefault.setStyle("PercentHeight", 					100);


//////ROOT SCROLLBAR

ScrollBarElement.StyleDefault = new StyleDefinition();
ScrollBarElement.StyleDefault.setStyle("ScrollTweenDuration", 						180); 			// number (milliseconds)
ScrollBarElement.StyleDefault.setStyle("LayoutDirection", 							"vertical");	// "vertical" || "horizontal"
ScrollBarElement.StyleDefault.setStyle("ClipContent", 								false);
ScrollBarElement.StyleDefault.setStyle("LayoutGap", 								-1); //Collapse borders
ScrollBarElement.StyleDefault.setStyle("LayoutHorizontalAlign", 					"center");
ScrollBarElement.StyleDefault.setStyle("LayoutVerticalAlign", 						"middle"); 
ScrollBarElement.StyleDefault.setStyle("ButtonTrackStyle", 							ScrollBarElement.ButtonTrackStyleDefault);
ScrollBarElement.StyleDefault.setStyle("ButtonIncrementStyle", 						ScrollBarElement.ButtonScrollArrowStyleDefault); 
ScrollBarElement.StyleDefault.setStyle("ButtonDecrementStyle", 						ScrollBarElement.ButtonScrollArrowStyleDefault);

//Applied dynamically based on LayoutDirection
//ScrollBarElement.StyleDefault.setStyle("ButtonTabStyle", 							ScrollBarElement.ButtonTabStyleDefault); 



/////////////ScrollBarElement Public Functions///////////////////

/**
 * @function setScrollPageSize
 * Sets the total number of scroll lines.
 * 
 * @param pageSize int
 * The total number of scroll lines.
 */
ScrollBarElement.prototype.setScrollPageSize = 
	function (pageSize)
	{
		if (this._scrollPageSize == pageSize)
			return;
	
		this._scrollPageSize = pageSize;
		this._invalidateLayout();
	};

/**
 * @function getScrollPageSize
 * Gets the total number of scroll lines.
 * 
 * @returns int
 * The total number of scroll lines.
 */	
ScrollBarElement.prototype.getScrollPageSize = 
	function ()
	{
		return this._scrollPageSize;
	};
	
/**
 * @function setScrollViewSize
 * Sets the number of scroll lines that fit within the view.
 * 
 * @param viewSize int
 * The number of scroll lines that fit within the view.
 */	
ScrollBarElement.prototype.setScrollViewSize = 
	function (viewSize)
	{
		if (this._scrollViewSize == viewSize)
			return;
		
		this._scrollViewSize = viewSize;
		this._invalidateLayout();
	};
	
/**
 * @function getScrollViewSize
 * Gets the number of scroll lines that fit within the view.
 * 
 * @returns int
 * The number of scroll lines that fit within the view.
 */	
ScrollBarElement.prototype.getScrollViewSize = 
	function ()
	{
		return this._scrollViewSize;
	};
	
/**
 * @function setScrollLineSize
 * Sets the number of lines to scroll when a scroll button is pressed.
 * 
 * @param lineSize int
 * The number of lines to scroll when a scroll button is pressed.
 */	
ScrollBarElement.prototype.setScrollLineSize = 
	function (lineSize)
	{
		this._scrollLineSize = lineSize;
	};		
	
/**
 * @function getScrollLineSize
 * Gets the number of lines to scroll when a scroll button is pressed.
 * 
 * @returns int
 * The number of lines to scroll when a scroll button is pressed.
 */	
ScrollBarElement.prototype.getScrollLineSize = 
	function ()
	{
		return this._scrollLineSize;
	};
	
/**
 * @function setScrollValue
 * Sets the position to scroll too. Range is 0 to (page size - view size).
 * 
 * @param value int
 * The position to scroll too.
 */	
ScrollBarElement.prototype.setScrollValue = 
	function (value)
	{
		if (this._scrollValue == value)
			return;
		
		this._scrollValue = value;
		this._invalidateLayout();
	};

/**
 * @function getScrollValue
 * Gets the scroll position.  Range is 0 to (page size - view size).
 * 
 * @returns int
 * The scroll position.
 */	
ScrollBarElement.prototype.getScrollValue = 
	function ()
	{
		return this._scrollValue;
	};

/**
 * @function startScrollTween
 * Starts a tween animation to scroll bar to the supplied scroll position.
 * 
 * @param tweenToValue int
 * The position to scroll too.
 */	
ScrollBarElement.prototype.startScrollTween = 
	function (tweenToValue)
	{
		var tweenDuration = this.getStyle("ScrollTweenDuration");
		if (tweenDuration > 0)
		{
			if (this._scrollTween == null)
			{
				this._scrollTween = new Tween();
				this._scrollTween.startVal = this._scrollValue;
				this._scrollTween.endVal = tweenToValue;
				this._scrollTween.duration = tweenDuration;
				this._scrollTween.startTime = Date.now();
				this._scrollTween.easingFunction = Tween.easeInOutSine;
				
				this.addEventListener("enterframe", this._onScrollBarEnterFrameInstance);
			}
			else
			{
				this._scrollTween.startVal = this._scrollValue;
				this._scrollTween.endVal = tweenToValue;
				this._scrollTween.startTime = Date.now();
				this._scrollTween.easingFunction = Tween.easeOutSine;
			}
		}
		else
		{
			this.endScrollTween();
			this.setScrollValue(tweenToValue);
			this.dispatchEvent(new ElementEvent("changed", false));
		}
	};
	
/**
 * @function endScrollTween
 * Ends the scroll tween animation. Immediately moves the scroll position to
 * the ending position if the tween is still running.
 */		
ScrollBarElement.prototype.endScrollTween = 
	function ()
	{
		if (this._scrollTween != null)
		{
			this.setScrollValue(this._scrollTween.endVal);
			this.removeEventListener("enterframe", this._onScrollBarEnterFrameInstance);
			this._scrollTween = null;
		}
	};	
	
/**
 * @function getTweenToValue
 * Gets the scroll position being tweened too.
 * 
 * @returns int
 * The scroll position beeing tweened too or null if no tween is running.
 */	
ScrollBarElement.prototype.getTweenToValue = 
	function ()
	{
		if (this._scrollTween == null)
			return null;
		
		return this._scrollTween.endVal;
	};
	
/////////////ScrollBarElement Internal Functions///////////////////

//@private container doesnt measure need to be notified by track & tab buttons	
ScrollBarElement.prototype._onTrackAndTabContainerMeasureComplete =
	function (event)
	{
		this._invalidateMeasure();
		this._invalidateLayout();
	};
	
//@private - only active when a tween is running.
ScrollBarElement.prototype._onScrollBarEnterFrame = 
	function (event)
	{
		var scrollValue = this._scrollTween.getValue(Date.now());
		
		if (scrollValue == this._scrollTween.endVal)
			this.endScrollTween();
		else
			this.setScrollValue(scrollValue);
		
		this.dispatchEvent(new ElementEvent("changed", false));
	};
	
//@Override	
ScrollBarElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		ScrollBarElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		this.endScrollTween();
	};		
	
/**
 * @function _onScrollButtonClick
 * Event handler for Buttons (increment, decrement, and track) "click" event. 
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
ScrollBarElement.prototype._onScrollButtonClick = 
	function (elementMouseEvent)
	{
		var incrementSize = null;
		
		var startScrollValue = this._scrollValue;
		if (this._scrollTween != null)
			startScrollValue = this._scrollTween.endVal;
		
		startScrollValue = Math.min(this._scrollPageSize - this._scrollViewSize, startScrollValue);
		startScrollValue = Math.max(0, startScrollValue);
		
		if (elementMouseEvent.getTarget() == this._buttonIncrement || 
			elementMouseEvent.getTarget() == this._buttonDecrement)
		{
			incrementSize = this.getScrollLineSize();
			
			if (elementMouseEvent.getTarget() == this._buttonDecrement)
				incrementSize = incrementSize * -1;
		}
		else if (elementMouseEvent.getTarget() == this._buttonTrack)
		{
			incrementSize = this._scrollViewSize * .75;
			
			if (this.getStyle("LayoutDirection") == "horizontal")
			{
				if (elementMouseEvent.getX() <= this._buttonTab._x + (this._buttonTab._width / 2))
					incrementSize = incrementSize * -1;
			}
			else //vertical
			{
				if (elementMouseEvent.getY() <= this._buttonTab._y + (this._buttonTab._height / 2))
					incrementSize = incrementSize * -1;
			}
		}
		
		var endScrollValue = startScrollValue + incrementSize;
		
		endScrollValue = Math.min(this._scrollPageSize - this._scrollViewSize, endScrollValue);
		endScrollValue = Math.max(0, endScrollValue);
		
		if (endScrollValue != startScrollValue)
			this.startScrollTween(endScrollValue);
	};

/**
 * @function _onScrollTabDrag
 * Event handler for Tab Button's "dragging" event. 
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
ScrollBarElement.prototype._onScrollTabDrag = 
	function (elementEvent)
	{
		var tabPosition = null;
		var trackSize = null;
		var tabSize = null;
		
		var direction = this.getStyle("LayoutDirection");
		var oldScrollValue = this._scrollValue;
		
		if (direction == "horizontal")
		{
			trackSize = this._buttonTrack._width;
			tabPosition = this._buttonTab._x - this._buttonTrack._x;
			tabSize = this._buttonTab._width;
		}
		else
		{
			trackSize = this._buttonTrack._height;
			tabPosition = this._buttonTab._y - this._buttonTrack._y;
			tabSize = this._buttonTab._height;
		}
		
		//Correct position
		if (tabPosition > trackSize - tabSize)
			tabPosition = trackSize - tabSize;
		if (tabPosition < 0)
			tabPosition = 0;
		
		trackSize = trackSize - tabSize;
		
		//Calculate new ScrollValue
		var scrollRange = this._scrollPageSize - this._scrollViewSize;
		var pixelsPerScaleUnit = trackSize / scrollRange;
		
		var newScrollValue = (tabPosition / pixelsPerScaleUnit);
		if (oldScrollValue != newScrollValue)
		{
			this.setScrollValue(newScrollValue);
			this.dispatchEvent(new ElementEvent("changed", false));
		}
		
		//Always invalidate layout, need to correct drag position.
		this._invalidateLayout();
	};

//@override
ScrollBarElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ScrollBarElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		//////Create Elements//////
		if (this._buttonDecrement == null)
		{
			this._buttonDecrement = new ButtonElement();
			this._buttonDecrement.addEventListener("click", this._onScrollButtonClickInstance);
			this.addElementAt(this._buttonDecrement, 0);
		}
		
		if (this._buttonTrack == null)
		{
			this._buttonTrack = new ButtonElement();
			this._buttonTrack.addEventListener("click", this._onScrollButtonClickInstance);
			this._trackAndTabContainer._addChild(this._buttonTrack);
		}
		
		if (this._buttonTab == null)
		{
			this._buttonTab = new ButtonElement();
			this._buttonTab.setStyle("Draggable", true);
			this._buttonTab.addEventListener("dragging", this._onScrollTabDragInstance);
			this._trackAndTabContainer._addChild(this._buttonTab);
		}
		
		if (this._buttonIncrement == null)
		{
			this._buttonIncrement = new ButtonElement();
			this._buttonIncrement.addEventListener("click", this._onScrollButtonClickInstance);
			this.addElementAt(this._buttonIncrement, this.getNumElements());
		}
		
		if ("LayoutDirection" in stylesMap)
		{
			this._invalidateMeasure();
			this._invalidateLayout();
		}
		else if ("Enabled" in stylesMap)
			this._invalidateLayout();
		

		var layoutDirection = this.getStyle("LayoutDirection");
		
		//We need to inject the default styles specific to LayoutDirection before other styling.
		if ("LayoutDirection" in stylesMap || "ButtonDecrementStyle" in stylesMap)
		{
			this._applySubStylesToElement("ButtonDecrementStyle", this._buttonDecrement);
			
			if (layoutDirection == "horizontal")
				this._buttonDecrement._addStyleDefinitionAt(ScrollBarElement.HButtonScrollArrowDecStyleDefault, 0, true);
			else
				this._buttonDecrement._addStyleDefinitionAt(ScrollBarElement.VButtonScrollArrowDecStyleDefault, 0, true);
		}
		
		if ("LayoutDirection" in stylesMap || "ButtonTrackStyle" in stylesMap)
		{
			this._applySubStylesToElement("ButtonTrackStyle", this._buttonTrack);
			
			if (layoutDirection == "horizontal")
				this._buttonTrack._addStyleDefinitionAt(ScrollBarElement.HButtonTrackStyleDefault, 0, true);
			else
				this._buttonTrack._addStyleDefinitionAt(ScrollBarElement.VButtonTrackStyleDefault, 0, true);
		}
		
		if ("LayoutDirection" in stylesMap || "ButtonTabStyle" in stylesMap)
		{
			this._applySubStylesToElement("ButtonTabStyle", this._buttonTab);
			
			if (layoutDirection == "horizontal")
				this._buttonTab._addStyleDefinitionAt(ScrollBarElement.HButtonTabStyleDefault, 0, true);
			else
				this._buttonTab._addStyleDefinitionAt(ScrollBarElement.VButtonTabStyleDefault, 0, true);
		}
		
		if ("LayoutDirection" in stylesMap || "ButtonIncrementStyle" in stylesMap)
		{
			this._applySubStylesToElement("ButtonIncrementStyle", this._buttonIncrement);
			
			if (layoutDirection == "horizontal")
				this._buttonIncrement._addStyleDefinitionAt(ScrollBarElement.HButtonScrollArrowIncStyleDefault, 0, true);
			else
				this._buttonIncrement._addStyleDefinitionAt(ScrollBarElement.VButtonScrollArrowIncStyleDefault, 0, true);
		}
		
		if ("ScrollTweenDuration" in stylesMap && this.getStyle("ScrollTweenDuration") == 0)
			this.endScrollTween();
	};
	
	
//@override
ScrollBarElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		//Get the ListContainer measured height
	
		//TODO: Fix this, its not efficient to changed measured size twice, always forces layout
		//		even if the measured size doesn't actually change.
		ScrollBarElement.base.prototype._doMeasure.call(this, padWidth, padHeight);
	
		var measuredWidth = this._measuredWidth;
		var measuredHeight = this._measuredHeight;
		
		//Account for the tab and track (container doesnt measure)
		
		//TODO: Handle rotation of tab?? 
		
		if (this.getStyle("LayoutDirection") == "vertical")
		{
			var tabMinHeight = this._buttonTab.getStyle("MinHeight");
			var trackWidth = this._buttonTrack._getStyledOrMeasuredWidth() + padWidth;
			var tabWidth = this._buttonTab._getStyledOrMeasuredWidth() + padWidth;
			
			measuredHeight += (tabMinHeight * 2);
			
			if (tabWidth > measuredWidth)
				measuredWidth = tabWidth;
			if (trackWidth > measuredWidth)
				measuredWidth = trackWidth;
		}
		else //horizontal
		{
			var tabMinWidth = this._buttonTab.getStyle("MinWidth");
			var tabHeight = this._buttonTab._getStyledOrMeasuredHeight() + padHeight;
			var trackHeight = this._buttonTrack._getStyledOrMeasuredHeight() + padHeight;
			
			measuredWidth += (tabMinWidth * 2);
			
			if (tabHeight > measuredHeight)
				measuredHeight = tabHeight;
			if (trackHeight > measuredHeight)
				measuredHeight = trackHeight;
		}
		
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};	
	
//@Override	
ScrollBarElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		ScrollBarElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		//Correct the scroll value (size reduction forces us to scroll up)
		this._scrollValue = Math.min(this._scrollValue, this._scrollPageSize - this._scrollViewSize);
		this._scrollValue = Math.max(this._scrollValue, 0);
		
		//Disable / Enable components
		if (this._scrollViewSize >= this._scrollPageSize || this.getStyle("Enabled") == false)
		{
			this._buttonIncrement.setStyle("Enabled", false);
			this._buttonDecrement.setStyle("Enabled", false);
			this._buttonTrack.setStyle("Enabled", false);
			this._buttonTab.setStyle("Visible", false);
		}
		else
		{
			this._buttonIncrement.clearStyle("Enabled");
			this._buttonDecrement.clearStyle("Enabled");
			this._buttonTrack.clearStyle("Enabled");
			this._buttonTab.clearStyle("Visible");
		}
		
		var availableTrackSize;
		var pixelsPerScaleUnit = 0;
		
		//TODO: Handle rotation of tab??
		var tabWidth = this._buttonTab.getStyle("Width");
		var tabMinWidth = this._buttonTab.getStyle("MinWidth");
		var tabMaxWidth = this._buttonTab.getStyle("MaxWidth");
		var tabPWidth = this._buttonTab.getStyle("PercentWidth");
		
		if (tabMinWidth == null)
			tabMinWidth = 0;
		if (tabMaxWidth == null)
			tabMaxWidth = Number.MAX_VALUE;
		
		var tabHeight = this._buttonTab.getStyle("Height");
		var tabMinHeight = this._buttonTab.getStyle("MinHeight");
		var tabMaxHeight = this._buttonTab.getStyle("MaxHeight");
		var tabPHeight = this._buttonTab.getStyle("PercentHeight");
		
		if (tabMinHeight == null)
			tabMinHeight = 0;
		if (tabMaxHeight == null)
			tabMaxHeight = Number.MAX_VALUE;
		
		var trackWidth = this._buttonTrack.getStyle("Width");
		var trackMinWidth = this._buttonTrack.getStyle("MinWidth");
		var trackMaxWidth = this._buttonTrack.getStyle("MaxWidth");
		var trackPWidth = this._buttonTrack.getStyle("PercentWidth");		
		
		if (trackMinWidth == null)
			trackMinWidth = 0;
		if (trackMaxWidth == null)
			trackMaxWidth = Number.MAX_VALUE;
		
		var trackHeight = this._buttonTrack.getStyle("Height");
		var trackMinHeight = this._buttonTrack.getStyle("MinHeight");
		var trackMaxHeight = this._buttonTrack.getStyle("MaxHeight");
		var trackPHeight = this._buttonTrack.getStyle("PercentHeight");
		
		if (trackMinHeight == null)
			trackMinHeight = 0;
		if (trackMaxHeight == null)
			trackMaxHeight = Number.MAX_VALUE;
		
		//Size and position the track and tab (their parent container doesnt layout or measure)
		var direction = this.getStyle("LayoutDirection");
		if (direction == "vertical")
		{
			if (tabHeight == null)
			{
				if (this._scrollPageSize > 0)
					tabHeight = Math.round(this._trackAndTabContainer._height * (this._scrollViewSize / this._scrollPageSize));
				else
					tabHeight = 0;
				
				tabHeight = Math.min(tabHeight, tabMaxHeight);
				tabHeight = Math.max(tabHeight, tabMinHeight);
			}
			
			var tabActualWidth = tabWidth;
			if (tabActualWidth == null)
			{
				if (tabPWidth != null)
					tabActualWidth = Math.round(this._trackAndTabContainer._width * (tabPWidth / 100));
				
				if (tabActualWidth == null)
					tabActualWidth = tabMinWidth;
				
				tabActualWidth = Math.min(tabActualWidth, tabMaxWidth);
				tabActualWidth = Math.max(tabActualWidth, tabMinWidth);
			}
			
			var trackActualWidth = trackWidth;
			if (trackActualWidth == null)
			{
				if (trackPWidth != null)
					trackActualWidth = Math.round(this._trackAndTabContainer._width * (trackPWidth / 100));
				
				if (trackActualWidth == null)
					trackActualWidth = trackMinWidth;
				
				trackActualWidth = Math.min(tabActualWidth, trackMaxWidth);
				trackActualWidth = Math.max(tabActualWidth, trackMinWidth);
			}
			
			if (this._scrollPageSize > this._scrollViewSize)
			{
				availableTrackSize = this._trackAndTabContainer._height - tabHeight;
				pixelsPerScaleUnit = availableTrackSize / (this._scrollPageSize - this._scrollViewSize);
			}
			
			this._buttonTrack._setActualSize(trackActualWidth, this._trackAndTabContainer._height);
			this._buttonTab._setActualSize(tabActualWidth, tabHeight);
			
			var hAlign = this.getStyle("LayoutHorizontalAlign");
			if (hAlign == "left")
			{
				this._buttonTrack._setActualPosition(0, 0);
				this._buttonTab._setActualPosition(0, Math.round(this._scrollValue * pixelsPerScaleUnit));
			}
			else if (hAlign == "center")
			{
				this._buttonTrack._setActualPosition(Math.round((this._trackAndTabContainer._width / 2) - (this._buttonTrack._width / 2)), 0);
				this._buttonTab._setActualPosition(Math.round((this._trackAndTabContainer._width / 2) - (this._buttonTab._width / 2)), Math.round(this._scrollValue * pixelsPerScaleUnit));
			}
			else //right
			{
				this._buttonTrack._setActualPosition(this._trackAndTabContainer._width - this._buttonTrack._width, 0);
				this._buttonTab._setActualPosition(this._trackAndTabContainer._width - this._buttonTab._width, Math.round(this._scrollValue * pixelsPerScaleUnit));
			}
		}
		else //horizontal
		{
			if (tabWidth == null)
			{
				if (this._scrollPageSize > 0)
					tabWidth = Math.round(this._trackAndTabContainer._width * (this._scrollViewSize / this._scrollPageSize));
				else
					tabWidth = 0;
				
				tabWidth = Math.min(tabWidth, tabMaxWidth);
				tabWidth = Math.max(tabWidth, tabMinWidth);
			}
			
			var tabActualHeight = tabHeight;
			if (tabActualHeight == null)
			{
				if (tabPHeight != null)
					tabActualHeight = Math.round(this._trackAndTabContainer._height * (tabPHeight / 100));
				
				if (tabActualHeight == null)
					tabActualHeight = tabMinHeight;
				
				tabActualHeight = Math.min(tabActualHeight, tabMaxHeight);
				tabActualHeight = Math.max(tabActualHeight, tabMinHeight);
			}
			
			var trackActualHeight = trackHeight;
			if (trackActualHeight == null)
			{
				if (trackPHeight != null)
					trackActualHeight = Math.round(this._trackAndTabContainer._height * (trackPHeight / 100));
				
				if (trackActualHeight == null)
					trackActualHeight = trackMinHeight;
				
				trackActualHeight = Math.min(tabActualHeight, trackMaxHeight);
				trackActualHeight = Math.max(tabActualHeight, trackMinHeight);
			}
			
			if (this._scrollPageSize > this._scrollViewSize)
			{
				availableTrackSize = this._trackAndTabContainer._width - tabWidth;
				pixelsPerScaleUnit = availableTrackSize / (this._scrollPageSize - this._scrollViewSize);
			}
			
			this._buttonTrack._setActualSize(this._trackAndTabContainer._width, trackActualHeight);
			this._buttonTab._setActualSize(tabWidth,tabActualHeight);
			
			var vAlign = this.getStyle("LayoutVerticalAlign");
			if (vAlign == "top")
			{
				this._buttonTrack._setActualPosition(0, 0);
				this._buttonTab._setActualPosition(Math.round(this._scrollValue * pixelsPerScaleUnit), 0);
			}
			else if (vAlign == "middle")
			{
				this._buttonTrack._setActualPosition(0, Math.round((this._trackAndTabContainer._height / 2) - (this._buttonTrack._height / 2)));
				this._buttonTab._setActualPosition(Math.round(this._scrollValue * pixelsPerScaleUnit), Math.round((this._trackAndTabContainer._height / 2) - (this._buttonTab._height / 2)));
			}
			else //bottom
			{
				this._buttonTrack._setActualPosition(0, this._trackAndTabContainer._height - this._buttonTrack._height);
				this._buttonTab._setActualPosition(Math.round(this._scrollValue * pixelsPerScaleUnit), this._trackAndTabContainer._height - this._buttonTab._height);
			}
		}
	};	
	
	

/**
 * @depends ContainerBaseElement.js
 */

///////////////////////////////////////////////////////////////////////////	
///////////////////////GridContainerElement////////////////////////////////

/**
 * @class GridContainerElement
 * @inherits ContainerBaseElement
 * 
 * GridContainerElement is used to layout children in a grid or table-like manner.
 * This is effectively a two dimensional ListContainerElement that also supports
 * elements spanning multiple rows and/or columns. 
 * 
 * The grid container does not respect any layout styling of cell elements (other
 * than measured/min sizes), rather it only respects layout styling of supplied row and
 * column definitions (GridContainerRowColumnDefinition). It is recommended that
 * other containers (such as ListContainerElements) are used for the cell elements
 * that wrap the content elements being added to the GridContainerElement. This way the 
 * cells may stretch as necessary and content elements can be independently aligned
 * within the cell without stretching the content element itself.
 * 
 * The GridContainerElement is more expensive than a ListContainerElement so should 
 * only be used when it is needed to maintain row/column alignment.
 * 
 * @constructor GridContainerElement 
 * Creates new GridContainerElement instance.
 */
function GridContainerElement()
{
	GridContainerElement.base.prototype.constructor.call(this);
	
	this._rowDefinitions = Object.create(null);		//Map by rowIndex
	this._columnDefinitions = Object.create(null);	//Map by columnIndex
	
	//Since the same definition may be used on multiple rows & columns
	//This stores the number of map entries by GridContainerRowColumnDefinition
	this._rowColumnDefinitionCount = [];			//{definition, count}
	
	this._gridCells = [];  							//{element, rowIndexStart, columnIndexStart, rowIndexEnd, columnIndexEnd} 
	
	var _self = this;
	
	this._onRowColumnDefinitionChangedInstance = 
		function (event)
		{
			_self._onRowColumnDefinitionChanged(event);
		};
}	
	
//Inherit from CanvasElement
GridContainerElement.prototype = Object.create(ContainerBaseElement.prototype);
GridContainerElement.prototype.constructor = GridContainerElement;
GridContainerElement.base = ContainerBaseElement;


/////////////Style Types///////////////////////////////

GridContainerElement._StyleTypes = Object.create(null);

/**
 * @style ClipContent boolean
 * @inheritable
 * 
 * Determines if out of bounds rendering is allowed. If true the element will clip all rendering
 * and children's rendering to the elements bounding box. This style is inheritable for container elements.
 */
GridContainerElement._StyleTypes.ClipContent = 				StyleableBase.EStyleType.INHERITABLE;		// number (true || false)

/**
 * @style LayoutVerticalGap int
 * 
 * Space in pixels to leave between rows.
 */
GridContainerElement._StyleTypes.LayoutVerticalGap = 		StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style LayoutHorizontalGap int
 * 
 * Space in pixels to leave between columns.
 */
GridContainerElement._StyleTypes.LayoutHorizontalGap = 		StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style LayoutVerticalAlign String
 * 
 * Vertical alignment to be used when the grid does not fill all available space. Allowable values are "top", "bottom", or "middle". 
 */
GridContainerElement._StyleTypes.LayoutVerticalAlign = 		StyleableBase.EStyleType.NORMAL;		// "top" || "bottom" || "middle" 

/**
 * @style LayoutHorizontalAlign String
 * 
 * Horizontal alignment to be used when the grid does not fill all available space. Allowable values are "left", "right", or "center". 
 */
GridContainerElement._StyleTypes.LayoutHorizontalAlign = 	StyleableBase.EStyleType.NORMAL;		//"left" || "right" || "center"


////////////Default Styles////////////////////////////

GridContainerElement.StyleDefault = new StyleDefinition();

GridContainerElement.StyleDefault.setStyle("LayoutVerticalGap", 				0);
GridContainerElement.StyleDefault.setStyle("LayoutHorizontalGap", 				0);
GridContainerElement.StyleDefault.setStyle("LayoutVerticalAlign", 				"top");
GridContainerElement.StyleDefault.setStyle("LayoutHorizontalAlign", 			"left");


////////PUBLIC///////////////////////

/**
 * @function setRowDefinition
 * Sets a row definition to the supplied row index.
 * 
 * @param definition GridContainerRowColumnDefinition
 * The definition to apply to the supplied row index.
 * 
 * @param index int
 * The row index to apply the definition.
 */
GridContainerElement.prototype.setRowDefinition = 
	function (definition, index)
	{
		var mapKey = Number(index).toString();
		var oldDefinition = this._rowDefinitions[mapKey];
		
		if (oldDefinition == definition)
			return;
		
		if (definition != null)
			this._rowDefinitions[mapKey] = definition;
		else
			delete this._rowDefinitions[mapKey];
		
		this._removeRowColumnDefinition(oldDefinition);
		this._addRowColumnDefinition(definition);
		
		this._invalidateMeasure();
		this._invalidateLayout();
	};

/**
 * @function getRowDefinition
 * Gets the row definition associated with the supplied row index.
 * 
 * @param index int
 * The row index to return the associated GridContainerRowColumnDefinition.
 * 
 * @returns GridContainerRowColumnDefinition
 * The GridContainerRowColumnDefinition associated with the supplied row index or null.
 */	
GridContainerElement.prototype.getRowDefinition = 
	function (index)
	{
		var mapKey = Number(index).toString();
		if (mapKey in this._rowDefinitions)
			return this._rowDefinitions[mapKey];
		
		return null;
	};

/**
 * @function clearRowDefinitions
 * Clears all row definitions.
 */		
GridContainerElement.prototype.clearRowDefinitions = 
	function ()
	{
		for (var mapKey in this._rowDefinitions)
			this._removeRowColumnDefinition(this._rowDefinitions[mapKey]);
		
		this._rowDefinitions = Object.create(null);		
		
		this._invalidateMeasure();
		this._invalidateLayout();
	};	
	
/**
 * @function setColumnDefinition
 * Sets a column definition to the supplied column index.
 * 
 * @param definition GridContainerRowColumnDefinition
 * The definition to apply to the supplied column index.
 * 
 * @param index int
 * The column index to apply the definition.
 */	
GridContainerElement.prototype.setColumnDefinition = 
	function (definition, index)
	{
		var mapKey = Number(index).toString();
		var oldDefinition = this._columnDefinitions[mapKey];
		
		if (oldDefinition == definition)
			return;
		
		if (definition != null)
			this._columnDefinitions[mapKey] = definition;
		else
			delete this._columnDefinitions[mapKey];
		
		this._removeRowColumnDefinition(oldDefinition);
		this._addRowColumnDefinition(definition);
		
		this._invalidateMeasure();
		this._invalidateLayout();
	};
	
/**
 * @function getColumnDefinition
 * Gets the column definition associated with the supplied column index.
 * 
 * @param index int
 * The column index to return the associated GridContainerRowColumnDefinition.
 * 
 * @returns GridContainerRowColumnDefinition
 * The GridContainerRowColumnDefinition associated with the supplied column index or null.
 */		
GridContainerElement.prototype.getColumnDefinition = 
	function (index)
	{
		var mapKey = Number(index).toString();
		if (mapKey in this._columnDefinitions)
			return this._columnDefinitions[mapKey];
		
		return null;
	};
	
/**
 * @function clearColumnDefinitions
 * Clears all column definitions.
 */		
GridContainerElement.prototype.clearColumnDefinitions = 
	function ()
	{
		for (var mapKey in this._columnDefinitions)
			this._removeRowColumnDefinition(this._columnDefinitions[mapKey]);
		
		this._columnDefinitions = Object.create(null);		
		
		this._invalidateMeasure();
		this._invalidateLayout();
	};	
	
/**
 * @function setCellElement
 * Sets an element to be used for the supplied row and column indexes
 * or spans. If any element(s) already occupies the supplied
 * row or column spans, they will be removed.
 * 
 * @param element CanvasElement
 * The element to insert into grid spanning the supplied row and column indexes.
 * 
 * @param rowIndexStart int
 * The row index to insert the element.
 * 
 * @param columnIndexStart int
 * The column index to insert the element.
 * 
 * @param rowIndexEnd int
 * Optional - If the element is to span multiple rows this is the last
 * row index (inclusive) the element should occupy. If omitted this will
 * be automatically set to rowIndexStart. 
 * 
 * @param columnIndexEnd int
 * Optional - If the element is to span multiple columns this is the last
 * column index (inclusive) the element should occupy. If omitted this will
 * be automatically set to columnIndexStart.
 */		
GridContainerElement.prototype.setCellElement = 
	function (element, rowIndexStart, columnIndexStart, rowIndexEnd, columnIndexEnd)
	{
		if (rowIndexEnd == null)
			rowIndexEnd = rowIndexStart;
		
		if (columnIndexEnd == null)
			columnIndexEnd = columnIndexStart;
	
		var swap;
		if (rowIndexStart > rowIndexEnd)
		{
			swap = rowIndexEnd;
			rowIndexEnd = rowIndexStart;
			rowIndexStart = swap;
		}
		
		if (columnIndexStart > columnIndexEnd)
		{
			swap = columnIndexEnd;
			columnIndexEnd = columnIndexStart;
			columnIndexStart = swap;
		}
		
		//Purge any overlapping elements (or ourself if we're moving an element to a different cell)
		for (var i = this._gridCells.length - 1; i >= 0; i--)
		{
			if (this._gridCells[i].element == element || 
				(this._gridCells[i].rowIndexStart <= rowIndexEnd && this._gridCells[i].rowIndexEnd >= rowIndexStart && 
				this._gridCells[i].columnIndexStart <= columnIndexEnd && this._gridCells[i].columnIndexEnd >= columnIndexStart))
			{
				this.removeElement(this._gridCells[i].element);
				this._gridCells.splice(i, 1);
			}
		}
		
		if (element != null)
		{
			this._gridCells.push({element:element, 
								rowIndexStart:rowIndexStart, 
								columnIndexStart:columnIndexStart, 
								rowIndexEnd:rowIndexEnd, 
								columnIndexEnd:columnIndexEnd});
			
			this.addElement(element);
		}
	};
	
/**
 * @function getCellElement
 * Gets the CanvasElement associated with (or spanning) the supplied row and column indexes.
 * 
 * @param rowIndex int
 * The row index to return the associated CanvasElement.
 * 
 * @param columnIndex int
 * The column index to return the associated CanvasElement.
 * 
 * @returns CanvasElement
 * The CanvasElement associated with the supplied row and column index or null.
 */	
GridContainerElement.prototype.getCellElement = 
	function (rowIndex, columnIndex)
	{
		for (var i = 0; i < this._gridCells.length; i++)
		{
			if (rowIndex >= this._gridCells[i].rowIndexStart && rowIndex <= this._gridCells[i].rowIndexEnd &&
				columnIndex >= this._gridCells[i].columnIndexStart && columnIndex <= this._gridCells[i].columnIndexEnd)
			{
				return this._gridCells[i].element;
			}
		}
		
		return null;
	};
	
/**
 * @function clearCellElements
 * Clears all element from the GridContainerElement.
 */	
GridContainerElement.prototype.clearCellElements = 
	function ()
	{
		for (var i = 0; i < this._gridCells.length; i++)
			this.removeElement(this._gridCells[i].element);
		
		this._gridCells.length = 0;
	};
	

////////STATIC INTERNAL/////////////////
	
////Sort comparators for measure / layout array sorting////
	
//Sort by span and maxStretch
GridContainerElement._stretchSpanMaxStretchComparator = 
	function (dataA, dataB)
	{
		//ascending
		if (dataA.span < dataB.stretchSpan)
			return -1;
		else if (dataA.span > dataB.stretchSpan)
			return 1;
		
		//ascending
		if (dataA.maxStretch < dataB.maxStretch)
			return -1;
		else if (dataA.maxStretch > dataB.maxStretch)
			return 1;
		
		return 0;
	};
	
//Sort by stretch priority (descending) and actual size	
GridContainerElement._stretchPriorityActualSizeComparator = 
	function (dataA, dataB)
	{
		//descending
		if (dataA.stretchPriority > dataB.stretchPriority)
			return -1;
		else if (dataA.stretchPriority < dataB.stretchPriority)
			return 1;
	
		//ascending
		if (dataA.actualSize < dataB.actualSize)
			return -1;
		else if (dataA.actualSize > dataB.actualSize)
			return 1;

		return 0;
	};
	
//Sort by index	
GridContainerElement._indexComparator = 
	function (dataA, dataB)
	{
		//ascending
		if (dataA.index < dataB.index)
			return -1;
		else if (dataA.index > dataB.index)
			return 1;
		
		return 0;
	};	
	
	
////////INTERNAL/////////////////////
	
/**
 * @function _onRowColumnDefinitionChanged
 * Event handler for row and columns GridContainerRowColumnDefinition "stylechanged" event. 
 * Updates the GridContainerElement.
 * 
 * @param styleChangedEvent StyleChangedEvent
 * The StyleChangedEvent to process.
 */		
GridContainerElement.prototype._onRowColumnDefinitionChanged = 
	function (event)
	{
		this._invalidateMeasure();
		this._invalidateLayout();
	};

//@private	
GridContainerElement.prototype._addRowColumnDefinition = 
	function (definition)
	{
		if (definition == null)
			return;
		
		for (var i = 0; i < this._rowColumnDefinitionCount.length; i++)
		{
			if (this._rowColumnDefinitionCount[i].definition == definition)
			{
				this._rowColumnDefinitionCount[i].count++;
				return;
			}
		}
		
		this._rowColumnDefinitionCount.push({definition:definition, count:1});
		
		if (this._manager != null)
			definition.addEventListener("stylechanged", this._onRowColumnDefinitionChangedInstance);
	};
	
//@private	
GridContainerElement.prototype._removeRowColumnDefinition = 
	function (definition)
	{
		if (definition == null)
			return;
		
		for (var i = 0; i < this._rowColumnDefinitionCount.length; i++)
		{
			if (this._rowColumnDefinitionCount[i].definition == definition)
			{
				this._rowColumnDefinitionCount[i].count--;
				
				if (this._rowColumnDefinitionCount[i].count == 0)
				{
					this._rowColumnDefinitionCount.splice(i, 1);
					
					if (this._manager != null)
						definition.removeEventListener("stylechanged", this._onRowColumnDefinitionChangedInstance);
				}
				
				return;
			}
		}
	};	
	
//@override	
GridContainerElement.prototype._onCanvasElementAdded = 
	function (addedRemovedEvent)
	{
		GridContainerElement.base.prototype._onCanvasElementAdded.call(this, addedRemovedEvent);
		
		for (var i = 0; i < this._rowColumnDefinitionCount.length; i++)
			this._rowColumnDefinitionCount[i].definition.addEventListener("stylechanged", this._onRowColumnDefinitionChangedInstance);
	};
	
//@override	
GridContainerElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		GridContainerElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		for (var i = 0; i < this._rowColumnDefinitionCount.length; i++)
			this._rowColumnDefinitionCount[i].definition.removeEventListener("stylechanged", this._onRowColumnDefinitionChangedInstance);
	};		
	
//@private	
GridContainerElement.prototype._createRowOrColumnData = 
	function (index, isRow)
	{
		var data = {size:null,
					percentSize:null,
					minSize:null,
					maxSize:null,
					stretchPriority:0,
					index:index,
					actualSize:0,
					actualPosition:0};
	
		var map = (isRow == true ? this._rowDefinitions : this._columnDefinitions);
		var mapKey = index.toString();
		
		if (map[mapKey] != null)
		{
			data.size = map[mapKey].getStyle("Size");
			
			if (data.size == null) //Stretch-able row/column
			{
				data.percentSize = map[mapKey].getStyle("PercentSize");
				data.minSize = map[mapKey].getStyle("MinSize");
				data.maxSize = map[mapKey].getStyle("MaxSize");
				
				if (data.percentSize == null)
					data.stretchPriority = map[mapKey].getStyle("StretchPriority");
				else
					data.stretchPriority = Number.MAX_VALUE;
				
				if (data.minSize != null)
					data.actualSize = data.minSize;
			}
			else //Fixed size row/column
			{
				data.stretchPriority = null;
				data.actualSize = data.size;
			}
		}
	
		if (data.minSize == null)
			data.minSize = 0;
		if (data.maxSize == null)
			data.maxSize = Number.MAX_VALUE;
		
		return data;
	};
	
//@private	
GridContainerElement.prototype._getRowsAndColumnsMeasuredData = 
	function ()
	{
		var layoutVerticalGap = this.getStyle("LayoutVerticalGap");
		var layoutHorizontalGap = this.getStyle("LayoutHorizontalGap");
	
		//Cache for row/column layout data, we use both map for lookup and array for sorting
		var rowDataMap = Object.create(null);
		var columnDataMap = Object.create(null);
		
		//Create data for all styled rows and columns
		var mapKey;
		for (mapKey in this._rowDefinitions)
			rowDataMap[mapKey] = this._createRowOrColumnData(Number(mapKey), true);
	
		for (mapKey in this._columnDefinitions)
			columnDataMap[mapKey] = this._createRowOrColumnData(Number(mapKey), false);
	
		var i;
		var i2;
		var i3;
		var cellData;
		var cellRowData;
		var cellColumnData;
		
		var rowSpan;			//Number of stretch-able rows a cell spans
		var columnSpan;			//Number of stretch-able columns a cell spans
		var rowMaxStretch;		//Maximum stretch priority of rows that cell spans
		var columnMaxStretch;	//Maximum stretch priority of columns that cell spans
		
		//Sorted array of cells (2 priority sort, first by span, second by maxStretch)
		var cellsByRowSorted = [];
		var cellsByColumnSorted = [];
		
		//Scan the cells, record data for stretch-able rows/columns and sort them by span & maxStretch.
		for (i = 0; i < this._gridCells.length; i++)
		{
			cellData = this._gridCells[i];
			
			rowStretchSpan = 0;
			columnStretchSpan = 0;
			rowMaxStretch = null;
			columnMaxStretch = null;
			
			//Scan the rows this cell spans.
			for (i2 = cellData.rowIndexStart; i2 <= cellData.rowIndexEnd; i2++)
			{
				//Get row data / create if does not exist
				if (rowDataMap[i2] == null)
					rowDataMap[i2] = this._createRowOrColumnData(i2, true);
				
				//If this is stretch-able row, increment span and adjust maxStretch
				if (rowDataMap[i2].stretchPriority != null)
				{
					rowStretchSpan++;
					
					if (rowMaxStretch == null)
						rowMaxStretch = rowDataMap[i2].stretchPriority;
					else
						rowMaxStretch = Math.max(rowMaxStretch, rowDataMap[i2].stretchPriority);
				}
			}
			
			//Scan the columns this cell spans.
			for (i2 = cellData.columnIndexStart; i2 <= cellData.columnIndexEnd; i2++)
			{
				//Get column data / create if does not exist
				if (columnDataMap[i2] == null)
					columnDataMap[i2] = this._createRowOrColumnData(i2, false);
				
				//If this column is stretch-able, increment span and adjust maxStretch
				if (columnDataMap[i2].stretchPriority != null)
				{
					columnStretchSpan++;
					
					if (columnMaxStretch == null)
						columnMaxStretch = columnDataMap[i2].stretchPriority;
					else
						columnMaxStretch = Math.max(columnMaxStretch, columnDataMap[i2].stretchPriority);
				}
			}
			
			//Store cell row data if this cell has stretch-able row(s)
			if (rowStretchSpan > 0)
			{
				cellRowData = {startIndex:cellData.rowIndexStart,
								endIndex:cellData.rowIndexEnd,
								stretchSpan:rowStretchSpan,
								maxStretch:rowMaxStretch,
								minSize:Math.max(cellData.element.getStyle("MinHeight"), cellData.element._measuredHeight),
								span:cellData.rowIndexEnd - cellData.rowIndexStart};
				
				cellsByRowSorted.push(cellRowData);
			}
			
			//Store cell column data if this cell has stretch-able column(s)
			if (columnStretchSpan > 0)
			{
				cellColumnData = {startIndex:cellData.columnIndexStart,
									endIndex:cellData.columnIndexEnd,
									stretchSpan:columnStretchSpan,
									maxStretch:columnMaxStretch,
									minSize:Math.max(cellData.element.getStyle("MinWidth"), cellData.element._measuredWidth),
									span:cellData.columnIndexEnd - cellData.columnIndexStart};
				
				cellsByColumnSorted.push(cellColumnData);
			}
		}
		
		//Sort cell data by stretch span and max stretch
		cellsByRowSorted.sort(GridContainerElement._stretchSpanMaxStretchComparator);
		cellsByColumnSorted.sort(GridContainerElement._stretchSpanMaxStretchComparator);
		
		//Row/Column calculation is identical, so these represent either when we loop twice (once for rows, again for columns)
		var layoutGap;
		var rowColumnDataMap;
		var cellsByRowColumnSorted;
		
		var actualSize;
		var distributeCount; 		//Number of rows/columns to distribute size too
		var distributeSize;  		//Current size to be distributed across distributeCount columns/rows
		var distributeSizeTotal;	//Total size to be distributed
		var stretchRowsColumns = [];
		
		//Calculate measured sizes of stretch-able rows / columns
		for (i = 0; i < 2; i++) //Run twice, rows then columns
		{
			layoutGap = (i == 0 ? layoutVerticalGap : layoutHorizontalGap);
			rowColumnDataMap = (i == 0 ? rowDataMap : columnDataMap);
			cellsByRowColumnSorted = (i == 0 ? cellsByRowSorted : cellsByColumnSorted);
			
			//Scan over cells
			for (i2 = 0; i2 < cellsByRowColumnSorted.length; i2++)
			{
				actualSize = 0;
				stretchRowsColumns.length = 0;
				cellData = cellsByRowColumnSorted[i2];
				
				//Scan the rows/columns this cell is spanning
				for (i3 = cellData.startIndex; i3 <= cellData.endIndex; i3++)
				{
					//Ignore rows/columns that size is maxed or are not stretch-able (fixed size)
					if (rowColumnDataMap[i3].actualSize < rowColumnDataMap[i3].maxSize && rowColumnDataMap[i3].stretchPriority != null)
						stretchRowsColumns.push(rowColumnDataMap[i3]);
					
					actualSize += rowColumnDataMap[i3].actualSize;
				}
				
				//Sort row/column data by stretch priority then current size (we stretch equally among smallest)
				stretchRowsColumns.sort(GridContainerElement._stretchPriorityActualSizeComparator);
				
				//Record total size that needs to be distributed, adjust for layout gap
				distributeSizeTotal = cellData.minSize - actualSize - (layoutGap * cellData.span);
				
				//Stretch the rows/columns
				while (distributeSizeTotal >= 1 && stretchRowsColumns.length > 0)
				{
					distributeCount = 0;
					distributeSize = distributeSizeTotal;
					
					//Scan rows/columns that we need to stretch
					for (i3 = 0; i3 < stretchRowsColumns.length; i3++)
					{
						//Equal size and priority
						if (stretchRowsColumns[i3].actualSize == stretchRowsColumns[0].actualSize && 
							stretchRowsColumns[i3].stretchPriority == stretchRowsColumns[0].stretchPriority)
						{
							distributeCount++;
							
							//Reduce distribute size to limiting max size
							if ((stretchRowsColumns[i3].maxSize - stretchRowsColumns[i3].actualSize) * distributeCount < distributeSize)
								distributeSize = (stretchRowsColumns[i3].maxSize - stretchRowsColumns[i3].actualSize) * distributeCount;
						}
						else
						{
							//Same priority but size is larger, stretch to this size.
							if (stretchRowsColumns[i3].stretchPriority == stretchRowsColumns[0].stretchPriority)
							{
								//Reduce distribute size to limiting size of larger row/column
								if ((stretchRowsColumns[i3].actualSize - stretchRowsColumns[0].actualSize) * distributeCount < distributeSize)
									distributeSize = (stretchRowsColumns[i3].actualSize - stretchRowsColumns[0].actualSize) * distributeCount;
							}
							
							break;
						}
					}
					
					//Distribute size equally across rows/columns to stretch
					if (distributeSize >= distributeCount)
					{
						//Divide for per row/column and round down
						distributeSize = Math.floor(distributeSize / distributeCount);
						distributeSizeTotal -= (distributeSize * distributeCount);
						
						for (i3 = 0; i3 < distributeCount; i3++)
							stretchRowsColumns[i3].actualSize += distributeSize;
					}
					else //Fractional size left over, distribute remainder by pixel 
					{
						distributeSizeTotal -= distributeSize;
						
						for (i3 = 0; i3 < distributeSize; i3++)
							stretchRowsColumns[i3].actualSize++;
					}
					
					//Purge rows/columns that are at max size
					if (distributeSizeTotal >= 1)
					{
						for (i3 = stretchRowsColumns.length - 1; i3 >= 0; i3--)
						{
							if (stretchRowsColumns[i3].actualSize >= stretchRowsColumns[i3].maxSize)
								stretchRowsColumns.splice(i3, 1);
						}
					}
				}
			}
		}
		
		return {rowDataMap:rowDataMap, columnDataMap:columnDataMap};
	};
	
//@override
GridContainerElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		GridContainerElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("LayoutVerticalGap" in stylesMap ||
			"LayoutHorizontalGap" in stylesMap)
		{
			this._invalidateMeasure();
			this._invalidateLayout();
		}
		else if ("LayoutVerticalAlign" in stylesMap || 
				"LayoutHorizontalAlign" in stylesMap)
		{
			this._invalidateLayout();
		}
	};	
	
//@override
GridContainerElement.prototype._doMeasure = 
	function (padWidth, padHeight)
	{
		var layoutVerticalGap = this.getStyle("LayoutVerticalGap");
		var layoutHorizontalGap = this.getStyle("LayoutHorizontalGap");
	
		//Get measured row and column sizes.
		var measuredRowsAndColumns = this._getRowsAndColumnsMeasuredData();
		
		//For convenience
		var rowDataMap = measuredRowsAndColumns.rowDataMap;
		var columnDataMap = measuredRowsAndColumns.columnDataMap;
		
		var rowsTotalSize = 0;
		var columnsTotalSize = 0;
		
		var addGap;
		var mapKey;
		
		addGap = false;
		for (mapKey in rowDataMap)
		{
			rowsTotalSize += rowDataMap[mapKey].actualSize;
			
			if (addGap == false)
				addGap = true;
			else
				rowsTotalSize += layoutVerticalGap;
		}

		addGap = false;
		for (mapKey in columnDataMap)
		{
			columnsTotalSize += columnDataMap[mapKey].actualSize;
			
			if (addGap == false)
				addGap = true;
			else
				columnsTotalSize += layoutHorizontalGap;
		}
		
		this._setMeasuredSize(padWidth + columnsTotalSize, padHeight + rowsTotalSize);
	};	
	
//@override
GridContainerElement.prototype._doLayout =
	function(paddingMetrics)
	{
		GridContainerElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		var layoutVerticalGap = this.getStyle("LayoutVerticalGap");
		var layoutHorizontalGap = this.getStyle("LayoutHorizontalGap");
	
		//Copy draw metrics from paddingMetrics (for convenience)
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		//Get measured row and column sizes.
		var measuredRowsAndColumns = this._getRowsAndColumnsMeasuredData();
		
		//For convenience
		var rowDataMap = measuredRowsAndColumns.rowDataMap;
		var columnDataMap = measuredRowsAndColumns.columnDataMap;
		
		var rowDataSorted = [];
		var columnDataSorted = [];
		
		var percentSizedRows = [];
		var percentSizedColumns = [];
		
		var rowsTotalPercent = 0;
		var columnsTotalPercent = 0;
		
		var rowsTotalGap = 0;
		var columnsTotalGap = 0;
		
		var rowsTotalSize = 0;
		var columnsTotalSize = 0;
		
		var rowsAvailableSize = 0;
		var columnsAvailableSize = 0;
		
		var mapKey;
		for (mapKey in rowDataMap)
		{
			rowDataSorted.push(rowDataMap[mapKey]);
			
			if (rowDataMap[mapKey].percentSize != null)
			{
				rowDataMap[mapKey].minSize = Math.max(rowDataMap[mapKey].actualSize, rowDataMap[mapKey].minSize);
				percentSizedRows.push(rowDataMap[mapKey]);
				rowsTotalPercent += rowDataMap[mapKey].percentSize;
			}
		}

		for (mapKey in columnDataMap)
		{
			columnDataSorted.push(columnDataMap[mapKey]);
			
			if (columnDataMap[mapKey].percentSize != null)
			{
				columnDataMap[mapKey].minSize = Math.max(columnDataMap[mapKey].actualSize, columnDataMap[mapKey].minSize);
				percentSizedColumns.push(columnDataMap[mapKey]);
				columnsTotalPercent += columnDataMap[mapKey].percentSize;
			}
		}

		//Sort row and column data by index
		rowDataSorted.sort(GridContainerElement._indexComparator);
		columnDataSorted.sort(GridContainerElement._indexComparator);
		
		//Calculate the total row/column gap size
		if (rowDataSorted.length > 1)
			rowsTotalGap = (rowDataSorted.length - 1) * layoutVerticalGap;
		
		if (columnDataSorted.length > 1)
			columnsTotalGap = (columnDataSorted.length - 1) * layoutHorizontalGap;
		
		//Get the total size of non-percent sized rows
		for (i = 0; i < rowDataSorted.length; i++)
		{
			if (rowDataSorted[i].percentSize == null)
				rowsTotalSize += rowDataSorted[i].actualSize;
		}
		
		//Add gap
		rowsTotalSize += rowsTotalGap;
		
		//Get the total size of non-percent sized columns
		for (i = 0; i < columnDataSorted.length; i++)
		{
			if (columnDataSorted[i].percentSize == null)
				columnsTotalSize += columnDataSorted[i].actualSize;
		}
		
		//Add gap
		columnsTotalSize += columnsTotalGap;		
		
		//Calculate the available size for percent sized rows/columns
		rowsAvailableSize = h - rowsTotalSize;
		if (rowsTotalPercent < 100 )
			rowsAvailableSize = Math.round(rowsAvailableSize * (rowsTotalPercent / 100));
		
		columnsAvailableSize = w - columnsTotalSize;
		if (columnsTotalPercent < 100)
			columnsAvailableSize = Math.round(columnsAvailableSize * (columnsTotalPercent / 100));
		
		//Calculate the size of percent sized rows
		CanvasElement._calculateMinMaxPercentSizes(percentSizedRows, rowsAvailableSize);
		
		//Calculate the size of percent sized columns
		CanvasElement._calculateMinMaxPercentSizes(percentSizedColumns, columnsAvailableSize);
		
		//Update total content size of rows
		for (i = 0; i < percentSizedRows.length; i++)
			rowsTotalSize += percentSizedRows[i].actualSize;
		
		//Update total content size of columns
		for (i = 0; i < percentSizedColumns.length; i++)
			columnsTotalSize += percentSizedColumns[i].actualSize;
		
		//Get rows/columns start position
		var actualX = x;
		var actualY = y;
		
		//Adjust vertical start position based on layout styles
		if (rowsTotalSize != h)
		{
			var layoutVerticalAlign = this.getStyle("LayoutVerticalAlign");
			
			if (layoutVerticalAlign == "middle")
				actualY += Math.round((h / 2) - (rowsTotalSize / 2));
			else if (layoutVerticalAlign == "bottom")
				actualY += (h - rowsTotalSize);
		}
		
		//Adjust horizontal start position based on layout styles
		if (columnsTotalSize != w)
		{
			var layoutHorizontalAlign = this.getStyle("LayoutHorizontalAlign");
			
			if (layoutHorizontalAlign == "center")
				actualX += Math.round((w / 2) - (columnsTotalSize / 2));
			else if (layoutHorizontalAlign == "right")
				actualX += (w - columnsTotalSize);
		}
		
		//Calculate position of rows/columns
		var lastSize;
		var layoutGap;
		var lastPosition;
		var rowColumnDataSorted;
		
		for (i = 0; i < 2; i++) //Run twice, rows then columns
		{
			lastSize = 0;
			layoutGap = (i == 0 ? layoutVerticalGap : layoutHorizontalGap);
			lastPosition = (i == 0 ? actualY : actualX);
			rowColumnDataSorted = (i == 0 ? rowDataSorted : columnDataSorted);
			
			for (i2 = 0; i2 < rowColumnDataSorted.length; i2++)
			{
				rowColumnDataSorted[i2].actualPosition = lastPosition + lastSize;
				
				//Insert gap
				if (i2 > 0)
					rowColumnDataSorted[i2].actualPosition += layoutGap;
				
				lastSize = rowColumnDataSorted[i2].actualSize;
				lastPosition = rowColumnDataSorted[i2].actualPosition;
			}
		}
		
		//Size & place cells
		var cellX;
		var cellY;
		var cellW;
		var cellH;
		var insertRowGap;
		var insertColumnGap;
		
		for (i = 0; i < this._gridCells.length; i++)
		{
			cell = this._gridCells[i];
			
			cellX = null;
			cellY = null;
			cellW = 0;
			cellH = 0;
			
			insertRowGap = false;
			insertColumnGap = false;
			
			for (i2 = cell.rowIndexStart; i2 <= cell.rowIndexEnd; i2++)
			{
				if (cellY == null)
					cellY = rowDataMap[i2].actualPosition;
				
				cellH += rowDataMap[i2].actualSize;
				
				if (insertRowGap == false)
					insertRowGap = true;
				else
					cellH += layoutVerticalGap;
			}
			
			for (i2 = cell.columnIndexStart; i2 <= cell.columnIndexEnd; i2++)
			{
				if (cellX == null)
					cellX = columnDataMap[i2].actualPosition;
				
				cellW += columnDataMap[i2].actualSize;
				
				if (insertColumnGap == false)
					insertColumnGap = true;
				else
					cellW += layoutHorizontalGap;
			}
			
			cell.element._setActualPosition(cellX, cellY);
			cell.element._setActualSize(cellW, cellH);
		}
	};
	
	
	


/**
 * @depends ContainerBaseElement.js
 */

///////////////////////////////////////////////////////////////////////////	
///////////////////////AnchorContainerElement//////////////////////////////

/**
 * @class AnchorContainerElement
 * @inherits ContainerBaseElement
 * 
 * The AnchorContainer can be used to lay out children via absolute or constraint positioning.
 * This container uses children's styles X, Y, Width, Height, PercentWidth, PercentHeight,
 * Top, Bottom, Left, Right, HorizontalCenter, VerticalCenter, MinWidth, MaxWidth, MinHeight, and MaxHeight. 
 * Nesting containers is the best way to quickly and simply build complex layouts.
 * 
 * X, Y, Width, and Height will override other styles with the same priority.
 * Higher priority styles always override lower priority styles. For example, setting "PercentWidth" via setStyle()
 * will override a "Width" style set via StyleDefinition. See CanvasElement getStyle() for priority chain info.
 * 
 * Elements Z index is determined by the order they are added (child index).
 * 
 * You may use styles such as Top and Bottom in conjunction to relatively size elements.
 * You may also combine styles such as Left or X and PercentWidth. Most styles are combine-able unless
 * they are in direct conflict with each other such as having a Left, Right, and Width which under
 * this scenario the Right style will be ignored. Exact behavior of conflicting styles is not defined 
 * and subject to change. 
 * 
 * @constructor AnchorContainerElement 
 * Creates new AnchorContainerElement instance.
 */
function AnchorContainerElement()
{
	AnchorContainerElement.base.prototype.constructor.call(this);
}

//Inherit from ContainerBaseElement
AnchorContainerElement.prototype = Object.create(ContainerBaseElement.prototype);
AnchorContainerElement.prototype.constructor = AnchorContainerElement;
AnchorContainerElement.base = ContainerBaseElement;	
	
//@Override
AnchorContainerElement.prototype._doMeasure = 
	function (padWidth, padHeight)
	{
		var measuredWidth = 0;
		var measuredHeight = 0;
		
		var child = null; //for convienence
		
		var x;
		var y;
		var width;
		var height;

		var top;
		var left;
		var bottom;
		var right;
		var hCenter;
		var vCenter;
		var rotateDegrees;
		var rotateCenterX;
		var rotateCenterY;
		var rotatedMetrics;
		
		var tempX;
		var tempY;
		var tempWidth;
		var tempHeight;
		var tempRotateDegrees;
		var tempRotateCenterX;
		var tempRotateCenterY;
		
		var xData;
		var leftData;
		
		var yData;
		var topData;
		
		for (var i = 0; i < this._elements.length; i++)
		{
			child = this._elements[i];
			if (child.getStyle("IncludeInLayout") == false)
				continue;
			
			var childSize = {width: 0, height:0};
			
			width = child._getStyledOrMeasuredWidth();
			height = child._getStyledOrMeasuredHeight();
			
			xData = child.getStyleData("X");
			leftData = child.getStyleData("Left");
			if (xData.comparePriority(leftData) >= 0)
				x = xData.value;
			else
				x = leftData.value;
			
			yData = child.getStyleData("Y");
			topData = child.getStyleData("Top");
			if (yData.comparePriority(topData) >= 0)
				y = yData.value;
			else
				y = topData.value
			
			bottom = child.getStyle("Bottom");
			right = child.getStyle("Right");
			
			hCenter = child.getStyle("HorizontalCenter");
			vCenter = child.getStyle("VerticalCenter");
			
			rotateDegrees = child.getStyle("RotateDegrees");
			rotateCenterX = child.getStyle("RotateCenterX");
			rotateCenterY = child.getStyle("RotateCenterY");
			
			if (rotateDegrees != 0)
			{
				//Record child's current x/y & w/h & rotation
				tempX = child._x;
				tempY = child._y;
				tempWidth = child._width;
				tempHeight = child._height;
				tempRotateDegrees = child._rotateDegrees;
				tempRotateCenterX = child._rotateCenterX;
				tempRotateCenterY = child._rotateCenterY;
				
				//TODO: Update getMetrics() so we can pass child values.
				//Spoof the rotation position/size so we can get parent metrics.
				child._x = x == null ? 0 : x;
				child._y = y == null ? 0 : y;
				child._width = width;
				child._height = height;
				child._rotateDegrees = rotateDegrees;
				child._rotateCenterX = rotateCenterX == null ? 0 : rotateCenterX;
				child._rotateCenterY = rotateCenterY == null ? 0 : rotateCenterY;
				
				//Get parent metrics for spoof position
				rotatedMetrics = child.getMetrics(this);
				
				//Put back current values
				child._x = tempX;
				child._y = tempY;
				child._width = tempWidth;
				child._height = tempHeight;
				child._rotateDegrees = tempRotateDegrees;
				child._rotateCenterX = tempRotateCenterX;
				child._rotateCenterY = tempRotateCenterY;
				
				if (rotateCenterX != null && rotateCenterY != null)
				{
					x = Math.max(rotatedMetrics.getX(), 0);
					y = Math.max(rotatedMetrics.getY(), 0);
				}
				
				childSize.width += Math.ceil(rotatedMetrics.getWidth());
				childSize.height += Math.ceil(rotatedMetrics.getHeight());
			}
			else //Non-Rotated Element
			{
				childSize.width += width;
				childSize.height += height;
			}
			
			if (right != null)
				childSize.width += right;
			if (bottom != null)
				childSize.height += bottom;
			
			if (x == null && right == null && hCenter != null)
				childSize.width += Math.abs(hCenter * 2);
			if (y == null && bottom == null && vCenter != null)
				childSize.height += Math.abs(vCenter * 2);
			
			if (x == null || x < 0)
				x = 0;
			if (y == null || y < 0)
				y = 0;
			
			childSize.width += x;
			childSize.height += y;
			
			measuredWidth = Math.max(measuredWidth, Math.ceil(childSize.width));
			measuredHeight = Math.max(measuredHeight, Math.ceil(childSize.height));
		}
		
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};
	
//@Override
AnchorContainerElement.prototype._doLayout =
	function(paddingMetrics)
	{
		AnchorContainerElement.base.prototype._doLayout.call(this, paddingMetrics);
	
		var child = null;
		
		var x = null;
		var y = null;
		var width = null;
		var height = null;
		var pWidth = null;
		var pHeight = null;
		var minWidth = null;
		var maxWidth = null;
		var minHeight = null;
		var maxHeight = null;		
		var top = null;
		var left = null;
		var bottom = null;
		var right = null;
		var hCenter = null;
		var vCenter = null;
		var rotateDegrees = null;
		var rotateCenterX = null;
		var rotateCenterY = null;
		var rotatedMetrics = null;
		
		var xData = null;
		var yData = null;
		var leftData = null;
		var topData = null;
		
		var widthData = null;
		var heightData = null;
		var percentWidthData = null;
		var percentHeightData = null;
		
		for (var i = 0; i < this._elements.length; i++)
		{
			child = this._elements[i];
			if (child.getStyle("IncludeInLayout") == false)
				continue;
			
			x = null;
			y = null;
			width = null;
			height = null;
			pWidth = null;
			pHeight = null;
			minWidth = null;
			maxWidth = null;
			minHeight = null;
			maxHeight = null;		
			top = null;
			left = null;
			bottom = null;
			right = null;
			hCenter = null;
			vCenter = null;
			rotateDegrees = null;
			rotateCenterX = null;
			rotateCenterY = null;
			rotatedMetrics = null;
			
			xData = child.getStyleData("X");
			leftData = child.getStyleData("Left");
			
			//Use x if equal or higher priority than left
			if (xData.comparePriority(leftData) >= 0)
				x = xData.value;
			else
				x = leftData.value;
			
			yData = child.getStyleData("Y");
			topData = child.getStyleData("Top");
			
			//Use y if equal or higher priority than top
			if (yData.comparePriority(topData) >= 0)
				y = yData.value;
			else
				y = topData.value;
			
			widthData = child.getStyleData("Width");
			percentWidthData = child.getStyleData("PercentWidth");
			
			//Use width if equal or higher priority than percent width
			if (widthData.comparePriority(percentWidthData) >= 0)
				width = widthData.value;
			else
				pWidth = percentWidthData.value;
			
			heightData = child.getStyleData("Height");
			percentHeightData = child.getStyleData("PercentHeight");
			
			//Use height if equal or higher priority than percent height
			if (heightData.comparePriority(percentHeightData) >= 0)
				height = heightData.value;
			else
				pHeight = percentHeightData.value;
				
			minWidth = child.getStyle("MinWidth");
			maxWidth = child.getStyle("MaxWidth");
			minHeight = child.getStyle("MinHeight");
			maxHeight = child.getStyle("MaxHeight");
			bottom = child.getStyle("Bottom");
			right = child.getStyle("Right");
			hCenter = child.getStyle("HorizontalCenter");
			vCenter = child.getStyle("VerticalCenter");
			rotateDegrees = child.getStyle("RotateDegrees");
			rotateCenterX = child.getStyle("RotateCenterX");
			rotateCenterY = child.getStyle("RotateCenterY");
			
			if (minWidth == null)
				minWidth = 0;
			if (minHeight == null)
				minHeight = 0;
			if (maxWidth == null)
				maxWidth = Number.MAX_VALUE;
			if (maxHeight == null)
				maxHeight = Number.MAX_VALUE;
			
			child._setActualRotation(rotateDegrees, rotateCenterX, rotateCenterY);
			
			if (rotateDegrees != 0)
			{
				if (width == null)
				{
					width = child._measuredWidth;
					width = Math.min(width, maxWidth);
					width = Math.max(width, minWidth);
				}
				
				if (height == null)
				{
					height = child._measuredHeight;
					height = Math.min(height, maxHeight);
					height = Math.max(height, minHeight);
				}
				
				child._setActualSize(width, height);
				
				if (rotateCenterX == null || rotateCenterY == null)
				{
					//prioritize x/y over left/top (but they're the same)
					if (x == null)
						x = left;
					if (y == null)
						y = top;
					
					if (x == null || y == null)
					{
						rotatedMetrics = child.getMetrics(this);
						
						width = Math.ceil(rotatedMetrics.getWidth());
						height = Math.ceil(rotatedMetrics.getHeight());
						
						if (x == null && right != null)
							x = this._width - width - right;
						
						if (y == null && bottom != null)
							y = this._height - height - bottom;
						
						if (x == null && hCenter != null)
							x = Math.round((this._width / 2) - (width / 2) + hCenter);
						
						if (y == null && vCenter != null)
							y = Math.round((this._height / 2) - (height / 2) + vCenter);
					}
					
					if (x == null)
						x = 0;
					if (y == null)
						y = 0;
					
					child._setRelativePosition(x, y, this);
				}
				else
				{
					if (x == null)
						x = 0;
					if (y == null)
						y = 0;
					
					child._setActualPosition(x, y);
				}
			}
			else //Non-Rotated Element
			{
				if (width == null)
				{
					if (x != null && right != null)
						width = this._width - x - right;
					else
					{
						if (pWidth != null)
							width = Math.round(this._width * (pWidth / 100));
						else
							width = child._measuredWidth;
					}
					
					width = Math.min(width, maxWidth);
					width = Math.max(width, minWidth);
				}
				
				if (height == null)
				{
					if (y != null && bottom != null)
						height = this._height - y - bottom;
					else
					{
						if (pHeight != null)
							height = Math.round(this._height * (pHeight / 100));
						else
							height = child._measuredHeight;
					}
					
					height = Math.min(height, maxHeight);
					height = Math.max(height, minHeight);
				}
				
				if (x == null && right != null)
					x = this._width - width - right;
				
				if (y == null && bottom != null)
					y = this._height - height - bottom;
				
				if (x == null && hCenter != null)
					x = Math.round((this._width / 2) - (width / 2) + hCenter);
				
				if (y == null && vCenter != null)
					y = Math.round((this._height / 2) - (height / 2) + vCenter);
				
				if (x == null)
					x = 0;
				if (y == null)
					y = 0;
				
				child._setActualPosition(x, y);
				child._setActualSize(width, height);
			}
		}
	};	
	
	


/**
 * @depends CanvasElement.js
 * @depends SolidFill.js
 */

//////////////////////////////////////////////////////////////
///////////////ColorPickerElement/////////////////////////////

/**
 * @class ColorPickerElement
 * @inherits CanvasElement
 * 
 * ColorPickerElement is a class used to select colors.  
 * 
 * @constructor ColorPickerElement 
 * Creates new ColorPickerElement instance.
 */

function ColorPickerElement() //extends CanvasElement
{
	ColorPickerElement.base.prototype.constructor.call(this);
	
	//Setup static fill
	if (ColorPickerElement._HueBarLinearGradient == null)
	{
		ColorPickerElement._HueBarLinearGradientFill = new LinearGradientFill();
		ColorPickerElement._HueBarLinearGradientFill.setStyle("GradientDegrees", 180); //Left to right
		ColorPickerElement._HueBarLinearGradientFill.setStyle("GradientColorStops", [[0,"#FF0000"],[.1666,"#FFFF00"],[.3333,"#00FF00"],[.5,"#00FFFF"],[.6666,"#0000FF"],[.8333,"#FF00FF"],[1,"#FF0000"]]);
	}
	
	//Setup static shape
	if (ColorPickerElement._CaretRoundedRectangleShape == null)
	{
		ColorPickerElement._CaretRoundedRectangleShape = new RoundedRectangleShape();
		ColorPickerElement._CaretRoundedRectangleShape.setStyle("CornerRadiusPercent", 50);
	}
	
	//Hsl background fill for picker area
	this._pickerAreaFill = new HslPickerFill();
	this._pickerAreaFill.setStyle("FillColor", "#FF0000");
	
	/////////////
	
	//Use list container to control layout
	this._rootListContainer = new ListContainerElement();
	this._rootListContainer.setStyle("LayoutGap", 5);
	
		this._hueBar = new CanvasElement();
		this._hueBar.setStyle("BackgroundFill", ColorPickerElement._HueBarLinearGradientFill);
		this._hueBar.setStyle("PercentWidth", 100);
		this._hueBar.setStyle("PercentHeight", 12);
		
		this._pickerAreaBorderContainer = new AnchorContainerElement();
		this._pickerAreaBorderContainer.setStyle("BorderType", "solid");
		this._pickerAreaBorderContainer.setStyle("BorderColor", "#CFCFCF");
		this._pickerAreaBorderContainer.setStyle("PercentWidth", 100);
		this._pickerAreaBorderContainer.setStyle("PercentHeight", 88);
		
			this._pickerArea = new CanvasElement();
			this._pickerArea.setStyle("BackgroundFill", this._pickerAreaFill);
			this._pickerArea.setStyle("Top", 1);
			this._pickerArea.setStyle("Bottom", 1);
			this._pickerArea.setStyle("Left", 1);
			this._pickerArea.setStyle("Right", 1);
			
		this._pickerAreaBorderContainer.addElement(this._pickerArea);
		
		this._colorSelectionContainer = new ListContainerElement();
		this._colorSelectionContainer.setStyle("LayoutDirection", "horizontal");
		this._colorSelectionContainer.setStyle("LayoutGap", 10);
		this._colorSelectionContainer.setStyle("PercentWidth", 100);
			
			this._textInputColor = new TextInputElement();
			this._textInputColor.setStyle("MaxChars", 6);
			this._textInputColor.setText("FFFFFF");
			
			this._selectedColorSwatch = new CanvasElement();
			this._selectedColorSwatch.setStyle("PercentHeight", 100);
			this._selectedColorSwatch.setStyle("PercentWidth", 100);
			this._selectedColorSwatch.setStyle("BorderType", "solid");
			this._selectedColorSwatch.setStyle("BorderColor", "#CFCFCF");
			this._selectedColorSwatch.setStyle("BackgroundFill", "#FFFFFF");
			
		this._colorSelectionContainer.addElement(this._textInputColor);
		this._colorSelectionContainer.addElement(this._selectedColorSwatch);	
			
	this._rootListContainer.addElement(this._hueBar);
	this._rootListContainer.addElement(this._pickerAreaBorderContainer);
	this._rootListContainer.addElement(this._colorSelectionContainer);

	//Put carets on top, we position these per current selection via _doLayout()
	this._hueCaret = new CanvasElement();
	this._hueCaret.setStyle("MouseEnabled", false);
	this._hueCaret.setStyle("BorderType", "solid");
	this._hueCaret.setStyle("BackgroundFill", "#FFFFFF");
	
	this._pickerCaret = new AnchorContainerElement();
	this._pickerCaret.setStyle("MouseEnabled", false);
	this._pickerCaret.setStyle("BackgroundShape", ColorPickerElement._CaretRoundedRectangleShape);
	this._pickerCaret.setStyle("BorderType", "solid");
	this._pickerCaret.setStyle("BorderColor", "#FFFFFF");
		
		this._pickerCaretInner = new CanvasElement();
		this._pickerCaretInner.setStyle("MouseEnabled", false);
		this._pickerCaretInner.setStyle("BackgroundShape", ColorPickerElement._CaretRoundedRectangleShape);
		this._pickerCaretInner.setStyle("BorderType", "solid");
		this._pickerCaretInner.setStyle("BorderColor", "#444444");
		this._pickerCaretInner.setStyle("Top", 1);
		this._pickerCaretInner.setStyle("Bottom", 1);
		this._pickerCaretInner.setStyle("Left", 1);
		this._pickerCaretInner.setStyle("Right", 1);
	
	this._pickerCaret.addElement(this._pickerCaretInner);	
		
	this._addChild(this._rootListContainer);
	this._addChild(this._hueCaret);
	this._addChild(this._pickerCaret);

	
	///////Event Handlers///////////
	
	//Private event handlers, different instance needed for each ColorPicker, proxy to prototype
	
	var _self = this;
	
	this._onChildContainerLayoutCompleteInstance = 
		function (event)
		{
			_self._onChildContainerLayoutComplete(event);
		};
	this._onTextInputColorMeasureCompleteInstance = 
		function (event)
		{
			_self._onTextInputColorMeasureComplete();
		};
	this._onHueBarMouseDownInstance = 
		function (mouseEvent)
		{
			_self._onHueBarMouseDown(mouseEvent);
		};
	this._onHueBarMouseMoveExInstance = 
		function (mouseEvent)
		{
			_self._onHueBarMouseMoveEx(mouseEvent);
		};
	this._onHueBarMouseUpInstance = 
		function (mouseEvent)
		{
			_self._onHueBarMouseUp(mouseEvent);
		};
	this._onPickerAreaMouseDownInstance = 
		function (mouseEvent)
		{
			_self._onPickerAreaMouseDown(mouseEvent);
		};
	this._onPickerAreaMouseMoveExInstance = 
		function (mouseEvent)
		{
			_self._onPickerAreaMouseMoveEx(mouseEvent);
		};
	this._onPickerAreaMouseUpInstance = 
		function (mouseEvent)
		{
			_self._onPickerAreaMouseUp(mouseEvent);
		};	
	this._onTextInputColorChangedInstance = 
		function (event)
		{
			_self._onTextInputColorChanged(event);
		};
		
	//For adjusting caret position when nested container layouts finished
	this._rootListContainer.addEventListener("layoutcomplete", this._onChildContainerLayoutCompleteInstance);	
	this._pickerAreaBorderContainer.addEventListener("layoutcomplete", this._onChildContainerLayoutCompleteInstance);
	
	//We're tweaking the TextInput's measured width
	this._textInputColor.addEventListener("measurecomplete", this._onTextInputColorMeasureCompleteInstance);	
	this._textInputColor.addEventListener("changed", this._onTextInputColorChangedInstance);
	
	this._hueBar.addEventListener("mousedown", this._onHueBarMouseDownInstance);
	this._hueBar.addEventListener("mouseup", this._onHueBarMouseUpInstance);
	
	this._pickerArea.addEventListener("mousedown", this._onPickerAreaMouseDownInstance);
	this._pickerArea.addEventListener("mouseup", this._onPickerAreaMouseUpInstance);
	
	
	////////////////////////////////
	
	this._selectedHue = 0;
	this._selectedSat = 0;
	this._selectedLight = 100;
}

//Inherit from CanvasElement
ColorPickerElement.prototype = Object.create(CanvasElement.prototype);
ColorPickerElement.prototype.constructor = ColorPickerElement;
ColorPickerElement.base = CanvasElement;


////////////Events/////////////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the ColorPicker's selection state changes as a result of user interaction.
 */

/////////////Style Types///////////////////////////////

ColorPickerElement._StyleTypes = Object.create(null);

/**
 * @style TextInputColorStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the TextInput color element.
 */
ColorPickerElement._StyleTypes.TextInputColorStyle = 			StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition


////////////Default Styles/////////////////////////////

ColorPickerElement.StyleDefault = new StyleDefinition();

ColorPickerElement.StyleDefault.setStyle("TextInputColorStyle", 	null);

ColorPickerElement.StyleDefault.setStyle("PaddingLeft", 			5);
ColorPickerElement.StyleDefault.setStyle("PaddingRight", 			5);
ColorPickerElement.StyleDefault.setStyle("PaddingBottom", 			5);
ColorPickerElement.StyleDefault.setStyle("PaddingTop", 				5);


/////////////Static////////////////////

//Static Internal, all color pickers can use the same fill/shape instances
ColorPickerElement._HueBarLinearGradientFill = null;
ColorPickerElement._CaretRoundedRectangleShape = null;

/**
 * @function hexToRgb
 * @static
 * Converts Hex color string to RGB values.
 * 
 * @param hex String
 * Hex string value formatted like "#FF0000"
 * 
 * @returns Object
 * An object {r:0, g:0, b:0} containing r, g, and b  
 * properties representing red, green, and blue channels.
 */
ColorPickerElement.hexToRgb = 
	function (hex)
	{
		var result = {r:0, g:0, b:0};
	
		//Get rgb from hex
		result.r = parseInt(hex.substr(1, 2), 16);
		result.g = parseInt(hex.substr(3, 2), 16);
		result.b = parseInt(hex.substr(5, 2), 16);
	
		return result;
	};

/**
 * @function rgbToHex
 * @static
 * Converts RGB values to hex format.
 * 
 * @param r Number
 * Value of the red channel (0-255)
 * 
 * @param g Number
 * Value of the green channel (0-255)
 * 
 * @param b Number
 * Value of the blue channel (0-255)
 * 
 * @returns String
 * Hex string value formatted like "#FF0000"
 */
ColorPickerElement.rgbToHex = 
	function (r, g, b)
	{
		var rx = r.toString(16);
		var gx = g.toString(16);
		var bx = b.toString(16);
		
		if (r < 16) 
			rx = '0' + rx;
		
		if (g < 16) 
			gx = '0' + gx;
		
		if (b < 16) 
			bx = '0' + bx;
		
		var result = '#' + rx + gx + bx;
		
		return result.toUpperCase();
	};

/**
 * @function rgbToHsl
 * @static
 * Converts RGB values to HSL values.
 * 
 * @param r Number
 * Value of the red channel (0-255)
 * 
 * @param g Number
 * Value of the green channel (0-255)
 * 
 * @param b Number
 * Value of the blue channel (0-255)
 * 
 * @returns Object
 * An object {h:0, s:0, l:0} containing h, s, and l  
 * properties representing hue, saturation, and lightness.
 * Note that fractional HSL values may be returned.
 */	
ColorPickerElement.rgbToHsl = 
	function (r, g, b) 
	{
		var red = r / 255;
		var green = g / 255;
		var blue = b / 255;

		var cmax = Math.max(red, green, blue);
		var cmin = Math.min(red, green, blue);
		var delta = cmax - cmin;
		
		var hue = 0;
		var sat = 0;
		var light = (cmax + cmin) / 2;
		
		var X = (1 - Math.abs(2 * light - 1));

		if (delta != 0)
		{
			if (cmax == red) 
				hue = ((green - blue) / delta);
			
			if (cmax == green)
				hue = 2 + ((blue - red) / delta);
			
			if (cmax == blue)
				hue = 4 + ((red - green) / delta);
			
			if (cmax != 0) 
				sat = delta / X;
		}

		var result = {h:0, s:0, l:0};
		
		result.h = CanvasElement.normalizeDegrees(60 * hue);
		result.s = sat * 100;
		result.l = light * 100;

		return result;
};	
	
/**
 * @function hslToRgb
 * @static
 * Converts HSL values to RGB values. Fractional HSL 
 * values are allowed.
 * 
 * @param h Number
 * Value of the hue (0 - 360)
 * 
 * @param s Number
 * Value of the saturation (0-100)
 * 
 * @param l Number
 * Value of lightness (0-100)
 * 
 * @returns Object
 * An object {r:0, g:0, b:0} containing r, g, and b  
 * properties representing red, green, and blue channels.
 */
ColorPickerElement.hslToRgb = 
	function (h, s, l) 
	{
		var sat = s / 100;
		var light = l / 100;
		var hue = h / 60;
		
		var c = sat * (1 - Math.abs(2 * light - 1));
		var x = c * (1 - Math.abs(hue % 2 - 1));
		var m = light - (c / 2);
	
		c = Math.trunc((c + m) * 255);
		x = Math.trunc((x + m) * 255);
		m = Math.trunc(m * 255);
	
		if (hue >= 0 && hue < 1)
			return {r:c, g:x, b:m};
		
		if (hue >= 1 && hue < 2)
			return {r:x, g:c, b:m};
			
		if (hue >= 2 && hue < 3)
			return {r:m, g:c, b:x};
		
		if (hue >= 3 && hue < 4)
			return {r:m, g:x, b:c};
			
		if (hue >= 4 && hue < 5)
			return {r:x, g:m, b:c};
			
		//if (hue >= 5 && hue <= 6)
			return {r:c, g:m, b:x};
	};


////////////Public////////////////
	
/**
 * @function setHexColor
 * Sets the selected color of the ColorPickerElement.
 * 
 * @param value String
 * RGB hex color value formatted as "#FF0000".
 */	
ColorPickerElement.prototype.setHexColor = 
	function (value)
	{
		value = this._fixInvalidHexColor(value);
		
		this._textInputColor.setText(value.slice(1));
		
		this._updateSelectedRgb();
	};
	
/**
 * @function getHexColor
 * Gets the selected color of the ColorPickerElement.
 * 
 * @returns String
 * RGB hex color value formatted as "#FF0000".
 */		
ColorPickerElement.prototype.getHexColor = 
	function ()
	{
		return this._fixInvalidHexColor(this._textInputColor.getText());
	};
	
	
////////////Internal//////////////
	
//@private	
ColorPickerElement.prototype._fixInvalidHexColor = 
	function (value)
	{
		value = value.toUpperCase();
	
		if (value.length == 0)
			value = "#";
		
		//Make sure the "#" token is present
		if (value[0] != "#")
			value = "#" + value;
		
		//Too short - pad zeros
		while (value.length < 7)
			value = value + "0";
		
		//Too long - truncate
		if (value.length > 7)
			value = value.slice(0, 7);
		
		//Replace invalid characters with "F"
		for (var i = 1; i < value.length; i++)
		{
			if (value[i] != "0" && 
				value[i] != "1" &&
				value[i] != "2" &&
				value[i] != "3" &&
				value[i] != "4" &&
				value[i] != "5" &&
				value[i] != "6" &&
				value[i] != "7" &&
				value[i] != "8" &&
				value[i] != "9" &&
				value[i] != "A" &&
				value[i] != "B" &&
				value[i] != "C" &&
				value[i] != "D" &&
				value[i] != "E" &&
				value[i] != "F")
			{
				value = value.slice(0, i) + "F" + value.slice(i + 1);
			}
		}
		
		return value;
	};	
	
//@private - Root list's layout complete handler, adjusts caret positions.
ColorPickerElement.prototype._onChildContainerLayoutComplete = 
	function (event)
	{
		this._layoutCarets();
	};
	
//@private - Adjust the TextInput measured width based on font size
ColorPickerElement.prototype._onTextInputColorMeasureComplete = 
	function (event)
	{
		var textInputPadding = this._textInputColor._getPaddingSize();
		var textWidth = CanvasElement._measureText("BBBBBB", this._textInputColor._getFontString());
		
		var textInputWidth = textWidth + textInputPadding.width + 10;
		var textInputHeight = this._textInputColor._getStyledOrMeasuredHeight();
		
		this._textInputColor._setMeasuredSize(textInputWidth, textInputHeight);
	};

/**
 * @function _onHueBarMouseDown
 * Event handler for the hue bar's "mousedown" event. 
 * Updates the ColorPicker's selected hue & color and dispatches changed event.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
ColorPickerElement.prototype._onHueBarMouseDown =
	function (elementMouseEvent)
	{
		//Listen for global mouse movement
		if (this.hasEventListener("mousemoveex", this._onHueBarMouseMoveExInstance) == false)
			this.addEventListener("mousemoveex", this._onHueBarMouseMoveExInstance);
		
		//Infer the hue via the mouse position
		var hue = (elementMouseEvent.getX() / this._hueBar._width) * 360;
		
		//Update hue
		if (hue != this._selectedHue)
		{
			this._selectedHue = hue;
			
			//Fix UI for new hue
			this._updateSelectedHue();
			this._updateSelectedHsl();
			
			//Dispatch changed event.
			if (this.hasEventListener("changed", null) == true)
				this.dispatchEvent(new ElementEvent("changed", false));
		}
	};
	
/**
 * @function _onHueBarMouseMoveEx
 * Event handler for the hue bar's "mousemoveex" event. 
 * Updates the ColorPicker's selected hue & color and dispatches changed event.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
ColorPickerElement.prototype._onHueBarMouseMoveEx = 
	function (elementMouseEvent)
	{
		//Translate global (Canvas based) mouse position to the position within the hue bar
		var mousePoint = {x:elementMouseEvent.getX(), y:elementMouseEvent.getY()};
		this._hueBar.translatePointFrom(mousePoint, this._manager);
	
		//Mouse is out of bounds, cap it
		if (mousePoint.x < 0)
			mousePoint.x = 0;
		if (mousePoint.x > this._hueBar._width)
			mousePoint.x = this._hueBar._width;
		
		//Infer hue via the mouse position
		var hue = (mousePoint.x / this._hueBar._width) * 360;
		
		//Update hue
		if (hue != this._selectedHue)
		{
			this._selectedHue = hue;
			
			//Fix UI for new hue
			this._updateSelectedHue();
			this._updateSelectedHsl();
			
			//Dispatch changed event.
			if (this.hasEventListener("changed", null) == true)
				this.dispatchEvent(new ElementEvent("changed", false));
		}
	};
	
/**
 * @function _onHueBarMouseUp
 * Event handler for the hue bar's "mouseup" event. 
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */	
ColorPickerElement.prototype._onHueBarMouseUp = 
	function (elementMouseEvent)
	{
		//Remove global mouse listener
		if (this.hasEventListener("mousemoveex", this._onHueBarMouseMoveExInstance) == true)
			this.removeEventListener("mousemoveex", this._onHueBarMouseMoveExInstance);
	};	
	
/**
 * @function _onPickerAreaMouseDown
 * Event handler for the picker area's "mousedown" event. 
 * Updates the ColorPicker's selected color and dispatches changed event.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
ColorPickerElement.prototype._onPickerAreaMouseDown = 
	function (elementMouseEvent)
	{
		//Add global mouse move listener
		if (this.hasEventListener("mousemoveex", this._onPickerAreaMouseMoveExInstance) == false)
			this.addEventListener("mousemoveex", this._onPickerAreaMouseMoveExInstance);
	
		//Infer saturation and lightness via mouse position
		this._selectedSat = (elementMouseEvent.getX() / this._pickerArea._width) * 100;
		this._selectedLight = 100 - ((elementMouseEvent.getY() / this._pickerArea._height) * 100);
		
		//Fix UI for new color
		this._updateSelectedHsl();
		
		//Dispatch changed event.
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};
	
/**
 * @function _onPickerAreaMouseMoveEx
 * Event handler for the picker area's "mousemoveex" event. 
 * Updates the ColorPicker's selected color and dispatches changed event.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
ColorPickerElement.prototype._onPickerAreaMouseMoveEx = 
	function (elementMouseEvent)
	{
		//Translate global (Canvas based) mouse position to the position within the picker area
		var mousePoint = {x:elementMouseEvent.getX(), y:elementMouseEvent.getY()};
		this._pickerArea.translatePointFrom(mousePoint, this._manager);
	
		//Mouse is out of bounds - cap it
		if (mousePoint.x < 0)
			mousePoint.x = 0;
		if (mousePoint.x > this._pickerArea._width)
			mousePoint.x = this._pickerArea._width;
		
		if (mousePoint.y < 0)
			mousePoint.y = 0;
		if (mousePoint.y > this._pickerArea._height)
			mousePoint.y = this._pickerArea._height;
		
		//Infer saturation and lightness via mouse position
		this._selectedSat = (mousePoint.x / this._pickerArea._width) * 100;
		this._selectedLight = 100 - ((mousePoint.y / this._pickerArea._height) * 100);
		
		//Fix UI for new color
		this._updateSelectedHsl();
		
		//Dispatch changed event.
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};	
	
/**
 * @function _onPickerAreaMouseUp
 * Event handler for the picker area's "mouseup" event. 
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */	
ColorPickerElement.prototype._onPickerAreaMouseUp = 
	function (elementMouseEvent)
	{
		//Remove global mouse listener
		if (this.hasEventListener("mousemoveex", this._onPickerAreaMouseMoveExInstance) == true)
			this.removeEventListener("mousemoveex", this._onPickerAreaMouseMoveExInstance);
	};	

/**
 * @function _updateSelectedHue
 * Updates the picker area's background hue.  
 */
ColorPickerElement.prototype._updateSelectedHue =
	function ()
	{
		//Get color of current hue
		var rgb = ColorPickerElement.hslToRgb(this._selectedHue, 100, 50);
		var rgbX = ColorPickerElement.rgbToHex(rgb.r, rgb.g, rgb.b);
		
		//Set picker area hue
		this._pickerAreaFill.setStyle("FillColor", rgbX);
		this._invalidateLayout();	//Need to move caret
	};	
	
/**
 * @function _updateSelectedHsl
 * Updates the selected color swatch and TextInput when color is set via HSL.
 */
ColorPickerElement.prototype._updateSelectedHsl = 
	function ()
	{
		//Translate hsl color selection to hex rgb string
		var rgb = ColorPickerElement.hslToRgb(this._selectedHue, this._selectedSat, this._selectedLight);
		var rgbX = ColorPickerElement.rgbToHex(rgb.r, rgb.g, rgb.b);
		
		//Fix UI
		this._selectedColorSwatch.setStyle("BackgroundFill", rgbX);
		this._textInputColor.setText(rgbX.slice(1));
		this._invalidateLayout();	//Need to move caret
	};
	
/**
 * @function _updateSelectedRgb
 * Updates the selected color swatch and HSL values when color set via RGB.
 */
ColorPickerElement.prototype._updateSelectedRgb = 
	function ()
	{
		var value = this._fixInvalidHexColor(this._textInputColor.getText());
	
		var rgb = ColorPickerElement.hexToRgb(value);
		
		//Translate to hsl
		var hsl = ColorPickerElement.rgbToHsl(rgb.r, rgb.g, rgb.b);
		
		//Update selection
		this._selectedHue = hsl.h;
		this._selectedSat = hsl.s;
		this._selectedLight = hsl.l;
		
		//Fix UI
		this._updateSelectedHue();
		this._selectedColorSwatch.setStyle("BackgroundFill", value);
		this._invalidateLayout();	//Need to move caret
	};
	
/**
 * @function _onTextInputColorChanged
 * Event handler for the text input's "changed" event. 
 * Updates the picker's selected color.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
ColorPickerElement.prototype._onTextInputColorChanged = 
	function (elementEvent)
	{
		this._updateSelectedRgb();
		
		//Dispatch changed event.
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};
	
//@override
ColorPickerElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ColorPickerElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		if ("TextInputColorStyle" in stylesMap)
			this._applySubStylesToElement("TextInputColorStyle", this._textInputColor);
	};	

//@override
ColorPickerElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var textInputHeight = this._textInputColor._getStyledOrMeasuredHeight();
		var totalGap = this._rootListContainer.getStyle("LayoutGap") * (this._rootListContainer.getNumElements() - 1);
		
		//176 x 176 measured picker area, 176 x 24 measured hue bar
		this._setMeasuredSize(padWidth + 176, padHeight + 24 + 176 + textInputHeight + totalGap);
	};

/**
 * @function _layoutCarets
 * Sizes and positions the hue bar and picker area carets.
 */
ColorPickerElement.prototype._layoutCarets = 
	function ()
	{
		//Children not done with layout
		if (this._rootListContainer._layoutInvalid == true ||
			this._pickerAreaBorderContainer._layoutInvalid == true) 
		{
			return;
		}
	
		//Hue caret
		this._hueCaret._setActualSize(4, this._hueBar._height + 2);
		
		var hueCaretPos = {x:0, y:-1};
		hueCaretPos.x = (this._selectedHue / 360) * this._hueBar._width;
		this.translatePointFrom(hueCaretPos, this._hueBar);
		
		hueCaretPos.x = Math.round(hueCaretPos.x - (this._hueCaret._width / 2));
		
		this._hueCaret._setActualPosition(hueCaretPos.x, hueCaretPos.y);
		
		//Picker caret
		this._pickerCaret._setActualSize(12, 12);
	
		//We have to layout based on the border container, since the picker area 
		//will not be sized yet since its a child of a nested anchor container.
		var pickerCaretPos = {x:0, y:0};
		pickerCaretPos.x = (this._selectedSat / 100) * (this._pickerArea._width);
		pickerCaretPos.y = ((100 - this._selectedLight) / 100) * (this._pickerArea._height);
		this.translatePointFrom(pickerCaretPos, this._pickerArea);
		
		pickerCaretPos.x = Math.round(pickerCaretPos.x - (this._pickerCaret._width / 2));
		pickerCaretPos.y = Math.round(pickerCaretPos.y - (this._pickerCaret._height / 2));
		
		this._pickerCaret._setActualPosition(pickerCaretPos.x, pickerCaretPos.y);
	};
	
//@override	
ColorPickerElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		ColorPickerElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		this._rootListContainer._setActualSize(w, h);
		this._rootListContainer._setActualPosition(x, y);
		
		this._layoutCarets();
	};	
	
	
/////////////HslPickerFill//////////////////////
	
//@private
function HslPickerFill() //Extends SolidFill
{
	HslPickerFill.base.prototype.constructor.call(this);
}

//Inherit from SolidFill
HslPickerFill.prototype = Object.create(SolidFill.prototype);
HslPickerFill.prototype.constructor = HslPickerFill;
HslPickerFill.base = SolidFill;

//@override
HslPickerFill.prototype.drawFill = 
	function (ctx, metrics)
	{
		//Do solid fill
		HslPickerFill.base.prototype.drawFill.call(this, ctx, metrics);
	
		var x = metrics.getX();
		var y = metrics.getY();
		var w = metrics.getWidth();
		var h = metrics.getHeight();
		
		var gradient;
		
		//Do saturation fill
		gradient = ctx.createLinearGradient(x, y, x + w, y);
		gradient.addColorStop(0, "hsla(0, 0%, 50%, 1)");
		gradient.addColorStop(1, "hsla(0, 0%, 50%, 0)");
		
		ctx.fillStyle = gradient;
		ctx.fill();
		
		//Do lightness fills
		gradient = ctx.createLinearGradient(x, y, x, y + h);
		gradient.addColorStop(0, "hsla(0, 0%, 100%, 1)");
		gradient.addColorStop(.5, "hsla(0, 0%, 100%, 0)");
		
		ctx.fillStyle = gradient;
		ctx.fill();
		
		gradient = ctx.createLinearGradient(x, y, x, y + h);
		gradient.addColorStop(.5, "hsla(0, 0%, 0%, 0)");
		gradient.addColorStop(1, "hsla(0, 0%, 0%, 1)");
		
		ctx.fillStyle = gradient;
		ctx.fill();
	};	
	


/**
 * @depends AnchorContainerElement.js
 */

//////////////////////////////////////////////////////////////
////////////////////CanvasManager/////////////////////////////

/**
 * @class CanvasManager
 * @inherits AnchorContainerElement
 * 
 * CanvasManager is the root of the display hierarchy, manages a single canvas, and is essentially
 * the brain of the system, its responsible things such as driving the component life cycle, 
 * managing CanvasElements, requesting render frames from the browser, etc.  
 * For elements to be rendered to the canvas, they must be added to CanvasManager, or be a descendant of
 * an element that has been added to CanvasManager. 
 * 
 * CanvasManager itself is a subclass of an AnchorContainer and can be used as such, although typically
 * for more complex layouts you will nest containers inside of CanvasManager.
 * 
 * @constructor CanvasManager 
 * Creates new CanvasManager instance.
 */

function CanvasManager()
{
	//Life cycle phases
	this._updateStylesQueue = new CmDepthQueue();
	this._updateMeasureQueue = new CmDepthQueue();
	this._updateLayoutQueue = new CmDepthQueue();
	this._updateRenderQueue = new CmDepthQueue();
	this._updateRedrawRegionQueue = new CmDepthQueue();
	this._updateTransformRegionQueue = new CmDepthQueue();
	this._compositeRenderQueue = new CmDepthQueue();
	
	//Used to store the add/remove events we need to dispatch after elements are added/removed from the display chain.
	//Adding and removing elements is a recursive process which must finish prior to dispatching any events.
	this._addRemoveDisplayChainQueue = new CmLinkedList();
	this._addRemoveDisplayChainQueueProcessing = false; //Recursion guard
	
	this._broadcastDispatcher = new EventDispatcher(); //Dispatches broadcast events.
	
	this._browserCursor = null;
	this._cursorChain = new CmLinkedList();	//Priority Chain (cursor)
	
	this._tabStopReverse = false;
	this._focusElement = null;				//Target with focus
	
	this._canvas = null;
	this._canvasContext = null;
	this._canvasRenderFramePending = false;
	
	this._mouseX = -1;
	this._mouseY = -1;
	
	this._rollOverInvalid = true;
	this._rollOverElement = null;	//Last roll over target.
	this._rollOverX = -1;			//Position within target (used for mousemove)
	this._rollOverY = -1;
	
	this._mouseDownElement = null; 	//Target to dispatch mouseup
	
	this._draggingElement = null;	//Target currently being dragged.
	this._draggingOffsetX = null;
	this._draggingOffsetY = null;	
	
	this._currentLocale = "en-us";
	
	this._redrawRegionCachePool = new CmRedrawRegionCachePool();
	this._redrawRegionPrevMetrics = null;
	
	//Now call base
	CanvasManager.base.prototype.constructor.call(this);

	this._alertQueue = [];
	this._alertModal = null;
	
	this._cursorContainer = new CanvasElement();
	this._cursorContainer.setStyle("MouseEnabled", false);
	this._addOverlayChild(this._cursorContainer);	
	
	var _self = this;
	
	//Private handlers, need instance for every CanvasManager
	this._onCursorDefinitionStyleChangedInstance = 
		function (styleChangedEvent)
		{
			_self._onCursorDefinitionStyleChanged(styleChangedEvent);
		};
		
	this._onCanvasFrame = 
		function ()
		{
			//Prevent double render frames if someone changes our associated canvas.
			if (_self._canvasContext == null)
			{
				_self._canvasRenderFramePending = false;
				return;
			}
			
			_self.updateNow();
			
			window.requestAnimationFrame(_self._onCanvasFrame);	
		};
	
	this._canvasResizeEventHandler = 
		function ()
		{
			var clientRect = _self._canvas.getBoundingClientRect();
			
			var w = Math.round(clientRect.width * window.devicePixelRatio);
			var h = Math.round(clientRect.height * window.devicePixelRatio);
		
			//Fix canvas manager size.
			_self.setStyle("Width", w);
			_self.setStyle("Height", h);
			_self._setActualSize(w, h);
			
			_self._redrawRegionPrevMetrics = null;
			_self._updateRedrawRegion(_self.getMetrics(null));
			
			_self.updateNow();
		};
	
	this._canvasFocusEventHandler = 
		function (browserEvent)
		{
			//Tab focus only (if focused via mouse we'll get the mouse event first)
			if (_self._focusElement == null && browserEvent.type == "focus")
			{
				if (_self._tabStopReverse == true)
					_self._updateFocusElement(_self._findChildTabStopReverse(_self, null), true);
				else
					_self._updateFocusElement(_self._findChildTabStopForward(_self, null), true);
			}
			else if (_self._focusElement != null && browserEvent.type == "blur")
				_self._updateFocusElement(null, true);
		};		
		
	this._canvasKeyboardEventHandler = 
		function (browserEvent)
		{
			if (browserEvent.type == "keydown")
			{
				if (browserEvent.key == "Tab" && browserEvent.shiftKey == true)
					_self._tabStopReverse = true;
				else if (browserEvent.key == "Tab" && browserEvent.shiftKey == false)
					_self._tabStopReverse = false;
			}
		
			if (_self._focusElement != null)
			{
				var keyboardEvent = new ElementKeyboardEvent(browserEvent.type, 
											browserEvent.key, browserEvent.which, 
											browserEvent.ctrlKey, browserEvent.altKey, 
											browserEvent.shiftKey, browserEvent.metaKey);
				
				_self._focusElement.dispatchEvent(keyboardEvent);
				
				if (keyboardEvent._canceled == true || keyboardEvent._defaultPrevented == true)
					browserEvent.preventDefault();
				else if (browserEvent.type == "keydown" && keyboardEvent.getKey() == "Tab")
				{
					var tabStopElement = null;
					var currentParent = _self._focusElement;
					var lastParent = null;
					
					if (_self._tabStopReverse == false)
					{
						while (currentParent != null)
						{
							tabStopElement = _self._findChildTabStopForward(currentParent, lastParent);
							
							if (tabStopElement != null)
								break;
							
							lastParent = currentParent;
							currentParent = currentParent._parent;
						}
					}
					else //Tab backwards
					{
						while (currentParent != null)
						{
							tabStopElement = _self._findChildTabStopReverse(currentParent, lastParent);
							
							if (tabStopElement != null)
								break;
							
							lastParent = currentParent;
							currentParent = currentParent._parent;
						}
					}
					
					_self._updateFocusElement(tabStopElement, true);
					if (tabStopElement != null)
						browserEvent.preventDefault();
				}
			}
		};
	
	this._canvasMouseEventHandler = 
		function(browserEvent)
		{
			//Translate mouse to local position
			var mousePoint = CanvasManager.getLocalMousePos(browserEvent, _self._canvas);
			
			var i = 0;

			if (browserEvent.type == "mouseup")
			{
				window.removeEventListener('mouseup', _self._canvasMouseEventHandler);

				_self._clearDraggingElement();
				_self._mouseDownElement._mouseIsDown = false;

				//Start at mousedown target, record parents up to canvas manager, fix state.
				var parentChain = [];
				parentChain.push(_self._mouseDownElement);
				
				while (parentChain[parentChain.length - 1]._parent != null)
				{
					parentChain[parentChain.length - 1]._parent._mouseIsDown = false;
					parentChain.push(parentChain[parentChain.length - 1]._parent);
				}

				var clickElement = null;
				var clickPoint = {x:0, y:0};
				
				//Adjust mouse point for target element to dispatch mouseup
				for (i = parentChain.length - 1; i >= 0; i--) //Start at CanvasManager child, work down to target.
				{
					//Rotate the point backwards so we can translate the point to the element's rotated plane.
					parentChain[i].rotatePoint(mousePoint, true);
					
					//Adjust the mouse point to within this element rather than its position in parent.
					mousePoint.x = mousePoint.x - parentChain[i]._x;
					mousePoint.y = mousePoint.y - parentChain[i]._y;
					
					//Dispatch click if we're still over the target element.
					if (mousePoint.x >= 0 && 
						mousePoint.x <= parentChain[i]._width &&
						mousePoint.y >= 0 &&
						mousePoint.y <= parentChain[i]._height)
					{
						clickElement = parentChain[i];
						
						clickPoint.x = mousePoint.x;
						clickPoint.y = mousePoint.y;
					}
				}
				
				_self._mouseDownElement = null;

				//Dispatch mouseup
				parentChain[0].dispatchEvent(new ElementMouseEvent("mouseup", mousePoint.x, mousePoint.y));
				
				//Dispatch click if we're still over the target element.
				if (clickElement != null)
					clickElement.dispatchEvent(new ElementMouseEvent("click", clickPoint.x, clickPoint.y));
			}
			else if (browserEvent.type == "wheel")
			{
				//Mouse is disabled on CanvasManager
				if (_self.getStyle("MouseEnabled") == false || _self.getStyle("Visible") == false)
					return;
				
				var currentElement = null;
				
				if (mousePoint.x >= 0 && mousePoint.x <= _self._width &&
					mousePoint.y >= 0 && mousePoint.y <= _self._height)
				{
					currentElement = _self;
					
					var foundChild = false;
					while (true)
					{
						foundChild = false;
						for (i = currentElement._children.length -1; i >= 0; i--)
						{
							//Skip element if mouse is disabled or visibility is off.
							if (currentElement._children[i].getStyle("MouseEnabled") == false || 
								currentElement._children[i].getStyle("Visible") == false)
								continue;
							
							//Rotate the point backwards so we can translate the point to the element's rotated plane.
							currentElement._children[i].rotatePoint(mousePoint, true);
							
							if (mousePoint.x >= currentElement._children[i]._x && 
								mousePoint.x <= currentElement._children[i]._x + currentElement._children[i]._width &&
								mousePoint.y >= currentElement._children[i]._y &&
								mousePoint.y <= currentElement._children[i]._y + currentElement._children[i]._height)
							{
								currentElement = currentElement._children[i];
								
								//Adjust the mouse point to within this element rather than its position in parent.
								mousePoint.x = mousePoint.x - currentElement._x;
								mousePoint.y = mousePoint.y - currentElement._y;
								
								foundChild = true;
								break;
							}
							
							//Rotate forwards, we're not over this child, undo the rotation.
							currentElement._children[i].rotatePoint(mousePoint, false);
						}
						
						if (foundChild == false)
							break;
					}
				}
				
				if (currentElement != null)
				{
					var deltaX = 0;
					if (browserEvent.deltaX > 0)
						deltaX = 1;
					else if (browserEvent.deltaX < 0)
						deltaX = -1;
					
					var deltaY = 0;
					if (browserEvent.deltaY > 0)
						deltaY = 1;
					else if (browserEvent.deltaY < 0)
						deltaY = -1;
					
					var mouseWheelEvent = new ElementMouseWheelEvent(mousePoint.x, mousePoint.y, deltaX, deltaY);
					currentElement.dispatchEvent(mouseWheelEvent);
					
					if (mouseWheelEvent._canceled == true || mouseWheelEvent._defaultPrevented == true)
						browserEvent.preventDefault();
				}
			}
			else if (browserEvent.type == "mousedown")
			{
				//Kill focus if we're not over the canvas				
				if (mousePoint.x < 0 || mousePoint.x > this._width || 
					mousePoint.y < 0 || mousePoint.y > this._height)
				{
					_self._updateFocusElement(null, false);
					return;
				}
					
				//Mouse is disabled on CanvasManager
				if (_self.getStyle("MouseEnabled") == false || _self.getStyle("Visible") == false || _self._mouseDownElement != null)
					return;
				
				var draggingElement = null;
				var draggingOffset = {x:0, y:0};
				
				var focusElement = null;
				var focusElementTabStop = -1;
				var currentElementTabStop = -1;
				
				var currentElement = _self; 
				var foundChild = false;
				while (true)
				{
					currentElement._mouseIsDown = true;
					
					//Only allow dragging if we're not in a container, or an AnchorContainer
					if (currentElement.getStyle("Draggable") == true && 
						(currentElement._parent instanceof AnchorContainerElement || !(currentElement._parent instanceof ContainerBaseElement)))
					{
						draggingElement = currentElement;
						draggingOffset = {x:mousePoint.x, y:mousePoint.y};
					}
				
					currentElementTabStop = currentElement.getStyle("TabStop");
					if (currentElementTabStop >= 0 || focusElementTabStop < 0)
					{
						focusElement = currentElement;
						focusElementTabStop = currentElementTabStop;
					}
					
					foundChild = false;
					for (i = currentElement._children.length -1; i >= 0; i--)
					{
						//Skip element if mouse is disabled or visibility is off.
						if (currentElement._children[i].getStyle("MouseEnabled") == false || 
							currentElement._children[i].getStyle("Visible") == false)
							continue;
						
						//Rotate the point backwards so we can translate the point to the element's rotated plane.
						currentElement._children[i].rotatePoint(mousePoint, true);
						
						if (mousePoint.x >= currentElement._children[i]._x && 
							mousePoint.x <= currentElement._children[i]._x + currentElement._children[i]._width &&
							mousePoint.y >= currentElement._children[i]._y &&
							mousePoint.y <= currentElement._children[i]._y + currentElement._children[i]._height)
						{
							currentElement = currentElement._children[i];
							
							//Adjust the mouse point to within this element rather than its position in parent.
							mousePoint.x = mousePoint.x - currentElement._x;
							mousePoint.y = mousePoint.y - currentElement._y;
							
							foundChild = true;
							break;
						}
						
						//Rotate forwards, we're not over this child, undo the rotation.
						currentElement._children[i].rotatePoint(mousePoint, false);
					}
					
					if (foundChild == false)
						break;
				}

				_self._mouseDownElement = currentElement;
				window.addEventListener('mouseup', _self._canvasMouseEventHandler, false);
					
				if (draggingElement != null)
					_self._setDraggingElement(draggingElement, draggingOffset.x, draggingOffset.y);
				
				currentElement.dispatchEvent(new ElementMouseEvent(browserEvent.type, mousePoint.x, mousePoint.y));
				
				_self._updateFocusElement(focusElement, false);
				
				//Always shut off focus ring (even if focus doesnt change)
				if (_self._focusElement != null)
					_self._focusElement._setRenderFocusRing(false);
			}
			else if (browserEvent.type == "mousemove")
			{
				//Mouse is disabled on CanvasManager
				if (_self.getStyle("MouseEnabled") == false)
					return;
				
				_self._mouseX = mousePoint.x;
				_self._mouseY = mousePoint.y;
				_self._rollOverInvalid = true;
				
				_self._updateCursor();
				
				//Adjust dragging element.
				if (_self._draggingElement != null)
				{
					//We use metrics relative to the parent of the element being dragged. We
					//want to keep the element within parent bounds even if its transformed (rotated).
					
					//Get drag element's metrics relative to its parent.
					var metrics = _self._draggingElement.getMetrics(_self._draggingElement._parent);
					
					//Get the drag offset relative to parent.
					var offset = {x: _self._draggingOffsetX, y: _self._draggingOffsetY};
					_self._draggingElement.translatePointTo(offset, _self._draggingElement._parent);
					
					//Get the mouse position relative to parent.
					var newPosition = {	x: mousePoint.x, y: mousePoint.y };
					_self.translatePointTo(newPosition, _self._draggingElement._parent);
					
					//Adjust mouse position for drag start offset.
					newPosition.x += metrics.getX() - offset.x;
					newPosition.y += metrics.getY() - offset.y;
					
					//Correct if we're out of bounds.
					if (newPosition.x < 0)
						newPosition.x = 0;
					if (newPosition.x > _self._draggingElement._parent._width - metrics.getWidth())
						newPosition.x = _self._draggingElement._parent._width - metrics.getWidth();
					if (newPosition.y < 0)
						newPosition.y = 0;
					if (newPosition.y > _self._draggingElement._parent._height - metrics.getHeight())
						newPosition.y = _self._draggingElement._parent._height - metrics.getHeight();
					
					//Set position relative to parent.
					_self._draggingElement._setRelativePosition(
							newPosition.x, 
							newPosition.y, 
							 _self._draggingElement._parent);
					
					//TODO: Can probably be smarter about this... Check style states
					if (_self._draggingElement._parent instanceof AnchorContainerElement)
					{
						if (_self._draggingElement.getStyle("RotateCenterX") == null || _self._draggingElement.getStyle("RotateCenterY") == null)
						{
							if (_self._draggingElement.getStyle("X") != null)
								_self._draggingElement.setStyle("X", newPosition.x);
							if (_self._draggingElement.getStyle("Y") != null)
								_self._draggingElement.setStyle("Y", newPosition.y);
						}
						else
						{
							if (_self._draggingElement.getStyle("X") != null)
								_self._draggingElement.setStyle("X", _self._draggingElement._x);
							if (_self._draggingElement.getStyle("Y") != null)
								_self._draggingElement.setStyle("Y", _self._draggingElement._y);
							
							_self._draggingElement.setStyle("RotateCenterX", _self._draggingElement._rotateCenterX);
							_self._draggingElement.setStyle("RotateCenterY", _self._draggingElement._rotateCenterY);
						}
					}
					
					//Dispatch dragging.
					_self._draggingElement.dispatchEvent(new ElementEvent("dragging", false));
				}
			}
		};
}

//Inherit from AnchorContainerElement
CanvasManager.prototype = Object.create(AnchorContainerElement.prototype);
CanvasManager.prototype.constructor = CanvasManager;
CanvasManager.base = AnchorContainerElement;	


/////////////Style Types///////////////////////////////

CanvasManager._StyleTypes = Object.create(null);

/**
 * @style ShowRedrawRegion boolean
 * 
 * When true the canvas redraw region will be displayed.
 */
CanvasManager._StyleTypes.ShowRedrawRegion = 								StyleableBase.EStyleType.NORMAL;		

/**
 * @style AlertModalClass CanvasElement
 * 
 * CanvasElement constructor to be used for the alert overlay. Defaults to CanvasElement.
 * This may be set to null if you do not wish to block interactivity with the underlying UI when
 * an alert is triggered. 
 */
CanvasManager._StyleTypes.AlertModalClass = 								StyleableBase.EStyleType.NORMAL;		

/**
 * @style AlertModalStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the alert modal overlay.
 * Default definition sets BackgroundFill to "rgba(180,180,180,.4)".
 * If styled to be transparent, the modal element will still block user interactivity.
 */
CanvasManager._StyleTypes.AlertModalStyle = 								StyleableBase.EStyleType.SUBSTYLE;		


/////////////Default Styles///////////////////////////////

CanvasManager.AlertModalStyle = new StyleDefinition();
CanvasManager.AlertModalStyle.setStyle("BackgroundFill", 					"rgba(180,180,180,.4)");


CanvasManager.StyleDefault = new StyleDefinition();

CanvasManager.StyleDefault.setStyle("ShowRedrawRegion", 					false);							// true || false
CanvasManager.StyleDefault.setStyle("AlertModalClass", 						CanvasElement);
CanvasManager.StyleDefault.setStyle("AlertModalStyle", 						CanvasManager.AlertModalStyle);	



///////////////////CanvasManager Public Functions//////////////////////

/**
 * @function setCanvas
 * Sets the canvas that CanvasManager should manage.
 * 
 * @param canvas Canvas
 * Reference to the DOM canvas that CanvasManager should manage.
 */
CanvasManager.prototype.setCanvas = 
	function (canvas)
	{
		if (this._canvas == canvas)
			return;
	
		var addedOrRemoved = (this._canvas == null || canvas == null);
		
		//Clean up old canvas
		if (this._canvas != null)
		{
			window.removeEventListener('mousedown', this._canvasMouseEventHandler, false);
			window.removeEventListener('mousemove', this._canvasMouseEventHandler, false);
			window.removeEventListener("wheel", this._canvasMouseEventHandler, false);
			window.removeEventListener("keydown", this._canvasKeyboardEventHandler, false);
			window.removeEventListener("keyup", this._canvasKeyboardEventHandler, false);
			window.removeEventListener("resize", this._canvasResizeEventHandler, false);
			
			this._canvas.removeEventListener("focus", this._canvasFocusEventHandler, true);
			this._canvas.removeEventListener("blur", this._canvasFocusEventHandler, true);
			
			this._canvas = null;
			this._canvasContext = null;
		}

		if (canvas != null)
		{
			this._canvas = canvas;
			this._canvasContext = canvas.getContext("2d");
			
			window.addEventListener("mousedown", this._canvasMouseEventHandler, false);
			window.addEventListener("mousemove", this._canvasMouseEventHandler, false);
			window.addEventListener("wheel", this._canvasMouseEventHandler, false);
			window.addEventListener("keydown", this._canvasKeyboardEventHandler, false);
			window.addEventListener("keyup", this._canvasKeyboardEventHandler, false);
			window.addEventListener("resize", this._canvasResizeEventHandler, false);
			
			this._canvas.addEventListener("focus", this._canvasFocusEventHandler, true);
			this._canvas.addEventListener("blur", this._canvasFocusEventHandler, true);
					
			this._canvas.tabIndex = 1;
			this._canvas.style.outline = "none";
			this._canvas.style.cursor = "default";
			
			if (navigator.userAgent.indexOf("Firefox") > 0)
				CanvasElement._browserType = "Firefox";
			else if (navigator.userAgent.indexOf("Chrome") > 0)
				CanvasElement._browserType = "Chrome";
			
			//Prevent double render frames if someone changes our associated canvas.
			if (this._canvasRenderFramePending == false)
			{
				this._canvasRenderFramePending = true;
				window.requestAnimationFrame(this._onCanvasFrame);	
			}
		}
		
		if (addedOrRemoved == true)
		{
			this._propagateChildData();
			this._processAddRemoveDisplayChainQueue();
		}
		
		if (this._canvas != null)
		{
			this._rollOverInvalid = true;
			this._canvasResizeEventHandler();
		}
	};

/**
 * @function getCanvas
 * Gets the DOM canvas reference CanvasManager is currently managing.
 * 
 * @returns Canvas
 * The DOM canvas reference CanvasManager is currently managing.
 */	
CanvasManager.prototype.getCanvas = 
	function ()
	{
		return this._canvas;
	};

/**
 * @function addAlert
 * Adds an element to be displayed as an alert. Only one alert may
 * be displayed at a time, subsequent will be queued and displayed in the
 * order they have been queued as previous alerts are removed. 
 * If an AlertModalClass style is set (defaults to CanvasElement), the modal 
 * element will be placed covering all content under the alert element.
 * 
 * @returns CanvasElement
 * The element supplied to addAlert().
 */		
CanvasManager.prototype.addAlert = 
	function (element)
	{
		this._alertQueue.push(element);
		
		if (this._alertQueue.length == 1)
		{
			if (this._alertModal != null)
				this.addElement(this._alertModal);
			
			this.addElement(element);
		}
		
		return element;
	};
	
/**
 * @function removeAlert
 * Removes an alert element. If this is the last alert, the modal
 * element will also be removed.
 * 
 * @param element CanvasElement
 * The element to remove from the alert list.
 * 
 * @returns CanvasElement
 * The element supplied to removeAlert() or null if invalid.
 */		
CanvasManager.prototype.removeAlert = 
	function (element)
	{
		var index = this._alertQueue.indexOf(element);
		if (index == -1)
			return null;
	
		this._alertQueue.splice(index, 1);
		
		//We removed the current alert
		if (index == 0)
		{
			this.removeElement(element);
			
			if (this._alertModal != null)
			{
				if (this._alertQueue.length == 0)
					this.removeElement(this._alertModal); //No more alerts, remove modal
				else
					this.setElementIndex(this._alertModal, this.getNumElements() - 1); //more alerts in queue, push modal to end
			}
			
			//Add next alert in queue
			if (this._alertQueue.length > 0)
				this.addElement(this._alertQueue[0]);
		}			
		
		return element;
	};
	
/**
 * @function setLocale
 * Sets the locale to be used when using localized strings. The actual
 * locale value is arbitrary, this simply dispatches an event to notify elements
 * that the locale has changed. Its up to implementors to store their locale strings
 * and update/lookup accordingly. CanvasManager defaults locale to "en-us". 
 * 
 * @param locale String
 * The locale to change too.
 */	
CanvasManager.prototype.setLocale = 
	function (locale)
	{
		if (this._currentLocale == locale)
			return;
		
		this._currentLocale = locale;
		
		if (this._broadcastDispatcher.hasEventListener("localechanged", null) == true)
			this._broadcastDispatcher.dispatchEvent(new DispatcherEvent("localechanged"));
	};
	
/**
 * @function getLocale
 * Gets CanvasManager's current locale.
 * 
 * @returns String
 * String representing CanvasManager's current locale.
 */	
CanvasManager.prototype.getLocale = 
	function ()
	{
		return this._currentLocale;
	};
	
/**
 * @function addCursor
 * Adds a cursor to be used when the mouse is over the canvas. Cursors are managed
 * as a priority chain. Element roll-over cursors use priority 0 so setting any explicit
 * cursor such as a busy cursor should use a priority higher than 0, unless you want Elements
 * to override the canvas cursor on roll-over.
 * 
 * @param cursorDefinition CursorDefinition
 * The cursor to add. This may be a custom CursorDefinition and CanvasManager will hide
 * the native browser cursor and render the custom cursor. It also may be a standard
 * browser CSS cursor String such as "text".
 * 
 * @param priority int
 * The priority to assign to the cursor. Higher priorities override lower priorities.
 * 
 * @returns Object
 * A "cursor instance" object that is to be used to remove the cursor.
 */	
CanvasManager.prototype.addCursor = 
	function (cursorDefinition, priority)
	{
		if (priority == null)
			priority = 0;
	
		if (cursorDefinition instanceof CursorDefinition)
		{
			if (cursorDefinition._addedCount == 0)
				cursorDefinition.addEventListener("stylechanged", this._onCursorDefinitionStyleChangedInstance);
				
			cursorDefinition._addedCount++;
		}
		
		var cursorInstance = new CmLinkedNode();
		cursorInstance.data = cursorDefinition;
		cursorInstance.priority = priority;
		
		var lastCursor = this._cursorChain.back;
		if (lastCursor == null)
			this._cursorChain.pushBack(cursorInstance);
		else
		{
			while (lastCursor != null && lastCursor.priority > cursorInstance.priority)
				lastCursor = lastCursor.prev;
			
			if (lastCursor == null)
				this._cursorChain.pushFront(cursorInstance);
			else
				this._cursorChain.insertAfter(cursorInstance, lastCursor);
		}
		
		this._updateCursor();
		
		return cursorInstance;
	};

/**
 * @function removeCursor
 * Removes a cursor via the cursor instance object returned by addCursor().
 * 
 * @param cursorInstance Object
 * The cursor instance Object returned by addCursor().
 */	
CanvasManager.prototype.removeCursor = 
	function (cursorInstance)
	{
		if (cursorDefinition instanceof CursorDefinition)
		{
			var cursorDefinition = cursorInstance.data;
			cursorDefinition._addedCount--;
				
			if (cursorDefinition._addedCount == 0)
				cursorDefinition.removeEventListener("stylechanged", this._onCursorDefinitionStyleChangedInstance);
		}

		this._cursorChain.removeNode(cursorInstance);
		this._updateCursor();
		
		return true;
	};	
	
/**
 * @function updateNow
 * This is an internal function and should conceivably *never* be called.
 * This forces a full pass of the component life cycle and is incredibly expensive.
 * The system calls this once per render frame with the only known exception being immediately after a canvas resize.
 * If you think you need to call this, you probably have a design problem.
 * Documentation added for unforeseen circumstances. 
 */	
CanvasManager.prototype.updateNow = 
	function ()
	{
		if (this._broadcastDispatcher.hasEventListener("enterframe", null) == true)
			this._broadcastDispatcher.dispatchEvent(new DispatcherEvent("enterframe"));
	
		//Process state updates.
		while (this._updateStylesQueue.length > 0 || 
				this._updateMeasureQueue.length > 0 || 
				this._updateLayoutQueue.length > 0 || 
				this._rollOverInvalid == true ||
				this._updateRenderQueue.length > 0)
		{
			//Process styles queue.
			while (this._updateStylesQueue.length > 0)
				this._updateStylesQueue.removeSmallest().data._validateStyles();
			
			//Process measure queue.
			while (this._updateMeasureQueue.length > 0 && 
					this._updateStylesQueue.length == 0)
			{
				this._updateMeasureQueue.removeLargest().data._validateMeasure();
			}
			
			//Process layout queue.
			while (this._updateLayoutQueue.length > 0 && 
					this._updateMeasureQueue.length == 0 && 
					this._updateStylesQueue.length == 0)
			{
				this._updateLayoutQueue.removeSmallest().data._validateLayout();
			}
			
			//Do rollover/rollout/mousemove
			if (this._rollOverInvalid == true && 
				this._updateLayoutQueue.length == 0 && 
				this._updateMeasureQueue.length == 0 && 
				this._updateStylesQueue.length == 0)
			{
				this._rollOverInvalid = false;
				
				var i;
				var currentElement = null;
				var mousePoint = {x: this._mouseX, y:this._mouseY};
				
				var lastRollOverTarget = this._rollOverElement;
				var lastRollOverX = this._rollOverX;
				var lastRollOverY = this._rollOverY;
				
				this._rollOverElement = null;
				
				var rollOverCommonParent = null;
				var rollOverElements = [];
				
				//Make sure we're over the canvas.				
				if (mousePoint.x >= 0 && mousePoint.x <= this._width &&
					mousePoint.y >= 0 && mousePoint.y <= this._height)
				{
					currentElement = this;
					if (currentElement._mouseIsOver == false)
					{
						rollOverElements.push(currentElement);
						currentElement._mouseIsOver = true;
					}
					else
						rollOverCommonParent = currentElement;
					
					this._rollOverElement = currentElement; 
					this._rollOverX = mousePoint.x;
					this._rollOverY = mousePoint.y;
					
					var foundChild = false;
					while (true)
					{
						foundChild = false;
						for (i = currentElement._children.length -1; i >= 0; i--)
						{
							//Skip element if mouse is disabled or not visible.
							if (currentElement._children[i].getStyle("MouseEnabled") == false ||
								currentElement._children[i].getStyle("Visible") == false)
								continue;
							
							//Rotate the point backwards so we can translate the point to the element's rotated plane.
							currentElement._children[i].rotatePoint(mousePoint, true);
							
							if (mousePoint.x >= currentElement._children[i]._x && 
								mousePoint.x <= currentElement._children[i]._x + currentElement._children[i]._width &&
								mousePoint.y >= currentElement._children[i]._y &&
								mousePoint.y <= currentElement._children[i]._y + currentElement._children[i]._height)
							{
								currentElement = currentElement._children[i];
								if (currentElement._mouseIsOver == false)
								{
									rollOverElements.push(currentElement);
									currentElement._mouseIsOver = true;
								}								
								else
									rollOverCommonParent = currentElement;
								
								//Adjust the mouse point to within this element rather than its position in parent.
								mousePoint.x = mousePoint.x - currentElement._x;
								mousePoint.y = mousePoint.y - currentElement._y;
								
								this._rollOverElement = currentElement;
								this._rollOverX = mousePoint.x;
								this._rollOverY = mousePoint.y;
								
								foundChild = true;
								break;
							}
							
							//Rotate forwards, we're not over this child, undo the rotation.
							currentElement._children[i].rotatePoint(mousePoint, false);
						}
						
						if (foundChild == false)
							break;
					}
				}

				if (currentElement != null && 
					(this._rollOverElement != lastRollOverTarget || 
					this._rollOverX != lastRollOverX || 
					this._rollOverY != lastRollOverY))
				{
					currentElement.dispatchEvent(new ElementMouseEvent("mousemove", mousePoint.x, mousePoint.y));
				}
				
				this._broadcastDispatcher.dispatchEvent(new ElementMouseEvent("mousemoveex", this._mouseX, this._mouseY));
				
				if (lastRollOverTarget != null && this._rollOverElement != lastRollOverTarget)
				{
					var rollOutElements = [];
					currentElement = lastRollOverTarget;
					while (currentElement != rollOverCommonParent)
					{
						currentElement._mouseIsOver = false;
						rollOutElements.push(currentElement);
						currentElement = currentElement._parent;
					}
					
					for (i = 0; i < rollOutElements.length; i++)
						rollOutElements[i].dispatchEvent(new ElementEvent("rollout", false));
				}
				
				for (i = 0; i < rollOverElements.length; i++)
					rollOverElements[i].dispatchEvent(new ElementEvent("rollover", false));
			}
			
			//Process render queue.
			while (this._updateRenderQueue.length > 0 && 
					this._rollOverInvalid == false &&
					this._updateLayoutQueue.length == 0 && 
					this._updateMeasureQueue.length == 0 && 
					this._updateStylesQueue.length == 0)
			{
				this._updateRenderQueue.removeSmallest().data._validateRender();
			}
		}
		
		//Update redraw region
		while (this._updateRedrawRegionQueue.length > 0)
			this._updateRedrawRegionQueue.removeSmallest().data._validateRedrawRegion(this._redrawRegionCachePool);
		
		this._redrawRegionCachePool.cleanup();
		
		//Update transform region
		while (this._updateTransformRegionQueue.length > 0)
			this._updateTransformRegionQueue.removeLargest().data._validateTransformRegion();
		
		//Render composite layers.
		while (this._compositeRenderQueue.length > 0)
			this._compositeRenderQueue.removeLargest().data._validateCompositeRender();
		
		//Render redraw region
		if (this._redrawRegionPrevMetrics != null)
			this._invalidateCompositeRender();
		
		if (this._broadcastDispatcher.hasEventListener("exitframe", null) == true)
			this._broadcastDispatcher.dispatchEvent(new DispatcherEvent("exitframe"));
	};

/////////////CanvasManager Static Public Functions///////////////	

/**
 * @function getLocalMousePos
 * @static
 * Translates browser mouse event coordinates to canvas relative coordinates.
 * The system automatically calls this and translates raw browser events to 
 * system events to be consumed by CanvasElements. You probably never need to call this.
 * 
 * @param event BrowserEvent
 * The browser mouse event.
 * 
 * @param canvas Canvas
 * The DOM Canvas reference to translate the mouse coordinates too.
 * 
 * @returns Object
 * A point object containing {x, y}.
 */	
CanvasManager.getLocalMousePos = 
	function (event, canvas)
	{
		//Reliable way to get position with canvas scaling.
		var rect = canvas.getBoundingClientRect();
		return {
			x: Math.round((event.clientX - rect.left) / (rect.right - rect.left) * canvas.width),
			y: Math.round((event.clientY - rect.top) / (rect.bottom - rect.top) * canvas.height)
		};
	};	

//////////////Internal Functions////////////////

//@override	
CanvasManager.prototype._updateCompositeCanvas = 
	function ()
	{
		if (this._compositeCanvas == null)
		{
			this._compositeCanvas = this._canvas;
			this._compositeCtx = this._canvasContext;
		}
		
		if (this._compositeCanvas.width != this._width || 
			this._compositeCanvas.height != this._height)
		{
			this._compositeCanvas.width = this._width;
			this._compositeCanvas.height = this._height;
			
			if (this._compositeCanvasMetrics == null)
				this._compositeCanvasMetrics = new DrawMetrics();
			
			this._compositeCanvasMetrics._x = 0;
			this._compositeCanvasMetrics._y = 0;
			this._compositeCanvasMetrics._width = this._width;
			this._compositeCanvasMetrics._height = this._height;
			
			//Expand the redraw region to this whole canvas.
			if (this._redrawRegionMetrics == null)
				this._redrawRegionMetrics = this._compositeCanvasMetrics.clone();
			else
				this._redrawRegionMetrics.mergeExpand(this._compositeCanvasMetrics);
		}
	};	
	
//@override (add redraw region visibility)	
CanvasManager.prototype._validateCompositeRender =
	function ()
	{
		if (this.getStyle("ShowRedrawRegion") == true)
		{
			var currentRegion = null;
			if (this._redrawRegionMetrics != null)
			{
				currentRegion = this._redrawRegionMetrics.clone();
				currentRegion._x -= 1;
				currentRegion._y -= 1;
				currentRegion._width += 2;
				currentRegion._height += 2;
				currentRegion.roundUp();
				
				//Expand the redraw region to this whole canvas.
				if (this._redrawRegionPrevMetrics != null)
					this._redrawRegionMetrics.mergeExpand(this._redrawRegionPrevMetrics);
			}
			else if (this._redrawRegionPrevMetrics != null)
				this._redrawRegionMetrics = this._redrawRegionPrevMetrics.clone();
			
			CanvasManager.base.prototype._validateCompositeRender.call(this);
			
			if (currentRegion != null)
			{
				this._canvasContext.lineWidth = 1;
				this._canvasContext.strokeStyle = "#FF0000";
				
				this._canvasContext.beginPath();
				this._canvasContext.moveTo(currentRegion._x + .5, currentRegion._y + .5);
				this._canvasContext.lineTo(currentRegion._x + currentRegion._width - .5, currentRegion._y + .5);
				this._canvasContext.lineTo(currentRegion._x + currentRegion._width - .5, currentRegion._y + currentRegion._height - .5);
				this._canvasContext.lineTo(currentRegion._x + .5, currentRegion._y + currentRegion._height - .5);
				this._canvasContext.closePath();
				this._canvasContext.stroke();	
			}
			
			this._redrawRegionPrevMetrics = currentRegion;
		}
		else
		{
			CanvasManager.base.prototype._validateCompositeRender.call(this);
			this._redrawRegionPrevMetrics = null;
		}
	};	
	
//@private	
CanvasManager.prototype._updateFocusElement = 
	function (newFocusElement, renderFocusRing)
	{
		if (newFocusElement != this._focusElement)
		{
			if (this._focusElement != null)
			{
				this._focusElement._isFocused = false;
				this._focusElement._setRenderFocusRing(false);
				
				if (this._focusElement.hasEventListener("focusout", null) == true)
					this._focusElement.dispatchEvent(new ElementEvent("focusout", false));
			}
			
			this._focusElement = newFocusElement;
			
			if (this._focusElement != null)
			{
				this._focusElement._isFocused = true;
				this._focusElement._setRenderFocusRing(renderFocusRing);
				
				if (this._focusElement.hasEventListener("focusin", null) == true)
					this._focusElement.dispatchEvent(new ElementEvent("focusin", false));
			}
		}
	};
	
//@private	
CanvasManager.prototype._findChildTabStopForward = 
	function (parent, afterChild)
	{
		var index = 0;
		if (afterChild != null)
			index = parent._children.indexOf(afterChild) + 1;
		
		for (var i = index; i < parent._children.length; i++)
		{
			if (parent._children[i].getStyle("MouseEnabled") == false ||
				parent._children[i].getStyle("Visible") == false || 
				parent._children[i].getStyle("Enabled") == false)
				continue;
			
			if (parent._children[i].getStyle("TabStop") >= 0)
				return parent._children[i];
			
			tabToElement = this._findChildTabStopForward(parent._children[i], null);
			if (tabToElement != null)
				return tabToElement;
		}
		
		return null;
	};

//@private	
CanvasManager.prototype._findChildTabStopReverse = 
	function (parent, beforeChild)
	{
		var index = parent._children.length - 1;
		if (beforeChild != null)
			index = parent._children.indexOf(beforeChild) - 1;
		
		for (var i = index; i >= 0; i--)
		{
			if (parent._children[i].getStyle("MouseEnabled") == false ||
				parent._children[i].getStyle("Visible") == false || 
				parent._children[i].getStyle("Enabled") == false)
				continue;
			
			var childTabStop = this._findChildTabStopReverse(parent._children[i], null);
			if (childTabStop != null)
				return childTabStop;
	
			if (parent._children[i].getStyle("TabStop") >= 0)
				return parent._children[i];
		}
		
		return null;
	};	
	
//@private	
CanvasManager.prototype._updateCursor = 
	function ()
	{
		var cursorDefinition = null;
		if (this._cursorChain.back != null)
			cursorDefinition = this._cursorChain.back.data;
		
		var displayedCursorElement = null;
		if (this._cursorContainer._getNumChildren() > 0)
			displayedCursorElement = this._cursorContainer._getChildAt(0);
		
		if (cursorDefinition != null)
		{
			var cursorElement = null;
			if (!(typeof cursorDefinition === "string" || cursorDefinition instanceof String))
			{
				var cursorClass = cursorDefinition.getStyle("CursorClass");
				
				if (cursorClass == null)
					cursorDefinition._cursorElement = null;
				else
				{
					if (cursorDefinition._cursorElement == null || 
						cursorDefinition._cursorElement.constructor != cursorClass)
					{
						cursorDefinition._cursorElement = new (cursorClass)();
						cursorDefinition._cursorElement.setStyleDefinitions(cursorDefinition.getStyle("CursorStyle"));
					}
					else
						cursorDefinition._cursorElement.setStyleDefinitions(cursorDefinition.getStyle("CursorStyle"));
					
					cursorElement = cursorDefinition._cursorElement;
				}
			}
			
			if (displayedCursorElement != cursorElement)
			{
				if (displayedCursorElement != null)
					this._cursorContainer._removeChild(displayedCursorElement);
				if (cursorElement != null)
					this._cursorContainer._addChild(cursorElement);
			}
			
			if (cursorElement != null)
			{
				if (this._browserCursor != "none")
				{
					this._browserCursor = "none";
					this._canvas.style.cursor = "none";
				}
					
				//Make visible if we're over canvas			
				if (this._mouseX >= 0 && this._mouseX <= this._width &&
					this._mouseY >= 0 && this._mouseY <= this._height)
				{
					var cursorWidth = cursorDefinition._cursorElement._getStyledOrMeasuredWidth();
					var cursorHeight = cursorDefinition._cursorElement._getStyledOrMeasuredHeight();
					var offsetX = cursorDefinition.getStyle("CursorOffsetX");
					var offsetY = cursorDefinition.getStyle("CursorOffsetY");
					
					cursorElement._setActualPosition(this._mouseX + offsetX, this._mouseY + offsetY);
					cursorElement._setActualSize(cursorWidth, cursorHeight);
					cursorElement.setStyle("Visible", true);
				}
				else //Hide' mouse is no longer over canvas
					cursorElement.setStyle("Visible", false);
			}
			else if (this._browserCursor != cursorDefinition)
			{
				this._browserCursor = cursorDefinition;
				this._canvas.style.cursor = cursorDefinition;
			}
		}
		else
		{
			if (displayedCursorElement != null)
				this._cursorContainer._removeChildAt(0);
			
			if (this._browserCursor != "default")
			{
				this._browserCursor = "default";
				this._canvas.style.cursor = "default";
			}
		}
	};
	
//@private	
CanvasManager.prototype._onCursorDefinitionStyleChanged = 
	function (styleChangedEvent)
	{
		var cursorDefinition = styleChangedEvent.getTarget();
		
		var styleName = styleChangedEvent.getStyleName();
		if (styleName == "CursorClass" && cursorDefinition._cursorElement != null)
		{
			var cursorClass = cursorDefinition.getStyle("CursorClass");
			if (cursorDefinition._cursorElement.constructor != cursorClass)
				cursorDefinition._cursorElement = null;
		}
		if (styleName == "CursorStyle" && cursorDefinition._cursorElement != null)
			cursorDefinition._cursorElement.setStyleDefinitions(this.getStyle("CursorStyle"));
		
		this._updateCursor();
	};
	
//@private	
CanvasManager.prototype._pushAddRemoveDisplayChainQueue = 
	function (element, type)
	{
		var node = new CmLinkedNode();
		node.data = {element:element, type:type};
		
		this._addRemoveDisplayChainQueue.pushBack(node);
	};

//@private	
CanvasManager.prototype._popAddRemoveDisplayChainQueue = 
	function ()
	{
		if (this._addRemoveDisplayChainQueue.length == 0)
			return null;
		
		var data = this._addRemoveDisplayChainQueue.front.data;
		this._addRemoveDisplayChainQueue.removeNode(this._addRemoveDisplayChainQueue.front);
		
		return data;
	};

//@private	
CanvasManager.prototype._processAddRemoveDisplayChainQueue = 
	function ()
	{
		//Recursion guard. An event may add or remove other elements, we dont want this function to recurse.
		if (this._addRemoveDisplayChainQueueProcessing == true)
			return;
		
		//Block recursion
		this._addRemoveDisplayChainQueueProcessing = true;
		
		var addRemoveData = this._popAddRemoveDisplayChainQueue();
		while (addRemoveData != null)
		{
			addRemoveData.element.dispatchEvent(new AddedRemovedEvent(addRemoveData.type, this));
			addRemoveData = this._popAddRemoveDisplayChainQueue();
		}
		
		//Queue emtpy, allow processing again.
		this._addRemoveDisplayChainQueueProcessing = false;
	};

//@private	
CanvasManager.prototype._clearDraggingElement = 
	function ()
	{
		if (this._draggingElement == null)
			return;

		this._draggingElement = null;
		this._draggingOffsetX = null;
		this._draggingOffsetY = null;
	};

//@private	
CanvasManager.prototype._setDraggingElement = 
	function (element, offsetX, offsetY)
	{
		if (this._draggingElement != null)
			return;

		this._draggingElement = element;
		this._draggingOffsetX = offsetX;
		this._draggingOffsetY = offsetY;
	};
	
//@override	
CanvasManager.prototype._doStylesUpdated = 
	function (stylesMap)
	{
		CanvasManager.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		var newModal = false;
		if ("AlertModalClass" in stylesMap)
		{
			var modalClass = this.getStyle("AlertModalClass");
			
			//Destroy if class is null or does not match existing
			if ((modalClass == null && this._alertModal != null) ||
				this._alertModal != null && this._alertModal.constructor != modalClass)
			{
				//Try to remove it (it may not be attached)
				this.removeElementAt(this.getElementIndex(this._alertModal));
				this._alertModal = null;
			}
			
			//Create
			if (modalClass != null && this._alertModal == null)
			{
				newModal = true;
				this._alertModal = new (modalClass)();
				this._alertModal.setStyle("IncludeInLayout", false); //Always false, we do this manually to prevent layout styles
				
				//Add behind first alert position
				if (this._alertQueue.length > 0)
					this.addElementAt(this._alertModal, this.getElementIndex(this._alertQueue[0]));
			}
		}
		
		if (this._alertModal != null && ("AlertModalStyle" in stylesMap || newModal == true))
			this._applySubStylesToElement("AlertModalStyle", this._alertModal);
	};
	
//@override	
CanvasManager.prototype._doLayout = 
	function (paddingMetrics)
	{
		CanvasManager.base.prototype._doLayout.call(this, paddingMetrics);
		
		this._alertModal._setActualSize(this._width, this._height);
		this._cursorContainer._setActualSize(this._width, this._height);
	};	



//////////Private Helper Classes////////////////////

function CmRedrawRegionCachePool()
{
	this.pointRawTl = {x:0, y:0};
	this.pointRawTr = {x:0, y:0};
	this.pointRawBr = {x:0, y:0};
	this.pointRawBl = {x:0, y:0};
	
	this.pointDrawableTl = {x:0, y:0};
	this.pointDrawableTr = {x:0, y:0};
	this.pointDrawableBr = {x:0, y:0};
	this.pointDrawableBl = {x:0, y:0};
	
	this.compositeMetrics = [];
	
	this.rawMetrics = new DrawMetrics();		
	this.drawableMetrics = new DrawMetrics();	
	this.clipMetrics = new DrawMetrics();
	this.shadowMetrics = new DrawMetrics();
};

CmRedrawRegionCachePool.prototype.cleanup = 
	function ()
	{
		this.compositeMetrics.length = 0;
	};

//Used exclusively by CanvasManager//

//Queue used for processing component cycles (styles, measure, layout) based on display chain depth.
function CmDepthQueue()
{
	this.depthArrayOfLists = []; //Array of CmLinkedList, index based on depth.
	this.length = 0;
	
	//Stores current start/end populated indexes of depthArrayOfLists for performance.
	this.minDepth = -1;
	this.maxDepth = -1;
}

CmDepthQueue.prototype.addNode = 
	function (node, depth)
	{
		var depthToIndex = depth - 1;
	
		if (this.depthArrayOfLists[depthToIndex] == null)
			this.depthArrayOfLists[depthToIndex] = new CmLinkedList();
		
		this.depthArrayOfLists[depthToIndex].pushBack(node);
		
		this.length = this.length + 1;
		
		if (depthToIndex < this.minDepth || this.minDepth == -1)
			this.minDepth = depthToIndex;
		if (depthToIndex > this.maxDepth)
			this.maxDepth = depthToIndex;
	};
	
CmDepthQueue.prototype.removeNode = 
	function (node, depth)
	{
		var depthToIndex = depth - 1; 
	
		this.depthArrayOfLists[depthToIndex].removeNode(node);
		
		this.length = this.length - 1;
		if (this.length == 0)
		{
			this.minDepth = -1;
			this.maxDepth = -1;
		}
	};
	
CmDepthQueue.prototype.removeSmallest = 
	function ()
	{
		if (this.length == 0)
			return null;
		
		for (var i = this.minDepth; i < this.depthArrayOfLists.length; i++)
		{
			this.minDepth = i;
			if (this.depthArrayOfLists[i] == null || this.depthArrayOfLists[i].length == 0)
				continue;
			
			var node = this.depthArrayOfLists[i].front;
			this.depthArrayOfLists[i].removeNode(node);
			
			this.length = this.length - 1;
			if (this.length == 0)
			{
				this.minDepth = -1;
				this.maxDepth = -1;
			}
			
			return node;
		}
	};
	
CmDepthQueue.prototype.removeLargest = 
	function ()
	{
		if (this.length == 0)
			return null;
		
		for (var i = this.maxDepth; i >= 0; i--)
		{
			this.maxDepth = i;
			if (this.depthArrayOfLists[i] == null || this.depthArrayOfLists[i].length == 0)
				continue;
			
			var node = this.depthArrayOfLists[i].back;
			this.depthArrayOfLists[i].removeNode(node);
			
			this.length = this.length - 1;
			if (this.length == 0)
			{
				this.minDepth = -1;
				this.maxDepth = -1;
			}
			
			return node;
		}
	};
	
//Basic linked list	
function CmLinkedList()
{
	this.front = null;
	this.back = null;
	
	this.length = 0;
}

CmLinkedList.prototype.pushFront = 
	function (cmLinkedNode)
	{
		this.length++;
		
		if (this.front == null)
		{
			cmLinkedNode.prev = null;
			cmLinkedNode.next = null;
			
			this.front = cmLinkedNode;
			this.back = cmLinkedNode;
		}
		else
		{
			cmLinkedNode.prev = null;
			cmLinkedNode.next = this.front;
			
			this.front.prev = cmLinkedNode;
			this.front = cmLinkedNode;
		}
	};
	
CmLinkedList.prototype.pushBack =
	function (cmLinkedNode)
	{
		this.length++;
	
		if (this.back == null)
		{
			cmLinkedNode.prev = null;
			cmLinkedNode.next = null;
			
			this.front = cmLinkedNode;
			this.back = cmLinkedNode;
		}
		else
		{
			cmLinkedNode.prev = this.back;
			cmLinkedNode.next = null;
			
			this.back.next = cmLinkedNode;
			this.back = cmLinkedNode;
		}
	};

CmLinkedList.prototype.insertBefore = 
	function (cmLinkedNode, beforeCmLinkedNode)	
	{
		this.length++;
		
		if (this.front == beforeCmLinkedNode)
			this.front = cmLinkedNode;
		
		if (beforeCmLinkedNode.prev != null)
			beforeCmLinkedNode.prev.next = cmLinkedNode;
		
		cmLinkedNode.prev = beforeCmLinkedNode.prev;
		cmLinkedNode.next = beforeCmLinkedNode;
		beforeCmLinkedNode.prev = cmLinkedNode;
	};

CmLinkedList.prototype.insertAfter = 
	function (cmLinkedNode, afterCmLinkedNode)
	{
		this.length++;
		
		if (this.back == afterCmLinkedNode)
			this.back = cmLinkedNode;
		
		if (afterCmLinkedNode.next != null)
			afterCmLinkedNode.next.prev = cmLinkedNode;
		
		cmLinkedNode.next = afterCmLinkedNode.next;
		cmLinkedNode.prev = afterCmLinkedNode;
		afterCmLinkedNode.next = cmLinkedNode;		
	};
	
CmLinkedList.prototype.removeNode = 
	function (cmLinkedNode)
	{
		if (cmLinkedNode == null)
			return null;
		
		this.length--;
		
		if (this.front == cmLinkedNode)
			this.front = cmLinkedNode.next;
		if (this.back == cmLinkedNode)
			this.back = cmLinkedNode.prev;
		
		if (cmLinkedNode.prev != null)
			cmLinkedNode.prev.next = cmLinkedNode.next;
		if (cmLinkedNode.next != null)
			cmLinkedNode.next.prev = cmLinkedNode.prev;
		
		cmLinkedNode.next = null;
		cmLinkedNode.prev = null;
	};
	
//Linked list iterator	
function CmLinkedNode()
{
	this.prev = null;
	this.next = null;
	
	this.data = null;
}





/**
 * @depends SkinnableElement.js
 */

/////////////////////////////////////////////////
//////////////////ButtonElement//////////////////

/**
 * @class ButtonElement
 * @inherits SkinnableElement
 * 
 * Button is a skin-able element that supports 4 states corresponding to mouse states
 * "up", "over", "down" and "disabled". It also has an optional label. 
 * 
 * Being a SkinnableElement, Button proxies its styles to its skins. 
 * You may assign custom skins and assign any styles you wish to apply to all skins to the Button itself. 
 * 
 * Button is used as a base class for many click-able elements such as
 * ToggleButton, Checkbox, RadioButton, etc. 
 * 
 * 
 * @constructor ButtonElement 
 * Creates new ButtonElement instance.
 */
function ButtonElement()
{
	ButtonElement.base.prototype.constructor.call(this);

	var _self = this;
	
	this._labelElement = null;
	
	//Private handler, need different instance for each button, proxy to prototype.	
	this._onButtonEventInstance = 
		function (elementEvent)
		{
			if (elementEvent.getType() == "mousedown")
				_self._onButtonMouseDown(elementEvent);
			else if (elementEvent.getType() == "mouseup")
				_self._onButtonMouseUp(elementEvent);
			else if (elementEvent.getType() == "click")
				_self._onButtonClick(elementEvent);
			else if (elementEvent.getType() == "rollover")
				_self._onButtonRollover(elementEvent);
			else if (elementEvent.getType() == "rollout")
				_self._onButtonRollout(elementEvent);
		};
		
	this.addEventListener("mousedown", this._onButtonEventInstance);
	this.addEventListener("mouseup", this._onButtonEventInstance);
	this.addEventListener("rollover", this._onButtonEventInstance);
	this.addEventListener("rollout", this._onButtonEventInstance);
	this.addEventListener("click", this._onButtonEventInstance);
}

//Inherit from SkinnableElement
ButtonElement.prototype = Object.create(SkinnableElement.prototype);
ButtonElement.prototype.constructor = ButtonElement;
ButtonElement.base = SkinnableElement;


/////////////Style Types///////////////////////////////

ButtonElement._StyleTypes = Object.create(null);

//New button specific styles.

/**
 * @style Text String
 * 
 * Text string to be displayed as the button label.
 */
ButtonElement._StyleTypes.Text = 						StyleableBase.EStyleType.NORMAL;		// "any string" || null

/**
 * @style SkinClass CanvasElement
 * 
 * The CanvasElement constructor type to apply to all skin states. 
 * Specific states such as UpSkinClass will override SkinClass when they are equal priority.
 */
ButtonElement._StyleTypes.SkinClass =					StyleableBase.EStyleType.NORMAL;	//Element constructor()

/**
 * @style UpSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the button skin when the button is in the "up" state. 
 * This will override SkinClass when equal or higher priority than SkinClass.
 */
ButtonElement._StyleTypes.UpSkinClass = 				StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style UpSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "up" state skin element.
 */
ButtonElement._StyleTypes.UpSkinStyle = 				StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style UpTextColor String
 * 
 * Hex color value to be used for the button label when the button is in the "up" state. Format like "#FF0000" (red).
 * This will override TextColor when equal or higher priority than TextColor.
 */
ButtonElement._StyleTypes.UpTextColor = 				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style OverSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the button skin when the button is in the "over" state. 
 * This will override SkinClass when equal or higher priority than SkinClass.
 */
ButtonElement._StyleTypes.OverSkinClass = 				StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style OverSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "over" state skin element.
 */
ButtonElement._StyleTypes.OverSkinStyle = 				StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style OverTextColor String
 * 
 * Hex color value to be used for the button label when the button is in the "over" state. Format like "#FF0000" (red).
 * This will override TextColor when equal or higher priority than TextColor.
 */
ButtonElement._StyleTypes.OverTextColor = 				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style DownSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the button skin when the button is in the "down" state. 
 * This will override SkinClass when equal or higher priority than SkinClass.
 */
ButtonElement._StyleTypes.DownSkinClass = 				StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style DownSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "down" state skin element.
 */
ButtonElement._StyleTypes.DownSkinStyle = 				StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style DownTextColor String
 * 
 * Hex color value to be used for the button label when the button is in the "down" state. Format like "#FF0000" (red).
 * This will override TextColor when equal or higher priority than TextColor.
 */
ButtonElement._StyleTypes.DownTextColor = 				StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style DisabledSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the button skin when the button is in the "disabled" state. 
 * This will override SkinClass when equal or higher priority than SkinClass.
 */
ButtonElement._StyleTypes.DisabledSkinClass = 			StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style DisabledSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "disabled" state skin element.
 */
ButtonElement._StyleTypes.DisabledSkinStyle = 			StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style DisabledTextColor String
 * 
 * Hex color value to be used for the button label when the button is in the "disabled" state. Format like "#FF0000" (red).
 * This will override TextColor when equal or higher priority than TextColor.
 */
ButtonElement._StyleTypes.DisabledTextColor = 			StyleableBase.EStyleType.NORMAL;		//"#000000"


//Change some of the text styles not to inherit, we'll set these to the label 
//so the label will use button defaults if no style explicitly set.

/**
 * @style TextHorizontalAlign String
 * 
 * Determines alignment when rendering text. Available values are "left", "center", and "right".
 */
ButtonElement._StyleTypes.TextHorizontalAlign =			StyleableBase.EStyleType.NORMAL;		// "left" || "center" || "right"

/**
 * @style TextVerticalAlign String
 * 
 * Determines the baseline when rendering text. Available values are "top", "middle", or "bottom".
 */
ButtonElement._StyleTypes.TextVerticalAlign =			StyleableBase.EStyleType.NORMAL;  		// "top" || "middle" || "bottom"


/////////Default Styles//////////////////////////////

ButtonElement.StyleDefault = new StyleDefinition();

//Override base class styles
ButtonElement.StyleDefault.setStyle("PaddingTop",						3);
ButtonElement.StyleDefault.setStyle("PaddingBottom",                    3);
ButtonElement.StyleDefault.setStyle("PaddingLeft",                      4);
ButtonElement.StyleDefault.setStyle("PaddingRight",                     4);

ButtonElement.StyleDefault.setStyle("TextHorizontalAlign", 				"center"); 
ButtonElement.StyleDefault.setStyle("TextVerticalAlign", 				"middle");

ButtonElement.StyleDefault.setStyle("TabStop", 							0);			// number

//ButtonElement specific styles.
ButtonElement.StyleDefault.setStyle("Text", 							null);
ButtonElement.StyleDefault.setStyle("SkinClass", 						CanvasElement); //Not necessary, just for completeness

ButtonElement.StyleDefault.setStyle("UpSkinClass", 						CanvasElement);
ButtonElement.StyleDefault.setStyle("OverSkinClass", 					CanvasElement);
ButtonElement.StyleDefault.setStyle("DownSkinClass", 					CanvasElement);
ButtonElement.StyleDefault.setStyle("DisabledSkinClass", 				CanvasElement);

ButtonElement.StyleDefault.setStyle("UpTextColor", 						"#000000");
ButtonElement.StyleDefault.setStyle("OverTextColor", 					"#000000");
ButtonElement.StyleDefault.setStyle("DownTextColor", 					"#000000");
ButtonElement.StyleDefault.setStyle("DisabledTextColor", 				"#888888");

//Skin Defaults////////////////////////////
ButtonElement.UpSkinStyleDefault = new StyleDefinition();

ButtonElement.UpSkinStyleDefault.setStyle("BorderType", 				"solid");
ButtonElement.UpSkinStyleDefault.setStyle("BorderThickness", 			1);
ButtonElement.UpSkinStyleDefault.setStyle("BorderColor", 				"#333333");
ButtonElement.UpSkinStyleDefault.setStyle("BackgroundFill", 			"#EBEBEB");

ButtonElement.OverSkinStyleDefault = new StyleDefinition();

ButtonElement.OverSkinStyleDefault.setStyle("BorderType", 				"solid");
ButtonElement.OverSkinStyleDefault.setStyle("BorderThickness", 			1);
ButtonElement.OverSkinStyleDefault.setStyle("BorderColor", 				"#333333");
ButtonElement.OverSkinStyleDefault.setStyle("BackgroundFill", 			"#DDDDDD");

ButtonElement.DownSkinStyleDefault = new StyleDefinition();

ButtonElement.DownSkinStyleDefault.setStyle("BorderType", 				"solid");
ButtonElement.DownSkinStyleDefault.setStyle("BorderThickness", 			1);
ButtonElement.DownSkinStyleDefault.setStyle("BorderColor", 				"#333333");
ButtonElement.DownSkinStyleDefault.setStyle("BackgroundFill", 			"#CCCCCC");

ButtonElement.DisabledSkinStyleDefault = new StyleDefinition();

ButtonElement.DisabledSkinStyleDefault.setStyle("BorderType", 			"solid");
ButtonElement.DisabledSkinStyleDefault.setStyle("BorderThickness", 		1);
ButtonElement.DisabledSkinStyleDefault.setStyle("BorderColor", 			"#999999");
ButtonElement.DisabledSkinStyleDefault.setStyle("BackgroundFill", 		"#ECECEC");
/////////////////////////////////////////////////

//Apply Skin Defaults
ButtonElement.StyleDefault.setStyle("UpSkinStyle", 						ButtonElement.UpSkinStyleDefault);
ButtonElement.StyleDefault.setStyle("OverSkinStyle", 					ButtonElement.OverSkinStyleDefault);
ButtonElement.StyleDefault.setStyle("DownSkinStyle", 					ButtonElement.DownSkinStyleDefault);
ButtonElement.StyleDefault.setStyle("DisabledSkinStyle", 				ButtonElement.DisabledSkinStyleDefault);


	
/////////////ButtonElement Protected Functions/////////////////////	
	
/**
 * @function _updateState
 * Called in response to mouse events, and when the Button is added to the display hierarchy (if mouse is enabled).
 * Updates the Button skin state.
 */
ButtonElement.prototype._updateState = 
	function ()
	{
		var newState = "up";
	
		if (this.getStyle("Enabled") == false)
			newState = "disabled";
		else
		{
			if (this._mouseIsDown == true)
				newState = "down";
			else if (this._mouseIsOver == true)
				newState = "over";
		}
		
		this.setStyle("SkinState", newState);
	};

/**
 * @function _onButtonMouseDown
 * Event handler for "mousedown" event. Updates the Button skin state.
 * Overriding this is more efficient than adding an additional "mousedown" event listener.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */	
ButtonElement.prototype._onButtonMouseDown = 
	function (elementMouseEvent)
	{
		this._updateState();
	};
	
/**
 * @function _onButtonMouseUp
 * Event handler for "mouseup" event. Updates the Button skin state.
 * Overriding this is more efficient than adding an additional "mouseup" event listener.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */		
ButtonElement.prototype._onButtonMouseUp = 
	function (elementMouseEvent)
	{
		this._updateState();
	};		

/**
 * @function _onButtonRollover
 * Event handler for "rollover" event. Updates the Button skin state.
 * Overriding this is more efficient than adding an additional "rollover" event listener.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
ButtonElement.prototype._onButtonRollover = 
	function (elementEvent)
	{
		this._updateState();
	};

/**
 * @function _onButtonRollout
 * Event handler for "rollout" event. Updates the Button skin state.
 * Overriding this is more efficient than adding an additional "rollout" event listener.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */		
ButtonElement.prototype._onButtonRollout = 
	function (elementEvent)
	{
		this._updateState();
	};	
	
/**
 * @function _onButtonClick
 * Event handler for "click" event. Cancels the event if the Button is disabled.
 * Overriding this is more efficient than adding an additional "click" event listener.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */			
ButtonElement.prototype._onButtonClick = 
	function (elementMouseEvent)
	{
		//Implementor will not expect a click event when button is disabled. 
		if (this.getStyle("Enabled") == false)
			elementMouseEvent.cancelEvent();
	};
	
//@override
ButtonElement.prototype._getSkinClass = 
	function (state)
	{
		var stateSkinClass = null;
	
		if (state == "up")
			stateSkinClass = this.getStyleData("UpSkinClass");
		else if (state == "over")
			stateSkinClass = this.getStyleData("OverSkinClass");
		else if (state == "down")
			stateSkinClass = this.getStyleData("DownSkinClass");
		else if (state == "disabled")
			stateSkinClass = this.getStyleData("DisabledSkinClass");
		
		var skinClass = this.getStyleData("SkinClass");
		
		//Shouldnt have null stateSkinClass
		if (stateSkinClass == null || skinClass.comparePriority(stateSkinClass) > 0) //Use skinClass if higher priority
			return skinClass.value;
		
		return stateSkinClass.value;
	};

//@override	
ButtonElement.prototype._getSubStyleNameForSkinState = 
	function (state)
	{
		if (state == "up")
			return "UpSkinStyle";
		if (state == "over")
			return "OverSkinStyle";
		if (state == "down")
			return "DownSkinStyle";
		if (state == "disabled")
			return "DisabledSkinStyle";
		
		return ButtonElement.base.prototype._getSubStyleNameForSkinState.call(this, state);
	};	
	
//@override
ButtonElement.prototype._changeState = 
	function (state)
	{
		ButtonElement.base.prototype._changeState.call(this, state);
		
		this._updateTextColor();
	};
	
/**
 * @function _getTextColor
 * Gets the text color to be used for the supplied state. 
 * Override this to add styles for additional states.
 * 
 * @param state String
 * String representing the state to return the text color style.
 * 
 * @returns string
 * Text color for the supplied state.
 */	
ButtonElement.prototype._getTextColor = 
	function (state)
	{
		var stateTextColor = null;
		
		if (state == "up")
			stateTextColor = this.getStyleData("UpTextColor");
		else if (state == "over")
			stateTextColor = this.getStyleData("OverTextColor");
		else if (state == "down")
			stateTextColor = this.getStyleData("DownTextColor");
		else if (state == "disabled")
			stateTextColor = this.getStyleData("DisabledTextColor");

		var textColor = this.getStyleData("TextColor");
		
		//Shouldnt have null stateTextColor
		if (stateTextColor == null || textColor.comparePriority(stateTextColor) > 0) //Use textColor if higher priority
			return textColor.value;
		
		return stateTextColor.value;
	};

/**
 * @function _updateTextColor
 * Updates the text color in response to state changes.
 */		
ButtonElement.prototype._updateTextColor = 
	function ()
	{
		if (this._labelElement == null)
			return;
		
		this._labelElement.setStyle("TextColor", this._getTextColor(this._currentSkinState));
	};
	
/**
 * @function _updateText
 * Updates the buttons text per styling. 
 * This function calls _updateLabelText(text)
 * Override this if you need to change the source of the label text and 
 * call _updateLabelText accordingly.
 */	
ButtonElement.prototype._updateText = 
	function ()
	{
		this._setLabelText(this.getStyle("Text"));
	};

/**
 * @function _setLabelText
 * Sets supplied text to the buttons label element, adds or destroys the label as necessary.
 * 
 * @param text String
 * String to be used for the label text.
 */		
ButtonElement.prototype._setLabelText = 
	function (text)
	{
		if (text == null || text == "")
		{
			if (this._labelElement != null)
			{
				this._removeChild(this._labelElement);
				this._labelElement = null;
			}
		}
		else
		{
			if (this._labelElement == null)
			{
				this._labelElement = this._createLabel();
				if (this._labelElement != null)
				{
					this._updateTextColor();
					this._addChild(this._labelElement);
				}
			}
			
			if (this._labelElement != null)
				this._labelElement.setStyle("Text", text);
		}	
	};
	
//@override
ButtonElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ButtonElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		////Update skin classes and sub styles.
		if ("SkinClass" in stylesMap || "UpSkinClass" in stylesMap)
			this._updateSkinClass("up");
		if ("UpSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("up");
		
		if ("SkinClass" in stylesMap || "OverSkinClass" in stylesMap)
			this._updateSkinClass("over");
		if ("OverSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("over");
		
		if ("SkinClass" in stylesMap || "DownSkinClass" in stylesMap)
			this._updateSkinClass("down");
		if ("DownSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("down");
		
		if ("SkinClass" in stylesMap || "DisabledSkinClass" in stylesMap)
			this._updateSkinClass("disabled");
		if ("DisabledSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("disabled");
		
		//Create / Destroy and proxy text to label.
		if ("Text" in stylesMap)
			this._updateText();
		
		//Only update the state if mouse is enabled, when disabled it means states are being manually controlled.
		if (this.getStyle("MouseEnabled") == false)
			this.clearStyle("SkinState");
		else if ("Enabled" in stylesMap)
			this._updateState();
		
		if ("TextHorizontalAlign" in stylesMap && this._labelElement != null)
			this._labelElement.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
		
		if ("TextVerticalAlign" in stylesMap && this._labelElement != null)
			this._labelElement.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
		
		//Always call (can optimize by checking for all text color styles)
		this._updateTextColor();
	};	
	
/**
 * @function _createLabel
 * Creates the Button's label instance when Text style is not null or empty.
 * 
 * @returns LabelElement
 * New LabelElement instance
 */	
ButtonElement.prototype._createLabel = 
	function ()
	{
		var label = new LabelElement();
	
		label.setStyle("MouseEnabled", false);
		label.setStyle("TextHorizontalAlign", this.getStyle("TextHorizontalAlign"));
		label.setStyle("TextVerticalAlign", this.getStyle("TextVerticalAlign"));
		
		label.setStyle("Padding", 0); //Wipe out default padding (no doubly padding, only this elements padding is necessary)
		
		return label;
	};
	
//@override
ButtonElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var measuredWidth = 0;
		var measuredHeight = 0;
	
		//Base size off of label.
		if (this._labelElement != null)
		{
			var labelWidth = this._labelElement._getStyledOrMeasuredWidth();
			var labelHeight = this._labelElement._getStyledOrMeasuredHeight();
			
			measuredWidth = labelWidth + padWidth;
			measuredHeight = labelHeight + padHeight;

			this._setMeasuredSize(measuredWidth, measuredHeight);
		}
		else
			ButtonElement.base.prototype._doMeasure.call(this, padWidth, padHeight);
	};

//@override	
ButtonElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		ButtonElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		if (this._labelElement != null)
		{
			this._labelElement._setActualPosition(paddingMetrics.getX(), paddingMetrics.getY());
			this._labelElement._setActualSize(paddingMetrics.getWidth(), paddingMetrics.getHeight());
		}
	};	
	


/**
 * @depends ButtonElement.js
 */

///////////////////////////////////////////////////////////////////////////	
///////////////////////ToggleButtonElement/////////////////////////////////

/**
 * @class ToggleButtonElement
 * @inherits ButtonElement
 * 
 * ToggleButton is identical to a button except that it adds "selected" versions of
 * the 4 button states and Toggles from selected to not-selected when clicked. It also
 * dispatches a "changed" event when the selected state changes.
 * 
 * ToggleButton selected states:
 * "selectedUp", "selectedOver", "selectedDown", "selectedDisabled".
 * 
 * Being a SkinnableElement, ToggleButton proxies its styles to its skins. 
 * You may assign custom skins and assign any styles you wish to apply to all skins to the ToggleButton itself. 
 * 
 * ToggleButton is a base class for components such as Checkbox and RadioButton.
 * 
 * 
 * @constructor ToggleButtonElement 
 * Creates new ToggleButtonElement instance.
 */
function ToggleButtonElement()
{
	ToggleButtonElement.base.prototype.constructor.call(this);
	
	this._isSelected = false;
}

//Inherit from ButtonElement
ToggleButtonElement.prototype = Object.create(ButtonElement.prototype);
ToggleButtonElement.prototype.constructor = ToggleButtonElement;
ToggleButtonElement.base = ButtonElement;

////////////Events/////////////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the ToggleButton's selection state changes as a result of user interaction.
 */


/////////////Style Types///////////////////////////////

ToggleButtonElement._StyleTypes = Object.create(null);

//New toggle button specific styles.

/**
 * @style AllowDeselect boolean
 * 
 * When false, the ToggleButton cannot be de-selected by the user and the "selectedOver" and "selectedDown" states are not used, 
 * as with the case for most tab or radio button type elements.
 */
ToggleButtonElement._StyleTypes.AllowDeselect = 				StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style SelectedUpSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the button skin when the button is in the "selectedUp" state. 
 * This will override SkinClass when equal or higher priority than SkinClass.
 */
ToggleButtonElement._StyleTypes.SelectedUpSkinClass = 			StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style SelectedUpSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "selectedUp" state skin element.
 */
ToggleButtonElement._StyleTypes.SelectedUpSkinStyle = 			StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style SelectedUpTextColor String
 * 
 * Hex color value to be used for the button label when the button is in the "selectedUp" state. Format like "#FF0000" (red).
 * This will override TextColor when equal or higher priority than TextColor.
 */
ToggleButtonElement._StyleTypes.SelectedUpTextColor = 			StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style SelectedOverSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the button skin when the button is in the "selectedOver" state. 
 * This will override SkinClass when equal or higher priority than SkinClass.
 */
ToggleButtonElement._StyleTypes.SelectedOverSkinClass = 		StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style SelectedOverSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "selectedOver" state skin element. 
 */
ToggleButtonElement._StyleTypes.SelectedOverSkinStyle = 		StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style SelectedOverTextColor String
 * 
 * Hex color value to be used for the button label when the button is in the "selectedOver" state. Format like "#FF0000" (red).
 * This will override TextColor when equal or higher priority than TextColor.
 */
ToggleButtonElement._StyleTypes.SelectedOverTextColor = 		StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style SelectedDownSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the button skin when the button is in the "selectedDown" state. 
 * This will override SkinClass when equal or higher priority than SkinClass.
 */
ToggleButtonElement._StyleTypes.SelectedDownSkinClass = 		StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style SelectedDownSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "selectedDown" state skin element. 
 */
ToggleButtonElement._StyleTypes.SelectedDownSkinStyle = 		StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style SelectedDownTextColor String
 * 
 * Hex color value to be used for the button label when the button is in the "selectedDown" state. Format like "#FF0000" (red).
 * This will override TextColor when equal or higher priority than TextColor.
 */
ToggleButtonElement._StyleTypes.SelectedDownTextColor = 		StyleableBase.EStyleType.NORMAL;		//"#000000"

/**
 * @style SelectedDisabledSkinClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the button skin when the button is in the "selectedDisabled" state. 
 * This will override SkinClass when equal or higher priority than SkinClass.
 */
ToggleButtonElement._StyleTypes.SelectedDisabledSkinClass = 	StyleableBase.EStyleType.NORMAL;		//Element constructor()

/**
 * @style SelectedDisabledSkinStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the "selectedDisabled" state skin element. 
 */
ToggleButtonElement._StyleTypes.SelectedDisabledSkinStyle = 	StyleableBase.EStyleType.SUBSTYLE;		//StyleDefinition

/**
 * @style SelectedDisabledTextColor String
 * 
 * Hex color value to be used for the button label when the button is in the "selectedDisabled" state. Format like "#FF0000" (red).
 * This will override TextColor when equal or higher priority than TextColor.
 */
ToggleButtonElement._StyleTypes.SelectedDisabledTextColor = 	StyleableBase.EStyleType.NORMAL;		//"#000000"


////////////Default Styles/////////////////////////////

ToggleButtonElement.StyleDefault = new StyleDefinition();

//ToggleButtonElement specific styles
ToggleButtonElement.StyleDefault.setStyle("AllowDeselect", 							true);

ToggleButtonElement.StyleDefault.setStyle("SelectedUpSkinClass", 					CanvasElement);
ToggleButtonElement.StyleDefault.setStyle("SelectedOverSkinClass", 					CanvasElement);
ToggleButtonElement.StyleDefault.setStyle("SelectedDownSkinClass", 					CanvasElement);
ToggleButtonElement.StyleDefault.setStyle("SelectedDisabledSkinClass", 				CanvasElement);

ToggleButtonElement.StyleDefault.setStyle("SelectedOverTextColor", 					"#000000");
ToggleButtonElement.StyleDefault.setStyle("SelectedUpTextColor", 					"#000000");
ToggleButtonElement.StyleDefault.setStyle("SelectedDownTextColor", 					"#000000");
ToggleButtonElement.StyleDefault.setStyle("SelectedDisabledTextColor", 				"#888888");

//Skin Defaults /////////////////////////////////////
ToggleButtonElement.SelectedUpSkinStyleDefault = new StyleDefinition();

ToggleButtonElement.SelectedUpSkinStyleDefault.setStyle("BorderType", 				"solid");
ToggleButtonElement.SelectedUpSkinStyleDefault.setStyle("BorderThickness", 			1);
ToggleButtonElement.SelectedUpSkinStyleDefault.setStyle("BorderColor", 				"#333333");
ToggleButtonElement.SelectedUpSkinStyleDefault.setStyle("BackgroundFill", 			"#CCCCCC");

ToggleButtonElement.SelectedOverSkinStyleDefault = new StyleDefinition();

ToggleButtonElement.SelectedOverSkinStyleDefault.setStyle("BorderType", 			"solid");
ToggleButtonElement.SelectedOverSkinStyleDefault.setStyle("BorderThickness", 		1);
ToggleButtonElement.SelectedOverSkinStyleDefault.setStyle("BorderColor", 			"#333333");
ToggleButtonElement.SelectedOverSkinStyleDefault.setStyle("BackgroundFill", 		"#BDBDBD");

ToggleButtonElement.SelectedDownSkinStyleDefault = new StyleDefinition();

ToggleButtonElement.SelectedDownSkinStyleDefault.setStyle("BorderType", 			"solid");
ToggleButtonElement.SelectedDownSkinStyleDefault.setStyle("BorderThickness", 		1);
ToggleButtonElement.SelectedDownSkinStyleDefault.setStyle("BorderColor", 			"#333333");
ToggleButtonElement.SelectedDownSkinStyleDefault.setStyle("BackgroundFill", 		"#B0B0B0");

ToggleButtonElement.SelectedDisabledSkinStyleDefault = new StyleDefinition();

ToggleButtonElement.SelectedDisabledSkinStyleDefault.setStyle("BorderType", 		"solid");
ToggleButtonElement.SelectedDisabledSkinStyleDefault.setStyle("BorderThickness", 	1);
ToggleButtonElement.SelectedDisabledSkinStyleDefault.setStyle("BorderColor", 		"#777777");
ToggleButtonElement.SelectedDisabledSkinStyleDefault.setStyle("BackgroundFill", 	"#C7C7C7");
///////////////////////////////////////////////////////

ToggleButtonElement.StyleDefault.setStyle("SelectedUpSkinStyle", 					ToggleButtonElement.SelectedUpSkinStyleDefault);
ToggleButtonElement.StyleDefault.setStyle("SelectedOverSkinStyle", 					ToggleButtonElement.SelectedOverSkinStyleDefault);
ToggleButtonElement.StyleDefault.setStyle("SelectedDownSkinStyle", 					ToggleButtonElement.SelectedDownSkinStyleDefault);
ToggleButtonElement.StyleDefault.setStyle("SelectedDisabledSkinStyle", 				ToggleButtonElement.SelectedDisabledSkinStyleDefault);


//////////////Public Functions/////////////////////////////////////////

/**
 * @function setSelected
 * Sets the selected state of the ToggleButton.
 * 
 * @param isSelected boolean
 * When true the toggle button is selected.
 */	
ToggleButtonElement.prototype.setSelected = 
	function (isSelected)
	{
		if (this._isSelected == isSelected)
			return;
		
		this._isSelected = isSelected;
		this._updateState();
	};
	
/**
 * @function getSelected
 * Gets the selected state of the ToggleButton.
 * 
 * @returns boolean
 * When true the toggle button is selected.
 */	
ToggleButtonElement.prototype.getSelected = 
	function ()
	{
		return this._isSelected;
	};



/////////////Internal Functions/////////////////////	

//@Override
ToggleButtonElement.prototype._updateState = 
	function ()
	{
		if (this._isSelected == false)
		{
			//Call base if we're not selected, handles non-selected states.
			ToggleButtonElement.base.prototype._updateState.call(this);
		}
		else
		{
			var newState = "selectedUp";
			
			if (this.getStyle("Enabled") == false)
				newState = "selectedDisabled";
			else if (this.getStyle("AllowDeselect") == true)
			{
				if (this._mouseIsDown == true)
					newState = "selectedDown";
				else if (this._mouseIsOver == true)
					newState = "selectedOver";
			}
			
			this.setStyle("SkinState", newState);
		}
	};

//@Override	
ToggleButtonElement.prototype._onButtonClick = 
	function (elementMouseEvent)
	{
		//Not calling base
	
		//Implementor will not expect a click event when button is disabled. 
		if (this.getStyle("Enabled") == false)
			elementMouseEvent.cancelEvent();
		else
		{
			if (this._isSelected == false || this.getStyle("AllowDeselect") == true) 
			{
				//Toggle selected state.
				this._isSelected = !this._isSelected;
				
				this._updateState();
				
				//Dispatch changed event.
				if (this.hasEventListener("changed", null) == true)
					this.dispatchEvent(new ElementEvent("changed", false));
			}	
		}
	};
	
//@override
ToggleButtonElement.prototype._getSkinClass = 
	function (state)
	{
		var stateSkinClass = null;
	
		if (state == "selectedUp")
			stateSkinClass = this.getStyleData("SelectedUpSkinClass");
		else if (state == "selectedOver")
			stateSkinClass = this.getStyleData("SelectedOverSkinClass");
		else if (state == "selectedDown")
			stateSkinClass = this.getStyleData("SelectedDownSkinClass");
		else if (state == "selectedDisabled")
			stateSkinClass = this.getStyleData("SelectedDisabledSkinClass");
		else //base class state
			return ToggleButtonElement.base.prototype._getSkinClass.call(this, state);
		
		var skinClass = this.getStyleData("SkinClass");
		
		if (skinClass.comparePriority(stateSkinClass) > 0) //Use skinClass if higher priority
			return skinClass.value;
		
		return stateSkinClass.value;
	};	
	
//@override	
ToggleButtonElement.prototype._getSubStyleNameForSkinState = 
	function (state)
	{
		if (state == "selectedUp")
			return "SelectedUpSkinStyle";
		if (state == "selectedOver")
			return "SelectedOverSkinStyle";
		if (state == "selectedDown")
			return "SelectedDownSkinStyle";
		if (state == "selectedDisabled")
			return "SelectedDisabledSkinStyle";
		
		return ToggleButtonElement.base.prototype._getSubStyleNameForSkinState.call(this, state);
	};		
	
//@Override
ToggleButtonElement.prototype._getTextColor = 
	function (state)
	{
		var stateTextColor = null;
	
		if (state == "selectedUp")
			stateTextColor = this.getStyleData("SelectedUpTextColor");
		else if (state == "selectedOver")
			stateTextColor = this.getStyleData("SelectedOverTextColor");
		else if (state == "selectedDown")
			stateTextColor = this.getStyleData("SelectedDownTextColor");
		else if (state == "selectedDisabled")
			stateTextColor = this.getStyleData("SelectedDisabledTextColor");
		else //base class state
			return ToggleButtonElement.base.prototype._getTextColor.call(this, state);
	
		var textColor = this.getStyleData("TextColor");
		
		if (textColor.comparePriority(stateTextColor) > 0) //Use textColor if higher priority
			return textColor.value;
		
		return stateTextColor.value;
	};

//@Override
ToggleButtonElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ToggleButtonElement.base.prototype._doStylesUpdated.call(this, stylesMap);
	
		////Update skin classes and sub styles.
		if ("SkinClass" in stylesMap || "SelectedUpSkinClass" in stylesMap)
			this._updateSkinClass("selectedUp");
		if ("SelectedUpSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("selectedUp");
		
		if ("SkinClass" in stylesMap || "SelectedOverSkinClass" in stylesMap)
			this._updateSkinClass("selectedOver");
		if ("SelectedOverSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("selectedOver");
		
		if ("SkinClass" in stylesMap || "SelectedDownSkinClass" in stylesMap)
			this._updateSkinClass("selectedDown");
		if ("SelectedDownSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("selectedDown");
		
		if ("SkinClass" in stylesMap || "SelectedDisabledSkinClass" in stylesMap)
			this._updateSkinClass("selectedDisabled");
		if ("SelectedDisabledSkinStyle" in stylesMap)
			this._updateSkinStyleDefinitions("selectedDisabled");
		
		if ("AllowDeselect" in stylesMap)
			this._updateState();
	};	
	

	
	


/**
 * @depends ToggleButtonElement.js
 * @depends RadioButtonSkinElement.js
 * @depends EllipseShape.js
 */

///////////////////////////////////////////////////////////////////////
/////////////////////RadioButtonElement////////////////////////////////

/**
 * @class RadioButtonElement
 * @inherits ToggleButtonElement
 * 
 * RadioButton is a skinned ToggleButton that adjusts the placement of the skin and label. 
 * ToggleButtonGroup may be used to group radio buttons so only 1 may be selected at a time.
 * 
 * When a label is in use, the skin is placed next to the label rather than underneath and is assumed to be square. 
 * When a label is not in use, the skin will span the entire bounding box.
 * 
 * Being a SkinnableElement, RadioButton proxies its styles to its skins. 
 * You may assign custom skins and assign any styles you wish to apply to all skins to the RadioButton itself. 
 * 
 * See the default skin RadioButtonSkinElement for additional skin styles.
 * 
 * @seealso RadioButtonSkinElement
 * @seealso ToggleButtonGroup
 * 
 * 
 * @constructor RadioButtonElement 
 * Creates new RadioButtonElement instance.
 */
function RadioButtonElement()
{
	RadioButtonElement.base.prototype.constructor.call(this);
}

//Inherit from ToggleButtonElement
RadioButtonElement.prototype = Object.create(ToggleButtonElement.prototype);
RadioButtonElement.prototype.constructor = RadioButtonElement;
RadioButtonElement.base = ToggleButtonElement;	


/////////////Style Types///////////////////////////////

RadioButtonElement._StyleTypes = Object.create(null);

//New RadioButtonElement specific styles

/**
 * @style LabelPlacement String
 * 
 * Determines if the label should be placed to the left or right of the skin. 
 * Allowable values are "left" or "right".
 */
RadioButtonElement._StyleTypes.LabelPlacement =						StyleableBase.EStyleType.NORMAL;		// "left" || "right"

/**
 * @style LabelGap Number
 * 
 * Determines distance in pixels the label should be placed from the skin.
 */
RadioButtonElement._StyleTypes.LabelGap =							StyleableBase.EStyleType.NORMAL;		// number



////////////Default Styles//////////////////////

RadioButtonElement.StyleDefault = new StyleDefinition();

//New RadioButton styles
RadioButtonElement.StyleDefault.setStyle("LabelPlacement", 						"right");
RadioButtonElement.StyleDefault.setStyle("LabelGap", 							5);

//Override base class styles
RadioButtonElement.StyleDefault.setStyle("AllowDeselect", 						false);

RadioButtonElement.StyleDefault.setStyle("PaddingTop",                          0);
RadioButtonElement.StyleDefault.setStyle("PaddingBottom",                       0);
RadioButtonElement.StyleDefault.setStyle("PaddingLeft",                         0);
RadioButtonElement.StyleDefault.setStyle("PaddingRight",                        0);

RadioButtonElement.StyleDefault.setStyle("TextHorizontalAlign", 				"left");
RadioButtonElement.StyleDefault.setStyle("TextVerticalAlign", 					"middle");

RadioButtonElement.StyleDefault.setStyle("SkinClass", 							RadioButtonSkinElement); //Not necessary, just for completeness

RadioButtonElement.StyleDefault.setStyle("UpSkinClass", 						RadioButtonSkinElement);
RadioButtonElement.StyleDefault.setStyle("OverSkinClass", 						RadioButtonSkinElement);
RadioButtonElement.StyleDefault.setStyle("DownSkinClass", 						RadioButtonSkinElement);
RadioButtonElement.StyleDefault.setStyle("DisabledSkinClass", 					RadioButtonSkinElement);

RadioButtonElement.StyleDefault.setStyle("SelectedUpSkinClass", 				RadioButtonSkinElement);
RadioButtonElement.StyleDefault.setStyle("SelectedOverSkinClass", 				RadioButtonSkinElement);
RadioButtonElement.StyleDefault.setStyle("SelectedDownSkinClass", 				RadioButtonSkinElement);
RadioButtonElement.StyleDefault.setStyle("SelectedDisabledSkinClass", 			RadioButtonSkinElement);


//Skin Defaults
RadioButtonElement.UpSkinStyleDefault = new StyleDefinition();

RadioButtonElement.UpSkinStyleDefault.setStyle("BackgroundShape",				new EllipseShape());
RadioButtonElement.UpSkinStyleDefault.setStyle("BorderType", 					"solid");
RadioButtonElement.UpSkinStyleDefault.setStyle("BorderThickness", 				1);
RadioButtonElement.UpSkinStyleDefault.setStyle("BorderColor", 					"#333333");
RadioButtonElement.UpSkinStyleDefault.setStyle("BackgroundFill", 				"#EBEBEB");
RadioButtonElement.UpSkinStyleDefault.setStyle("CheckColor", 					"#000000");

RadioButtonElement.OverSkinStyleDefault = new StyleDefinition();

RadioButtonElement.OverSkinStyleDefault.setStyle("BackgroundShape",				new EllipseShape());
RadioButtonElement.OverSkinStyleDefault.setStyle("BorderType", 					"solid");
RadioButtonElement.OverSkinStyleDefault.setStyle("BorderThickness", 			1);
RadioButtonElement.OverSkinStyleDefault.setStyle("BorderColor", 				"#333333");
RadioButtonElement.OverSkinStyleDefault.setStyle("BackgroundFill", 				"#DDDDDD");
RadioButtonElement.OverSkinStyleDefault.setStyle("CheckColor", 					"#000000");

RadioButtonElement.DownSkinStyleDefault = new StyleDefinition();

RadioButtonElement.DownSkinStyleDefault.setStyle("BackgroundShape",				new EllipseShape());
RadioButtonElement.DownSkinStyleDefault.setStyle("BorderType", 					"solid");
RadioButtonElement.DownSkinStyleDefault.setStyle("BorderThickness", 			1);
RadioButtonElement.DownSkinStyleDefault.setStyle("BorderColor", 				"#333333");
RadioButtonElement.DownSkinStyleDefault.setStyle("BackgroundFill", 				"#CCCCCC");
RadioButtonElement.DownSkinStyleDefault.setStyle("CheckColor", 					"#000000");

RadioButtonElement.DisabledSkinStyleDefault = new StyleDefinition();

RadioButtonElement.DisabledSkinStyleDefault.setStyle("BackgroundShape",			new EllipseShape());
RadioButtonElement.DisabledSkinStyleDefault.setStyle("BorderType", 				"solid");
RadioButtonElement.DisabledSkinStyleDefault.setStyle("BorderThickness", 		1);
RadioButtonElement.DisabledSkinStyleDefault.setStyle("BorderColor", 			"#999999");
RadioButtonElement.DisabledSkinStyleDefault.setStyle("BackgroundFill", 			"#ECECEC");
RadioButtonElement.DisabledSkinStyleDefault.setStyle("CheckColor", 				"#777777");

//Apply Skin Defaults
RadioButtonElement.StyleDefault.setStyle("UpSkinStyle", 						RadioButtonElement.UpSkinStyleDefault);
RadioButtonElement.StyleDefault.setStyle("OverSkinStyle", 						RadioButtonElement.OverSkinStyleDefault);
RadioButtonElement.StyleDefault.setStyle("DownSkinStyle", 						RadioButtonElement.DownSkinStyleDefault);
RadioButtonElement.StyleDefault.setStyle("DisabledSkinStyle", 					RadioButtonElement.DisabledSkinStyleDefault);

RadioButtonElement.StyleDefault.setStyle("SelectedUpSkinStyle", 				RadioButtonElement.UpSkinStyleDefault);
RadioButtonElement.StyleDefault.setStyle("SelectedOverSkinStyle", 				RadioButtonElement.OverSkinStyleDefault);
RadioButtonElement.StyleDefault.setStyle("SelectedDownSkinStyle", 				RadioButtonElement.DownSkinStyleDefault);
RadioButtonElement.StyleDefault.setStyle("SelectedDisabledSkinStyle", 			RadioButtonElement.DisabledSkinStyleDefault);


/////////////Internal Functions/////////////////////	

//@override
RadioButtonElement.prototype._doStylesUpdated = 
	function (stylesMap)
	{
		RadioButtonElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("LabelGap" in stylesMap)
		{
			this._invalidateMeasure();
			this._invalidateLayout();
		}
		else if ("LabelPlacement" in stylesMap)
			this._invalidateLayout();
	};

//@override
RadioButtonElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var measuredWidth = 0;
		var measuredHeight = 0;
		
		if (this._labelElement != null)
		{
			var labelWidth = this._labelElement._getStyledOrMeasuredWidth();
			var labelHeight = this._labelElement._getStyledOrMeasuredHeight();
			
			measuredHeight = padHeight + labelHeight;
			measuredWidth = measuredHeight + padWidth + labelWidth + this.getStyle("LabelGap");
		}
		else
		{
		    measuredHeight = padHeight + 14;
		    measuredWidth = padWidth + 14;
		}
		
		this._setMeasuredSize(measuredWidth, measuredHeight);
	};

//@override	
RadioButtonElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		if (this._labelElement != null)
		{
			var labelPlacement = this.getStyle("LabelPlacement");
			var labelGap = this.getStyle("LabelGap");
			
			for (var prop in this._skins)
			{
				this._skins[prop]._setActualSize(this._height, this._height);
				
				if (labelPlacement == "left")
					this._skins[prop]._setActualPosition(this._width - this._height, 0);
				else
					this._skins[prop]._setActualPosition(0, 0);
			}
			
			if (labelPlacement == "left")
				this._labelElement._setActualPosition(paddingMetrics.getX(), paddingMetrics.getY());
			else
				this._labelElement._setActualPosition(this._height + labelGap + paddingMetrics.getX(), paddingMetrics.getY());
			
			this._labelElement._setActualSize(paddingMetrics.getWidth() - labelGap - this._height, paddingMetrics.getHeight());
		}
		else
		{
			for (var prop in this._skins)
			{
				this._skins[prop]._setActualSize(this._width, this._height);
				this._skins[prop]._setActualPosition(0, 0);
			}
		}
	};	




/**
 * @depends ButtonElement.js
 * @depends DropdownArrowButtonSkinElement.js
 * @depends Tween.js
 */

//////////////////////////////////////////////////////////////
////////////////DropdownBaseElement///////////////////////////

/**
 * @class DropdownBaseElement
 * @inherits ButtonElement
 * 
 * DropdownBaseElement is an abstract base class for buttons that display 
 * a pop up when clicked.
 * 
 * The DropdownBase itself contains a child button which is used to render
 * the divider line and arrow. DropdownBase proxies its SkinState style to the arrow
 * button so the arrow button will change states along with the DropdownBase itself.
 * See the default skin for the arrow button, DropdownArrowButtonSkinElement for additional styles.
 * 
 * @seealso DropdownArrowButtonSkinElement
 * 
 * 
 * @constructor DropdownBaseElement 
 * Creates new DropdownBaseElement instance.
 */
function DropdownBaseElement()
{
	DropdownBaseElement.base.prototype.constructor.call(this);

	this._arrowButton = null;
	
	this._popupElement = null;
	this._openCloseTween = null;
	this._tweenIsOpening = true;
	
	
	//////////////////
	
	var _self = this;
	
	//Private event listeners, need an instance for each DropdownBaseElement, proxy to prototype.
		
	this._onDropdownBaseManagerCaptureEventInstance = 
		function (event)
		{
			_self._onDropdownBaseManagerCaptureEvent(event);
		};
		
	this._onDropdownBaseManagerResizeEventInstance = 
		function (event)
		{
			_self._onDropdownBaseManagerResizeEvent(event);
		};
		
	this._onDropdownBaseEnterFrameInstance = 
		function (event)
		{
			_self._onDropdownBaseEnterFrame(event);
		};
}

//Inherit from ButtonElement
DropdownBaseElement.prototype = Object.create(ButtonElement.prototype);
DropdownBaseElement.prototype.constructor = DropdownBaseElement;
DropdownBaseElement.base = ButtonElement;

////////////Events///////////////////////////////

/**
 * @event opened DispatcherEvent
 * Dispatched when the pop up is opened as a result of user input.
 * 
 * @event closed DispatcherEvent
 * Dispatched when the pop up is closed as a result of user input.
 */

/////////////Style Types/////////////////////////

DropdownBaseElement._StyleTypes = Object.create(null);

/**
 * @style ArrowButtonClass CanvasElement
 * The CanvasElement or subclass constructor to be used for the arrow icon. Defaults to Button. 
 * Note that Dropdown proxies its SkinState style to the arrow button so the arrow will change states with the Dropdown.
 */
DropdownBaseElement._StyleTypes.ArrowButtonClass = 				StyleableBase.EStyleType.NORMAL; 		// CanvasElement constructor

/**
 * @style ArrowButtonStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to apply to the arrow icon class.
 */
DropdownBaseElement._StyleTypes.ArrowButtonStyle = 				StyleableBase.EStyleType.SUBSTYLE; 		// StyleDefinition

/**
 * @style OpenCloseTweenDuration Number
 * Duration in milliseconds the open and close animation should run.
 */
DropdownBaseElement._StyleTypes.OpenCloseTweenDuration = 		StyleableBase.EStyleType.NORMAL; 		// number (milliseconds)

/**
 * @style OpenCloseTweenEasingFunction Function
 * Easing function used on the open and close animations. Defaults to Tween.easeInOutSine().
 */
DropdownBaseElement._StyleTypes.OpenCloseTweenEasingFunction = 	StyleableBase.EStyleType.NORMAL; 		// function (fraction) { return fraction} - see Tween.easing


////////////Default Styles////////////////////

DropdownBaseElement.ArrowButtonSkinStyleDefault = new StyleDefinition();
DropdownBaseElement.ArrowButtonSkinStyleDefault.setStyle("BorderType", 					null);
DropdownBaseElement.ArrowButtonSkinStyleDefault.setStyle("BackgroundFill", 				null);

DropdownBaseElement.ArrowButtonDisabledSkinStyleDefault = new StyleDefinition();
DropdownBaseElement.ArrowButtonDisabledSkinStyleDefault.setStyle("BorderType", 			null);
DropdownBaseElement.ArrowButtonDisabledSkinStyleDefault.setStyle("BackgroundFill", 		null);
DropdownBaseElement.ArrowButtonDisabledSkinStyleDefault.setStyle("ArrowColor", 			"#888888");
DropdownBaseElement.ArrowButtonDisabledSkinStyleDefault.setStyle("LineColor", 			"#888888");

/////Arrow default style///////
DropdownBaseElement.ArrowButtonStyleDefault = new StyleDefinition();
DropdownBaseElement.ArrowButtonStyleDefault.setStyle("SkinClass", 					DropdownArrowButtonSkinElement);

//Note that SkinState is proxied to the arrow button, so the arrow will change state along with the Dropdown (unless you turn mouse back on)
DropdownBaseElement.ArrowButtonStyleDefault.setStyle("MouseEnabled", 				false);

DropdownBaseElement.ArrowButtonStyleDefault.setStyle("UpSkinStyle", 				DropdownBaseElement.ArrowButtonSkinStyleDefault);
DropdownBaseElement.ArrowButtonStyleDefault.setStyle("OverSkinStyle", 				DropdownBaseElement.ArrowButtonSkinStyleDefault);
DropdownBaseElement.ArrowButtonStyleDefault.setStyle("DownSkinStyle", 				DropdownBaseElement.ArrowButtonSkinStyleDefault);
DropdownBaseElement.ArrowButtonStyleDefault.setStyle("DisabledSkinStyle", 			DropdownBaseElement.ArrowButtonDisabledSkinStyleDefault);
///////////////////////////////

DropdownBaseElement.StyleDefault = new StyleDefinition();
DropdownBaseElement.StyleDefault.setStyle("PaddingTop",								3);
DropdownBaseElement.StyleDefault.setStyle("PaddingBottom",							3);
DropdownBaseElement.StyleDefault.setStyle("PaddingRight",							4);
DropdownBaseElement.StyleDefault.setStyle("PaddingLeft",							4);

DropdownBaseElement.StyleDefault.setStyle("ArrowButtonClass", 						ButtonElement); 								// Element constructor
DropdownBaseElement.StyleDefault.setStyle("ArrowButtonStyle", 						DropdownBaseElement.ArrowButtonStyleDefault); 		// StyleDefinition
DropdownBaseElement.StyleDefault.setStyle("TextHorizontalAlign", 					"left"); 								
DropdownBaseElement.StyleDefault.setStyle("OpenCloseTweenDuration", 				300); 											// number (milliseconds)
DropdownBaseElement.StyleDefault.setStyle("OpenCloseTweenEasingFunction", 			Tween.easeInOutSine); 							// function (fraction) { return fraction}


/////////Style Proxy Maps/////////////////////////////

//Proxy map for styles we want to pass to the arrow button.
DropdownBaseElement._ArrowButtonProxyMap = Object.create(null);
DropdownBaseElement._ArrowButtonProxyMap.SkinState = 						true;
DropdownBaseElement._ArrowButtonProxyMap._Arbitrary = 						true;


/////////////Public///////////////////////////////

/**
 * @function open
 * Opens the pop up.
 * 
 * @param animate boolean
 * When true animates the appearance of the pop-up.
 */	
DropdownBaseElement.prototype.open = 
	function (animate)
	{
		if (this._manager == null)
			return;
	
		//Create popup element
		if (this._popupElement == null)
		{
			this._popupElement = this._createPopup();
			
			if (this._popupElement == null)
				return;
			
			this._popupElement._owner = this;
		}
		
		//Add the pop-up list. Wait for layoutcomplete to adjust positioning and size (will set openHeight once done)
		var added = this._addPopup(); 
		
		var tweenDuration = this.getStyle("OpenCloseTweenDuration");
		
		if (animate == false || tweenDuration <= 0)
		{
			this._endOpenCloseTween();
			this._updateTweenPosition(this._getOpenedTweenValue());
		}
		else
		{
			if (this._openCloseTween != null) //Tween running
			{
				if (this._openCloseTween.startVal != this._getClosedTweenValue()) //Reverse if closing, ignore if opening.
					this._reverseTween();
			}
			else if (added == true) //Only start tween if popup did not already exist
			{
				this._openCloseTween = new Tween();
				this._openCloseTween.duration = tweenDuration;
				this._openCloseTween.startTime = Date.now();
				this._openCloseTween.easingFunction = this.getStyle("OpenCloseTweenEasingFunction");
				
				this._tweenIsOpening = true;
				
				this.addEventListener("enterframe", this._onDropdownBaseEnterFrameInstance);
			}
		}
	};
	
/**
 * @function close
 * Closes the pop up.
 * 
 * @param animate boolean
 * When true animates the disappearance of the pop-up.
 */		
DropdownBaseElement.prototype.close = 
	function (animate)
	{
		if (this._popupElement == null)
			return;
	
		var tweenDuration = this.getStyle("OpenCloseTweenDuration");
	
		if (animate == false || tweenDuration <= 0)
		{
			this._endOpenCloseTween();		
			this._removePopup();
		}
		else 
		{
			if (this._openCloseTween != null) //Tween running
			{
				if (this._openCloseTween.startVal == this._getClosedTweenValue()) //Reverse if opening, ignore if closing.
					this._reverseTween();
			}
			else if (this._popupElement._parent != null) //Dont close if already closed
			{
				this._openCloseTween = new Tween();
				this._openCloseTween.duration = tweenDuration;
				this._openCloseTween.startTime = Date.now();
				this._openCloseTween.easingFunction = this.getStyle("OpenCloseTweenEasingFunction");
				
				this._tweenIsOpening = false;
				
				this.addEventListener("enterframe", this._onDropdownBaseEnterFrameInstance);
			}
		}
	};

	
/////////////Internal///////////////////////////////	
	
/**
 * @function _createPopup
 * Stub for creating the pop up element to be added to the pop up container.
 * Override this and return a new CanvasElement instance to be used as the pop up.
 * 
 * @returns CanvasElement
 * New pop up instance.
 */	
DropdownBaseElement.prototype._createPopup = 
	function ()
	{
		return null;
	};	
	
/**
 * @function _removePopup
 * Removes the pop up container from manager and cleans up event listeners.
 * 
 * @returns bool
 * Returns true if the pop up was removed, false if the pop up does not exist.
 */	
DropdownBaseElement.prototype._removePopup = 
	function ()
	{
		if (this._popupElement._parent == null)
			return false;
	
		this._manager.removeCaptureListener("wheel", this._onDropdownBaseManagerCaptureEventInstance);
		this._manager.removeCaptureListener("mousedown", this._onDropdownBaseManagerCaptureEventInstance);
		this._manager.removeEventListener("resize", this._onDropdownBaseManagerResizeEventInstance);
		this._manager.removeElement(this._popupElement);
		
		return true;
	};

/**
 * @function _addPopup
 * Adds the pop up container to manager and registers event listeners.
 * 
 * @returns bool
 * Returns true if the pop up was added, false if the pop up already exists.
 */		
DropdownBaseElement.prototype._addPopup = 
	function ()
	{
		if (this._popupElement._parent != null)
			return false;
		
		this._manager.addElement(this._popupElement);
		this._manager.addCaptureListener("wheel", this._onDropdownBaseManagerCaptureEventInstance);
		this._manager.addCaptureListener("mousedown", this._onDropdownBaseManagerCaptureEventInstance);
		this._manager.addEventListener("resize", this._onDropdownBaseManagerResizeEventInstance);
		
		return true;
	};
	
/**
 * @function _getTweenRunning
 * Returns true if the open/close tween is running. 
 * Needed by subclasses for positioning the pop up element.
 * 
 * @returns boolean
 * True if the open/close tween is running, otherwise false.
 */		
DropdownBaseElement.prototype._getTweenRunning = 
	function ()
	{
		if (this._openCloseTween == null)
			return false;
		
		return true;
	};
	
//@private	
DropdownBaseElement.prototype._onDropdownBaseEnterFrame = 
	function (event)
	{
		//Update tween start/stops
		if (this._tweenIsOpening == true)
		{
			this._openCloseTween.startVal = this._getClosedTweenValue();
			this._openCloseTween.endVal = this._getOpenedTweenValue();
		}
		else
		{
			this._openCloseTween.startVal = this._getOpenedTweenValue();
			this._openCloseTween.endVal = this._getClosedTweenValue();
		}
	
		//Get current tween value
		var value = this._openCloseTween.getValue(Date.now());
		
		//Update popup
		this._updateTweenPosition(value);
		
		//Handle tween finished
		if (value == this._openCloseTween.endVal)
		{
			if (value == this._getClosedTweenValue())
				this.close(false);			//Finished closing
			else
				this._endOpenCloseTween();	//Finished opening
		}
	};
	
//@private
DropdownBaseElement.prototype._endOpenCloseTween = 
	function ()
	{
		if (this._openCloseTween != null)
		{
			this.removeEventListener("enterframe", this._onDropdownBaseEnterFrameInstance);
			this._openCloseTween = null;
		}
	};

/**
 * @function _getOpenedTweenValue
 * Returns value to be used for the pop up's fully open position.
 * Override this to supply values for the open/close tween start & stop.
 * 
 * @returns Number
 * Value for open/closed tween's fully opened position.
 */	
DropdownBaseElement.prototype._getOpenedTweenValue = 
	function ()
	{
		return 1;
	};	

/**
 * @function _getClosedTweenValue
 * Returns value to be used for the pop up's fully closed position.
 * Override this to supply values for the open/close tween start & stop.
 * 
 * @returns Number
 * Value for open/closed tween's fully closed position.
 */	
DropdownBaseElement.prototype._getClosedTweenValue = 
	function ()
	{
		return 0;
	};
	
/**
 * @function _updateTweenPosition
 * Stub for updating the popup when the tween value changes.
 * Override this to adjust the popup during the open/close tween.
 * 
 * @param value Number
 * Current tween value.
 */
DropdownBaseElement.prototype._updateTweenPosition = 
	function (value)
	{
		//Stub for override
	};
	
/**
 * @function _onDropdownBaseManagerCaptureEvent
 * Capture event handler for CanvasManager "wheel" and "mousedown". Used to close 
 * the pop up when events happen outside the DropdownBase or pop up elements. 
 * Only active when pop up is open.
 * 
 * @param event ElementEvent
 * ElementEvent to process.
 */	
DropdownBaseElement.prototype._onDropdownBaseManagerCaptureEvent = 
	function (event)
	{
		//Check if the popup is in this target's parent chain.
		var target = event.getTarget();
		
		while (target != null)
		{
			//Yes, leave the drop down open
			if (target == this._popupElement || (event.getType() == "mousedown" && target == this))
				return;
			
			target = target._parent;
		}
		
		//Kill the drop down, event happened outside the popup list.
		this.close(false);
		
		if (this.hasEventListener("closed", null) == true)
			this.dispatchEvent(new DispatcherEvent("closed"));
	};
	
/**
 * @function _onDropdownManagerResizeEvent
 * Capture event handler for CanvasManager "resize". Used to close the pop up.
 * Only active when pop up is open.
 * 
 * @param event DispatcherEvent
 * DispatcherEvent to process.
 */		
DropdownBaseElement.prototype._onDropdownBaseManagerResizeEvent = 
	function (event)
	{
		this.close(false);
		
		if (this.hasEventListener("closed", null) == true)
			this.dispatchEvent(new DispatcherEvent("closed"));
	};

//@override	
DropdownBaseElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		DropdownBaseElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		this.close(false);
	};	

//@private	
DropdownBaseElement.prototype._reverseTween = 
	function ()
	{
		//Reverse direction flag
		this._tweenIsOpening = !this._tweenIsOpening;
	
		//Fix tween start time
		var now = Date.now();
		var elapsed = now - this._openCloseTween.startTime;
		
		this._openCloseTween.startTime = now + elapsed - this._openCloseTween.duration;		
	};
	
//@override	
DropdownBaseElement.prototype._onButtonClick = 
	function (elementMouseEvent)
	{
		//Just cancels event if we're disabled.
		DropdownBaseElement.base.prototype._onButtonClick.call(this, elementMouseEvent);
		
		if (elementMouseEvent.getIsCanceled() == true)
			return;
		
		if (this._openCloseTween != null)
		{
			this._reverseTween();
			
			if (this._tweenIsOpening == true)
			{
				if (this.hasEventListener("opened", null) == true)
					this.dispatchEvent(new DispatcherEvent("opened"));
			}
			else
			{
				if (this.hasEventListener("closed", null) == true)
					this.dispatchEvent(new DispatcherEvent("closed"));
			}
		}
		else 
		{
			if (this._popupElement == null || this._popupElement._parent == null)
			{
				this.open(true);
				
				if (this.hasEventListener("opened", null) == true)
					this.dispatchEvent(new DispatcherEvent("opened"));
			}
			else
			{
				this.close(true);
				
				if (this.hasEventListener("closed", null) == true)
					this.dispatchEvent(new DispatcherEvent("closed"));
			}
		}
	};	
	
//@private	
DropdownBaseElement.prototype._updateArrowButton = 
	function ()
	{
		var arrowClass = this.getStyle("ArrowButtonClass");
		
		if (arrowClass == null)
		{
			if (this._arrowButton != null)
			{
				this._removeChild(this._arrowButton);
				this._arrowButton = null;
			}
		}
		else
		{
			if (this._arrowButton == null || this._arrowButton.constructor != arrowClass)
			{
				var newArrowButton = new (arrowClass)();
				newArrowButton._setStyleProxy(new StyleProxy(this, DropdownBaseElement._ArrowButtonProxyMap));
				
				if (this._arrowButton != null)
					this._removeChild(this._arrowButton);
				
				this._arrowButton = newArrowButton;
				
				this._addChild(this._arrowButton);
			}
			
			this._applySubStylesToElement("ArrowButtonStyle", this._arrowButton);
		}
	};
	
//@override
DropdownBaseElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DropdownBaseElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("ArrowButtonClass" in stylesMap || "ArrowButtonStyle" in stylesMap)
			this._updateArrowButton();
	};
	
//@override
DropdownBaseElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var textHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
		var textWidth = 20;
		
		if (this._labelElement != null)
			textWidth = this._labelElement._getStyledOrMeasuredWidth();
		
		var measuredWidth = textWidth + padWidth; 
		var measuredHeight = textHeight + padHeight;
		
		if (this._arrowButton != null)
		{
			var arrowWidth = this._arrowButton.getStyle("Width");
			var arrowHeight = this._arrowButton.getStyle("Height");
			
			if (arrowHeight != null && arrowHeight > measuredHeight)
				measuredHeight = arrowHeight;
			if (arrowWidth != null)
				measuredWidth += arrowWidth;
			else
				measuredWidth += Math.round(measuredHeight * .85);
		}

		this._setMeasuredSize(measuredWidth, measuredHeight);
	};	
	
//@override	
DropdownBaseElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		DropdownBaseElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		if (this._arrowButton != null)
		{
			var x = paddingMetrics.getX();
			var y = paddingMetrics.getY();
			var w = paddingMetrics.getWidth();
			var h = paddingMetrics.getHeight();
			
			var arrowWidth = this._arrowButton.getStyle("Width");
			var arrowHeight = this._arrowButton.getStyle("Height");
			
			if (arrowHeight == null)
				arrowHeight = this._height;
			if (arrowWidth == null)
				arrowWidth = this._height * .85;
			
			if (this._width < arrowWidth)
			{
				this._arrowButton._setActualSize(0, 0);
				this._labelElement._setActualSize(0, 0);
			}
			else
			{
				if (this._labelElement != null)
				{
					this._labelElement._setActualPosition(x, y);
					this._labelElement._setActualSize(w - arrowWidth, h);
				}
					
				this._arrowButton._setActualPosition(this._width - arrowWidth, y + (h / 2) - (arrowHeight / 2));
				this._arrowButton._setActualSize(arrowWidth, arrowHeight);
			}
		}
	};


/**
 * @depends DropdownBaseElement.js
 * @depends DataRendererLabelElement.js
 * @depends DataListElement.js
 */

//////////////////////////////////////////////////////////////
//////////////////DropdownElement/////////////////////////////

/**
 * @class DropdownElement
 * @inherits DropdownBaseElement
 * 
 * DropdownElement is a compound button that creates a pop-up drop-down list which the user
 * can select a value which is then displayed by the Dropdown. The values
 * in the list are generated by a supplied ListCollection and associated styling.
 * 
 * The Dropdown button itself contains a child button which is used to render
 * the divider line and arrow. Dropdown proxies its SkinState style to the arrow
 * button so the arrow button will change states along with the Dropdown itself.
 * See the default item renderer, DataRendererLabelElement and the default 
 * skin for the arrow button, DropdownArrowButtonSkinElement for additional styles.
 * 
 * @seealso DropdownArrowButtonSkinElement
 * 
 * 
 * @constructor DropdownElement 
 * Creates new DropdownElement instance.
 */
function DropdownElement()
{
	DropdownElement.base.prototype.constructor.call(this);

	this._listCollection = null; //Data collection
	
	this._selectedIndex = -1;
	this._selectedItem = null;
	
	this._popupContainer = null; //Same as base._popupElement;
	this._dataListPopup = null; 
	
	this._openDirection = null;
	this._openHeight = null;
	this._dropdownManagerMetrics = null;
	
	this._sampledTextWidth = null;
	

	/////////////////////
	
	var _self = this;
	
	//Private event listener, need an instance for each DropdownElement, proxy to prototype.
	this._onDropdownListCollectionChangedInstance = 
		function (collectionChangedEvent)
		{
			_self._onDropdownListCollectionChanged(collectionChangedEvent);
		};
		
	this._onDropdownDataListPopupChangedInstance = 
		function (event)
		{
			_self._onDropdownDataListPopupChanged(event);
		};
	
	this._onDropdownDataListPopupListItemClickedInstance = 
		function (event)
		{
			_self._onDropdownDataListPopupListItemClicked(event);
		};
		
	this._onDropdownDataListPopupLayoutCompleteInstance = 
		function (event)
		{
			_self._onDropdownDataListPopupLayoutComplete(event);
		};
}

//Inherit from ButtonElement
DropdownElement.prototype = Object.create(DropdownBaseElement.prototype);
DropdownElement.prototype.constructor = DropdownElement;
DropdownElement.base = DropdownBaseElement;


////////////Events///////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the drop down selection changes as a result of user input.
 * 
 * @event listitemclick ElementListItemClickEvent
 * Dispatched when a DataRenderer in the popup list is clicked. Includes associated collection item/index.
 */


/////////////Style Types/////////////////////////

DropdownElement._StyleTypes = Object.create(null);

/**
 * @style ItemLabelFunction Function
 * A function that returns a text string per a supplied collection item.
 * function (itemData) { return "" }
 */
DropdownElement._StyleTypes.ItemLabelFunction = 			StyleableBase.EStyleType.NORMAL; 		// function (itemData) { return "" }

/**
 * @style PopupDataListStyle StyleDefinition
 * The StyleDefinition or [StyleDefinition] array to apply to the pop up list element.
 */
DropdownElement._StyleTypes.PopupDataListStyle = 			StyleableBase.EStyleType.SUBSTYLE; 		// StyleDefinition

/**
 * @style MaxPopupHeight Number
 * Maximum height in pixels of the pop up list element.
 */
DropdownElement._StyleTypes.MaxPopupHeight = 				StyleableBase.EStyleType.NORMAL; 		// number

/**
 * @style PopupDataListClipTopOrBottom Number
 * 
 * Size in pixels to clip off the pop up list. Clips top when opening down, bottom when opening up. 
 * Defaults to 1 to collapse pop up list and dropdown default borders.
 */
DropdownElement._StyleTypes.PopupDataListClipTopOrBottom = 	StyleableBase.EStyleType.NORMAL; 		// number


////////////Default Styles////////////////////

//DataList Scrollbar style
DropdownElement.DataListScrollBarStyleDefault = new StyleDefinition();
DropdownElement.DataListScrollBarStyleDefault.setStyle("Padding", -1);			//Expand by 1px to share borders

DropdownElement.DataListItemUpSkinStyleDefault = new StyleDefinition();
DropdownElement.DataListItemUpSkinStyleDefault.setStyle("BackgroundFill", 		"#FFFFFF");

DropdownElement.DataListItemAltSkinStyleDefault = new StyleDefinition();
DropdownElement.DataListItemAltSkinStyleDefault.setStyle("BackgroundFill", 		"#F0F0F0");

//DataList ListItem style
DropdownElement.DataListItemStyleDefault = new StyleDefinition();
DropdownElement.DataListItemStyleDefault.setStyle("UpSkinStyle", 				DropdownElement.DataListItemUpSkinStyleDefault);
DropdownElement.DataListItemStyleDefault.setStyle("AltSkinStyle", 				DropdownElement.DataListItemAltSkinStyleDefault);

//DataList style
DropdownElement.DataListStyleDefault = new StyleDefinition();
DropdownElement.DataListStyleDefault.setStyle("ScrollBarStyle", 				DropdownElement.DataListScrollBarStyleDefault);
DropdownElement.DataListStyleDefault.setStyle("ListItemClass", 					DataRendererLabelElement);	//Same as DataList default (not needed)
DropdownElement.DataListStyleDefault.setStyle("ListItemStyle", 					DropdownElement.DataListItemStyleDefault);										
DropdownElement.DataListStyleDefault.setStyle("BorderType", 					"solid");
DropdownElement.DataListStyleDefault.setStyle("BorderThickness", 				1);
DropdownElement.DataListStyleDefault.setStyle("PaddingTop",						1);
DropdownElement.DataListStyleDefault.setStyle("PaddingBottom",					1);
DropdownElement.DataListStyleDefault.setStyle("PaddingLeft",					1);
DropdownElement.DataListStyleDefault.setStyle("PaddingRight",					1);
///////////////////////////////////

DropdownElement.StyleDefault = new StyleDefinition();

DropdownElement.StyleDefault.setStyle("PopupDataListStyle", 					DropdownElement.DataListStyleDefault); 			// StyleDefinition
DropdownElement.StyleDefault.setStyle("MaxPopupHeight", 						200); 											// number
DropdownElement.StyleDefault.setStyle("PopupDataListClipTopOrBottom", 			1); 											// number
DropdownElement.StyleDefault.setStyle("ItemLabelFunction", 						DataListElement.DefaultItemLabelFunction); 		// function (itemData) { return "" }


/////////Style Proxy Maps/////////////////////////////

//Proxy map for styles we want to pass to the DataList popup.
DropdownElement._PopupDataListProxyMap = Object.create(null);
DropdownElement._PopupDataListProxyMap.ItemLabelFunction = 				true;
DropdownElement._PopupDataListProxyMap._Arbitrary = 					true;


/////////////Public///////////////////////////////

/**
 * @function setSelectedIndex
 * Sets the selection collection index. Also updates selected item.
 * 
 * @param index int
 * Collection index to select.
 */
DropdownElement.prototype.setSelectedIndex = 
	function (index)
	{
		if (this._selectedIndex == index)
			return false;
		
		if (this._listCollection == null || index > this._listCollection.length -1)
			return false;
		
		if (index < -1)
			index = -1;
		
		if (this._dataListPopup != null)
			this._dataListPopup.setSelectedIndex(index);
		
		this._selectedIndex = index;
		this._selectedItem = this._listCollection.getItemAt(index);
		this._updateText();

		return true;
	};

/**
 * @function getSelectedIndex
 * Gets the selected collection index.
 * 
 * @returns int
 * Selected collection index or -1 if none selected.
 */	
DropdownElement.prototype.getSelectedIndex = 
	function ()
	{
		return this._selectedIndex;
	};
	
/**
 * @function setSelectedItem
 * Sets the collection item to select, also updates selected index.
 * 
 * @param item Object
 * Collection item to select.
 */	
DropdownElement.prototype.setSelectedItem = 
	function (item)
	{
		var index = this._listCollection.getItemIndex(item);
		this.setSelectedIndex(index);
	};
	
/**
 * @function getSelectedItem
 * Gets the selected collection item.
 * 
 * @returns Object
 * Selected collection item or null if none selected.
 */	
DropdownElement.prototype.getSelectedItem = 
	function ()
	{
		return this._selectedItem;
	};
	
/**
 * @function setListCollection
 * Sets the ListCollection to be used as the data-provider.
 * 
 * @param listCollection ListCollection
 * ListCollection to be used as the data-provider.
 */	
DropdownElement.prototype.setListCollection = 
	function (listCollection)
	{
		if (this._listCollection == listCollection)
			return;
	
		if (this._manager == null)
		{
			this._listCollection = listCollection;
		}
		else
		{
			if (this._listCollection != null)
				this._listCollection.removeEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance);
			
			this._listCollection = listCollection;
			
			if (this._listCollection != null)
				this._listCollection.addEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance);
		}
		
		//Fix selected index/item
		if (this._listCollection == null)
		{
			this._selectedIndex = -1;
			this._selectedItem = null;
		}
		else
		{
			if (this._selectedItem != null)
			{
				this._selectedIndex = this._listCollection.getItemIndex(this._selectedItem);
				
				if (this._selectedIndex == -1)
					this._selectedItem = null;
			}
		}
		
		this._updateText();
		this._sampledTextWidth = null;
		this._invalidateMeasure();
		
		if (this._dataListPopup != null)
			this._dataListPopup.setListCollection(listCollection);
	};	

	
/////////////Internal///////////////////////////////	

//@override
DropdownElement.prototype._createPopup = 
	function ()
	{
		this._popupContainer = new CanvasElement();
		this._popupContainer.setStyle("ClipContent", true);
		
		this._dataListPopup = new DataListElement();
		this._dataListPopup._owner = this; //Set owner - base sets its on the container
		this._popupContainer._addChild(this._dataListPopup);
		
		this._dataListPopup._setStyleProxy(new StyleProxy(this, DropdownElement._PopupDataListProxyMap));
		this._applySubStylesToElement("PopupDataListStyle", this._dataListPopup);
		
		this._dataListPopup.setListCollection(this._listCollection);
		this._dataListPopup.setSelectedIndex(this._selectedIndex);
		
		this._dataListPopup.addEventListener("changed", this._onDropdownDataListPopupChangedInstance);
		this._dataListPopup.addEventListener("listitemclick", this._onDropdownDataListPopupListItemClickedInstance);
		this._dataListPopup.addEventListener("layoutcomplete", this._onDropdownDataListPopupLayoutCompleteInstance);
		
		return this._popupContainer;
	};	
	
//@override
DropdownElement.prototype._addPopup = 
	function ()
	{
		if (DropdownElement.base.prototype._addPopup.call(this) == false)
			return false;
		
		this._openDirection = "down";
		this._openHeight = this.getStyle("MaxPopupHeight");
		this._dropdownManagerMetrics = this.getMetrics(this._manager);
		
		this._popupContainer.setStyle("Height", 0);
		this._onDropdownDataListPopupLayoutComplete(null);
		
		return true;
	};
	
//@override
DropdownElement.prototype._removePopup = 
	function ()
	{
		if (DropdownElement.base.prototype._removePopup.call(this) == false)
			return false;
		
		this._openDirection = null;
		this._openHeight = null;
		this._dropdownManagerMetrics = null;
		
		return true;
	};

//@override	
DropdownElement.prototype._getOpenedTweenValue = 
	function ()
	{
		return this._openHeight;
	};

	
//@override	
DropdownElement.prototype._updateTweenPosition = 
	function (value)
	{
		this._popupContainer.setStyle("Height", value);
		
		if (this._openDirection == "up")
		{
			this._popupContainer.setStyle("Y", this._dropdownManagerMetrics._y - value);
			this._dataListPopup._setActualPosition(0, 0);
		}
		else //if (this._openDirection == "down")
		{
			this._popupContainer.setStyle("Y", this._dropdownManagerMetrics._y + this._dropdownManagerMetrics._height);
			this._dataListPopup._setActualPosition(0, value - this._dataListPopup._height);
		}
	};

/**
 * @function _onDropdownDataListPopupLayoutComplete
 * Event handler for pop up list "layoutcomplete". 
 * Updates the pop up list height after content size is known and determines
 * if drop down opens up or down depending on available space.
 * 
 * @param event DispatcherEvent
 * DispatcherEvent to process.
 */		
DropdownElement.prototype._onDropdownDataListPopupLayoutComplete =
	function (event)
	{
		var maxHeight = this.getStyle("MaxPopupHeight");
		var height = null;
		
		if (this._dataListPopup.getStyle("ListDirection") == "horizontal")
			height = maxHeight;
		else
		{
			//Get actual Popup list height.
			var contentSize = this._dataListPopup._getContentSize();
			
			if (contentSize < maxHeight)
			{
				if (this._listCollection != null && this._dataListPopup._getNumRenderers() < this._listCollection.getLength())
					height = maxHeight;
				else
					height = contentSize;
			}
			else //contentSize >= maxHeight
				height = maxHeight;
		}
		
		//Determine open up/down and correct if not enough available space.
		var availableBottom = this._manager._height - (this._dropdownManagerMetrics._y + this._dropdownManagerMetrics._height);
		if (availableBottom >= height)
		{
			this._openDirection = "down";
			this._openHeight = height;
		}
		else //if (availableBottom < height)
		{
			var availableTop = this._dropdownManagerMetrics._y;
			if (availableTop >= height)
			{
				this._openDirection = "up";
				this._openHeight = height;
			}
			else //if (availableTop < height)
			{
				if (availableBottom >= availableTop)
				{
					this._openDirection = "down";
					this._openHeight = availableBottom;
				}
				else
				{
					this._openDirection = "up";
					this._openHeight = availableTop;
				}
			}
		}
		
		this._popupContainer.setStyle("X", this._dropdownManagerMetrics._x);
		this._popupContainer.setStyle("Width", this._dropdownManagerMetrics._width);
		this._dataListPopup._setActualSize(this._dropdownManagerMetrics._width, this._openHeight);

		this._openHeight -= this.getStyle("PopupDataListClipTopOrBottom");
		
		//Fix position if tween is not running
		if (this._getTweenRunning() == false)
			this._updateTweenPosition(this._openHeight);
	};
	
/**
 * @function _onDropdownDataListPopupChanged
 * Event handler for pop up list "changed" event. Updates selected item/index and re-dispatches "changed" event.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to process.
 */	
DropdownElement.prototype._onDropdownDataListPopupChanged = 
	function (elementEvent)
	{
		this.setSelectedIndex(this._dataListPopup.getSelectedIndex());
		
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};

/**
 * @function _onDropdownDataListPopupListItemClicked
 * Event handler for pop up list "listitemclick" event. 
 * Re-dispatches the list event and closes the Dropdown.
 * 
 * @param elementListItemClickEvent ElementListItemClickEvent
 * ElementListItemClickEvent to process.
 */		
DropdownElement.prototype._onDropdownDataListPopupListItemClicked = 
	function (elementListItemClickEvent)
	{
		//Just proxy the event from the popup list
		if (this.hasEventListener("listitemclick", null) == true)
			this.dispatchEvent(elementListItemClickEvent.clone());
		
		var shouldClose = true;
		if (this._dataListPopup.getStyle("Selectable") == false ||
			(elementListItemClickEvent.getItem().hasOwnProperty("selectable") == true &&
			elementListItemClickEvent.getItem().selectable == false))
		{
			shouldClose = false;
		}
		
		if (shouldClose == true)
		{
			this.close(true);
			
			if (this.hasEventListener("closed", null) == true)
				this.dispatchEvent(new DispatcherEvent("closed"));
		}
	};
	
//@override	
DropdownElement.prototype._updateText = 
	function ()
	{
		var text = null;
		var labelFunction = this.getStyle("ItemLabelFunction");
		
		if (this._selectedItem == null || labelFunction == null)
			text = this.getStyle("Text");
		else
			text = labelFunction(this._selectedItem);
		
		this._setLabelText(text);
	};	
	
/**
 * @function _onDropdownListCollectionChanged
 * Event handler for the ListCollection data-providers "collectionchanged" event. 
 * 
 * @param collectionChangedEvent CollectionChangedEvent
 * CollectionChangedEvent to process.
 */	
DropdownElement.prototype._onDropdownListCollectionChanged = 
	function (collectionChangedEvent)
	{
		//Room to optimize here
//		var type = collectionChangedEvent.getKind();
//		var index = collectionChangedEvent.getIndex();
	
		//Fix selected index/item 
		if (this._selectedItem != null)
		{
			this._selectedIndex = this._listCollection.getItemIndex(this._selectedItem);
			
			if (this._selectedIndex == -1)
				this._selectedItem = null;
		}
		
		this._updateText();
		this._sampledTextWidth = null;
		this._invalidateMeasure();
	};	
	
//@Override	
DropdownElement.prototype._onCanvasElementAdded = 
	function (addedRemovedEvent)
	{
		DropdownElement.base.prototype._onCanvasElementAdded.call(this, addedRemovedEvent);
	
		if (this._listCollection != null && this._listCollection.hasEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance) == false)
			this._listCollection.addEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance);
	};

//@Override	
DropdownElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		DropdownElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		if (this._listCollection != null && this._listCollection.hasEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance) == true)
			this._listCollection.removeEventListener("collectionchanged", this._onDropdownListCollectionChangedInstance);
		
		this.close(false);
	};	

//@override
DropdownElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DropdownElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("ItemLabelFunction" in stylesMap)
		{
			this._sampledTextWidth = null;
			this._invalidateMeasure();
			this._updateText();
		}
		
		if ("PopupDataListStyle" in stylesMap && this._dataListPopup != null)
			this._applySubStylesToElement("PopupDataListStyle", this._dataListPopup);
		
		if ("TextStyle" in stylesMap ||
			"TextFont" in stylesMap ||
			"TextSize" in stylesMap ||
			"TextHorizontalAlign" in stylesMap ||
			"TextVerticalAlign" in stylesMap || 
			"TextLinePaddingTop" in stylesMap ||
			"TextLinePaddingBottom" in stylesMap ||
			"Text" in stylesMap)
		{
			this._sampledTextWidth = null;
			this._invalidateMeasure();
		}
	};
	
/**
 * @function _sampleTextWidths
 * Measures text width of first 10 ListCollection items for measurement.
 * 
 * @returns Number
 * Largest text width in pixels.
 */	
DropdownElement.prototype._sampleTextWidths = 
	function ()
	{
		var labelFont = this._getFontString();
		
		var text = this.getStyle("Text");
		if (text == null)
			text = "";
		
		var measuredTextWidth = CanvasElement._measureText(text, labelFont);
		
		//Sample the first 10 items.
		var labelFunction = this.getStyle("ItemLabelFunction");
		if (this._listCollection != null && labelFunction != null)
		{
			var textWidth = 0;
			for (var i = 0; i < 10; i++)
			{
				if (i == this._listCollection.getLength())
					break;
				
				textWidth = CanvasElement._measureText(labelFunction(this._listCollection.getItemAt(i)), labelFont);
				
				if (textWidth > measuredTextWidth)
					measuredTextWidth = textWidth;
			}
		}
		
		return measuredTextWidth;
	};
	
//@override
DropdownElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		if (this._sampledTextWidth == null)
			this._sampledTextWidth = this._sampleTextWidths();
		
		var textHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
		
		var measuredWidth = this._sampledTextWidth + padWidth + 20; //Add some extra space
		var measuredHeight = textHeight + padHeight;
		
		if (this._arrowButton != null)
		{
			var arrowWidth = this._arrowButton.getStyle("Width");
			var arrowHeight = this._arrowButton.getStyle("Height");
			
			if (arrowHeight != null && arrowHeight > measuredHeight)
				measuredHeight = arrowHeight;
			if (arrowWidth != null)
				measuredWidth += arrowWidth;
			else
				measuredWidth += Math.round(measuredHeight * .85);
		}

		this._setMeasuredSize(measuredWidth, measuredHeight);
	};	
	
//@override	
DropdownElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		DropdownElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		if (this._openDirection != null) //dropdown open
		{
			//Update the dropdown metrics
			this._dropdownManagerMetrics = this.getMetrics(this._manager);
			
			//Update the widths of the popup container and list. (Heights handled by tween / list layoutcomplete)
			//This is here so that when the Dropdown is using measured width, and the collection changes,
			//it may change the width of the dropdown button, so we need to make sure we keep the widths in sync.
			this._popupContainer.setStyle("Width", this._dropdownManagerMetrics._width);
			this._popupContainer.setStyle("X", this._dropdownManagerMetrics._x);
			
			this._dataListPopup._setActualSize(this._dropdownManagerMetrics._width, this._dataListPopup._height);
		}
	};
	
	


/**
 * @depends DropdownBaseElement.js
 */

//////////////////////////////////////////////////////////////
///////////////DatePickerButtonElement///////////////////////

/**
 * @class DatePickerButtonElement
 * @inherits DropdownBaseElement
 * 
 * DatePickerButtonElement is a compound button that creates a pop-up DatePicker
 * where the user can select a date which is then displayed on the button. 
 * 
 * The DatePickerButtonElement button itself contains a child button which is used to render
 * the divider line and arrow. DatePickerButtonElement proxies its SkinState style to the arrow
 * button so the arrow button will change states along with the DatePickerButtonElement itself.
 * See the default skin for the arrow button DropdownArrowButtonSkinElement for additional styles.
 * 
 * @seealso DropdownArrowButtonSkinElement
 * 
 * 
 * @constructor DatePickerButtonElement 
 * Creates new DatePickerButtonElement instance.
 */
function DatePickerButtonElement()
{
	DatePickerButtonElement.base.prototype.constructor.call(this);

	this._datePickerPopup = null;
	this._selectedDate = null;

	
	////////////////
	
	var _self = this;
	
	//Private event listeners, need an instance for each DatePickerButton, proxy to prototype.
		
	this._onDatePickerChangedInstance = 
		function (event)
		{
			_self._onDatePickerChanged(event);
		};
	this._onDatePickerLayoutCompleteInstance = 
		function (event)
		{
			_self._onDatePickerLayoutComplete(event);
		};	
}

//Inherit from ButtonElement
DatePickerButtonElement.prototype = Object.create(DropdownBaseElement.prototype);
DatePickerButtonElement.prototype.constructor = DatePickerButtonElement;
DatePickerButtonElement.base = DropdownBaseElement;


////////////Static///////////////////////////////

DatePickerButtonElement.DefaultDateFormatLabelFunction = 
	function (date)
	{
		var year = date.getFullYear().toString();
		var month = (date.getMonth() + 1).toString();
		var day = date.getDate().toString();
		
		while (month.length < 2)
			month = "0" + month;
		
		while (day.length < 2)
			day = "0" + day;
		
		return year + "-" + month + "-" + day;
	};

	
////////////Events///////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the date selection changes as a result of user input.
 */


/////////////Style Types/////////////////////////

DatePickerButtonElement._StyleTypes = Object.create(null);

/**
 * @style AllowDeselect boolean
 * 
 * When false, the DatePicker day ToggleButtons cannot be de-selected by the user and the "selectedOver" and "selectedDown" states are not used.
 */
DatePickerButtonElement._StyleTypes.AllowDeselect = 					StyleableBase.EStyleType.NORMAL;		// true || false

/**
 * @style DateFormatLabelFunction Function
 * 
 * A function that accepts a date and returns a string to be displayed as the date label.
 * Signature: function (date) { return "" }
 * The default label function returns returns international date format "YYYY-MM-DD".
 */
DatePickerButtonElement._StyleTypes.DateFormatLabelFunction = 			StyleableBase.EStyleType.NORMAL; 

/**
 * @style PopupDatePickerStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the pop up DatePicker element.
 */
DatePickerButtonElement._StyleTypes.PopupDatePickerStyle = 				StyleableBase.EStyleType.SUBSTYLE; 		// StyleDefinition

/**
 * @style PopupDatePickerDistance Number
 * 
 * Vertical distance in pixels to place the DatePicker pop up from the button.
 * Defaults to -1 to collapse default 1 pixel borders.
 */
DatePickerButtonElement._StyleTypes.PopupDatePickerDistance = 			StyleableBase.EStyleType.NORMAL; 		


////////////Default Styles////////////////////

////DatePickerButton default style/////
DatePickerButtonElement.StyleDefault = new StyleDefinition();
DatePickerButtonElement.StyleDefault.setStyle("AllowDeselect",							false);
DatePickerButtonElement.StyleDefault.setStyle("DateFormatLabelFunction",				DatePickerButtonElement.DefaultDateFormatLabelFunction)		
DatePickerButtonElement.StyleDefault.setStyle("PopupDatePickerStyle", 					null);
DatePickerButtonElement.StyleDefault.setStyle("PopupDatePickerDistance", 				-1);			


/////////////Public///////////////////////////////

/**
 * @function setSelectedDate
 * Sets the selected date of the DatePickerButton.
 * 
 * @param date Date
 * Date to set as the selected date or null for no selection.
 */		
DatePickerButtonElement.prototype.setSelectedDate = 
	function (date)
	{
		if (date instanceof Date == false)
			return;
	
		this._selectedDate = new Date(date.getTime());
	
		if (this._datePickerPopup != null)
			this._datePickerPopup.setSelectedDate(this._selectedDate);
		
		this._updateText();
	};
	
/**
 * @function getSelectedDate
 * Gets the selected date of the DatePickerButton.
 * 
 * @returns Date
 * Currently selected date or null if none selected.
 */	
DatePickerButtonElement.prototype.getSelectedDate = 
	function ()
	{
		return new Date(this._selectedDate.getTime());
	};

	
/////////////Internal///////////////////////////////	
	
//@override
DatePickerButtonElement.prototype._createPopup = 
	function ()
	{
		this._datePickerPopup = new DatePickerElement();
		
		this._datePickerPopup.addEventListener("changed", this._onDatePickerChangedInstance);	
		this._datePickerPopup.addEventListener("layoutcomplete", this._onDatePickerLayoutCompleteInstance);
		
		this._applySubStylesToElement("PopupDatePickerStyle", this._datePickerPopup);
		this._datePickerPopup.setSelectedDate(this._selectedDate);
		this._datePickerPopup.setStyle("AllowDeselect", this.getStyle("AllowDeselect"));
		
		return this._datePickerPopup;
	};	

//@override	
DatePickerButtonElement.prototype._updateTweenPosition = 
	function (value)
	{
		this._datePickerPopup.setStyle("Alpha", value);
	};	
	
/**
 * @function _onDatePickerLayoutComplete
 * Event handler for pop up DatePicker "layoutcomplete". 
 * Updates the pop up size when content size is known and determines
 * position of the pop up depending on available space.
 * 
 * @param event DispatcherEvent
 * DispatcherEvent to process.
 */		
DatePickerButtonElement.prototype._onDatePickerLayoutComplete =
	function (event)
	{
		this._layoutDatePickerPopup();
	};
	
/**
 * @function _onDatePickerChangedInstance
 * Event handler for pop up DatePicker "changed" event. 
 * Updates date label and re-dispatches "changed" event.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to process.
 */	
DatePickerButtonElement.prototype._onDatePickerChanged = 
	function (elementEvent)
	{
		this.setSelectedDate(this._datePickerPopup.getSelectedDate());
		
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	
		this.close(true);
		
		if (this.hasEventListener("closed", null) == true)
			this.dispatchEvent(new ElementEvent("closed", false));
	};

/**
 * @function _updateText
 * Updates the date label text via the styled DateFormatLabelFunction
 */
DatePickerButtonElement.prototype._updateText = 
	function ()
	{
		var labelFunction = this.getStyle("DateFormatLabelFunction");
		
		if (this._selectedDate != null && labelFunction != null)
			text = labelFunction(this._selectedDate);
		else
			text = this.getStyle("Text");
		
		this._setLabelText(text);
	};

//@override
DatePickerButtonElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DatePickerButtonElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("PopupDatePickerStyle" in stylesMap && this._datePickerPopup != null)
		{
			this._applySubStylesToElement("PopupDatePickerStyle", this._datePickerPopup);
			this._invalidateLayout();
		}
		
		if ("AllowDeselect" in stylesMap && this._datePickerPopup != null)
			this._datePickerPopup.setStyle("AllowDeselect", this.getStyle("AllowDeselect"));
		
		if ("PopupDatePickerDistance" in stylesMap)
			this._invalidateLayout();
		
		if ("DateFormatLabelFunction" in stylesMap)
			this._updateText();
		
		
			
	};
	
//@override
DatePickerButtonElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		var fontString = this._getFontString();
			
		var dateTextWidth = 20;
		var labelFunction = this.getStyle("DateFormatLabelFunction");
		if (labelFunction != null)
			dateTextWidth = CanvasElement._measureText(labelFunction(new Date()), fontString);
		
		var textLabelWidth = 20;
		var textLabel = this.getStyle("Text");
		if (textLabel != null)
			textLabelWidth = CanvasElement._measureText(textLabel, fontString);
		
		var textWidth = Math.max(dateTextWidth, textLabelWidth);		
		var textHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
		
		var arrowWidth = null;
		var arrowHeight = null;
		
		if (this._arrowButton != null)
		{
			arrowWidth = this._arrowButton.getStyle("Width");
			arrowHeight = this._arrowButton.getStyle("Height");
		}
		
		if (arrowHeight == null)
			arrowHeight = textHeight + padHeight;
		if (arrowWidth == null)
			arrowWidth = Math.round(arrowHeight * .85); 
		
		var h = Math.ceil(Math.max(arrowHeight, textHeight + padHeight));
		var w = Math.ceil(padWidth + textWidth + arrowWidth);
		
		this._setMeasuredSize(w, h);
	};	
	
/**
 * @function _layoutDatePickerPopup
 * Sizes and positions the DatePicker pop up.
 */	
DatePickerButtonElement.prototype._layoutDatePickerPopup = 
	function ()
	{
		//DatePicker not displayed - bail.
		if (this._datePickerPopup == null ||
			this._datePickerPopup._parent == null || 
			this._datePickerPopup._layoutInvalid == true)
		{
			return;
		}
	
		var managerMetrics = this.getMetrics(this._manager);
		
		var pickerDistance = this.getStyle("PopupDatePickerDistance");
		
		var pickerWidth = this._datePickerPopup.getStyle("Width");
		if (pickerWidth == null)
			pickerWidth = this._datePickerPopup._measuredWidth;
		
		var pickerHeight = this._datePickerPopup.getStyle("Height");
		if (pickerHeight == null)
			pickerHeight = this._datePickerPopup._measuredHeight;
		
		//Figure out the available space around the button that we have to place the pop up
		var availableBottom = this._manager._height - (managerMetrics._y + managerMetrics._height) - pickerDistance;
		var availableTop = managerMetrics._y - pickerDistance;
		var availableRight = this._manager._width - managerMetrics._x;
		var availableLeft = managerMetrics._x + managerMetrics._width;
		
		var pickerX = 0;
		var pickerY = 0;
		
		//Open bottom
		if (availableBottom > pickerHeight || pickerHeight > availableTop)
			pickerY = managerMetrics._y + managerMetrics._height + pickerDistance;
		else //Open top
			pickerY = managerMetrics._y - pickerHeight - pickerDistance;

		//Left aligned
		if (availableRight > pickerWidth || pickerWidth < availableRight)
			pickerX = managerMetrics._x;
		else //Right aligned
			pickerX = managerMetrics._x + managerMetrics._width - pickerWidth;
		
		this._datePickerPopup.setStyle("X", pickerX);
		this._datePickerPopup.setStyle("Y", pickerY);
		this._datePickerPopup.setStyle("Width", pickerWidth);
		this._datePickerPopup.setStyle("Height", pickerHeight);
	};
	
//@override	
DatePickerButtonElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		DatePickerButtonElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		this._layoutDatePickerPopup();
		
		var x = paddingMetrics.getX();
		var y = paddingMetrics.getY();
		var w = paddingMetrics.getWidth();
		var h = paddingMetrics.getHeight();
		
		var arrowWidth = 0;
		var arrowHeight = 0;
		
		if (this._arrowButton != null)
		{
			var x = paddingMetrics.getX();
			var y = paddingMetrics.getY();
			var w = paddingMetrics.getWidth();
			var h = paddingMetrics.getHeight();
			
			var iconWidth = this._arrowButton.getStyle("Width");
			var iconHeight = this._arrowButton.getStyle("Height");
			
			if (iconHeight == null)
				iconHeight = this._height;
			if (iconWidth == null)
				iconWidth = this._height * .85;
			
			if (this._width < iconWidth)
			{
				this._arrowButton._setActualSize(0, 0);
				this._labelElement._setActualSize(0, 0);
			}
			else
			{
				if (this._labelElement != null)
				{
					this._labelElement._setActualPosition(x, y);
					this._labelElement._setActualSize(w - iconWidth, h);
				}
					
				this._arrowButton._setActualPosition(this._width - iconWidth, y + (h / 2) - (iconHeight / 2));
				this._arrowButton._setActualSize(iconWidth, iconHeight);
			}
		}
	};
	
	


/**
 * @depends ButtonElement.js
 */

//////////////////////////////////////////////////////////////////
/////////////DataGridHeaderItemRenderer////////////////////////////

/**
 * @class DataGridHeaderItemRenderer
 * @inherits ButtonElement
 * 
 * Default DataGrid HeaderItem renderer based on Button. 
 * Adds sort icons.
 * 
 * @constructor DataGridHeaderItemRenderer 
 * Creates new DataGridHeaderItemRenderer instance.
 */
function DataGridHeaderItemRenderer()
{
	DataGridHeaderItemRenderer.base.prototype.constructor.call(this);
	
	this._sortAscIcon = null;
	this._sortDescIcon = null;
	this._currentSortDirection = null;
}

//Inherit from ButtonElement
DataGridHeaderItemRenderer.prototype = Object.create(ButtonElement.prototype);
DataGridHeaderItemRenderer.prototype.constructor = DataGridHeaderItemRenderer;
DataGridHeaderItemRenderer.base = ButtonElement;

//////Style Types//////////////////////
DataGridHeaderItemRenderer._StyleTypes = Object.create(null);

/**
 * @style SortAscIconClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the ascending sort icon. 
 * Defaults to Button. HeaderItemRenderer proxies its SkinState style to the sort icons so
 * the sort icons will change state along with the HeaderItemRenderer.
 */
DataGridHeaderItemRenderer._StyleTypes.SortAscIconClass =			StyleableBase.EStyleType.NORMAL;		// CanvasElement constructor

/**
 * @style SortAscIconStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply ascending sort icon element.
 */
DataGridHeaderItemRenderer._StyleTypes.SortAscIconStyle =			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style SortDescIconClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the descending sort icon. 
 * Defaults to Button. HeaderItemRenderer proxies its SkinState style to the sort icons so
 * the sort icons will change state along with the HeaderItemRenderer.
 */
DataGridHeaderItemRenderer._StyleTypes.SortDescIconClass =			StyleableBase.EStyleType.NORMAL;		// CanvasElement constructor

/**
 * @style SortDescIconStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply descending sort icon element.
 */
DataGridHeaderItemRenderer._StyleTypes.SortDescIconStyle =			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style SortIconGap Number
 * 
 * Distance in pixels between the sort icon and the header label.
 */
DataGridHeaderItemRenderer._StyleTypes.SortIconGap =					StyleableBase.EStyleType.NORMAL;		// number

/**
 * @style SortIconPlacement String
 * 
 * Determines placement of the sort icon. Allowable values are "left" or "right".
 */
DataGridHeaderItemRenderer._StyleTypes.SortIconPlacement =				StyleableBase.EStyleType.NORMAL;		// "left" || "right"


/////////Default Styles///////////////

//Make disabled skin look like "up" skin (just not click-able)
DataGridHeaderItemRenderer.DisabledSkinStyleDefault = new StyleDefinition();
DataGridHeaderItemRenderer.DisabledSkinStyleDefault.setStyle("BackgroundFill", 			"#EBEBEB");
DataGridHeaderItemRenderer.DisabledSkinStyleDefault.setStyle("BorderType", 				null);

//Other up/over/down skins (kill border)
DataGridHeaderItemRenderer.SkinStyleDefault = new StyleDefinition();
DataGridHeaderItemRenderer.SkinStyleDefault.setStyle("BorderType", 						null);


DataGridHeaderItemRenderer.StyleDefault = new StyleDefinition();
DataGridHeaderItemRenderer.StyleDefault.setStyle("UpSkinStyle", 						DataGridHeaderItemRenderer.SkinStyleDefault);
DataGridHeaderItemRenderer.StyleDefault.setStyle("OverSkinStyle", 						DataGridHeaderItemRenderer.SkinStyleDefault);
DataGridHeaderItemRenderer.StyleDefault.setStyle("DownSkinStyle", 						DataGridHeaderItemRenderer.SkinStyleDefault);
DataGridHeaderItemRenderer.StyleDefault.setStyle("DisabledSkinStyle", 					DataGridHeaderItemRenderer.DisabledSkinStyleDefault);
DataGridHeaderItemRenderer.StyleDefault.setStyle("DisabledTextColor", 					"#000000");

DataGridHeaderItemRenderer.StyleDefault.setStyle("TextHorizontalAlign", 				"left");
DataGridHeaderItemRenderer.StyleDefault.setStyle("PaddingTop",							4);
DataGridHeaderItemRenderer.StyleDefault.setStyle("PaddingBottom",						4);
DataGridHeaderItemRenderer.StyleDefault.setStyle("PaddingLeft",							8);
DataGridHeaderItemRenderer.StyleDefault.setStyle("PaddingRight",						8);

/////Sort Icon default styles //////

//Ascending Sort Icon
DataGridHeaderItemRenderer.SortAscIconSkinBgShapeDefault = new ArrowShape();
DataGridHeaderItemRenderer.SortAscIconSkinBgShapeDefault.setStyle("Direction", 				"up");

DataGridHeaderItemRenderer.SortAscIconSkinStyleDefault = new StyleDefinition();
DataGridHeaderItemRenderer.SortAscIconSkinStyleDefault.setStyle("BorderType", 				null);
DataGridHeaderItemRenderer.SortAscIconSkinStyleDefault.setStyle("BackgroundFill", 			"#444444");
DataGridHeaderItemRenderer.SortAscIconSkinStyleDefault.setStyle("BackgroundShape", 			DataGridHeaderItemRenderer.SortAscIconSkinBgShapeDefault);

DataGridHeaderItemRenderer.SortAscIconStyleDefault = new StyleDefinition();
DataGridHeaderItemRenderer.SortAscIconStyleDefault.setStyle("UpSkinStyle", 					DataGridHeaderItemRenderer.SortAscIconSkinStyleDefault);
DataGridHeaderItemRenderer.SortAscIconStyleDefault.setStyle("OverSkinStyle", 				DataGridHeaderItemRenderer.SortAscIconSkinStyleDefault);
DataGridHeaderItemRenderer.SortAscIconStyleDefault.setStyle("DownSkinStyle", 				DataGridHeaderItemRenderer.SortAscIconSkinStyleDefault);
//Note that SkinState is proxied to the sort icons, so the sort icons will change state along with the HeaderRenderer (unless you turn mouse back on)
DataGridHeaderItemRenderer.SortAscIconStyleDefault.setStyle("MouseEnabled", 				false);

//Descending Sort Icon
DataGridHeaderItemRenderer.SortDescIconSkinBgShapeDefault = new ArrowShape();
DataGridHeaderItemRenderer.SortDescIconSkinBgShapeDefault.setStyle("Direction", 			"down");

DataGridHeaderItemRenderer.SortDescIconSkinStyleDefault = new StyleDefinition();
DataGridHeaderItemRenderer.SortDescIconSkinStyleDefault.setStyle("BorderType", 				null);
DataGridHeaderItemRenderer.SortDescIconSkinStyleDefault.setStyle("BackgroundFill", 			"#444444");
DataGridHeaderItemRenderer.SortDescIconSkinStyleDefault.setStyle("BackgroundShape", 		DataGridHeaderItemRenderer.SortDescIconSkinBgShapeDefault);

DataGridHeaderItemRenderer.SortDescIconStyleDefault = new StyleDefinition();
DataGridHeaderItemRenderer.SortDescIconStyleDefault.setStyle("UpSkinStyle", 				DataGridHeaderItemRenderer.SortDescIconSkinStyleDefault);
DataGridHeaderItemRenderer.SortDescIconStyleDefault.setStyle("OverSkinStyle", 				DataGridHeaderItemRenderer.SortDescIconSkinStyleDefault);
DataGridHeaderItemRenderer.SortDescIconStyleDefault.setStyle("DownSkinStyle", 				DataGridHeaderItemRenderer.SortDescIconSkinStyleDefault);

//Note that SkinState is proxied to the sort icons, so the sort icons will change state along with the HeaderRenderer (unless you turn mouse back on)
DataGridHeaderItemRenderer.SortDescIconStyleDefault.setStyle("MouseEnabled", 				false);

///////////////////////////////////

DataGridHeaderItemRenderer.StyleDefault.setStyle("SortAscIconClass",						ButtonElement);											// CanvasElement constructor
DataGridHeaderItemRenderer.StyleDefault.setStyle("SortAscIconStyle",						DataGridHeaderItemRenderer.SortAscIconStyleDefault);	// StyleDefinition
DataGridHeaderItemRenderer.StyleDefault.setStyle("SortDescIconClass",						ButtonElement);											// CanvasElement constructor
DataGridHeaderItemRenderer.StyleDefault.setStyle("SortDescIconStyle",						DataGridHeaderItemRenderer.SortDescIconStyleDefault);	// StyleDefinition

DataGridHeaderItemRenderer.StyleDefault.setStyle("SortIconGap",								3);			// number
DataGridHeaderItemRenderer.StyleDefault.setStyle("SortIconPlacement",						"right");	// "left" || "right"



//////Proxy////////////////////////////
DataGridHeaderItemRenderer._SortIconProxyMap = Object.create(null);

//Proxy the SkinState this way we can trigger icon skin changes when our skin changes without
//impacting the functionality of a custom icon. We'll also disable mouse on the default style.
DataGridHeaderItemRenderer._SortIconProxyMap.SkinState = 		true;


///////////Internal/////////////////////

/**
 * @function _createSortIcon
 * Generates a sort icon based on the IconClass styles.
 * 
 * @param isDecending boolean
 * True if the icon should be a descending icon, false otherwise.
 * 
 * @returns CanvasElement
 * The resulting sort icon element instance.
 */
DataGridHeaderItemRenderer.prototype._createSortIcon = 
	function (isDecending)
	{
		var iconClass = null;
		
		if (isDecending == true)
			iconClass = this.getStyle("SortDescIconClass");
		else
			iconClass = this.getStyle("SortAscIconClass");
		
		var newIcon = new (iconClass)();
		newIcon._setStyleProxy(new StyleProxy(this,DataGridHeaderItemRenderer._SortIconProxyMap));
		
		if (isDecending == true)
			this._applySubStylesToElement("SortDescIconStyle", newIcon);
		else
			this._applySubStylesToElement("SortAscIconStyle", newIcon);
		
		return newIcon;
	};

/**
 * @function _updateSortIcons
 * Updates sort icons in response to style and list data changes.
 */
DataGridHeaderItemRenderer.prototype._updateSortIcons = 
	function ()
	{
		if (this._listData == null)
			return;
	
		var listData = this._listData;
		var dataCollection = listData._parentGrid._listCollection;
		
		var sortDirection = null;
		
		if (dataCollection != null && 
			dataCollection._collectionSort != null && 
			dataCollection._collectionSort == listData._parentGrid._gridColumns[listData._columnIndex].getStyle("CollectionSort"))
		{
			if (dataCollection._collectionSort._isDecending == true)
				sortDirection = "down";
			else
				sortDirection = "up";
		}

		if (this._currentSortDirection != sortDirection)
		{
			this._currentSortDirection = sortDirection;
			this._invalidateLayout();
		}
		
		if (sortDirection == null)
		{
			if (this._sortAscIcon != null)
				this._sortAscIcon.setStyle("Visible", false);
			if (this._sortDescIcon != null)
				this._sortDescIcon.setStyle("Visible", false);
		}
		else if (sortDirection == "up")
		{
			var upIconClass = this.getStyle("SortAscIconClass");
			
			if (upIconClass == null)
			{
				if (this._sortAscIcon != null)
				{
					this._removeChild(this._sortAscIcon);
					this._sortAscIcon = null;
				}
			}
			else
			{
				if (this._sortAscIcon == null)
				{
					this._sortAscIcon = this._createSortIcon(false);
					this._addChild(this._sortAscIcon);
				}
				else if (this._sortAscIcon.constructor != upIconClass)
				{ //Icon Class changed
					this._removeChild(this._sortAscIcon);
					this._sortAscIcon = this._createSortIcon(false);
					this._addChild(this._sortAscIcon);
				}
				else
					this._applySubStylesToElement("SortAscIconStyle", this._sortAscIcon);
				
				if (this._sortDescIcon != null)
					this._sortDescIcon.setStyle("Visible", false);
				
				this._sortAscIcon.setStyle("Visible", true);
			}
		}
		else if (sortDirection == "down")
		{
			var downIconClass = this.getStyle("SortDescIconClass");
			
			if (downIconClass == null)
			{
				if (this._sortDescIcon != null)
				{
					this._removeChild(this._sortDescIcon);
					this._sortDescIcon = null;
				}
			}
			else
			{
				if (this._sortDescIcon == null)
				{
					this._sortDescIcon = this._createSortIcon(true);
					this._addChild(this._sortDescIcon);
				}
				else if (this._sortDescIcon.constructor != downIconClass)
				{ //Icon Class changed
					this._removeChild(this._sortDescIcon);
					this._sortDescIcon = this._createSortIcon(true);
					this._addChild(this._sortDescIcon);
				}
				else
					this._applySubStylesToElement("SortDescIconStyle", this._sortDescIcon);
				
				if (this._sortAscIcon != null)
					this._sortAscIcon.setStyle("Visible", false);
				
				this._sortDescIcon.setStyle("Visible", true);
			}
		}
	};

//@Override
DataGridHeaderItemRenderer.prototype._setListData = 
	function (listData, itemData)
	{
		DataGridHeaderItemRenderer.base.prototype._setListData.call(this, listData, itemData);
		
		if (listData == null)
			return;
		
		this.setStyle("Text", listData._parentGrid._gridColumns[listData._columnIndex].getStyle("HeaderText"));
		this._updateSortIcons();
	};

//@Override
DataGridHeaderItemRenderer.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DataGridHeaderItemRenderer.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("SortAscIconClass" in stylesMap ||
			"SortAscIconStyle" in stylesMap ||
			"SortDescIconClass" in stylesMap ||
			"SortDescIconStyle" in stylesMap)
		{
			this._updateSortIcons();
		}
		
		if ("SortIconGap" in stylesMap || "SortIconPlacement" in stylesMap)
			this._invalidateLayout();
	};	
	
//@Override	
DataGridHeaderItemRenderer.prototype._doLayout = 
	function (paddingMetrics)
	{
		DataGridHeaderItemRenderer.base.prototype._doLayout.call(this, paddingMetrics);
	
		//Fix label position to leave room for sort indicator.
		if (this._labelElement != null && this._currentSortDirection != null)
		{
			var sortIcon = null;
			if (this._currentSortDirection == "up")
				sortIcon = this._sortAscIcon;
			else
				sortIcon = this._sortDescIcon;
			
			if (sortIcon != null)
			{
				var x = paddingMetrics.getX();
				var y = paddingMetrics.getY();
				var w = paddingMetrics.getWidth();
				var h = paddingMetrics.getHeight();
				
				var iconWidth = sortIcon.getStyle("Width");
				var iconHeight = sortIcon.getStyle("Height");
				
				if (iconHeight == null)
					iconHeight = paddingMetrics.getHeight() * .35;
				if (iconWidth == null)
					iconWidth = iconHeight * 1.5;
				
				if (w < iconWidth)
				{
					sortIcon._setActualSize(0, 0);
					this._labelElement._setActualSize(0, 0);
				}
				else
				{
					var gap = this.getStyle("SortIconGap");
					var iconPlacement = this.getStyle("SortIconPlacement");
					
					if (iconPlacement == "left")
					{
						this._labelElement._setActualPosition(x + iconWidth + gap, y);
						this._labelElement._setActualSize(w - iconWidth - gap, h);
						
						sortIcon._setActualPosition(x, y + (h / 2) - (iconHeight / 2));
						sortIcon._setActualSize(iconWidth, iconHeight);
					}
					else // "right"
					{
						this._labelElement._setActualPosition(x, y);
						this._labelElement._setActualSize(w - iconWidth - gap, h);
						
						sortIcon._setActualPosition(x + w - iconWidth, y + (h / 2) - (iconHeight / 2));
						sortIcon._setActualSize(iconWidth, iconHeight);
					}
				}
			}
		}
	};	


/**
 * @depends DataGridDataRenderer.js
 * @depends DataGridHeaderColumnDividerSkinElement.js
 * @depends ButtonElement.js
 */

//////////////////////////////////////////////////////////////////
/////////////DataGridHeaderElement////////////////////////////////

/**
 * @class DataGridHeaderElement
 * @inherits DataGridDataRenderer
 * 
 * Default DataGrid header element. 
 * Renders header items per parent grid's columns. 
 * Adds drag-able column dividers and updates parent grid's column widths.
 * 
 * 
 * @constructor DataGridHeaderElement 
 * Creates new DataGridHeaderElement instance.
 */
function DataGridHeaderElement()
{
	DataGridHeaderElement.base.prototype.constructor.call(this);
	
	var _self = this;
	
	this._onColumnDividerDragInstance = 
		function (elementEvent)
		{
			_self._onColumnDividerDrag(elementEvent);
		};
}
	
//Inherit from DataGridDataRenderer
DataGridHeaderElement.prototype = Object.create(DataGridDataRenderer.prototype);
DataGridHeaderElement.prototype.constructor = DataGridHeaderElement;
DataGridHeaderElement.base = DataGridDataRenderer;	

/////////////Style Types/////////////////////////

DataGridHeaderElement._StyleTypes = Object.create(null);

/**
 * @style ColumnDividerClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the draggable column divider (defaults to Button). 
 */
DataGridHeaderElement._StyleTypes.ColumnDividerClass = 		StyleableBase.EStyleType.NORMAL; 	// Element constructor()

/**
 * @style ColumnDividerStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the column divider element.
 * See default skin class is DataGridHeaderColumnDividerSkinElement for additional styles.
 * 
 * @seealso DataGridHeaderColumnDividerSkinElement
 */
DataGridHeaderElement._StyleTypes.ColumnDividerStyle = 		StyleableBase.EStyleType.SUBSTYLE; 	// StyleDefinition

/**
 * @style DraggableColumns boolean
 * 
 * When true, column dividers are draggable.
 */
DataGridHeaderElement._StyleTypes.DraggableColumns = 		StyleableBase.EStyleType.NORMAL; 	// StyleDefinition


////////////Default Styles////////////////////

DataGridHeaderElement.StyleDefault = new StyleDefinition();
DataGridHeaderElement.StyleDefault.setStyle("PaddingBottom", 				1);
DataGridHeaderElement.StyleDefault.setStyle("BorderType", 					"solid");
DataGridHeaderElement.StyleDefault.setStyle("BorderThickness", 				1);
DataGridHeaderElement.StyleDefault.setStyle("BorderColor", 					"#000000");

//Column Divider button style
DataGridHeaderElement.ColumnDividerSkinStyleDefault = new StyleDefinition();
DataGridHeaderElement.ColumnDividerSkinStyleDefault.setStyle("DividerLineColor", 		"#777777");
DataGridHeaderElement.ColumnDividerSkinStyleDefault.setStyle("DividerArrowColor", 		"#444444");
DataGridHeaderElement.ColumnDividerSkinStyleDefault.setStyle("BorderType", 				null);
DataGridHeaderElement.ColumnDividerSkinStyleDefault.setStyle("BackgroundFill", 			null);

DataGridHeaderElement.ColumnDividerStyleDefault = new StyleDefinition();
DataGridHeaderElement.ColumnDividerStyleDefault.setStyle("SkinClass", 				DataGridHeaderColumnDividerSkinElement); 
DataGridHeaderElement.ColumnDividerStyleDefault.setStyle("Width", 					11);
DataGridHeaderElement.ColumnDividerStyleDefault.setStyle("TabStop", 				-1);
DataGridHeaderElement.ColumnDividerStyleDefault.setStyle("UpSkinStyle", 			DataGridHeaderElement.ColumnDividerSkinStyleDefault);
DataGridHeaderElement.ColumnDividerStyleDefault.setStyle("OverSkinStyle", 			DataGridHeaderElement.ColumnDividerSkinStyleDefault);
DataGridHeaderElement.ColumnDividerStyleDefault.setStyle("DownSkinStyle", 			DataGridHeaderElement.ColumnDividerSkinStyleDefault);
DataGridHeaderElement.ColumnDividerStyleDefault.setStyle("DisabledSkinStyle", 		DataGridHeaderElement.ColumnDividerSkinStyleDefault);

DataGridHeaderElement.StyleDefault.setStyle("ColumnDividerClass", 					ButtonElement);
DataGridHeaderElement.StyleDefault.setStyle("ColumnDividerStyle", 					DataGridHeaderElement.ColumnDividerStyleDefault); 

DataGridHeaderElement.StyleDefault.setStyle("DraggableColumns", 					true);



////////Internal////////////////////////////////

/**
 * @function _onColumnDividerDrag
 * Event handler for column divider "dragging" event. Updates the header item renderers and 
 * parent grid column widths.
 * 
 * @param elementEvent ElementEvent
 * The ElementEvent to process.
 */	
DataGridHeaderElement.prototype._onColumnDividerDrag = 
	function (elementEvent)
	{
		if (this._listData == null)
			return;
		
		var parentGrid = this._listData._parentList;
		var dividerRenderer = null;
		var expectedPosition = 0;
		var totalSize = 0;
		var totalPercent = 0;
		var i;
		
		//Record column data
		var columnData = []; 
		for (i = 0; i < parentGrid._gridColumns.length; i++)
		{
			columnData.push(
				{actualSize:parentGrid._columnSizes[i], 
				percentSize:parentGrid._columnPercents[i],
				oldPercentSize:parentGrid._columnPercents[i],
				minSize:parentGrid._gridColumns[i].getStyle("MinSize"),
				minPercent:0});
			
			totalSize += columnData[i].actualSize;
			totalPercent += columnData[i].percentSize;
		}
		
		//Min percent per column based on its min pixel size.
		for (i = 0; i < columnData.length; i++)
			columnData[i].minPercent = columnData[i].minSize / totalSize * totalPercent;
		
		//Calculate new column widths
		var currentColumn = 0;
		for (i = columnData.length; i < this._itemRenderersContainer._children.length; i++)
		{
			dividerRenderer = this._itemRenderersContainer._children[i];
			currentColumn = i - columnData.length;
			
			expectedPosition += columnData[currentColumn].actualSize;
			
			if (dividerRenderer == elementEvent.getCurrentTarget())
			{
				//Columns left of the divider we adjust by pixel and re-calculate their approximate percents.
				//Columns right of the divider we re-calculate their percents, and then determine the pixel size.
				
				expectedPosition = Math.round(expectedPosition - (dividerRenderer._width / 2)); //Round here
				
				var minPosition = expectedPosition;
				var maxPosition = expectedPosition;
				
				//Record "right" side column data, determine maximum slide position.
				var remainingPercent = 0;
				var remainingSize = 0;
				var remainingColumns = [];
				for (var i2 = currentColumn + 1; i2 < columnData.length; i2++)
				{
					remainingColumns.push(columnData[i2]);
					remainingPercent += columnData[i2].percentSize;
					remainingSize += columnData[i2].actualSize;
					maxPosition += columnData[i2].actualSize - columnData[currentColumn].minSize;
				}
				
				//Minimum slide position.
				for (var i2 = currentColumn; i2 >= 0; i2--)
					minPosition -= (columnData[i2].actualSize - columnData[i2].minSize);

				//Correct if we're outside of min/max
				var actualPosition = dividerRenderer._x;
				if (actualPosition < minPosition)
					actualPosition = minPosition;
				if (actualPosition > maxPosition)
					actualPosition = maxPosition;
				
				//Adjust left side columns
				var percentDelta = 0;
				var availableDelta = actualPosition - expectedPosition;
				remainingSize -= availableDelta;
				
				for (var i2 = currentColumn; i2 >= 0; i2--)
				{
					//Adjust size
					if (columnData[i2].actualSize + availableDelta < columnData[i2].minSize)
					{
						availableDelta -= columnData[i2].actualSize - columnData[i2].minSize;
						columnData[i2].actualSize = columnData[i2].minSize;
					}
					else
					{
						columnData[i2].actualSize += availableDelta;
						availableDelta = 0;
					}
					
					//Calculate column's new percent
					columnData[i2].percentSize = columnData[i2].actualSize / totalSize * totalPercent;
					
					//Add up the percent delta to distribute to "right" side columns.
					percentDelta += columnData[i2].percentSize - columnData[i2].oldPercentSize;
					
					if (availableDelta == 0)
						break;
				}
				
				//Calculate new percentages for remaining columns.
				var percentColumns = remainingColumns.slice();		
				var done = false;
				while (done == false)
				{
					done = true;
					for (var i2 = 0; i2 < percentColumns.length; i2++)
					{
						//Break the percent delta up by ratio.
						var delta = percentDelta * (percentColumns[i2].oldPercentSize / remainingPercent);
						
						//We hit minimum percent, use the minimum, remove it from the calculation and restart.
						if (percentColumns[i2].oldPercentSize - delta < percentColumns[i2].minPercent)
						{
							percentColumns[i2].percentSize = percentColumns[i2].minPercent;
							remainingPercent -= percentColumns[i2].minPercent;
							percentDelta -= (percentColumns[i2].oldPercentSize - percentColumns[i2].percentSize);
							
							percentColumns.splice(i2, 1);
							
							done = false;
							break;
						}
						else
							percentColumns[i2].percentSize = percentColumns[i2].oldPercentSize - delta;
					}
				}
	
				//Translate remaining column percents to actual widths.
				CanvasElement._calculateMinMaxPercentSizes(remainingColumns, remainingSize);
				
				break;
			}
		}
		
		//Update Grids column data.
		for (i = 0; i < columnData.length; i++)
		{
			parentGrid._columnSizes[i] = columnData[i].actualSize;
			parentGrid._columnPercents[i] = columnData[i].percentSize;
		}
		
		this._invalidateLayout();
		parentGrid._invalidateLayout(); //For gridlines
		parentGrid._invalidateListRenderersLayout(); //Adjust columns
	};

//@Override
DataGridHeaderElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DataGridHeaderElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("ColumnDividerClass" in stylesMap || "ColumnDividerStyle" in stylesMap)
			this._setListData(this._listData, this._itemData);
		
		if ("DraggableColumns" in stylesMap && this._listData != null)
		{
			var parentGrid = this._listData._parentList;
			var draggableColumns = this.getStyle("DraggableColumns");
			var dividerRenderer = null;
			var hasListener = null;
			
			for (var i = parentGrid._gridColumns.length; i < this._itemRenderersContainer._children.length; i++)
			{
				dividerRenderer = this._itemRenderersContainer._children[i];
				dividerRenderer.setStyle("Draggable", draggableColumns);
				
				hasListener = dividerRenderer.hasEventListener("dragging", this._onColumnDividerDragInstance);
				
				if (draggableColumns == true && hasListener == false)
					dividerRenderer.addEventListener("dragging", this._onColumnDividerDragInstance);
				else if (draggableColumns == false && hasListener == true)
					dividerRenderer.removeEventListener("dragging", this._onColumnDividerDragInstance);
				
				if (draggableColumns == true)
					dividerRenderer.setStyle("Enabled", true);
				else
					dividerRenderer.setStyle("Enabled", false);
			}
		}
	};
	
//@Override
DataGridHeaderElement.prototype._setListData = 
	function (listData, itemData)
	{
		// Call base.base
		DataGridDataRenderer.base.prototype._setListData.call(this, listData, itemData);
		
		if (listData == null)
			return;
		
		var i = 0;
		var renderer = null;
		for (i = 0; i < listData._parentList._gridColumns.length; i++)
		{
			renderer = this._itemRenderersContainer._getChildAt(i);
			
			if (renderer == null)
			{
				renderer = listData._parentList._createHeaderItemRenderer(i);
				this._itemRenderersContainer._addChildAt(renderer, i);
			}
			else
			{
				columnDef = listData._parentList._gridColumns[i];
				
				if (renderer.constructor != columnDef.getStyle("HeaderItemClass"))
				{ //Renderer Class changed
					
					this._itemRenderersContainer._removeChildAt(i);
					renderer = listData._parentList._createHeaderItemRenderer(i);
					this._itemRenderersContainer._addChildAt(renderer, i);
				}
				else
				{ //Update DataGridData
					
					listData._parentList._updateHeaderItemRendererData(renderer, i);
				}
			}
		}
		
		var dividerClass = this.getStyle("ColumnDividerClass");
		var draggableColumns = this.getStyle("DraggableColumns");
		
		var totalElements = listData._parentList._gridColumns.length;
		
		if (dividerClass != null)
			totalElements = (totalElements * 2) - 1;
		
		for (var i2 = i; i2 < totalElements; i2++)
		{
			renderer = this._itemRenderersContainer._getChildAt(i2);
			
			if (renderer != null && renderer.constructor != dividerClass)
			{
				this._itemRenderersContainer._removeChildAt(i2);
				renderer = null;
			}
			
			if (renderer == null)
			{
				renderer = new (dividerClass)();
				this._applySubStylesToElement("ColumnDividerStyle", renderer);
				renderer.setStyle("Draggable", draggableColumns);
				
				if (draggableColumns == true)
				{
					renderer.addEventListener("dragging", this._onColumnDividerDragInstance);
					renderer.setStyle("Enabled", true);
				}
				else
					renderer.setStyle("Enabled", false);
				
				this._itemRenderersContainer._addChildAt(renderer, i2);
			}
			else
				this._applySubStylesToElement("ColumnDividerStyle", renderer);
		}
		
		//Purge excess renderers.
		while (this._itemRenderersContainer._children.length > totalElements)
			this._itemRenderersContainer._removeChildAt(this._itemRenderersContainer._children.length - 1);
		
		//Invalidate, the item renderer container doesnt measure so wont do it for us.
		this._invalidateMeasure();
		this._invalidateLayout();
	};

//@Override	
DataGridHeaderElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		DataGridHeaderElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		if (this._listData == null)
			return;
		
		var parentGrid = this._listData._parentList;
		var dividerRenderer = null;
		var currentPosition = 0;
		var columnSize = 0;
		
		//Size / Position divider lines
		for (var i = parentGrid._columnSizes.length; i < this._itemRenderersContainer._children.length; i++)
		{
			dividerRenderer = this._itemRenderersContainer._children[i];
			columnSize = parentGrid._columnSizes[i - parentGrid._columnSizes.length];
			currentPosition += columnSize;
			
			var dividerHeight = dividerRenderer.getStyle("Height");
			if (dividerHeight == null)
				dividerHeight = Math.round(this._itemRenderersContainer._height * .7);
			
			dividerRenderer._setActualSize(dividerRenderer._getStyledOrMeasuredWidth(), dividerHeight);
			dividerRenderer._setActualPosition(currentPosition - (dividerRenderer._getStyledOrMeasuredWidth() / 2), (this._itemRenderersContainer._height / 2) - (dividerRenderer._height / 2));
		}
	};
	
	


/**
 * @depends DataListElement.js
 * @depends DataGridDataRenderer.js
 * @depends DataGridHeaderElement.js
 * @depends CanvasElement.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////////DataGridElement/////////////////////////////////		
	
/**
 * @class DataGridElement
 * @inherits DataListElement
 * 
 * DataGridElement is a data-driven container that renders a header and rows
 * via a supplied ListCollection, DataGridColumnDefinition(s), and Data/Item Renderers.
 * A scroll bar will be added if the collection size exceeds the available area. 
 * DataGridElement only renders visible DataRenderers so collection size does not impact performance
 * and allows the header, rows, header item, and row item classes to be specified and styled if desired.
 * 
 * The default header class is DataGridHeaderElement.
 * The default ListItem DataRenderer (renders a row) is DataGridDataRenderer.
 * 
 * Default header/row ItemRenderers are supplied by the DataGridColumnDefinition and are as follows.
 * The default HeaderItem DataRenderer is DataGridHeaderItemRenderer.
 * The default RowItem DataRenderer DataGridLabelItemRenderer.
 * 
 * 
 * @seealso DataGridHeaderElement
 * @seealso DataGridDataRenderer
 * @seealso DataGridColumnDefinition
 * @seealso DataGridHeaderItemRenderer
 * @seealso DataGridLabelItemRenderer
 * 
 * 
 * @constructor DataGridElement 
 * Creates new DataGridElement instance.
 */
function DataGridElement()
{
	DataGridElement.base.prototype.constructor.call(this);
	
	/**
	 * @member _gridColumns Array
	 * Read Only - Array of DataGridColumnDefinition.
	 */
	this._gridColumns = [];
	
	this._columnSizes = [];
	this._columnPercents = [];	
	
	this._selectedColumnIndex = -1;
	
	this._overIndex = -1; //row index
	this._overColumnIndex = -1;
	
	this._gridHeader = null;
	
	this._gridLineContainer = new CanvasElement();
	this._gridLineContainer.setStyle("MouseEnabled", false);
	this._gridLineContainer.setStyle("ClipContent", true);
	this._addChild(this._gridLineContainer);
	
	var _self = this;
	
	//Private event handler, proxy to prototype.
	this._onDataGridColumnDefinitionChangedInstance = 
		function (styleChangedEvent)
		{
			_self._onDataGridColumnDefinitionChanged(styleChangedEvent);
		};
	this._onDataGridRowItemClickInstance = 
		function (elementMouseEvent)
		{
			_self._onDataGridRowItemClick(elementMouseEvent);
		};
	this._onDataGridRowItemRolloverInstance = 
		function (event)
		{
			_self._onDataGridRowItemRollover(event);
		};
	this._onDataGridRowItemRolloutInstance = 
		function (event)
		{
			_self._onDataGridRowItemRollout(event);
		};	
	this._onDataGridHeaderItemClickInstance = 
		function (elementMouseEvent)
		{
			_self._onDataGridHeaderItemClick(elementMouseEvent);
		};
	this._onGridLineContainerMeasureCompleteInstance = 
		function (event)
		{
			_self._onGridLineContainerMeasureComplete(event);
		};
		
	
	this._gridLineContainer.addEventListener("measurecomplete", this._onGridLineContainerMeasureCompleteInstance);	
}

//Inherit from DataListElement
DataGridElement.prototype = Object.create(DataListElement.prototype);
DataGridElement.prototype.constructor = DataGridElement;
DataGridElement.base = DataListElement;

////////////Events////////////////////////////////////////

/**
 * @event listitemclick ElementGridItemClickEvent
 * Dispatched when an ItemRenderer or header is clicked. Includes associated collection item/index.
 */


/////////////Style Types////////////////////////////////////////////

DataGridElement._StyleTypes = Object.create(null);

/**
 * @style HeaderClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the DataGrid header. Default is DataGridHeaderElement.
 * 
 * @seealso DataGridHeaderElement
 */
DataGridElement._StyleTypes.HeaderClass = 						StyleableBase.EStyleType.NORMAL;		// Element constructor()

/**
 * @style HeaderStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the header element.
 */
DataGridElement._StyleTypes.HeaderStyle = 						StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style GridLinesPriority String
 * 
 * Determines which set of grid lines will be rendered first. Allowable values are "vertical" or "horizontal".
 */
DataGridElement._StyleTypes.GridLinesPriority = 				StyleableBase.EStyleType.NORMAL;		// "vertical" || "horizontal" (Which lines are drawn first / below)

/**
 * @style VerticalGridLinesClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the DataGrid vertical grid lines. Default is CanvasElement.
 */
DataGridElement._StyleTypes.VerticalGridLinesClass = 			StyleableBase.EStyleType.NORMAL;		// Element constructor()

/**
 * @style VerticalGridLinesStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the vertical grid line elements.
 */
DataGridElement._StyleTypes.VerticalGridLinesStyle = 			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style HorizontalGridLinesClass CanvasElement
 * 
 * The CanvasElement constructor to be used for the DataGrid horizontal grid lines. Default is null.
 */
DataGridElement._StyleTypes.HorizontalGridLinesClass = 			StyleableBase.EStyleType.NORMAL;		// Element constructor()

/**
 * @style HorizontalGridLinesStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the horizontal grid line elements.
 */
DataGridElement._StyleTypes.HorizontalGridLinesStyle = 			StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition


////////////Default Styles/////////////////////////////////////////

DataGridElement.StyleDefault = new StyleDefinition();

/////GridLines default style //////
DataGridElement.GridLineStyleDefault = new StyleDefinition();
DataGridElement.GridLineStyleDefault.setStyle("Width", 					1);				// number
DataGridElement.GridLineStyleDefault.setStyle("Height", 				1); 			// number
DataGridElement.GridLineStyleDefault.setStyle("BackgroundFill", 		"#BBBBBB");		// "#000000"
///////////////////////////////////

/////ScrollBar default style //////
DataGridElement.ScrollBarStyleDefault = new StyleDefinition();
DataGridElement.ScrollBarStyleDefault.setStyle("Padding", 				-1);			// number
///////////////////////////////////

//Override base class styles
DataGridElement.StyleDefault.setStyle("ListItemClass", 					DataGridDataRenderer); 					// Element constructor()	
DataGridElement.StyleDefault.setStyle("ListItemStyle", 					null); 									// StyleDefinition

DataGridElement.StyleDefault.setStyle("BorderType",		 				"solid"); 	
DataGridElement.StyleDefault.setStyle("BorderThickness",	 			1);
DataGridElement.StyleDefault.setStyle("PaddingTop",	 					1);
DataGridElement.StyleDefault.setStyle("PaddingBottom", 					1);
DataGridElement.StyleDefault.setStyle("PaddingLeft",					1);
DataGridElement.StyleDefault.setStyle("PaddingRight", 					1);
DataGridElement.StyleDefault.setStyle("ScrollBarStyle", 				DataGridElement.ScrollBarStyleDefault);	// StyleDefinition

//DataGrid specific
DataGridElement.StyleDefault.setStyle("HeaderClass", 					DataGridHeaderElement); 				// Element constructor()
DataGridElement.StyleDefault.setStyle("HeaderStyle", 					null); 									// StyleDefinition

DataGridElement.StyleDefault.setStyle("GridLinesPriority", 				"vertical"); 							// "vertical" || "horizontal"

DataGridElement.StyleDefault.setStyle("VerticalGridLinesClass", 		CanvasElement); 						// Element constructor()
DataGridElement.StyleDefault.setStyle("VerticalGridLinesStyle", 		DataGridElement.GridLineStyleDefault); 	// StyleDefinition

DataGridElement.StyleDefault.setStyle("HorizontalGridLinesClass", 		CanvasElement); 						// Element constructor()
DataGridElement.StyleDefault.setStyle("HorizontalGridLinesStyle", 		DataGridElement.GridLineStyleDefault); 	// StyleDefinition
DataGridElement.StyleDefault.setStyle("TabStop", 						0);


///////////Public//////////////////////////////////

/**
 * @function addColumnDefinition
 * Adds a column definition to be rendered by the DataGrid.
 * 
 * @param columnDefinition DataGridColumnDefinition
 * Column definition to be rendered by the DataGrid.
 */
DataGridElement.prototype.addColumnDefinition = 
	function (columnDefinition)
	{
		return this.addColumnDefinitionAt(columnDefinition, this._gridColumns.length);
	};
	
/**
 * @function addColumnDefinitionAt
 * Adds a column definition to be rendered by the DataGrid at a supplied column index.
 * 
 * @param columnDefinition DataGridColumnDefinition
 * Column definition to be rendered by the DataGrid.
 * 
 * @param index int
 * The index to insert the column definition.
 * 
 * @returns DataGridColumnDefinition
 * The added DataGridColumnDefinition or null if could not be added.
 */	
DataGridElement.prototype.addColumnDefinitionAt = 
	function (columnDefinition, index)
	{
		if (!(columnDefinition instanceof DataGridColumnDefinition))
			return null;
		
		this._gridColumns.splice(index, 0, columnDefinition);
		this._columnPercents.splice(index, 0, columnDefinition.getStyle("PercentSize"));
		
		if (this._manager != null)
			columnDefinition.addEventListener("stylechanged", this._onDataGridColumnDefinitionChangedInstance);
		
		this._columnSizes = []; //Force grid to re-calculate column sizes
		this._columnsChanged();
		
		return columnDefinition;
	};
	
/**
 * @function getColumnDefinitionAt
 * Gets the DataGridColumnDefinition at a supplied column index.
 * 
 * @param index int
 * The index to return the DataGridColumnDefinition.
 * 
 * @returns DataGridColumnDefinition
 * The DataGridColumnDefinition at the supplied index.
 */	
DataGridElement.prototype.getColumnDefinitionAt = 
	function (index)
	{
		if (index < 0 || index >= this._gridColumns.length)
			return null;
	
		return this._gridColumns[index];
	};	
	
/**
 * @function removeColumnDefinition
 * Removes a column definition from the DataGrid.
 * 
 * @param columnDefinition DataGridColumnDefinition
 * Column definition to be removed.
 * 
 * @returns DataGridColumnDefinition
 * The removed column definition.
 */	
DataGridElement.prototype.removeColumnDefinition = 
	function (columnDefinition)
	{
		return this.removeColumnDefinitionAt(this._gridColumns.indexOf(columnDefinition));
	};

/**
 * @function removeColumnDefinitionAt
 * Removes a column definition from the DataGrid.
 * 
 * @param index int
 * Column index of definition to be removed.
 * 
 * @returns DataGridColumnDefinition
 * The removed column definition or null if the index was out of range.
 */		
DataGridElement.prototype.removeColumnDefinitionAt = 
	function (index)
	{
		if (index < 0 || index >= this._gridColumns.length)
			return null;
		
		var removed = this._gridColumns.splice(index, 1)[0]; //Returns array of removed items.
		this._columnPercents.splice(index, 1);
		
		if (this._manager != null)
			removed.removeEventListener("stylechanged", this._onDataGridColumnDefinitionChangedInstance);
		
		this._columnSizes = []; //Force grid to re-calculate column sizes
		this._columnsChanged();
		
		return removed;
	};

/**
 * @function getNumColumns
 * Gets the number of column definitions.
 * 
 * @returns int
 * Number of column definitions.
 */		
DataGridElement.prototype.getNumColumns = 
	function ()
	{
		return this._gridColumns.length;
	};
	
/**
 * @function setSelectedIndex
 * @override
 * 
 * Sets the selected collection (row) index and column index. 
 * When both row and column is specified the associated cell is selected.
 * 
 * @param rowIndex int
 * The collection (row) index to be selected or -1 for none.
 * 
 * @param columnIndex int
 * the column index to be selected or -1 for none.
 * 
 * @returns bool
 * Returns true if the selection changed.
 */	
DataGridElement.prototype.setSelectedIndex = 
	function (rowIndex, columnIndex)
	{
		if (this._selectedIndex == rowIndex && this._selectedColumnIndex == columnIndex)
			return false;
		
		if (rowIndex > this._listCollection.length -1)
			return false;
		
		if (columnIndex > this._gridColumns.length - 1)
			return false;
		
		if (rowIndex < -1)
			rowIndex = -1;
		
		if (columnIndex < -1)
			columnIndex = -1;
		
		this._selectedIndex = rowIndex;
		this._selectedColumnIndex = columnIndex;
		
		var selectionData = {rowIndex:this._selectedIndex, columnIndex:this._selectedColumnIndex, rowOverIndex:this._overIndex, columnOverIndex:this._columnOverIndex};
		
		for (var i = 0; i < this._contentPane._children.length; i++)
			this._contentPane._children[i]._setListSelected(selectionData);
		
		return true;
	};	
	
/**
 * @function getSelectedIndex
 * @override
 * Gets the selected collection (row) and column index. 
 * 
 * @returns Object
 * Returns and object containing row and column indexes {row:-1, column:-1}
 */		
DataGridElement.prototype.getSelectedIndex = 
	function ()
	{
		return {row:this._selectedIndex, column:this._selectedColumnIndex};
	};	
	
	
///////////Internal////////////////////////////////
	
//@private
DataGridElement.prototype._onGridLineContainerMeasureComplete = 
	function (event)
	{
		this._invalidateLayout();
	};
	
/**
 * @function _onDataGridColumnDefinitionChanged
 * Event handler for DataGridColumnDefinition "stylechanged" event. Updates the DataGrid column.
 * 
 * @param styleChangedEvent StyleChangedEvent
 * The StyleChangedEvent to process.
 */	
DataGridElement.prototype._onDataGridColumnDefinitionChanged = 
	function (styleChangedEvent)
	{
		var styleName = styleChangedEvent.getStyleName();
		
		if (styleName == "PercentSize")
		{
			var columnIndex = this._gridColumns.indexOf(styleChangedEvent.getTarget());
			this._columnPercents[columnIndex] = styleChangedEvent.getNewValue();
			this._columnSizes = []; //Force grid to re-calculate column sizes
			this._invalidateLayout();
		}
		else if (styleName == "MinSize")
		{
			this._columnSizes = []; //Force grid to re-calculate column sizes
			this._invalidateLayout();
		}
		else
			this._columnsChanged();
	};

/**
 * @function _columnsChanged
 * Called in response to columns being added/removed or their styles changed.
 * Updates the DataGrid columns.
 */	
DataGridElement.prototype._columnsChanged = 
	function ()
	{
		//Refresh all the ListData, data hasnt changed, but this
		//also tells the renderer to inspect and adjust the columns.
		if (this._gridHeader != null)
		{
			this._gridHeader._setListData(
				this._gridHeader._listData,
				null);
		}
		
		var renderer = null;
		for (var i = 0; i < this._contentPane._children.length; i++)
		{
			renderer = this._contentPane._children[i];
			this._updateRendererData(renderer, renderer._listData._itemIndex);
		}
		
		this._invalidateLayout();
	};

//@override	
DataGridElement.prototype._onDataListCollectionChanged = 
	function (collectionChangedEvent)
	{
		DataGridElement.base.prototype._onDataListCollectionChanged.call(this, collectionChangedEvent);
		
		//Sort may have happened, update the header's data so it can adjust sort icon.
		if (collectionChangedEvent.getKind() == "reset" && this._gridHeader != null)
		{
			//Data hasnt actually changed.
			this._gridHeader._setListData(
				this._gridHeader._listData,
				null);
		}
	};
	
//@override	
DataGridElement.prototype._onCanvasElementAdded = 
	function (addedRemovedEvent)
	{
		DataGridElement.base.prototype._onCanvasElementAdded.call(this, addedRemovedEvent);
	
		for (var i = 0; i < this._gridColumns.length; i++)
			this._gridColumns[i].addEventListener("stylechanged", this._onDataGridColumnDefinitionChangedInstance);
	};

//@override	
DataGridElement.prototype._onCanvasElementRemoved = 
	function (addedRemovedEvent)
	{
		DataGridElement.base.prototype._onCanvasElementRemoved.call(this, addedRemovedEvent);
		
		for (var i = 0; i < this._gridColumns.length; i++)
			this._gridColumns[i].removeEventListener("stylechanged", this._onDataGridColumnDefinitionChangedInstance);
	};		
	
//@override		
DataGridElement.prototype._createRenderer = 
	function (itemIndex)
	{
		var newRenderer = new (this.getStyle("ListItemClass"))();
		this._applySubStylesToElement("ListItemStyle", newRenderer);
		this._updateRendererData(newRenderer, itemIndex);
		
		return newRenderer;
	};	

/**
 * @function _createHeaderItemRenderer
 * Generates a header ItemRenderer base on the column definition HeaderItemClass style.
 * 
 * @param columnIndex int
 * Column index associated with the header ItemRenderer.
 * 
 * @returns CanvasElement
 * The new header ItemRenderer instance.
 */		
DataGridElement.prototype._createHeaderItemRenderer = 
	function (columnIndex)
	{
		var columnDefinition = this._gridColumns[columnIndex];
		
		var headerItemClass = columnDefinition.getStyle("HeaderItemClass");
		var newRenderer = new (headerItemClass)();
		columnDefinition._applySubStylesToElement("HeaderItemStyle", newRenderer);
		
		this._updateHeaderItemRendererData(newRenderer, columnIndex);		
		
		newRenderer.addEventListener("click", this._onDataGridHeaderItemClickInstance);
		
		return newRenderer;
	};
	
/**
 * @function _updateHeaderItemRendererData
 * Updates the header ItemRenderer list data.
 * 
 * @param renderer CanvasElement
 * Header ItemRenderer to update.
 * 
 * @param columnIndex int
 * Column index to associate with the header ItemRenderer.
 */		
DataGridElement.prototype._updateHeaderItemRendererData = 
	function (renderer, columnIndex)
	{
		var columnDefinition = this._gridColumns[columnIndex];
		
		//Optimize, only create new data if its actually changed.
		var listData = null;
		if (renderer._listData != null && renderer._listData._columnIndex == columnIndex)
			listData = renderer._listData;
		else
			listData = new DataGridItemData(this, -1, columnIndex);
		
		columnDefinition._applySubStylesToElement("HeaderItemStyle", renderer);
		
		renderer._setListData(
			listData,
			null);
	};
	
/**
 * @function _onDataGridHeaderItemClick
 * Event handler for header ItemRenderer "click" event. Sorts the collection if a 
 * CollectionSort is assigned to the DataGridColumDefinition and dispatched "listitemclick" event.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */			
DataGridElement.prototype._onDataGridHeaderItemClick = 
	function (elementMouseEvent)
	{
		var columnIndex = elementMouseEvent.getCurrentTarget()._listData._columnIndex;
		
		var collectionSort = this._gridColumns[columnIndex].getStyle("CollectionSort");
		if (this._listCollection != null && collectionSort != null && collectionSort instanceof CollectionSort)
		{
			if (this._listCollection._collectionSort != collectionSort)
			{
				collectionSort._isDecending = false;
				this._listCollection.setCollectionSort(collectionSort);
				this._listCollection.sort();
			}
			else
			{
				collectionSort._isDecending = !(collectionSort._isDecending);
				this._listCollection.sort();
			}
		}
		
		this.dispatchEvent(new ElementGridItemClickEvent(-1, columnIndex, null));
	};
	
/**
 * @function _createRowItemRenderer
 * Generates a row ItemRenderer base on the column definition RowItemClass style.
 * 
 * @param itemIndex int
 * Collection item index to associate with the row ItemRenderer.
 * 
 * @param columnIndex int
 * Column index to associate with the row ItemRenderer.
 * 
 * @returns CanvasElement
 * The new row ItemRenderer instance.
 */		
DataGridElement.prototype._createRowItemRenderer = 
	function (itemIndex, columnIndex)
	{
		var columnDefinition = this._gridColumns[columnIndex];
	
		var rowItemClass = columnDefinition.getStyle("RowItemClass");
		var newRenderer = new (rowItemClass)();
		columnDefinition._applySubStylesToElement("RowItemStyle", newRenderer);
		
		this._updateRowItemRendererData(newRenderer, itemIndex, columnIndex);		
		
		newRenderer.addEventListener("click", this._onDataGridRowItemClickInstance);
		newRenderer.addEventListener("rollover", this._onDataGridRowItemRolloverInstance);
		newRenderer.addEventListener("rollout", this._onDataGridRowItemRolloutInstance);
		
		return newRenderer;
	};

DataGridElement.prototype._onDataGridRowItemRollover = 
	function (event)
	{
		this._overIndex = event.getCurrentTarget()._listData._itemIndex;
		this._columnOverIndex = event.getCurrentTarget()._listData._columnIndex;
		
		var renderer = null;
		for (var i = 0; i < this._contentPane._children.length; i++)
		{
			renderer = this._contentPane._children[i];
			renderer._setListSelected({rowIndex:this._selectedIndex, columnIndex:this._selectedColumnIndex, rowOverIndex:this._overIndex, columnOverIndex:this._columnOverIndex});
		}
	};
	
DataGridElement.prototype._onDataGridRowItemRollout = 
	function (event)
	{
		this._overIndex = -1;
		this._columnOverIndex = -1;
		
		var renderer = null;
		for (var i = 0; i < this._contentPane._children.length; i++)
		{
			renderer = this._contentPane._children[i];
			renderer._setListSelected({rowIndex:this._selectedIndex, columnIndex:this._selectedColumnIndex, rowOverIndex:this._overIndex, columnOverIndex:this._columnOverIndex});
		}
	};	
	
//@override	
DataGridElement.prototype._updateRendererData = 
	function (renderer, itemIndex)
	{
		var listData = null;
		
		//Optimize, dont create new data if already exists.
		if (renderer._listData != null)
		{
			listData = renderer._listData;
			listData._parentList = this;
			listData._itemIndex = itemIndex;
		}
		else
			listData = new DataListData(this, itemIndex);
	
		//Always call the function even if data has not changed, this indicates to the
		//renderer to inspect its parent related data and it may make changes even if
		//this data is the same. An example is changes to a DataGrid's columns.
		renderer._setListData(
			listData,
			this._listCollection.getItemAt(itemIndex));
		
		renderer._setListSelected({rowIndex:this._selectedIndex, columnIndex:this._selectedColumnIndex, rowOverIndex:this._overIndex, columnOverIndex:this._columnOverIndex});
	};	
	
/**
 * @function _onDataGridRowItemClick
 * Event handler for the row ItemRenderer "click" event. Updates selected index/item and dispatches "listitemclick" and "changed" events.
 * 
 * @param elementMouseEvent ElementMouseEvent
 * The ElementMouseEvent to process.
 */
DataGridElement.prototype._onDataGridRowItemClick = 
	function (elementMouseEvent)
	{
		var itemIndex = elementMouseEvent.getCurrentTarget()._listData._itemIndex;
		var columnIndex = elementMouseEvent.getCurrentTarget()._listData._columnIndex;
		var itemData = elementMouseEvent.getCurrentTarget()._itemData;
		
		var columnDef = this._gridColumns[columnIndex];
		
		var gridSelectable = this.getStyle("Selectable");
		var columnSelectable = columnDef.getStyle("Selectable");
		var elementSelectable = elementMouseEvent.getCurrentTarget().getStyle("Selectable"); 
		
		if (elementSelectable === "undefined")
			elementSelectable = true;
		
		var dispatchChanged = false;
		
		if (gridSelectable && columnSelectable && elementSelectable)
		{
			var selectRow = itemIndex;
			var selectColumn = columnIndex;
			
			if (columnDef.getStyle("SelectionType") == "row")
				selectColumn = -1;
			else if (columnDef.getStyle("SelectionType") == "column")
				selectRow = -1;
			
			if (this.setSelectedIndex(selectRow, selectColumn) == true)
				dispatchChanged = true;
		}
		
		this.dispatchEvent(new ElementGridItemClickEvent(itemIndex, columnIndex, itemData));
		
		if (dispatchChanged == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};	
	
/**
 * @function _updateRowItemRendererData
 * Updates the row ItemRenderer list data.
 * 
 * @param renderer CanvasElement
 * Row ItemRenderer to update.
 * 
 * @param itemIndex int
 * Collection item index to associate with the row ItemRenderer.
 * 
 * @param columnIndex int
 * Column index to associate with the row ItemRenderer.
 */		
DataGridElement.prototype._updateRowItemRendererData = 
	function (renderer, itemIndex, columnIndex)
	{
		var columnDefinition = this._gridColumns[columnIndex];
		
		//Optimize, only create new data if its actually changed.
		var listData = null;
		if (renderer._listData != null && renderer._listData._columnIndex == columnIndex && renderer._listData._itemIndex == itemIndex)
			listData = renderer._listData;
		else
			listData = new DataGridItemData(this, itemIndex, columnIndex);
		
		columnDefinition._applySubStylesToElement("RowItemStyle", renderer);
	
		renderer._setListData(
			listData,
			this._listCollection.getItemAt(itemIndex));
	};
	
//@override
DataGridElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		DataGridElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("HeaderClass" in stylesMap)
		{
			var headerClass = this.getStyle("HeaderClass");
			
			//Destroy if class is null or does not match existing
			if ((headerClass == null && this._gridHeader != null) ||
				this._gridHeader != null && this._gridHeader.constructor != headerClass)
			{
				this._removeChild(this._gridHeader);
				this._gridHeader = null;
			}
			
			//Create
			if (headerClass != null && this._gridHeader == null)
			{
				this._gridHeader = new (headerClass)();
				
				this._gridHeader._setListData(
					new DataListData(this, -1),
					null);
				
				this._addChild(this._gridHeader);
			}
			
			this._invalidateLayout();
		}
		
		if ("HeaderStyle" in stylesMap && this._gridHeader != null)
			this._applySubStylesToElement("HeaderStyle", this._gridHeader);
		
		if ("GridLinesPriority" in stylesMap ||
			"VerticalGridLinesClass" in stylesMap ||
			"VerticalGridLinesStyle" in stylesMap ||
			"HorizontalGridLinesClass" in stylesMap ||
			"HorizontalGridLinesStyle" in stylesMap)
		{
			this._invalidateLayout();
		}
	};		
	
//@override
DataGridElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		this._setMeasuredSize(padWidth + 16, padWidth + 16);
	};
	
/**
 * @function _createGridLine
 * Generates a grid line element based on vertical/horizontal GridLinesClass style.
 * 
 * @param direction String
 * The grid line direction "vertical" or "horizontal"
 * 
 * @returns CanvasElement
 * The new grid line element.
 */		
DataGridElement.prototype._createGridLine = 
	function (direction)
	{
		var line = null;
		if (direction == "vertical")
		{
			line = new (this.getStyle("VerticalGridLinesClass"))();
			this._applySubStylesToElement("VerticalGridLinesStyle", line);
		}
		else
		{
			line = new (this.getStyle("HorizontalGridLinesClass"))();
			this._applySubStylesToElement("HorizontalGridLinesStyle", line);
		}
		
		return line;
	};
	
//@Override	
DataGridElement.prototype._doLayout = 
	function (paddingMetrics)
	{
		if (this._gridHeader != null)
		{
			var headerHeight = this._gridHeader._getStyledOrMeasuredHeight();
			
			var adjustedPadding = new DrawMetrics();
			adjustedPadding._x = paddingMetrics._x;
			adjustedPadding._y = paddingMetrics._y + headerHeight;
			adjustedPadding._width = paddingMetrics._width;
			adjustedPadding._height = paddingMetrics._height - headerHeight;
			
			//Adjust the padding so base() leaves us room for the header
			DataGridElement.base.prototype._doLayout.call(this, adjustedPadding);
		}
		else
			DataGridElement.base.prototype._doLayout.call(this, paddingMetrics);
		
		//Base makes multiple passes, no reason to run below if we're waiting for the DataList to finish anyway.
		if (this._layoutInvalid == true)
			return;
		
		//Size / Position the grid header
		if (this._gridHeader != null)
		{
			this._gridHeader._setActualPosition(this._contentPane._x, paddingMetrics.getY());
			this._gridHeader._setActualSize(paddingMetrics.getWidth(), this._gridHeader._getStyledOrMeasuredHeight());
		}
		
		var i;
		var calculateColumnSizes = false;
		
		//Determine if we need to recalculate column widths (new columns or size change)
		if (this._columnSizes.length != this._gridColumns.length)
			calculateColumnSizes = true;
		else
		{
			var totalSize = 0;
			for (i = 0; i < this._columnSizes.length; i++)
				totalSize += this._columnSizes[i];
			
			if (totalSize != this._contentPane._width)
				calculateColumnSizes = true;
		}
		
		if (calculateColumnSizes == true)
		{
			var columnData = [];
			
			//Record column size info.
			for (i = 0; i < this._gridColumns.length; i++)
			{
				columnData.push(
					{percentSize: this._columnPercents[i], //We dont use column style, its maintained separately. Header can change the percents.
					minSize: this._gridColumns[i].getStyle("MinSize")});
			}
			
			//Calculate actual widths.
			CanvasElement._calculateMinMaxPercentSizes(columnData, this._contentPane._width);
			
			//Update recorded sizes.
			var newColumnSizes = [];
			for (i = 0; i < columnData.length; i++)
				newColumnSizes.push(columnData[i].actualSize);
			
			this._columnSizes = newColumnSizes;
			
			//Invalidate children.
			if (this._gridHeader != null)
				this._gridHeader._invalidateLayout();
			
			this._invalidateListRenderersLayout();
		}
		
		////////Grid Lines//////////////////////////////////////////////////////////////////////////
		this._gridLineContainer._setActualPosition(this._contentPane._x, this._contentPane._y);
		this._gridLineContainer._setActualSize(this._contentPane._width, this._contentPane._height);
		
		var itemIndex = null;
		var lineIndex = 0;
		var gridLine = null;
		var rowRenderer = null;
		var verticalComplete = false;
		var horizontalComplete = false;
		var linePriority = this.getStyle("GridLinesPriority");
		var verticalClass = this.getStyle("VerticalGridLinesClass");
		var horizontalClass = this.getStyle("HorizontalGridLinesClass");
		
		while (verticalComplete == false || horizontalComplete == false)
		{
			if ((linePriority == "horizontal" && horizontalComplete == false) || (verticalComplete == true && horizontalComplete == false))
			{
				if (horizontalClass != null)
				{
					for (i = 0; i < this._contentPane._children.length; i++)
					{
						rowRenderer = this._contentPane._children[i];
						itemIndex = rowRenderer._listData._itemIndex;
						if (itemIndex == 0)
							continue;
						
						gridLine = this._gridLineContainer._getChildAt(lineIndex);
						if (gridLine == null)
						{
							gridLine = this._createGridLine("horizontal");
							this._gridLineContainer._addChildAt(gridLine, lineIndex);
						}
						else if (gridLine.constructor != horizontalClass)
						{
							this._gridLineContainer._removeChildAt(lineIndex);
							gridLine = this._createGridLine("horizontal");
							this._gridLineContainer._addChildAt(gridLine, lineIndex);
						}
						else
							this._applySubStylesToElement("HorizontalGridLinesStyle", gridLine);
						
						gridLine._setActualSize(this._gridLineContainer._width, gridLine.getStyle("Height"));
						gridLine._setActualPosition(0, rowRenderer._y - (gridLine._height / 2));
						
						lineIndex++;
					}
				}
				
				horizontalComplete = true;
			}
			
			if ((linePriority == "vertical" && verticalComplete == false) || (horizontalComplete == true && verticalComplete == false))
			{
				if (verticalClass != null)
				{
					var linePosition = 0;
					for (i = 0; i < this._columnSizes.length - 1; i++)
					{
						linePosition += this._columnSizes[i];
						gridLine = this._gridLineContainer._getChildAt(lineIndex);
						
						if (gridLine == null)
						{
							gridLine = this._createGridLine("vertical");
							this._gridLineContainer._addChildAt(gridLine, lineIndex);
						}
						else if (gridLine.constructor != verticalClass)
						{
							this._gridLineContainer._removeChildAt(lineIndex);
							gridLine = this._createGridLine("vertical");
							this._gridLineContainer._addChildAt(gridLine, lineIndex);
						}
						else
							this._applySubStylesToElement("VerticalGridLinesStyle", gridLine);
						
						gridLine._setActualSize(gridLine.getStyle("Width"), this._gridLineContainer._height);
						gridLine._setActualPosition(linePosition - (gridLine._width / 2), 0);
						
						lineIndex++;
					}
				}
				
				verticalComplete = true;
			}
		}
		
		//Purge excess line renderers.
		while (lineIndex < this._gridLineContainer._children.length)
			this._gridLineContainer._removeChildAt(this._gridLineContainer._children.length - 1);
	};
	
	




/**
 * @depends StyleableBase.js
 * @depends DataGridHeaderItemRenderer.js
 * @depends DataGridLabelItemRenderer.js
 */

///////////////////////////////////////////////////////////////////////
///////////////////DataGridColumnDefinition////////////////////////////		
	
/**
 * @class DataGridColumnDefinition
 * @inherits StyleableBase
 * 
 * DataGridColumnDefinition defines and stores styles necessary for the DataGrid to render columns.
 * 
 * The default HeaderItemClass is DataGridHeaderItemRenderer.
 * The default RowItemClass is DataGridLabelItemRenderer.
 * 
 * 
 * @seealso DataGridElement
 * @seealso DataGridHeaderItemRenderer
 * @seealso DataGridLabelItemRenderer
 * 
 * @constructor DataGridColumnDefinition 
 * Creates new DataGridColumnDefinition instance.
 */
function DataGridColumnDefinition()
{
	DataGridColumnDefinition.base.prototype.constructor.call(this);
	
	//Set a default alphabetical sort.  Need a different sort instance for each column.
	this.setStyle("CollectionSort", new CollectionSort(function (objA, objB) { return objA.col2 < objB.col2 ? -1 : objA.col2 > objB.col2 ? 1 : 0; }));
}
	
//Inherit from StyleableBase
DataGridColumnDefinition.prototype = Object.create(StyleableBase.prototype);
DataGridColumnDefinition.prototype.constructor = DataGridColumnDefinition;
DataGridColumnDefinition.base = StyleableBase;

/////////////Style Types///////////////////////////////

DataGridColumnDefinition._StyleTypes = Object.create(null);

/**
 * @style PercentSize Number
 * 
 * The percentage of available space the column should consume. Percentages
 * are allowed to add to more than 100 and will consume all of the available space. 
 */
DataGridColumnDefinition._StyleTypes.PercentSize = 					StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style MinSize Number
 * 
 * Minimum number of pixels the column should consume.
 */
DataGridColumnDefinition._StyleTypes.MinSize = 						StyleableBase.EStyleType.NORMAL;		// number || null

/**
 * @style HeaderText String
 * Text label to be used for the column header.
 */
DataGridColumnDefinition._StyleTypes.HeaderText = 					StyleableBase.EStyleType.NORMAL;		// "string"

/**
 * @style HeaderItemClass CanvasElement
 * 
 * The DataRenderer CanvasElement constructor to be used for the column header. 
 */
DataGridColumnDefinition._StyleTypes.HeaderItemClass = 				StyleableBase.EStyleType.NORMAL;		// Element constructor()

/**
 * @style HeaderItemStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the HeaderItem DataRenderer.
 */
DataGridColumnDefinition._StyleTypes.HeaderItemStyle = 				StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style CollectionSort CollectionSort
 * 
 * CollectionSort to be used to sort the column.
 * Default column sort uses alphabetic compare. 
 * You may null this style to prevent the column from sorting, and / or disable the column header button.
 */
DataGridColumnDefinition._StyleTypes.CollectionSort = 				StyleableBase.EStyleType.NORMAL;		// CollectionSort() 

/**
 * @style RowItemClass CanvasElement
 * 
 * The DataRenderer CanvasElement constructor to be used for this columns rows. 
 */
DataGridColumnDefinition._StyleTypes.RowItemClass = 				StyleableBase.EStyleType.NORMAL;		// Element constructor()

/**
 * @style RowItemStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the RowItem DataRenderer.
 */
DataGridColumnDefinition._StyleTypes.RowItemStyle = 				StyleableBase.EStyleType.SUBSTYLE;		// StyleDefinition

/**
 * @style RowItemLabelFunction Function
 * 
 * A function that returns a text string per a supplied collection item and columnIndex. 
 * function (itemData, columnIndex) { return "" }
 */
DataGridColumnDefinition._StyleTypes.RowItemLabelFunction = 		StyleableBase.EStyleType.NORMAL;		// function (data, columnIndex) { return "" }

/**
 * @style SelectionType string
 * 
 * Determines the selection type that the column will use. Allowable values are "row", "column", or "cell".
 * Row selection will only affect other columns also set to "row".
 */
DataGridColumnDefinition._StyleTypes.SelectionType = 				StyleableBase.EStyleType.NORMAL;		

/**
 * @style Selectable boolean
 * 
 * When true, items in this column can be selected and the DataGrid will dispatch "changed" events.
 * When SelectionType is set to "row" or "column", related items will also be set to a "selected" state.
 */
DataGridColumnDefinition._StyleTypes.Selectable = 					StyleableBase.EStyleType.NORMAL;	// true || false

/**
 * @style Highlightable boolean
 * 
 * When true, items in this column will be highlighted on mouseover, "over" state. 
 * When SelectionType is set to "row" or "column", related items will also be set to an "over" state.
 */
DataGridColumnDefinition._StyleTypes.Highlightable = 				StyleableBase.EStyleType.NORMAL;	// true || false



/////////Default Styles///////////////////////////////

DataGridColumnDefinition.StyleDefault = new StyleDefinition();

DataGridColumnDefinition.StyleDefault.setStyle("PercentSize", 				100);							// number || null
DataGridColumnDefinition.StyleDefault.setStyle("MinSize", 					12);							// number || null
DataGridColumnDefinition.StyleDefault.setStyle("SelectionType", 			"row");							// "row" || "column" || "cell"
DataGridColumnDefinition.StyleDefault.setStyle("Selectable", 				true);							// true || false
DataGridColumnDefinition.StyleDefault.setStyle("Highlightable", 			true);							// true || false

DataGridColumnDefinition.StyleDefault.setStyle("HeaderText", 				"");							// "string"
DataGridColumnDefinition.StyleDefault.setStyle("HeaderItemClass", 			DataGridHeaderItemRenderer);	// Element constructor()
DataGridColumnDefinition.StyleDefault.setStyle("HeaderItemStyle", 			null);							// StyleDefinition
DataGridColumnDefinition.StyleDefault.setStyle("CollectionSort", 			null);

DataGridColumnDefinition.StyleDefault.setStyle("RowItemClass", 				DataGridLabelItemRenderer);		// Element constructor()
DataGridColumnDefinition.StyleDefault.setStyle("RowItemStyle", 				null);							// StyleDefinition
DataGridColumnDefinition.StyleDefault.setStyle("RowItemLabelFunction", 		null);							// function (data, columnIndex) { return "" }




/**
 * @depends CanvasElement.js
 * @depends DropdownBaseElement.js
 */

//////////////////////////////////////////////////////////////
///////////////ColorPickerButtonElement///////////////////////

/**
 * @class ColorPickerButtonElement
 * @inherits DropdownBaseElement
 * 
 * ColorPickerButtonElement is a compound button that creates a pop-up ColorPicker
 * which the user can select a color which is then displayed on the button. 
 * 
 * The ColorPickerButtonElement button itself contains a child button which is used to render
 * the divider line and arrow. ColorPickerButtonElement proxies its SkinState style to the arrow
 * button so the arrow button will change states along with the ColorPickerButtonElement itself.
 * See the default skin for the arrow button DropdownArrowButtonSkinElement for additional styles.
 * 
 * @seealso DropdownArrowButtonSkinElement
 * 
 * 
 * @constructor ColorPickerButtonElement 
 * Creates new ColorPickerButtonElement instance.
 */
function ColorPickerButtonElement()
{
	ColorPickerButtonElement.base.prototype.constructor.call(this);

	this._colorSwatch = null;
	this._colorPickerPopup = null;
	this._selectedHexColor = "#FFFFFF";
	
	////////////////
	
	var _self = this;
	
	this._onColorPickerChangedInstance = 
		function (event)
		{
			_self._onColorPickerChanged(event);
		};
	this._onColorPickerKeydownInstance = 
		function (keyboardEvent)
		{
			_self._onColorPickerKeydown(keyboardEvent);
		};	
	this._onColorPickerLayoutCompleteInstance = 
		function (event)
		{
			_self._onColorPickerLayoutComplete(event);
		};	
}

//Inherit from ButtonElement
ColorPickerButtonElement.prototype = Object.create(DropdownBaseElement.prototype);
ColorPickerButtonElement.prototype.constructor = ColorPickerButtonElement;
ColorPickerButtonElement.base = DropdownBaseElement;


////////////Events///////////////////////////////

/**
 * @event changed ElementEvent
 * Dispatched when the color selection changes as a result of user input.
 */


/////////////Style Types/////////////////////////

ColorPickerButtonElement._StyleTypes = Object.create(null);

/**
 * @style PopupColorPickerStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the pop up ColorPicker element.
 */
ColorPickerButtonElement._StyleTypes.PopupColorPickerStyle = 			StyleableBase.EStyleType.SUBSTYLE; 		// StyleDefinition

/**
 * @style ColorSwatchClass CanvasElement
 * 
 * The CanvasElement or subclass constructor to be used for the color swatch icon. Defaults to CanvasElement. 
 * Note that ColorPickerButton sets the swatch class's BackgroundFill style to the selected color
 */
ColorPickerButtonElement._StyleTypes.ColorSwatchClass = 				StyleableBase.EStyleType.NORMAL; 		// CanvasElement constructor

/**
 * @style ColorSwatchStyle StyleDefinition
 * 
 * The StyleDefinition or [StyleDefinition] array to apply to the color swatch. Note that the
 * ColorPickerButton sets the color swatch's BackgroundFill style to the selected color.
 */
ColorPickerButtonElement._StyleTypes.ColorSwatchStyle = 				StyleableBase.EStyleType.SUBSTYLE; 		// StyleDefinition

/**
 * @style PopupColorPickerDistance Number
 * 
 * Vertical distance in pixels to place the ColorPicker pop up from the button.
 * Defaults to -1 to collapse default 1 pixel borders.
 */
ColorPickerButtonElement._StyleTypes.PopupColorPickerDistance = 		StyleableBase.EStyleType.NORMAL; 		


////////////Default Styles////////////////////

////Color swatch default style////
ColorPickerButtonElement.ColorSwatchStyleDefault = new StyleDefinition();
ColorPickerButtonElement.ColorSwatchStyleDefault.setStyle("BorderType", "solid");
ColorPickerButtonElement.ColorSwatchStyleDefault.setStyle("BorderColor", "#CFCFCF");

////ColorPicker pop up default style////
ColorPickerButtonElement.PopupColorPickerStyleDefault = new StyleDefinition();
ColorPickerButtonElement.PopupColorPickerStyleDefault.setStyle("BorderType", "solid");
ColorPickerButtonElement.PopupColorPickerStyleDefault.setStyle("BackgroundFill", "#FFFFFF");

////ColorPickerButton default style/////
ColorPickerButtonElement.StyleDefault = new StyleDefinition();
ColorPickerButtonElement.StyleDefault.setStyle("PopupColorPickerStyle", 				ColorPickerButtonElement.PopupColorPickerStyleDefault);
ColorPickerButtonElement.StyleDefault.setStyle("PopupColorPickerDistance", 				-1);			
ColorPickerButtonElement.StyleDefault.setStyle("ColorSwatchClass", 						CanvasElement);
ColorPickerButtonElement.StyleDefault.setStyle("ColorSwatchStyle", 						ColorPickerButtonElement.ColorSwatchStyleDefault); 			


/////////////Public///////////////////////////////

/**
 * @function setHexColor
 * Sets the selected color of the ColorPickerElement.
 * 
 * @param value String
 * RGB hex color value formatted as "#FF0000".
 */	
ColorPickerButtonElement.prototype.setHexColor = 
	function (value)
	{
		this._selectedHexColor = value;
		
		if (this._colorSwatch != null)
			this._colorSwatch.setStyle("BackgroundFill", this._selectedHexColor);
		
		if (this._colorPickerPopup != null)
			this._colorPickerPopup.setHexColor(value);
	};
	
/**
 * @function getHexColor
 * Gets the selected color of the ColorPickerElement.
 * 
 * @returns String
 * RGB hex color value formatted as "#FF0000".
 */		
ColorPickerButtonElement.prototype.getHexColor = 
	function ()
	{
		return this._selectedHexColor;
	};


/////////////Internal///////////////////////////////	
	
//@override
ColorPickerButtonElement.prototype._createPopup = 
	function ()
	{
		this._colorPickerPopup = new ColorPickerElement();
		
		this._colorPickerPopup.addEventListener("changed", this._onColorPickerChangedInstance);	
		this._colorPickerPopup.addEventListener("keydown", this._onColorPickerKeydownInstance);
		this._colorPickerPopup.addEventListener("layoutcomplete", this._onColorPickerLayoutCompleteInstance);
		
		this._applySubStylesToElement("PopupColorPickerStyle", this._colorPickerPopup);
		this._colorPickerPopup.setHexColor(this._selectedHexColor);
		
		return this._colorPickerPopup;
	};	

//@override	
ColorPickerButtonElement.prototype._updateTweenPosition = 
	function (value)
	{
		this._colorPickerPopup.setStyle("Alpha", value);
	};

/**
 * @function _onColorPickerLayoutComplete
 * Event handler for pop up ColorPicker "layoutcomplete". 
 * Updates the pop up size content size is known and determines
 * position the pop up opens depending on available space.
 * 
 * @param event DispatcherEvent
 * DispatcherEvent to process.
 */		
ColorPickerButtonElement.prototype._onColorPickerLayoutComplete =
	function (event)
	{
		this._layoutColorPickerPopup();
	};
	
/**
 * @function _onColorPickerChangedInstance
 * Event handler for pop up ColorPicker "changed" event. 
 * Updates selected color and re-dispatches "changed" event.
 * 
 * @param elementEvent ElementEvent
 * ElementEvent to process.
 */	
ColorPickerButtonElement.prototype._onColorPickerChanged = 
	function (elementEvent)
	{
		this._selectedHexColor = this._colorPickerPopup.getHexColor();
	
		if (this._colorSwatch != null)
			this._colorSwatch.setStyle("BackgroundFill", this._selectedHexColor);
		
		if (this.hasEventListener("changed", null) == true)
			this.dispatchEvent(new ElementEvent("changed", false));
	};

/**
 * @function _onColorPickerKeydown
 * Event handler for pop up ColorPicker "keydown" event. 
 * Closes the pop up when enter or tab is pressed. 
 * 
 * @param elementKeyboardEvent ElementKeyboardEvent
 * ElementKeyboardEvent to process.
 */	
ColorPickerButtonElement.prototype._onColorPickerKeydown = 
	function (elementKeyboardEvent)
	{
		if (elementKeyboardEvent.getKeyCode() == 13 ||
			elementKeyboardEvent.getKeyCode() == 9)
		{
			this.close(true);
			
			//Dispatch closed event.
			if (this.hasEventListener("closed", null) == true)
				this.dispatchEvent(new ElementEvent("closed", false));
		}
	};
	
//@private	
ColorPickerButtonElement.prototype._updateColorSwatch = 
	function ()
	{
		var colorSwatchClass = this.getStyle("ColorSwatchClass");
		
		if (colorSwatchClass == null)
		{
			if (this._colorSwatch != null)
			{
				this._removeChild(this._colorSwatch);
				this._colorSwatch = null;
			}
		}
		else
		{
			if (this._colorSwatch == null || this._colorSwatch.constructor != colorSwatchClass)
			{
				var newColorSwatch = new (colorSwatchClass)();
				
				if (this._colorSwatch != null)
					this._removeChild(this._colorSwatch);
				
				this._colorSwatch = newColorSwatch;

				this._addChild(this._colorSwatch);
			}
			
			this._applySubStylesToElement("ColorSwatchStyle", this._colorSwatch);
			this._colorSwatch.setStyle("BackgroundFill", this._selectedHexColor);
		}
	};

//@override
ColorPickerButtonElement.prototype._doStylesUpdated =
	function (stylesMap)
	{
		ColorPickerButtonElement.base.prototype._doStylesUpdated.call(this, stylesMap);
		
		if ("ColorSwatchClass" in stylesMap || "ColorSwatchStyle" in stylesMap)
		{
			this._updateColorSwatch();
			this._invalidateMeasure();
			this._invalidateLayout();
		}
		
		if ("PopupColorPickerStyle" in stylesMap && this._colorPickerPopup != null)
		{
			this._applySubStylesToElement("PopupColorPickerStyle", this._colorPickerPopup);
			this._invalidateLayout();
		}
		
		if ("PopupColorPickerDistance" in stylesMap)
			this._invalidateLayout();
	};
	
//@override
ColorPickerButtonElement.prototype._doMeasure = 
	function(padWidth, padHeight)
	{
		//We still use the text height for measuring so the sizing is the same as DropDown
		var textHeight = this.getStyle("TextSize") + this.getStyle("TextLinePaddingTop") + this.getStyle("TextLinePaddingBottom");
		
		var arrowWidth = null;
		var arrowHeight = null;
		
		if (this._arrowButton != null)
		{
			arrowWidth = this._arrowButton.getStyle("Width");
			arrowHeight = this._arrowButton.getStyle("Height");
		}
		
		if (arrowHeight == null)
			arrowHeight = textHeight + padHeight;
		if (arrowWidth == null)
			arrowWidth = Math.round(arrowHeight * .85); 
			
		var swatchWidth = null;
		var swatchHeight = null;
		
		if (this._colorSwatch != null)
		{
			swatchWidth = this._colorSwatch.getStyle("Width");
			swatchHeight = this._colorSwatch.getStyle("Height");
		}
		
		if (swatchWidth == null)
			swatchWidth = 60;
		if (swatchHeight == null)
			swatchHeight = textHeight;
		
		var h = Math.max(swatchHeight + padHeight, arrowHeight, textHeight + padHeight);
		var w = padWidth + swatchWidth + arrowWidth;
		
		this._setMeasuredSize(w, h);
	};	
	
/**
 * @function _layoutColorPickerPopup
 * Sizes and positions the ColorPicker pop up.
 */	
ColorPickerButtonElement.prototype._layoutColorPickerPopup = 
	function ()
	{
		//Color picker not displayed - bail.
		if (this._colorPickerPopup == null ||
			this._colorPickerPopup._parent == null || 
			this._colorPickerPopup._layoutInvalid == true)
		{
			return;
		}
	
		var managerMetrics = this.getMetrics(this._manager);
		
		var colorPickerDistance = this.getStyle("PopupColorPickerDistance");
		
		var colorPickerWidth = this._colorPickerPopup.getStyle("Width");
		if (colorPickerWidth == null)
			colorPickerWidth = this._colorPickerPopup._measuredWidth;
		
		var colorPickerHeight = this._colorPickerPopup.getStyle("Height");
		if (colorPickerHeight == null)
			colorPickerHeight = this._colorPickerPopup._m