# PEP 8 — общепринятый стиль кода на языке Python

Весь код на языке программирования Python пишется в соответствии с общепринятым стилем, который изложен в документе PEP 8. Соответствие какому-то единому стандарту необходимо для повышения читаемости и понимаемости кода как самим автором, так и другими программистами.

## Внешний вид кода
### Отступы
Во-первых, оформление блоков кода делается 4 пробелами. Хотя и возможно использование табуляции для проставления отступов, но этот способ является менее предпочтительным. Стоит заметить, что Python3 не допускает использование табуляции и пробелов одновременно.

Во-вторых, есть два способа выравнивания элементов, обернутых в скобки (круглые, квадратные или фигурные). Первое — это вертикальное выравнивание, второе — использование висячего отступа. Следует помнить, что во втором случае на первой линии не должно быть аргументов, а последующие строки должны выравниваться с одинаковым отступом.

##### Правильно:

```python
# Вертикальное выравнивание
foo = long_function_name(var_one, var_two,
                         var_three, var_four)

# Использование висячего отступа.  Добавлены пробелы, для отделения аргументов от блока кода.
def long_function_name(
        var_one, var_two, var_three,
        var_four):
    print(var_one)
```

##### Неправильно:

```python
# Аргументы на первой линии запрещены, если не используется вертикальное выравнивание
foo = long_function_name(var_one, var_two,
    var_three, var_four)

# Требуется больший отступ для выделения висячего отступа
def long_function_name(
    var_one, var_two, var_three,
    var_four):
    print(var_one)

```

##### Опционально:

```python
# Нет необходимости в большем количестве отступов
foo = long_function_name(
  var_one, var_two,
  var_three, var_four)

```

Закрывающие скобки в многострочных конструкциях могут находиться на последней строке с аргументами функций или элементами коллекций, на отдельной строке под первым непробельным символом предыдущей строки либо на отдельной строке под первым символом строки, начинающей многострочную конструкцию.

```python
my_list = [
    1, 2, 3,
    4, 5, 6,
    ]
result = some_function_that_takes_arguments(
    'a', 'b', 'c',
    'd', 'e', 'f',
    )

my_another_list = [
    1, 2, 3,
    4, 5, 6,
]
result = another_function_that_takes_arguments(
    'a', 'b', 'c',
    'd', 'e', 'f',
)

```

### Максимальная длина строки
Длина строки не должна превышать 79 символов. Строки документации и комментариев ограничиваются 72 символами. Для этого используют переносы строк внутри скобок. Однако бинарные операторы и некоторые конструкции требуют использования обратного слеша для явного указания, что используется многострочная конструкция.

```python
# Такие конструкции требуют использование обратного слеша
with open('/path/to/some/file/you/want/to/read') as file_1, \
        open('/path/to/some/file/being/written', 'w') as file_2:
    file_2.write(file_1.read())
    
sum_of_elements = value1 + \
    value2

```

С другой стороны, если выражение обернуто в скобки, то бинарные операторы уже не требуют обратного слеша

```python
income = (gross_wages
          + taxable_interest
          + (dividends - qualified_dividends)
          - ira_deduction
          - student_loan_interest)

```

### Пустые строки
Функции верхнего уровня и определения классов отделяются двумя пустыми строками. Методы внутри класса разделяются одной пустой строкой. Можно использовать пустые строки для логического разделения кода.

### Кодировка
Кодировка Python должна быть UTF-8. Файлы не должны иметь объявления кодировки.

Начиная с версии Python 3.0 в стандартной библиотеке действует следующее соглашение: все идентификаторы обязаны содержать только [ASCII](https://ru.wikipedia.org/wiki/ASCII) символы, и означать английские слова везде, где это возможно (во многих случаях используются сокращения или неанглийские технические термины). Кроме того, строки и комментарии тоже должны содержать лишь ASCII символы. Исключения составляют: (а) test case, тестирующий не-ASCII особенности программы, и (б) имена авторов. Авторы, чьи имена основаны не на латинском алфавите, должны транслитерировать свои имена в латиницу.

Проектам с открытым кодом для широкой аудитории также рекомендуется использовать это соглашение.

### Импорты
Каждый импорт должен быть на отдельной строке.

##### Правильно:
```python
import os
import sys
```

##### Неправильно:
```python
import os, sys
```

##### Но можно писать и так:
```python
from subprocess import Popen, PIPE
```

Импорты всегда помещаются в начале файла, сразу после комментариев к модулю и строк документации, и перед объявлением констант.

Импорты должны быть сгруппированы в следующем порядке:

1. импорты из стандартной библиотеки
2. импорты сторонних библиотек
3. импорты модулей текущего проекта

Вставляйте пустую строку между каждой группой импортов.

### Пробелы
#### Избегайте пробелов в следующих ситуациях
Непосредственно внутри скобок:

```python
# Правильно
spam(ham[1], {eggs: 2})

# Неправильно
spam( ham[ 1 ], { eggs: 2 } )
```

Непосредственно перед запятой, точкой с запятой или двоеточием:

```python
# Правильно
if x == 4: print(x, y); x, y = y, x

# Неправильно
if x == 4 : print(x , y) ; x , y = y , x

```

Сразу перед открывающей скобкой, после которой начинается список аргументов при вызове функции:

```python
# Правильно
spam(1)

# Неправильно
spam (1)

```

Сразу перед открывающей скобкой, после которой следует индекс или срез:

```python
# Правильно
my_dict['key'] = my_list[index]

# Неправильно
my_dict ['key'] = my_list [index]

```

Избегайте пробелов в конце строки. Например, из-за оставленного пробела после обратного слеша следующая строка может не считаться как продолжение предыдущей.

#### Вместе с бинарными операторами
Всегда окружайте эти бинарные операторы одним пробелом с каждой стороны: присваивания (`=`, `+=`, `-=` и другие), сравнения (`==`, `<`, `>`, `!=`, `<=`, `>=`, `in`, `not in`, `is`, `is not`), логические (`and`, `or`, `not`).

Если используются арифметические операторы с разными приоритетами, попробуйте добавить пробелы вокруг операторов с самым низким приоритетом. Используйте свои собственные суждения, однако никогда не используйте более одного пробела и всегда используйте одинаковое количество пробелов по обе стороны бинарного оператора.

##### Правильно:

```python
i = i + 1
submitted += 1
x = x*2 - 1
hypot2 = x*x + y*y
c = (a+b) * (a-b)

```

##### Неправильно:

```python
i=i+1
submitted +=1
x = x * 2 - 1
hypot2 = x * x + y * y
c = (a + b) * (a - b)

```
Не используйте пробелы вокруг знака =, если он используется для обозначения именованного аргумента или значения параметров по умолчанию.

##### Правильно:

```python
def complex(real, imag=0.0):
    return magic(r=real, i=imag)

```

##### Неправильно:

```python
def complex(real, imag = 0.0):
    return magic(r = real, i = imag)

```

### Комментарии
Комментарии, противоречащие коду, хуже, чем отсутствие комментариев. Всегда исправляйте комментарии, если меняете код!

Комментарии должны являться законченными предложениями. Если комментарий — фраза или предложение, первое слово должно быть написано с большой буквы, если только это не имя переменной, которая начинается с маленькой буквы (никогда не изменяйте регистр переменной!).

Если комментарий короткий, можно опустить точку в конце предложения. Блок комментариев обычно состоит из одного или более абзацев, составленных из полноценных предложений, поэтому каждое предложение должно оканчиваться точкой.

Ставьте два пробела после точки в конце предложения.

Программисты, которые не говорят на английском языке, пожалуйста, пишите комментарии на английском, если только вы не уверены на 120%, что ваш код никогда не будут читать люди, не знающие вашего родного языка.

#### Блоки комментариев
Блок комментариев обычно объясняет код (весь или только некоторую часть), идущий после блока, и должен иметь тот же отступ, что и сам код. Каждая строчка такого блока должна начинаться с символа # и одного пробела после него (если только сам текст комментария не имеет отступа).

Абзацы внутри блока комментариев разделяются строкой, состоящей из одного символа #.

#### Inline комментарии
Старайтесь реже использовать подобные комментарии.

Такой комментарий находится в той же строке, что и инструкция. Inline комментарии должны отделяться по крайней мере двумя пробелами от инструкции. Они должны начинаться с символа # и одного пробела.

Комментарии в строке с кодом не нужны и только отвлекают от чтения, если они объясняют очевидное. Не пишите вот так:

```python
x = x + 1                 # Increment x
```

Впрочем, такие комментарии иногда полезны:

```python
x = x + 1                 # Compensate for border
```

#### Строки документации
Пишите документацию для всех публичных модулей, функций, классов, методов. Строки документации необязательны для приватных методов, но лучше написать, что делает метод. Комментарий нужно писать после строки с def.

[PEP 257](https://www.python.org/dev/peps/pep-0257/) объясняет, как правильно и хорошо документировать. Заметьте, очень важно, чтобы закрывающие кавычки стояли на отдельной строке. А еще лучше, если перед ними будет ещё и пустая строка, например:

```python
"""Return a foobang

Optional plotz says to frobnicate the bizbaz first.

"""
```

### Соглашение об именах
Соглашения по именованию переменных в Python немного туманны, поэтому их список никогда не будет полным — тем не менее, ниже мы приводим список рекомендаций, действующих на данный момент. Новые модули и пакеты должны быть написаны согласно этим стандартам, но если в какой-либо уже существующей библиотеке эти правила нарушаются, предпочтительнее писать в едином с ней стиле.

- Имена модулей должны записываться коротко, маленькими буквами.
- Подчеркивания допускаются, если это улучшает читаемость. В именах пакетов предпочтительнее не использовать подчеркивание.
- Имена классов пишутся в виде много идущих подряд слов, каждое с заглавной буквы. Если в названии есть аббревиатура, то она пишется заглавными буквами.
- Исключение тоже является классом.
- Имена функций пишутся с маленькой буквы, между словами используют подчеркивание.
- Стиль именования функций, когда слова пишутся подряд с заглавной буквы, кроме первого слова, допустим только в тех местах, где уже преобладает такой стиль, для сохранения обратной совместимости.
- Имена переменных или открытых атрибутов класса состоят из маленьких букв, слова разделяются символами подчеркивания.
- Имена глобальных констант состоят из заглавных букв, слова разделяются символами подчеркивания.

##### Например:

- `mymodule` — модуль или пакет
- `UserClassName` — класс
- `ExceptionsAreAlsoClasses` — исключение
- `function_name` — функция или метод
- `notDesiredFunctionName` — менее предпочтительное имя функции
- `variable_name` — переменная или открытый атрибут класса
- `GLOBAL_CONSTANT` — глобальная константа

#### Символы подчеркивания
Кроме разделения слов внутри названий символы подчеркивания могут выполнять другие роли:

- Один символ подчеркивания в начале атрибута или функции говорит о том, что этот атрибут или эта функция для каких-то внутренних нужд.
- Один символ подчеркивания в конце названия используется, чтобы избежать конфликта с каким-либо зарезервированным словом.
- Два символа подчеркивания в начале названий скрываемых функций или атрибутов.
- По два символа подчеркивания в начале и в конце названия используется у так называемых "магических" методов. Эти методы описаны в документации, и изобретать свои не следует.

### Общие замечания
Никогда не используйте символы l (маленькая латинская буква «эль»), O (заглавная латинская буква «о») или I (заглавная латинская буква «ай») как однобуквенные идентификаторы. В некоторых шрифтах эти символы неотличимы от цифры один и нуля. Если очень нужно l, пишите вместо неё заглавную L.

Имена должны содержать только символы ASCII и означать только английские слова.

Если имя — часть [API](https://ru.wikipedia.org/wiki/API), то оно должно быть согласовано со стилем кода интерфейса, а не реализации.

Имена, которые видны пользователю как часть публичного API, должны следовать конвенциям, которые отражают использование, а не реализацию.

---

Как отмечалось выше, необходимо следить за оформлением кода. Однако, не всегда возможно отследить все ошибки, особенно в большом проекте. Для этого следует исползовать линтеры - программные средства, до некоторой степени контролирующие выполнение PEP 8. С некоторыми из них можно ознакомиться по ссылкам:

https://pypi.org/project/pep8/

https://habr.com/company/dataart/blog/318776/

---

Ознакомится с полной версией PEP8 на английском языке можно по ссылке: https://www.python.org/dev/peps/pep-0008/

