///
/// Copyright (C) 2015-2016, Dependable Systems Laboratory, EPFL
/// Copyright (C) 2014-2016, Cyberhaven
///
/// Permission is hereby granted, free of charge, to any person obtaining a copy
/// of this software and associated documentation files (the "Software"), to deal
/// in the Software without restriction, including without limitation the rights
/// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
/// copies of the Software, and to permit persons to whom the Software is
/// furnished to do so, subject to the following conditions:
///
/// The above copyright notice and this permission notice shall be included in all
/// copies or substantial portions of the Software.
///
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
///

#ifndef S2E_PLUGINS_POVGEN_H
#define S2E_PLUGINS_POVGEN_H

#include <inttypes.h>
#include <s2e/Plugin.h>
#include <s2e/Plugins/OSMonitors/Linux/DecreeMonitor.h>
#include <s2e/Plugins/OSMonitors/Support/ModuleExecutionDetector.h>
#include <s2e/Plugins/OSMonitors/Support/ProcessExecutionDetector.h>
#include <s2e/Plugins/Searchers/SeedSearcher.h>
#include <sstream>
#include <string>

#include "PovGenerator.h"

namespace s2e {
namespace plugins {
namespace pov {

class DecreePovGeneratorState;

/** Handler required for KLEE interpreter */
class DecreePovGenerator : public PovGenerator {
    S2E_PLUGIN

private:
    typedef std::pair<std::string, std::vector<unsigned char>> VarValuePair;
    typedef std::vector<VarValuePair> ConcreteInputs;

    typedef std::set<u_int64_t> AddressSet;
    AddressSet m_povAddresses; // addresses at which POVs were generated
    unsigned m_numPOVs;        // number of POVs generated so far

    static const std::string C_HEADER, C_FOOTER;
    DecreeMonitor *m_monitor;
    ProcessExecutionDetector *m_detector;
    ModuleExecutionDetector *m_modules;
    seeds::SeedSearcher *m_seedSearcher;

    std::string generatePoV(uint64_t seedIndex, const DecreePovGeneratorState *plgState, const PovOptions &opt,
                            const VariableRemapping &remapping, const klee::Assignment &solution,
                            const klee::ConstraintManager &constraints);

    void generatePoV(S2EExecutionState *state, const PovOptions &opt, std::string &cPov);

public:
    DecreePovGenerator(S2E *s2e);

    void initialize();

    bool generatePoV(S2EExecutionState *state, const PovOptions &opt, const std::string &filePrefix,
                     std::vector<std::string> &filePaths);

    static bool isReceive(const klee::Array *array);
    static bool isRandom(const klee::Array *array);
    static bool isRandomRead(const klee::ref<klee::Expr> &e);
    static bool isReceiveRead(const klee::ref<klee::Expr> &e);

    sigc::signal<void, S2EExecutionState *, const ModuleDescriptor &> onRandomInputFork;

private:
    void unmergeSelects(S2EExecutionState *state, const klee::Assignment &assignment);

    void onStateFork(S2EExecutionState *state, const std::vector<S2EExecutionState *> &newStates,
                     const std::vector<klee::ref<klee::Expr>> &newConditions);

    void onRandom(S2EExecutionState *state, uint64_t pid, const std::vector<klee::ref<klee::Expr>> &data);

    void onWrite(S2EExecutionState *state, uint64_t pid, uint64_t fd, const std::vector<klee::ref<klee::Expr>> &data,
                 klee::ref<klee::Expr> sizeExpr);

    void onSymbolicRead(S2EExecutionState *state, uint64_t pid, uint64_t fd, uint64_t size,
                        const std::vector<std::pair<std::vector<klee::ref<klee::Expr>>, std::string>> &data,
                        klee::ref<klee::Expr> sizeExpr);

    void onConcreteRead(S2EExecutionState *state, uint64_t pid, uint64_t fd, const std::vector<uint8_t> &data);

    void generateNegotiate(std::stringstream &ss, const PovOptions &opt);
    void generateReadSecret(std::stringstream &ss, const PovOptions &opt);
};
} // namespace pov
} // namespace plugins
} // namespace s2e

#endif
