# Path joiner and normalizer #

*PathJoin* is PHP library for normalizing and joining file system paths. The
purpose of this library is to make easier to work with file system paths
irregardless of the platform and the system directory separator.

The purpose of file path normalization is to provide a single consistent file
path representation. In other words, the normalization in this library will
resolve `.` and `..` directory references and also condense multiple directory
separators into one. This makes it much easier to avoid common problems when
comparing paths against each other.

While PHP provides a built in function `realpath()`, it is not usable in every
case since it works by using the file system. This library simply combines and
normalizes the paths using string handling. There is no requirement for the
files or directories to be readable or even exist.

The API documentation is available at: http://kit.riimu.net/api/pathjoin/

[![Travis](https://img.shields.io/travis/Riimu/Kit-PathJoin.svg?style=flat-square)](https://travis-ci.org/Riimu/Kit-PathJoin)
[![Scrutinizer](https://img.shields.io/scrutinizer/g/Riimu/Kit-PathJoin.svg?style=flat-square)](https://scrutinizer-ci.com/g/Riimu/Kit-PathJoin/)
[![Scrutinizer Coverage](https://img.shields.io/scrutinizer/coverage/g/Riimu/Kit-PathJoin.svg?style=flat-square)](https://scrutinizer-ci.com/g/Riimu/Kit-PathJoin/)
[![Packagist](https://img.shields.io/packagist/v/riimu/kit-pathjoin.svg?style=flat-square)](https://packagist.org/packages/riimu/kit-pathjoin)

## Requirements ##

  * The minimum supported PHP version is 5.6

## Installation ##

### Installation with Composer ###

The easiest way to install this library is to use Composer to handle your
dependencies. In order to install this library via Composer, simply follow
these two steps:

  1. Acquire the `composer.phar` by running the Composer
     [Command-line installation](https://getcomposer.org/download/)
     in your project root.

  2. Once you have run the installation script, you should have the `composer.phar`
     file in you project root and you can run the following command:

     ```
     php composer.phar require "riimu/kit-pathjoin:^1.2"
     ```

After installing this library via Composer, you can load the library by
including the `vendor/autoload.php` file that was generated by Composer during
the installation.

### Adding the library as a dependency ###

If you are already familiar with how to use Composer, you may alternatively add
the library as a dependency by adding the following `composer.json` file to your
project and running the `composer install` command:

```json
{
    "require": {
        "riimu/kit-pathjoin": "^1.2"
    }
}
```

### Manual installation ###

If you do not wish to use Composer to load the library, you may also download
the library manually by downloading the [latest release](https://github.com/Riimu/Kit-PathJoin/releases/latest)
and extracting the `src` folder to your project. You may then include the
provided `src/autoload.php` file to load the library classes.

## Usage ##

This library provides two convenient methods, `Path::normalize()` and
`Path::join()`. Both of these methods work in a very similar fashion. The main
difference is that while the `join()` method can accept multiple paths to join,
the `normalize()` will only accept a single path. Both of the methods will
return a normalized path as the result.

The following example will contain numerous different use cases of the library:

```php
<?php

require 'vendor/autoload.php';
use Riimu\Kit\PathJoin\Path;

// Both of the following will output 'foo/bar' on Unix and 'foo\bar' on Windows
echo Path::normalize('foo/bar') . PHP_EOL;
echo Path::join('foo', 'bar') . PHP_EOL;

// The join method accepts multiple arguments or a single array
echo Path::join('foo', 'bar', 'baz') . PHP_EOL;   // outputs 'foo/bar/baz'
echo Path::join(['foo', 'bar', 'baz']) . PHP_EOL; // outputs 'foo/bar/baz'

// The '.' and '..' directory references will be resolved in the paths
echo Path::normalize('foo/./bar/../baz') . PHP_EOL;     // outputs 'foo/baz'
echo Path::join(['foo/./', 'bar', '../baz']) . PHP_EOL; // outputs 'foo/baz'

// Only the first path can denote an absolute path in the join method
echo Path::join('/foo', '/bar/baz') . PHP_EOL;     // outputs '/foo/bar/baz'
echo Path::join('foo', '/bar') . PHP_EOL;          // outputs 'foo/bar'
echo Path::join('foo', '../bar', 'baz') . PHP_EOL; // outputs 'bar/baz'
echo Path::join('', '/bar', 'baz') . PHP_EOL;      // outputs 'bar/baz'

// Relative paths can start with a '..', but absolute paths cannot
echo Path::join('/foo', '../../bar', 'baz') . PHP_EOL; // outputs '/bar/baz'
echo Path::join('foo', '../../bar', 'baz') . PHP_EOL;  // outputs '../bar/baz'

// Empty paths will result in a '.'
echo Path::normalize('foo/..') . PHP_EOL;
echo Path::join('foo', 'bar', '../..') . PHP_EOL;
```

The `Path::normalize()` also accepts a second parameter `$prependDrive` that
takes a boolean value and defaults to true. On Windows platforms, the drive
letter is important part of the absolute path. Thus, when the parameter is set
to true, the method will prepend the drive letter of the current working
directory to absolute paths if the absolute path does not provide one itself.

The following example is true for Windows systems, if the working directory is
located on the C: drive:

```php
<?php

require 'vendor/autoload.php';
use Riimu\Kit\PathJoin\Path;

echo Path::normalize('/foo/bar') . PHP_EOL;        // outputs 'C:\foo\Bar'
echo Path::normalize('D:/foo/bar') . PHP_EOL;      // outputs 'D:\foo\Bar'
echo Path::normalize('/foo/bar', false) . PHP_EOL; // outputs '\foo\Bar'
```

## Credits ##

This library is Copyright (c) 2014-2017 Riikka Kalliomäki.

See LICENSE for license and copying information.
