package de.ullisroboterseite.UrsAI2UDPv3;

// Autor: https://UllisRoboterSeite.de
// Doku:  https://UllisRoboterSeite.de/android-AI2-UDP.html
// Created: 2018-01-25
//
// Version 1 (2018-01-25)
// -------------------------
// - Basis-Version
//
// Version 2 (2018-12-08)
// -------------------------
// Seit Dezember 2018 gab es Probleme mit Netzwerkzugriffen. Diese Zugriffe müssen nun in separaten Threads ausgeführt werden.
// Die Komponente wurde entsprechend umgeschrieben.
// Zusätzlich wurde die Eigenschaft InitErr eingeführt, über die Fehler bei der Initialisierung der Komponente abgerufen werden können.
//
// Version 3 (2019-01-10)
// -------------------------
// - aufgeräumt und neu strukturiert
//
// Version 3.1 (2019-04-19)
// -------------------------
// - BinaryMode hinzugefügt
//
// Version 3.2 (2020-02-17)
// -------------------------
// - DropOwnBroadcast wurde nach Start des Listeners nicht weitergereicht.
// - Die Bestimmung der Local-Host-IP per Enumration über die Netzwerkinterfaces führt zu Problemen
//   bei Endgeräten, die mehr als ein Interface besitzen. DropOwnBroadcast hat nicht funktioniert.
//   DropOwnBroadcast vergleicht nun die Absender-Adresse mit allen vorhandenen NIC-Adressen.
//   LocalHost ist die Adresse, mit der das Internet erreicht wird (google, 8.8.8.8).
//

import com.google.appinventor.components.annotations.*;
import com.google.appinventor.components.common.PropertyTypeConstants;
import com.google.appinventor.components.runtime.*;
import com.google.appinventor.components.runtime.util.*;

import java.net.*;
import android.os.Handler;

@DesignerComponent(version = 3, //
        versionName = "3.2", //
        description = "AI2 extension block for UDP communication.", //
        category = com.google.appinventor.components.common.ComponentCategory.EXTENSION, //
        nonVisible = true, //
        helpUrl = "http://UllisRoboterSeite.de/android-AI2-UDP.html", //
        iconName = "aiwebres/udp.png")
@SimpleObject(external = true)
@UsesPermissions(permissionNames = "android.permission.INTERNET,android.permission.WAKE_LOCK,android.permission.ACCESS_NETWORK_STATE")
public class UrsAI2UDPv3 extends AndroidNonvisibleComponent implements Component, UrsReceiverEventListener {
    public static final int VERSION = 3;
    private volatile boolean _DropOwnBroadcast = true; // Datagramme von der eigenen IP sollen unterdrückt werden
    private volatile boolean _BinaryMode = false; // Die Nachricht enthält Code mit Byte-Daten
    private volatile String errMsg = ""; // letzte Fehlermeldung

    private volatile boolean isStopping = false; // Stopp-Vorgang läuft
    private volatile boolean isStarting = false; // Start-Vorgang läuft
    private volatile boolean isXmitting = false; // Anzahl Xmit-Vorgang läuft

    final UrsAI2UDPv3 thisInstance = this;

    public UrsAI2UDPv3(ComponentContainer container) {
        super(container.$form());
        UrsUdpReceiver.addListener(this);
    } // ctor

    // #region Properties
    @SimpleProperty(description = "true: (Broadcast) Messages sent to yourself are dropped.")
    public boolean DropSentToYourself() {
        return _DropOwnBroadcast;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_BOOLEAN, defaultValue = "True")
    @SimpleProperty(description = "true: (Broadcast) sent to yourself are dropped.")
    public void DropSentToYourself(boolean value) {
        _DropOwnBroadcast = value;
        UrsUdpReceiver.setDropOwnbroadcast(value);
    }

    @SimpleProperty(description = "true: Binary data expected.")
    public boolean BinaryMode() {
        return _BinaryMode;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_BOOLEAN, defaultValue = "False")
    @SimpleProperty(description = "true: Binary data expected.")
    public void BinaryMode(boolean value) {
        _BinaryMode = value;
    }

    @SimpleProperty(description = "Returns a text message about the last error.")
    public String LastErrorMessage() {
        return errMsg;
    }

    @SimpleProperty(description = "true: UDP server is running.")
    public boolean IsRunning() {
        return UrsUdpReceiver.IsRunning();
    }
    // #endregion

    // #region Events
    @SimpleEvent(description = "A datagram has been received.")
    public void DataReceived(String Data, String RemoteIP, int RemotePort) {
        EventDispatcher.dispatchEvent(this, "DataReceived", Data, RemoteIP, RemotePort);
    }

    @Override
    @SimpleEvent(description = "The UDP listening server was startet.")
    public void ServerStarted(final String LocalIP, final int LocalPort) {
        handler.post(new Runnable() {
            public void run() {
                EventDispatcher.dispatchEvent(thisInstance, "ServerStarted", LocalIP, LocalPort);
            } // run
        }); // post
    }

    @SimpleEvent(description = "The UDP listening server has stopped.")
    public void ServerStopped() {
        EventDispatcher.dispatchEvent(this, "ServerStopped");
    }
    // #endregion

    @SimpleFunction(description = "Returns the IP address of the local host.")
    public String LocalHost(String Default) {
        String lh = UrsUdpHelper.getLocalHostString();
        if (lh.isEmpty())
            return Default;
        else
            return lh;
    }

    @SimpleFunction(description = "Returns the IPv4 addresses of all known network interfaces.")
    public YailList getNICList() {
        return YailList.makeList(UrsUdpHelper.getNICNames());
    }

    // Returncodes sind:
    // 0: erfolgreicher Versand
    // 1: IP-Adressangabe kann nicht in IPv4 umgewandelt werden
    // (s. JavaInetAddress.getByName)
    // 2: Unter LocalPort kann kein Socket angelegt werden
    // 3: Fehler beim Versenden
    // 4: String kann nicht konvertiert werden
    @SimpleFunction(description = "Send a datagram")
    public int Xmit(String RemoteIP, int RemotePort, int LocalPort, String Message) {
        byte[] bytes; // Daten

        while (isXmitting) // ggf warten bis vorhergehender Xmit abgeschlossen
            ;
        isXmitting = true;
        while (isStarting) // Ggf. auf Abschluss des Start-Vorgangs warten
            ;
        while (isStopping) // Ggf. auf Abdschluss des Stopp-Vorgangs warten
            ;

        errMsg = ""; // Alte Fehlermeldung löschen

        if (_BinaryMode) {
            // Input konvertieren

            try {
                bytes = StringToBytes(Message);
            } catch (Exception e) {
                errMsg = "Cannot convert the input";
                isXmitting = false;
                return 4;
            }
        } else {
            bytes = Message.getBytes();
        }

        int rtc = UrsUdpXmitter.Xmit(RemoteIP, RemotePort, LocalPort, bytes, UrsUdpReceiver.ListenSocket);
        if (rtc != 0) {
            errMsg = UrsUdpXmitter.errMsg;
        }
        isXmitting = false;
        return rtc;
    } // Xmit

    byte[] StringToBytes(String inp) {
        inp = inp.replace(',', ';');
        String[] bs = inp.split(";");
        byte[] bytes = new byte[bs.length];

        for (int i = 0; i < bs.length; i++) {
            int b = Integer.decode(bs[i].trim());
            if (b > 255 || b < 0)
                throw new NumberFormatException();
            bytes[i] = (byte) b;
        }
        return bytes;
    }

    // Listener starten

    // Erstellt den Listener-Thread und startet ihn
    // Falls in Listener-Thread aktiv ist, wird er vorher terminiert
    // Returncodes sind:
    // 0: erfolgreicher Start
    // 1: DatagrammSocket konnte nicht angelegt werden
    // 2: Startvorgang bevor vorhergehender Start abgeschlossen wurde
    // 3: Lokaler Host konnte nicht ermittelt werden.
    @SimpleFunction(description = "Start the UDP listening server.")
    public int StartListening(int LocalPort) {
        errMsg = "";
        if (isStarting) { // Es läuft bereits ein Start-Vorgang
            errMsg = "Another start process is active";
            return 2;
        }
        isStarting = true; // Start-Vorgang begonnen
        while (isStopping) // Ggf. warten bis Stopp-Vorgang beendet ist
            ;
        while (isXmitting) // Ggf. warten bis Xmit beendet
            ;
        InetAddress localHost = UrsUdpHelper.getLocalHost();
        if (localHost == null)
            return 3;
        int rtc = UrsUdpReceiver.BeginListening(localHost, LocalPort, _DropOwnBroadcast);
        isStarting = false; // Start-Vorgang beendet
        if (rtc > 0)
            errMsg = "No acces to local port: " + LocalPort + ". Port in use?";
        return rtc;
    } // StartListening

    @SimpleFunction(description = "Stops the UDP listening server.")
    public void StopListening() {
        if (isStopping) // Es läuft bereits ein Stopp-Vorgang
            return;
        isStopping = true; // Stopp-Vorgang gestartet
        while (isStarting) // Ggf. warten bis der Startvorgang beendet ist
            ;
        while (isXmitting) // Ggf. warten bis Xmit beendet
            ;
        UrsUdpReceiver.StopListening();
        isStopping = false; // Stopp-Vorgang beendet
    } // StopListening

    // #region Interface UrsReceiverEventListener
    final Handler handler = new Handler();

    @Override
    public void PacketReceived(DatagramPacket packet) {
        final String RemoteIP = packet.getAddress().getHostAddress();
        final int RemotePort = packet.getPort();

        String result;

        if (_BinaryMode) {
            byte[] d = packet.getData();
            result = "";
            for (int i = 0; i < d.length; i++) {
                int temp = d[i];
                if (temp < 0)
                    temp += 256;
                result += ";" + temp;
            }
            if (result.length() > 0)
                result = result.substring(1);
        } else {
            result = new String(packet.getData(), 0, packet.getLength());
        }

        final String received = result;

        handler.post(new Runnable() {
            public void run() {
                DataReceived(received, RemoteIP, RemotePort);
            } // run
        }); // post
    } // PacketReceived

    @Override
    public void ListenerThreadStopped(int ErrorCode) {
        handler.post(new Runnable() {
            public void run() {
                ServerStopped();
            } // run
        }); // post
    } // ListenerThreadStopped

    // DEBUG
/*
    @Override
    @SimpleEvent(description = "Returns Debugger Infos.")
    public void DebuggerLog(final String Source, final String Msg) {
        handler.post(new Runnable() { // Ggf. wird dieses Ereignis in einem Threadausgelöst.
            public void run() {
                EventDispatcher.dispatchEvent(thisInstance, "DebuggerLog", Source, Msg);
            }
        });

    }
    */
    // #region
} // class UrsAI2UDP