﻿/*
 * MemorySharp Library
 * http://www.binarysharp.com/
 *
 * Copyright (C) 2012-2016 Jämes Ménétrey (a.k.a. ZenLulz).
 * This library is released under the MIT License.
 * See the file LICENSE for more information.
*/

using System.Runtime.InteropServices;
using Binarysharp.MSharp.Internals;

namespace Binarysharp.MSharp.Native
{
    #region ClientIdStruct
    /// <summary>
    /// A structure containing the process and thread identifier.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct ClientIdStruct
    {
        /// <summary>
        /// The process identifier.
        /// </summary>
        public IntPtr UniqueProcess;

        /// <summary>
        /// The thread identifier.
        /// </summary>
        public IntPtr UniqueThread;
    }
    #endregion ClientIdStruct

    #region FlashInfo
    /// <summary>
    /// Contains the flash status for a window and the number of times the system should flash the window.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct FlashInfo
    {
        public int Size;
        public IntPtr Hwnd;
        public FlashWindowFlags Flags;
        public uint Count;
        public int Timeout;
    }
    #endregion FlashInfo

    #region HardwareInput
    /// <summary>
    /// Contains information about a simulated message generated by an input device other than a keyboard or mouse.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct HardwareInput
    {
        /// <summary>
        /// The message generated by the input hardware.
        /// </summary>
        public int Message;

        /// <summary>
        /// The low-order word of the lParam parameter for <see cref="Message" />.
        /// </summary>
        public short WParamL;

        /// <summary>
        /// The high-order word of the lParam parameter for <see cref="Message" />.
        /// </summary>
        public short WParamH;
    }
    #endregion HardwareInput

    #region Input
    /// <summary>
    /// Used by <see cref="NativeMethods.SendInput" /> to store information for synthesizing input events such as keystrokes,
    /// mouse movement, and mouse clicks.
    /// </summary>
    [StructLayout(LayoutKind.Explicit)]
    public struct Input
    {
        /// <summary>
        /// Constructor that specifies a type.
        /// </summary>
        /// <param name="type">The type if the input event.</param>
        public Input(InputTypes type) : this()
        {
            Type = type;
        }

        /// <summary>
        /// The type of the input event.
        /// </summary>
        [FieldOffset(0)] public InputTypes Type;

        /// <summary>
        /// The information about a simulated mouse event.
        /// </summary>
        [FieldOffset(sizeof(int))] public MouseInput Mouse;

        /// <summary>
        /// The information about a simulated keyboard event.
        /// </summary>
        [FieldOffset(sizeof(int))] public KeyboardInput Keyboard;

        /// <summary>
        /// The information about a simulated hardware event.
        /// </summary>
        [FieldOffset(sizeof(int))] public HardwareInput Hardware;
    }
    #endregion Input

    #region KeyboardInput
    /// <summary>
    /// Contains information about a simulated keyboard event.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct KeyboardInput
    {
        /// <summary>
        /// A virtual-key code. The code must be a value in the range 1 to 254. If the <see cref="Flags" /> member specifies
        /// KEYEVENTF_UNICODE, wVk must be 0.
        /// </summary>
        public Keys VirtualKey;

        /// <summary>
        /// A hardware scan code for the key.
        /// If <see cref="Flags" /> specifies KEYEVENTF_UNICODE, wScan specifies a Unicode character which is to be sent to the
        /// foreground application.
        /// </summary>
        public short ScanCode;

        /// <summary>
        /// Specifies various aspects of a keystroke.
        /// </summary>
        public KeyboardFlags Flags;

        /// <summary>
        /// The time stamp for the event, in milliseconds. If this parameter is zero, the system will provide its own time stamp.
        /// </summary>
        public int Time;

        /// <summary>
        /// An additional value associated with the keystroke. Use the GetMessageExtraInfo function to obtain this information.
        /// </summary>
        public IntPtr ExtraInfo;
    }
    #endregion KeyboardInput

    #region LdtEntry
    /// <summary>
    /// Describes an entry in the descriptor table. This structure is valid only on x86-based systems.
    /// </summary>
    /// <remarks>This is a simplified version of the original structure.</remarks>
    [StructLayout(LayoutKind.Sequential)]
    public struct LdtEntry
    {
        /// <summary>
        /// The low-order part of the address of the last byte in the segment.
        /// </summary>
        public ushort LimitLow;

        /// <summary>
        /// The low-order part of the base address of the segment.
        /// </summary>
        public ushort BaseLow;

        /// <summary>
        /// Middle bits (16–23) of the base address of the segment.
        /// </summary>
        public byte BaseMid;

        /// <summary>
        /// Values of the Type, Dpl, and Pres members in the Bits structure (not implemented).
        /// </summary>
        public byte Flag1;

        /// <summary>
        /// Values of the LimitHi, Sys, Reserved_0, Default_Big, and Granularity members in the Bits structure (not implemented).
        /// </summary>
        public byte Flag2;

        /// <summary>
        /// High bits (24–31) of the base address of the segment.
        /// </summary>
        public byte BaseHi;
    }
    #endregion LdtEntry

    #region M128A
    /// <summary>
    /// A structure that represents an integer with a size of 16 bytes.
    /// </summary>
    [StructLayout(LayoutKind.Explicit, Size = M128AMetaData.TotalSize)]
    public struct M128A
    {
        /// <summary>
        /// The low part.
        /// </summary>
        [FieldOffset(M128AMetaData.Offsets.Low)] public ulong Low;

        /// <summary>
        /// The high part.
        /// </summary>
        [FieldOffset(M128AMetaData.Offsets.High)] public long High;
    }
    #endregion M128A

    #region MemoryBasicInformation
    /// <summary>
    /// Contains information about a range of pages in the virtual address space of a process. The VirtualQuery and
    /// <see cref="NativeMethods.VirtualQueryEx" /> functions use this structure.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct MemoryBasicInformation
    {
        /// <summary>
        /// A pointer to the base address of the region of pages.
        /// </summary>
        public IntPtr BaseAddress;

        /// <summary>
        /// A pointer to the base address of a range of pages allocated by the VirtualAlloc function. The page pointed to by the
        /// BaseAddress member is contained within this allocation range.
        /// </summary>
        public IntPtr AllocationBase;

        /// <summary>
        /// The memory protection option when the region was initially allocated. This member can be one of the memory protection
        /// constants or 0 if the caller does not have access.
        /// </summary>
        public MemoryProtectionFlags AllocationProtect;

        /// <summary>
        /// The size of the region beginning at the base address in which all pages have identical attributes, in bytes.
        /// </summary>
        public IntPtr RegionSize;

        /// <summary>
        /// The state of the pages in the region.
        /// </summary>
        public MemoryStateFlags State;

        /// <summary>
        /// The access protection of the pages in the region. This member is one of the values listed for the AllocationProtect
        /// member.
        /// </summary>
        public MemoryProtectionFlags Protect;

        /// <summary>
        /// The type of pages in the region.
        /// </summary>
        public MemoryTypeFlags Type;
    }
    #endregion MemoryBasicInformation

    #region MouseInput
    /// <summary>
    /// Contains information about a simulated mouse event.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct MouseInput
    {
        /// <summary>
        /// The absolute position of the mouse, or the amount of motion since the last mouse event was generated, depending on the
        /// value of the <see cref="Flags" /> member.
        /// Absolute data is specified as the x coordinate of the mouse; relative data is specified as the number of pixels moved.
        /// </summary>
        public int DeltaX;

        /// <summary>
        /// The absolute position of the mouse, or the amount of motion since the last mouse event was generated, depending on the
        /// value of the <see cref="Flags" /> member.
        /// Absolute data is specified as the y coordinate of the mouse; relative data is specified as the number of pixels moved.
        /// </summary>
        public int DeltaY;

        /// <summary>
        /// If <see cref="Flags" /> contains <see cref="MouseFlags.Wheel" />, then mouseData specifies the amount of wheel
        /// movement.
        /// A positive value indicates that the wheel was rotated forward, away from the user; a negative value indicates that the
        /// wheel was rotated backward, toward the user.
        /// One wheel click is defined as WHEEL_DELTA, which is 120.
        /// Windows Vista: If dwFlags contains <see cref="MouseFlags.HWheel" />, then dwData specifies the amount of wheel
        /// movement.
        /// A positive value indicates that the wheel was rotated to the right; a negative value indicates that the wheel was
        /// rotated to the left.
        /// One wheel click is defined as WHEEL_DELTA, which is 120.
        /// If dwFlags does not contain <see cref="MouseFlags.Wheel" />, <see cref="MouseFlags.XDown" />, or
        /// <see cref="MouseFlags.XUp" />, then mouseData should be zero.
        /// If dwFlags contains <see cref="MouseFlags.XDown" /> or <see cref="MouseFlags.XUp" />, then mouseData specifies which X
        /// buttons were pressed or released.
        /// This value may be any combination of the following flags.
        /// XBUTTON1 = 0x1
        /// XBUTTON2 = 0x2
        /// </summary>
        public int MouseData;

        /// <summary>
        /// A set of bit flags that specify various aspects of mouse motion and button clicks.
        /// The bits in this member can be any reasonable combination of the following values.
        /// The bit flags that specify mouse button status are set to indicate changes in status, not ongoing conditions.
        /// For example, if the left mouse button is pressed and held down, <see cref="MouseFlags.LeftDown" /> is set when the left
        /// button is first pressed, but not for subsequent motions. Similarly, <see cref="MouseFlags.LeftUp" /> is set only when
        /// the button is first released.
        /// You cannot specify both the <see cref="MouseFlags.Wheel" /> flag and either <see cref="MouseFlags.XDown" /> or
        /// <see cref="MouseFlags.XUp" /> flags
        /// simultaneously in the dwFlags parameter, because they both require use of the mouseData field.
        /// </summary>
        public MouseFlags Flags;

        /// <summary>
        /// The time stamp for the event, in milliseconds. If this parameter is 0, the system will provide its own time stamp.
        /// </summary>
        public int Time;

        /// <summary>
        /// An additional value associated with the mouse event. An application calls GetMessageExtraInfo to obtain this extra
        /// information.
        /// </summary>
        public IntPtr ExtraInfo;
    }
    #endregion MouseInput

    #region Point
    /// <summary>
    /// The <see cref="Point" /> structure defines the x and y coordinates of a point.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct Point
    {
        /// <summary>
        /// The x-coordinate of the point.
        /// </summary>
        public int X;

        /// <summary>
        /// The y-coordinate of the point.
        /// </summary>
        public int Y;

        /// <summary>
        /// Returns a string that represents the current object.
        /// </summary>
        public override string ToString()
        {
            return string.Format("X = {0} Y = {1}", X, Y);
        }
    }
    #endregion Point

    #region ProcessBasicInformation
    /// <summary>
    /// Structure containing basic information about a process.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct ProcessBasicInformation
    {
        /// <summary>
        /// The exit status.
        /// </summary>
        public IntPtr ExitStatus;

        /// <summary>
        /// The base address of Process Environment Block.
        /// </summary>
        public IntPtr PebBaseAddress;

        /// <summary>
        /// The affinity mask.
        /// </summary>
        public IntPtr AffinityMask;

        /// <summary>
        /// The base priority.
        /// </summary>
        public IntPtr BasePriority;

        /// <summary>
        /// The process id.
        /// </summary>
        public IntPtr ProcessId;

        /// <summary>
        /// The process id of the parent process.
        /// </summary>
        public IntPtr InheritedFromUniqueProcessId;

        /// <summary>
        /// The size of this structure.
        /// </summary>
        public int Size => MarshalType<ProcessBasicInformation>.Size;
    }
    #endregion ProcessBasicInformation

    #region ThreadBasicInformation
    /// <summary>
    /// Structure containing basic information about a thread.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct ThreadBasicInformation
    {
        /// <summary>
        /// the exit status (NTSTATUS).
        /// </summary>
        public uint ExitStatus;

        /// <summary>
        /// The base address of Thread Environment Block.
        /// </summary>
        public IntPtr TebBaseAddress;

        /// <summary>
        /// The process and thread identifiers.
        /// </summary>
        public ClientIdStruct ClientIdStruct;

        /// <summary>
        /// The affinity mask.
        /// </summary>
        public IntPtr AffinityMask;

        /// <summary>
        /// The priority.
        /// </summary>
        public int Priority;

        /// <summary>
        /// The base priority.
        /// </summary>
        public int BasePriority;
    }
    #endregion ThreadBasicInformation

    #region ThreadContext32
    /// <summary>
    /// Represents a thread context for 32-bit processes on Windows. Create a new instance using the constructor with the
    /// flags.
    /// </summary>
    /// <remarks>
    /// This structure is blittable and therefore, does not require any marshaling from Platform Invoke.
    /// Referenced on https://docs.microsoft.com/en-us/windows/desktop/api/winnt/ns-winnt-_wow64_context.
    /// Refer to the header file WinNT.h for definitions of this structure for each processor architecture.
    /// </remarks>
    [StructLayout(LayoutKind.Explicit, Size = ThreadContext32Metadata.TotalSize)]
    public unsafe struct ThreadContext32
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="ThreadContext32" /> struct.
        /// </summary>
        /// <param name="flags">Determines which registers are returned or set during the context query.</param>
        public ThreadContext32(ThreadContextFlags flags) : this()
        {
            ContextFlags = flags;
        }

        /// <summary>
        /// Determines which registers are returned or set when using <see cref="NativeMethods.GetThreadContext" /> or
        /// <see cref="NativeMethods.SetThreadContext" />.
        /// If the context record is used as an INPUT parameter, then for each portion of the context record controlled by a flag
        /// whose value is set, it is assumed that portion of the
        /// context record contains valid context. If the context record is being used to modify a threads context, then only that
        /// portion of the threads context will be modified.
        /// If the context record is used as an INPUT/OUTPUT parameter to capture the context of a thread, then only those portions
        /// of the thread's context corresponding to set flags will be returned.
        /// The context record is never used as an OUTPUT only parameter.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.ContextFlags)]
        public ThreadContextFlags ContextFlags;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.DebugRegisters" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Dr0)]
        public uint Dr0;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.DebugRegisters" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Dr1)]
        public uint Dr1;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.DebugRegisters" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Dr2)]
        public uint Dr2;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.DebugRegisters" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Dr3)]
        public uint Dr3;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.DebugRegisters" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Dr6)]
        public uint Dr6;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.DebugRegisters" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Dr7)]
        public uint Dr7;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.ControlWord)]
        public uint ControlWord;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.StatusWord)]
        public uint StatusWord;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.TagWord)]
        public uint TagWord;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.ErrorOffset)]
        public uint ErrorOffset;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.ErrorSelector)]
        public uint ErrorSelector;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.DataOffset)]
        public uint DataOffset;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.DataSelector)]
        public uint DataSelector;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.RegisterArea)]
        public fixed byte RegisterArea[ThreadContext32Metadata.Sizes.RegisterAreaSize];

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.FloatingPoint" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Cr0NpxState)]
        public uint Cr0NpxState;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Segments" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.SegGs)]
        public uint SegGs;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Segments" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.SegFs)]
        public uint SegFs;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Segments" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.SegEs)]
        public uint SegEs;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Segments" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.SegDs)]
        public uint SegDs;

        /// <summary>
        /// This register is specified/returned if the ContextFlags word contains the flag
        /// <see cref="ThreadContextFlags.Integer" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Edi)]
        public uint Edi;

        /// <summary>
        /// This register is specified/returned if the ContextFlags word contains the flag
        /// <see cref="ThreadContextFlags.Integer" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Esi)]
        public uint Esi;

        /// <summary>
        /// This register is specified/returned if the ContextFlags word contains the flag
        /// <see cref="ThreadContextFlags.Integer" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Ebx)]
        public uint Ebx;

        /// <summary>
        /// This register is specified/returned if the ContextFlags word contains the flag
        /// <see cref="ThreadContextFlags.Integer" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Edx)]
        public uint Edx;

        /// <summary>
        /// This register is specified/returned if the ContextFlags word contains the flag
        /// <see cref="ThreadContextFlags.Integer" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Ecx)]
        public uint Ecx;

        /// <summary>
        /// This register is specified/returned if the ContextFlags word contains the flag
        /// <see cref="ThreadContextFlags.Integer" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Eax)]
        public uint Eax;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Control" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Ebp)]
        public uint Ebp;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Control" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Eip)]
        public uint Eip;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Control" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.SegCs)]
        public uint SegCs;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Control" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.EFlags)]
        public uint EFlags;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Control" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.Esp)]
        public uint Esp;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag <see cref="ThreadContextFlags.Control" />.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.SegSs)]
        public uint SegSs;

        /// <summary>
        /// This is specified/returned if <see cref="ContextFlags" /> contains the flag
        /// <see cref="ThreadContextFlags.ExtendedRegisters" />.
        /// The format and contexts are processor specific.
        /// </summary>
        [FieldOffset(ThreadContext32Metadata.Offsets.ExtendedRegisters)]
        public fixed byte ExtendedRegisters[ThreadContext32Metadata.Sizes.ExtendedRegistersSize];
    }
    #endregion ThreadContext32

    #region ThreadContext64
    /// <summary>
    /// Represents a thread context for 64-bit processes on Windows. Create a new instance using the constructor with the
    /// flags.
    /// </summary>
    /// <remarks>
    /// This structure is blittable and therefore, does not require any marshaling from Platform Invoke.
    /// Referenced on https://docs.microsoft.com/en-us/windows/desktop/api/winnt/ns-winnt-_context.
    /// Refer to the header file WinNT.h for definitions of this structure for each processor architecture.
    /// </remarks>
    [StructLayout(LayoutKind.Explicit, Size = ThreadContext64Metadata.TotalSize)]
    public unsafe struct ThreadContext64
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="ThreadContext64" /> struct.
        /// </summary>
        /// <param name="flags">Determines which registers are returned or set during the context query.</param>
        public ThreadContext64(ThreadContextFlags flags) : this()
        {
            ContextFlags = flags;
        }

        [FieldOffset(ThreadContext64Metadata.Offsets.P1Home)]
        public ulong P1Home;

        [FieldOffset(ThreadContext64Metadata.Offsets.P2Home)]
        public ulong P2Home;

        [FieldOffset(ThreadContext64Metadata.Offsets.P3Home)]
        public ulong P3Home;

        [FieldOffset(ThreadContext64Metadata.Offsets.P4Home)]
        public ulong P4Home;

        [FieldOffset(ThreadContext64Metadata.Offsets.P5Home)]
        public ulong P5Home;

        [FieldOffset(ThreadContext64Metadata.Offsets.P6Home)]
        public ulong P6Home;

        /// <summary>
        /// Determines which registers are returned or set when using <see cref="NativeMethods.GetThreadContext" /> or
        /// <see cref="NativeMethods.SetThreadContext" />.
        /// If the context record is used as an INPUT parameter, then for each portion of the context record controlled by a flag
        /// whose value is set, it is assumed that portion of the
        /// context record contains valid context. If the context record is being used to modify a threads context, then only that
        /// portion of the threads context will be modified.
        /// If the context record is used as an INPUT/OUTPUT parameter to capture the context of a thread, then only those portions
        /// of the thread's context corresponding to set flags will be returned.
        /// The context record is never used as an OUTPUT only parameter.
        /// </summary>
        [FieldOffset(ThreadContext64Metadata.Offsets.ContextFlags)]
        public ThreadContextFlags ContextFlags;

        [FieldOffset(ThreadContext64Metadata.Offsets.MxCsr)]
        public uint MxCsr;

        [FieldOffset(ThreadContext64Metadata.Offsets.SegCs)]
        public ushort SegCs;

        [FieldOffset(ThreadContext64Metadata.Offsets.SegDs)]
        public ushort SegDs;

        [FieldOffset(ThreadContext64Metadata.Offsets.SegEs)]
        public ushort SegEs;

        [FieldOffset(ThreadContext64Metadata.Offsets.SegFs)]
        public ushort SegFs;

        [FieldOffset(ThreadContext64Metadata.Offsets.SegGs)]
        public ushort SegGs;

        [FieldOffset(ThreadContext64Metadata.Offsets.SegSs)]
        public ushort SegSs;

        [FieldOffset(ThreadContext64Metadata.Offsets.EFlags)]
        public uint EFlags;

        [FieldOffset(ThreadContext64Metadata.Offsets.Dr0)]
        public ulong Dr0;

        [FieldOffset(ThreadContext64Metadata.Offsets.Dr1)]
        public ulong Dr1;

        [FieldOffset(ThreadContext64Metadata.Offsets.Dr2)]
        public ulong Dr2;

        [FieldOffset(ThreadContext64Metadata.Offsets.Dr3)]
        public ulong Dr3;

        [FieldOffset(ThreadContext64Metadata.Offsets.Dr6)]
        public ulong Dr6;

        [FieldOffset(ThreadContext64Metadata.Offsets.Dr7)]
        public ulong Dr7;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rax)]
        public ulong Rax;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rcx)]
        public ulong Rcx;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rdx)]
        public ulong Rdx;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rbx)]
        public ulong Rbx;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rsp)]
        public ulong Rsp;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rbp)]
        public ulong Rbp;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rsi)]
        public ulong Rsi;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rdi)]
        public ulong Rdi;

        [FieldOffset(ThreadContext64Metadata.Offsets.R8)]
        public ulong R8;

        [FieldOffset(ThreadContext64Metadata.Offsets.R9)]
        public ulong R9;

        [FieldOffset(ThreadContext64Metadata.Offsets.R10)]
        public ulong R10;

        [FieldOffset(ThreadContext64Metadata.Offsets.R11)]
        public ulong R11;

        [FieldOffset(ThreadContext64Metadata.Offsets.R12)]
        public ulong R12;

        [FieldOffset(ThreadContext64Metadata.Offsets.R13)]
        public ulong R13;

        [FieldOffset(ThreadContext64Metadata.Offsets.R14)]
        public ulong R14;

        [FieldOffset(ThreadContext64Metadata.Offsets.R15)]
        public ulong R15;

        [FieldOffset(ThreadContext64Metadata.Offsets.Rip)]
        public ulong Rip;

        [FieldOffset(ThreadContext64Metadata.Offsets.Header)]
        public fixed ulong Header[ThreadContext64Metadata.Sizes.Header];

        [FieldOffset(ThreadContext64Metadata.Offsets.Legacy)]
        public fixed ulong Legacy[ThreadContext64Metadata.Sizes.Legacy];

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm0)]
        public M128A Xmm0;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm1)]
        public M128A Xmm1;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm2)]
        public M128A Xmm2;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm3)]
        public M128A Xmm3;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm4)]
        public M128A Xmm4;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm5)]
        public M128A Xmm5;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm6)]
        public M128A Xmm6;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm7)]
        public M128A Xmm7;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm8)]
        public M128A Xmm8;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm9)]
        public M128A Xmm9;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm10)]
        public M128A Xmm10;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm11)]
        public M128A Xmm11;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm12)]
        public M128A Xmm12;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm13)]
        public M128A Xmm13;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm14)]
        public M128A Xmm14;

        [FieldOffset(ThreadContext64Metadata.Offsets.Xmm15)]
        public M128A Xmm15;

        [FieldOffset(ThreadContext64Metadata.Offsets.VectorRegister)]
        public fixed ulong VectorRegister[ThreadContext64Metadata.Sizes.VectorRegister];

        [FieldOffset(ThreadContext64Metadata.Offsets.VectorControl)]
        public ulong VectorControl;

        [FieldOffset(ThreadContext64Metadata.Offsets.DebugControl)]
        public ulong DebugControl;

        [FieldOffset(ThreadContext64Metadata.Offsets.LastBranchToRip)]
        public ulong LastBranchToRip;

        [FieldOffset(ThreadContext64Metadata.Offsets.LastBranchFromRip)]
        public ulong LastBranchFromRip;

        [FieldOffset(ThreadContext64Metadata.Offsets.LastExceptionToRip)]
        public ulong LastExceptionToRip;

        [FieldOffset(ThreadContext64Metadata.Offsets.LastExceptionFromRip)]
        public ulong LastExceptionFromRip;
    }
    #endregion ThreadContext64

    #region Rectangle
    /// <summary>
    /// The <see cref="Rectangle" /> structure defines the coordinates of the upper-left and lower-right corners of a
    /// rectangle.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct Rectangle
    {
        /// <summary>
        /// The x-coordinate of the upper-left corner of the rectangle.
        /// </summary>
        public int Left;

        /// <summary>
        /// The y-coordinate of the upper-left corner of the rectangle.
        /// </summary>
        public int Top;

        /// <summary>
        /// The x-coordinate of the lower-right corner of the rectangle.
        /// </summary>
        public int Right;

        /// <summary>
        /// The y-coordinate of the lower-right corner of the rectangle.
        /// </summary>
        public int Bottom;

        /// <summary>
        /// Gets or sets the height of the element.
        /// </summary>
        public int Height
        {
            get => Bottom - Top;
            set => Bottom = Top + value;
        }

        /// <summary>
        /// Gets or sets the width of the element.
        /// </summary>
        public int Width
        {
            get => Right - Left;
            set => Right = Left + value;
        }

        /// <summary>
        /// Returns a string that represents the current object.
        /// </summary>
        public override string ToString()
        {
            return string.Format("Left = {0} Top = {1} Height = {2} Width = {3}", Left, Top, Height, Width);
        }
    }
    #endregion Rectangle

    #region WindowPlacement
    /// <summary>
    /// Contains information about the placement of a window on the screen.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct WindowPlacement
    {
        /// <summary>
        /// The length of the structure, in bytes. Before calling the GetWindowPlacement or SetWindowPlacement functions, set this
        /// member to sizeof(WINDOWPLACEMENT).
        /// </summary>
        public int Length;

        /// <summary>
        /// Specifies flags that control the position of the minimized window and the method by which the window is restored.
        /// </summary>
        public int Flags;

        /// <summary>
        /// The current show state of the window.
        /// </summary>
        public WindowStates ShowCmd;

        /// <summary>
        /// The coordinates of the window's upper-left corner when the window is minimized.
        /// </summary>
        public Point MinPosition;

        /// <summary>
        /// The coordinates of the window's upper-left corner when the window is maximized.
        /// </summary>
        public Point MaxPosition;

        /// <summary>
        /// The window's coordinates when the window is in the restored position.
        /// </summary>
        public Rectangle NormalPosition;
    }
    #endregion WindowPlacement
}