# Testing for ORM Injection

## Summary

[Object Relational Mapping (ORM) Injection](https://capec.mitre.org/data/definitions/109.html) is an attack using SQL Injection against an ORM generated data access object model. From the point of view of a tester, this attack is virtually identical to a SQL Injection attack. However, the injection vulnerability exists in code generated by the ORM layer.

The benefits of using an ORM tool include quick generation of an object layer to communicate to a relational database, standardize code templates for these objects, and that they usually provide a set of safe functions to protect against SQL Injection attacks. ORM generated objects can use SQL or in some cases, a variant of SQL, to perform CRUD (Create, Read, Update, Delete) operations on a database. It is possible, however, for a web application using ORM generated objects to be vulnerable to SQL Injection attacks if methods can accept unsanitized input parameters.

## How to Test

ORM layers can be prone to vulnerabilities, as they extend the surface of attack. Instead of directly targeting the application with SQL queries, you'd be focusing on abusing the ORM layer to send malicious SQL queries.

### Identify the ORM Layer

To efficiently test and understand what's happening between your requests and the backend queries, and as with everything related to conducting proper testing, it is essential to identify the technology being used. By following the [information gathering](../01-Information_Gathering/README.md) chapter, you should be aware of the technology being used by the application at hand. Check this [list mapping languages to their respective ORMs](https://en.wikipedia.org/wiki/List_of_object-relational_mapping_software).

### Abusing the ORM Layer

After identifying the possible ORM being used, it becomes essential to understand how its parser is functioning, and study methods to abuse it, or even maybe if the application is using an old version, identify CVEs pertaining to the library being used. Sometimes, ORM layers are not properly implemented, and thus allow for the tester to conduct normal [SQL Injection](05-Testing_for_SQL_Injection.md), without worrying about the ORM layer.

#### Weak ORM Implementation

A vulnerable scenario where the ORM layer was not implemented properly, taken from [SANS](https://software-security.sans.org/developer-how-to/fix-sql-injection-in-java-hibernate):

```java
List results = session.createQuery("from Orders as orders where orders.id = " + currentOrder.getId()).list();
List results = session.createSQLQuery("Select * from Books where author = " + book.getAuthor()).list();
```

The above didn't implement the positional parameter, which allows the developer to replace the input with a `?`. An example would be as such:

```java
Query hqlQuery = session.createQuery("from Orders as orders where orders.id = ?");
List results = hqlQuery.setString(0, "123-ADB-567-QTWYTFDL").list(); // 0 is the first position, where it is dynamically replaced by the string set
```

This implementation leaves the validation and sanitization to be done by the ORM layer, and the only way to bypass it would be by identifying an issue with the ORM layer.

#### Vulnerable ORM Layer

ORM layers are code, third-party libraries most of the time. They can be vulnerable just like any other piece of code. One example could be the [sequelize ORM npm library](https://snyk.io/blog/sequelize-orm-npm-library-found-vulnerable-to-sql-injection-attacks/) which was found to be vulnerable in 2019. In another research done by [RIPS Tech](https://www.ripstech.com/), bypasses were identified in the [hibernate ORM used by Java](https://hibernate.org/orm/).

Based on their [blog article](https://blog.ripstech.com/2020/exploiting-hibernate-injections/), a cheat sheet that could allow the tester to identify issues could be outlined as follows:

| DBMS       | SQL Injection                                                         |
|------------|-----------------------------------------------------------------------|
| MySQL      | `abc\' INTO OUTFILE --`                                               |
<!-- markdownlint-disable-next-line MD055 MD056 -->
| PostgreSQL | `$$='$$=chr(61) || chr(0x27) and 1=pg_sleep(2) || version()'`         |
| Oracle     | `NVL(TO_CHAR(DBMS_XMLGEN.getxml('select 1 where 1337>1')),'1')!='1'`  |
| MS SQL     | `1<LEN(%C2%A0(select%C2%A0top%C2%A01%C2%A0name%C2%A0from%C2%A0users)` |

Another example would include the [Laravel Query-Builder](https://laravel.com/docs/7.x/queries), which was found to be [vulnerable in 2019](https://freek.dev/1317-an-important-security-release-for-laravel-query-builder).

## References

- [Wikipedia - ORM](https://en.wikipedia.org/wiki/Object-relational_mapping)
- [New Methods for Exploiting ORM Injections in Java Applications (HITB16)](https://insinuator.net/2016/06/new-methods-for-exploiting-orm-injections-in-java-applications-hitb16/)
- [HITB2016 Slides - ORM Injections in Java Applications](https://archive.conference.hitb.org/hitbsecconf2016ams/sessions/new-methods-for-exploiting-orm-injections-in-java-applications/)
- [Fixing SQL Injection: ORM is not enough](https://snyk.io/blog/sql-injection-orm-vulnerabilities/)
- [PayloadsAllTheThings - HQL Injection](https://github.com/swisskyrepo/PayloadsAllTheThings/blob/master/SQL%20Injection/HQL%20Injection.md)
