/// ## References
///
/// (USB HID to PS/2)[https://web.archive.org/web/20030701121507/http://microsoft.com/hwdev/download/tech/input/translate.pdf]
use core::mem;

/// PS/2 Scanset2 to USB HID translator
#[derive(Default)]
pub struct Translator {
	state: State,
	release: bool,
}

#[derive(Clone, Copy, Default)]
enum State {
	#[default]
	None,
	Escape,
	LongEscape(PauseState),
}

#[derive(Clone, Copy)]
enum PauseState {
	Wait14,
	Wait77,
}

impl Translator {
	pub fn push<'a>(&mut self, byte: u8, buf: &'a mut [u8; 8]) -> Option<(bool, &'a [u8])> {
		match byte {
			0xe0 => {
				if !matches!(self.state, State::None) {
					log!("escape sequence sent when state is not none");
				}
				self.state = State::Escape;
				None
			}
			0xe1 => {
				if !matches!(self.state, State::None) {
					log!("long escape sequence sent when state is not none");
				}
				self.state = State::LongEscape(PauseState::Wait14);
				None
			}
			0xf0 => {
				if self.release && !matches!(self.state, State::LongEscape(_)) {
					log!("release already set");
				}
				self.release = true;
				None
			}
			_ => {
				let b = match self.state {
					State::None => translate_single(byte),
					State::Escape if byte == 0x12 => {
						// Just ignore, I'm done with this multi-scancode bullshit.
						// (This is prepended to print screen, which is E0 7C)
						self.state = State::None;
						return None;
					}
					State::Escape => translate_escaped(byte),
					State::LongEscape(PauseState::Wait14) if byte == 0x14 => {
						self.state = State::LongEscape(PauseState::Wait77);
						return None;
					}
					State::LongEscape(PauseState::Wait77) if byte == 0x77 => Some(0x48),
					State::LongEscape(_) => None,
				};
				if let Some(b) = b {
					buf[0] = b;
					self.state = State::None;
					Some((mem::take(&mut self.release), &buf[..1]))
				} else {
					log!("unknown sequence");
					self.state = State::None;
					self.release = false;
					None
				}
			}
		}
	}
}

macro_rules! map {
	{ [$a:ident] $($usb:literal $ps2:literal)* } => {
		Some(match $a {
			$($ps2 => $usb,)*
			_ => return None,
		})
	};
}

fn translate_single(byte: u8) -> Option<u8> {
	map! {
		[byte]
		0x4 0x1c
		0x5 0x32
		0x6 0x21
		0x7 0x23
		0x8 0x24
		0x9 0x2b
		0xa 0x34
		0xb 0x33
		0xc 0x43
		0xd 0x3b
		0xe 0x42
		0xf 0x4b
		0x10 0x3a
		0x11 0x31
		0x12 0x44
		0x13 0x4d
		0x14 0x15
		0x15 0x2d
		0x16 0x1b
		0x17 0x2c
		0x18 0x3c
		0x19 0x2a
		0x1a 0x1d
		0x1b 0x22
		0x1c 0x35
		0x1d 0x1a
		0x1e 0x16
		0x1f 0x1e
		0x20 0x26
		0x21 0x25
		0x22 0x2e
		0x23 0x36
		0x24 0x3d
		0x25 0x3e
		0x26 0x46
		0x27 0x45
		0x28 0x5a
		0x29 0x76
		0x2a 0x66
		0x2b 0x0d
		0x2c 0x29
		0x2d 0x4e
		0x2e 0x55
		0x2f 0x54
		0x30 0x5b
		0x31 0x5d
		//0x32 0x5d // bruh
		0x33 0x4c
		0x34 0x52
		0x35 0x0e
		0x36 0x41
		0x37 0x49
		0x38 0x4a
		0x39 0x58
		0x3a 0x05
		0x3b 0x06
		0x3c 0x04
		0x3d 0x0c
		0x3e 0x03
		0x3f 0x0b
		0x40 0x83
		0x41 0x0a
		0x42 0x01
		0x43 0x09
		0x44 0x78
		0x45 0x07

		0x47 0x7e

		0x53 0x77

		0x55 0x7c
		0x56 0x7b
		0x57 0x79

		0x59 0x69
		0x5a 0x72
		0x5b 0x7a
		0x5c 0x6b
		0x5d 0x73
		0x5e 0x74
		0x5f 0x6c
		0x60 0x75
		0x61 0x7d
		0x62 0x70
		0x63 0x71
		0x64 0x61

		0x67 0x0f
		0x68 0x2f
		0x69 0x37
		0x6a 0x3f

		0x85 0x6d

		0x87 0x51
		0x88 0x13
		0x89 0x6a
		0x8a 0x64
		0x8b 0x67
		0x8c 0x27

		0x90 0xf2
		0x91 0xf1
		0x92 0x63
		0x93 0x62
		0x94 0x5f

		0xe0 0x14
		0xe1 0x12
		0xe2 0x11

		0xe5 0x59
	}
}

fn translate_escaped(byte: u8) -> Option<u8> {
	map! {
		[byte]
		0x46 0x7c

		0x48 0x7e

		0x49 0x70
		0x4a 0x6c
		0x4b 0x7d
		0x4c 0x71
		0x4d 0x69
		0x4e 0x7a
		0x4f 0x74
		0x50 0x6b
		0x51 0x72
		0x52 0x75

		0x54 0x4a

		0x58 0x5a

		0x65 0x2f

		0xe3 0x1f
		0xe4 0x14

		0xe6 0x11
		0xe7 0x27

		// TODO bunch of USB HID page 0x0c stuff
	}
}
