import { Component, Inject, OnDestroy, OnInit, inject, signal } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  ReactiveFormsModule,
  Validators
} from '@angular/forms';
import { XButtonComponent } from '@ng-nest/ui/button';
import { XDialogModule, XDialogRef, X_DIALOG_DATA } from '@ng-nest/ui/dialog';
import { XInputComponent } from '@ng-nest/ui/input';
import { XLoadingComponent } from '@ng-nest/ui/loading';
import { XMessageService } from '@ng-nest/ui/message';
import { UserService } from '@ui/api';
import { Subject, finalize, tap } from 'rxjs';

@Component({
  selector: 'app-reset-password',
  imports: [
    ReactiveFormsModule,
    XLoadingComponent,
    XInputComponent,
    XButtonComponent,
    XDialogModule
  ],
  templateUrl: './reset-password.component.html'
})
export class ResetPasswordComponent implements OnInit, OnDestroy {
  dialogRef = inject(XDialogRef<ResetPasswordComponent>);
  user = inject(UserService);
  fb = inject(FormBuilder);
  message = inject(XMessageService);

  id = signal('');

  formLoading = signal(false);
  saveLoading = signal(false);

  form!: FormGroup;

  $destroy = new Subject<void>();

  constructor(@Inject(X_DIALOG_DATA) public data: { id: string; saveSuccess: () => void }) {}

  ngOnInit(): void {
    const { id } = this.data;
    this.id.set(id);
    this.form = this.fb.group({
      password: [null, [Validators.required]],
      checkPassword: [null, [Validators.required, this.confirmationValidator]]
    });
  }

  ngOnDestroy(): void {
    this.$destroy.next();
    this.$destroy.complete();
  }

  save() {
    const value = this.form.getRawValue();
    delete value.checkPassword;
    this.saveLoading.set(true);
    this.user
      .resetPassword(this.id(), value)
      .pipe(
        tap((x) => {
          this.message.success(x);
          this.dialogRef.close();
        }),
        finalize(() => this.saveLoading.set(false))
      )
      .subscribe();
  }

  confirmationValidator = (control: FormControl): { [s: string]: boolean } => {
    if (!control.value) {
      return { required: true };
    } else if (!this.passwordCheck(control.value)) {
      return { confirm: true, error: true };
    }
    return {};
  };

  passwordCheck = (value: string) => {
    return value === this.form.controls['password'].value;
  };
}
