# chapter13 占位符类型作为模板参数（例如`auto`）
自从C++17起，你可以使用占位符类型（`auto`和`decltype(auto)`）作为非类型模板参数的类型。
这意味着我们可以写出泛型代码来处理不同类型的非类型模板参数。

## 13.1 使用`auto`模板参数
自从C++17起，你可以使用`auto`来声明非类型模板参数。例如：
```cpp
template<auto N> class S {
    ...
};
```
这允许我们为不同类型实例化非类型模板参数`N`：
```cpp
S<42> s1;   // OK：S中N的类型是int
S<'a'> s2;  // OK：S中N的类型是char
```
然而，你不能使用这个特性来实例化一些不允许作为模板参数的类型：
```cpp
S<2.5> s3;  // ERROR：模板参数的类型不能是double
```
我们甚至还可以用指明类型的版本作为部分特化版：
```cpp
template<int N> class S<N> {
    ...
};
```
甚至还支持类模板参数推导。例如：
```cpp
template<typename T, auto N>
class A {
public:
    A(const std::array<T, N>&) {
    }
    A(T(&)[N]) {
    }
    ...
};
```
这个类可以推导出`T`的类型、`N`的类型、`N`的值：
```cpp
A a2{"hello"};  // OK，推导为A<const char, 6>，N的类型是std::size_t

std::array<double, 10> sa1;
A a1{sa1};      // OK，推导为A<double, 10>，N的类型是std::size_t
```
你也可以修饰`auto`，例如，可以确保参数类型必须是个指针：
```cpp
template<const auto* P> struct S;
```
另外，通过使用可变参数模板，你可以使用多个不同类型的模板参数来实例化模板：
```cpp
template<auto... VS> class HeteroValueList {
};
```
也可以用多个相同类型的参数：
```cpp
template<auto V1, decltype(V1)... VS> class HomoValueList {
};
```
例如：
```cpp
HeteroValueList<1, 2, 3> vals1;         // OK
HeteroValueList<1, 'a', true> vals2;    // OK
HomoValueList<1, 2, 3> vals3;           // OK
HomoValueList<1, 'a', true> vals4;      // ERROR
```

### 13.1.1 字符和字符串模板参数
这个特性的一个应用就是你可以定义一个既可能是字符也可能是字符串的模板参数。
例如，我们可以像下面这样改进用折叠表达式输出任意数量参数
的方法：

```cpp
#include <iostream>

template<auto Sep = ' ', typename First, typename... Args>
void print(const First& first, const Args&... args) {
    std::cout << first;
    auto outWithSep = [] (const auto& arg) {
                          std::cout << Sep << arg;
                      };
    (... , outWithSep(args));
    std::cout << '\n';
}
```

将默认的参数分隔符`Sep`设置为空格，我们可以实现和之前相同的效果：
```cpp
template<auto Sep = ' ', typename First, typename... Args>
void print(const First& firstarg, const Args&... args) {
    ...
}
```
我们仍然可以像之前一样调用：
```cpp
std::string s{"world"};
print(7.5, "hello", s);      // 打印出：7.5 hello world
```
然而，通过把分隔符`Sep`参数化，我们也可以显示指明另一个字符作为分隔符：
```cpp
print<'-'>(7.5, "hello", s); // 打印出：7.5-hello-world
```
甚至，因为使用了`auto`，我们甚至可以传递被声明为无链接
的字符串字面量作为分隔符：
```cpp
static const char sep[] = ", ";
print<sep>(7.5, "hello", s); // 打印出：7.5, hello, world
```
另外，我们也可以传递任何其他可以用作模板参数的类型：
```cpp
print<-11>(7.5, "hello", s); // 打印出：7.5-11hello-11world
```

### 13.1.2 定义元编程常量
`auto`模板参数特性的另一个应用是可以让我们更轻易的定义编译期常量。

原本的下列代码：
```cpp
template<typename T, T v>
struct constant
{
    static constexpr T value = v;
};

using i = constant<int, 42>;
using c = constant<char, 'x'>;
using b = constant<bool, true>;
```
现在可以简单的实现为：
```cpp
template<auto v>
struct constant
{
    static constexpr auto value = v;
};

using i = constant<42>;
using c = constant<'x'>;
using b = constant<true>;
```
同样，原本的下列代码：
```cpp
template<typename T, T... Elements>
struct sequence {
};

using indexes = sequence<int, 0, 3, 4>;
```
现在可以简单的实现为：
```cpp
template<auto... Elements>
struct sequence {
};

using indexes = sequence<0, 3, 4>;
```
你现在甚至可以定义一个持有若干不同类型的值的编译期对象（类似于一个简单的tuple）：
```cpp
using tuple = sequence<0, 'h', true>;
```

## 13.2 使用`auto`作为变量模板的参数
你也可以使用`auto`作为模板参数来实现 *变量模板(variable templates)* 。

例如，下面的声明定义了一个变量模板`arr`，它的模板参数分别是元素的类型和数量：
```cpp
template<typename T, auto N> std::array<T, N> arr;
```
在每个编译单元中，所有对`arr<int, 10>`的引用将指向同一个全局对象。
而`arr<long, 10>`和`arr<int, 10u>`将指向其他对象
（每一个都可以在所有编译单元中使用）。

作为一个完整的例子，考虑如下的头文件：

```cpp
#ifndef VARTMPLAUTO_HPP
#define VARTMPLAUTO_HPP

#include <array>
template<typename T, auto N> std::array<T, N> arr{};

void printArr();

#endif // VARTMPLAUTO_HPP
```

这里，我们可以在一个编译单元内修改两个变量模板的不同实例：

```cpp
#include "vartmplauto.hpp"

int main()
{
    arr<int, 5>[0] = 17;
    arr<int, 5>[3] = 42;
    arr<int, 5u>[1] = 11;
    arr<int, 5u>[3] = 33;
    printArr();
}
```

另一个编译单元内可以打印这两个变量模板：

```cpp
#include "vartmplauto.hpp"
#include <iostream>

void printArr()
{
    std::cout << "arr<int, 5>:  ";
    for (const auto& elem : arr<int, 5>) {
        std::cout << elem << ' ';
    }
    std::cout << "\narr<int, 5u>: ";
    for (const auto& elem : arr<int, 5u>) {
        std::cout << elem << ' ';
    }
    std::cout << '\n';
}
```

程序的输出将是：
```
arr<int, 5>:  17 0 0 42 0
arr<int, 5u>: 0 11 0 33 0
```
用同样的方式你可以声明一个任意类型的常量变量模板，类型可以通过初始值推导出来：
```cpp
template<auto N> constexpr auto val = N; // 自从C++17起OK
```
之后可以像下面这样使用：
```cpp
auto v1 = val<5>;       // v1 == 5，v1的类型为int
auto v2 = val<true>;    // v2 == true，v2的类型为bool
auto v3 = val<'a'>;     // v3 == 'a'，v3的类型为char
```
这里解释了发生了什么：
```cpp
std::is_same_v<decltype(val<5>), int>       // 返回false
std::is_same_v<decltype(val<5>), const int> // 返回true
std::is_same_v<decltype(v1), int>           // 返回true（因为auto会退化）
```

## 13.3 使用`decltype(auto)`模板参数
你现在也可以使用另一个占位类型`decltype(auto)`（C++14引入）作为模板参数。
注意，这个占位类型的推导有非常特殊的规则。根据`decltype`的规则，如果使用
`decltype(auto)`来推导 *表达式(expressions)* 而不是变量名，
那么推导的结果将依赖于表达式的值类型：

- prvalue（例如临时变量）推导出 *type* 
- lvalue（例如有名字的对象）推导出 *type&* 
- xvalue（例如用`std::move()`标记的对象）推导出 *type&&* 

这意味着你很容易就会把模板参数推导为引用，这可能导致一些令人惊奇的效果。

例如：

```cpp
#include <iostream>

template<decltype(auto) N>
struct S {
    void printN() const {
        std::cout << "N: " << N << '\n';
    }
};

static const int c = 42;
static int v = 42;

int main()
{
    S<c> s1;        // N的类型推导为const int 42
    S<(c)> s2;      // N的类型推导为const int&，N是c的引用
    s1.printN();
    s2.printN();

    S<(v)> s3;      // N的类型推导为int&，N是v的引用
    v = 77;
    s3.printN();    // 打印出：N: 77
}
```

