# Chapter10 编译期`if`语句
通过使用语法`if constexpr(...)`，编译器可以计算编译期的条件表达式来在编译期决定使用
一个`if`语句的 *then* 的部分还是 *else* 的部分。其余部分的代码将会被丢弃，这意味着
它们甚至不会被生成。然而这并不意味着被丢弃的部分完全被忽略，
这些部分中的代码也会像没使用的模板一样进行语法检查。

例如：

```cpp
#include <string>

template <typename T>
std::string asString(T x)
{
    if constexpr(std::is_same_v<T, std::string>) {
        return x;                   // 如果T不能自动转换为string该语句将无效
    }
    else if constexpr(std::is_arithmetic_v<T>) {
        return std::to_string(x);   // 如果T不是数字类型该语句将无效
    }
    else {
        return std::string(x);      // 如果不能转换为string该语句将无效
    }
}
```

通过使用`if constexpr`我们在编译期就可以决定我们是简单返回传入的字符串、
对传入的数字调用`to_string()`还是使用构造函数来把传入的参数转换为
`std::string`。无效的调用将被 *丢弃* ，因此下面的代码能够通过编译
（如果使用运行时`if`语句则不能通过编译）：

```cpp
#include "ifcomptime.hpp"
#include <iostream>

int main()
{
    std::cout << asString(42) << '\n';
    std::cout << asString(std::string("hello")) << '\n';
    std::cout << asString("hello") << '\n';
}

```

## 10.1 编译期`if`语句的动机
如果我们在上面的例子中使用运行时`if`，下面的代码将永远不能通过编译：

```cpp
#include <string>

template <typename T>
std::string asString(T x)
{
    if (std::is_same_v<T, std::string>) {
        return x;                   // 如果不能自动转换为string会导致ERROR
    }
    else if (std::is_numeric_v<T>) {
        return std::to_string(x);   // 如果不是数字将导致ERROR
    }
    else {
        return std::string(x);      // 如果不能转换为string将导致ERROR
    }
}

```

这是因为模板在实例化时整个模板会作为一个整体进行编译。
然而`if`语句的条件表达式的检查是运行时特性。
即使在编译期就能确定条件表达式的值一定是`false`， *then* 的部分也必须能通过编译。
因此，当传递一个`std::string`或者字符串字面量时，会因为`std::to_string()`无效
而导致编译失败。此外，当传递一个数字值时，将会因为第一个和第三个返回语句无效而导致编译失败。

使用编译期`if`语句时， *then* 部分和 *else* 部分中不可能被用到的部分将成为
 *丢弃的语句* ：

- 当传递一个`std::string`时，第一个`if`语句的 *else* 部分将被丢弃。
- 当传递一个数字时，第一个`if`语句的 *then* 部分和最后的 *else* 部分将被丢弃。
- 当传递一个字符串字面量（类型为`const char*`）时，第一和第二个`if`语句
的 *then* 部分将被丢弃。

因此，在每一个实例化中，无效的分支都会在编译时被丢弃，所以代码能成功编译。

注意被丢弃的语句并不是被忽略了。即使是被忽略的语句也必须符合正确的语法，
并且所有和模板参数无关的调用也必须正确。
事实上，模板编译的第一个阶段（ *定义期间* ）将会检查语法和所有与模板无关的名称是否有效。
所有的`static_asserts`也必须有效，即使所在的分支没有被编译。

例如：
```cpp
template<typename T>
void foo(T t)
{
    if constexpr(std::is_integral_v<T>) {
        if (t > 0) {
            foo(t-1);   // OK
        }
    }
    else {
        undeclared(t); // 如果未被声明且未被丢弃将导致错误
        undeclared();  // 如果未声明将导致错误（即使被丢弃也一样）
        static_assert(false, "no integral"); // 总是会进行断言（即使被丢弃也一样）
    }
}
```
对于一个符合标准的编译器来说，上面的例子 *永远* 不能通过编译的原因有两个：

- 即使`T`是一个整数类型，如下调用：
```cpp
undeclared(); // 如果未声明将导致错误（即使被丢弃也一样）
```
如果该函数未定义时即使处于被丢弃的 *else* 部分也会导致错误，因为这个调用并不依赖于模板参数。
- 如下断言
```cpp
static_assert(false, "no integral"); // 总是会进行断言（即使被丢弃也一样）
```
即使处于被丢弃的 *else* 部分也总是会断言失败，因为它也不依赖于模板参数。
一个使用编译期条件的静态断言没有这个问题：
```cpp
static_assert(!std::is_integral_v<T>, "no integral");
```

注意有一些编译器（例如Visual C++ 2013和2015）并没有正确实现模板编译的两个阶段。
它们把第一个阶段（ *定义期间* ）的大部分工作推迟到了第二个阶段（ *实例化期间* ），
因此有些无效的函数调用甚至一些错误的语法都可能通过编译。

## 10.2 使用编译期`if`语句
理论上讲，只要条件表达式是编译期的表达式你就可以像使用运行期`if`一样使用编译期`if`。
你也可以混合使用编译期和运行期的`if`：
```cpp
if constexpr (std::is_integral_v<std::remove_reference_t<T>>) {
    if (val > 10) {
        if constexpr (std::numeric_limits<char>::is_signed) {
            ...
        }
        else {
            ...
        }
    }
    else {
        ...
    }
}
else {
    ...
}
```
注意你不能在函数体之外使用`if constexpr`。
因此，你不能使用它来替换预处理器的条件编译。

### 10.2.1 编译期`if`的注意事项
使用编译期`if`时可能会导致一些并不明显的后果。这将在接下来的小节中讨论。

#### 编译期`if`影响返回值类型
编译期`if`可能会影响函数的返回值类型。例如，下面的代码总能通过编译，
但返回值的类型可能会不同：
```cpp
auto foo()
{
    if constexpr (sizeof(int) > 4) {
        return 42;
    }
    else {
        return 42u;
    }
}
```
这里，因为我们使用了`auto`，返回值的类型将依赖于返回语句，
而执行哪条返回语句又依赖于`int`的字节数：

- 如果大于4字节，返回`42`的返回语句将会生效，因此返回值类型是`int`。
- 否则，返回`42u`的返回语句将生效，因此返回值类型是`unsigned int`。

这种情况下有`if constexpr`语句的函数可能返回完全不同的类型。
例如，如果我们不写 *else* 部分，返回值将会是`int`或者`void`：
```cpp
auto foo()  // 返回值类型可能是int或者void
{
    if constexpr (sizeof(int) > 4) {
        return 42;
    }
}
```
注意这里如果使用运行期`if`那么代码将永远不能通过编译，
因为推导返回值类型时会考虑到所有可能的返回值类型，因此推导会有歧义。

#### 即使在 *then* 部分返回也要考虑 *else* 部分
运行期`if`有一个模式不能应用于编译期`if`：
如果代码在 *then* 和 *else* 部分都会返回，
那么在运行期`if`中你可以跳过`else`部分。
也就是说，
```cpp
if (...) {
    return a;
}
else {
    return b;
}
```
可以写成：
```cpp
if (...) {
    return a;
}
return b;
```
但这个模式不能应用于编译期`if`，因为在第二种写法里，
返回值类型将同时依赖于两个返回语句而不是依赖其中一个，这会导致行为发生改变。
例如，如果按照上面的示例修改代码，那么 *也许能也许不能* 通过编译：
```cpp
auto foo()
{
    if constexpr (sizeof(int) > 4) {
        return 42;
    }
    return 42u;
}
```
如果条件表达式为true（`int`大于4字节），编译器将会推导出两个不同的返回值类型，
这会导致错误。否则，将只会有一条有效的返回语句，因此代码能通过编译。

#### 编译期短路求值
考虑如下代码：
```cpp
template<typename T>
constexpr auto foo(const T& val)
{
    if constexpr(std::is_integral<T>::value) {
        if constexpr (T{} < 10) {
            return val * 2;
        }
    }
    return val;
}
```
这里我们使用了两个编译期条件来决定是直接返回传入的值还是返回传入值的两倍。

下面的代码都能编译：
```cpp
constexpr auto x1 = foo(42);    // 返回84
constexpr auto x2 = foo("hi");  // OK，返回"hi"
```
运行时`if`的条件表达式会进行短路求值（当`&&`左侧为`false`时停止求值，
当`||`左侧为`true`时停止求值）。
这可能会导致你希望编译期`if`也会短路求值：
```cpp
template<typename T>
constexpr auto bar(const T& val)
{
    if constexpr (std::is_integral<T>::value && T{} < 10) {
        return val * 2;
    }
    return val;
}
```
然而，编译期`if`的条件表达式总是作为整体实例化并且必须整体有效，
这意味着如果传递一个不能进行`<10`运算的类型将不能通过编译：
```cpp
constexpr auto x2 = bar("hi");  // 编译期ERROR
```
因此，编译期`if`在实例化时并不短路求值。
如果后边的条件的有效性依赖于前边的条件，那你需要把条件进行嵌套。
例如，你必须写成如下形式：
```cpp
if constexpr (std::is_same_v<MyType, T>) {
    if constepxr (T::i == 42) {
        ...
    }
}
```
而不是写成：
```cpp
if constexpr (std::is_same_v<MyType, T> && T::i == 42) {
    ...
}
```

### 10.2.2 其他编译期`if`的示例
#### 完美返回泛型值
编译期`if`的一个应用就是先对返回值进行一些处理，再进行完美转发。
因为`decltype(auto)`不能推导为`void`（因为`void`是不完全类型），
所以你必须像下面这么写：

```cpp
#include <functional>   // for std::forward()
#include <type_traits>  // for std::is_same<> and std::invoke_result<>

template<typename Callable, typename... Args>
decltype(auto) call(Callable op, Args&&... args)
{
    if constexpr(std::is_void_v<std::invoke_result_t<Callable, Args...>>) {
        // 返回值类型是void：
        op(std::forward<Args>(args)...);
        ... // 在返回前进行一些处理
        return;
    }
    else {
        // 返回值类型不是void:
        decltype(auto) ret{op(std::forward<Args>(args)...)};
        ... // 在返回前用ret进行一些处理
        return ret;
    }
}

```

函数的返回值类型可以推导为`void`，
但`ret`的声明不能推导为`void`，
因此必须把`op`返回`void`的情况单独处理。

#### 使用编译期`if`进行类型分发
编译期`if`的一个典型应用是类型分发。在C++17之前，
你必须为每一个想处理的类型重载一个单独的函数。现在，有了编译期`if`，
你可以把所有的逻辑放在一个函数里。

例如，如下的重载版本的`std::advance()`算法：
```cpp
template<typename Iterator, typename Distance>
void advance(Iterator& pos, Distance n) {
    using cat = typename std::iterator_traits<Iterator>::iterator_category;
    advanceImpl(pos, n, cat{}); // 根据迭代器类型进行分发
}

template<typename Iterator, typename Distance>
void advanceImpl(Iterator& pos, Distance n, std::random_access_iterator_tag) {
    pos += n;
}

template<typename Iterator, typename Distance>
void advanceImpl(Iterator& pos, Distance n, std::bidirectional_iterator_tag) {
    if (n >= 0) {
        while (n--) {
            ++pos;
        }
    }
    else {
        while (n++) {
            --pos;
        }
    }
}

template<typename Iterator, typename Distance>
void advanceImpl(Iterator& pos, Distance n, std::input_iterator_tag) {
    while (n--) {
        ++pos;
    }
}
```
现在可以把所有实现都放在同一个函数中：
```cpp
template<typename Iterator, typename Distance>
void advance(Iterator& pos, Distance n) {
    using cat = typename std::iterator_traits<Iterator>::iterator_category;
    if constexpr (std::is_convertible_v<cat, std::random_access_iterator_tag>) {
        pos += n;
    }
    else if constexpr (std::is_convertible_v<cat, std::bidirectional_access_iterator_tag>) {
        if (n >= 0) {
            while (n--) {
                ++pos;
            }
        }
        else {
            while (n++) {
                --pos;
            }
        }
    }
    else {  // input_iterator_tag
        while (n--) {
            ++pos;
        }
    }
}
```
这里我们就像是有了一个编译期`switch`，每一个`if constexpr`语句就像是一个
case。然而，注意例子中的两种实现还是有一处不同的：

- 重载函数的版本遵循 **最佳匹配** 语义。
- 编译期`if`的版本遵循 **最先匹配** 语义。

另一个类型分发的例子是使用编译期`if`实现`get<>()`重载
来实现结构化绑定接口。

第三个例子是在用作`std::variant<>`访问器
的泛型lambda中处理不同的类型。

## 10.3 带初始化的编译期`if`语句
注意编译期`if`语句也可以使用新的带初始化的形式。
例如，如果有一个`constexpr`函数`foo()`，你可以这样写：
```cpp
template<typename T>
void bar(const T x)
{
    if constexpr (auto obj = foo(x); std::is_same_v<decltype(obj), T>) {
        std::cout << "foo(x) yields same type\n";
        ...
    }
    else {
        std::cout << "foo(x) yields different type\n";
        ...
    }
}
```
如果有一个参数类型也为`T`的`constexpr`函数`foo()`，
你就可以根据`foo(x)`是否返回与`x`相同的类型来进行不同的处理。

如果要根据`foo(x)`返回的值来进行判定，那么可以写：
```cpp
constexpr auto c = ...;
if constexpr (constexpr auto obj = foo(c); obj == 0) {
    std::cout << "foo() == 0\n";
    ...
}
```
注意如果想在条件语句中使用`obj`的值，
那么`obj`必须要声明为`constexpr`。

## 10.4 在模板之外使用编译期`if`
`if constexpr`可以在任何函数中使用，而并非仅限于模板。
只要条件表达式是编译期的，并且可以转换成`bool`类型。
然而，在普通函数里使用时 *then* 和 *else* 部分的所有语句都必须有效，
即使有可能被丢弃。

例如，下面的代码不能通过编译，因为`undeclared()`的调用必须是有效的，
即使`char`是有符号数导致 *else* 部分被丢弃也一样：
```cpp
#include <limits>

template<typename T>
void foo(T t);

int main()
{
    if constexpr(std::numeric_limits<char>::is_signed) {
        foo(42);        // OK
    }
    else {
        undeclared(42); // 未声明时总是ERROR（即使被丢弃）
    }
}
```

下面的代码也永远不能成功编译，因为总有一个静态断言会失败：
```cpp
if constexpr(std::numeric_limits<char>::is_signed) {
    static_assert(std::numeric_limits<char>::is_signed);
}
else {
    static_assert(!std::numeric_limits<char>::is_signed);
}
```
在泛型代码之外使用编译期`if`的唯一好处是被丢弃的部分不会成为最终程序的一部分，
这将减小生成的可执行程序的大小。例如，在如下程序中：
```cpp
#include <limits>
#include <string>
#include <array>

int main()
{
    if (!std::numeric_limits<char>::is_signed) {
        static std::array<std::string, 1000> arr1;
        ...
    }
    else {
        static std::array<std::string, 1000> arr2;
        ...
    }
}
```
要么`arr1`要么`arr2`会成为最终可执行程序的一部分，
但不可能两者都是。
