# Chapter7 新属性和属性特性
自从C++11起，就可以指明 *属性(attributes)* （允许或者禁用某些警告的注解）。
C++17引入了新的属性，还扩展了属性的使用场景，这样可以带来一些便利。

## 7.1 `[[nodiscard]]`属性
新属性`[[nodiscard]]`可以鼓励编译器在某个函数的返回值未被使用时给出警告
（这并不意味着编译器一定要给出警告）。

`[[nodiscard]]`通常应该用于防止某些因为返回值未被使用导致的不当行为。
这些不当行为可能是（译者注：请配合下边的例子理解这些不当行为）：

-  **内存泄露** ，例如返回值中含有动态分配的内存，但并未使用。
-  **未知的或出乎意料的行为** ，例如因为没有使用返回值而导致了一些奇怪的行为。
-  **不必要的开销** ，例如因为返回值没被使用而进行了一些无意义的行为。

这里有一些该属性发挥作用的例子：

- 申请资源但自身并不释放，而是将资源返回等待其他函数释放的函数应该被标记为
`[[nodiscard]]`。一个典型的例子是申请内存的函数，
例如`malloc()`函数或者分配器的`allocate()`成员函数。

然而，注意有些函数 *可能* 会返回一个无需再处理的值。例如，程序员在调用`realloc()`函数的时候，可能会有意的将参数`new_size`设为0来达到释放内存的目的，这种情况下的返回值无需之后调用`free()`函数释放。
因此，如果把`realloc()`函数标记为`[[nodiscard]]`将会适得其反。当然，并不推荐使用`realloc()`函数释放内存，因为那是未定义行为。
- 有时如果没有使用返回值将导致函数行为和预期不同，一个很好的例子是`std::async()`
（C++11引入）。`std::async()`会在后台异步地执行一个任务并返回一个可以用来等待
任务执行结束的句柄（也可以通过它获取返回值或者异常）。然而，如果返回值没有被使用的话该调用
将变成同步的调用，因为在启动任务的语句结束之后未被使用的返回值的析构函数会立即执行，而析构
函数会阻塞等待任务运行结束。因此，不使用返回值导致的结果与`std::async()`的目的
完全矛盾。将`std::async()`标记为`[[nodiscard]]`可以让编译器给出警告。
- 另一个例子是成员函数`empty()`，它的作用是检查一个对象（容器/字符串）是否
为空。程序员经常误用该函数来“清空”容器（删除所有元素）：
```cpp
cont.empty();
```
这种对`empty()`的误用并没有使用返回值，所以`[[nodiscard]]`可以检查出这种误用：
```cpp
class MyContainer {
    ...
public:
    [[nodiscard]] bool empty() const noexcept;
    ...
};
```
这里的属性标记可以帮助检查这种逻辑错误。

如果因为某些原因你不想使用一个被标记为`[[nodiscard]]`的函数的返回值，
你可以把返回值转换为`void`：
```cpp
(void)coll.empty(); // 禁止[[nodiscard]]警告
```
注意如果成员函数被覆盖或者隐藏时基类中标记的属性不会被继承：
```cpp
struct B {
    [[nodiscard]] int* foo();
};

struct D : B {
    int* foo();
};

B b;
b.foo();        // 警告
(void)b.foo();  // 没有警告

D d;
d.foo();        // 没有警告
```
因此你需要给派生类里相应的成员函数再次标记`[[nodiscard]]`
（除非有某些原因导致你不想在派生类里确保返回值必须被使用）。

你可以把属性标记在函数前的所有修饰符之前，也可以标记在函数名之后：
```cpp
class C {
    ...
    [[nodiscard]] friend bool operator== (const C&, const C&);
    friend bool operator!= [[nodiscard]] (const C&, const C&);
};
```
把属性放在`friend`和`bool`之间或者`bool`和`operator==`
之间是错误的。

尽管这个特性从C++17起引入，但它还没有在标准库中使用。因为这个提案出现的太晚了，所以最
需要它的`std::async()`也还没有使用它。不过这里讨论的所有例子，将在下一次C++标准
中实现（见C++20中通过的<https://wg21.link/p0600r1>提案）。

为了保证代码的可移植性，你应该使用`[[nodiscard]]`而不是一些不可移植的方案
（例如gcc和clang的
`[[gnu:warn_unused_result]]`或者Visual C++的`_Check_return_`）。

当定义`new()`运算符时，
你应该用`[[nodiscard]]`对该函数进行标记，
例如定义一个追踪所有`new`调用的头文件。

## 7.2 `[[maybe_unused]]`属性
新的属性`[[maybe_unused]]`可以避免编译器在某个变量未被使用时发出警告。
新的属性可以应用于类的声明、使用`typedef`或者`using`定义的类型、
一个变量、一个非静态数据成员、一个函数、一个枚举类型、一个枚举值等场景。

例如其中一个作用是定义一个可能不会使用的参数：
```cpp
void foo(int val, [[maybe_unused]] std::string msg)
{
#ifdef DEBUG
    log(msg);
#endif
    ...
}
```
另一个例子是定义一个可能不会使用的成员：
```cpp
class MyStruct {
    char c;
    int i;
    [[maybe_unused]] char makeLargerSize[100];
    ...
};
```
注意你不能对一条语句应用`[[maybe_unused]]`。
因此，你不能直接用`[[maybe_unused]]`来抵消
`[[nodiscard]]`的作用：

```cpp
[[nodiscard]] void* foo();
int main()
{
    foo();                              // 警告：返回值没有使用
    [[maybe_unused]] foo();             // 错误：maybe_unused不允许出现在此
    [[maybe_unused]] auto x = foo();    // OK
}
```

## 7.3 `[[fallthrough]]`属性
新的属性`[[fallthrough]]`可以避免编译器在`switch`语句中某一个标签
缺少`break`语句时发出警告。例如：
```cpp
void commentPlace(int place)
{
    switch (place) {
        case 1:
            std::cout << "very ";
            [[fallthrough]];
        case 2:
            std::cout << "well\n";
            break;
        default:
            std::cout << "OK\n";
            break;
    }
}
```
这个例子中参数为1时将输出：
```
very well
```
`case 1`和`case 2`中的语句都会被执行。
注意这个属性必须被用作单独的语句，还要有分号结尾。
另外在`switch`语句的最后一个分支不能使用它。

## 7.4 通用的属性扩展
自从C++17起下列有关属性的通用特性变得可用：

- 属性现在可以用来标记命名空间。例如，你可以像下面这样弃用一个命名空间：
```cpp
namespace [[deprecated]] DraftAPI {
    ...
}
```
这也可以应用于内联的和匿名的命名空间。
- 属性现在可以标记枚举子（枚举类型的值）。
例如你可以像下面这样引入一个新的枚举值作为某个已有枚举值（并且现在已经被废弃）的替代：
```cpp
enum class City { Berlin = 0,
                  NewYork = 1,
                  Mumbai = 2,
                  Bombay [[deprecated]] = Mumbai,
                  ... };
```
这里`Mumbai`和`Bombay`代表同一个城市的数字码，但使用`Bombay`
已经被标记为废弃的。注意对于枚举值，属性被放置在标识符 *之后* 。
- 用户自定义的属性一般应该定义在自定义的命名空间中。现在可以使用`using`前缀
来避免为每一个属性重复输入命名空间。也就是说，如下代码：
```cpp
[[MyLib::WebService, MyLib::RestService, MyLib::doc("html")]] void foo();
```
可以被替换为
```cpp
[[using MyLib: WebService, RestService, doc("html")]] void foo();
```
注意在使用了`using`前缀时重复命名空间将导致错误：
```cpp
[[using MyLib: MyLib::doc("html")]] void foo(); // ERROR
```
