# Chapter3 内联变量
出于可移植性和易于整合的目的，在头文件中提供完整的类和库的定义是很重要的。
然而，在C++17之前，只有当这个库既不提供也不需要全局对象的时候才可以这样做。

自从C++17开始，你可以在头文件中以`inline`的方式 *定义* 全局变量/对象：
```cpp
class MyClass {
    inline static std::string msg{"OK"}; // OK（自C++17起）
    ...
};

inline MyClass myGlobalObj;  // 即使被多个CPP文件包含也OK
```
只要一个编译单元内没有重复的定义即可。此例中的定义即使被多个编译单元使用，
也会指向同一个对象。

## 3.1 内联变量产生的动机
在C++里不允许在类里初始化非常量静态成员：
```cpp
class MyClass {
    static std::string msg{"OK"};   // 编译期ERROR
    ...
};
```
可以在类定义的外部定义并初始化非常量静态成员，但如果被多个CPP文件同时包含的话又会引发新的错误：
```cpp
class MyClass {
    static std::string msg;
    ...
};
std::string MyClass::msg{"OK"}; // 如果被多个CPP文件包含会导致链接ERROR
```
根据 *一次定义原则* (ODR)，一个变量或实体的定义只能出现在一个编译单元内——
除非该变量或实体被定义为`inline`的。

即使使用预处理来进行保护也没有用：
```cpp
#ifndef MYHEADER_HPP
#define MYHEADER_HPP

class MyClass {
    static std::string msg;
    ...
};
std::string MyClass::msg{"OK"}; // 如果被多个CPP文件包含会导致链接ERROR

#endif
```
问题并不在于头文件是否可能被重复包含多次，而是两个不同的CPP文件都包含了这个头文件，
因而都定义了`MyClass::msg`。

出于同样的原因，如果你在头文件中定义了一个类的实例对象
也会出现相同的链接错误：
```cpp
class MyClass {
    ...
};
MyClass myGlobalObject; // 如果被多个CPP文件包含会导致链接ERROR
```

#### 解决方法
对于一些场景，这里有一些解决方法：

- 你可以在一个`class/struct`的定义中初始化数字或枚举类型的常量静态成员：
```cpp
class MyClass {
    static const bool trace = false;    // OK，字面类型
    ...
};
```
然而，这种方法只能初始化字面类型，例如基本的整数、浮点数、指针类型或者
用常量表达式初始化了所有内部非静态成员的类，并且该类不能有用户自定义的或虚的析构函数。
另外，如果你需要获取这个静态常量成员的地址（例如你想定义一个它的引用）的话
那么你必须在那个编译单元内定义它并且不能在其他编译单元内再次定义。
- 你可以定义一个返回`static`的局部变量的内联函数：
```cpp
inline std::string& getMsg() {
    static std::string msg{"OK"};
    return msg;
}
```
- 你可以定义一个返回该值的`static`的成员函数：
```cpp
class MyClass {
    static std::string& getMsg() {
        static std::string msg{"OK"};
        return msg;
    }
    ...
};
```
- 你可以使用变量模板（自C++14起）：
```cpp
template<typename T = std::string>
T myGlobalMsg{"OK"};
```
- 你可以为静态成员定义一个模板类：
```cpp
template<typename = void>
class MyClassStatics
{
    static std::string msg;
};

template<typename T>
std::string MyClassStatics<T>::msg{"OK"};
```
然后继承它：
```cpp
class MyClass : public MyClassStatics<>
{
    ...
};
```

然而，所有这些方法都会导致签名重载，可读性也会变差，使用该变量的方式也变得不同。
另外，全局变量的初始化可能会推迟到第一次使用时。
所以那些假设变量一开始就已经初始化的写法是不可行的（例如使用一个对象来监控整个程序的过程）。

## 3.2 使用内联变量
现在，使用了`inline`修饰符之后，即使定义所在的头文件被多个CPP文件包含，
也只会有一个全局对象：
```cpp
class MyClass {
    inline static std::string msg{"OK"};    // 自从C++17起OK
    ...
};

inline MyClass myGlobalObj; // 即使被多个CPP文件包含也OK
```
这里使用的`inline`和函数声明时的`inline`有相同的语义：

- 它可以在多个编译单元中定义，只要所有定义都是相同的。
- 它必须在每个使用它的编译单元中定义

将变量定义在头文件里，然后多个CPP文件再都包含这个头文件，就可以满足上述两个要求。
程序的行为就好像只有一个变量一样。

你甚至可以利用它在头文件中定义原子类型：
```cpp
inline std::atomic<bool> ready{false};
```
像通常一样，当你定义`std::atomic`类型的变量时必须进行初始化。

注意你仍然必须确保在你初始化内联变量之前它们的类型必须是完整的。
例如，如果一个`struct`或者`class`有一个自身类型的`static`成员，
那么这个成员只能在类型声明之后再进行定义：
```cpp
struct MyType {
    int value;
    MyType(int i) : value{i} {
    }
    // 一个存储该类型最大值的静态对象
    static MyType max;  // 这里只能进行声明
    ...
};
inline MyType MyType::max{0};
```
另一个使用内联变量的例子见追踪所有`new`调用的头文件。

## 3.3 `constexpr static`成员现在隐含`inline`
对于静态成员，`constexpr`修饰符现在隐含着`inline`。
自从C++17起，如下声明 *定义* 了静态数据成员`n`：
```cpp
struct D {
    static constexpr int n = 5; // C++11/C++14: 声明
                                // 自从C++17起: 定义
}
```
和下边的代码等价：
```cpp
struct D {
    inline static constexpr int n = 5;
};
```
注意在C++17之前，你就可以只有声明没有定义。考虑如下声明：
```cpp
struct D {
    static constexpr int n = 5;
};
```
如果不需要`D::n`的定义的话只有上面的声明就够了，
例如当`D::n`以值传递时：
```cpp
std::cout << D::n;          // OK，ostream::operator<<(int)只需要D::n的值
```
如果`D::n`以引用传递到一个非内联函数，并且该函数调用没有被优化掉的话，
该调用将会导致错误。例如：
```cpp
int twice(const int& i);

std::cout << twice(D::n);   // 通常会导致ERROR
```
这段代码违反了 *一次定义原则* (ODR)。如果编译器进行了优化，那么这段代码可能会像预期一样工作
也可能会因为缺少定义导致链接错误。如果不进行优化，那么几乎肯定会因为缺少`D::n`的定义而
导致错误。
如果创建一个`D::n`的指针那么更可能因为缺少定义导致链接错误（但在某些编译模式下仍然可能正常编译）：
```cpp
const int* p = &D::n;       // 通常会导致ERROR
```
因此在C++17之前，你必须在一个编译单元内定义`D::n`:
```cpp
constexpr int D::n;         // C++11/C++14: 定义
                            // 自从C++17起: 多余的声明（已被废弃）
```
现在当使用C++17进行构建时，类中的声明本身就成了定义，因此即使没有上边的定义，
上面的所有例子现在也都可以正常工作。上边的定义现在仍然有效但已经成了废弃的多余声明。

## 3.4 内联变量和`thread_local`
通过使用`thread_local`你可以为每个线程创建一个内联变量：
```cpp
struct ThreadData {
    inline static thread_local std::string name;    // 每个线程都有自己的name
    ...
};

inline thread_local std::vector<std::string> cache; // 每个线程都有一份cache
```
作为一个完整的例子，考虑如下头文件：

```cpp
#include <string>
#include <iostream>

struct MyData {
    inline static std::string gName = "global";             // 整个程序中只有一个
    inline static thread_local std::string tName = "tls";   // 每个线程有一个
    std::string lName = "local";                            // 每个实例有一个
    ...
    void print(const std::string& msg) const {
        std::cout << msg << '\n';
        std::cout << "- gName: " << gName << '\n';
        std::cout << "- tName: " << tName << '\n';
        std::cout << "- lName: " << lName << '\n';
    }
};

inline thread_local MyData myThreadData;    // 每个线程一个对象
```

你可以在包含`main()`的编译单元内使用它：

```cpp
#include "inlinethreadlocal.hpp"
#include <thread>

void foo();

int main()
{
    myThreadData.print("main() begin:");

    myThreadData.gName = "thraed1 name";
    myThreadData.tName = "thread1 name";
    myThreadData.lName = "thread1 name";
    myThreadData.print("main() later:");

    std::thread t(foo);
    t.join();
    myThreadData.print("main() end:");
}
```

你也可以在另一个定义了`foo()`函数的编译单元内使用这个头文件，
这个函数会在另一个线程中被调用：

```cpp
#include "inlinethreadlocal.hpp"

void foo()
{
    myThreadData.print("foo() begin:");

    myThreadData.gName = "thread2 name";
    myThreadData.tName = "thread2 name";
    myThreadData.lName = "thread2 name";
    myThreadData.print("foo() end:");
}
```

程序的输出如下：
```
main() begin:
- gName: global
- tName: tls
- lName: local
main() later:
- gName: thread1 name
- tName: thread1 name
- lName: thread1 name
foo() begin:
- gName: thread1 name
- tName: tls
- lName: local
foo() end:
- gName: thread2 name
- tName: thread2 name
- lName: thread2 name
main() end:
- gName: thread2 name
- tName: thread1 name
- lName: thread1 name
```
