unit Groq.Functions.Core;

{-------------------------------------------------------------------------------

      Github repository :  https://github.com/MaxiDonkey/DelphiGroqCloud
      Visit the Github repository for the documentation and use examples

 ------------------------------------------------------------------------------}

interface

uses
  System.SysUtils, System.Classes, System.JSON;

type
  /// <summary>
  /// Interface defining the core structure and functionality of a function in the system.
  /// </summary>
  /// <remarks>
  /// This interface outlines the basic properties and methods that any function implementation must include.
  /// </remarks>
  IFunctionCore = interface
    ['{DA7C210F-8940-48E9-B933-1AEBFD7A81B8}']
    /// <summary>
    /// Retrieves the description of the function.
    /// </summary>
    function GetDescription: string;
    /// <summary>
    /// Retrieves the name of the function.
    /// </summary>
    function GetName: string;
    /// <summary>
    /// Retrieves the InputSchema required by the function, represented as a JSON schema.
    /// </summary>
    function GetInputSchema: string;
    /// <summary>
    /// Executes the function with the provided arguments and returns the result as a string.
    /// </summary>
    /// <param name="Arguments">The arguments passed to the function in JSON format.</param>
    /// <returns>The result of the function execution as a string.</returns>
    function Execute(const Arguments: string): string;
     /// <summary>
    /// Converts the TFunctionCore instance to a JSON object that contains its type and representation.
    /// </summary>
    /// <returns>A JSON object representing the function instance.</returns>
    function ToJson: TJSONObject;
    /// <summary>
    /// Creates a string representation of the TFunctionCore instance in JSON format, including its description, name, and InputSchema.
    /// </summary>
    /// <returns>A string representation of the function in JSON format.</returns>
    function ToString: string;
    /// <summary>
    /// A brief description of the function's purpose, used by the model to determine when and how to call the function.
    /// </summary>
    property Description: string read GetDescription;
    /// <summary>
    /// The unique identifier of the function that will be called. It must only contain characters from a-z, A-Z, 0-9, underscores, or dashes, and should not exceed 64 characters in length.
    /// </summary>
    property Name: string read GetName;
    //// <summary>
    /// The InputSchema required by the function, specified as a JSON schema. If no InputSchema are required, use the schema: {"type": "object", "properties": {}}.
    /// </summary>
    property InputSchema: string read GetInputSchema;
  end;

  /// <summary>
  /// Abstract base class for implementing core function behavior.
  /// </summary>
  /// <remarks>
  /// This class provides basic implementations for some methods and defines the structure that derived classes must follow.
  /// </remarks>
  TFunctionCore = class abstract(TinterfacedObject, IFunctionCore)
  protected
    /// <summary>
    /// Retrieves the description of the function. Derived classes must implement this method.
    /// </summary>
    function GetDescription: string; virtual; abstract;
     /// <summary>
    /// Retrieves the name of the function. Derived classes must implement this method.
    /// </summary>
    function GetName: string; virtual; abstract;
    /// <summary>
    /// Retrieves the InputSchema required by the function, represented as a JSON schema. Derived classes must implement this method.
    /// </summary>
    function GetInputSchema: string; virtual; abstract;
  public
    /// <summary>
    /// Executes the function with the provided arguments and returns the result as a string. Derived classes must implement this method.
    /// </summary>
    /// <param name="Arguments">The arguments passed to the function in JSON format.</param>
    /// <returns>The result of the function execution as a string.</returns>
    function Execute(const Arguments: string): string; virtual; abstract;
    /// <summary>
    /// Converts the TFunctionCore instance to a JSON object that contains its type and representation.
    /// </summary>
    /// <returns>A JSON object representing the function instance.</returns>
    function ToJson: TJSONObject;
    /// <summary>
    /// Creates a string representation of the TFunctionCore instance in JSON format, including its description, name, and InputSchema.
    /// </summary>
    /// <returns>A string representation of the function in JSON format.</returns>
    function ToString: string; override;
    /// <summary>
    /// A brief description of the function's purpose, used by the model to determine when and how to call the function.
    /// </summary>
    property Description: string read GetDescription;
    /// <summary>
    /// The unique identifier of the function that will be called. It must only contain characters from a-z, A-Z, 0-9, underscores, or dashes, and should not exceed 64 characters in length.
    /// </summary>
    property Name: string read GetName;
    /// <summary>
    /// The InputSchema required by the function, specified as a JSON schema. If no InputSchema are required, use the schema: {"type": "object", "properties": {}}.
    /// </summary>
    property InputSchema: string read GetInputSchema;
  end;

implementation

{ TFunctionCore }

function TFunctionCore.ToJson: TJSONObject;
begin
  Result := TJSONObject.Create;
  try
    Result.AddPair('type', 'function');
    var JSONFunction := TJSONObject.Create
          .AddPair('name', Name)
          .AddPair('description', Description)
          .AddPair('parameters', TJSONObject.ParseJSONValue(InputSchema));
    Result.AddPair('function', JSONFunction);
  except
    on E: Exception do
      begin
        Result.Free;
        raise;
      end;
  end;
end;

function TFunctionCore.ToString: string;
begin
  with TStringWriter.Create do
    try
      Write('"type": "%s"', ['function']);
      Write(',"function": {');
      Write('"name": "%s"', [Name]);
      Write(',"description": "%s"', [Description]);
      Write(',"parameters": %s}', [InputSchema]);
      Result := Format('{%s}', [ToString]);
    finally
      Free;
    end;

  {--- Format output }
  var JSON := TJSONObject.ParseJSONValue(Result);
  try
    Result := JSON.Format();
  finally
    JSON.Free;
  end;
end;

end.
