meta:
  id: dicom
  title: Digital Imaging and Communications in Medicine (DICOM) file format
  file-extension: dcm
  license: MIT
  endian: le
doc-ref: 'http://dicom.nema.org/medical/dicom/current/output/html/part10.html#chapter_7'
doc: |
  DICOM (Digital Imaging and Communications in Medicine), AKA NEMA
  PS3, AKA ISO 12052:2006, is a file format and network protocol
  standard for medical imaging purposes. This parser covers file
  format, typically written by various medical equipment, such as
  radiography, computer tomography scans, MRT, ultrasonography, etc.

  DICOM defines two transfer syntaxes: implicit and explicit. This
  top-level parser attempts to autodetect and handle both of them. If
  any problems arise, one can use `file_explicit` and `file_implicit`
  subtypes to force parsing in particular transfer syntax.
seq:
  - id: file_header
    type: t_file_header
  - id: elements
    type: t_data_element_implicit
    repeat: eos
types:
  t_file_header:
    seq:
      - id: preamble
        size: 128
      - id: magic
        contents: 'DICM'
  t_data_element_explicit:
    doc-ref: 'http://dicom.nema.org/medical/dicom/current/output/html/part05.html#sect_7.1.2'
    seq:
      - id: tag_group
        type: u2
      - id: tag_elem
        type: u2
      - id: vr
        type: str
        encoding: ASCII
        size: 2
        if: not is_forced_implicit
      - id: reserved
        type: u2
        if: is_long_len and not is_forced_implicit
      - id: value_len
        type:
          switch-on: is_long_len
          cases:
            false: u2
            true: u4
      - id: value
        size: value_len
        if: value_len != 0xffff_ffff
      - id: items
        type: seq_item
        repeat: until
        repeat-until: _.tag_elem == 0xe0dd
        if: vr == 'SQ' and value_len == 0xffff_ffff
      - id: elements_implicit
        type: t_data_element_implicit
        repeat: eos
        if: is_transfer_syntax_change_implicit
    instances:
      is_forced_implicit:
        value: tag_group == 0xfffe
      is_long_len:
        value: >
          is_forced_implicit or
          vr == 'OB' or
          vr == 'OD' or
          vr == 'OF' or
          vr == 'OL' or
          vr == 'OW' or
          vr == 'SQ' or
          vr == 'UC' or
          vr == 'UR' or
          vr == 'UT' or
          vr == 'UN'
#      p_is_transfer_syntax_change_non_implicit:
#        # '1.2.840.10008.1.2.1\0' (Explicit VR Little Endian)
#        # See http://www.dicomlibrary.com/dicom/transfer-syntax/
#        value: value != [49, 46, 50, 46, 56, 52, 48, 46, 49, 48, 48, 48, 56, 46, 49, 46, 50, 46, 49, 0]
      is_transfer_syntax_change_implicit:
        value: false
      tag:
        value: 'tag_group << 16 | tag_elem'
        enum: tags
  t_data_element_implicit:
    doc-ref: 'http://dicom.nema.org/medical/dicom/current/output/html/part05.html#sect_7.1.2'
    seq:
      - id: tag_group
        type: u2
      - id: tag_elem
        type: u2
      - id: vr
        type: str
        encoding: ASCII
        size: 2
        if: is_forced_explicit
      - id: reserved
        type: u2
        if: is_long_len and is_forced_explicit
      - id: value_len
        type:
          switch-on: 'is_forced_explicit ? is_long_len : true'
          cases:
            false: u2
            true: u4
      - id: value
        size: value_len
        if: value_len != 0xffff_ffff
      - id: items
        type: seq_item
        repeat: until
        repeat-until: _.tag_elem == 0xe0dd
        if: vr == 'SQ' and value_len == 0xffff_ffff
      - id: elements
        type: t_data_element_explicit
        repeat: eos
        if: is_transfer_syntax_change_explicit
    instances:
      is_forced_explicit:
        value: tag_group == 0x0002
      is_long_len:
        value: >
          is_forced_explicit and (
          vr == 'OB' or
          vr == 'OD' or
          vr == 'OF' or
          vr == 'OL' or
          vr == 'OW' or
          vr == 'SQ' or
          vr == 'UC' or
          vr == 'UR' or
          vr == 'UT' or
          vr == 'UN')
      tag:
        value: 'tag_group << 16 | tag_elem'
        enum: tags
      p_is_transfer_syntax_change_explicit:
        # '1.2.840.10008.1.2.1\0' (Explicit VR Little Endian)
        # See http://www.dicomlibrary.com/dicom/transfer-syntax/
        value: value == [49, 46, 50, 46, 56, 52, 48, 46, 49, 48, 48, 48, 56, 46, 49, 46, 50, 46, 49, 0]
      is_transfer_syntax_change_explicit:
        value: p_is_transfer_syntax_change_explicit
  seq_item:
    seq:
      - id: tag_group
        contents: [0xfe, 0xff]
      - id: tag_elem
        type: u2
      - id: value_len
        type: u4
      - id: value
        size: value_len
        if: value_len != 0xffff_ffff
      - id: items
        type: t_data_element_explicit
        repeat: until
        repeat-until: _.tag_group == 0xfffe and _.tag_elem == 0xe00d
        if: value_len == 0xffff_ffff
enums:
  tags:
    0x00020000: file_meta_information_group_length
    0x00020001: file_meta_information_version
    0x00020002: media_storage_sop_class_uid
    0x00020003: media_storage_sop_instance_uid
    0x00020010: transfer_syntax_uid
    0x00020012: implementation_class_uid
    0x00020013: implementation_version_name
    0x00020016: source_application_entity_title
    0x00020017: sending_application_entity_title
    0x00020018: receiving_application_entity_title
    0x00020100: private_information_creator_uid
    0x00020102: private_information
    0x00041130: file_set_id
    0x00041141: file_set_descriptor_file_id
    0x00041142: specific_character_set_of_file_set_descriptor_file
    0x00041200: offset_of_the_first_directory_record_of_the_root_directory_entity
    0x00041202: offset_of_the_last_directory_record_of_the_root_directory_entity
    0x00041212: file_set_consistency_flag
    0x00041220: directory_record_sequence
    0x00041400: offset_of_the_next_directory_record
    0x00041410: record_in_use_flag
    0x00041420: offset_of_referenced_lower_level_directory_entity
    0x00041430: directory_record_type
    0x00041432: private_record_uid
    0x00041500: referenced_file_id
    0x00041504: mrdr_directory_record_offset
    0x00041510: referenced_sop_class_uid_in_file
    0x00041511: referenced_sop_instance_uid_in_file
    0x00041512: referenced_transfer_syntax_uid_in_file
    0x0004151A: referenced_related_general_sop_class_uid_in_file
    0x00041600: number_of_references
    0x00080001: length_to_end
    0x00080005: specific_character_set
    0x00080006: language_code_sequence
    0x00080008: image_type
    0x00080010: recognition_code
    0x00080012: instance_creation_date
    0x00080013: instance_creation_time
    0x00080014: instance_creator_uid
    0x00080015: instance_coercion_datetime
    0x00080016: sop_class_uid
    0x00080018: sop_instance_uid
    0x0008001A: related_general_sop_class_uid
    0x0008001B: original_specialized_sop_class_uid
    0x00080020: study_date
    0x00080021: series_date
    0x00080022: acquisition_date
    0x00080023: content_date
    0x00080024: overlay_date
    0x00080025: curve_date
    0x0008002A: acquisition_datetime
    0x00080030: study_time
    0x00080031: series_time
    0x00080032: acquisition_time
    0x00080033: content_time
    0x00080034: overlay_time
    0x00080035: curve_time
    0x00080040: data_set_type
    0x00080041: data_set_subtype
    0x00080042: nuclear_medicine_series_type
    0x00080050: accession_number
    0x00080051: issuer_of_accession_number_sequence
    0x00080052: query_retrieve_level
    0x00080053: query_retrieve_view
    0x00080054: retrieve_ae_title
    0x00080055: station_ae_title
    0x00080056: instance_availability
    0x00080058: failed_sop_instance_uid_list
    0x00080060: modality
    0x00080061: modalities_in_study
    0x00080062: sop_classes_in_study
    0x00080064: conversion_type
    0x00080068: presentation_intent_type
    0x00080070: manufacturer
    0x00080080: institution_name
    0x00080081: institution_address
    0x00080082: institution_code_sequence
    0x00080090: referring_physician_s_name
    0x00080092: referring_physician_s_address
    0x00080094: referring_physician_s_telephone_numbers
    0x00080096: referring_physician_identification_sequence
    0x0008009C: consulting_physician_s_name
    0x0008009D: consulting_physician_identification_sequence
    0x00080100: code_value
    0x00080101: extended_code_value
    0x00080102: coding_scheme_designator
    0x00080103: coding_scheme_version
    0x00080104: code_meaning
    0x00080105: mapping_resource
    0x00080106: context_group_version
    0x00080107: context_group_local_version
    0x00080108: extended_code_meaning
    0x0008010B: context_group_extension_flag
    0x0008010C: coding_scheme_uid
    0x0008010D: context_group_extension_creator_uid
    0x0008010F: context_identifier
    0x00080110: coding_scheme_identification_sequence
    0x00080112: coding_scheme_registry
    0x00080114: coding_scheme_external_id
    0x00080115: coding_scheme_name
    0x00080116: coding_scheme_responsible_organization
    0x00080117: context_uid
    0x00080118: mapping_resource_uid
    0x00080119: long_code_value
    0x00080120: urn_code_value
    0x00080121: equivalent_code_sequence
    0x00080122: mapping_resource_name
    0x00080123: context_group_identification_sequence
    0x00080124: mapping_resource_identification_sequence
    0x00080201: timezone_offset_from_utc
    0x00080300: private_data_element_characteristics_sequence
    0x00080301: private_group_reference
    0x00080302: private_creator_reference
    0x00080303: block_identifying_information_status
    0x00080304: nonidentifying_private_elements
    0x00080306: identifying_private_elements
    0x00080305: deidentification_action_sequence
    0x00080307: deidentification_action
    0x00081000: network_id
    0x00081010: station_name
    0x00081030: study_description
    0x00081032: procedure_code_sequence
    0x0008103E: series_description
    0x0008103F: series_description_code_sequence
    0x00081040: institutional_department_name
    0x00081048: physician_s__of_record
    0x00081049: physician_s__of_record_identification_sequence
    0x00081050: performing_physician_s_name
    0x00081052: performing_physician_identification_sequence
    0x00081060: name_of_physician_s__reading_study
    0x00081062: physician_s__reading_study_identification_sequence
    0x00081070: operators__name
    0x00081072: operator_identification_sequence
    0x00081080: admitting_diagnoses_description
    0x00081084: admitting_diagnoses_code_sequence
    0x00081090: manufacturer_s_model_name
    0x00081100: referenced_results_sequence
    0x00081110: referenced_study_sequence
    0x00081111: referenced_performed_procedure_step_sequence
    0x00081115: referenced_series_sequence
    0x00081120: referenced_patient_sequence
    0x00081125: referenced_visit_sequence
    0x00081130: referenced_overlay_sequence
    0x00081134: referenced_stereometric_instance_sequence
    0x0008113A: referenced_waveform_sequence
    0x00081140: referenced_image_sequence
    0x00081145: referenced_curve_sequence
    0x0008114A: referenced_instance_sequence
    0x0008114B: referenced_real_world_value_mapping_instance_sequence
    0x00081150: referenced_sop_class_uid
    0x00081155: referenced_sop_instance_uid
    0x0008115A: sop_classes_supported
    0x00081160: referenced_frame_number
    0x00081161: simple_frame_list
    0x00081162: calculated_frame_list
    0x00081163: time_range
    0x00081164: frame_extraction_sequence
    0x00081167: multi_frame_source_sop_instance_uid
    0x00081190: retrieve_url
    0x00081195: transaction_uid
    0x00081196: warning_reason
    0x00081197: failure_reason
    0x00081198: failed_sop_sequence
    0x00081199: referenced_sop_sequence
    0x0008119A: other_failures_sequence
    0x00081200: studies_containing_other_referenced_instances_sequence
    0x00081250: related_series_sequence
    0x00082110: lossy_image_compression
    0x00082111: derivation_description
    0x00082112: source_image_sequence
    0x00082120: stage_name
    0x00082122: stage_number
    0x00082124: number_of_stages
    0x00082127: view_name
    0x00082128: view_number
    0x00082129: number_of_event_timers
    0x0008212A: number_of_views_in_stage
    0x00082130: event_elapsed_time_s_
    0x00082132: event_timer_name_s_
    0x00082133: event_timer_sequence
    0x00082134: event_time_offset
    0x00082135: event_code_sequence
    0x00082142: start_trim
    0x00082143: stop_trim
    0x00082144: recommended_display_frame_rate
    0x00082200: transducer_position
    0x00082204: transducer_orientation
    0x00082208: anatomic_structure
    0x00082218: anatomic_region_sequence
    0x00082220: anatomic_region_modifier_sequence
    0x00082228: primary_anatomic_structure_sequence
    0x00082229: anatomic_structure__space_or_region_sequence
    0x00082230: primary_anatomic_structure_modifier_sequence
    0x00082240: transducer_position_sequence
    0x00082242: transducer_position_modifier_sequence
    0x00082244: transducer_orientation_sequence
    0x00082246: transducer_orientation_modifier_sequence
    0x00082251: anatomic_structure_space_or_region_code_sequence__trial_
    0x00082253: anatomic_portal_of_entrance_code_sequence__trial_
    0x00082255: anatomic_approach_direction_code_sequence__trial_
    0x00082256: anatomic_perspective_description__trial_
    0x00082257: anatomic_perspective_code_sequence__trial_
    0x00082258: anatomic_location_of_examining_instrument_description__trial_
    0x00082259: anatomic_location_of_examining_instrument_code_sequence__trial_
    0x0008225A: anatomic_structure_space_or_region_modifier_code_sequence__trial_
    0x0008225C: on_axis_background_anatomic_structure_code_sequence__trial_
    0x00083001: alternate_representation_sequence
    0x00083010: irradiation_event_uid
    0x00083011: source_irradiation_event_sequence
    0x00083012: radiopharmaceutical_administration_event_uid
    0x00084000: identifying_comments
    0x00089007: frame_type
    0x00089092: referenced_image_evidence_sequence
    0x00089121: referenced_raw_data_sequence
    0x00089123: creator_version_uid
    0x00089124: derivation_image_sequence
    0x00089154: source_image_evidence_sequence
    0x00089205: pixel_presentation
    0x00089206: volumetric_properties
    0x00089207: volume_based_calculation_technique
    0x00089208: complex_image_component
    0x00089209: acquisition_contrast
    0x00089215: derivation_code_sequence
    0x00089237: referenced_presentation_state_sequence
    0x00089410: referenced_other_plane_sequence
    0x00089458: frame_display_sequence
    0x00089459: recommended_display_frame_rate_in_float
    0x00089460: skip_frame_range_flag
    0x00100010: patient_s_name
    0x00100020: patient_id
    0x00100021: issuer_of_patient_id
    0x00100022: type_of_patient_id
    0x00100024: issuer_of_patient_id_qualifiers_sequence
    0x00100026: source_patient_group_identification_sequence
    0x00100027: group_of_patients_identification_sequence
    0x00100028: subject_relative_position_in_image
    0x00100030: patient_s_birth_date
    0x00100032: patient_s_birth_time
    0x00100033: patient_s_birth_date_in_alternative_calendar
    0x00100034: patient_s_death_date_in_alternative_calendar
    0x00100035: patient_s_alternative_calendar
    0x00100040: patient_s_sex
    0x00100050: patient_s_insurance_plan_code_sequence
    0x00100101: patient_s_primary_language_code_sequence
    0x00100102: patient_s_primary_language_modifier_code_sequence
    0x00100200: quality_control_subject
    0x00100201: quality_control_subject_type_code_sequence
    0x00100212: strain_description
    0x00100213: strain_nomenclature
    0x00100214: strain_stock_number
    0x00100215: strain_source_registry_code_sequence
    0x00100216: strain_stock_sequence
    0x00100217: strain_source
    0x00100218: strain_additional_information
    0x00100219: strain_code_sequence
    0x00101000: other_patient_ids
    0x00101001: other_patient_names
    0x00101002: other_patient_ids_sequence
    0x00101005: patient_s_birth_name
    0x00101010: patient_s_age
    0x00101020: patient_s_size
    0x00101021: patient_s_size_code_sequence
    0x00101030: patient_s_weight
    0x00101040: patient_s_address
    0x00101050: insurance_plan_identification
    0x00101060: patient_s_mother_s_birth_name
    0x00101080: military_rank
    0x00101081: branch_of_service
    0x00101090: medical_record_locator
    0x00101100: referenced_patient_photo_sequence
    0x00102000: medical_alerts
    0x00102110: allergies
    0x00102150: country_of_residence
    0x00102152: region_of_residence
    0x00102154: patient_s_telephone_numbers
    0x00102155: patient_s_telecom_information
    0x00102160: ethnic_group
    0x00102180: occupation
    0x001021A0: smoking_status
    0x001021B0: additional_patient_history
    0x001021C0: pregnancy_status
    0x001021D0: last_menstrual_date
    0x001021F0: patient_s_religious_preference
    0x00102201: patient_species_description
    0x00102202: patient_species_code_sequence
    0x00102203: patient_s_sex_neutered
    0x00102210: anatomical_orientation_type
    0x00102292: patient_breed_description
    0x00102293: patient_breed_code_sequence
    0x00102294: breed_registration_sequence
    0x00102295: breed_registration_number
    0x00102296: breed_registry_code_sequence
    0x00102297: responsible_person
    0x00102298: responsible_person_role
    0x00102299: responsible_organization
    0x00104000: patient_comments
    0x00109431: examined_body_thickness
    0x00120010: clinical_trial_sponsor_name
    0x00120020: clinical_trial_protocol_id
    0x00120021: clinical_trial_protocol_name
    0x00120030: clinical_trial_site_id
    0x00120031: clinical_trial_site_name
    0x00120040: clinical_trial_subject_id
    0x00120042: clinical_trial_subject_reading_id
    0x00120050: clinical_trial_time_point_id
    0x00120051: clinical_trial_time_point_description
    0x00120060: clinical_trial_coordinating_center_name
    0x00120062: patient_identity_removed
    0x00120063: de_identification_method
    0x00120064: de_identification_method_code_sequence
    0x00120071: clinical_trial_series_id
    0x00120072: clinical_trial_series_description
    0x00120081: clinical_trial_protocol_ethics_committee_name
    0x00120082: clinical_trial_protocol_ethics_committee_approval_number
    0x00120083: consent_for_clinical_trial_use_sequence
    0x00120084: distribution_type
    0x00120085: consent_for_distribution_flag
    0x00140023: cad_file_format
    0x00140024: component_reference_system
    0x00140025: component_manufacturing_procedure
    0x00140028: component_manufacturer
    0x00140030: material_thickness
    0x00140032: material_pipe_diameter
    0x00140034: material_isolation_diameter
    0x00140042: material_grade
    0x00140044: material_properties_description
    0x00140045: material_properties_file_format__retired_
    0x00140046: material_notes
    0x00140050: component_shape
    0x00140052: curvature_type
    0x00140054: outer_diameter
    0x00140056: inner_diameter
    0x00140100: component_welder_ids
    0x00140101: secondary_approval_status
    0x00140102: secondary_review_date
    0x00140103: secondary_review_time
    0x00140104: secondary_reviewer_name
    0x00140105: repair_id
    0x00140106: multiple_component_approval_sequence
    0x00140107: other_approval_status
    0x00140108: other_secondary_approval_status
    0x00141010: actual_environmental_conditions
    0x00141020: expiry_date
    0x00141040: environmental_conditions
    0x00142002: evaluator_sequence
    0x00142004: evaluator_number
    0x00142006: evaluator_name
    0x00142008: evaluation_attempt
    0x00142012: indication_sequence
    0x00142014: indication_number
    0x00142016: indication_label
    0x00142018: indication_description
    0x0014201A: indication_type
    0x0014201C: indication_disposition
    0x0014201E: indication_roi_sequence
    0x00142030: indication_physical_property_sequence
    0x00142032: property_label
    0x00142202: coordinate_system_number_of_axes
    0x00142204: coordinate_system_axes_sequence
    0x00142206: coordinate_system_axis_description
    0x00142208: coordinate_system_data_set_mapping
    0x0014220A: coordinate_system_axis_number
    0x0014220C: coordinate_system_axis_type
    0x0014220E: coordinate_system_axis_units
    0x00142210: coordinate_system_axis_values
    0x00142220: coordinate_system_transform_sequence
    0x00142222: transform_description
    0x00142224: transform_number_of_axes
    0x00142226: transform_order_of_axes
    0x00142228: transformed_axis_units
    0x0014222A: coordinate_system_transform_rotation_and_scale_matrix
    0x0014222C: coordinate_system_transform_translation_matrix
    0x00143011: internal_detector_frame_time
    0x00143012: number_of_frames_integrated
    0x00143020: detector_temperature_sequence
    0x00143022: sensor_name
    0x00143024: horizontal_offset_of_sensor
    0x00143026: vertical_offset_of_sensor
    0x00143028: sensor_temperature
    0x00143040: dark_current_sequence
    0x00143050: dark_current_counts
    0x00143060: gain_correction_reference_sequence
    0x00143070: air_counts
    0x00143071: kv_used_in_gain_calibration
    0x00143072: ma_used_in_gain_calibration
    0x00143073: number_of_frames_used_for_integration
    0x00143074: filter_material_used_in_gain_calibration
    0x00143075: filter_thickness_used_in_gain_calibration
    0x00143076: date_of_gain_calibration
    0x00143077: time_of_gain_calibration
    0x00143080: bad_pixel_image
    0x00143099: calibration_notes
    0x00144002: pulser_equipment_sequence
    0x00144004: pulser_type
    0x00144006: pulser_notes
    0x00144008: receiver_equipment_sequence
    0x0014400A: amplifier_type
    0x0014400C: receiver_notes
    0x0014400E: pre_amplifier_equipment_sequence
    0x0014400F: pre_amplifier_notes
    0x00144010: transmit_transducer_sequence
    0x00144011: receive_transducer_sequence
    0x00144012: number_of_elements
    0x00144013: element_shape
    0x00144014: element_dimension_a
    0x00144015: element_dimension_b
    0x00144016: element_pitch_a
    0x00144017: measured_beam_dimension_a
    0x00144018: measured_beam_dimension_b
    0x00144019: location_of_measured_beam_diameter
    0x0014401A: nominal_frequency
    0x0014401B: measured_center_frequency
    0x0014401C: measured_bandwidth
    0x0014401D: element_pitch_b
    0x00144020: pulser_settings_sequence
    0x00144022: pulse_width
    0x00144024: excitation_frequency
    0x00144026: modulation_type
    0x00144028: damping
    0x00144030: receiver_settings_sequence
    0x00144031: acquired_soundpath_length
    0x00144032: acquisition_compression_type
    0x00144033: acquisition_sample_size
    0x00144034: rectifier_smoothing
    0x00144035: dac_sequence
    0x00144036: dac_type
    0x00144038: dac_gain_points
    0x0014403A: dac_time_points
    0x0014403C: dac_amplitude
    0x00144040: pre_amplifier_settings_sequence
    0x00144050: transmit_transducer_settings_sequence
    0x00144051: receive_transducer_settings_sequence
    0x00144052: incident_angle
    0x00144054: coupling_technique
    0x00144056: coupling_medium
    0x00144057: coupling_velocity
    0x00144058: probe_center_location_x
    0x00144059: probe_center_location_z
    0x0014405A: sound_path_length
    0x0014405C: delay_law_identifier
    0x00144060: gate_settings_sequence
    0x00144062: gate_threshold
    0x00144064: velocity_of_sound
    0x00144070: calibration_settings_sequence
    0x00144072: calibration_procedure
    0x00144074: procedure_version
    0x00144076: procedure_creation_date
    0x00144078: procedure_expiration_date
    0x0014407A: procedure_last_modified_date
    0x0014407C: calibration_time
    0x0014407E: calibration_date
    0x00144080: probe_drive_equipment_sequence
    0x00144081: drive_type
    0x00144082: probe_drive_notes
    0x00144083: drive_probe_sequence
    0x00144084: probe_inductance
    0x00144085: probe_resistance
    0x00144086: receive_probe_sequence
    0x00144087: probe_drive_settings_sequence
    0x00144088: bridge_resistors
    0x00144089: probe_orientation_angle
    0x0014408B: user_selected_gain_y
    0x0014408C: user_selected_phase
    0x0014408D: user_selected_offset_x
    0x0014408E: user_selected_offset_y
    0x00144091: channel_settings_sequence
    0x00144092: channel_threshold
    0x0014409A: scanner_settings_sequence
    0x0014409B: scan_procedure
    0x0014409C: translation_rate_x
    0x0014409D: translation_rate_y
    0x0014409F: channel_overlap
    0x001440A0: image_quality_indicator_type
    0x001440A1: image_quality_indicator_material
    0x001440A2: image_quality_indicator_size
    0x00145002: linac_energy
    0x00145004: linac_output
    0x00145100: active_aperture
    0x00145101: total_aperture
    0x00145102: aperture_elevation
    0x00145103: main_lobe_angle
    0x00145104: main_roof_angle
    0x00145105: connector_type
    0x00145106: wedge_model_number
    0x00145107: wedge_angle_float
    0x00145108: wedge_roof_angle
    0x00145109: wedge_element_1_position
    0x0014510A: wedge_material_velocity
    0x0014510B: wedge_material
    0x0014510C: wedge_offset_z
    0x0014510D: wedge_origin_offset_x
    0x0014510E: wedge_time_delay
    0x0014510F: wedge_name
    0x00145110: wedge_manufacturer_name
    0x00145111: wedge_description
    0x00145112: nominal_beam_angle
    0x00145113: wedge_offset_x
    0x00145114: wedge_offset_y
    0x00145115: wedge_total_length
    0x00145116: wedge_in_contact_length
    0x00145117: wedge_front_gap
    0x00145118: wedge_total_height
    0x00145119: wedge_front_height
    0x0014511A: wedge_rear_height
    0x0014511B: wedge_total_width
    0x0014511C: wedge_in_contact_width
    0x0014511D: wedge_chamfer_height
    0x0014511E: wedge_curve
    0x0014511F: radius_along_the_wedge
    0x00180010: contrast_bolus_agent
    0x00180012: contrast_bolus_agent_sequence
    0x00180013: contrast_bolus_t1_relaxivity
    0x00180014: contrast_bolus_administration_route_sequence
    0x00180015: body_part_examined
    0x00180020: scanning_sequence
    0x00180021: sequence_variant
    0x00180022: scan_options
    0x00180023: mr_acquisition_type
    0x00180024: sequence_name
    0x00180025: angio_flag
    0x00180026: intervention_drug_information_sequence
    0x00180027: intervention_drug_stop_time
    0x00180028: intervention_drug_dose
    0x00180029: intervention_drug_code_sequence
    0x0018002A: additional_drug_sequence
    0x00180030: radionuclide
    0x00180031: radiopharmaceutical
    0x00180032: energy_window_centerline
    0x00180033: energy_window_total_width
    0x00180034: intervention_drug_name
    0x00180035: intervention_drug_start_time
    0x00180036: intervention_sequence
    0x00180037: therapy_type
    0x00180038: intervention_status
    0x00180039: therapy_description
    0x0018003A: intervention_description
    0x00180040: cine_rate
    0x00180042: initial_cine_run_state
    0x00180050: slice_thickness
    0x00180060: kvp
    0x00180070: counts_accumulated
    0x00180071: acquisition_termination_condition
    0x00180072: effective_duration
    0x00180073: acquisition_start_condition
    0x00180074: acquisition_start_condition_data
    0x00180075: acquisition_termination_condition_data
    0x00180080: repetition_time
    0x00180081: echo_time
    0x00180082: inversion_time
    0x00180083: number_of_averages
    0x00180084: imaging_frequency
    0x00180085: imaged_nucleus
    0x00180086: echo_number_s_
    0x00180087: magnetic_field_strength
    0x00180088: spacing_between_slices
    0x00180089: number_of_phase_encoding_steps
    0x00180090: data_collection_diameter
    0x00180091: echo_train_length
    0x00180093: percent_sampling
    0x00180094: percent_phase_field_of_view
    0x00180095: pixel_bandwidth
    0x00181000: device_serial_number
    0x00181002: device_uid
    0x00181003: device_id
    0x00181004: plate_id
    0x00181005: generator_id
    0x00181006: grid_id
    0x00181007: cassette_id
    0x00181008: gantry_id
    0x00181010: secondary_capture_device_id
    0x00181011: hardcopy_creation_device_id
    0x00181012: date_of_secondary_capture
    0x00181014: time_of_secondary_capture
    0x00181016: secondary_capture_device_manufacturer
    0x00181017: hardcopy_device_manufacturer
    0x00181018: secondary_capture_device_manufacturer_s_model_name
    0x00181019: secondary_capture_device_software_versions
    0x0018101A: hardcopy_device_software_version
    0x0018101B: hardcopy_device_manufacturer_s_model_name
    0x00181020: software_version_s_
    0x00181022: video_image_format_acquired
    0x00181023: digital_image_format_acquired
    0x00181030: protocol_name
    0x00181040: contrast_bolus_route
    0x00181041: contrast_bolus_volume
    0x00181042: contrast_bolus_start_time
    0x00181043: contrast_bolus_stop_time
    0x00181044: contrast_bolus_total_dose
    0x00181045: syringe_counts
    0x00181046: contrast_flow_rate
    0x00181047: contrast_flow_duration
    0x00181048: contrast_bolus_ingredient
    0x00181049: contrast_bolus_ingredient_concentration
    0x00181050: spatial_resolution
    0x00181060: trigger_time
    0x00181061: trigger_source_or_type
    0x00181062: nominal_interval
    0x00181063: frame_time
    0x00181064: cardiac_framing_type
    0x00181065: frame_time_vector
    0x00181066: frame_delay
    0x00181067: image_trigger_delay
    0x00181068: multiplex_group_time_offset
    0x00181069: trigger_time_offset
    0x0018106A: synchronization_trigger
    0x0018106C: synchronization_channel
    0x0018106E: trigger_sample_position
    0x00181070: radiopharmaceutical_route
    0x00181071: radiopharmaceutical_volume
    0x00181072: radiopharmaceutical_start_time
    0x00181073: radiopharmaceutical_stop_time
    0x00181074: radionuclide_total_dose
    0x00181075: radionuclide_half_life
    0x00181076: radionuclide_positron_fraction
    0x00181077: radiopharmaceutical_specific_activity
    0x00181078: radiopharmaceutical_start_datetime
    0x00181079: radiopharmaceutical_stop_datetime
    0x00181080: beat_rejection_flag
    0x00181081: low_r_r_value
    0x00181082: high_r_r_value
    0x00181083: intervals_acquired
    0x00181084: intervals_rejected
    0x00181085: pvc_rejection
    0x00181086: skip_beats
    0x00181088: heart_rate
    0x00181090: cardiac_number_of_images
    0x00181094: trigger_window
    0x00181100: reconstruction_diameter
    0x00181110: distance_source_to_detector
    0x00181111: distance_source_to_patient
    0x00181114: estimated_radiographic_magnification_factor
    0x00181120: gantry_detector_tilt
    0x00181121: gantry_detector_slew
    0x00181130: table_height
    0x00181131: table_traverse
    0x00181134: table_motion
    0x00181135: table_vertical_increment
    0x00181136: table_lateral_increment
    0x00181137: table_longitudinal_increment
    0x00181138: table_angle
    0x0018113A: table_type
    0x00181140: rotation_direction
    0x00181141: angular_position
    0x00181142: radial_position
    0x00181143: scan_arc
    0x00181144: angular_step
    0x00181145: center_of_rotation_offset
    0x00181146: rotation_offset
    0x00181147: field_of_view_shape
    0x00181149: field_of_view_dimension_s_
    0x00181150: exposure_time
    0x00181151: x_ray_tube_current
    0x00181152: exposure
    0x00181153: exposure_in__as
    0x00181154: average_pulse_width
    0x00181155: radiation_setting
    0x00181156: rectification_type
    0x0018115A: radiation_mode
    0x0018115E: image_and_fluoroscopy_area_dose_product
    0x00181160: filter_type
    0x00181161: type_of_filters
    0x00181162: intensifier_size
    0x00181164: imager_pixel_spacing
    0x00181166: grid
    0x00181170: generator_power
    0x00181180: collimator_grid_name
    0x00181181: collimator_type
    0x00181182: focal_distance
    0x00181183: x_focus_center
    0x00181184: y_focus_center
    0x00181190: focal_spot_s_
    0x00181191: anode_target_material
    0x001811A0: body_part_thickness
    0x001811A2: compression_force
    0x001811A4: paddle_description
    0x00181200: date_of_last_calibration
    0x00181201: time_of_last_calibration
    0x00181202: datetime_of_last_calibration
    0x00181210: convolution_kernel
    0x00181240: upper_lower_pixel_values
    0x00181242: actual_frame_duration
    0x00181243: count_rate
    0x00181244: preferred_playback_sequencing
    0x00181250: receive_coil_name
    0x00181251: transmit_coil_name
    0x00181260: plate_type
    0x00181261: phosphor_type
    0x00181271: water_equivalent_diameter
    0x00181272: water_equivalent_diameter_calculation_method_code_sequence
    0x00181300: scan_velocity
    0x00181301: whole_body_technique
    0x00181302: scan_length
    0x00181310: acquisition_matrix
    0x00181312: in_plane_phase_encoding_direction
    0x00181314: flip_angle
    0x00181315: variable_flip_angle_flag
    0x00181316: sar
    0x00181318: db_dt
    0x00181320: b1rms
    0x00181400: acquisition_device_processing_description
    0x00181401: acquisition_device_processing_code
    0x00181402: cassette_orientation
    0x00181403: cassette_size
    0x00181404: exposures_on_plate
    0x00181405: relative_x_ray_exposure
    0x00181411: exposure_index
    0x00181412: target_exposure_index
    0x00181413: deviation_index
    0x00181450: column_angulation
    0x00181460: tomo_layer_height
    0x00181470: tomo_angle
    0x00181480: tomo_time
    0x00181490: tomo_type
    0x00181491: tomo_class
    0x00181495: number_of_tomosynthesis_source_images
    0x00181500: positioner_motion
    0x00181508: positioner_type
    0x00181510: positioner_primary_angle
    0x00181511: positioner_secondary_angle
    0x00181520: positioner_primary_angle_increment
    0x00181521: positioner_secondary_angle_increment
    0x00181530: detector_primary_angle
    0x00181531: detector_secondary_angle
    0x00181600: shutter_shape
    0x00181602: shutter_left_vertical_edge
    0x00181604: shutter_right_vertical_edge
    0x00181606: shutter_upper_horizontal_edge
    0x00181608: shutter_lower_horizontal_edge
    0x00181610: center_of_circular_shutter
    0x00181612: radius_of_circular_shutter
    0x00181620: vertices_of_the_polygonal_shutter
    0x00181622: shutter_presentation_value
    0x00181623: shutter_overlay_group
    0x00181624: shutter_presentation_color_cielab_value
    0x00181700: collimator_shape
    0x00181702: collimator_left_vertical_edge
    0x00181704: collimator_right_vertical_edge
    0x00181706: collimator_upper_horizontal_edge
    0x00181708: collimator_lower_horizontal_edge
    0x00181710: center_of_circular_collimator
    0x00181712: radius_of_circular_collimator
    0x00181720: vertices_of_the_polygonal_collimator
    0x00181800: acquisition_time_synchronized
    0x00181801: time_source
    0x00181802: time_distribution_protocol
    0x00181803: ntp_source_address
    0x00182001: page_number_vector
    0x00182002: frame_label_vector
    0x00182003: frame_primary_angle_vector
    0x00182004: frame_secondary_angle_vector
    0x00182005: slice_location_vector
    0x00182006: display_window_label_vector
    0x00182010: nominal_scanned_pixel_spacing
    0x00182020: digitizing_device_transport_direction
    0x00182030: rotation_of_scanned_film
    0x00182041: biopsy_target_sequence
    0x00182042: target_uid
    0x00182043: localizing_cursor_position
    0x00182044: calculated_target_position
    0x00182045: target_label
    0x00182046: displayed_z_value
    0x00183100: ivus_acquisition
    0x00183101: ivus_pullback_rate
    0x00183102: ivus_gated_rate
    0x00183103: ivus_pullback_start_frame_number
    0x00183104: ivus_pullback_stop_frame_number
    0x00183105: lesion_number
    0x00184000: acquisition_comments
    0x00185000: output_power
    0x00185010: transducer_data
    0x00185012: focus_depth
    0x00185020: processing_function
    0x00185021: postprocessing_function
    0x00185022: mechanical_index
    0x00185024: bone_thermal_index
    0x00185026: cranial_thermal_index
    0x00185027: soft_tissue_thermal_index
    0x00185028: soft_tissue_focus_thermal_index
    0x00185029: soft_tissue_surface_thermal_index
    0x00185030: dynamic_range
    0x00185040: total_gain
    0x00185050: depth_of_scan_field
    0x00185100: patient_position
    0x00185101: view_position
    0x00185104: projection_eponymous_name_code_sequence
    0x00185210: image_transformation_matrix
    0x00185212: image_translation_vector
    0x00186000: sensitivity
    0x00186011: sequence_of_ultrasound_regions
    0x00186012: region_spatial_format
    0x00186014: region_data_type
    0x00186016: region_flags
    0x00186018: region_location_min_x0
    0x0018601A: region_location_min_y0
    0x0018601C: region_location_max_x1
    0x0018601E: region_location_max_y1
    0x00186020: reference_pixel_x0
    0x00186022: reference_pixel_y0
    0x00186024: physical_units_x_direction
    0x00186026: physical_units_y_direction
    0x00186028: reference_pixel_physical_value_x
    0x0018602A: reference_pixel_physical_value_y
    0x0018602C: physical_delta_x
    0x0018602E: physical_delta_y
    0x00186030: transducer_frequency
    0x00186031: transducer_type
    0x00186032: pulse_repetition_frequency
    0x00186034: doppler_correction_angle
    0x00186036: steering_angle
    0x00186038: doppler_sample_volume_x_position__retired_
    0x00186039: doppler_sample_volume_x_position
    0x0018603A: doppler_sample_volume_y_position__retired_
    0x0018603B: doppler_sample_volume_y_position
    0x0018603C: tm_line_position_x0__retired_
    0x0018603D: tm_line_position_x0
    0x0018603E: tm_line_position_y0__retired_
    0x0018603F: tm_line_position_y0
    0x00186040: tm_line_position_x1__retired_
    0x00186041: tm_line_position_x1
    0x00186042: tm_line_position_y1__retired_
    0x00186043: tm_line_position_y1
    0x00186044: pixel_component_organization
    0x00186046: pixel_component_mask
    0x00186048: pixel_component_range_start
    0x0018604A: pixel_component_range_stop
    0x0018604C: pixel_component_physical_units
    0x0018604E: pixel_component_data_type
    0x00186050: number_of_table_break_points
    0x00186052: table_of_x_break_points
    0x00186054: table_of_y_break_points
    0x00186056: number_of_table_entries
    0x00186058: table_of_pixel_values
    0x0018605A: table_of_parameter_values
    0x00186060: r_wave_time_vector
    0x00187000: detector_conditions_nominal_flag
    0x00187001: detector_temperature
    0x00187004: detector_type
    0x00187005: detector_configuration
    0x00187006: detector_description
    0x00187008: detector_mode
    0x0018700A: detector_id
    0x0018700C: date_of_last_detector_calibration
    0x0018700E: time_of_last_detector_calibration
    0x00187010: exposures_on_detector_since_last_calibration
    0x00187011: exposures_on_detector_since_manufactured
    0x00187012: detector_time_since_last_exposure
    0x00187014: detector_active_time
    0x00187016: detector_activation_offset_from_exposure
    0x0018701A: detector_binning
    0x00187020: detector_element_physical_size
    0x00187022: detector_element_spacing
    0x00187024: detector_active_shape
    0x00187026: detector_active_dimension_s_
    0x00187028: detector_active_origin
    0x0018702A: detector_manufacturer_name
    0x0018702B: detector_manufacturer_s_model_name
    0x00187030: field_of_view_origin
    0x00187032: field_of_view_rotation
    0x00187034: field_of_view_horizontal_flip
    0x00187036: pixel_data_area_origin_relative_to_fov
    0x00187038: pixel_data_area_rotation_angle_relative_to_fov
    0x00187040: grid_absorbing_material
    0x00187041: grid_spacing_material
    0x00187042: grid_thickness
    0x00187044: grid_pitch
    0x00187046: grid_aspect_ratio
    0x00187048: grid_period
    0x0018704C: grid_focal_distance
    0x00187050: filter_material
    0x00187052: filter_thickness_minimum
    0x00187054: filter_thickness_maximum
    0x00187056: filter_beam_path_length_minimum
    0x00187058: filter_beam_path_length_maximum
    0x00187060: exposure_control_mode
    0x00187062: exposure_control_mode_description
    0x00187064: exposure_status
    0x00187065: phototimer_setting
    0x00188150: exposure_time_in__s
    0x00188151: x_ray_tube_current_in__a
    0x00189004: content_qualification
    0x00189005: pulse_sequence_name
    0x00189006: mr_imaging_modifier_sequence
    0x00189008: echo_pulse_sequence
    0x00189009: inversion_recovery
    0x00189010: flow_compensation
    0x00189011: multiple_spin_echo
    0x00189012: multi_planar_excitation
    0x00189014: phase_contrast
    0x00189015: time_of_flight_contrast
    0x00189016: spoiling
    0x00189017: steady_state_pulse_sequence
    0x00189018: echo_planar_pulse_sequence
    0x00189019: tag_angle_first_axis
    0x00189020: magnetization_transfer
    0x00189021: t2_preparation
    0x00189022: blood_signal_nulling
    0x00189024: saturation_recovery
    0x00189025: spectrally_selected_suppression
    0x00189026: spectrally_selected_excitation
    0x00189027: spatial_pre_saturation
    0x00189028: tagging
    0x00189029: oversampling_phase
    0x00189030: tag_spacing_first_dimension
    0x00189032: geometry_of_k_space_traversal
    0x00189033: segmented_k_space_traversal
    0x00189034: rectilinear_phase_encode_reordering
    0x00189035: tag_thickness
    0x00189036: partial_fourier_direction
    0x00189037: cardiac_synchronization_technique
    0x00189041: receive_coil_manufacturer_name
    0x00189042: mr_receive_coil_sequence
    0x00189043: receive_coil_type
    0x00189044: quadrature_receive_coil
    0x00189045: multi_coil_definition_sequence
    0x00189046: multi_coil_configuration
    0x00189047: multi_coil_element_name
    0x00189048: multi_coil_element_used
    0x00189049: mr_transmit_coil_sequence
    0x00189050: transmit_coil_manufacturer_name
    0x00189051: transmit_coil_type
    0x00189052: spectral_width
    0x00189053: chemical_shift_reference
    0x00189054: volume_localization_technique
    0x00189058: mr_acquisition_frequency_encoding_steps
    0x00189059: de_coupling
    0x00189060: de_coupled_nucleus
    0x00189061: de_coupling_frequency
    0x00189062: de_coupling_method
    0x00189063: de_coupling_chemical_shift_reference
    0x00189064: k_space_filtering
    0x00189065: time_domain_filtering
    0x00189066: number_of_zero_fills
    0x00189067: baseline_correction
    0x00189069: parallel_reduction_factor_in_plane
    0x00189070: cardiac_r_r_interval_specified
    0x00189073: acquisition_duration
    0x00189074: frame_acquisition_datetime
    0x00189075: diffusion_directionality
    0x00189076: diffusion_gradient_direction_sequence
    0x00189077: parallel_acquisition
    0x00189078: parallel_acquisition_technique
    0x00189079: inversion_times
    0x00189080: metabolite_map_description
    0x00189081: partial_fourier
    0x00189082: effective_echo_time
    0x00189083: metabolite_map_code_sequence
    0x00189084: chemical_shift_sequence
    0x00189085: cardiac_signal_source
    0x00189087: diffusion_b_value
    0x00189089: diffusion_gradient_orientation
    0x00189090: velocity_encoding_direction
    0x00189091: velocity_encoding_minimum_value
    0x00189092: velocity_encoding_acquisition_sequence
    0x00189093: number_of_k_space_trajectories
    0x00189094: coverage_of_k_space
    0x00189095: spectroscopy_acquisition_phase_rows
    0x00189096: parallel_reduction_factor_in_plane__retired_
    0x00189098: transmitter_frequency
    0x00189100: resonant_nucleus
    0x00189101: frequency_correction
    0x00189103: mr_spectroscopy_fov_geometry_sequence
    0x00189104: slab_thickness
    0x00189105: slab_orientation
    0x00189106: mid_slab_position
    0x00189107: mr_spatial_saturation_sequence
    0x00189112: mr_timing_and_related_parameters_sequence
    0x00189114: mr_echo_sequence
    0x00189115: mr_modifier_sequence
    0x00189117: mr_diffusion_sequence
    0x00189118: cardiac_synchronization_sequence
    0x00189119: mr_averages_sequence
    0x00189125: mr_fov_geometry_sequence
    0x00189126: volume_localization_sequence
    0x00189127: spectroscopy_acquisition_data_columns
    0x00189147: diffusion_anisotropy_type
    0x00189151: frame_reference_datetime
    0x00189152: mr_metabolite_map_sequence
    0x00189155: parallel_reduction_factor_out_of_plane
    0x00189159: spectroscopy_acquisition_out_of_plane_phase_steps
    0x00189166: bulk_motion_status
    0x00189168: parallel_reduction_factor_second_in_plane
    0x00189169: cardiac_beat_rejection_technique
    0x00189170: respiratory_motion_compensation_technique
    0x00189171: respiratory_signal_source
    0x00189172: bulk_motion_compensation_technique
    0x00189173: bulk_motion_signal_source
    0x00189174: applicable_safety_standard_agency
    0x00189175: applicable_safety_standard_description
    0x00189176: operating_mode_sequence
    0x00189177: operating_mode_type
    0x00189178: operating_mode
    0x00189179: specific_absorption_rate_definition
    0x00189180: gradient_output_type
    0x00189181: specific_absorption_rate_value
    0x00189182: gradient_output
    0x00189183: flow_compensation_direction
    0x00189184: tagging_delay
    0x00189185: respiratory_motion_compensation_technique_description
    0x00189186: respiratory_signal_source_id
    0x00189195: chemical_shift_minimum_integration_limit_in_hz
    0x00189196: chemical_shift_maximum_integration_limit_in_hz
    0x00189197: mr_velocity_encoding_sequence
    0x00189198: first_order_phase_correction
    0x00189199: water_referenced_phase_correction
    0x00189200: mr_spectroscopy_acquisition_type
    0x00189214: respiratory_cycle_position
    0x00189217: velocity_encoding_maximum_value
    0x00189218: tag_spacing_second_dimension
    0x00189219: tag_angle_second_axis
    0x00189220: frame_acquisition_duration
    0x00189226: mr_image_frame_type_sequence
    0x00189227: mr_spectroscopy_frame_type_sequence
    0x00189231: mr_acquisition_phase_encoding_steps_in_plane
    0x00189232: mr_acquisition_phase_encoding_steps_out_of_plane
    0x00189234: spectroscopy_acquisition_phase_columns
    0x00189236: cardiac_cycle_position
    0x00189239: specific_absorption_rate_sequence
    0x00189240: rf_echo_train_length
    0x00189241: gradient_echo_train_length
    0x00189250: arterial_spin_labeling_contrast
    0x00189251: mr_arterial_spin_labeling_sequence
    0x00189252: asl_technique_description
    0x00189253: asl_slab_number
    0x00189254: asl_slab_thickness
    0x00189255: asl_slab_orientation
    0x00189256: asl_mid_slab_position
    0x00189257: asl_context
    0x00189258: asl_pulse_train_duration
    0x00189259: asl_crusher_flag
    0x0018925A: asl_crusher_flow_limit
    0x0018925B: asl_crusher_description
    0x0018925C: asl_bolus_cut_off_flag
    0x0018925D: asl_bolus_cut_off_timing_sequence
    0x0018925E: asl_bolus_cut_off_technique
    0x0018925F: asl_bolus_cut_off_delay_time
    0x00189260: asl_slab_sequence
    0x00189295: chemical_shift_minimum_integration_limit_in_ppm
    0x00189296: chemical_shift_maximum_integration_limit_in_ppm
    0x00189297: water_reference_acquisition
    0x00189298: echo_peak_position
    0x00189301: ct_acquisition_type_sequence
    0x00189302: acquisition_type
    0x00189303: tube_angle
    0x00189304: ct_acquisition_details_sequence
    0x00189305: revolution_time
    0x00189306: single_collimation_width
    0x00189307: total_collimation_width
    0x00189308: ct_table_dynamics_sequence
    0x00189309: table_speed
    0x00189310: table_feed_per_rotation
    0x00189311: spiral_pitch_factor
    0x00189312: ct_geometry_sequence
    0x00189313: data_collection_center__patient_
    0x00189314: ct_reconstruction_sequence
    0x00189315: reconstruction_algorithm
    0x00189316: convolution_kernel_group
    0x00189317: reconstruction_field_of_view
    0x00189318: reconstruction_target_center__patient_
    0x00189319: reconstruction_angle
    0x00189320: image_filter
    0x00189321: ct_exposure_sequence
    0x00189322: reconstruction_pixel_spacing
    0x00189323: exposure_modulation_type
    0x00189324: estimated_dose_saving
    0x00189325: ct_x_ray_details_sequence
    0x00189326: ct_position_sequence
    0x00189327: table_position
    0x00189328: exposure_time_in_ms
    0x00189329: ct_image_frame_type_sequence
    0x00189330: x_ray_tube_current_in_ma
    0x00189332: exposure_in_mas
    0x00189333: constant_volume_flag
    0x00189334: fluoroscopy_flag
    0x00189335: distance_source_to_data_collection_center
    0x00189337: contrast_bolus_agent_number
    0x00189338: contrast_bolus_ingredient_code_sequence
    0x00189340: contrast_administration_profile_sequence
    0x00189341: contrast_bolus_usage_sequence
    0x00189342: contrast_bolus_agent_administered
    0x00189343: contrast_bolus_agent_detected
    0x00189344: contrast_bolus_agent_phase
    0x00189345: ctdivol
    0x00189346: ctdi_phantom_type_code_sequence
    0x00189351: calcium_scoring_mass_factor_patient
    0x00189352: calcium_scoring_mass_factor_device
    0x00189353: energy_weighting_factor
    0x00189360: ct_additional_x_ray_source_sequence
    0x00189401: projection_pixel_calibration_sequence
    0x00189402: distance_source_to_isocenter
    0x00189403: distance_object_to_table_top
    0x00189404: object_pixel_spacing_in_center_of_beam
    0x00189405: positioner_position_sequence
    0x00189406: table_position_sequence
    0x00189407: collimator_shape_sequence
    0x00189410: planes_in_acquisition
    0x00189412: xa_xrf_frame_characteristics_sequence
    0x00189417: frame_acquisition_sequence
    0x00189420: x_ray_receptor_type
    0x00189423: acquisition_protocol_name
    0x00189424: acquisition_protocol_description
    0x00189425: contrast_bolus_ingredient_opaque
    0x00189426: distance_receptor_plane_to_detector_housing
    0x00189427: intensifier_active_shape
    0x00189428: intensifier_active_dimension_s_
    0x00189429: physical_detector_size
    0x00189430: position_of_isocenter_projection
    0x00189432: field_of_view_sequence
    0x00189433: field_of_view_description
    0x00189434: exposure_control_sensing_regions_sequence
    0x00189435: exposure_control_sensing_region_shape
    0x00189436: exposure_control_sensing_region_left_vertical_edge
    0x00189437: exposure_control_sensing_region_right_vertical_edge
    0x00189438: exposure_control_sensing_region_upper_horizontal_edge
    0x00189439: exposure_control_sensing_region_lower_horizontal_edge
    0x00189440: center_of_circular_exposure_control_sensing_region
    0x00189441: radius_of_circular_exposure_control_sensing_region
    0x00189442: vertices_of_the_polygonal_exposure_control_sensing_region
    0x00189447: column_angulation__patient_
    0x00189449: beam_angle
    0x00189451: frame_detector_parameters_sequence
    0x00189452: calculated_anatomy_thickness
    0x00189455: calibration_sequence
    0x00189456: object_thickness_sequence
    0x00189457: plane_identification
    0x00189461: field_of_view_dimension_s__in_float
    0x00189462: isocenter_reference_system_sequence
    0x00189463: positioner_isocenter_primary_angle
    0x00189464: positioner_isocenter_secondary_angle
    0x00189465: positioner_isocenter_detector_rotation_angle
    0x00189466: table_x_position_to_isocenter
    0x00189467: table_y_position_to_isocenter
    0x00189468: table_z_position_to_isocenter
    0x00189469: table_horizontal_rotation_angle
    0x00189470: table_head_tilt_angle
    0x00189471: table_cradle_tilt_angle
    0x00189472: frame_display_shutter_sequence
    0x00189473: acquired_image_area_dose_product
    0x00189474: c_arm_positioner_tabletop_relationship
    0x00189476: x_ray_geometry_sequence
    0x00189477: irradiation_event_identification_sequence
    0x00189504: x_ray_3d_frame_type_sequence
    0x00189506: contributing_sources_sequence
    0x00189507: x_ray_3d_acquisition_sequence
    0x00189508: primary_positioner_scan_arc
    0x00189509: secondary_positioner_scan_arc
    0x00189510: primary_positioner_scan_start_angle
    0x00189511: secondary_positioner_scan_start_angle
    0x00189514: primary_positioner_increment
    0x00189515: secondary_positioner_increment
    0x00189516: start_acquisition_datetime
    0x00189517: end_acquisition_datetime
    0x00189518: primary_positioner_increment_sign
    0x00189519: secondary_positioner_increment_sign
    0x00189524: application_name
    0x00189525: application_version
    0x00189526: application_manufacturer
    0x00189527: algorithm_type
    0x00189528: algorithm_description
    0x00189530: x_ray_3d_reconstruction_sequence
    0x00189531: reconstruction_description
    0x00189538: per_projection_acquisition_sequence
    0x00189541: detector_position_sequence
    0x00189542: x_ray_acquisition_dose_sequence
    0x00189543: x_ray_source_isocenter_primary_angle
    0x00189544: x_ray_source_isocenter_secondary_angle
    0x00189545: breast_support_isocenter_primary_angle
    0x00189546: breast_support_isocenter_secondary_angle
    0x00189547: breast_support_x_position_to_isocenter
    0x00189548: breast_support_y_position_to_isocenter
    0x00189549: breast_support_z_position_to_isocenter
    0x00189550: detector_isocenter_primary_angle
    0x00189551: detector_isocenter_secondary_angle
    0x00189552: detector_x_position_to_isocenter
    0x00189553: detector_y_position_to_isocenter
    0x00189554: detector_z_position_to_isocenter
    0x00189555: x_ray_grid_sequence
    0x00189556: x_ray_filter_sequence
    0x00189557: detector_active_area_tlhc_position
    0x00189558: detector_active_area_orientation
    0x00189559: positioner_primary_angle_direction
    0x00189601: diffusion_b_matrix_sequence
    0x00189602: diffusion_b_value_xx
    0x00189603: diffusion_b_value_xy
    0x00189604: diffusion_b_value_xz
    0x00189605: diffusion_b_value_yy
    0x00189606: diffusion_b_value_yz
    0x00189607: diffusion_b_value_zz
    0x00189621: functional_mr_sequence
    0x00189622: functional_settling_phase_frames_present
    0x00189623: functional_sync_pulse
    0x00189624: settling_phase_frame
    0x00189701: decay_correction_datetime
    0x00189715: start_density_threshold
    0x00189716: start_relative_density_difference_threshold
    0x00189717: start_cardiac_trigger_count_threshold
    0x00189718: start_respiratory_trigger_count_threshold
    0x00189719: termination_counts_threshold
    0x00189720: termination_density_threshold
    0x00189721: termination_relative_density_threshold
    0x00189722: termination_time_threshold
    0x00189723: termination_cardiac_trigger_count_threshold
    0x00189724: termination_respiratory_trigger_count_threshold
    0x00189725: detector_geometry
    0x00189726: transverse_detector_separation
    0x00189727: axial_detector_dimension
    0x00189729: radiopharmaceutical_agent_number
    0x00189732: pet_frame_acquisition_sequence
    0x00189733: pet_detector_motion_details_sequence
    0x00189734: pet_table_dynamics_sequence
    0x00189735: pet_position_sequence
    0x00189736: pet_frame_correction_factors_sequence
    0x00189737: radiopharmaceutical_usage_sequence
    0x00189738: attenuation_correction_source
    0x00189739: number_of_iterations
    0x00189740: number_of_subsets
    0x00189749: pet_reconstruction_sequence
    0x00189751: pet_frame_type_sequence
    0x00189755: time_of_flight_information_used
    0x00189756: reconstruction_type
    0x00189758: decay_corrected
    0x00189759: attenuation_corrected
    0x00189760: scatter_corrected
    0x00189761: dead_time_corrected
    0x00189762: gantry_motion_corrected
    0x00189763: patient_motion_corrected
    0x00189764: count_loss_normalization_corrected
    0x00189765: randoms_corrected
    0x00189766: non_uniform_radial_sampling_corrected
    0x00189767: sensitivity_calibrated
    0x00189768: detector_normalization_correction
    0x00189769: iterative_reconstruction_method
    0x00189770: attenuation_correction_temporal_relationship
    0x00189771: patient_physiological_state_sequence
    0x00189772: patient_physiological_state_code_sequence
    0x00189801: depth_s__of_focus
    0x00189803: excluded_intervals_sequence
    0x00189804: exclusion_start_datetime
    0x00189805: exclusion_duration
    0x00189806: us_image_description_sequence
    0x00189807: image_data_type_sequence
    0x00189808: data_type
    0x00189809: transducer_scan_pattern_code_sequence
    0x0018980B: aliased_data_type
    0x0018980C: position_measuring_device_used
    0x0018980D: transducer_geometry_code_sequence
    0x0018980E: transducer_beam_steering_code_sequence
    0x0018980F: transducer_application_code_sequence
    0x00189810: zero_velocity_pixel_value
    0x0018A001: contributing_equipment_sequence
    0x0018A002: contribution_datetime
    0x0018A003: contribution_description
    0x0020000D: study_instance_uid
    0x0020000E: series_instance_uid
    0x00200010: study_id
    0x00200011: series_number
    0x00200012: acquisition_number
    0x00200013: instance_number
    0x00200014: isotope_number
    0x00200015: phase_number
    0x00200016: interval_number
    0x00200017: time_slot_number
    0x00200018: angle_number
    0x00200019: item_number
    0x00200020: patient_orientation
    0x00200022: overlay_number
    0x00200024: curve_number
    0x00200026: lut_number
    0x00200030: image_position
    0x00200032: image_position__patient_
    0x00200035: image_orientation
    0x00200037: image_orientation__patient_
    0x00200050: location
    0x00200052: frame_of_reference_uid
    0x00200060: laterality
    0x00200062: image_laterality
    0x00200070: image_geometry_type
    0x00200080: masking_image
    0x002000AA: report_number
    0x00200100: temporal_position_identifier
    0x00200105: number_of_temporal_positions
    0x00200110: temporal_resolution
    0x00200200: synchronization_frame_of_reference_uid
    0x00200242: sop_instance_uid_of_concatenation_source
    0x00201000: series_in_study
    0x00201001: acquisitions_in_series
    0x00201002: images_in_acquisition
    0x00201003: images_in_series
    0x00201004: acquisitions_in_study
    0x00201005: images_in_study
    0x00201020: reference
    0x00201040: position_reference_indicator
    0x00201041: slice_location
    0x00201070: other_study_numbers
    0x00201200: number_of_patient_related_studies
    0x00201202: number_of_patient_related_series
    0x00201204: number_of_patient_related_instances
    0x00201206: number_of_study_related_series
    0x00201208: number_of_study_related_instances
    0x00201209: number_of_series_related_instances
    0x00203401: modifying_device_id
    0x00203402: modified_image_id
    0x00203403: modified_image_date
    0x00203404: modifying_device_manufacturer
    0x00203405: modified_image_time
    0x00203406: modified_image_description
    0x00204000: image_comments
    0x00205000: original_image_identification
    0x00205002: original_image_identification_nomenclature
    0x00209056: stack_id
    0x00209057: in_stack_position_number
    0x00209071: frame_anatomy_sequence
    0x00209072: frame_laterality
    0x00209111: frame_content_sequence
    0x00209113: plane_position_sequence
    0x00209116: plane_orientation_sequence
    0x00209128: temporal_position_index
    0x00209153: nominal_cardiac_trigger_delay_time
    0x00209154: nominal_cardiac_trigger_time_prior_to_r_peak
    0x00209155: actual_cardiac_trigger_time_prior_to_r_peak
    0x00209156: frame_acquisition_number
    0x00209157: dimension_index_values
    0x00209158: frame_comments
    0x00209161: concatenation_uid
    0x00209162: in_concatenation_number
    0x00209163: in_concatenation_total_number
    0x00209164: dimension_organization_uid
    0x00209165: dimension_index_pointer
    0x00209167: functional_group_pointer
    0x00209170: unassigned_shared_converted_attributes_sequence
    0x00209171: unassigned_per_frame_converted_attributes_sequence
    0x00209172: conversion_source_attributes_sequence
    0x00209213: dimension_index_private_creator
    0x00209221: dimension_organization_sequence
    0x00209222: dimension_index_sequence
    0x00209228: concatenation_frame_offset_number
    0x00209238: functional_group_private_creator
    0x00209241: nominal_percentage_of_cardiac_phase
    0x00209245: nominal_percentage_of_respiratory_phase
    0x00209246: starting_respiratory_amplitude
    0x00209247: starting_respiratory_phase
    0x00209248: ending_respiratory_amplitude
    0x00209249: ending_respiratory_phase
    0x00209250: respiratory_trigger_type
    0x00209251: r_r_interval_time_nominal
    0x00209252: actual_cardiac_trigger_delay_time
    0x00209253: respiratory_synchronization_sequence
    0x00209254: respiratory_interval_time
    0x00209255: nominal_respiratory_trigger_delay_time
    0x00209256: respiratory_trigger_delay_threshold
    0x00209257: actual_respiratory_trigger_delay_time
    0x00209301: image_position__volume_
    0x00209302: image_orientation__volume_
    0x00209307: ultrasound_acquisition_geometry
    0x00209308: apex_position
    0x00209309: volume_to_transducer_mapping_matrix
    0x0020930A: volume_to_table_mapping_matrix
    0x0020930B: volume_to_transducer_relationship
    0x0020930C: patient_frame_of_reference_source
    0x0020930D: temporal_position_time_offset
    0x0020930E: plane_position__volume__sequence
    0x0020930F: plane_orientation__volume__sequence
    0x00209310: temporal_position_sequence
    0x00209311: dimension_organization_type
    0x00209312: volume_frame_of_reference_uid
    0x00209313: table_frame_of_reference_uid
    0x00209421: dimension_description_label
    0x00209450: patient_orientation_in_frame_sequence
    0x00209453: frame_label
    0x00209518: acquisition_index
    0x00209529: contributing_sop_instances_reference_sequence
    0x00209536: reconstruction_index
    0x00220001: light_path_filter_pass_through_wavelength
    0x00220002: light_path_filter_pass_band
    0x00220003: image_path_filter_pass_through_wavelength
    0x00220004: image_path_filter_pass_band
    0x00220005: patient_eye_movement_commanded
    0x00220006: patient_eye_movement_command_code_sequence
    0x00220007: spherical_lens_power
    0x00220008: cylinder_lens_power
    0x00220009: cylinder_axis
    0x0022000A: emmetropic_magnification
    0x0022000B: intra_ocular_pressure
    0x0022000C: horizontal_field_of_view
    0x0022000D: pupil_dilated
    0x0022000E: degree_of_dilation
    0x00220010: stereo_baseline_angle
    0x00220011: stereo_baseline_displacement
    0x00220012: stereo_horizontal_pixel_offset
    0x00220013: stereo_vertical_pixel_offset
    0x00220014: stereo_rotation
    0x00220015: acquisition_device_type_code_sequence
    0x00220016: illumination_type_code_sequence
    0x00220017: light_path_filter_type_stack_code_sequence
    0x00220018: image_path_filter_type_stack_code_sequence
    0x00220019: lenses_code_sequence
    0x0022001A: channel_description_code_sequence
    0x0022001B: refractive_state_sequence
    0x0022001C: mydriatic_agent_code_sequence
    0x0022001D: relative_image_position_code_sequence
    0x0022001E: camera_angle_of_view
    0x00220020: stereo_pairs_sequence
    0x00220021: left_image_sequence
    0x00220022: right_image_sequence
    0x00220028: stereo_pairs_present
    0x00220030: axial_length_of_the_eye
    0x00220031: ophthalmic_frame_location_sequence
    0x00220032: reference_coordinates
    0x00220035: depth_spatial_resolution
    0x00220036: maximum_depth_distortion
    0x00220037: along_scan_spatial_resolution
    0x00220038: maximum_along_scan_distortion
    0x00220039: ophthalmic_image_orientation
    0x00220041: depth_of_transverse_image
    0x00220042: mydriatic_agent_concentration_units_sequence
    0x00220048: across_scan_spatial_resolution
    0x00220049: maximum_across_scan_distortion
    0x0022004E: mydriatic_agent_concentration
    0x00220055: illumination_wave_length
    0x00220056: illumination_power
    0x00220057: illumination_bandwidth
    0x00220058: mydriatic_agent_sequence
    0x00221007: ophthalmic_axial_measurements_right_eye_sequence
    0x00221008: ophthalmic_axial_measurements_left_eye_sequence
    0x00221009: ophthalmic_axial_measurements_device_type
    0x00221010: ophthalmic_axial_length_measurements_type
    0x00221012: ophthalmic_axial_length_sequence
    0x00221019: ophthalmic_axial_length
    0x00221024: lens_status_code_sequence
    0x00221025: vitreous_status_code_sequence
    0x00221028: iol_formula_code_sequence
    0x00221029: iol_formula_detail
    0x00221033: keratometer_index
    0x00221035: source_of_ophthalmic_axial_length_code_sequence
    0x00221037: target_refraction
    0x00221039: refractive_procedure_occurred
    0x00221040: refractive_surgery_type_code_sequence
    0x00221044: ophthalmic_ultrasound_method_code_sequence
    0x00221050: ophthalmic_axial_length_measurements_sequence
    0x00221053: iol_power
    0x00221054: predicted_refractive_error
    0x00221059: ophthalmic_axial_length_velocity
    0x00221065: lens_status_description
    0x00221066: vitreous_status_description
    0x00221090: iol_power_sequence
    0x00221092: lens_constant_sequence
    0x00221093: iol_manufacturer
    0x00221094: lens_constant_description
    0x00221095: implant_name
    0x00221096: keratometry_measurement_type_code_sequence
    0x00221097: implant_part_number
    0x00221100: referenced_ophthalmic_axial_measurements_sequence
    0x00221101: ophthalmic_axial_length_measurements_segment_name_code_sequence
    0x00221103: refractive_error_before_refractive_surgery_code_sequence
    0x00221121: iol_power_for_exact_emmetropia
    0x00221122: iol_power_for_exact_target_refraction
    0x00221125: anterior_chamber_depth_definition_code_sequence
    0x00221127: lens_thickness_sequence
    0x00221128: anterior_chamber_depth_sequence
    0x00221130: lens_thickness
    0x00221131: anterior_chamber_depth
    0x00221132: source_of_lens_thickness_data_code_sequence
    0x00221133: source_of_anterior_chamber_depth_data_code_sequence
    0x00221134: source_of_refractive_measurements_sequence
    0x00221135: source_of_refractive_measurements_code_sequence
    0x00221140: ophthalmic_axial_length_measurement_modified
    0x00221150: ophthalmic_axial_length_data_source_code_sequence
    0x00221153: ophthalmic_axial_length_acquisition_method_code_sequence
    0x00221155: signal_to_noise_ratio
    0x00221159: ophthalmic_axial_length_data_source_description
    0x00221210: ophthalmic_axial_length_measurements_total_length_sequence
    0x00221211: ophthalmic_axial_length_measurements_segmental_length_sequence
    0x00221212: ophthalmic_axial_length_measurements_length_summation_sequence
    0x00221220: ultrasound_ophthalmic_axial_length_measurements_sequence
    0x00221225: optical_ophthalmic_axial_length_measurements_sequence
    0x00221230: ultrasound_selected_ophthalmic_axial_length_sequence
    0x00221250: ophthalmic_axial_length_selection_method_code_sequence
    0x00221255: optical_selected_ophthalmic_axial_length_sequence
    0x00221257: selected_segmental_ophthalmic_axial_length_sequence
    0x00221260: selected_total_ophthalmic_axial_length_sequence
    0x00221262: ophthalmic_axial_length_quality_metric_sequence
    0x00221265: ophthalmic_axial_length_quality_metric_type_code_sequence
    0x00221273: ophthalmic_axial_length_quality_metric_type_description
    0x00221300: intraocular_lens_calculations_right_eye_sequence
    0x00221310: intraocular_lens_calculations_left_eye_sequence
    0x00221330: referenced_ophthalmic_axial_length_measurement_qc_image_sequence
    0x00221415: ophthalmic_mapping_device_type
    0x00221420: acquisition_method_code_sequence
    0x00221423: acquisition_method_algorithm_sequence
    0x00221436: ophthalmic_thickness_map_type_code_sequence
    0x00221443: ophthalmic_thickness_mapping_normals_sequence
    0x00221445: retinal_thickness_definition_code_sequence
    0x00221450: pixel_value_mapping_to_coded_concept_sequence
    0x00221452: mapped_pixel_value
    0x00221454: pixel_value_mapping_explanation
    0x00221458: ophthalmic_thickness_map_quality_threshold_sequence
    0x00221460: ophthalmic_thickness_map_threshold_quality_rating
    0x00221463: anatomic_structure_reference_point
    0x00221465: registration_to_localizer_sequence
    0x00221466: registered_localizer_units
    0x00221467: registered_localizer_top_left_hand_corner
    0x00221468: registered_localizer_bottom_right_hand_corner
    0x00221470: ophthalmic_thickness_map_quality_rating_sequence
    0x00221472: relevant_opt_attributes_sequence
    0x00221512: transformation_method_code_sequence
    0x00221513: transformation_algorithm_sequence
    0x00221515: ophthalmic_axial_length_method
    0x00221517: ophthalmic_fov
    0x00221518: two_dimensional_to_three_dimensional_map_sequence
    0x00221525: wide_field_ophthalmic_photography_quality_rating_sequence
    0x00221526: wide_field_ophthalmic_photography_quality_threshold_sequence
    0x00221527: wide_field_ophthalmic_photography_threshold_quality_rating
    0x00221528: x_coordinates_center_pixel_view_angle
    0x00221529: y_coordinates_center_pixel_view_angle
    0x00221530: number_of_map_points
    0x00221531: two_dimensional_to_three_dimensional_map_data
    0x00240010: visual_field_horizontal_extent
    0x00240011: visual_field_vertical_extent
    0x00240012: visual_field_shape
    0x00240016: screening_test_mode_code_sequence
    0x00240018: maximum_stimulus_luminance
    0x00240020: background_luminance
    0x00240021: stimulus_color_code_sequence
    0x00240024: background_illumination_color_code_sequence
    0x00240025: stimulus_area
    0x00240028: stimulus_presentation_time
    0x00240032: fixation_sequence
    0x00240033: fixation_monitoring_code_sequence
    0x00240034: visual_field_catch_trial_sequence
    0x00240035: fixation_checked_quantity
    0x00240036: patient_not_properly_fixated_quantity
    0x00240037: presented_visual_stimuli_data_flag
    0x00240038: number_of_visual_stimuli
    0x00240039: excessive_fixation_losses_data_flag
    0x00240040: excessive_fixation_losses
    0x00240042: stimuli_retesting_quantity
    0x00240044: comments_on_patient_s_performance_of_visual_field
    0x00240045: false_negatives_estimate_flag
    0x00240046: false_negatives_estimate
    0x00240048: negative_catch_trials_quantity
    0x00240050: false_negatives_quantity
    0x00240051: excessive_false_negatives_data_flag
    0x00240052: excessive_false_negatives
    0x00240053: false_positives_estimate_flag
    0x00240054: false_positives_estimate
    0x00240055: catch_trials_data_flag
    0x00240056: positive_catch_trials_quantity
    0x00240057: test_point_normals_data_flag
    0x00240058: test_point_normals_sequence
    0x00240059: global_deviation_probability_normals_flag
    0x00240060: false_positives_quantity
    0x00240061: excessive_false_positives_data_flag
    0x00240062: excessive_false_positives
    0x00240063: visual_field_test_normals_flag
    0x00240064: results_normals_sequence
    0x00240065: age_corrected_sensitivity_deviation_algorithm_sequence
    0x00240066: global_deviation_from_normal
    0x00240067: generalized_defect_sensitivity_deviation_algorithm_sequence
    0x00240068: localized_deviation_from_normal
    0x00240069: patient_reliability_indicator
    0x00240070: visual_field_mean_sensitivity
    0x00240071: global_deviation_probability
    0x00240072: local_deviation_probability_normals_flag
    0x00240073: localized_deviation_probability
    0x00240074: short_term_fluctuation_calculated
    0x00240075: short_term_fluctuation
    0x00240076: short_term_fluctuation_probability_calculated
    0x00240077: short_term_fluctuation_probability
    0x00240078: corrected_localized_deviation_from_normal_calculated
    0x00240079: corrected_localized_deviation_from_normal
    0x00240080: corrected_localized_deviation_from_normal_probability_calculated
    0x00240081: corrected_localized_deviation_from_normal_probability
    0x00240083: global_deviation_probability_sequence
    0x00240085: localized_deviation_probability_sequence
    0x00240086: foveal_sensitivity_measured
    0x00240087: foveal_sensitivity
    0x00240088: visual_field_test_duration
    0x00240089: visual_field_test_point_sequence
    0x00240090: visual_field_test_point_x_coordinate
    0x00240091: visual_field_test_point_y_coordinate
    0x00240092: age_corrected_sensitivity_deviation_value
    0x00240093: stimulus_results
    0x00240094: sensitivity_value
    0x00240095: retest_stimulus_seen
    0x00240096: retest_sensitivity_value
    0x00240097: visual_field_test_point_normals_sequence
    0x00240098: quantified_defect
    0x00240100: age_corrected_sensitivity_deviation_probability_value
    0x00240102: generalized_defect_corrected_sensitivity_deviation_flag
    0x00240103: generalized_defect_corrected_sensitivity_deviation_value
    0x00240104: generalized_defect_corrected_sensitivity_deviation_probability_value
    0x00240105: minimum_sensitivity_value
    0x00240106: blind_spot_localized
    0x00240107: blind_spot_x_coordinate
    0x00240108: blind_spot_y_coordinate
    0x00240110: visual_acuity_measurement_sequence
    0x00240112: refractive_parameters_used_on_patient_sequence
    0x00240113: measurement_laterality
    0x00240114: ophthalmic_patient_clinical_information_left_eye_sequence
    0x00240115: ophthalmic_patient_clinical_information_right_eye_sequence
    0x00240117: foveal_point_normative_data_flag
    0x00240118: foveal_point_probability_value
    0x00240120: screening_baseline_measured
    0x00240122: screening_baseline_measured_sequence
    0x00240124: screening_baseline_type
    0x00240126: screening_baseline_value
    0x00240202: algorithm_source
    0x00240306: data_set_name
    0x00240307: data_set_version
    0x00240308: data_set_source
    0x00240309: data_set_description
    0x00240317: visual_field_test_reliability_global_index_sequence
    0x00240320: visual_field_global_results_index_sequence
    0x00240325: data_observation_sequence
    0x00240338: index_normals_flag
    0x00240341: index_probability
    0x00240344: index_probability_sequence
    0x00280002: samples_per_pixel
    0x00280003: samples_per_pixel_used
    0x00280004: photometric_interpretation
    0x00280005: image_dimensions
    0x00280006: planar_configuration
    0x00280008: number_of_frames
    0x00280009: frame_increment_pointer
    0x0028000A: frame_dimension_pointer
    0x00280010: rows
    0x00280011: columns
    0x00280012: planes
    0x00280014: ultrasound_color_data_present
    0x00280030: pixel_spacing
    0x00280031: zoom_factor
    0x00280032: zoom_center
    0x00280034: pixel_aspect_ratio
    0x00280040: image_format
    0x00280050: manipulated_image
    0x00280051: corrected_image
    0x0028005F: compression_recognition_code
    0x00280060: compression_code
    0x00280061: compression_originator
    0x00280062: compression_label
    0x00280063: compression_description
    0x00280065: compression_sequence
    0x00280066: compression_step_pointers
    0x00280068: repeat_interval
    0x00280069: bits_grouped
    0x00280070: perimeter_table
    0x00280071: perimeter_value
    0x00280080: predictor_rows
    0x00280081: predictor_columns
    0x00280082: predictor_constants
    0x00280090: blocked_pixels
    0x00280091: block_rows
    0x00280092: block_columns
    0x00280093: row_overlap
    0x00280094: column_overlap
    0x00280100: bits_allocated
    0x00280101: bits_stored
    0x00280102: high_bit
    0x00280103: pixel_representation
    0x00280104: smallest_valid_pixel_value
    0x00280105: largest_valid_pixel_value
    0x00280106: smallest_image_pixel_value
    0x00280107: largest_image_pixel_value
    0x00280108: smallest_pixel_value_in_series
    0x00280109: largest_pixel_value_in_series
    0x00280110: smallest_image_pixel_value_in_plane
    0x00280111: largest_image_pixel_value_in_plane
    0x00280120: pixel_padding_value
    0x00280121: pixel_padding_range_limit
    0x00280122: float_pixel_padding_value
    0x00280123: double_float_pixel_padding_value
    0x00280124: float_pixel_padding_range_limit
    0x00280125: double_float_pixel_padding_range_limit
    0x00280200: image_location
    0x00280300: quality_control_image
    0x00280301: burned_in_annotation
    0x00280302: recognizable_visual_features
    0x00280303: longitudinal_temporal_information_modified
    0x00280304: referenced_color_palette_instance_uid
    0x00280400: transform_label
    0x00280401: transform_version_number
    0x00280402: number_of_transform_steps
    0x00280403: sequence_of_compressed_data
    0x00280404: details_of_coefficients
    0x00280700: dct_label
    0x00280701: data_block_description
    0x00280702: data_block
    0x00280710: normalization_factor_format
    0x00280720: zonal_map_number_format
    0x00280721: zonal_map_location
    0x00280722: zonal_map_format
    0x00280730: adaptive_map_format
    0x00280740: code_number_format
    0x00280A02: pixel_spacing_calibration_type
    0x00280A04: pixel_spacing_calibration_description
    0x00281040: pixel_intensity_relationship
    0x00281041: pixel_intensity_relationship_sign
    0x00281050: window_center
    0x00281051: window_width
    0x00281052: rescale_intercept
    0x00281053: rescale_slope
    0x00281054: rescale_type
    0x00281055: window_center___width_explanation
    0x00281056: voi_lut_function
    0x00281080: gray_scale
    0x00281090: recommended_viewing_mode
    0x00281100: gray_lookup_table_descriptor
    0x00281101: red_palette_color_lookup_table_descriptor
    0x00281102: green_palette_color_lookup_table_descriptor
    0x00281103: blue_palette_color_lookup_table_descriptor
    0x00281104: alpha_palette_color_lookup_table_descriptor
    0x00281111: large_red_palette_color_lookup_table_descriptor
    0x00281112: large_green_palette_color_lookup_table_descriptor
    0x00281113: large_blue_palette_color_lookup_table_descriptor
    0x00281199: palette_color_lookup_table_uid
    0x00281200: gray_lookup_table_data
    0x00281201: red_palette_color_lookup_table_data
    0x00281202: green_palette_color_lookup_table_data
    0x00281203: blue_palette_color_lookup_table_data
    0x00281204: alpha_palette_color_lookup_table_data
    0x00281211: large_red_palette_color_lookup_table_data
    0x00281212: large_green_palette_color_lookup_table_data
    0x00281213: large_blue_palette_color_lookup_table_data
    0x00281214: large_palette_color_lookup_table_uid
    0x00281221: segmented_red_palette_color_lookup_table_data
    0x00281222: segmented_green_palette_color_lookup_table_data
    0x00281223: segmented_blue_palette_color_lookup_table_data
    0x00281224: segmented_alpha_palette_color_lookup_table_data
    0x00281300: breast_implant_present
    0x00281350: partial_view
    0x00281351: partial_view_description
    0x00281352: partial_view_code_sequence
    0x0028135A: spatial_locations_preserved
    0x00281401: data_frame_assignment_sequence
    0x00281402: data_path_assignment
    0x00281403: bits_mapped_to_color_lookup_table
    0x00281404: blending_lut_1_sequence
    0x00281405: blending_lut_1_transfer_function
    0x00281406: blending_weight_constant
    0x00281407: blending_lookup_table_descriptor
    0x00281408: blending_lookup_table_data
    0x0028140B: enhanced_palette_color_lookup_table_sequence
    0x0028140C: blending_lut_2_sequence
    0x0028140D: blending_lut_2_transfer_function
    0x0028140E: data_path_id
    0x0028140F: rgb_lut_transfer_function
    0x00281410: alpha_lut_transfer_function
    0x00282000: icc_profile
    0x00282002: color_space
    0x00282110: lossy_image_compression
    0x00282112: lossy_image_compression_ratio
    0x00282114: lossy_image_compression_method
    0x00283000: modality_lut_sequence
    0x00283002: lut_descriptor
    0x00283003: lut_explanation
    0x00283004: modality_lut_type
    0x00283006: lut_data
    0x00283010: voi_lut_sequence
    0x00283110: softcopy_voi_lut_sequence
    0x00284000: image_presentation_comments
    0x00285000: bi_plane_acquisition_sequence
    0x00286010: representative_frame_number
    0x00286020: frame_numbers_of_interest__foi_
    0x00286022: frame_of_interest_description
    0x00286023: frame_of_interest_type
    0x00286030: mask_pointer_s_
    0x00286040: r_wave_pointer
    0x00286100: mask_subtraction_sequence
    0x00286101: mask_operation
    0x00286102: applicable_frame_range
    0x00286110: mask_frame_numbers
    0x00286112: contrast_frame_averaging
    0x00286114: mask_sub_pixel_shift
    0x00286120: tid_offset
    0x00286190: mask_operation_explanation
    0x00287000: equipment_administrator_sequence
    0x00287001: number_of_display_subsystems
    0x00287002: current_configuration_id
    0x00287003: display_subsystem_id
    0x00287004: display_subsystem_name
    0x00287005: display_subsystem_description
    0x00287006: system_status
    0x00287007: system_status_comment
    0x00287008: target_luminance_characteristics_sequence
    0x00287009: luminance_characteristics_id
    0x0028700A: display_subsystem_configuration_sequence
    0x0028700B: configuration_id
    0x0028700C: configuration_name
    0x0028700D: configuration_description
    0x0028700E: referenced_target_luminance_characteristics_id
    0x0028700F: qa_results_sequence
    0x00287010: display_subsystem_qa_results_sequence
    0x00287011: configuration_qa_results_sequence
    0x00287012: measurement_equipment_sequence
    0x00287013: measurement_functions
    0x00287014: measurement_equipment_type
    0x00287015: visual_evaluation_result_sequence
    0x00287016: display_calibration_result_sequence
    0x00287017: ddl_value
    0x00287018: ciexy_white_point
    0x00287019: display_function_type
    0x0028701A: gamma_value
    0x0028701B: number_of_luminance_points
    0x0028701C: luminance_response_sequence
    0x0028701D: target_minimum_luminance
    0x0028701E: target_maximum_luminance
    0x0028701F: luminance_value
    0x00287020: luminance_response_description
    0x00287021: white_point_flag
    0x00287022: display_device_type_code_sequence
    0x00287023: display_subsystem_sequence
    0x00287024: luminance_result_sequence
    0x00287025: ambient_light_value_source
    0x00287026: measured_characteristics
    0x00287027: luminance_uniformity_result_sequence
    0x00287028: visual_evaluation_test_sequence
    0x00287029: test_result
    0x0028702A: test_result_comment
    0x0028702B: test_image_validation
    0x0028702C: test_pattern_code_sequence
    0x0028702D: measurement_pattern_code_sequence
    0x0028702E: visual_evaluation_method_code_sequence
    0x00287FE0: pixel_data_provider_url
    0x00289001: data_point_rows
    0x00289002: data_point_columns
    0x00289003: signal_domain_columns
    0x00289099: largest_monochrome_pixel_value
    0x00289108: data_representation
    0x00289110: pixel_measures_sequence
    0x00289132: frame_voi_lut_sequence
    0x00289145: pixel_value_transformation_sequence
    0x00289235: signal_domain_rows
    0x00289411: display_filter_percentage
    0x00289415: frame_pixel_shift_sequence
    0x00289416: subtraction_item_id
    0x00289422: pixel_intensity_relationship_lut_sequence
    0x00289443: frame_pixel_data_properties_sequence
    0x00289444: geometrical_properties
    0x00289445: geometric_maximum_distortion
    0x00289446: image_processing_applied
    0x00289454: mask_selection_mode
    0x00289474: lut_function
    0x00289478: mask_visibility_percentage
    0x00289501: pixel_shift_sequence
    0x00289502: region_pixel_shift_sequence
    0x00289503: vertices_of_the_region
    0x00289505: multi_frame_presentation_sequence
    0x00289506: pixel_shift_frame_range
    0x00289507: lut_frame_range
    0x00289520: image_to_equipment_mapping_matrix
    0x00289537: equipment_coordinate_system_identification
    0x0032000A: study_status_id
    0x0032000C: study_priority_id
    0x00320012: study_id_issuer
    0x00320032: study_verified_date
    0x00320033: study_verified_time
    0x00320034: study_read_date
    0x00320035: study_read_time
    0x00321000: scheduled_study_start_date
    0x00321001: scheduled_study_start_time
    0x00321010: scheduled_study_stop_date
    0x00321011: scheduled_study_stop_time
    0x00321020: scheduled_study_location
    0x00321021: scheduled_study_location_ae_title
    0x00321030: reason_for_study
    0x00321031: requesting_physician_identification_sequence
    0x00321032: requesting_physician
    0x00321033: requesting_service
    0x00321034: requesting_service_code_sequence
    0x00321040: study_arrival_date
    0x00321041: study_arrival_time
    0x00321050: study_completion_date
    0x00321051: study_completion_time
    0x00321055: study_component_status_id
    0x00321060: requested_procedure_description
    0x00321064: requested_procedure_code_sequence
    0x00321070: requested_contrast_agent
    0x00324000: study_comments
    0x00380004: referenced_patient_alias_sequence
    0x00380008: visit_status_id
    0x00380010: admission_id
    0x00380011: issuer_of_admission_id
    0x00380014: issuer_of_admission_id_sequence
    0x00380016: route_of_admissions
    0x0038001A: scheduled_admission_date
    0x0038001B: scheduled_admission_time
    0x0038001C: scheduled_discharge_date
    0x0038001D: scheduled_discharge_time
    0x0038001E: scheduled_patient_institution_residence
    0x00380020: admitting_date
    0x00380021: admitting_time
    0x00380030: discharge_date
    0x00380032: discharge_time
    0x00380040: discharge_diagnosis_description
    0x00380044: discharge_diagnosis_code_sequence
    0x00380050: special_needs
    0x00380060: service_episode_id
    0x00380061: issuer_of_service_episode_id
    0x00380062: service_episode_description
    0x00380064: issuer_of_service_episode_id_sequence
    0x00380100: pertinent_documents_sequence
    0x00380101: pertinent_resources_sequence
    0x00380102: resource_description
    0x00380300: current_patient_location
    0x00380400: patient_s_institution_residence
    0x00380500: patient_state
    0x00380502: patient_clinical_trial_participation_sequence
    0x00384000: visit_comments
    0x003A0004: waveform_originality
    0x003A0005: number_of_waveform_channels
    0x003A0010: number_of_waveform_samples
    0x003A001A: sampling_frequency
    0x003A0020: multiplex_group_label
    0x003A0200: channel_definition_sequence
    0x003A0202: waveform_channel_number
    0x003A0203: channel_label
    0x003A0205: channel_status
    0x003A0208: channel_source_sequence
    0x003A0209: channel_source_modifiers_sequence
    0x003A020A: source_waveform_sequence
    0x003A020C: channel_derivation_description
    0x003A0210: channel_sensitivity
    0x003A0211: channel_sensitivity_units_sequence
    0x003A0212: channel_sensitivity_correction_factor
    0x003A0213: channel_baseline
    0x003A0214: channel_time_skew
    0x003A0215: channel_sample_skew
    0x003A0218: channel_offset
    0x003A021A: waveform_bits_stored
    0x003A0220: filter_low_frequency
    0x003A0221: filter_high_frequency
    0x003A0222: notch_filter_frequency
    0x003A0223: notch_filter_bandwidth
    0x003A0230: waveform_data_display_scale
    0x003A0231: waveform_display_background_cielab_value
    0x003A0240: waveform_presentation_group_sequence
    0x003A0241: presentation_group_number
    0x003A0242: channel_display_sequence
    0x003A0244: channel_recommended_display_cielab_value
    0x003A0245: channel_position
    0x003A0246: display_shading_flag
    0x003A0247: fractional_channel_display_scale
    0x003A0248: absolute_channel_display_scale
    0x003A0300: multiplexed_audio_channels_description_code_sequence
    0x003A0301: channel_identification_code
    0x003A0302: channel_mode
    0x00400001: scheduled_station_ae_title
    0x00400002: scheduled_procedure_step_start_date
    0x00400003: scheduled_procedure_step_start_time
    0x00400004: scheduled_procedure_step_end_date
    0x00400005: scheduled_procedure_step_end_time
    0x00400006: scheduled_performing_physician_s_name
    0x00400007: scheduled_procedure_step_description
    0x00400008: scheduled_protocol_code_sequence
    0x00400009: scheduled_procedure_step_id
    0x0040000A: stage_code_sequence
    0x0040000B: scheduled_performing_physician_identification_sequence
    0x00400010: scheduled_station_name
    0x00400011: scheduled_procedure_step_location
    0x00400012: pre_medication
    0x00400020: scheduled_procedure_step_status
    0x00400026: order_placer_identifier_sequence
    0x00400027: order_filler_identifier_sequence
    0x00400031: local_namespace_entity_id
    0x00400032: universal_entity_id
    0x00400033: universal_entity_id_type
    0x00400035: identifier_type_code
    0x00400036: assigning_facility_sequence
    0x00400039: assigning_jurisdiction_code_sequence
    0x0040003A: assigning_agency_or_department_code_sequence
    0x00400100: scheduled_procedure_step_sequence
    0x00400220: referenced_non_image_composite_sop_instance_sequence
    0x00400241: performed_station_ae_title
    0x00400242: performed_station_name
    0x00400243: performed_location
    0x00400244: performed_procedure_step_start_date
    0x00400245: performed_procedure_step_start_time
    0x00400250: performed_procedure_step_end_date
    0x00400251: performed_procedure_step_end_time
    0x00400252: performed_procedure_step_status
    0x00400253: performed_procedure_step_id
    0x00400254: performed_procedure_step_description
    0x00400255: performed_procedure_type_description
    0x00400260: performed_protocol_code_sequence
    0x00400261: performed_protocol_type
    0x00400270: scheduled_step_attributes_sequence
    0x00400275: request_attributes_sequence
    0x00400280: comments_on_the_performed_procedure_step
    0x00400281: performed_procedure_step_discontinuation_reason_code_sequence
    0x00400293: quantity_sequence
    0x00400294: quantity
    0x00400295: measuring_units_sequence
    0x00400296: billing_item_sequence
    0x00400300: total_time_of_fluoroscopy
    0x00400301: total_number_of_exposures
    0x00400302: entrance_dose
    0x00400303: exposed_area
    0x00400306: distance_source_to_entrance
    0x00400307: distance_source_to_support
    0x0040030E: exposure_dose_sequence
    0x00400310: comments_on_radiation_dose
    0x00400312: x_ray_output
    0x00400314: half_value_layer
    0x00400316: organ_dose
    0x00400318: organ_exposed
    0x00400320: billing_procedure_step_sequence
    0x00400321: film_consumption_sequence
    0x00400324: billing_supplies_and_devices_sequence
    0x00400330: referenced_procedure_step_sequence
    0x00400340: performed_series_sequence
    0x00400400: comments_on_the_scheduled_procedure_step
    0x00400440: protocol_context_sequence
    0x00400441: content_item_modifier_sequence
    0x00400500: scheduled_specimen_sequence
    0x0040050A: specimen_accession_number
    0x00400512: container_identifier
    0x00400513: issuer_of_the_container_identifier_sequence
    0x00400515: alternate_container_identifier_sequence
    0x00400518: container_type_code_sequence
    0x0040051A: container_description
    0x00400520: container_component_sequence
    0x00400550: specimen_sequence
    0x00400551: specimen_identifier
    0x00400552: specimen_description_sequence__trial_
    0x00400553: specimen_description__trial_
    0x00400554: specimen_uid
    0x00400555: acquisition_context_sequence
    0x00400556: acquisition_context_description
    0x0040059A: specimen_type_code_sequence
    0x00400560: specimen_description_sequence
    0x00400562: issuer_of_the_specimen_identifier_sequence
    0x00400600: specimen_short_description
    0x00400602: specimen_detailed_description
    0x00400610: specimen_preparation_sequence
    0x00400612: specimen_preparation_step_content_item_sequence
    0x00400620: specimen_localization_content_item_sequence
    0x004006FA: slide_identifier
    0x0040071A: image_center_point_coordinates_sequence
    0x0040072A: x_offset_in_slide_coordinate_system
    0x0040073A: y_offset_in_slide_coordinate_system
    0x0040074A: z_offset_in_slide_coordinate_system
    0x004008D8: pixel_spacing_sequence
    0x004008DA: coordinate_system_axis_code_sequence
    0x004008EA: measurement_units_code_sequence
    0x004009F8: vital_stain_code_sequence__trial_
    0x00401001: requested_procedure_id
    0x00401002: reason_for_the_requested_procedure
    0x00401003: requested_procedure_priority
    0x00401004: patient_transport_arrangements
    0x00401005: requested_procedure_location
    0x00401006: placer_order_number___procedure
    0x00401007: filler_order_number___procedure
    0x00401008: confidentiality_code
    0x00401009: reporting_priority
    0x0040100A: reason_for_requested_procedure_code_sequence
    0x00401010: names_of_intended_recipients_of_results
    0x00401011: intended_recipients_of_results_identification_sequence
    0x00401012: reason_for_performed_procedure_code_sequence
    0x00401060: requested_procedure_description__trial_
    0x00401101: person_identification_code_sequence
    0x00401102: person_s_address
    0x00401103: person_s_telephone_numbers
    0x00401104: person_s_telecom_information
    0x00401400: requested_procedure_comments
    0x00402001: reason_for_the_imaging_service_request
    0x00402004: issue_date_of_imaging_service_request
    0x00402005: issue_time_of_imaging_service_request
    0x00402006: placer_order_number___imaging_service_request__retired_
    0x00402007: filler_order_number___imaging_service_request__retired_
    0x00402008: order_entered_by
    0x00402009: order_enterer_s_location
    0x00402010: order_callback_phone_number
    0x00402011: order_callback_telecom_information
    0x00402016: placer_order_number___imaging_service_request
    0x00402017: filler_order_number___imaging_service_request
    0x00402400: imaging_service_request_comments
    0x00403001: confidentiality_constraint_on_patient_data_description
    0x00404001: general_purpose_scheduled_procedure_step_status
    0x00404002: general_purpose_performed_procedure_step_status
    0x00404003: general_purpose_scheduled_procedure_step_priority
    0x00404004: scheduled_processing_applications_code_sequence
    0x00404005: scheduled_procedure_step_start_datetime
    0x00404006: multiple_copies_flag
    0x00404007: performed_processing_applications_code_sequence
    0x00404009: human_performer_code_sequence
    0x00404010: scheduled_procedure_step_modification_datetime
    0x00404011: expected_completion_datetime
    0x00404015: resulting_general_purpose_performed_procedure_steps_sequence
    0x00404016: referenced_general_purpose_scheduled_procedure_step_sequence
    0x00404018: scheduled_workitem_code_sequence
    0x00404019: performed_workitem_code_sequence
    0x00404020: input_availability_flag
    0x00404021: input_information_sequence
    0x00404022: relevant_information_sequence
    0x00404023: referenced_general_purpose_scheduled_procedure_step_transaction_uid
    0x00404025: scheduled_station_name_code_sequence
    0x00404026: scheduled_station_class_code_sequence
    0x00404027: scheduled_station_geographic_location_code_sequence
    0x00404028: performed_station_name_code_sequence
    0x00404029: performed_station_class_code_sequence
    0x00404030: performed_station_geographic_location_code_sequence
    0x00404031: requested_subsequent_workitem_code_sequence
    0x00404032: non_dicom_output_code_sequence
    0x00404033: output_information_sequence
    0x00404034: scheduled_human_performers_sequence
    0x00404035: actual_human_performers_sequence
    0x00404036: human_performer_s_organization
    0x00404037: human_performer_s_name
    0x00404040: raw_data_handling
    0x00404041: input_readiness_state
    0x00404050: performed_procedure_step_start_datetime
    0x00404051: performed_procedure_step_end_datetime
    0x00404052: procedure_step_cancellation_datetime
    0x00404070: output_destination_sequence
    0x00404071: dicom_storage_sequence
    0x00404072: stow_rs_storage_sequence
    0x00404073: storage_url
    0x00404074: xds_storage_sequence
    0x00408302: entrance_dose_in_mgy
    0x00409092: parametric_map_frame_type_sequence
    0x00409094: referenced_image_real_world_value_mapping_sequence
    0x00409096: real_world_value_mapping_sequence
    0x00409098: pixel_value_mapping_code_sequence
    0x00409210: lut_label
    0x00409211: real_world_value_last_value_mapped
    0x00409212: real_world_value_lut_data
    0x00409213: double_float_real_world_value_last_value_mapped
    0x00409214: double_float_real_world_value_first_value_mapped
    0x00409216: real_world_value_first_value_mapped
    0x00409220: quantity_definition_sequence
    0x00409224: real_world_value_intercept
    0x00409225: real_world_value_slope
    0x0040A007: findings_flag__trial_
    0x0040A010: relationship_type
    0x0040A020: findings_sequence__trial_
    0x0040A021: findings_group_uid__trial_
    0x0040A022: referenced_findings_group_uid__trial_
    0x0040A023: findings_group_recording_date__trial_
    0x0040A024: findings_group_recording_time__trial_
    0x0040A026: findings_source_category_code_sequence__trial_
    0x0040A027: verifying_organization
    0x0040A028: documenting_organization_identifier_code_sequence__trial_
    0x0040A030: verification_datetime
    0x0040A032: observation_datetime
    0x0040A040: value_type
    0x0040A043: concept_name_code_sequence
    0x0040A047: measurement_precision_description__trial_
    0x0040A050: continuity_of_content
    0x0040A057: urgency_or_priority_alerts__trial_
    0x0040A060: sequencing_indicator__trial_
    0x0040A066: document_identifier_code_sequence__trial_
    0x0040A067: document_author__trial_
    0x0040A068: document_author_identifier_code_sequence__trial_
    0x0040A070: identifier_code_sequence__trial_
    0x0040A073: verifying_observer_sequence
    0x0040A074: object_binary_identifier__trial_
    0x0040A075: verifying_observer_name
    0x0040A076: documenting_observer_identifier_code_sequence__trial_
    0x0040A078: author_observer_sequence
    0x0040A07A: participant_sequence
    0x0040A07C: custodial_organization_sequence
    0x0040A080: participation_type
    0x0040A082: participation_datetime
    0x0040A084: observer_type
    0x0040A085: procedure_identifier_code_sequence__trial_
    0x0040A088: verifying_observer_identification_code_sequence
    0x0040A089: object_directory_binary_identifier__trial_
    0x0040A090: equivalent_cda_document_sequence
    0x0040A0B0: referenced_waveform_channels
    0x0040A110: date_of_document_or_verbal_transaction__trial_
    0x0040A112: time_of_document_creation_or_verbal_transaction__trial_
    0x0040A120: datetime
    0x0040A121: date
    0x0040A122: time
    0x0040A123: person_name
    0x0040A124: uid
    0x0040A125: report_status_id__trial_
    0x0040A130: temporal_range_type
    0x0040A132: referenced_sample_positions
    0x0040A136: referenced_frame_numbers
    0x0040A138: referenced_time_offsets
    0x0040A13A: referenced_datetime
    0x0040A160: text_value
    0x0040A161: floating_point_value
    0x0040A162: rational_numerator_value
    0x0040A163: rational_denominator_value
    0x0040A167: observation_category_code_sequence__trial_
    0x0040A168: concept_code_sequence
    0x0040A16A: bibliographic_citation__trial_
    0x0040A170: purpose_of_reference_code_sequence
    0x0040A171: observation_uid
    0x0040A172: referenced_observation_uid__trial_
    0x0040A173: referenced_observation_class__trial_
    0x0040A174: referenced_object_observation_class__trial_
    0x0040A180: annotation_group_number
    0x0040A192: observation_date__trial_
    0x0040A193: observation_time__trial_
    0x0040A194: measurement_automation__trial_
    0x0040A195: modifier_code_sequence
    0x0040A224: identification_description__trial_
    0x0040A290: coordinates_set_geometric_type__trial_
    0x0040A296: algorithm_code_sequence__trial_
    0x0040A297: algorithm_description__trial_
    0x0040A29A: pixel_coordinates_set__trial_
    0x0040A300: measured_value_sequence
    0x0040A301: numeric_value_qualifier_code_sequence
    0x0040A307: current_observer__trial_
    0x0040A30A: numeric_value
    0x0040A313: referenced_accession_sequence__trial_
    0x0040A33A: report_status_comment__trial_
    0x0040A340: procedure_context_sequence__trial_
    0x0040A352: verbal_source__trial_
    0x0040A353: address__trial_
    0x0040A354: telephone_number__trial_
    0x0040A358: verbal_source_identifier_code_sequence__trial_
    0x0040A360: predecessor_documents_sequence
    0x0040A370: referenced_request_sequence
    0x0040A372: performed_procedure_code_sequence
    0x0040A375: current_requested_procedure_evidence_sequence
    0x0040A380: report_detail_sequence__trial_
    0x0040A385: pertinent_other_evidence_sequence
    0x0040A390: hl7_structured_document_reference_sequence
    0x0040A402: observation_subject_uid__trial_
    0x0040A403: observation_subject_class__trial_
    0x0040A404: observation_subject_type_code_sequence__trial_
    0x0040A491: completion_flag
    0x0040A492: completion_flag_description
    0x0040A493: verification_flag
    0x0040A494: archive_requested
    0x0040A496: preliminary_flag
    0x0040A504: content_template_sequence
    0x0040A525: identical_documents_sequence
    0x0040A600: observation_subject_context_flag__trial_
    0x0040A601: observer_context_flag__trial_
    0x0040A603: procedure_context_flag__trial_
    0x0040A730: content_sequence
    0x0040A731: relationship_sequence__trial_
    0x0040A732: relationship_type_code_sequence__trial_
    0x0040A744: language_code_sequence__trial_
    0x0040A992: uniform_resource_locator__trial_
    0x0040B020: waveform_annotation_sequence
    0x0040DB00: template_identifier
    0x0040DB06: template_version
    0x0040DB07: template_local_version
    0x0040DB0B: template_extension_flag
    0x0040DB0C: template_extension_organization_uid
    0x0040DB0D: template_extension_creator_uid
    0x0040DB73: referenced_content_item_identifier
    0x0040E001: hl7_instance_identifier
    0x0040E004: hl7_document_effective_time
    0x0040E006: hl7_document_type_code_sequence
    0x0040E008: document_class_code_sequence
    0x0040E010: retrieve_uri
    0x0040E011: retrieve_location_uid
    0x0040E020: type_of_instances
    0x0040E021: dicom_retrieval_sequence
    0x0040E022: dicom_media_retrieval_sequence
    0x0040E023: wado_retrieval_sequence
    0x0040E024: xds_retrieval_sequence
    0x0040E025: wado_rs_retrieval_sequence
    0x0040E030: repository_unique_id
    0x0040E031: home_community_id
    0x00420010: document_title
    0x00420011: encapsulated_document
    0x00420012: mime_type_of_encapsulated_document
    0x00420013: source_instance_sequence
    0x00420014: list_of_mime_types
    0x00440001: product_package_identifier
    0x00440002: substance_administration_approval
    0x00440003: approval_status_further_description
    0x00440004: approval_status_datetime
    0x00440007: product_type_code_sequence
    0x00440008: product_name
    0x00440009: product_description
    0x0044000A: product_lot_identifier
    0x0044000B: product_expiration_datetime
    0x00440010: substance_administration_datetime
    0x00440011: substance_administration_notes
    0x00440012: substance_administration_device_id
    0x00440013: product_parameter_sequence
    0x00440019: substance_administration_parameter_sequence
    0x00460012: lens_description
    0x00460014: right_lens_sequence
    0x00460015: left_lens_sequence
    0x00460016: unspecified_laterality_lens_sequence
    0x00460018: cylinder_sequence
    0x00460028: prism_sequence
    0x00460030: horizontal_prism_power
    0x00460032: horizontal_prism_base
    0x00460034: vertical_prism_power
    0x00460036: vertical_prism_base
    0x00460038: lens_segment_type
    0x00460040: optical_transmittance
    0x00460042: channel_width
    0x00460044: pupil_size
    0x00460046: corneal_size
    0x00460050: autorefraction_right_eye_sequence
    0x00460052: autorefraction_left_eye_sequence
    0x00460060: distance_pupillary_distance
    0x00460062: near_pupillary_distance
    0x00460063: intermediate_pupillary_distance
    0x00460064: other_pupillary_distance
    0x00460070: keratometry_right_eye_sequence
    0x00460071: keratometry_left_eye_sequence
    0x00460074: steep_keratometric_axis_sequence
    0x00460075: radius_of_curvature
    0x00460076: keratometric_power
    0x00460077: keratometric_axis
    0x00460080: flat_keratometric_axis_sequence
    0x00460092: background_color
    0x00460094: optotype
    0x00460095: optotype_presentation
    0x00460097: subjective_refraction_right_eye_sequence
    0x00460098: subjective_refraction_left_eye_sequence
    0x00460100: add_near_sequence
    0x00460101: add_intermediate_sequence
    0x00460102: add_other_sequence
    0x00460104: add_power
    0x00460106: viewing_distance
    0x00460121: visual_acuity_type_code_sequence
    0x00460122: visual_acuity_right_eye_sequence
    0x00460123: visual_acuity_left_eye_sequence
    0x00460124: visual_acuity_both_eyes_open_sequence
    0x00460125: viewing_distance_type
    0x00460135: visual_acuity_modifiers
    0x00460137: decimal_visual_acuity
    0x00460139: optotype_detailed_definition
    0x00460145: referenced_refractive_measurements_sequence
    0x00460146: sphere_power
    0x00460147: cylinder_power
    0x00460201: corneal_topography_surface
    0x00460202: corneal_vertex_location
    0x00460203: pupil_centroid_x_coordinate
    0x00460204: pupil_centroid_y_coordinate
    0x00460205: equivalent_pupil_radius
    0x00460207: corneal_topography_map_type_code_sequence
    0x00460208: vertices_of_the_outline_of_pupil
    0x00460210: corneal_topography_mapping_normals_sequence
    0x00460211: maximum_corneal_curvature_sequence
    0x00460212: maximum_corneal_curvature
    0x00460213: maximum_corneal_curvature_location
    0x00460215: minimum_keratometric_sequence
    0x00460218: simulated_keratometric_cylinder_sequence
    0x00460220: average_corneal_power
    0x00460224: corneal_i_s_value
    0x00460227: analyzed_area
    0x00460230: surface_regularity_index
    0x00460232: surface_asymmetry_index
    0x00460234: corneal_eccentricity_index
    0x00460236: keratoconus_prediction_index
    0x00460238: decimal_potential_visual_acuity
    0x00460242: corneal_topography_map_quality_evaluation
    0x00460244: source_image_corneal_processed_data_sequence
    0x00460247: corneal_point_location
    0x00460248: corneal_point_estimated
    0x00460249: axial_power
    0x00460250: tangential_power
    0x00460251: refractive_power
    0x00460252: relative_elevation
    0x00460253: corneal_wavefront
    0x00480001: imaged_volume_width
    0x00480002: imaged_volume_height
    0x00480003: imaged_volume_depth
    0x00480006: total_pixel_matrix_columns
    0x00480007: total_pixel_matrix_rows
    0x00480008: total_pixel_matrix_origin_sequence
    0x00480010: specimen_label_in_image
    0x00480011: focus_method
    0x00480012: extended_depth_of_field
    0x00480013: number_of_focal_planes
    0x00480014: distance_between_focal_planes
    0x00480015: recommended_absent_pixel_cielab_value
    0x00480100: illuminator_type_code_sequence
    0x00480102: image_orientation__slide_
    0x00480105: optical_path_sequence
    0x00480106: optical_path_identifier
    0x00480107: optical_path_description
    0x00480108: illumination_color_code_sequence
    0x00480110: specimen_reference_sequence
    0x00480111: condenser_lens_power
    0x00480112: objective_lens_power
    0x00480113: objective_lens_numerical_aperture
    0x00480120: palette_color_lookup_table_sequence
    0x00480200: referenced_image_navigation_sequence
    0x00480201: top_left_hand_corner_of_localizer_area
    0x00480202: bottom_right_hand_corner_of_localizer_area
    0x00480207: optical_path_identification_sequence
    0x0048021A: plane_position__slide__sequence
    0x0048021E: column_position_in_total_image_pixel_matrix
    0x0048021F: row_position_in_total_image_pixel_matrix
    0x00480301: pixel_origin_interpretation
    0x00500004: calibration_image
    0x00500010: device_sequence
    0x00500012: container_component_type_code_sequence
    0x00500013: container_component_thickness
    0x00500014: device_length
    0x00500015: container_component_width
    0x00500016: device_diameter
    0x00500017: device_diameter_units
    0x00500018: device_volume
    0x00500019: inter_marker_distance
    0x0050001A: container_component_material
    0x0050001B: container_component_id
    0x0050001C: container_component_length
    0x0050001D: container_component_diameter
    0x0050001E: container_component_description
    0x00500020: device_description
    0x00520001: contrast_bolus_ingredient_percent_by_volume
    0x00520002: oct_focal_distance
    0x00520003: beam_spot_size
    0x00520004: effective_refractive_index
    0x00520006: oct_acquisition_domain
    0x00520007: oct_optical_center_wavelength
    0x00520008: axial_resolution
    0x00520009: ranging_depth
    0x00520011: a_line_rate
    0x00520012: a_lines_per_frame
    0x00520013: catheter_rotational_rate
    0x00520014: a_line_pixel_spacing
    0x00520016: mode_of_percutaneous_access_sequence
    0x00520025: intravascular_oct_frame_type_sequence
    0x00520026: oct_z_offset_applied
    0x00520027: intravascular_frame_content_sequence
    0x00520028: intravascular_longitudinal_distance
    0x00520029: intravascular_oct_frame_content_sequence
    0x00520030: oct_z_offset_correction
    0x00520031: catheter_direction_of_rotation
    0x00520033: seam_line_location
    0x00520034: first_a_line_location
    0x00520036: seam_line_index
    0x00520038: number_of_padded_a_lines
    0x00520039: interpolation_type
    0x0052003A: refractive_index_applied
    0x00540010: energy_window_vector
    0x00540011: number_of_energy_windows
    0x00540012: energy_window_information_sequence
    0x00540013: energy_window_range_sequence
    0x00540014: energy_window_lower_limit
    0x00540015: energy_window_upper_limit
    0x00540016: radiopharmaceutical_information_sequence
    0x00540017: residual_syringe_counts
    0x00540018: energy_window_name
    0x00540020: detector_vector
    0x00540021: number_of_detectors
    0x00540022: detector_information_sequence
    0x00540030: phase_vector
    0x00540031: number_of_phases
    0x00540032: phase_information_sequence
    0x00540033: number_of_frames_in_phase
    0x00540036: phase_delay
    0x00540038: pause_between_frames
    0x00540039: phase_description
    0x00540050: rotation_vector
    0x00540051: number_of_rotations
    0x00540052: rotation_information_sequence
    0x00540053: number_of_frames_in_rotation
    0x00540060: r_r_interval_vector
    0x00540061: number_of_r_r_intervals
    0x00540062: gated_information_sequence
    0x00540063: data_information_sequence
    0x00540070: time_slot_vector
    0x00540071: number_of_time_slots
    0x00540072: time_slot_information_sequence
    0x00540073: time_slot_time
    0x00540080: slice_vector
    0x00540081: number_of_slices
    0x00540090: angular_view_vector
    0x00540100: time_slice_vector
    0x00540101: number_of_time_slices
    0x00540200: start_angle
    0x00540202: type_of_detector_motion
    0x00540210: trigger_vector
    0x00540211: number_of_triggers_in_phase
    0x00540220: view_code_sequence
    0x00540222: view_modifier_code_sequence
    0x00540300: radionuclide_code_sequence
    0x00540302: administration_route_code_sequence
    0x00540304: radiopharmaceutical_code_sequence
    0x00540306: calibration_data_sequence
    0x00540308: energy_window_number
    0x00540400: image_id
    0x00540410: patient_orientation_code_sequence
    0x00540412: patient_orientation_modifier_code_sequence
    0x00540414: patient_gantry_relationship_code_sequence
    0x00540500: slice_progression_direction
    0x00540501: scan_progression_direction
    0x00541000: series_type
    0x00541001: units
    0x00541002: counts_source
    0x00541004: reprojection_method
    0x00541006: suv_type
    0x00541100: randoms_correction_method
    0x00541101: attenuation_correction_method
    0x00541102: decay_correction
    0x00541103: reconstruction_method
    0x00541104: detector_lines_of_response_used
    0x00541105: scatter_correction_method
    0x00541200: axial_acceptance
    0x00541201: axial_mash
    0x00541202: transverse_mash
    0x00541203: detector_element_size
    0x00541210: coincidence_window_width
    0x00541220: secondary_counts_type
    0x00541300: frame_reference_time
    0x00541310: primary__prompts__counts_accumulated
    0x00541311: secondary_counts_accumulated
    0x00541320: slice_sensitivity_factor
    0x00541321: decay_factor
    0x00541322: dose_calibration_factor
    0x00541323: scatter_fraction_factor
    0x00541324: dead_time_factor
    0x00541330: image_index
    0x00541400: counts_included
    0x00541401: dead_time_correction_flag
    0x00603000: histogram_sequence
    0x00603002: histogram_number_of_bins
    0x00603004: histogram_first_bin_value
    0x00603006: histogram_last_bin_value
    0x00603008: histogram_bin_width
    0x00603010: histogram_explanation
    0x00603020: histogram_data
    0x00620001: segmentation_type
    0x00620002: segment_sequence
    0x00620003: segmented_property_category_code_sequence
    0x00620004: segment_number
    0x00620005: segment_label
    0x00620006: segment_description
    0x00620008: segment_algorithm_type
    0x00620009: segment_algorithm_name
    0x0062000A: segment_identification_sequence
    0x0062000B: referenced_segment_number
    0x0062000C: recommended_display_grayscale_value
    0x0062000D: recommended_display_cielab_value
    0x0062000E: maximum_fractional_value
    0x0062000F: segmented_property_type_code_sequence
    0x00620010: segmentation_fractional_type
    0x00620011: segmented_property_type_modifier_code_sequence
    0x00620012: used_segments_sequence
    0x00620020: tracking_id
    0x00620021: tracking_uid
    0x00640002: deformable_registration_sequence
    0x00640003: source_frame_of_reference_uid
    0x00640005: deformable_registration_grid_sequence
    0x00640007: grid_dimensions
    0x00640008: grid_resolution
    0x00640009: vector_grid_data
    0x0064000F: pre_deformation_matrix_registration_sequence
    0x00640010: post_deformation_matrix_registration_sequence
    0x00660001: number_of_surfaces
    0x00660002: surface_sequence
    0x00660003: surface_number
    0x00660004: surface_comments
    0x00660009: surface_processing
    0x0066000A: surface_processing_ratio
    0x0066000B: surface_processing_description
    0x0066000C: recommended_presentation_opacity
    0x0066000D: recommended_presentation_type
    0x0066000E: finite_volume
    0x00660010: manifold
    0x00660011: surface_points_sequence
    0x00660012: surface_points_normals_sequence
    0x00660013: surface_mesh_primitives_sequence
    0x00660015: number_of_surface_points
    0x00660016: point_coordinates_data
    0x00660017: point_position_accuracy
    0x00660018: mean_point_distance
    0x00660019: maximum_point_distance
    0x0066001A: points_bounding_box_coordinates
    0x0066001B: axis_of_rotation
    0x0066001C: center_of_rotation
    0x0066001E: number_of_vectors
    0x0066001F: vector_dimensionality
    0x00660020: vector_accuracy
    0x00660021: vector_coordinate_data
    0x00660023: triangle_point_index_list
    0x00660024: edge_point_index_list
    0x00660025: vertex_point_index_list
    0x00660026: triangle_strip_sequence
    0x00660027: triangle_fan_sequence
    0x00660028: line_sequence
    0x00660029: primitive_point_index_list
    0x0066002A: surface_count
    0x0066002B: referenced_surface_sequence
    0x0066002C: referenced_surface_number
    0x0066002D: segment_surface_generation_algorithm_identification_sequence
    0x0066002E: segment_surface_source_instance_sequence
    0x0066002F: algorithm_family_code_sequence
    0x00660030: algorithm_name_code_sequence
    0x00660031: algorithm_version
    0x00660032: algorithm_parameters
    0x00660034: facet_sequence
    0x00660035: surface_processing_algorithm_identification_sequence
    0x00660036: algorithm_name
    0x00660037: recommended_point_radius
    0x00660038: recommended_line_thickness
    0x00660040: long_primitive_point_index_list
    0x00660041: long_triangle_point_index_list
    0x00660042: long_edge_point_index_list
    0x00660043: long_vertex_point_index_list
    0x00660101: track_set_sequence
    0x00660102: track_sequence
    0x00660103: recommended_display_cielab_value_list
    0x00660104: tracking_algorithm_identification_sequence
    0x00660105: track_set_number
    0x00660106: track_set_label
    0x00660107: track_set_description
    0x00660108: track_set_anatomical_type_code_sequence
    0x00660121: measurements_sequence
    0x00660124: track_set_statistics_sequence
    0x00660125: floating_point_values
    0x00660129: track_point_index_list
    0x00660130: track_statistics_sequence
    0x00660132: measurement_values_sequence
    0x00660133: diffusion_acquisition_code_sequence
    0x00660134: diffusion_model_code_sequence
    0x00686210: implant_size
    0x00686221: implant_template_version
    0x00686222: replaced_implant_template_sequence
    0x00686223: implant_type
    0x00686224: derivation_implant_template_sequence
    0x00686225: original_implant_template_sequence
    0x00686226: effective_datetime
    0x00686230: implant_target_anatomy_sequence
    0x00686260: information_from_manufacturer_sequence
    0x00686265: notification_from_manufacturer_sequence
    0x00686270: information_issue_datetime
    0x00686280: information_summary
    0x006862A0: implant_regulatory_disapproval_code_sequence
    0x006862A5: overall_template_spatial_tolerance
    0x006862C0: hpgl_document_sequence
    0x006862D0: hpgl_document_id
    0x006862D5: hpgl_document_label
    0x006862E0: view_orientation_code_sequence
    0x006862F0: view_orientation_modifier
    0x006862F2: hpgl_document_scaling
    0x00686300: hpgl_document
    0x00686310: hpgl_contour_pen_number
    0x00686320: hpgl_pen_sequence
    0x00686330: hpgl_pen_number
    0x00686340: hpgl_pen_label
    0x00686345: hpgl_pen_description
    0x00686346: recommended_rotation_point
    0x00686347: bounding_rectangle
    0x00686350: implant_template_3d_model_surface_number
    0x00686360: surface_model_description_sequence
    0x00686380: surface_model_label
    0x00686390: surface_model_scaling_factor
    0x006863A0: materials_code_sequence
    0x006863A4: coating_materials_code_sequence
    0x006863A8: implant_type_code_sequence
    0x006863AC: fixation_method_code_sequence
    0x006863B0: mating_feature_sets_sequence
    0x006863C0: mating_feature_set_id
    0x006863D0: mating_feature_set_label
    0x006863E0: mating_feature_sequence
    0x006863F0: mating_feature_id
    0x00686400: mating_feature_degree_of_freedom_sequence
    0x00686410: degree_of_freedom_id
    0x00686420: degree_of_freedom_type
    0x00686430: tag_2d_mating_feature_coordinates_sequence
    0x00686440: referenced_hpgl_document_id
    0x00686450: tag_2d_mating_point
    0x00686460: tag_2d_mating_axes
    0x00686470: tag_2d_degree_of_freedom_sequence
    0x00686490: tag_3d_degree_of_freedom_axis
    0x006864A0: range_of_freedom
    0x006864C0: tag_3d_mating_point
    0x006864D0: tag_3d_mating_axes
    0x006864F0: tag_2d_degree_of_freedom_axis
    0x00686500: planning_landmark_point_sequence
    0x00686510: planning_landmark_line_sequence
    0x00686520: planning_landmark_plane_sequence
    0x00686530: planning_landmark_id
    0x00686540: planning_landmark_description
    0x00686545: planning_landmark_identification_code_sequence
    0x00686550: tag_2d_point_coordinates_sequence
    0x00686560: tag_2d_point_coordinates
    0x00686590: tag_3d_point_coordinates
    0x006865A0: tag_2d_line_coordinates_sequence
    0x006865B0: tag_2d_line_coordinates
    0x006865D0: tag_3d_line_coordinates
    0x006865E0: tag_2d_plane_coordinates_sequence
    0x006865F0: tag_2d_plane_intersection
    0x00686610: tag_3d_plane_origin
    0x00686620: tag_3d_plane_normal
    0x00700001: graphic_annotation_sequence
    0x00700002: graphic_layer
    0x00700003: bounding_box_annotation_units
    0x00700004: anchor_point_annotation_units
    0x00700005: graphic_annotation_units
    0x00700006: unformatted_text_value
    0x00700008: text_object_sequence
    0x00700009: graphic_object_sequence
    0x00700010: bounding_box_top_left_hand_corner
    0x00700011: bounding_box_bottom_right_hand_corner
    0x00700012: bounding_box_text_horizontal_justification
    0x00700014: anchor_point
    0x00700015: anchor_point_visibility
    0x00700020: graphic_dimensions
    0x00700021: number_of_graphic_points
    0x00700022: graphic_data
    0x00700023: graphic_type
    0x00700024: graphic_filled
    0x00700040: image_rotation__retired_
    0x00700041: image_horizontal_flip
    0x00700042: image_rotation
    0x00700050: displayed_area_top_left_hand_corner__trial_
    0x00700051: displayed_area_bottom_right_hand_corner__trial_
    0x00700052: displayed_area_top_left_hand_corner
    0x00700053: displayed_area_bottom_right_hand_corner
    0x0070005A: displayed_area_selection_sequence
    0x00700060: graphic_layer_sequence
    0x00700062: graphic_layer_order
    0x00700066: graphic_layer_recommended_display_grayscale_value
    0x00700067: graphic_layer_recommended_display_rgb_value
    0x00700068: graphic_layer_description
    0x00700080: content_label
    0x00700081: content_description
    0x00700082: presentation_creation_date
    0x00700083: presentation_creation_time
    0x00700084: content_creator_s_name
    0x00700086: content_creator_s_identification_code_sequence
    0x00700087: alternate_content_description_sequence
    0x00700100: presentation_size_mode
    0x00700101: presentation_pixel_spacing
    0x00700102: presentation_pixel_aspect_ratio
    0x00700103: presentation_pixel_magnification_ratio
    0x00700207: graphic_group_label
    0x00700208: graphic_group_description
    0x00700209: compound_graphic_sequence
    0x00700226: compound_graphic_instance_id
    0x00700227: font_name
    0x00700228: font_name_type
    0x00700229: css_font_name
    0x00700230: rotation_angle
    0x00700231: text_style_sequence
    0x00700232: line_style_sequence
    0x00700233: fill_style_sequence
    0x00700234: graphic_group_sequence
    0x00700241: text_color_cielab_value
    0x00700242: horizontal_alignment
    0x00700243: vertical_alignment
    0x00700244: shadow_style
    0x00700245: shadow_offset_x
    0x00700246: shadow_offset_y
    0x00700247: shadow_color_cielab_value
    0x00700248: underlined
    0x00700249: bold
    0x00700250: italic
    0x00700251: pattern_on_color_cielab_value
    0x00700252: pattern_off_color_cielab_value
    0x00700253: line_thickness
    0x00700254: line_dashing_style
    0x00700255: line_pattern
    0x00700256: fill_pattern
    0x00700257: fill_mode
    0x00700258: shadow_opacity
    0x00700261: gap_length
    0x00700262: diameter_of_visibility
    0x00700273: rotation_point
    0x00700274: tick_alignment
    0x00700278: show_tick_label
    0x00700279: tick_label_alignment
    0x00700282: compound_graphic_units
    0x00700284: pattern_on_opacity
    0x00700285: pattern_off_opacity
    0x00700287: major_ticks_sequence
    0x00700288: tick_position
    0x00700289: tick_label
    0x00700294: compound_graphic_type
    0x00700295: graphic_group_id
    0x00700306: shape_type
    0x00700308: registration_sequence
    0x00700309: matrix_registration_sequence
    0x0070030A: matrix_sequence
    0x0070030B: frame_of_reference_to_displayed_coordinate_system_transformation_matrix
    0x0070030C: frame_of_reference_transformation_matrix_type
    0x0070030D: registration_type_code_sequence
    0x0070030F: fiducial_description
    0x00700310: fiducial_identifier
    0x00700311: fiducial_identifier_code_sequence
    0x00700312: contour_uncertainty_radius
    0x00700314: used_fiducials_sequence
    0x00700318: graphic_coordinates_data_sequence
    0x0070031A: fiducial_uid
    0x0070031C: fiducial_set_sequence
    0x0070031E: fiducial_sequence
    0x00700401: graphic_layer_recommended_display_cielab_value
    0x00700402: blending_sequence
    0x00700403: relative_opacity
    0x00700404: referenced_spatial_registration_sequence
    0x00700405: blending_position
    0x00701101: presentation_display_collection_uid
    0x00701102: presentation_sequence_collection_uid
    0x00701103: presentation_sequence_position_index
    0x00701104: rendered_image_reference_sequence
    0x00701201: volumetric_presentation_state_input_sequence
    0x00701202: presentation_input_type
    0x00701203: input_sequence_position_index
    0x00701204: crop
    0x00701205: cropping_specification_index
    0x00701206: compositing_method
    0x00701207: volumetric_presentation_input_number
    0x00701208: image_volume_geometry
    0x00701301: volume_cropping_sequence
    0x00701302: volume_cropping_method
    0x00701303: bounding_box_crop
    0x00701304: oblique_cropping_plane_sequence
    0x00701305: plane
    0x00701306: plane_normal
    0x00701309: cropping_specification_number
    0x00701501: multi_planar_reconstruction_style
    0x00701502: mpr_thickness_type
    0x00701503: mpr_slab_thickness
    0x00701505: mpr_top_left_hand_corner
    0x00701507: mpr_view_width_direction
    0x00701508: mpr_view_width
    0x0070150C: number_of_volumetric_curve_points
    0x0070150D: volumetric_curve_points
    0x00701511: mpr_view_height_direction
    0x00701512: mpr_view_height
    0x00701801: presentation_state_classification_component_sequence
    0x00701802: component_type
    0x00701803: component_input_sequence
    0x00701804: volumetric_presentation_input_index
    0x00701805: presentation_state_compositor_component_sequence
    0x00701806: weighting_transfer_function_sequence
    0x00701807: weighting_lookup_table_descriptor
    0x00701808: weighting_lookup_table_data
    0x00701901: volumetric_annotation_sequence
    0x00701903: referenced_structured_context_sequence
    0x00701904: referenced_content_item
    0x00701905: volumetric_presentation_input_annotation_sequence
    0x00701907: annotation_clipping
    0x00701A01: presentation_animation_style
    0x00701A03: recommended_animation_rate
    0x00701A04: animation_curve_sequence
    0x00701A05: animation_step_size
    0x00720002: hanging_protocol_name
    0x00720004: hanging_protocol_description
    0x00720006: hanging_protocol_level
    0x00720008: hanging_protocol_creator
    0x0072000A: hanging_protocol_creation_datetime
    0x0072000C: hanging_protocol_definition_sequence
    0x0072000E: hanging_protocol_user_identification_code_sequence
    0x00720010: hanging_protocol_user_group_name
    0x00720012: source_hanging_protocol_sequence
    0x00720014: number_of_priors_referenced
    0x00720020: image_sets_sequence
    0x00720022: image_set_selector_sequence
    0x00720024: image_set_selector_usage_flag
    0x00720026: selector_attribute
    0x00720028: selector_value_number
    0x00720030: time_based_image_sets_sequence
    0x00720032: image_set_number
    0x00720034: image_set_selector_category
    0x00720038: relative_time
    0x0072003A: relative_time_units
    0x0072003C: abstract_prior_value
    0x0072003E: abstract_prior_code_sequence
    0x00720040: image_set_label
    0x00720050: selector_attribute_vr
    0x00720052: selector_sequence_pointer
    0x00720054: selector_sequence_pointer_private_creator
    0x00720056: selector_attribute_private_creator
    0x0072005E: selector_ae_value
    0x0072005F: selector_as_value
    0x00720060: selector_at_value
    0x00720061: selector_da_value
    0x00720062: selector_cs_value
    0x00720063: selector_dt_value
    0x00720064: selector_is_value
    0x00720065: selector_ob_value
    0x00720066: selector_lo_value
    0x00720067: selector_of_value
    0x00720068: selector_lt_value
    0x00720069: selector_ow_value
    0x0072006A: selector_pn_value
    0x0072006B: selector_tm_value
    0x0072006C: selector_sh_value
    0x0072006D: selector_un_value
    0x0072006E: selector_st_value
    0x0072006F: selector_uc_value
    0x00720070: selector_ut_value
    0x00720071: selector_ur_value
    0x00720072: selector_ds_value
    0x00720073: selector_od_value
    0x00720074: selector_fd_value
    0x00720075: selector_ol_value
    0x00720076: selector_fl_value
    0x00720078: selector_ul_value
    0x0072007A: selector_us_value
    0x0072007C: selector_sl_value
    0x0072007E: selector_ss_value
    0x0072007F: selector_ui_value
    0x00720080: selector_code_sequence_value
    0x00720100: number_of_screens
    0x00720102: nominal_screen_definition_sequence
    0x00720104: number_of_vertical_pixels
    0x00720106: number_of_horizontal_pixels
    0x00720108: display_environment_spatial_position
    0x0072010A: screen_minimum_grayscale_bit_depth
    0x0072010C: screen_minimum_color_bit_depth
    0x0072010E: application_maximum_repaint_time
    0x00720200: display_sets_sequence
    0x00720202: display_set_number
    0x00720203: display_set_label
    0x00720204: display_set_presentation_group
    0x00720206: display_set_presentation_group_description
    0x00720208: partial_data_display_handling
    0x00720210: synchronized_scrolling_sequence
    0x00720212: display_set_scrolling_group
    0x00720214: navigation_indicator_sequence
    0x00720216: navigation_display_set
    0x00720218: reference_display_sets
    0x00720300: image_boxes_sequence
    0x00720302: image_box_number
    0x00720304: image_box_layout_type
    0x00720306: image_box_tile_horizontal_dimension
    0x00720308: image_box_tile_vertical_dimension
    0x00720310: image_box_scroll_direction
    0x00720312: image_box_small_scroll_type
    0x00720314: image_box_small_scroll_amount
    0x00720316: image_box_large_scroll_type
    0x00720318: image_box_large_scroll_amount
    0x00720320: image_box_overlap_priority
    0x00720330: cine_relative_to_real_time
    0x00720400: filter_operations_sequence
    0x00720402: filter_by_category
    0x00720404: filter_by_attribute_presence
    0x00720406: filter_by_operator
    0x00720420: structured_display_background_cielab_value
    0x00720421: empty_image_box_cielab_value
    0x00720422: structured_display_image_box_sequence
    0x00720424: structured_display_text_box_sequence
    0x00720427: referenced_first_frame_sequence
    0x00720430: image_box_synchronization_sequence
    0x00720432: synchronized_image_box_list
    0x00720434: type_of_synchronization
    0x00720500: blending_operation_type
    0x00720510: reformatting_operation_type
    0x00720512: reformatting_thickness
    0x00720514: reformatting_interval
    0x00720516: reformatting_operation_initial_view_direction
    0x00720520: tag_3d_rendering_type
    0x00720600: sorting_operations_sequence
    0x00720602: sort_by_category
    0x00720604: sorting_direction
    0x00720700: display_set_patient_orientation
    0x00720702: voi_type
    0x00720704: pseudo_color_type
    0x00720705: pseudo_color_palette_instance_reference_sequence
    0x00720706: show_grayscale_inverted
    0x00720710: show_image_true_size_flag
    0x00720712: show_graphic_annotation_flag
    0x00720714: show_patient_demographics_flag
    0x00720716: show_acquisition_techniques_flag
    0x00720717: display_set_horizontal_justification
    0x00720718: display_set_vertical_justification
    0x00740120: continuation_start_meterset
    0x00740121: continuation_end_meterset
    0x00741000: procedure_step_state
    0x00741002: procedure_step_progress_information_sequence
    0x00741004: procedure_step_progress
    0x00741006: procedure_step_progress_description
    0x00741008: procedure_step_communications_uri_sequence
    0x0074100A: contact_uri
    0x0074100C: contact_display_name
    0x0074100E: procedure_step_discontinuation_reason_code_sequence
    0x00741020: beam_task_sequence
    0x00741022: beam_task_type
    0x00741024: beam_order_index__trial_
    0x00741025: autosequence_flag
    0x00741026: table_top_vertical_adjusted_position
    0x00741027: table_top_longitudinal_adjusted_position
    0x00741028: table_top_lateral_adjusted_position
    0x0074102A: patient_support_adjusted_angle
    0x0074102B: table_top_eccentric_adjusted_angle
    0x0074102C: table_top_pitch_adjusted_angle
    0x0074102D: table_top_roll_adjusted_angle
    0x00741030: delivery_verification_image_sequence
    0x00741032: verification_image_timing
    0x00741034: double_exposure_flag
    0x00741036: double_exposure_ordering
    0x00741038: double_exposure_meterset__trial_
    0x0074103A: double_exposure_field_delta__trial_
    0x00741040: related_reference_rt_image_sequence
    0x00741042: general_machine_verification_sequence
    0x00741044: conventional_machine_verification_sequence
    0x00741046: ion_machine_verification_sequence
    0x00741048: failed_attributes_sequence
    0x0074104A: overridden_attributes_sequence
    0x0074104C: conventional_control_point_verification_sequence
    0x0074104E: ion_control_point_verification_sequence
    0x00741050: attribute_occurrence_sequence
    0x00741052: attribute_occurrence_pointer
    0x00741054: attribute_item_selector
    0x00741056: attribute_occurrence_private_creator
    0x00741057: selector_sequence_pointer_items
    0x00741200: scheduled_procedure_step_priority
    0x00741202: worklist_label
    0x00741204: procedure_step_label
    0x00741210: scheduled_processing_parameters_sequence
    0x00741212: performed_processing_parameters_sequence
    0x00741216: unified_procedure_step_performed_procedure_sequence
    0x00741220: related_procedure_step_sequence
    0x00741222: procedure_step_relationship_type
    0x00741224: replaced_procedure_step_sequence
    0x00741230: deletion_lock
    0x00741234: receiving_ae
    0x00741236: requesting_ae
    0x00741238: reason_for_cancellation
    0x00741242: scp_status
    0x00741244: subscription_list_status
    0x00741246: unified_procedure_step_list_status
    0x00741324: beam_order_index
    0x00741338: double_exposure_meterset
    0x0074133A: double_exposure_field_delta
    0x00741401: brachy_task_sequence
    0x00741402: continuation_start_total_reference_air_kerma
    0x00741403: continuation_end_total_reference_air_kerma
    0x00741404: continuation_pulse_number
    0x00741405: channel_delivery_order_sequence
    0x00741406: referenced_channel_number
    0x00741407: start_cumulative_time_weight
    0x00741408: end_cumulative_time_weight
    0x00741409: omitted_channel_sequence
    0x0074140A: reason_for_channel_omission
    0x0074140B: reason_for_channel_omission_description
    0x0074140C: channel_delivery_order_index
    0x0074140D: channel_delivery_continuation_sequence
    0x0074140E: omitted_application_setup_sequence
    0x00760001: implant_assembly_template_name
    0x00760003: implant_assembly_template_issuer
    0x00760006: implant_assembly_template_version
    0x00760008: replaced_implant_assembly_template_sequence
    0x0076000A: implant_assembly_template_type
    0x0076000C: original_implant_assembly_template_sequence
    0x0076000E: derivation_implant_assembly_template_sequence
    0x00760010: implant_assembly_template_target_anatomy_sequence
    0x00760020: procedure_type_code_sequence
    0x00760030: surgical_technique
    0x00760032: component_types_sequence
    0x00760034: component_type_code_sequence
    0x00760036: exclusive_component_type
    0x00760038: mandatory_component_type
    0x00760040: component_sequence
    0x00760055: component_id
    0x00760060: component_assembly_sequence
    0x00760070: component_1_referenced_id
    0x00760080: component_1_referenced_mating_feature_set_id
    0x00760090: component_1_referenced_mating_feature_id
    0x007600A0: component_2_referenced_id
    0x007600B0: component_2_referenced_mating_feature_set_id
    0x007600C0: component_2_referenced_mating_feature_id
    0x00780001: implant_template_group_name
    0x00780010: implant_template_group_description
    0x00780020: implant_template_group_issuer
    0x00780024: implant_template_group_version
    0x00780026: replaced_implant_template_group_sequence
    0x00780028: implant_template_group_target_anatomy_sequence
    0x0078002A: implant_template_group_members_sequence
    0x0078002E: implant_template_group_member_id
    0x00780050: tag_3d_implant_template_group_member_matching_point
    0x00780060: tag_3d_implant_template_group_member_matching_axes
    0x00780070: implant_template_group_member_matching_2d_coordinates_sequence
    0x00780090: tag_2d_implant_template_group_member_matching_point
    0x007800A0: tag_2d_implant_template_group_member_matching_axes
    0x007800B0: implant_template_group_variation_dimension_sequence
    0x007800B2: implant_template_group_variation_dimension_name
    0x007800B4: implant_template_group_variation_dimension_rank_sequence
    0x007800B6: referenced_implant_template_group_member_id
    0x007800B8: implant_template_group_variation_dimension_rank
    0x00800001: surface_scan_acquisition_type_code_sequence
    0x00800002: surface_scan_mode_code_sequence
    0x00800003: registration_method_code_sequence
    0x00800004: shot_duration_time
    0x00800005: shot_offset_time
    0x00800006: surface_point_presentation_value_data
    0x00800007: surface_point_color_cielab_value_data
    0x00800008: uv_mapping_sequence
    0x00800009: texture_label
    0x00800010: u_value_data
    0x00800011: v_value_data
    0x00800012: referenced_texture_sequence
    0x00800013: referenced_surface_data_sequence
    0x00820001: assessment_summary
    0x00820003: assessment_summary_description
    0x00820004: assessed_sop_instance_sequence
    0x00820005: referenced_comparison_sop_instance_sequence
    0x00820006: number_of_assessment_observations
    0x00820007: assessment_observations_sequence
    0x00820008: observation_significance
    0x0082000A: observation_description
    0x0082000C: structured_constraint_observation_sequence
    0x00820010: assessed_attribute_value_sequence
    0x00820016: assessment_set_id
    0x00820017: assessment_requester_sequence
    0x00820018: selector_attribute_name
    0x00820019: selector_attribute_keyword
    0x00820021: assessment_type_code_sequence
    0x00820022: observation_basis_code_sequence
    0x00820023: assessment_label
    0x00820032: constraint_type
    0x00820033: specification_selection_guidance
    0x00820034: constraint_value_sequence
    0x00820035: recommended_default_value_sequence
    0x00820036: constraint_violation_significance
    0x00820037: constraint_violation_condition
    0x00880130: storage_media_file_set_id
    0x00880140: storage_media_file_set_uid
    0x00880200: icon_image_sequence
    0x00880904: topic_title
    0x00880906: topic_subject
    0x00880910: topic_author
    0x00880912: topic_keywords
    0x01000410: sop_instance_status
    0x01000420: sop_authorization_datetime
    0x01000424: sop_authorization_comment
    0x01000426: authorization_equipment_certification_number
    0x04000005: mac_id_number
    0x04000010: mac_calculation_transfer_syntax_uid
    0x04000015: mac_algorithm
    0x04000020: data_elements_signed
    0x04000100: digital_signature_uid
    0x04000105: digital_signature_datetime
    0x04000110: certificate_type
    0x04000115: certificate_of_signer
    0x04000120: signature
    0x04000305: certified_timestamp_type
    0x04000310: certified_timestamp
    0x04000401: digital_signature_purpose_code_sequence
    0x04000402: referenced_digital_signature_sequence
    0x04000403: referenced_sop_instance_mac_sequence
    0x04000404: mac
    0x04000500: encrypted_attributes_sequence
    0x04000510: encrypted_content_transfer_syntax_uid
    0x04000520: encrypted_content
    0x04000550: modified_attributes_sequence
    0x04000561: original_attributes_sequence
    0x04000562: attribute_modification_datetime
    0x04000563: modifying_system
    0x04000564: source_of_previous_values
    0x04000565: reason_for_the_attribute_modification
    0x20000010: number_of_copies
    0x2000001E: printer_configuration_sequence
    0x20000020: print_priority
    0x20000030: medium_type
    0x20000040: film_destination
    0x20000050: film_session_label
    0x20000060: memory_allocation
    0x20000061: maximum_memory_allocation
    0x20000062: color_image_printing_flag
    0x20000063: collation_flag
    0x20000065: annotation_flag
    0x20000067: image_overlay_flag
    0x20000069: presentation_lut_flag
    0x2000006A: image_box_presentation_lut_flag
    0x200000A0: memory_bit_depth
    0x200000A1: printing_bit_depth
    0x200000A2: media_installed_sequence
    0x200000A4: other_media_available_sequence
    0x200000A8: supported_image_display_formats_sequence
    0x20000500: referenced_film_box_sequence
    0x20000510: referenced_stored_print_sequence
    0x20100010: image_display_format
    0x20100030: annotation_display_format_id
    0x20100040: film_orientation
    0x20100050: film_size_id
    0x20100052: printer_resolution_id
    0x20100054: default_printer_resolution_id
    0x20100060: magnification_type
    0x20100080: smoothing_type
    0x201000A6: default_magnification_type
    0x201000A7: other_magnification_types_available
    0x201000A8: default_smoothing_type
    0x201000A9: other_smoothing_types_available
    0x20100100: border_density
    0x20100110: empty_image_density
    0x20100120: min_density
    0x20100130: max_density
    0x20100140: trim
    0x20100150: configuration_information
    0x20100152: configuration_information_description
    0x20100154: maximum_collated_films
    0x2010015E: illumination
    0x20100160: reflected_ambient_light
    0x20100376: printer_pixel_spacing
    0x20100500: referenced_film_session_sequence
    0x20100510: referenced_image_box_sequence
    0x20100520: referenced_basic_annotation_box_sequence
    0x20200010: image_box_position
    0x20200020: polarity
    0x20200030: requested_image_size
    0x20200040: requested_decimate_crop_behavior
    0x20200050: requested_resolution_id
    0x202000A0: requested_image_size_flag
    0x202000A2: decimate_crop_result
    0x20200110: basic_grayscale_image_sequence
    0x20200111: basic_color_image_sequence
    0x20200130: referenced_image_overlay_box_sequence
    0x20200140: referenced_voi_lut_box_sequence
    0x20300010: annotation_position
    0x20300020: text_string
    0x20400010: referenced_overlay_plane_sequence
    0x20400011: referenced_overlay_plane_groups
    0x20400020: overlay_pixel_data_sequence
    0x20400060: overlay_magnification_type
    0x20400070: overlay_smoothing_type
    0x20400072: overlay_or_image_magnification
    0x20400074: magnify_to_number_of_columns
    0x20400080: overlay_foreground_density
    0x20400082: overlay_background_density
    0x20400090: overlay_mode
    0x20400100: threshold_density
    0x20400500: referenced_image_box_sequence__retired_
    0x20500010: presentation_lut_sequence
    0x20500020: presentation_lut_shape
    0x20500500: referenced_presentation_lut_sequence
    0x21000010: print_job_id
    0x21000020: execution_status
    0x21000030: execution_status_info
    0x21000040: creation_date
    0x21000050: creation_time
    0x21000070: originator
    0x21000140: destination_ae
    0x21000160: owner_id
    0x21000170: number_of_films
    0x21000500: referenced_print_job_sequence__pull_stored_print_
    0x21100010: printer_status
    0x21100020: printer_status_info
    0x21100030: printer_name
    0x21100099: print_queue_id
    0x21200010: queue_status
    0x21200050: print_job_description_sequence
    0x21200070: referenced_print_job_sequence
    0x21300010: print_management_capabilities_sequence
    0x21300015: printer_characteristics_sequence
    0x21300030: film_box_content_sequence
    0x21300040: image_box_content_sequence
    0x21300050: annotation_content_sequence
    0x21300060: image_overlay_box_content_sequence
    0x21300080: presentation_lut_content_sequence
    0x213000A0: proposed_study_sequence
    0x213000C0: original_image_sequence
    0x22000001: label_using_information_extracted_from_instances
    0x22000002: label_text
    0x22000003: label_style_selection
    0x22000004: media_disposition
    0x22000005: barcode_value
    0x22000006: barcode_symbology
    0x22000007: allow_media_splitting
    0x22000008: include_non_dicom_objects
    0x22000009: include_display_application
    0x2200000A: preserve_composite_instances_after_media_creation
    0x2200000B: total_number_of_pieces_of_media_created
    0x2200000C: requested_media_application_profile
    0x2200000D: referenced_storage_media_sequence
    0x2200000E: failure_attributes
    0x2200000F: allow_lossy_compression
    0x22000020: request_priority
    0x30020002: rt_image_label
    0x30020003: rt_image_name
    0x30020004: rt_image_description
    0x3002000A: reported_values_origin
    0x3002000C: rt_image_plane
    0x3002000D: x_ray_image_receptor_translation
    0x3002000E: x_ray_image_receptor_angle
    0x30020010: rt_image_orientation
    0x30020011: image_plane_pixel_spacing
    0x30020012: rt_image_position
    0x30020020: radiation_machine_name
    0x30020022: radiation_machine_sad
    0x30020024: radiation_machine_ssd
    0x30020026: rt_image_sid
    0x30020028: source_to_reference_object_distance
    0x30020029: fraction_number
    0x30020030: exposure_sequence
    0x30020032: meterset_exposure
    0x30020034: diaphragm_position
    0x30020040: fluence_map_sequence
    0x30020041: fluence_data_source
    0x30020042: fluence_data_scale
    0x30020050: primary_fluence_mode_sequence
    0x30020051: fluence_mode
    0x30020052: fluence_mode_id
    0x30040001: dvh_type
    0x30040002: dose_units
    0x30040004: dose_type
    0x30040005: spatial_transform_of_dose
    0x30040006: dose_comment
    0x30040008: normalization_point
    0x3004000A: dose_summation_type
    0x3004000C: grid_frame_offset_vector
    0x3004000E: dose_grid_scaling
    0x30040010: rt_dose_roi_sequence
    0x30040012: dose_value
    0x30040014: tissue_heterogeneity_correction
    0x30040040: dvh_normalization_point
    0x30040042: dvh_normalization_dose_value
    0x30040050: dvh_sequence
    0x30040052: dvh_dose_scaling
    0x30040054: dvh_volume_units
    0x30040056: dvh_number_of_bins
    0x30040058: dvh_data
    0x30040060: dvh_referenced_roi_sequence
    0x30040062: dvh_roi_contribution_type
    0x30040070: dvh_minimum_dose
    0x30040072: dvh_maximum_dose
    0x30040074: dvh_mean_dose
    0x30060002: structure_set_label
    0x30060004: structure_set_name
    0x30060006: structure_set_description
    0x30060008: structure_set_date
    0x30060009: structure_set_time
    0x30060010: referenced_frame_of_reference_sequence
    0x30060012: rt_referenced_study_sequence
    0x30060014: rt_referenced_series_sequence
    0x30060016: contour_image_sequence
    0x30060018: predecessor_structure_set_sequence
    0x30060020: structure_set_roi_sequence
    0x30060022: roi_number
    0x30060024: referenced_frame_of_reference_uid
    0x30060026: roi_name
    0x30060028: roi_description
    0x3006002A: roi_display_color
    0x3006002C: roi_volume
    0x30060030: rt_related_roi_sequence
    0x30060033: rt_roi_relationship
    0x30060036: roi_generation_algorithm
    0x30060038: roi_generation_description
    0x30060039: roi_contour_sequence
    0x30060040: contour_sequence
    0x30060042: contour_geometric_type
    0x30060044: contour_slab_thickness
    0x30060045: contour_offset_vector
    0x30060046: number_of_contour_points
    0x30060048: contour_number
    0x30060049: attached_contours
    0x30060050: contour_data
    0x30060080: rt_roi_observations_sequence
    0x30060082: observation_number
    0x30060084: referenced_roi_number
    0x30060085: roi_observation_label
    0x30060086: rt_roi_identification_code_sequence
    0x30060088: roi_observation_description
    0x300600A0: related_rt_roi_observations_sequence
    0x300600A4: rt_roi_interpreted_type
    0x300600A6: roi_interpreter
    0x300600B0: roi_physical_properties_sequence
    0x300600B2: roi_physical_property
    0x300600B4: roi_physical_property_value
    0x300600B6: roi_elemental_composition_sequence
    0x300600B7: roi_elemental_composition_atomic_number
    0x300600B8: roi_elemental_composition_atomic_mass_fraction
    0x300600B9: additional_rt_roi_identification_code_sequence
    0x300600C0: frame_of_reference_relationship_sequence
    0x300600C2: related_frame_of_reference_uid
    0x300600C4: frame_of_reference_transformation_type
    0x300600C6: frame_of_reference_transformation_matrix
    0x300600C8: frame_of_reference_transformation_comment
    0x30080010: measured_dose_reference_sequence
    0x30080012: measured_dose_description
    0x30080014: measured_dose_type
    0x30080016: measured_dose_value
    0x30080020: treatment_session_beam_sequence
    0x30080021: treatment_session_ion_beam_sequence
    0x30080022: current_fraction_number
    0x30080024: treatment_control_point_date
    0x30080025: treatment_control_point_time
    0x3008002A: treatment_termination_status
    0x3008002B: treatment_termination_code
    0x3008002C: treatment_verification_status
    0x30080030: referenced_treatment_record_sequence
    0x30080032: specified_primary_meterset
    0x30080033: specified_secondary_meterset
    0x30080036: delivered_primary_meterset
    0x30080037: delivered_secondary_meterset
    0x3008003A: specified_treatment_time
    0x3008003B: delivered_treatment_time
    0x30080040: control_point_delivery_sequence
    0x30080041: ion_control_point_delivery_sequence
    0x30080042: specified_meterset
    0x30080044: delivered_meterset
    0x30080045: meterset_rate_set
    0x30080046: meterset_rate_delivered
    0x30080047: scan_spot_metersets_delivered
    0x30080048: dose_rate_delivered
    0x30080050: treatment_summary_calculated_dose_reference_sequence
    0x30080052: cumulative_dose_to_dose_reference
    0x30080054: first_treatment_date
    0x30080056: most_recent_treatment_date
    0x3008005A: number_of_fractions_delivered
    0x30080060: override_sequence
    0x30080061: parameter_sequence_pointer
    0x30080062: override_parameter_pointer
    0x30080063: parameter_item_index
    0x30080064: measured_dose_reference_number
    0x30080065: parameter_pointer
    0x30080066: override_reason
    0x30080068: corrected_parameter_sequence
    0x3008006A: correction_value
    0x30080070: calculated_dose_reference_sequence
    0x30080072: calculated_dose_reference_number
    0x30080074: calculated_dose_reference_description
    0x30080076: calculated_dose_reference_dose_value
    0x30080078: start_meterset
    0x3008007A: end_meterset
    0x30080080: referenced_measured_dose_reference_sequence
    0x30080082: referenced_measured_dose_reference_number
    0x30080090: referenced_calculated_dose_reference_sequence
    0x30080092: referenced_calculated_dose_reference_number
    0x300800A0: beam_limiting_device_leaf_pairs_sequence
    0x300800B0: recorded_wedge_sequence
    0x300800C0: recorded_compensator_sequence
    0x300800D0: recorded_block_sequence
    0x300800E0: treatment_summary_measured_dose_reference_sequence
    0x300800F0: recorded_snout_sequence
    0x300800F2: recorded_range_shifter_sequence
    0x300800F4: recorded_lateral_spreading_device_sequence
    0x300800F6: recorded_range_modulator_sequence
    0x30080100: recorded_source_sequence
    0x30080105: source_serial_number
    0x30080110: treatment_session_application_setup_sequence
    0x30080116: application_setup_check
    0x30080120: recorded_brachy_accessory_device_sequence
    0x30080122: referenced_brachy_accessory_device_number
    0x30080130: recorded_channel_sequence
    0x30080132: specified_channel_total_time
    0x30080134: delivered_channel_total_time
    0x30080136: specified_number_of_pulses
    0x30080138: delivered_number_of_pulses
    0x3008013A: specified_pulse_repetition_interval
    0x3008013C: delivered_pulse_repetition_interval
    0x30080140: recorded_source_applicator_sequence
    0x30080142: referenced_source_applicator_number
    0x30080150: recorded_channel_shield_sequence
    0x30080152: referenced_channel_shield_number
    0x30080160: brachy_control_point_delivered_sequence
    0x30080162: safe_position_exit_date
    0x30080164: safe_position_exit_time
    0x30080166: safe_position_return_date
    0x30080168: safe_position_return_time
    0x30080171: pulse_specific_brachy_control_point_delivered_sequence
    0x30080172: pulse_number
    0x30080173: brachy_pulse_control_point_delivered_sequence
    0x30080200: current_treatment_status
    0x30080202: treatment_status_comment
    0x30080220: fraction_group_summary_sequence
    0x30080223: referenced_fraction_number
    0x30080224: fraction_group_type
    0x30080230: beam_stopper_position
    0x30080240: fraction_status_summary_sequence
    0x30080250: treatment_date
    0x30080251: treatment_time
    0x300A0002: rt_plan_label
    0x300A0003: rt_plan_name
    0x300A0004: rt_plan_description
    0x300A0006: rt_plan_date
    0x300A0007: rt_plan_time
    0x300A0009: treatment_protocols
    0x300A000A: plan_intent
    0x300A000B: treatment_sites
    0x300A000C: rt_plan_geometry
    0x300A000E: prescription_description
    0x300A0010: dose_reference_sequence
    0x300A0012: dose_reference_number
    0x300A0013: dose_reference_uid
    0x300A0014: dose_reference_structure_type
    0x300A0015: nominal_beam_energy_unit
    0x300A0016: dose_reference_description
    0x300A0018: dose_reference_point_coordinates
    0x300A001A: nominal_prior_dose
    0x300A0020: dose_reference_type
    0x300A0021: constraint_weight
    0x300A0022: delivery_warning_dose
    0x300A0023: delivery_maximum_dose
    0x300A0025: target_minimum_dose
    0x300A0026: target_prescription_dose
    0x300A0027: target_maximum_dose
    0x300A0028: target_underdose_volume_fraction
    0x300A002A: organ_at_risk_full_volume_dose
    0x300A002B: organ_at_risk_limit_dose
    0x300A002C: organ_at_risk_maximum_dose
    0x300A002D: organ_at_risk_overdose_volume_fraction
    0x300A0040: tolerance_table_sequence
    0x300A0042: tolerance_table_number
    0x300A0043: tolerance_table_label
    0x300A0044: gantry_angle_tolerance
    0x300A0046: beam_limiting_device_angle_tolerance
    0x300A0048: beam_limiting_device_tolerance_sequence
    0x300A004A: beam_limiting_device_position_tolerance
    0x300A004B: snout_position_tolerance
    0x300A004C: patient_support_angle_tolerance
    0x300A004E: table_top_eccentric_angle_tolerance
    0x300A004F: table_top_pitch_angle_tolerance
    0x300A0050: table_top_roll_angle_tolerance
    0x300A0051: table_top_vertical_position_tolerance
    0x300A0052: table_top_longitudinal_position_tolerance
    0x300A0053: table_top_lateral_position_tolerance
    0x300A0055: rt_plan_relationship
    0x300A0070: fraction_group_sequence
    0x300A0071: fraction_group_number
    0x300A0072: fraction_group_description
    0x300A0078: number_of_fractions_planned
    0x300A0079: number_of_fraction_pattern_digits_per_day
    0x300A007A: repeat_fraction_cycle_length
    0x300A007B: fraction_pattern
    0x300A0080: number_of_beams
    0x300A0082: beam_dose_specification_point
    0x300A0084: beam_dose
    0x300A0086: beam_meterset
    0x300A0088: beam_dose_point_depth
    0x300A0089: beam_dose_point_equivalent_depth
    0x300A008A: beam_dose_point_ssd
    0x300A008B: beam_dose_meaning
    0x300A008C: beam_dose_verification_control_point_sequence
    0x300A008D: average_beam_dose_point_depth
    0x300A008E: average_beam_dose_point_equivalent_depth
    0x300A008F: average_beam_dose_point_ssd
    0x300A0090: beam_dose_type
    0x300A0091: alternate_beam_dose
    0x300A0092: alternate_beam_dose_type
    0x300A00A0: number_of_brachy_application_setups
    0x300A00A2: brachy_application_setup_dose_specification_point
    0x300A00A4: brachy_application_setup_dose
    0x300A00B0: beam_sequence
    0x300A00B2: treatment_machine_name
    0x300A00B3: primary_dosimeter_unit
    0x300A00B4: source_axis_distance
    0x300A00B6: beam_limiting_device_sequence
    0x300A00B8: rt_beam_limiting_device_type
    0x300A00BA: source_to_beam_limiting_device_distance
    0x300A00BB: isocenter_to_beam_limiting_device_distance
    0x300A00BC: number_of_leaf_jaw_pairs
    0x300A00BE: leaf_position_boundaries
    0x300A00C0: beam_number
    0x300A00C2: beam_name
    0x300A00C3: beam_description
    0x300A00C4: beam_type
    0x300A00C5: beam_delivery_duration_limit
    0x300A00C6: radiation_type
    0x300A00C7: high_dose_technique_type
    0x300A00C8: reference_image_number
    0x300A00CA: planned_verification_image_sequence
    0x300A00CC: imaging_device_specific_acquisition_parameters
    0x300A00CE: treatment_delivery_type
    0x300A00D0: number_of_wedges
    0x300A00D1: wedge_sequence
    0x300A00D2: wedge_number
    0x300A00D3: wedge_type
    0x300A00D4: wedge_id
    0x300A00D5: wedge_angle
    0x300A00D6: wedge_factor
    0x300A00D7: total_wedge_tray_water_equivalent_thickness
    0x300A00D8: wedge_orientation
    0x300A00D9: isocenter_to_wedge_tray_distance
    0x300A00DA: source_to_wedge_tray_distance
    0x300A00DB: wedge_thin_edge_position
    0x300A00DC: bolus_id
    0x300A00DD: bolus_description
    0x300A00DE: effective_wedge_angle
    0x300A00E0: number_of_compensators
    0x300A00E1: material_id
    0x300A00E2: total_compensator_tray_factor
    0x300A00E3: compensator_sequence
    0x300A00E4: compensator_number
    0x300A00E5: compensator_id
    0x300A00E6: source_to_compensator_tray_distance
    0x300A00E7: compensator_rows
    0x300A00E8: compensator_columns
    0x300A00E9: compensator_pixel_spacing
    0x300A00EA: compensator_position
    0x300A00EB: compensator_transmission_data
    0x300A00EC: compensator_thickness_data
    0x300A00ED: number_of_boli
    0x300A00EE: compensator_type
    0x300A00EF: compensator_tray_id
    0x300A00F0: number_of_blocks
    0x300A00F2: total_block_tray_factor
    0x300A00F3: total_block_tray_water_equivalent_thickness
    0x300A00F4: block_sequence
    0x300A00F5: block_tray_id
    0x300A00F6: source_to_block_tray_distance
    0x300A00F7: isocenter_to_block_tray_distance
    0x300A00F8: block_type
    0x300A00F9: accessory_code
    0x300A00FA: block_divergence
    0x300A00FB: block_mounting_position
    0x300A00FC: block_number
    0x300A00FE: block_name
    0x300A0100: block_thickness
    0x300A0102: block_transmission
    0x300A0104: block_number_of_points
    0x300A0106: block_data
    0x300A0107: applicator_sequence
    0x300A0108: applicator_id
    0x300A0109: applicator_type
    0x300A010A: applicator_description
    0x300A010C: cumulative_dose_reference_coefficient
    0x300A010E: final_cumulative_meterset_weight
    0x300A0110: number_of_control_points
    0x300A0111: control_point_sequence
    0x300A0112: control_point_index
    0x300A0114: nominal_beam_energy
    0x300A0115: dose_rate_set
    0x300A0116: wedge_position_sequence
    0x300A0118: wedge_position
    0x300A011A: beam_limiting_device_position_sequence
    0x300A011C: leaf_jaw_positions
    0x300A011E: gantry_angle
    0x300A011F: gantry_rotation_direction
    0x300A0120: beam_limiting_device_angle
    0x300A0121: beam_limiting_device_rotation_direction
    0x300A0122: patient_support_angle
    0x300A0123: patient_support_rotation_direction
    0x300A0124: table_top_eccentric_axis_distance
    0x300A0125: table_top_eccentric_angle
    0x300A0126: table_top_eccentric_rotation_direction
    0x300A0128: table_top_vertical_position
    0x300A0129: table_top_longitudinal_position
    0x300A012A: table_top_lateral_position
    0x300A012C: isocenter_position
    0x300A012E: surface_entry_point
    0x300A0130: source_to_surface_distance
    0x300A0131: average_beam_dose_point_source_to_external_contour_distance
    0x300A0132: source_to_external_contour_distance
    0x300A0133: external_contour_entry_point
    0x300A0134: cumulative_meterset_weight
    0x300A0140: table_top_pitch_angle
    0x300A0142: table_top_pitch_rotation_direction
    0x300A0144: table_top_roll_angle
    0x300A0146: table_top_roll_rotation_direction
    0x300A0148: head_fixation_angle
    0x300A014A: gantry_pitch_angle
    0x300A014C: gantry_pitch_rotation_direction
    0x300A014E: gantry_pitch_angle_tolerance
    0x300A0150: fixation_eye
    0x300A0151: chair_head_frame_position
    0x300A0152: head_fixation_angle_tolerance
    0x300A0153: chair_head_frame_position_tolerance
    0x300A0154: fixation_light_azimuthal_angle_tolerance
    0x300A0155: fixation_light_polar_angle_tolerance
    0x300A0180: patient_setup_sequence
    0x300A0182: patient_setup_number
    0x300A0183: patient_setup_label
    0x300A0184: patient_additional_position
    0x300A0190: fixation_device_sequence
    0x300A0192: fixation_device_type
    0x300A0194: fixation_device_label
    0x300A0196: fixation_device_description
    0x300A0198: fixation_device_position
    0x300A0199: fixation_device_pitch_angle
    0x300A019A: fixation_device_roll_angle
    0x300A01A0: shielding_device_sequence
    0x300A01A2: shielding_device_type
    0x300A01A4: shielding_device_label
    0x300A01A6: shielding_device_description
    0x300A01A8: shielding_device_position
    0x300A01B0: setup_technique
    0x300A01B2: setup_technique_description
    0x300A01B4: setup_device_sequence
    0x300A01B6: setup_device_type
    0x300A01B8: setup_device_label
    0x300A01BA: setup_device_description
    0x300A01BC: setup_device_parameter
    0x300A01D0: setup_reference_description
    0x300A01D2: table_top_vertical_setup_displacement
    0x300A01D4: table_top_longitudinal_setup_displacement
    0x300A01D6: table_top_lateral_setup_displacement
    0x300A0200: brachy_treatment_technique
    0x300A0202: brachy_treatment_type
    0x300A0206: treatment_machine_sequence
    0x300A0210: source_sequence
    0x300A0212: source_number
    0x300A0214: source_type
    0x300A0216: source_manufacturer
    0x300A0218: active_source_diameter
    0x300A021A: active_source_length
    0x300A021B: source_model_id
    0x300A021C: source_description
    0x300A0222: source_encapsulation_nominal_thickness
    0x300A0224: source_encapsulation_nominal_transmission
    0x300A0226: source_isotope_name
    0x300A0228: source_isotope_half_life
    0x300A0229: source_strength_units
    0x300A022A: reference_air_kerma_rate
    0x300A022B: source_strength
    0x300A022C: source_strength_reference_date
    0x300A022E: source_strength_reference_time
    0x300A0230: application_setup_sequence
    0x300A0232: application_setup_type
    0x300A0234: application_setup_number
    0x300A0236: application_setup_name
    0x300A0238: application_setup_manufacturer
    0x300A0240: template_number
    0x300A0242: template_type
    0x300A0244: template_name
    0x300A0250: total_reference_air_kerma
    0x300A0260: brachy_accessory_device_sequence
    0x300A0262: brachy_accessory_device_number
    0x300A0263: brachy_accessory_device_id
    0x300A0264: brachy_accessory_device_type
    0x300A0266: brachy_accessory_device_name
    0x300A026A: brachy_accessory_device_nominal_thickness
    0x300A026C: brachy_accessory_device_nominal_transmission
    0x300A0280: channel_sequence
    0x300A0282: channel_number
    0x300A0284: channel_length
    0x300A0286: channel_total_time
    0x300A0288: source_movement_type
    0x300A028A: number_of_pulses
    0x300A028C: pulse_repetition_interval
    0x300A0290: source_applicator_number
    0x300A0291: source_applicator_id
    0x300A0292: source_applicator_type
    0x300A0294: source_applicator_name
    0x300A0296: source_applicator_length
    0x300A0298: source_applicator_manufacturer
    0x300A029C: source_applicator_wall_nominal_thickness
    0x300A029E: source_applicator_wall_nominal_transmission
    0x300A02A0: source_applicator_step_size
    0x300A02A2: transfer_tube_number
    0x300A02A4: transfer_tube_length
    0x300A02B0: channel_shield_sequence
    0x300A02B2: channel_shield_number
    0x300A02B3: channel_shield_id
    0x300A02B4: channel_shield_name
    0x300A02B8: channel_shield_nominal_thickness
    0x300A02BA: channel_shield_nominal_transmission
    0x300A02C8: final_cumulative_time_weight
    0x300A02D0: brachy_control_point_sequence
    0x300A02D2: control_point_relative_position
    0x300A02D4: control_point_3d_position
    0x300A02D6: cumulative_time_weight
    0x300A02E0: compensator_divergence
    0x300A02E1: compensator_mounting_position
    0x300A02E2: source_to_compensator_distance
    0x300A02E3: total_compensator_tray_water_equivalent_thickness
    0x300A02E4: isocenter_to_compensator_tray_distance
    0x300A02E5: compensator_column_offset
    0x300A02E6: isocenter_to_compensator_distances
    0x300A02E7: compensator_relative_stopping_power_ratio
    0x300A02E8: compensator_milling_tool_diameter
    0x300A02EA: ion_range_compensator_sequence
    0x300A02EB: compensator_description
    0x300A0302: radiation_mass_number
    0x300A0304: radiation_atomic_number
    0x300A0306: radiation_charge_state
    0x300A0308: scan_mode
    0x300A0309: modulated_scan_mode_type
    0x300A030A: virtual_source_axis_distances
    0x300A030C: snout_sequence
    0x300A030D: snout_position
    0x300A030F: snout_id
    0x300A0312: number_of_range_shifters
    0x300A0314: range_shifter_sequence
    0x300A0316: range_shifter_number
    0x300A0318: range_shifter_id
    0x300A0320: range_shifter_type
    0x300A0322: range_shifter_description
    0x300A0330: number_of_lateral_spreading_devices
    0x300A0332: lateral_spreading_device_sequence
    0x300A0334: lateral_spreading_device_number
    0x300A0336: lateral_spreading_device_id
    0x300A0338: lateral_spreading_device_type
    0x300A033A: lateral_spreading_device_description
    0x300A033C: lateral_spreading_device_water_equivalent_thickness
    0x300A0340: number_of_range_modulators
    0x300A0342: range_modulator_sequence
    0x300A0344: range_modulator_number
    0x300A0346: range_modulator_id
    0x300A0348: range_modulator_type
    0x300A034A: range_modulator_description
    0x300A034C: beam_current_modulation_id
    0x300A0350: patient_support_type
    0x300A0352: patient_support_id
    0x300A0354: patient_support_accessory_code
    0x300A0355: tray_accessory_code
    0x300A0356: fixation_light_azimuthal_angle
    0x300A0358: fixation_light_polar_angle
    0x300A035A: meterset_rate
    0x300A0360: range_shifter_settings_sequence
    0x300A0362: range_shifter_setting
    0x300A0364: isocenter_to_range_shifter_distance
    0x300A0366: range_shifter_water_equivalent_thickness
    0x300A0370: lateral_spreading_device_settings_sequence
    0x300A0372: lateral_spreading_device_setting
    0x300A0374: isocenter_to_lateral_spreading_device_distance
    0x300A0380: range_modulator_settings_sequence
    0x300A0382: range_modulator_gating_start_value
    0x300A0384: range_modulator_gating_stop_value
    0x300A0386: range_modulator_gating_start_water_equivalent_thickness
    0x300A0388: range_modulator_gating_stop_water_equivalent_thickness
    0x300A038A: isocenter_to_range_modulator_distance
    0x300A0390: scan_spot_tune_id
    0x300A0391: scan_spot_prescribed_indices
    0x300A0392: number_of_scan_spot_positions
    0x300A0393: scan_spot_reordered
    0x300A0394: scan_spot_position_map
    0x300A0395: scan_spot_reordering_allowed
    0x300A0396: scan_spot_meterset_weights
    0x300A0398: scanning_spot_size
    0x300A039A: number_of_paintings
    0x300A03A0: ion_tolerance_table_sequence
    0x300A03A2: ion_beam_sequence
    0x300A03A4: ion_beam_limiting_device_sequence
    0x300A03A6: ion_block_sequence
    0x300A03A8: ion_control_point_sequence
    0x300A03AA: ion_wedge_sequence
    0x300A03AC: ion_wedge_position_sequence
    0x300A0401: referenced_setup_image_sequence
    0x300A0402: setup_image_comment
    0x300A0410: motion_synchronization_sequence
    0x300A0412: control_point_orientation
    0x300A0420: general_accessory_sequence
    0x300A0421: general_accessory_id
    0x300A0422: general_accessory_description
    0x300A0423: general_accessory_type
    0x300A0424: general_accessory_number
    0x300A0425: source_to_general_accessory_distance
    0x300A0431: applicator_geometry_sequence
    0x300A0432: applicator_aperture_shape
    0x300A0433: applicator_opening
    0x300A0434: applicator_opening_x
    0x300A0435: applicator_opening_y
    0x300A0436: source_to_applicator_mounting_position_distance
    0x300A0440: number_of_block_slab_items
    0x300A0441: block_slab_sequence
    0x300A0442: block_slab_thickness
    0x300A0443: block_slab_number
    0x300A0450: device_motion_control_sequence
    0x300A0451: device_motion_execution_mode
    0x300A0452: device_motion_observation_mode
    0x300A0453: device_motion_parameter_code_sequence
    0x300A0501: distal_depth_fraction
    0x300A0502: distal_depth
    0x300A0503: nominal_range_modulation_fractions
    0x300A0504: nominal_range_modulated_region_depths
    0x300A0505: depth_dose_parameters_sequence
    0x300A0506: delivered_depth_dose_parameters_sequence
    0x300A0507: delivered_distal_depth_fraction
    0x300A0508: delivered_distal_depth
    0x300A0509: delivered_nominal_range_modulation_fractions
    0x300A0510: delivered_nominal_range_modulated_region_depths
    0x300A0511: delivered_reference_dose_definition
    0x300A0512: reference_dose_definition
    0x300C0002: referenced_rt_plan_sequence
    0x300C0004: referenced_beam_sequence
    0x300C0006: referenced_beam_number
    0x300C0007: referenced_reference_image_number
    0x300C0008: start_cumulative_meterset_weight
    0x300C0009: end_cumulative_meterset_weight
    0x300C000A: referenced_brachy_application_setup_sequence
    0x300C000C: referenced_brachy_application_setup_number
    0x300C000E: referenced_source_number
    0x300C0020: referenced_fraction_group_sequence
    0x300C0022: referenced_fraction_group_number
    0x300C0040: referenced_verification_image_sequence
    0x300C0042: referenced_reference_image_sequence
    0x300C0050: referenced_dose_reference_sequence
    0x300C0051: referenced_dose_reference_number
    0x300C0055: brachy_referenced_dose_reference_sequence
    0x300C0060: referenced_structure_set_sequence
    0x300C006A: referenced_patient_setup_number
    0x300C0080: referenced_dose_sequence
    0x300C00A0: referenced_tolerance_table_number
    0x300C00B0: referenced_bolus_sequence
    0x300C00C0: referenced_wedge_number
    0x300C00D0: referenced_compensator_number
    0x300C00E0: referenced_block_number
    0x300C00F0: referenced_control_point_index
    0x300C00F2: referenced_control_point_sequence
    0x300C00F4: referenced_start_control_point_index
    0x300C00F6: referenced_stop_control_point_index
    0x300C0100: referenced_range_shifter_number
    0x300C0102: referenced_lateral_spreading_device_number
    0x300C0104: referenced_range_modulator_number
    0x300C0111: omitted_beam_task_sequence
    0x300C0112: reason_for_omission
    0x300C0113: reason_for_omission_description
    0x300E0002: approval_status
    0x300E0004: review_date
    0x300E0005: review_time
    0x300E0008: reviewer_name
    0x40000010: arbitrary
    0x40004000: text_comments
    0x40080040: results_id
    0x40080042: results_id_issuer
    0x40080050: referenced_interpretation_sequence
    0x400800FF: report_production_status__trial_
    0x40080100: interpretation_recorded_date
    0x40080101: interpretation_recorded_time
    0x40080102: interpretation_recorder
    0x40080103: reference_to_recorded_sound
    0x40080108: interpretation_transcription_date
    0x40080109: interpretation_transcription_time
    0x4008010A: interpretation_transcriber
    0x4008010B: interpretation_text
    0x4008010C: interpretation_author
    0x40080111: interpretation_approver_sequence
    0x40080112: interpretation_approval_date
    0x40080113: interpretation_approval_time
    0x40080114: physician_approving_interpretation
    0x40080115: interpretation_diagnosis_description
    0x40080117: interpretation_diagnosis_code_sequence
    0x40080118: results_distribution_list_sequence
    0x40080119: distribution_name
    0x4008011A: distribution_address
    0x40080200: interpretation_id
    0x40080202: interpretation_id_issuer
    0x40080210: interpretation_type_id
    0x40080212: interpretation_status_id
    0x40080300: impressions
    0x40084000: results_comments
    0x40100001: low_energy_detectors
    0x40100002: high_energy_detectors
    0x40100004: detector_geometry_sequence
    0x40101001: threat_roi_voxel_sequence
    0x40101004: threat_roi_base
    0x40101005: threat_roi_extents
    0x40101006: threat_roi_bitmap
    0x40101007: route_segment_id
    0x40101008: gantry_type
    0x40101009: ooi_owner_type
    0x4010100A: route_segment_sequence
    0x40101010: potential_threat_object_id
    0x40101011: threat_sequence
    0x40101012: threat_category
    0x40101013: threat_category_description
    0x40101014: atd_ability_assessment
    0x40101015: atd_assessment_flag
    0x40101016: atd_assessment_probability
    0x40101017: mass
    0x40101018: density
    0x40101019: z_effective
    0x4010101A: boarding_pass_id
    0x4010101B: center_of_mass
    0x4010101C: center_of_pto
    0x4010101D: bounding_polygon
    0x4010101E: route_segment_start_location_id
    0x4010101F: route_segment_end_location_id
    0x40101020: route_segment_location_id_type
    0x40101021: abort_reason
    0x40101023: volume_of_pto
    0x40101024: abort_flag
    0x40101025: route_segment_start_time
    0x40101026: route_segment_end_time
    0x40101027: tdr_type
    0x40101028: international_route_segment
    0x40101029: threat_detection_algorithm_and_version
    0x4010102A: assigned_location
    0x4010102B: alarm_decision_time
    0x40101031: alarm_decision
    0x40101033: number_of_total_objects
    0x40101034: number_of_alarm_objects
    0x40101037: pto_representation_sequence
    0x40101038: atd_assessment_sequence
    0x40101039: tip_type
    0x4010103A: version
    0x40101041: ooi_owner_creation_time
    0x40101042: ooi_type
    0x40101043: ooi_size
    0x40101044: acquisition_status
    0x40101045: basis_materials_code_sequence
    0x40101046: phantom_type
    0x40101047: ooi_owner_sequence
    0x40101048: scan_type
    0x40101051: itinerary_id
    0x40101052: itinerary_id_type
    0x40101053: itinerary_id_assigning_authority
    0x40101054: route_id
    0x40101055: route_id_assigning_authority
    0x40101056: inbound_arrival_type
    0x40101058: carrier_id
    0x40101059: carrier_id_assigning_authority
    0x40101060: source_orientation
    0x40101061: source_position
    0x40101062: belt_height
    0x40101064: algorithm_routing_code_sequence
    0x40101067: transport_classification
    0x40101068: ooi_type_descriptor
    0x40101069: total_processing_time
    0x4010106C: detector_calibration_data
    0x4010106D: additional_screening_performed
    0x4010106E: additional_inspection_selection_criteria
    0x4010106F: additional_inspection_method_sequence
    0x40101070: ait_device_type
    0x40101071: qr_measurements_sequence
    0x40101072: target_material_sequence
    0x40101073: snr_threshold
    0x40101075: image_scale_representation
    0x40101076: referenced_pto_sequence
    0x40101077: referenced_tdr_instance_sequence
    0x40101078: pto_location_description
    0x40101079: anomaly_locator_indicator_sequence
    0x4010107A: anomaly_locator_indicator
    0x4010107B: pto_region_sequence
    0x4010107C: inspection_selection_criteria
    0x4010107D: secondary_inspection_method_sequence
    0x4010107E: prcs_to_rcs_orientation
    0x4FFE0001: mac_parameters_sequence
    0x52009229: shared_functional_groups_sequence
    0x52009230: per_frame_functional_groups_sequence
    0x54000100: waveform_sequence
    0x54000110: channel_minimum_value
    0x54000112: channel_maximum_value
    0x54001004: waveform_bits_allocated
    0x54001006: waveform_sample_interpretation
    0x5400100A: waveform_padding_value
    0x54001010: waveform_data
    0x56000010: first_order_phase_correction_angle
    0x56000020: spectroscopy_data
    0x7FE00008: float_pixel_data
    0x7FE00009: double_float_pixel_data
    0x7FE00010: pixel_data
    0x7FE00020: coefficients_sdvn
    0x7FE00030: coefficients_sdhn
    0x7FE00040: coefficients_sddn
    0xFFFAFFFA: digital_signatures_sequence
    0xFFFCFFFC: data_set_trailing_padding
    0xFFFEE000: item
    0xFFFEE00D: item_delimitation_item
    0xFFFEE0DD: sequence_delimitation_item
