const { extname } = require('path')

const DELIMETER = '@'

/**
 * Get image size variations
 * @param {string}   src                    Path to image with filename
 * @param {object[]} images             Specific object with information about each image.
 *                                          Generated by `grunt-image-size` task
 * @param {string}   [delimeter=DELIMETER]  Delimeters, which marks filename suffix. For instance,
 *                                          `@` for `image@200.jpg`, which is 200px variant
 * @return {object[]} Set of images size variations
 */
const getImageSet = (src, images, delimeter = DELIMETER) => {
  const filepath = src.replace(new RegExp(`${extname(src)}$`), '')

  return images.filter(({ name }) =>
    name.startsWith(`${filepath}${DELIMETER}`) || name === src
  )
}

/**
 * Make `srcset` content for `<img>` tag
 * @param {object[]} set Specific object with set of images, which represetns same image
 *                       in varius sizes
 * @return {string} Formed `srcset` attribute
 */
const printImageSet = (set) =>
  set.reduce((srcset, { name, width }, i) => {
    const isLast = i + 1 === set.length

    return `${srcset}${name} ${width}w${isLast ? '' : ', '}`
  }, '')

/**
 * Get image size and its srcset
 * @param {string}   src        Path to image with filename
 * @param {object[]} images Specific object with information about each image.
 *                              Generated by `grunt-image-size` task
 * @return {object} Image dimensions info
 */
const imageSize = (src, images) => {
  if (typeof src !== 'string') throw new Error(`[imageSize] \`src\` must be a string, \`${typeof src}\` provided`)

  if (src.startsWith('http') || src.startsWith('ftp')) return { src, srcset: () => null }

  if (!src.startsWith('/')) throw new Error(`[imageSize] document-relative urls not supported yet, \`${src}\` provided`)
  if (!Array.isArray(images)) throw new Error(`[imageSize] \`images\` must be an array, \`${typeof images}\` provided`)

  const image = images.find((e) => src === e.name)

  if (!image) throw new Error(`[imageSize] image \`${src}\` did not match any image data`)

  return {
    src,
    width: image.width,
    height: image.height,
    set: () => getImageSet(src, images),
    srcset: () => printImageSet(getImageSet(src, images))
  }
}

module.exports = imageSize
