/*!
 * @file      lorawan_key_config.h
 *
 * @brief     End device lorawan key configuration
 *
 * The Clear BSD License
 * Copyright Semtech Corporation 2021. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the Semtech corporation nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY
 * THIS LICENSE. THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SEMTECH CORPORATION BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef LORAWAN_KEY_CONFIG_H
#define LORAWAN_KEY_CONFIG_H

#ifdef __cplusplus
extern "C" {
#endif

/*
 * -----------------------------------------------------------------------------
 * --- DEPENDENCIES ------------------------------------------------------------
 */

#include <stdint.h>

/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC MACROS -----------------------------------------------------------
 */

/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC CONSTANTS --------------------------------------------------------
 */

/*!
 ******************************************************************************
 ********************************** WARNING ***********************************
 ******************************************************************************
  The crypto-element implementation supports both 1.0.x and 1.1.x LoRaWAN
  versions of the specification.
  Thus it has been decided to use the 1.1.x keys and EUI name definitions.
  The below table shows the names equivalence between versions:
               +---------------------+-------------------------+
               |       1.0.x         |          1.1.x          |
               +=====================+=========================+
               | LORAWAN_DEVICE_EUI  | LORAWAN_DEVICE_EUI      |
               +---------------------+-------------------------+
               | LORAWAN_APP_EUI     | LORAWAN_JOIN_EUI        |
               +---------------------+-------------------------+
               | LORAWAN_APP_KEY     | LORAWAN_NWK_KEY         |
               +---------------------+-------------------------+

 ******************************************************************************
 ******************************************************************************
 ******************************************************************************
 */

/*!
 * @brief IEEE Organizationally Unique Identifier ( OUI ) (big endian)
 * @remark This is unique to a company or organization
 */
#define IEEE_OUI 0xFE, 0xFF, 0xFF

/*!
 * @brief Tracker device IEEE EUI (big endian)
 *
 * @remark In this application the value is automatically generated by calling
 *         BoardGetUniqueId function
 */
#define LORAWAN_DEVICE_EUI_DEFAULT             \
    {                                          \
        IEEE_OUI, 0xFF, 0xFD, 0xFF, 0x00, 0x00 \
    }
#define LORAWAN_DEVICE_EUI_LEN 8

/*!
 * @brief App/Join server IEEE EUI (big endian)
 */
#define LORAWAN_JOIN_EUI_DEFAULT                       \
    {                                                  \
        0x00, 0x16, 0xC0, 0x01, 0xFF, 0xFE, 0x00, 0x01 \
    }
#define LORAWAN_JOIN_EUI_LEN 8

/*!
 * @brief loRaWAN Application Key (big endian)
 */
#define LORAWAN_APP_KEY_DEFAULT                                                                        \
    {                                                                                                  \
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 \
    }
#define LORAWAN_APP_KEY_LEN 16

/*!
 * @brief LoRaWAN region - see @ref smtc_modem_region_t
 */
#define LORAWAN_REGION_DEFAULT SMTC_MODEM_REGION_EU_868

/*!
 * @brief LoRaWAN regulatory sub region country. define LoRaWAN subregion countries, 0 means disable, 1 means enable
 */
#define ACTIVATE_SUB_REGION_JAPAN 0

/*!
 * @brief LoRaWAN class - see @ref smtc_modem_class_t
 */
#define LORAWAN_CLASS_DEFAULT SMTC_MODEM_CLASS_A

#ifndef LORAWAN_DEVICE_EUI
#define LORAWAN_DEVICE_EUI LORAWAN_DEVICE_EUI_DEFAULT
#endif  // LORAWAN_DEVICE_EUI

#ifndef LORAWAN_JOIN_EUI
#define LORAWAN_JOIN_EUI LORAWAN_JOIN_EUI_DEFAULT
#endif  // LORAWAN_JOIN_EUI

#ifndef LORAWAN_APP_KEY
#define LORAWAN_APP_KEY LORAWAN_APP_KEY_DEFAULT
#endif  // LORAWAN_APP_KEY

#ifndef LORAWAN_REGION
#define LORAWAN_REGION LORAWAN_REGION_DEFAULT
#endif  // LORAWAN_REGION

#ifndef LORAWAN_CLASS
#define LORAWAN_CLASS LORAWAN_CLASS_DEFAULT
#endif  // LORAWAN_CLASS

/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC TYPES ------------------------------------------------------------
 */

/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC FUNCTIONS PROTOTYPES ---------------------------------------------
 */

#ifdef __cplusplus
}
#endif

#endif  // LORAWAN_KEY_CONFIG_H

/* --- EOF ------------------------------------------------------------------ */
