use std::{
    collections::HashMap,
    sync::{Arc, Mutex},
};

use accord::packets::ServerboundPacket;
use config::Config;
use tokio::sync::mpsc;

use druid::{
    im::Vector,
    kurbo::Insets,
    widget::{Button, Checkbox, Flex, Label, List, Svg, SvgData, TextBox, ViewSwitcher},
    AppLauncher, Color, Data, Env, Event, FontDescriptor, FontFamily, ImageBuf, Lens, UnitPoint,
    Widget, WidgetExt, WindowDesc,
};

use serde::{Deserialize, Serialize};

use flexi_logger::Logger;

mod controllers;
use controllers::*;

mod connection_handler;
use connection_handler::*;

mod config;

mod widgets;
use widgets::*;

//TODO: Loading up past messages

#[derive(Serialize, Deserialize)]
pub struct Theme {
    pub background1: String,
    pub background2: String,
    pub text_color1: String,
    pub color1: String,
    pub highlight: String,
    pub border: f64,
}

impl Default for Theme {
    fn default() -> Self {
        Self {
            background1: "#200730".to_string(),
            background2: "#030009".to_string(),
            text_color1: "#6ef3e7".to_string(),
            color1: "#7521ee29".to_string(),
            highlight: "#77777777".to_string(),
            border: 4.5,
        }
    }
}

/// Represents a message on the server
#[derive(Debug, Data, Lens, Clone, PartialEq, Eq)]
pub struct Message {
    pub sender_id: i64,
    pub sender: String,
    pub date: String,
    pub content: String,
    pub is_image: bool,
}

/// Views in accord-gui application
#[derive(Debug, Data, Clone, Copy, PartialEq, Eq)]
enum Views {
    /// Starting view. Login prompt
    Connect,
    /// Main view, with messages etc.
    Main,
}

#[derive(Debug, Lens, Data, Clone)]
struct AppState {
    current_view: Views,
    info_label_text: Arc<String>,
    input_text1: Arc<String>,
    input_text2: Arc<String>,
    input_text3: Arc<String>,
    remember_login: bool,
    input_text4: Arc<String>,
    /// For sending commands to [`ConnectionHandler`]
    connection_handler_tx: Arc<mpsc::Sender<ConnectionHandlerCommand>>,
    /// List of connected users
    user_list: Vector<String>,
    /// Cached messages
    messages: Vector<Message>,
    images_from_links: bool,
}

fn init_logger() {
    Logger::try_with_env_or_str("warn")
        .unwrap()
        .start()
        .unwrap();
}

// This could be not static, but oh well
// TODO: Maybe this should be just set in Env?
static mut THEME: Option<Theme> = None;

pub const GUI_COMMAND: druid::Selector<GuiCommand> = druid::Selector::new("gui_command");

fn main() {
    init_logger();

    let config = config::load_config();

    // I solemnly swear this is the only place in which we mutate THEME
    unsafe {
        THEME = Some(config.theme.expect("Theme should be loaded from config!"));
    }

    let connection_handler = ConnectionHandler {};
    let (tx, rx) = mpsc::channel(16);

    // Cache of images
    let dled_images = Arc::new(Mutex::new(HashMap::new()));

    let main_window = WindowDesc::new(ui_builder(Arc::clone(&dled_images))).title("accord");

    let data = AppState {
        current_view: Views::Connect,
        info_label_text: Arc::new("".to_string()),
        input_text1: Arc::new(config.address.clone()),
        input_text2: Arc::new(config.username.clone()),
        input_text3: Arc::new("".to_string()),
        remember_login: config.remember_login,
        input_text4: Arc::new("".to_string()),
        connection_handler_tx: Arc::new(tx),
        user_list: Vector::new(),
        messages: Vector::new(),
        images_from_links: config.images_from_links,
    };

    let launcher = AppLauncher::with_window(main_window).delegate(Delegate {
        dled_images,
        rt: tokio::runtime::Runtime::new().unwrap(),
    });

    let event_sink = launcher.get_external_handle();

    std::thread::spawn(move || {
        connection_handler.main_loop(rx, event_sink);
    });

    launcher.launch(data).unwrap();
}

/// Connect to server using data from input textboxes
fn connect_click(data: &mut AppState) {
    let addr = try_parse_addr(&data.input_text1);
    if accord::utils::verify_username(&*data.input_text2) {
        data.info_label_text = Arc::new("Connecting...".to_string());
        data.connection_handler_tx
            .blocking_send(ConnectionHandlerCommand::Connect(
                addr,
                data.input_text2.to_string(),
                data.input_text3.to_string(),
            ))
            .unwrap();
        config::save_config(config_from_appstate(data)).unwrap();
    } else {
        log::warn!("Invalid username");
        data.info_label_text = Arc::new("Invalid username".to_string());
    };
}

/// Send message to server
fn send_message_click(data: &mut AppState) {
    let s = data.input_text4.clone();
    if accord::utils::verify_message(&*s) {
        let p = if let Some(command) = s.strip_prefix('/') {
            ServerboundPacket::Command(command.to_string())
        } else {
            ServerboundPacket::Message(s.to_string())
        };
        data.connection_handler_tx
            .blocking_send(ConnectionHandlerCommand::Write(p))
            .unwrap();
        data.input_text4 = Arc::new(String::new());
    } else {
        data.info_label_text = Arc::new("Invalid message".to_string());
    };
}

// Less typing
fn unwrap_from_hex(s: &str) -> Color {
    Color::from_hex_str(s).unwrap()
}

/// Builds UI of connect view
fn connect_view() -> impl Widget<AppState> {
    let font = FontDescriptor::new(FontFamily::SYSTEM_UI).with_size(20.0);
    let theme = unsafe {
        // We only read
        THEME.as_ref().unwrap()
    };

    let input_label_c = |s: &str| -> druid::widget::Align<AppState> {
        Label::new(s)
            .with_font(font.clone())
            .with_text_color(unwrap_from_hex(&theme.text_color1))
            .padding(7.0)
            .center()
    };
    let input_box_c = || -> TextBox<Arc<String>> {
        TextBox::new()
            .with_font(font.clone())
            .with_text_color(unwrap_from_hex(&theme.text_color1))
    };

    let info_label = Label::dynamic(|data, _env| format!("{}", data))
        .with_text_color(Color::YELLOW)
        .with_font(font.clone())
        .padding(5.0)
        .lens(AppState::info_label_text);
    let label1 = input_label_c("Address:");
    let label2 = input_label_c("Username:");
    let label3 = input_label_c("Password:");
    let button = Button::new("Connect")
        .on_click(|_, data, _| connect_click(data))
        .padding(5.0);
    let input1 = input_box_c().lens(AppState::input_text1).expand_width();
    let input2 = input_box_c().lens(AppState::input_text2).expand_width();
    let input3 = input_box_c()
        .lens(AppState::input_text3)
        .expand_width()
        .controller(TakeFocusConnect);
    let checkbox = Checkbox::new("Remember login").lens(AppState::remember_login);

    let checkbox2 = Checkbox::new("Images from links").lens(AppState::images_from_links);

    let accord_logo_data = match include_str!("resources/accord-logo.svg").parse::<SvgData>() {
        Ok(svg) => svg,
        Err(err) => {
            log::error!("{}", err);
            log::error!("Using an empty SVG instead.");
            SvgData::default()
        }
    };
    let accord_logo = Svg::new(accord_logo_data).fill_mode(druid::widget::FillStrat::ScaleDown);

    Flex::column()
        .with_child(
            accord_logo
                .fix_width(300.0)
                .align_vertical(UnitPoint::BOTTOM),
        )
        .with_child(info_label)
        .with_child(
            Flex::column()
                .with_child(
                    Flex::row()
                        .with_child(label1)
                        .with_flex_child(input1, 1.0)
                        .fix_width(250.0),
                )
                .with_child(
                    Flex::row()
                        .with_child(label2)
                        .with_flex_child(input2, 1.0)
                        .fix_width(250.0),
                )
                .with_child(
                    Flex::row()
                        .with_child(label3)
                        .with_flex_child(input3, 1.0)
                        .fix_width(250.0),
                )
                .with_child(checkbox)
                .with_child(button)
                .with_child(checkbox2)
                .padding(10.0)
                .fix_width(350.0)
                .padding((-30.0, 5.0, -20.0, 5.0))
                .cut_corners(0.0, 20.0, 20.0, 0.0)
                .with_border(unwrap_from_hex(&theme.highlight), theme.border)
                .with_background(unwrap_from_hex(&theme.color1)),
        )
        .align_vertical(UnitPoint::new(0.0, 0.25))
}

/// Builds a [`Widget`] showing a message
fn message(dled_images: Arc<Mutex<HashMap<String, ImageBuf>>>) -> impl Widget<Message> {
    let theme = unsafe {
        // We only read
        THEME.as_ref().unwrap()
    };

    let font = FontDescriptor::new(FontFamily::SYSTEM_UI).with_size(17.0);
    let content_label = Label::dynamic(|d: &String, _e: &_| d.clone())
        .with_font(font.clone())
        .with_text_color(unwrap_from_hex(&theme.text_color1))
        .with_line_break_mode(druid::widget::LineBreaking::WordWrap)
        .lens(Message::content);
    let image_from_link = ImageMessage::new(content_label, dled_images);
    Flex::row()
        .cross_axis_alignment(druid::widget::CrossAxisAlignment::Start)
        .with_child(
            Label::dynamic(|data: &Message, _env| {
                if data.sender.is_empty() {
                    "".to_string()
                } else {
                    format!("{} {}:", data.sender, data.date)
                }
            })
            .with_text_color(unwrap_from_hex(&theme.text_color1))
            .with_font(font.with_weight(druid::FontWeight::BOLD)),
        )
        .with_default_spacer()
        .with_flex_child(Flex::column().with_child(image_from_link), 1.0)
        .padding(Insets::uniform_xy(5.0, 5.0))
        .cut_corners_sym(10.0)
        .with_background(unwrap_from_hex(&theme.color1))
        .with_border(unwrap_from_hex(&theme.highlight), theme.border)
        .padding(Insets::uniform_xy(0.0, 1.0))
}

/// Parses address from string.
/// If string contains `':'`, it assumes it's "ADDRESS:PORT",
/// else it assumes it's just the address.
fn try_parse_addr(s: &str) -> String {
    if s.contains(':') {
        s.to_owned()
    } else {
        format!("{}:{}", s, accord::DEFAULT_PORT)
    }
}

/// Builds UI of main view
fn main_view(dled_images: Arc<Mutex<HashMap<String, ImageBuf>>>) -> impl Widget<AppState> {
    let theme = unsafe {
        // We only read
        THEME.as_ref().unwrap()
    };
    let user_list_font = FontDescriptor::new(FontFamily::SYSTEM_UI)
        .with_size(15.0)
        .with_weight(druid::FontWeight::BOLD);

    let info_label = Label::dynamic(|data, _env| format!("{}", data))
        .with_text_color(Color::YELLOW)
        .lens(AppState::info_label_text);

    let accord_logo_data = match include_str!("resources/accord-logo.svg").parse::<SvgData>() {
        Ok(svg) => svg,
        Err(err) => {
            log::error!("{}", err);
            log::error!("Using an empty SVG instead.");
            SvgData::default()
        }
    };
    let accord_logo = Svg::new(accord_logo_data).fill_mode(druid::widget::FillStrat::ScaleDown);

    let user_list_widget = Flex::column()
        .cross_axis_alignment(druid::widget::CrossAxisAlignment::Start)
        .with_flex_child(
            List::new(move || Label::raw().with_font(user_list_font.clone()))
                .lens(AppState::user_list),
            1.0,
        )
        .with_child(Label::new("").fix_width(100.0))
        .expand_height()
        .padding((10.0, 5.0, 5.0, 5.0))
        .cut_corners(10.0, 0.0, 0.0, 10.0)
        .with_border(unwrap_from_hex(&theme.highlight), theme.border)
        .with_background(unwrap_from_hex(&theme.color1))
        .padding((0.0, 0.0, 5.0, 0.0));

    let messages_list_widget = List::new(move || message(Arc::clone(&dled_images)))
        .controller(ListController)
        .scroll()
        .vertical()
        .controller(ScrollController::new())
        .expand_height()
        .lens(AppState::messages);

    let input_text_box = TextBox::multiline()
        .lens(AppState::input_text4)
        .expand_width()
        .controller(TakeFocusMain)
        .controller(MessageTextBoxController);

    let send_button =
        Button::new("Send").on_click(|_ctx, data: &mut AppState, _env| send_message_click(data));

    Flex::column()
        .cross_axis_alignment(druid::widget::CrossAxisAlignment::Start)
        .with_child(accord_logo.fix_height(80.0).center())
        .with_child(info_label)
        .with_flex_child(
            Flex::row()
                .cross_axis_alignment(druid::widget::CrossAxisAlignment::Start)
                .with_child(user_list_widget)
                .with_flex_child(messages_list_widget, 1.0),
            1.0,
        )
        .with_default_spacer()
        .with_child(
            Flex::row()
                .with_flex_child(input_text_box, 1.0)
                .with_default_spacer()
                .with_child(send_button),
        )
        .padding(20.0)
}

/// Builds root widget
fn ui_builder(dled_images: Arc<Mutex<HashMap<String, ImageBuf>>>) -> impl Widget<AppState> {
    let theme = unsafe {
        // We only read
        THEME.as_ref().unwrap()
    };
    Flex::column()
        .with_flex_child(
            ViewSwitcher::new(
                |data: &AppState, _env| data.current_view,
                move |selector, _data, _env| match *selector {
                    Views::Connect => Box::new(connect_view()),
                    Views::Main => Box::new(main_view(Arc::clone(&dled_images))),
                },
            ),
            1.0,
        )
        .background(druid::LinearGradient::new(
            UnitPoint::BOTTOM,
            UnitPoint::TOP,
            (
                unwrap_from_hex(&theme.background2),
                unwrap_from_hex(&theme.background1),
            ),
        ))
}

/// Main delegate for this app
struct Delegate {
    dled_images: Arc<Mutex<HashMap<String, ImageBuf>>>,
    rt: tokio::runtime::Runtime,
}

/// Construct [`Config`] from [`AppState`]
fn config_from_appstate(data: &AppState) -> Config {
    let (address, username) = if data.remember_login {
        (data.input_text1.to_string(), data.input_text2.to_string())
    } else {
        Default::default()
    };
    Config {
        address,
        username,
        remember_login: data.remember_login,
        images_from_links: data.images_from_links,
        theme: None,
    }
}

impl druid::AppDelegate<AppState> for Delegate {
    fn event(
        &mut self,
        ctx: &mut druid::DelegateCtx,
        _window_id: druid::WindowId,
        event: Event,
        data: &mut AppState,
        _env: &Env,
    ) -> Option<Event> {
        use druid::keyboard_types::Key;
        match event {
            Event::KeyDown(ref kevent) => match kevent.key {
                Key::Enter => {
                    match data.current_view {
                        Views::Connect => connect_click(data),
                        Views::Main => send_message_click(data),
                    }
                    None
                }
                Key::PageUp => {
                    ctx.submit_command(controllers::SCROLL.with(-1.0));
                    None
                }
                Key::PageDown => {
                    ctx.submit_command(controllers::SCROLL.with(1.0));
                    None
                }
                _ => Some(event),
            },
            _ => Some(event),
        }
    }

    fn command(
        &mut self,
        ctx: &mut druid::DelegateCtx,
        _target: druid::Target,
        cmd: &druid::Command,
        data: &mut AppState,
        _env: &Env,
    ) -> druid::Handled {
        if let Some(command) = cmd.get(GUI_COMMAND) {
            match command {
                GuiCommand::AddMessage(m) => {
                    data.messages.push_back(m.clone());

                    // Try to get image from message link
                    if data.images_from_links {
                        let dled_images = Arc::clone(&self.dled_images);
                        let link = m.content.clone();
                        let event_sink = ctx.get_external_handle();
                        self.rt.spawn(async move {
                            try_get_image_from_link(&link, dled_images, event_sink).await;
                        });
                    }
                }
                GuiCommand::Connected => {
                    data.info_label_text = Arc::new(String::new());
                    data.current_view = Views::Main;
                }
                GuiCommand::ConnectionEnded(m) => {
                    data.messages = Vector::new();
                    data.info_label_text = Arc::new(m.to_string());
                    data.current_view = Views::Connect;
                }
                GuiCommand::SendImage(image_bytes) => {
                    let v = image_bytes.to_vec();
                    let p = ServerboundPacket::ImageMessage(v);
                    data.connection_handler_tx
                        .blocking_send(ConnectionHandlerCommand::Write(p))
                        .unwrap();
                }
                GuiCommand::StoreImage(hash, img_bytes) => {
                    let img_buf = ImageBuf::from_data(img_bytes).unwrap();

                    let mut dled_images = self.dled_images.lock().unwrap();
                    dled_images.insert(hash.to_string(), img_buf);
                    ctx.submit_command(
                        druid::Selector::<String>::new("image_downloaded").with(hash.to_string()),
                    );
                }
                GuiCommand::UpdateUserList(user_list) => data.user_list = user_list.into(),
            };
        };
        druid::Handled::No
    }
}

/// Tries to download and image from the link and stores it in `dled_images` cache.
///
/// Returns `true` on success.
async fn try_get_image_from_link(
    link: &str,
    dled_images: Arc<Mutex<HashMap<String, ImageBuf>>>,
    event_sink: druid::ExtEventSink,
) -> bool {
    if !dled_images.lock().unwrap().contains_key(link) {
        let client = reqwest::ClientBuilder::new()
            .timeout(std::time::Duration::from_secs(10))
            .build()
            .unwrap();

        // We get just head first to see if it's an image
        let req = client.head(link).build();
        let resp = match req {
            Ok(req) => client.execute(req).await,
            Err(_) => return false,
        };
        match resp {
            Ok(resp) => {
                if resp.status() == reqwest::StatusCode::OK
                    && resp.headers().get("content-type").map_or(false, |v| {
                        v.to_str().map_or(false, |s| s.starts_with("image/"))
                    })
                    && resp.headers().get("content-length").map_or(false, |v| {
                        v.to_str().map_or(false, |s| {
                            s.parse::<u32>().map_or(false, |l| {
                                l < 31457280 // 30 MB
                            })
                        })
                    })
                {
                    let req = client.get(link).build().unwrap();

                    let resp = match client.execute(req).await {
                        Ok(resp) => resp,
                        Err(_) => return false,
                    };

                    let img_bytes = resp.bytes().await.unwrap();
                    let img_buf = ImageBuf::from_data(&img_bytes).unwrap();

                    let mut dled_images = dled_images.lock().unwrap();
                    dled_images.insert(link.to_string(), img_buf);
                    event_sink
                        .submit_command(
                            druid::Selector::<String>::new("image_downloaded"),
                            link.to_string(),
                            druid::Target::Auto,
                        )
                        .unwrap();
                }
            }
            Err(e) => {
                log::warn!("Error when getting image: {}", e);
                return false;
            }
        };
    };

    true
}
