import sys
import json
from pathlib import Path

current_file = Path(__file__).parent.parent.absolute()
dataset_path = Path(__file__).parent.parent.parent / "design_patterns"
sys.path.append(str(current_file))

from modules.descriptor_validator import DescriptorValidator
from modules.solidity_scanner import SolidityScanner
from modules.config import settings

scanner: SolidityScanner


def test_prepare_descriptors():
    global scanner
    desc_validator = DescriptorValidator(f"{current_file}/descriptors/")
    if not desc_validator.load_schema(schema_path=f"{current_file}/modules/data/descriptor_schema.json"):
        raise Exception('Error loading schema')
    settings.descriptors = desc_validator.load_descriptors()
    if not settings.descriptors:
        raise Exception('Error loading descriptors')
    scanner = SolidityScanner()
    assert True

def test_accessrestriction_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Authorization/accessrestriction_pattern.sol")
    excepted_result: str = '{"AccessRestriction": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 13, "match_statement": "onlyafter"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "24", "match_statement": "require(_condition)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_ownership_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Authorization/ownership_pattern.sol")
    excepted_result: str = '{"Ownable": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 25, "match_statement": "onlyowner"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "26", "match_statement": "require(_owner == msg.sender,\'ownable: caller is not the owner\')"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 25, "match_statement": "onlyowner"}, "comparison": {"result": true, "line_match": "26", "match_statement": "_owner == msg.sender"}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_commitreveal_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Behavioral/commitreveal_pattern.sol")
    excepted_result: str = '{"CommitReveal": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": true, "line_match": "19", "match_statement": "logcommit(_commit,msg.sender)"}, "fn_definition": {"result": true, "line_match": 13, "match_statement": "commit"}, "event_emit": {"result": true, "line_match": 10, "match_statement": "logcommit"}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "43", "match_statement": "require(bytes(usercommit.status)[0] == \'r\')"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_guardcheck_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Behavioral/guardcheck_pattern.sol")
    excepted_result: str = '{"GuardCheck": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": true, "line_match": 14, "match_statement": "Check Block"}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "7", "match_statement": "require(addr != address(0))"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": true, "line_match": "19", "match_statement": "payable(addr).transfer(transferamount)"}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_oracle_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Behavioral/oracle_pattern.sol")
    excepted_result: str = '{"Oracle": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 13, "match_statement": "onlyby"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "14", "match_statement": "require(msg.sender == account)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": true, "line_match": 17, "match_statement": "query"}, "var_definition": {"result": false}, "event_emit": {"result": true, "line_match": 12, "match_statement": "newrequest"}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}, "OracleConsumer": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 32, "match_statement": "onlyby"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "33", "match_statement": "require(msg.sender == account)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": true, "line_match": "36", "match_statement": "oracle.query(\'usd\',this.oracleresponse)"}, "fn_definition": {"result": false}, "var_definition": {"result": true, "line_match": 31, "match_statement": "oracle_addr"}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_pulloverpush_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Behavioral/pulloverpush_pattern.sol")
    excepted_result: str = '{"Auction": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": true, "line_match": 22, "match_statement": "Check Block"}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "11", "match_statement": "require(msg.value >= highestbid)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": true, "line_match": "23", "match_statement": "payable(msg.sender).transfer(refund)"}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_randomness_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Behavioral/randomness_pattern.sol")
    excepted_result: str = '{"SimpleRandom": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": true, "line_match": "7", "match_statement": "uint(blockhash(block.number - 1))"}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_statemachine_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Behavioral/statemachine_pattern.sol")
    excepted_result: str = '{"DepositLock": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": true, "line_match": 33, "match_statement": "Check Block"}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "16", "match_statement": "require(stage == _stage)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": true, "line_match": "34", "match_statement": "payable(msg.sender).transfer(amount)"}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": true, "line_match": 15, "match_statement": "atstage"}, "enum_definition": {"result": true, "line_match": 6, "match_statement": "stages"}, "fn_definition": {"result": true, "line_match": 24, "match_statement": "nextstage"}, "fn_call": {"result": true, "line_match": "20", "match_statement": "nextstage()"}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_memoryarraybuilding_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Economic/memoryarraybuilding_pattern.sol")
    excepted_result: str = '{"MemoryArrayBuilding": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": true, "line_match": 18, "match_statement": "getItemIDsByOwner"}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_stringequalitycmp_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Economic/stringequalitycmp_pattern.sol")
    excepted_result: str = '{"SimpleContractExample": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": true, "line_match": "7", "match_statement": "keccak256(abi.encodepacked(_a)) == keccak256(abi.encodepacked(_b))"}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_tightvariablepacking_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Economic/tightvariablepacking_pattern.sol")
    excepted_result: str = '{"StructPackingExample": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": true, "line_match": 6, "match_statement": "CheapStruct"}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_autodeprecation_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Lifecycle/autodeprecation_pattern.sol")
    excepted_result: str = '{"Deprecatable": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 18, "match_statement": "whendeprecated"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 18, "match_statement": "whendeprecated"}, "comparison": {"result": true, "line_match": "11", "match_statement": "block.timestamp > expires"}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_mortal_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Lifecycle/mortal_pattern.sol")
    excepted_result: str = '{"Mortal": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 9, "match_statement": "onlyowner"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": true, "line_match": "11", "match_statement": "selfdestruct(payable(owner()))"}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": true, "line_match": 7, "match_statement": "ownable"}, "modifier": {"result": true, "line_match": 9, "match_statement": "onlyowner"}, "comparison": {"result": true, "line_match": "15", "match_statement": "msg.sender == owner()"}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_datasegregation_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Maintenance/datasegregation_pattern.sol")
    excepted_result: str = '{"Storage": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": true, "line_match": 6, "match_statement": "uintstorage"}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}, "Logic": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}, "Proxy": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_register_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Maintenance/register_pattern.sol")
    excepted_result: str = '{"Register": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 11, "match_statement": "onlyowner"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "26", "match_statement": "require(success)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": true, "line_match": 7, "match_statement": "ownable"}, "modifier": {"result": true, "line_match": 11, "match_statement": "onlyowner"}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": true, "line_match": "25", "match_statement": "currentversion.delegatecall(msg.data)"}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_relay_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Maintenance/relay_pattern.sol")
    excepted_result: str = '{"Relay": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 14, "match_statement": "onlyowner"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "22", "match_statement": "require(success)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": true, "line_match": 7, "match_statement": "ownable"}, "modifier": {"result": true, "line_match": 14, "match_statement": "onlyowner"}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": true, "line_match": "21", "match_statement": "currentversion.delegatecall(msg.data)"}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_satellite_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Maintenance/satellite_pattern.sol")
    excepted_result: str = '{"sumSatellite": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}, "Core": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 17, "match_statement": "onlyowner"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": true, "line_match": 15, "match_statement": "ownable"}, "modifier": {"result": true, "line_match": 17, "match_statement": "onlyowner"}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_balancelimit_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Security/balancelimit_pattern.sol")
    excepted_result: str = '{"BalanceLimit": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": true, "line_match": "12", "match_statement": "address(this).balance <= limit"}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "12", "match_statement": "require(address(this).balance <= limit)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_checkeffectsinteraction():
    scanner.parse_solidity_file(f"{dataset_path}/Security/checkeffectsinteraction.sol")
    excepted_result: str = '{"ChecksEffectsInteractions": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": true, "line_match": "13", "match_statement": "balances[msg.sender] >= amount"}}, "Check Effects Interaction": {"check_effects_interaction": {"result": true, "line_match": 15, "match_statement": "Check Block"}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "13", "match_statement": "require(balances[msg.sender] >= amount)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": true, "line_match": "17", "match_statement": "payable(msg.sender).transfer(amount)"}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_emergencystop_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Security/emergencystop_pattern.sol")
    excepted_result: str = '{"EmergencyStop": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 15, "match_statement": "enableinemergency"}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": true, "line_match": 21, "match_statement": "contractstopped = !contractstopped"}, "fn_definition": {"result": true, "line_match": 20, "match_statement": "togglecontractstopped"}, "fn_call": {"result": false}, "modifier": {"result": true, "line_match": 15, "match_statement": "enableinemergency"}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": true, "line_match": 7, "match_statement": "ownable"}, "modifier": {"result": true, "line_match": 20, "match_statement": "onlyowner"}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_mutex_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Security/mutex_pattern.sol")
    excepted_result: str = '{"Mutex": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "10", "match_statement": "require(!locked)"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": true, "line_match": 9, "match_statement": "noreentrancy"}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_ratelimit_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Security/ratelimit_pattern.sol")
    excepted_result: str = '{"RateLimiter": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": false}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": true, "line_match": 21, "match_statement": "ratelimited"}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_rejector_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Security/rejector_pattern.sol")
    excepted_result: str = '{"Rejector": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": false}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "6", "match_statement": "revert()"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": true, "line_match": "6", "match_statement": "revert()"}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

def test_speedbump_pattern():
    scanner.parse_solidity_file(f"{dataset_path}/Security/speedbump_pattern.sol")
    excepted_result: str = '{"SpeedBumps": {"Access Restriction": {"inheritance": {"result": false}, "modifier": {"result": false}}, "Auto Deprecation": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Balance Limit": {"comparison": {"result": false}}, "Check Effects Interaction": {"check_effects_interaction": {"result": true, "line_match": 31, "match_statement": "Check Block"}}, "Circuit Breaker": {"inheritance": {"result": false}, "state_toggle": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}, "modifier": {"result": false}}, "Commit & Reveal": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "event_emit": {"result": false}}, "Eternal Storage": {"eternal_storage": {"result": false}}, "Guard Check": {"fn_call": {"result": true, "line_match": "32", "match_statement": "require(payable(msg.sender).send(amounttowithdraw))"}}, "Memory Array Building": {"memory_array_building": {"result": false}}, "Mortal": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Mutex": {"inheritance": {"result": false}, "modifier": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "Oracle": {"fn_call": {"result": false}, "fn_definition": {"result": false}, "var_definition": {"result": false}, "event_emit": {"result": false}}, "Ownership": {"inheritance": {"result": false}, "modifier": {"result": false}, "comparison": {"result": false}}, "Pull Over Push": {"inheritance": {"result": false}, "fn_call": {"result": false}}, "Randomness": {"fn_call": {"result": false}}, "Rate Limit": {"modifier": {"result": false}, "inheritance": {"result": false}}, "Rejector": {"rejector": {"result": false}}, "Relay": {"inheritance": {"result": false}, "relay": {"result": false}}, "State Machine": {"modifier": {"result": false}, "enum_definition": {"result": false}, "fn_definition": {"result": false}, "fn_call": {"result": false}}, "String Equality": {"comparison": {"result": false}}, "Tight Variable Packing": {"tight_variable_packing": {"result": false}}}}'
    result = scanner.get_design_pattern_statistics()
    assert json.dumps(result) == excepted_result

