#ifdef _WIN32
#include <windows.h>
#else
#include <cerrno>
#include <sys/ioctl.h>
#include <unistd.h>
#include "utf8proc.h"
#include "../tty.hpp"
#endif

#include "../exception.hpp"
#include "../platforms/platform.hpp"
#include "map"

std::pair<std::size_t, std::size_t> Term::Private::get_term_size() {
#ifdef _WIN32
    if (GetStdHandle(STD_OUTPUT_HANDLE) == INVALID_HANDLE_VALUE) {
        throw Term::Exception("GetStdHandle(STD_OUTPUT_HANDLE) failed");
    }
    CONSOLE_SCREEN_BUFFER_INFO inf;
    if (GetConsoleScreenBufferInfo(GetStdHandle(STD_OUTPUT_HANDLE), &inf)) {
        std::size_t cols = inf.srWindow.Right - inf.srWindow.Left + 1;
        std::size_t rows = inf.srWindow.Bottom - inf.srWindow.Top + 1;
        return {rows, cols};
    } else {
        // This happens when we are not connected to a terminal
        throw Term::Exception("Couldn't get terminal size. Is it connected to a TTY?");
    }
#else
    struct winsize ws{};
    if (ioctl(STDOUT_FILENO, TIOCGWINSZ, &ws) == -1 || ws.ws_col == 0) {
        // This happens when we are not connected to a terminal
        throw Term::Exception("Couldn't get terminal size. Is it connected to a TTY?");
    } else {
        return std::pair<std::size_t, std::size_t>{ws.ws_row, ws.ws_col};
    }
#endif
}

short Term::Private::c32_display_width(const char32_t& c32) {
#ifdef _WIN32
    static const std::map<char32_t, short> codepoint_width = {
            {0x6, 1},
            {0x7, 0},
            {0x8, -1},
            {0x9, 4},
            {0xa, -1},
            {0xc, 1},
            {0xd, -1},
            {0xa3, 1},
            {0xa4, 2},
            {0xa6, 1},
            {0xa7, 2},
            {0xad, 1},
            {0xae, 2},
            {0xb0, 1},
            {0xb1, 2},
            {0xb5, 1},
            {0xb6, 2},
            {0xbb, 1},
            {0xbf, 2},
            {0xc5, 1},
            {0xc6, 2},
            {0xcf, 1},
            {0xd0, 2},
            {0xd6, 1},
            {0xd8, 2},
            {0xdd, 1},
            {0xe1, 2},
            {0xe5, 1},
            {0xe6, 2},
            {0xe7, 1},
            {0xea, 2},
            {0xef, 1},
            {0xf0, 2},
            {0xf1, 1},
            {0xf3, 2},
            {0xf6, 1},
            {0xfa, 2},
            {0xfb, 1},
            {0xfc, 2},
            {0xfd, 1},
            {0xfe, 2},
            {0x100, 1},
            {0x101, 2},
            {0x110, 1},
            {0x111, 2},
            {0x112, 1},
            {0x113, 2},
            {0x11a, 1},
            {0x11b, 2},
            {0x125, 1},
            {0x127, 2},
            {0x131, 1},
            {0x132, 2},
            {0x137, 1},
            {0x138, 2},
            {0x13e, 1},
            {0x13f, 2},
            {0x140, 1},
            {0x141, 2},
            {0x143, 1},
            {0x144, 2},
            {0x147, 1},
            {0x14b, 2},
            {0x14c, 1},
            {0x14d, 2},
            {0x151, 1},
            {0x153, 2},
            {0x165, 1},
            {0x166, 2},
            {0x16a, 1},
            {0x16b, 2},
            {0x1cd, 1},
            {0x1ce, 2},
            {0x1d1, 1},
            {0x1d2, 2},
            {0x1d3, 1},
            {0x1d4, 2},
            {0x1d5, 1},
            {0x1d6, 2},
            {0x1d7, 1},
            {0x1d8, 2},
            {0x1d9, 1},
            {0x1da, 2},
            {0x1db, 1},
            {0x1dc, 2},
            {0x250, 1},
            {0x251, 2},
            {0x260, 1},
            {0x261, 2},
            {0x2c3, 1},
            {0x2c4, 2},
            {0x308, 1},
            {0x309, 2},
            {0x30c, 1},
            {0x30e, 2},
            {0x30f, 1},
            {0x326, 2},
            {0x328, 1},
            {0x331, 2},
            {0x332, 1},
            {0x36f, 2},
            {0x390, 1},
            {0x398, 2},
            {0x399, 1},
            {0x3a1, 2},
            {0x3a2, 1},
            {0x3a9, 2},
            {0x3b0, 1},
            {0x3b5, 2},
            {0x3b6, 1},
            {0x3b8, 2},
            {0x3b9, 1},
            {0x3bd, 2},
            {0x3be, 1},
            {0x3c1, 2},
            {0x3c2, 1},
            {0x3c9, 2},
            {0x400, 1},
            {0x401, 2},
            {0x40f, 1},
            {0x432, 2},
            {0x433, 1},
            {0x441, 2},
            {0x442, 1},
            {0x44f, 2},
            {0x450, 1},
            {0x451, 2},
            {0x10ff, 1},
            {0x115f, 2},
            {0x2013, 1},
            {0x2015, 2},
            {0x201f, 1},
            {0x2021, 2},
            {0x2024, 1},
            {0x2026, 2},
            {0x202f, 1},
            {0x2030, 2},
            {0x203a, 1},
            {0x203b, 2},
            {0x203d, 1},
            {0x203e, 2},
            {0x20ab, 1},
            {0x20ac, 2},
            {0x2102, 1},
            {0x2103, 2},
            {0x2104, 1},
            {0x2105, 2},
            {0x2108, 1},
            {0x2109, 2},
            {0x2115, 1},
            {0x2116, 2},
            {0x2120, 1},
            {0x2122, 2},
            {0x2125, 1},
            {0x2126, 2},
            {0x212a, 1},
            {0x212b, 2},
            {0x2152, 1},
            {0x2154, 2},
            {0x215a, 1},
            {0x215e, 2},
            {0x2160, 1},
            {0x216b, 2},
            {0x2171, 1},
            {0x2179, 2},
            {0x2188, 1},
            {0x2189, 2},
            {0x218f, 1},
            {0x2199, 2},
            {0x21a8, 1},
            {0x21aa, 2},
            {0x21b7, 1},
            {0x21b9, 2},
            {0x21d1, 1},
            {0x21d2, 2},
            {0x21d3, 1},
            {0x21d4, 2},
            {0x21e6, 1},
            {0x21e7, 2},
            {0x21ff, 1},
            {0x2200, 2},
            {0x2201, 1},
            {0x2203, 2},
            {0x2206, 1},
            {0x2208, 2},
            {0x220a, 1},
            {0x220b, 2},
            {0x220e, 1},
            {0x220f, 2},
            {0x2210, 1},
            {0x2211, 2},
            {0x2219, 1},
            {0x221a, 2},
            {0x221c, 1},
            {0x2220, 2},
            {0x2222, 1},
            {0x2223, 2},
            {0x2224, 1},
            {0x2225, 2},
            {0x2226, 1},
            {0x222a, 2},
            {0x222b, 1},
            {0x222c, 2},
            {0x222d, 1},
            {0x222e, 2},
            {0x2233, 1},
            {0x2237, 2},
            {0x223b, 1},
            {0x223d, 2},
            {0x2247, 1},
            {0x2248, 2},
            {0x224b, 1},
            {0x224c, 2},
            {0x2251, 1},
            {0x2252, 2},
            {0x225f, 1},
            {0x2261, 2},
            {0x2263, 1},
            {0x2267, 2},
            {0x2269, 1},
            {0x226b, 2},
            {0x226d, 1},
            {0x226f, 2},
            {0x2281, 1},
            {0x2283, 2},
            {0x2285, 1},
            {0x2287, 2},
            {0x2294, 1},
            {0x2295, 2},
            {0x2298, 1},
            {0x2299, 2},
            {0x22a4, 1},
            {0x22a5, 2},
            {0x22be, 1},
            {0x22bf, 2},
            {0x2311, 1},
            {0x2312, 2},
            {0x2319, 1},
            {0x231b, 2},
            {0x2327, 1},
            {0x232a, 2},
            {0x23ce, 1},
            {0x23cf, 2},
            {0x23e8, 1},
            {0x23f3, 2},
            {0x23f7, 1},
            {0x23fa, 2},
            {0x245f, 1},
            {0x24e9, 2},
            {0x24ea, 1},
            {0x24ff, 2},
            {0x259f, 1},
            {0x25a1, 2},
            {0x25a2, 1},
            {0x25a9, 2},
            {0x25b1, 1},
            {0x25b3, 2},
            {0x25b5, 1},
            {0x25b7, 2},
            {0x25bb, 1},
            {0x25bd, 2},
            {0x25bf, 1},
            {0x25c1, 2},
            {0x25c5, 1},
            {0x25c8, 2},
            {0x25ca, 1},
            {0x25cb, 2},
            {0x25cd, 1},
            {0x25d1, 2},
            {0x25e1, 1},
            {0x25e5, 2},
            {0x25ee, 1},
            {0x25ef, 2},
            {0x25fa, 1},
            {0x25fe, 2},
            {0x25ff, 1},
            {0x2606, 2},
            {0x2608, 1},
            {0x2609, 2},
            {0x260d, 1},
            {0x260f, 2},
            {0x2610, 1},
            {0x2611, 2},
            {0x2613, 1},
            {0x2615, 2},
            {0x2617, 1},
            {0x2618, 2},
            {0x261b, 1},
            {0x261e, 2},
            {0x261f, 1},
            {0x2620, 2},
            {0x2621, 1},
            {0x2623, 2},
            {0x2625, 1},
            {0x2626, 2},
            {0x2629, 1},
            {0x262a, 2},
            {0x262d, 1},
            {0x262f, 2},
            {0x2637, 1},
            {0x263a, 2},
            {0x263f, 1},
            {0x2640, 2},
            {0x2641, 1},
            {0x2642, 2},
            {0x2647, 1},
            {0x2653, 2},
            {0x265e, 1},
            {0x2661, 2},
            {0x2662, 1},
            {0x266a, 2},
            {0x266b, 1},
            {0x266d, 2},
            {0x266e, 1},
            {0x266f, 2},
            {0x267a, 1},
            {0x267b, 2},
            {0x267d, 1},
            {0x267f, 2},
            {0x2691, 1},
            {0x2697, 2},
            {0x2698, 1},
            {0x2699, 2},
            {0x269a, 1},
            {0x269c, 2},
            {0x269d, 1},
            {0x26a1, 2},
            {0x26a9, 1},
            {0x26ab, 2},
            {0x26af, 1},
            {0x26b1, 2},
            {0x26bc, 1},
            {0x26bf, 2},
            {0x26c3, 1},
            {0x26e1, 2},
            {0x26e2, 1},
            {0x26e3, 2},
            {0x26e7, 1},
            {0x26ff, 2},
            {0x2701, 1},
            {0x2702, 2},
            {0x2704, 1},
            {0x2705, 2},
            {0x2707, 1},
            {0x270d, 2},
            {0x270e, 1},
            {0x270f, 2},
            {0x2711, 1},
            {0x2712, 2},
            {0x2713, 1},
            {0x2714, 2},
            {0x2715, 1},
            {0x2716, 2},
            {0x271c, 1},
            {0x271d, 2},
            {0x2720, 1},
            {0x2721, 2},
            {0x2727, 1},
            {0x2728, 2},
            {0x2732, 1},
            {0x2734, 2},
            {0x273c, 1},
            {0x273d, 2},
            {0x2743, 1},
            {0x2744, 2},
            {0x2746, 1},
            {0x2747, 2},
            {0x274b, 1},
            {0x274c, 2},
            {0x274d, 1},
            {0x274e, 2},
            {0x2752, 1},
            {0x2755, 2},
            {0x2756, 1},
            {0x2757, 2},
            {0x2762, 1},
            {0x2764, 2},
            {0x2775, 1},
            {0x277f, 2},
            {0x2794, 1},
            {0x2797, 2},
            {0x27a0, 1},
            {0x27a1, 2},
            {0x27af, 1},
            {0x27b0, 2},
            {0x27be, 1},
            {0x27bf, 2},
            {0x2933, 1},
            {0x2935, 2},
            {0x2b04, 1},
            {0x2b07, 2},
            {0x2b1a, 1},
            {0x2b1c, 2},
            {0x2b4f, 1},
            {0x2b50, 2},
            {0x2b54, 1},
            {0x2b59, 2},
            {0x2e7f, 1},
            {0x303e, 2},
            {0x3040, 1},
            {0x3096, 2},
            {0x3098, 1},
            {0x30ff, 2},
            {0x3104, 1},
            {0x312e, 2},
            {0x3130, 1},
            {0x318e, 2},
            {0x318f, 1},
            {0x4dbf, 2},
            {0x4dff, 1},
            {0xa4c6, 2},
            {0xa95f, 1},
            {0xa97c, 2},
            {0xabff, 1},
            {0xd7a3, 2},
            {0xdfff, 1},
            {0xe7c6, 2},
            {0xe7c8, 1},
            {0xfaff, 2},
            {0xfdff, 1},
            {0xfe1f, 2},
            {0xfe2f, 1},
            {0xfe6b, 2},
            {0xff00, 1},
            {0xff60, 2},
            {0xffdf, 1},
            {0xffe6, 2},
            {0xfffc, 1},
            {0xfffd, 2},
            {0xffff, 1}
    };
    short char_width = 1;
    auto it = codepoint_width.lower_bound(c32);
    if (it != codepoint_width.end()) {
        char_width = it->second;
    }
    return char_width;
#else
    return utf8proc_charwidth(static_cast<wchar_t>(c32));
#endif
}
