#ifdef _WIN32
#include <windows.h>
#else
#include <cerrno>
#include <sys/ioctl.h>
#include <unistd.h>
#endif

#include "../exception.hpp"
#include "../input.hpp"
#include "../tty.hpp"

char Term::Platform::read_raw_stdin() {
    char c = static_cast<char>(getchar());
    if (c >= 0) {
        return c;
    } else if (c == EOF) {
        //In non-raw (blocking) mode this happens when the input file
        //ends. In such a case, return the End of Transmission (EOT)
        //character (Ctrl-D)
        return 0x04;
    } else {
        throw Exception("getchar() failed.");
    }
}

bool Term::Platform::read_raw(char* s) {
    //Do nothing when TTY is not connected.
    if (!is_stdin_a_tty()) {
        return false;
    }
#ifdef _WIN32
    DWORD n_read = 0;
    GetNumberOfConsoleInputEvents(GetStdHandle(STD_INPUT_HANDLE), &n_read);
    if (n_read >= 1) {
        INPUT_RECORD buf;
        if (!ReadConsoleInputW(GetStdHandle(STD_INPUT_HANDLE), &buf, 1, &n_read)) {
            throw Exception("ReadConsoleInput() failed.");
        }
        if (n_read == 1) {
            switch (buf.EventType) {
                case KEY_EVENT: {
                    WORD skip = buf.Event.KeyEvent.wVirtualKeyCode; //Skip them for now.
                    if (skip == VK_SHIFT || skip == VK_LWIN || skip == VK_RWIN || skip == VK_APPS
                    || skip == VK_CONTROL || skip == VK_MENU || skip == VK_CAPITAL) {
                        return false;
                    }
                    if (buf.Event.KeyEvent.bKeyDown) {
                        *s = buf.Event.KeyEvent.uChar.AsciiChar;
                        return true;
                    } else {
                        return false;
                    }
                }
                case FOCUS_EVENT:
                case MENU_EVENT:
                case MOUSE_EVENT:
                case WINDOW_BUFFER_SIZE_EVENT:
                default:
                    return false;
            }
        } else {
            throw Exception("kbhit() and ReadConsoleInput() inconsistent.");
        }
    } else {
        return false;
    }
#else
    ::ssize_t nread = ::read(0, s, 1);
    if (nread == -1 && errno != EAGAIN) {
        throw Term::Exception("read() failed");
    }
    return nread == 1;
#endif
}
