/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package org.key_project.slicing.ui;

import java.util.Collection;
import java.util.Optional;
import javax.swing.*;

/**
 * Utility class to generate HTML tables for UI purposes.
 *
 * @author Arne Keller
 */
public final class HtmlFactory {
    private HtmlFactory() {
    }

    /**
     * Generate an HTML table using the given column labels and row data.
     * <p>
     * If clickable[i] is true, the cells in the i-th column will be links pointing to
     * '#i_idx', where i is the column index and idx is generated by a call to the provided
     * IndexFactory.
     * </p>
     * <p>
     * If the alignment array is present, it must contain an entry for each column (null, "right"
     * are useful values). The cells in that column will be aligned accordingly (default/null:
     * left alignment).
     * </p>
     *
     * @param columnNames column labels
     * @param clickable whether the cells in a column should be clickable links
     * @param alignment text alignment of each column
     * @param rows row data to display in the table
     * @param indexFactory index factory
     * @return HTML string
     */
    public static String generateTable(
            Collection<String> columnNames,
            boolean[] clickable,
            Optional<String[]> alignment,
            Collection<Collection<String>> rows,
            IndexFactory indexFactory) {
        if (columnNames.size() != clickable.length) {
            throw new IllegalArgumentException();
        }

        StringBuilder stats = new StringBuilder("<table><thead>");
        for (String a : columnNames) {
            stats.append("<td>").append(a).append("</td>");
        }
        stats.append("</thead><tbody>");

        for (Collection<String> row : rows) {
            stats.append("<tr>");
            int i = 0;
            for (String cell : row) {
                stats.append("<td");
                if (alignment.isPresent()) {
                    String align = alignment.get()[i];
                    if (align != null) {
                        stats.append(" style='text-align:")
                                .append(align)
                                .append("'");
                    }
                }
                stats.append(">");
                if (clickable[i]) {
                    stats.append("<a href='#");
                    stats.append(i);
                    stats.append("_");
                    stats.append(indexFactory.nextIndex());
                    stats.append("'>");
                }
                if (cell.startsWith("<")) {
                    // assume cell content is HTML
                    stats.append(cell);
                } else {
                    stats.append(escapeText(cell));
                }
                if (clickable[i]) {
                    stats.append("</a>");
                }
                stats.append("</td>");
                i++;
            }
            stats.append("</tr>");
        }

        stats.append("</tbody></table>");
        return stats.toString();
    }

    /**
     * Create a Swing component (JEditorPane) showing the given HTML document.
     *
     * @param html HTML document
     * @return Swing component showing that HTML
     */
    public static JComponent createComponent(String html) {
        JEditorPane htmlContent = new JEditorPane("text/html", html);
        htmlContent.setEditable(false);
        htmlContent.setBorder(BorderFactory.createEmptyBorder());
        htmlContent.setCaretPosition(0);
        return htmlContent;
    }

    private static String escapeText(String text) {
        return text
                .replace("&", "&amp;")
                .replace("<", "&lt;")
                .replace(">", "&gt;");
    }
}
