/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package de.uka.ilkd.key.util.removegenerics;

import recoder.CrossReferenceServiceConfiguration;
import recoder.abstraction.Type;
import recoder.java.declaration.TypeArgumentDeclaration;
import recoder.java.reference.TypeReference;
import recoder.kit.ProblemReport;
import recoder.kit.TypeKit;
import recoder.list.generic.ASTList;
import recoder.service.CrossReferenceSourceInfo;

/**
 * Handle a type reference in the generic deletion process.
 *
 * If the type reference references a type avar or an array over tv, it must be replaced. if the
 * type var is ("extends"-) bounded than <b>with the first boundary(!)</b> otherwise with
 * java.lang.Object.
 *
 * @author MU
 *
 */

public class ResolveTypeReference extends GenericResolutionTransformation {

    private final TypeReference reference;

    private TypeReference replaceWith;

    private final CrossReferenceSourceInfo sourceInfo;

    public ResolveTypeReference(TypeReference reference, CrossReferenceServiceConfiguration sc) {
        super(sc);
        this.reference = reference;
        sourceInfo = sc.getCrossReferenceSourceInfo();
    }

    @Override
    public ProblemReport analyze() {
        ASTList<TypeArgumentDeclaration> typeArguments = reference.getTypeArguments();
        if (typeArguments != null && !typeArguments.isEmpty()) {
            return EQUIVALENCE;
        }

        Type type = sourceInfo.getType(reference);

        Type replaceType = targetType(type);

        if (replaceType != null && !replaceType.equals(type)) {
            replaceWith = TypeKit.createTypeReference(getProgramFactory(), replaceType);
            return EQUIVALENCE;
        }

        return IDENTITY;
    }

    @Override
    public void transform() {
        if (replaceWith != null) {
            replace(reference, replaceWith);
        } else {
            reference.setTypeArguments(null);
        }
    }

}
