/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package de.uka.ilkd.key.smt.newsmt2;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import de.uka.ilkd.key.control.DefaultUserInterfaceControl;
import de.uka.ilkd.key.control.KeYEnvironment;
import de.uka.ilkd.key.logic.Term;
import de.uka.ilkd.key.nparser.KeyIO;
import de.uka.ilkd.key.proof.Proof;
import de.uka.ilkd.key.proof.io.ProofSaver;

import org.key_project.util.Streams;

import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Tag;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static de.uka.ilkd.key.logic.equality.RenamingTermProperty.RENAMING_TERM_PROPERTY;

/**
 * This test case makes sure that all KeY formulas which are translated to axioms in SMT can
 * actually be proved in KeY.
 *
 * @author Mattias Ulbrich
 */
@Tag("slow")
public class ProveSMTLemmasTest {
    private static final Logger LOGGER = LoggerFactory.getLogger(ProveSMTLemmasTest.class);
    private static String HEADER;

    @BeforeAll
    public static void setUpPreamble() throws IOException {
        HEADER =
            Streams.toString(ProveSMTLemmasTest.class.getResourceAsStream("smt-lemma-header.key"));
    }

    @AfterAll
    public static void tearDownClass() {
        HEADER = null;
    }

    @ParameterizedTest
    @MethodSource("data")
    public void testSMTLemmaSoundness(String name, String lemmaString) throws Exception {
        URL proofFile = getClass().getResource("SMT_lemma_" + name + ".proof");
        Path path;
        if (proofFile != null) {
            assert proofFile.getProtocol().equals("file");
            path = Paths.get(proofFile.toURI());
        } else {
            path = Files.createTempFile("SMT_lemma_" + name + "_", ".key");
            Files.writeString(path, HEADER + "\\problem { " + lemmaString + "}");
        }

        File file = path.toFile();

        LOGGER.info("Now processing file {}", file);

        KeYEnvironment<DefaultUserInterfaceControl> env = KeYEnvironment.load(file);
        try {
            Proof loadedProof = env.getLoadedProof();
            env.getProofControl().startAndWaitForAutoMode(loadedProof);
            if (!loadedProof.closed()) {
                File saveFile = new File(file.getAbsoluteFile() + ".proof");
                ProofSaver saver = new ProofSaver(loadedProof, saveFile);
                saver.save();
                Assertions.fail("Proof does not close. See " + file + " and " + saveFile);
            } else {
                if (proofFile == null) {
                    // delete temp files
                    file.delete();
                } else {
                    // and check if proofs are actually for the right theorem!
                    KeyIO io = new KeyIO(loadedProof.getServices());
                    Term parsedLemma = io.parseExpression(lemmaString);
                    Term actual = loadedProof.root().sequent().succedent().get(0).formula();
                    if (!actual.equalsModProperty(parsedLemma, RENAMING_TERM_PROPERTY)) {
                        LOGGER.info("Stored : {}", parsedLemma);
                        LOGGER.warn("Proven : {}", actual);
                        Assertions.fail("The proven lemma is different from the stored one.");
                    }
                }

            }
        } finally {
            env.dispose();
        }
    }

    public static List<String[]> data() throws IOException {
        URL url = DefinedSymbolsHandler.class.getResource("DefinedSymbolsHandler.preamble.xml");
        if (url == null) {
            throw new FileNotFoundException(
                "Cannot find resource file which should have been generated by gradle");
        }

        Properties props = new Properties();
        try (InputStream in = url.openStream()) {
            props.loadFromXML(in);
        }

        List<String[]> result = new ArrayList<>();

        for (String name : props.stringPropertyNames()) {
            if (name.matches(".*\\.dl(\\.[0-9]+)?")) {
                String[] params = { name, props.getProperty(name) };
                result.add(params);
            }
        }

        return result;
    }

}
