/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package de.uka.ilkd.key.strategy.termgenerator;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import de.uka.ilkd.key.java.Services;
import de.uka.ilkd.key.ldt.IntegerLDT;
import de.uka.ilkd.key.logic.PosInOccurrence;
import de.uka.ilkd.key.logic.Semisequent;
import de.uka.ilkd.key.logic.Sequent;
import de.uka.ilkd.key.logic.SequentFormula;
import de.uka.ilkd.key.logic.Term;
import de.uka.ilkd.key.logic.TermServices;
import de.uka.ilkd.key.logic.label.TermLabelState;
import de.uka.ilkd.key.logic.op.Equality;
import de.uka.ilkd.key.logic.op.JFunction;
import de.uka.ilkd.key.logic.op.OperatorSV;
import de.uka.ilkd.key.logic.sort.GenericSort;
import de.uka.ilkd.key.proof.Goal;
import de.uka.ilkd.key.rule.RuleApp;
import de.uka.ilkd.key.rule.SyntacticalReplaceVisitor;
import de.uka.ilkd.key.rule.Taclet;
import de.uka.ilkd.key.rule.TacletApp;
import de.uka.ilkd.key.rule.inst.SVInstantiations;
import de.uka.ilkd.key.strategy.feature.MutableState;
import de.uka.ilkd.key.strategy.quantifierHeuristics.Constraint;
import de.uka.ilkd.key.strategy.quantifierHeuristics.EqualityConstraint;
import de.uka.ilkd.key.strategy.quantifierHeuristics.Metavariable;
import de.uka.ilkd.key.strategy.quantifierHeuristics.PredictCostProver;
import de.uka.ilkd.key.strategy.quantifierHeuristics.Substitution;

import org.key_project.logic.Name;
import org.key_project.logic.sort.Sort;
import org.key_project.util.collection.DefaultImmutableMap;
import org.key_project.util.collection.DefaultImmutableSet;
import org.key_project.util.collection.ImmutableList;
import org.key_project.util.collection.ImmutableSLList;
import org.key_project.util.collection.ImmutableSet;

public class TriggeredInstantiations implements TermGenerator {

    public static TermGenerator create(boolean skipConditions) {
        return new TriggeredInstantiations(skipConditions);
    }

    private Sequent last = Sequent.EMPTY_SEQUENT;
    private Set<Term> lastCandidates = new HashSet<>();
    private ImmutableSet<Term> lastAxioms = DefaultImmutableSet.nil();

    private final boolean checkConditions;

    /**
     *
     * @param checkConditions boolean indicating if conditions should be checked
     */
    public TriggeredInstantiations(boolean checkConditions) {
        this.checkConditions = checkConditions;
    }

    /**
     * Generates all instances
     */
    @Override
    public Iterator<Term> generate(RuleApp app, PosInOccurrence pos, Goal goal,
            MutableState mState) {
        if (app instanceof TacletApp tapp) {

            final Services services = goal.proof().getServices();
            final Taclet taclet = tapp.taclet();

            final Set<Term> terms;
            final Set<Term> axiomSet;
            ImmutableSet<Term> axioms = DefaultImmutableSet.nil();


            final Sequent seq = goal.sequent();
            if (seq != last) {
                terms = new HashSet<>();
                axiomSet = new HashSet<>();
                computeAxiomAndCandidateSets(seq, terms, axiomSet, services);
                for (Term axiom : axiomSet) {
                    axioms = axioms.add(axiom);
                }

                synchronized (this) {
                    last = seq;
                    lastCandidates = terms;
                    lastAxioms = axioms;
                }
            } else {
                synchronized (this) {
                    terms = lastCandidates;
                    axioms = lastAxioms;
                }
            }

            if (taclet.hasTrigger()) {

                final Term comprehension = pos.subTerm();

                if (tapp.uninstantiatedVars().size() <= 1) {
                    SVInstantiations svInst = tapp.instantiations();

                    final OperatorSV sv = taclet.getTrigger().triggerVar();
                    final Sort svSort;
                    if (sv.sort() instanceof GenericSort) {
                        svSort = svInst.getGenericSortInstantiations().getRealSort(sv, services);
                    } else {
                        svSort = sv.sort();
                    }

                    final Metavariable mv = new Metavariable(new Name("$MV$" + sv.name()), svSort);

                    final Term trigger = instantiateTerm(taclet.getTrigger().getTerm(), services,
                        svInst.replace(sv, services.getTermFactory().createTerm(mv), services));

                    final Set<Term> instances =
                        computeInstances(services, comprehension, mv, trigger, terms, axioms, tapp);

                    return instances.iterator();
                } else {
                    // at the moment instantiations with more than one
                    // missing taclet variable not supported
                    return ImmutableSLList.<Term>nil().iterator();
                }
            } else {
                return ImmutableSLList.<Term>nil().iterator();
            }

        } else {
            throw new IllegalArgumentException("At the moment only taclets are supported.");
        }

    }

    private Term instantiateTerm(final Term term, final Services services,
            SVInstantiations svInst) {
        final SyntacticalReplaceVisitor syn = new SyntacticalReplaceVisitor(new TermLabelState(),
            null, null, svInst, null, null, null, services);
        term.execPostOrder(syn);
        return syn.getTerm();
    }

    private void computeAxiomAndCandidateSets(final Sequent seq, final Set<Term> terms,
            final Set<Term> axioms, Services services) {
        final IntegerLDT integerLDT = services.getTypeConverter().getIntegerLDT();
        collectAxiomsAndCandidateTerms(terms, axioms, integerLDT, seq.antecedent(), true, services);
        collectAxiomsAndCandidateTerms(terms, axioms, integerLDT, seq.succedent(), false, services);
    }

    private void collectAxiomsAndCandidateTerms(final Set<Term> terms, final Set<Term> axioms,
            final IntegerLDT integerLDT, Semisequent antecedent, boolean inAntecedent,
            TermServices services) {

        for (SequentFormula sf : antecedent) {
            collectTerms(sf.formula(), terms, integerLDT);
            if (sf.formula().op() instanceof JFunction
                    || sf.formula().op() == Equality.EQUALS) {
                axioms.add(
                    inAntecedent ? sf.formula() : services.getTermBuilder().not(sf.formula()));
            }
        }
    }

    private boolean isAvoidConditionProvable(Term cond, ImmutableSet<Term> axioms,
            Services services) {

        long cost = PredictCostProver.computerInstanceCost(
            new Substitution(DefaultImmutableMap.nilMap()), cond,
            axioms, services);
        return cost == -1;
    }

    private HashSet<Term> computeInstances(Services services, final Term comprehension,
            final Metavariable mv, final Term trigger, Set<Term> terms, ImmutableSet<Term> axioms,
            TacletApp app) {

        final HashSet<Term> instances = new HashSet<>();
        final HashSet<Term> alreadyChecked = new HashSet<>();

        for (final Term t : terms) {
            boolean addToInstances = true;
            Constraint c = EqualityConstraint.BOTTOM.unify(trigger, t, services);
            if (c.isSatisfiable()) {
                final Term middle = c.getInstantiation(mv, services);
                if (middle != null && !alreadyChecked.contains(middle)) {
                    alreadyChecked.add(middle);
                    if (!checkConditions && app.taclet().getTrigger().hasAvoidConditions()) {
                        ImmutableList<Term> conditions =
                            instantiateConditions(services, app, middle);
                        for (Term condition : conditions) {
                            if (isAvoidConditionProvable(condition, axioms, services)) {
                                addToInstances = false;
                                break;
                            }
                        }
                    }
                    if (addToInstances) {
                        instances.add(middle);
                    }
                }
            }
        }
        return instances;
    }

    private ImmutableList<Term> instantiateConditions(Services services, TacletApp app,
            final Term middle) {
        ImmutableList<Term> conditions;
        conditions = ImmutableSLList.nil();
        for (Term singleAvoidCond : app.taclet().getTrigger().avoidConditions()) {
            conditions =
                conditions.append(instantiateTerm(singleAvoidCond, services, app.instantiations()
                        .replace(app.taclet().getTrigger().triggerVar(), middle, services)));
        }
        return conditions;
    }

    private void collectTerms(Term instanceCandidate, Set<Term> terms, IntegerLDT intLDT) {
        if (instanceCandidate.freeVars().isEmpty()
                && !instanceCandidate.containsJavaBlockRecursive()) {
            terms.add(instanceCandidate);
        }
        if (intLDT.getNumberSymbol() != instanceCandidate.op()) {
            for (int i = 0; i < instanceCandidate.arity(); i++) {
                collectTerms(instanceCandidate.sub(i), terms, intLDT);
            }
        }
    }

}
