/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package de.uka.ilkd.key.speclang;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.UnaryOperator;

import de.uka.ilkd.key.informationflow.po.InfFlowContractPO;
import de.uka.ilkd.key.java.Services;
import de.uka.ilkd.key.java.abstraction.KeYJavaType;
import de.uka.ilkd.key.java.declaration.modifier.VisibilityModifier;
import de.uka.ilkd.key.logic.Term;
import de.uka.ilkd.key.logic.op.IObserverFunction;
import de.uka.ilkd.key.logic.op.IProgramMethod;
import de.uka.ilkd.key.logic.op.LocationVariable;
import de.uka.ilkd.key.logic.op.Modality;
import de.uka.ilkd.key.pp.LogicPrinter;
import de.uka.ilkd.key.proof.init.ContractPO;
import de.uka.ilkd.key.proof.init.InitConfig;
import de.uka.ilkd.key.proof.init.ProofOblInput;
import de.uka.ilkd.key.util.InfFlowSpec;

import org.key_project.util.collection.ImmutableList;



/**
 * Standard implementation of the DependencyContract interface.
 */
public final class InformationFlowContractImpl implements InformationFlowContract {

    private final int id;
    private final KeYJavaType forClass;
    private final IProgramMethod pm;
    private final KeYJavaType specifiedIn;
    private final String baseName;
    private final String name;
    private final Term origPre;
    /** The original free precondition. */
    private final Term origFreePre;
    private final Term origMby;
    private final Term origModifiable;
    private final Modality.JavaModalityKind modality;
    private final Term origSelf;
    private final ImmutableList<Term> origParams;
    private final Term origResult;
    private final Term origExc;
    private final Term origAtPre;
    private final boolean toBeSaved;
    private final Term origDep;
    private final ImmutableList<InfFlowSpec> origInfFlowSpecs;

    /**
     * If a method is strictly pure, it has no modifiable clause which could anonymised.
     *
     * @see #hasModifiableClause()
     */
    final boolean hasRealModifiableClause;


    // -------------------------------------------------------------------------
    // constructors
    // -------------------------------------------------------------------------

    private InformationFlowContractImpl(
            String baseName, String name, KeYJavaType forClass,
            IProgramMethod pm, KeYJavaType specifiedIn, Modality.JavaModalityKind modalityKind,
            Term pre, Term freePre,
            Term mby, Term modifiable, boolean hasRealModifiable, Term self,
            ImmutableList<Term> params,
            Term result, Term exc, Term heapAtPre, Term dep,
            ImmutableList<InfFlowSpec> infFlowSpecs, boolean toBeSaved, int id) {
        assert baseName != null;
        assert forClass != null;
        assert pm != null;
        assert modalityKind != null;
        assert pre != null;
        assert freePre != null;
        assert modifiable != null;
        assert (self == null) == pm.isStatic();
        assert params != null;
        // assert params.size() == pm.getParameterDeclarationCount();
        if (result == null) {
            // can happen for if-contracts generated by if-loop-invariants.
            // to be fixed?
            // assert (pm.isVoid() || pm.isConstructor()) : "resultVar == null for method "+pm;
        } else {
            assert (!pm.isVoid() && !pm.isConstructor())
                    : "non-null result variable for void method or constructor " + pm
                        + " with return type " + pm.getReturnType();
        }
        assert exc != null;
        // assert dep != null;
        assert infFlowSpecs != null;

        this.baseName = baseName;
        this.name = name != null ? name
                : ContractFactory.generateContractName(baseName, forClass, pm, specifiedIn, id);
        this.forClass = forClass;
        this.pm = pm;
        this.specifiedIn = specifiedIn;
        this.origPre = pre;
        this.origFreePre = freePre;
        this.origMby = mby;
        this.origModifiable = modifiable;
        this.origSelf = self;
        this.origParams = params;
        this.origResult = result;
        this.origExc = exc;
        this.origAtPre = heapAtPre;
        this.id = id;
        this.modality = modalityKind;
        this.hasRealModifiableClause = hasRealModifiable;
        this.toBeSaved = toBeSaved;
        this.origDep = dep;
        this.origInfFlowSpecs = infFlowSpecs;
    }


    public InformationFlowContractImpl(String baseName, KeYJavaType forClass, IProgramMethod pm,
            KeYJavaType specifiedIn, Modality.JavaModalityKind modalityKind, Term pre, Term freePre,
            Term mby, Term modifiable,
            boolean hasRealModifiable, Term self, ImmutableList<Term> params, Term result, Term exc,
            Term heapAtPre, Term dep, ImmutableList<InfFlowSpec> infFlowSpecs, boolean toBeSaved) {
        this(baseName, null, forClass, pm, specifiedIn, modalityKind, pre, freePre, mby, modifiable,
            hasRealModifiable, self, params, result, exc, heapAtPre, dep, infFlowSpecs, toBeSaved,
            INVALID_ID);
    }


    // -------------------------------------------------------------------------
    // public interface
    // -------------------------------------------------------------------------

    @Override
    public InformationFlowContract map(UnaryOperator<Term> op, Services services) {
        return new InformationFlowContractImpl(baseName, name, forClass, pm, specifiedIn, modality,
            op.apply(origPre), op.apply(origFreePre), op.apply(origMby), op.apply(origModifiable),
            hasRealModifiableClause, origSelf,
            origParams.stream().map(op).collect(ImmutableList.collector()), op.apply(origResult),
            op.apply(origExc), op.apply(origAtPre), op.apply(origDep),
            origInfFlowSpecs.stream().map(spec -> spec.map(op)).collect(ImmutableList.collector()),
            toBeSaved, id);
    }

    @Override
    public String getName() {
        return name;
    }


    @Override
    public int id() {
        return id;
    }


    @Override
    public KeYJavaType getKJT() {
        return forClass;
    }


    @Override
    public IProgramMethod getTarget() {
        return pm;
    }


    @Override
    public boolean hasMby() {
        return origMby != null;
    }


    @Override
    public String getBaseName() {
        return baseName;
    }


    @Override
    public Term getPre() {
        return origPre;
    }


    @Override
    public Term getFreePre() {
        return origFreePre;
    }


    @Override
    public Term getModifiable() {
        return origModifiable;
    }


    @Override
    public Term getMby() {
        return origMby;
    }


    @Override
    public Modality.JavaModalityKind getModalityKind() {
        return modality;
    }


    @Override
    public Term getSelf() {
        if (origSelf == null) {
            assert pm.isStatic() : "missing self variable in non-static method contract";
            return null;
        }
        return origSelf;
    }


    @Override
    public ImmutableList<Term> getParams() {
        return origParams;
    }


    @Override
    public Term getResult() {
        return origResult;
    }


    @Override
    public Term getExc() {
        return origExc;
    }


    @Override
    public Term getAtPre() {
        return origAtPre;
    }


    @Override
    public boolean isReadOnlyContract() {
        return origModifiable.toString().equals("empty");
    }


    @Override
    public String getTypeName() {
        return ContractFactory.generateContractTypeName(baseName, forClass, pm, specifiedIn);
    }


    @Override
    public boolean hasModifiableClause() {
        return hasRealModifiableClause;
    }


    @Override
    public boolean hasInfFlowSpec() {
        // TODO (KIT): This is always true (except null)
        return !(origInfFlowSpecs == InfFlowSpec.EMPTY_INF_FLOW_SPEC);
    }


    @Override
    public String getHTMLText(Services services) {
        return "<html>" + getHTMLBody(services) + "</html>";
    }


    public String getHTMLBody(Services services) {
        return "<html>" + getHTMLSignature() + getHTMLFor(origPre, "pre", services)
            + getHTMLFor(origFreePre, "free_pre", services)
            + getHTMLFor(origModifiable, "modifiable", services)
            + (hasRealModifiableClause ? "" : "<b>, creates no new objects</b>")
            + getHTMLFor(origMby, "measured-by", services) + "<br><b>termination</b> " + modality
            + getHTMLFor(origInfFlowSpecs, "determines", services) + "</html>";
    }


    private String getHTMLSignature() {
        return "<i>" + LogicPrinter.escapeHTML(getHTMLSignatureBody().toString(), false) + "</i>";
    }


    private StringBuffer getHTMLSignatureBody() {
        final StringBuffer sig = new StringBuffer();
        if (origResult != null) {
            sig.append(origResult);
            sig.append(" = ");
        } else if (pm.isConstructor()) {
            sig.append(origSelf);
            sig.append(" = new ");
        }
        if (!pm.isStatic() && !pm.isConstructor()) {
            sig.append(origSelf);
            sig.append(".");
        }
        sig.append(pm.getName());
        sig.append("(");
        for (Term pv : origParams) {
            sig.append(pv.toString()).append(", ");
        }
        if (!origParams.isEmpty()) {
            sig.setLength(sig.length() - 2);
        }
        sig.append(")");
        sig.append(" catch(");
        sig.append(origExc);
        sig.append(")");
        return sig;
    }


    private String getHTMLFor(Term originalTerm, String htmlName, Services services) {
        if (originalTerm == null) {
            return "";
        } else {
            final String quickPrint = LogicPrinter.quickPrintTerm(originalTerm, services);
            return "<br>" + "<b>" + htmlName + "</b> " + LogicPrinter.escapeHTML(quickPrint, false);
        }
    }


    private String getHTMLFor(Iterable<Term> originalTerms, Services services) {
        StringBuilder result;
        final Iterator<Term> it = originalTerms.iterator();
        if (!it.hasNext()) {
            result = new StringBuilder(
                LogicPrinter.quickPrintTerm(services.getTermBuilder().empty(), services));
        } else {
            result = new StringBuilder();
        }
        while (it.hasNext()) {
            Term term = it.next();
            final String quickPrint = LogicPrinter.quickPrintTerm(term, services);
            result.append(LogicPrinter.escapeHTML(quickPrint, false));
            if (it.hasNext()) {
                result.append(", ");
            }
        }
        return result.toString();
    }


    private String getHTMLFor(ImmutableList<InfFlowSpec> originalInfFlowSpecs, String htmlName,
            Services services) {
        StringBuilder infFlowSpecString = new StringBuilder();
        if (hasInfFlowSpec()) {
            infFlowSpecString = new StringBuilder("<br><b>" + htmlName + "</b> ");
            Iterator<InfFlowSpec> it = originalInfFlowSpecs.iterator();
            while (it.hasNext()) {
                final InfFlowSpec infFlowSpec = it.next();
                infFlowSpecString.append(getHTMLFor(infFlowSpec.postExpressions, services));
                infFlowSpecString.append(" <b>by</b> ")
                        .append(getHTMLFor(infFlowSpec.preExpressions, services));
                if (!infFlowSpec.newObjects.isEmpty()) {
                    infFlowSpecString.append(", <b>new objects</b> ")
                            .append(getHTMLFor(infFlowSpec.newObjects, services));
                }
                if (it.hasNext()) {
                    infFlowSpecString.append("<br>" + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;"
                        + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + "<b>and</b> ");
                }
            }
        }
        return infFlowSpecString.toString();
    }


    @Override
    public String toString() {
        // TODO: all fields should be printed!!
        return name + ":: kjt: " + forClass + "; pm: " + pm + "; modality: " + modality + "; pre: "
            + origPre + "; origFreePre: " + origFreePre + "; mby: " + origMby + "; modifiable: "
            + origModifiable
            + "; selfVar: " + origSelf + "; paramVars: " + origParams + "; id:" + id;
    }


    @Override
    public ContractPO createProofObl(InitConfig initConfig) {
        return (ContractPO) createProofObl(initConfig, this);
    }


    @Override
    public ProofOblInput getProofObl(Services services) {
        return services.getSpecificationRepository().getPO(this);
    }

    @Override
    public ProofOblInput createProofObl(InitConfig initConfig, Contract contract) {
        return new InfFlowContractPO(initConfig, (InformationFlowContract) contract);
    }

    @Override
    public ProofOblInput createProofObl(InitConfig initConfig, Contract contract,
            boolean addSymbolicExecutionLabel) {
        if (addSymbolicExecutionLabel) {
            throw new IllegalStateException("Symbolic Execution API is not supported.");
        } else {
            return createProofObl(initConfig, contract);
        }
    }

    @Override
    public String getDisplayName() {
        return ContractFactory.generateDisplayName(baseName, forClass, pm, specifiedIn, id);
    }

    @Override
    public String getPODisplayName() {
        return "Method Contract";
    }


    @Override
    public VisibilityModifier getVisibility() {
        assert false; // this is currently not applicable for contracts
        return null;
    }


    @Override
    public boolean transactionApplicableContract() {
        return false;
    }


    @Override
    public KeYJavaType getSpecifiedIn() {
        return specifiedIn;
    }


    @Override
    public InformationFlowContract setID(int newId) {
        return new InformationFlowContractImpl(baseName, null, forClass, pm, specifiedIn, modality,
            origPre, origFreePre, origMby, origModifiable, hasRealModifiableClause, origSelf,
            origParams,
            origResult, origExc, origAtPre, origDep, origInfFlowSpecs, toBeSaved, newId);
    }


    @Override
    public InformationFlowContract setTarget(KeYJavaType newKJT, IObserverFunction newPM) {
        assert newPM instanceof IProgramMethod;
        return new InformationFlowContractImpl(baseName, null, newKJT, (IProgramMethod) newPM,
            specifiedIn, modality, origPre, origFreePre, origMby, origModifiable,
            hasRealModifiableClause,
            origSelf, origParams, origResult, origExc, origAtPre, origDep, origInfFlowSpecs,
            toBeSaved, id);
    }


    @Override
    public InformationFlowContract setName(String name) {
        return new InformationFlowContractImpl(baseName, name, forClass, pm, specifiedIn, modality,
            origPre, origFreePre, origMby, origModifiable, hasRealModifiableClause, origSelf,
            origParams,
            origResult, origExc, origAtPre, origDep, origInfFlowSpecs, toBeSaved, id);
    }


    @Override
    public InformationFlowContract setModality(Modality.JavaModalityKind modalityKind) {
        return new InformationFlowContractImpl(baseName, name, forClass, pm, specifiedIn,
            modalityKind,
            origPre, origFreePre, origMby, origModifiable, hasRealModifiableClause, origSelf,
            origParams,
            origResult, origExc, origAtPre, origDep, origInfFlowSpecs, toBeSaved, id);
    }


    @Override
    public InformationFlowContract setModifiable(Term modifiable) {
        return new InformationFlowContractImpl(baseName, name, forClass, pm, specifiedIn, modality,
            origPre, origFreePre, origMby, modifiable, hasRealModifiableClause, origSelf,
            origParams,
            origResult, origExc, origAtPre, origDep, origInfFlowSpecs, toBeSaved, id);
    }


    @Override
    public Term getDep() {
        return origDep;
    }


    @Override
    public ImmutableList<InfFlowSpec> getInfFlowSpecs() {
        return origInfFlowSpecs;
    }


    @Override
    public boolean equals(Contract c) {
        if (!(c instanceof InformationFlowContract ifc)) {
            return false;
        }
        assert name != null;
        assert forClass != null;
        assert pm != null;
        assert modality != null;
        assert origPre != null;
        assert origFreePre != null;
        assert origModifiable != null;
        assert origParams != null;
        assert origDep != null;
        assert origInfFlowSpecs != null;
        return name.equals(ifc.getName()) && forClass.equals(ifc.getKJT())
                && pm.equals(ifc.getTarget()) && modality.equals(ifc.getModalityKind())
                && origPre.equals(ifc.getPre()) && origFreePre.equals(ifc.getFreePre())
                && (origMby != null ? origMby.equals(ifc.getMby()) : ifc.getMby() == null)
                && origModifiable.equals(ifc.getModifiable())
                && (origSelf != null ? origSelf.equals(ifc.getSelf()) : ifc.getSelf() == null)
                && origParams.equals(ifc.getParams())
                && (origResult != null ? origResult.equals(ifc.getResult())
                        : ifc.getResult() == null)
                && origExc.equals(ifc.getExc()) && origAtPre.equals(ifc.getAtPre())
                && id == ifc.id() && toBeSaved == ifc.toBeSaved() && origDep.equals(ifc.getDep())
                && origInfFlowSpecs.equals(ifc.getInfFlowSpecs());
    }


    @Override
    public boolean toBeSaved() {
        return false; // because information flow contracts currently cannot
                      // be specified directly in DL
    }


    @Override
    public String proofToString(Services services) {
        throw new UnsupportedOperationException("Operation not supported.");
    }


    @Override
    public String getPlainText(Services services) {
        return getHTMLText(services); // TODO: return real plaintext...
    }


    @Override
    public Term getGlobalDefs(LocationVariable heap, Term heapTerm, Term selfTerm,
            ImmutableList<Term> paramTerms, Services services) {
        // information flow contracts do not have global defs (yet?)
        throw new UnsupportedOperationException("Not supported yet.");
    }


    @Override
    public boolean hasSelfVar() {
        return origSelf != null;
    }



    // the following code is legacy code

    @Override
    @Deprecated
    public Term getPre(LocationVariable heap, LocationVariable selfVar,
            ImmutableList<LocationVariable> paramVars,
            Map<LocationVariable, LocationVariable> atPreVars, Services services) {
        throw new UnsupportedOperationException(
            "Not supported any more. " + "Please use the POSnippetFactory instead.");
    }

    @Override
    @Deprecated
    public Term getPre(List<LocationVariable> heapContext, LocationVariable selfVar,
            ImmutableList<LocationVariable> paramVars,
            Map<LocationVariable, LocationVariable> atPreVars, Services services) {
        throw new UnsupportedOperationException(
            "Not supported any more. " + "Please use the POSnippetFactory instead.");

    }

    @Override
    @Deprecated
    public Term getPre(LocationVariable heap, Term heapTerm, Term selfTerm,
            ImmutableList<Term> paramTerms, Map<LocationVariable, Term> atPres, Services services) {
        throw new UnsupportedOperationException(
            "Not supported any more. " + "Please use the POSnippetFactory instead.");

    }

    @Override
    @Deprecated
    public Term getPre(List<LocationVariable> heapContext, Map<LocationVariable, Term> heapTerms,
            Term selfTerm, ImmutableList<Term> paramTerms, Map<LocationVariable, Term> atPres,
            Services services) {
        throw new UnsupportedOperationException(
            "Not supported any more. " + "Please use the POSnippetFactory instead.");

    }

    @Override
    @Deprecated
    public Term getMby(LocationVariable selfVar, ImmutableList<LocationVariable> paramVars,
            Services services) {
        throw new UnsupportedOperationException(
            "Not supported any more. " + "Please use the POSnippetFactory instead.");
    }


    @Override
    @Deprecated
    public Term getMby(Map<LocationVariable, Term> heapTerms, Term selfTerm,
            ImmutableList<Term> paramTerms, Map<LocationVariable, Term> atPres, Services services) {
        throw new UnsupportedOperationException(
            "Not supported any more. " + "Please use the POSnippetFactory instead.");
    }


    @Override
    public OriginalVariables getOrigVars() {
        // TODO Auto-generated method stub
        return null;
    }


    @Override
    public Term getDep(LocationVariable heap, boolean atPre, LocationVariable selfVar,
            ImmutableList<LocationVariable> paramVars,
            Map<LocationVariable, LocationVariable> atPreVars, Services services) {
        return null;
    }


    @Override
    public Term getDep(LocationVariable heap, boolean atPre, Term heapTerm, Term selfTerm,
            ImmutableList<Term> paramTerms, Map<LocationVariable, Term> atPres, Services services) {
        return null;
    }


    @Override
    public Term getRequires(LocationVariable heap) {
        return null;
    }


    @Override
    public Term getModifiable(LocationVariable heap) {
        return null;
    }


    @Override
    public Term getAccessible(LocationVariable heap) {
        return null;
    }


    @Override
    public Term getGlobalDefs() {
        // information flow contracts do not have global defs (yet?)
        throw new UnsupportedOperationException("Not supported yet.");
    }
}
