/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package de.uka.ilkd.key.proof.io;

import java.io.StringReader;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import de.uka.ilkd.key.axiom_abstraction.AbstractDomainElement;
import de.uka.ilkd.key.axiom_abstraction.predicateabstraction.AbstractPredicateAbstractionLattice;
import de.uka.ilkd.key.axiom_abstraction.predicateabstraction.AbstractionPredicate;
import de.uka.ilkd.key.axiom_abstraction.predicateabstraction.SimplePredicateAbstractionLattice;
import de.uka.ilkd.key.java.ProgramElement;
import de.uka.ilkd.key.java.Services;
import de.uka.ilkd.key.logic.*;
import de.uka.ilkd.key.logic.op.*;
import de.uka.ilkd.key.logic.op.QuantifiableVariable;
import de.uka.ilkd.key.parser.DefaultTermParser;
import de.uka.ilkd.key.parser.ParserException;
import de.uka.ilkd.key.pp.AbbrevMap;
import de.uka.ilkd.key.proof.Goal;
import de.uka.ilkd.key.proof.Node;
import de.uka.ilkd.key.proof.Proof;
import de.uka.ilkd.key.proof.init.ProblemInitializer;
import de.uka.ilkd.key.proof.io.intermediate.AppNodeIntermediate;
import de.uka.ilkd.key.proof.io.intermediate.BranchNodeIntermediate;
import de.uka.ilkd.key.proof.io.intermediate.BuiltInAppIntermediate;
import de.uka.ilkd.key.proof.io.intermediate.MergeAppIntermediate;
import de.uka.ilkd.key.proof.io.intermediate.MergePartnerAppIntermediate;
import de.uka.ilkd.key.proof.io.intermediate.NodeIntermediate;
import de.uka.ilkd.key.proof.io.intermediate.SMTAppIntermediate;
import de.uka.ilkd.key.proof.io.intermediate.TacletAppIntermediate;
import de.uka.ilkd.key.prover.impl.PerfScope;
import de.uka.ilkd.key.rule.*;
import de.uka.ilkd.key.rule.merge.MergePartner;
import de.uka.ilkd.key.rule.merge.MergeProcedure;
import de.uka.ilkd.key.rule.merge.MergeRuleBuiltInRuleApp;
import de.uka.ilkd.key.rule.merge.procedures.MergeWithPredicateAbstraction;
import de.uka.ilkd.key.rule.merge.procedures.MergeWithPredicateAbstractionFactory;
import de.uka.ilkd.key.settings.DefaultSMTSettings;
import de.uka.ilkd.key.settings.ProofIndependentSMTSettings;
import de.uka.ilkd.key.settings.ProofIndependentSettings;
import de.uka.ilkd.key.smt.*;
import de.uka.ilkd.key.smt.SMTSolverResult.ThreeValuedTruth;
import de.uka.ilkd.key.speclang.Contract;
import de.uka.ilkd.key.speclang.OperationContract;
import de.uka.ilkd.key.util.ProgressMonitor;
import de.uka.ilkd.key.util.mergerule.MergeRuleUtils;
import de.uka.ilkd.key.util.mergerule.SymbolicExecutionStateWithProgCnt;

import org.key_project.logic.Name;
import org.key_project.logic.Named;
import org.key_project.logic.sort.Sort;
import org.key_project.util.collection.DefaultImmutableSet;
import org.key_project.util.collection.ImmutableList;
import org.key_project.util.collection.ImmutableSLList;
import org.key_project.util.collection.ImmutableSet;
import org.key_project.util.collection.Pair;

import org.jspecify.annotations.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static de.uka.ilkd.key.util.mergerule.MergeRuleUtils.sequentToSETriple;


/**
 * This class is responsible for generating a KeY proof from an intermediate representation
 * generated by {@link IntermediatePresentationProofFileParser}.
 * <p>
 *
 * Replay is started using
 * {@link #replay(ProblemInitializer.ProblemInitializerListener, ProgressMonitor)}. In the course of
 * replaying, new nodes are added to the supplied proof object. The last goal touched during replay
 * can be obtained by {@link #getLastSelectedGoal()}.
 *
 * TODO: Check if joining with more than one partner works out of the box. Potential problem:
 * Different order may result in syntactically different nodes.
 *
 * @see IntermediatePresentationProofFileParser
 *
 * @author Dominic Scheurer
 */
public class IntermediateProofReplayer {
    /**
     * Set as {@link #getStatus()} if the proof contains SMT steps that didn't reload successfully.
     * Usually occurs if the timeout is set too low.
     */
    public static final String SMT_NOT_RUN =
        "Your proof has been loaded, but SMT solvers have not been run";

    private static final String ERROR_LOADING_PROOF_LINE = "Error loading proof.\n";
    private static final String NOT_APPLICABLE =
        " not available or not applicable in this context.";
    private static final Logger LOGGER = LoggerFactory.getLogger(IntermediateProofReplayer.class);


    /** The problem loader, for reporting errors */
    private final AbstractProblemLoader loader;
    /** The proof object into which to load the replayed proof */
    private Proof proof = null;

    /** Encountered errors */
    private final List<Throwable> errors = new LinkedList<>();
    /** Error status */
    private String status = "";

    /** Stores open branches */
    private final LinkedList<Pair<Node, NodeIntermediate>> queue =
        new LinkedList<>();

    /**
     * Used by the node merging during the proof replay.
     *
     * @param node the other node to be merged with
     * @param pio pio of ??? (ask Dominic Steinhöfel)
     * @param intermediate representation of the node during the replay
     * @see MergePartnerAppIntermediate
     */
    private record PartnerNode(Node node, PosInOccurrence pio, NodeIntermediate intermediate) {}

    /** Maps join node IDs to previously seen join partners */
    private final HashMap<Integer, HashSet<PartnerNode>> joinPartnerNodes = new HashMap<>();

    /** The current open goal */
    private Goal currGoal = null;

    /**
     * Constructs a new {@link IntermediateProofReplayer}.
     *
     * @param loader The problem loader, for reporting errors.
     * @param proof The proof object into which to load the replayed proof.
     * @param parserResult the result of the proof file parser to be replayed
     */
    public IntermediateProofReplayer(AbstractProblemLoader loader, Proof proof,
            IntermediatePresentationProofFileParser.Result parserResult) {
        this.proof = proof;
        this.loader = loader;

        queue.addFirst(
            new Pair<>(proof.root(), parserResult.parsedResult()));
    }

    /**
     * Constructs a new {@link IntermediateProofReplayer} without initializing the queue of
     * intermediate parsing results. Note that
     * {@link #replay(ProblemInitializer.ProblemInitializerListener, ProgressMonitor)} will not
     * work as expected when using this constructor, but other methods will.
     *
     * @param loader The problem loader, for reporting errors.
     * @param proof The proof object into which to load the replayed proof.
     */
    protected IntermediateProofReplayer(AbstractProblemLoader loader, Proof proof) {
        this.proof = proof;
        this.loader = loader;
    }

    /**
     * @return the lastSelectedGoal
     */
    public Goal getLastSelectedGoal() {
        return currGoal;
    }

    /**
     * Starts the actual replay process. Results are stored in the supplied proof object; the last
     * selected goal may be obtained by {@link #getLastSelectedGoal()}.
     * Note: This method deletes the intermediate proof tree!
     *
     * @param listener problem initializer listener for the current proof (may be null)
     * @param progressMonitor progress monitor used to report replay progress (may be null)
     * @return result of the replay procedure (see {@link Result})
     */
    public Result replay(ProblemInitializer.ProblemInitializerListener listener,
            ProgressMonitor progressMonitor) {
        return replay(listener, progressMonitor, true);
    }

    /**
     * Starts the actual replay process. Results are stored in the supplied
     * proof object; the last selected goal may be obtained by
     * {@link #getLastSelectedGoal()}.
     *
     * @param listener problem initializer listener for the current proof
     * @param progressMonitor progress monitor used to report replay progress
     * @param deleteIntermediateTree indicates if the intermediate proof tree should be
     *        deleted (set to false if it shal be kept for further use)
     * @return result of the replay procedure (see {@link Result})
     */
    public Result replay(ProblemInitializer.ProblemInitializerListener listener,
            ProgressMonitor progressMonitor, boolean deleteIntermediateTree) {
        // initialize progress monitoring
        int stepIndex = 0;
        int reportInterval = 1;
        int max = 0;
        var time = System.nanoTime();
        if (listener != null && progressMonitor != null) {
            max = !queue.isEmpty() && queue.peekFirst().second != null
                    ? queue.peekFirst().second.countAllChildren()
                    : 1;
            listener.reportStatus(this, "Replaying proof", max);
            reportInterval = Math.max(1, Integer.highestOneBit(max / 256));
        }

        while (!queue.isEmpty()) {
            // periodically report replay progress
            if (listener != null && progressMonitor != null && stepIndex % reportInterval == 0) {
                progressMonitor.setProgress(stepIndex);
            }
            stepIndex++;

            final Pair<Node, NodeIntermediate> currentP = queue.pollFirst();
            final Node currNode = currentP.first;
            final NodeIntermediate currNodeInterm = currentP.second;
            currGoal = proof.getOpenGoal(currNode);

            try {
                if (currNodeInterm instanceof BranchNodeIntermediate) {
                    assert currNodeInterm.getChildren().size() <= 1
                            : "Branch node should have exactly one child.";
                    if (currNodeInterm.getChildren().size() == 1) {
                        currNode.getNodeInfo().setBranchLabel(
                            ((BranchNodeIntermediate) currNodeInterm).getBranchTitle());
                        queue.addFirst(new Pair<>(currNode,
                            currNodeInterm.getChildren().get(0)));
                    }
                } else if (currNodeInterm instanceof AppNodeIntermediate currInterm) {

                    currNode.getNodeInfo().setNotes(currInterm.getNotes());

                    // Register name proposals
                    proof.getServices().getNameRecorder()
                            .setProposals(currInterm.getIntermediateRuleApp().getNewNames());

                    if (currInterm.getIntermediateRuleApp() instanceof TacletAppIntermediate) {
                        TacletAppIntermediate appInterm =
                            (TacletAppIntermediate) currInterm.getIntermediateRuleApp();

                        try {
                            currGoal.apply(constructTacletApp(appInterm, currGoal));

                            final Iterator<Node> children = currNode.childrenIterator();
                            final LinkedList<NodeIntermediate> intermChildren =
                                currInterm.getChildren();

                            addChildren(children, intermChildren);

                            // set information about SUCCESSFUL rule application
                            currNode.getNodeInfo().setInteractiveRuleApplication(
                                currInterm.isInteractiveRuleApplication());
                            currNode.getNodeInfo()
                                    .setScriptRuleApplication(currInterm.isScriptRuleApplication());

                            if (deleteIntermediateTree) {
                                // Children are no longer needed, set them to null
                                // to free memory.
                                currInterm.setChildren(null);
                            }


                        } catch (Exception | AssertionError e) {
                            reportError(ERROR_LOADING_PROOF_LINE + "Line " + appInterm.getLineNr()
                                + ", goal " + currGoal.node().serialNr() + ", rule "
                                + appInterm.getRuleName() + NOT_APPLICABLE, e);
                        }

                    } else if (currInterm
                            .getIntermediateRuleApp() instanceof BuiltInAppIntermediate) {
                        BuiltInAppIntermediate appInterm =
                            (BuiltInAppIntermediate) currInterm.getIntermediateRuleApp();

                        if (appInterm instanceof MergeAppIntermediate joinAppInterm) {
                            HashSet<PartnerNode> partnerNodesInfo =
                                joinPartnerNodes.get(((MergeAppIntermediate) appInterm).getId());

                            if (partnerNodesInfo == null
                                    || partnerNodesInfo.size() < joinAppInterm.getNrPartners()) {
                                // In case of an exception happening during the
                                // replay process, it can happen that the queue
                                // is
                                // empty when reaching this point. Then, we may
                                // not
                                // add the join node to the end of the queue
                                // since
                                // this will result in non-termination.

                                if (queue.isEmpty()) {
                                    continue;
                                }

                                // Wait until all partners are found: Add node
                                // at the end of the queue. NOTE: DO NOT CHANGE
                                // THIS to adding the node to the front! This
                                // will
                                // result in non-termination!
                                queue.addLast(
                                    new Pair<>(currNode, currNodeInterm));
                            } else {
                                try {
                                    final Services services = proof.getServices();

                                    MergeRuleBuiltInRuleApp joinApp = instantiateJoinApp(
                                        joinAppInterm, currNode, partnerNodesInfo, services);

                                    assert joinApp.complete()
                                            : "Join app should be automatically completed in replay";

                                    currGoal.apply(joinApp);

                                    final Iterator<Node> childrenIterator =
                                        currNode.childrenIterator();
                                    for (NodeIntermediate child : currInterm.getChildren()) {
                                        queue.addFirst(new Pair<>(
                                            childrenIterator.next(), child));
                                    }

                                    // Now add children of partner nodes
                                    for (PartnerNode partnerNodeInfo : partnerNodesInfo) {
                                        Iterator<Node> children =
                                            partnerNodeInfo.node.childrenIterator();
                                        LinkedList<NodeIntermediate> intermChildren =
                                            partnerNodeInfo.intermediate.getChildren();

                                        addChildren(children, intermChildren);
                                    }
                                } catch (SkipSMTRuleException | BuiltInConstructionException e) {
                                    reportError(
                                        ERROR_LOADING_PROOF_LINE + "Line " + appInterm.getLineNr()
                                            + ", goal " + currGoal.node().serialNr() + ", rule "
                                            + appInterm.getRuleName() + NOT_APPLICABLE,
                                        e);
                                }
                            }
                        } else if (appInterm instanceof MergePartnerAppIntermediate joinPartnerApp) {
                            // Register this partner node
                            HashSet<PartnerNode> partnerNodeInfo =
                                joinPartnerNodes.computeIfAbsent(joinPartnerApp.getMergeNodeId(),
                                    k -> new HashSet<>());

                            partnerNodeInfo.add(new PartnerNode(
                                currNode,
                                PosInOccurrence.findInSequent(currGoal.sequent(),
                                    appInterm.getPosInfo().first, appInterm.getPosInfo().second),
                                currNodeInterm));
                        } else {
                            try {
                                IBuiltInRuleApp app = constructBuiltinApp(appInterm, currGoal);
                                if (!app.complete()) {
                                    app = app.tryToInstantiate(currGoal);
                                }
                                currGoal.apply(app);

                                final Iterator<Node> children = currNode.childrenIterator();
                                LinkedList<NodeIntermediate> intermChildren =
                                    currInterm.getChildren();

                                addChildren(children, intermChildren);
                            } catch (SkipSMTRuleException e) {
                                // silently continue; status will be reported
                                // via
                                // polling
                            } catch (BuiltInConstructionException | AssertionError
                                    | RuntimeException e) {
                                reportError(ERROR_LOADING_PROOF_LINE + "Line "
                                    + appInterm.getLineNr() + ", goal " + currGoal.node().serialNr()
                                    + ", rule " + appInterm.getRuleName() + NOT_APPLICABLE, e);
                            }
                        }
                    }
                }
            } catch (Throwable throwable) {
                // Default exception catcher -- proof should not stop loading
                // if anything goes wrong, but instead continue with the next
                // node in the queue.
                reportError(ERROR_LOADING_PROOF_LINE, throwable);
            }
        }
        if (listener != null) {
            listener.reportStatus(this, "Proof loaded.");
        }

        if (listener != null && progressMonitor != null) {
            progressMonitor.setProgress(max);
        }
        LOGGER.debug("Proof replay took " + PerfScope.formatTime(System.nanoTime() - time));
        return new Result(status, errors, currGoal);
    }

    /**
     * Adds the pairs of proof node children and intermediate children to the queue. At the moment,
     * they are added in the order they were parsed. For the future, it may be sensible to choose a
     * different procedure, for instance one that minimizes the number of open goals per time
     * interval to save memory. Note that in this case, some test cases might be adapted which
     * depend on fixed node serial numbers.
     *
     * @param children Iterator of proof node children.
     * @param intermChildren List of corresponding intermediate children.
     */
    private void addChildren(Iterator<Node> children, LinkedList<NodeIntermediate> intermChildren) {
        int i = 0;
        while (!currGoal.node().isClosed() && children.hasNext() && intermChildren.size() > 0) {

            // NOTE: In the case of an unfinished proof, there
            // is another node after the last application which
            // is not represented by an intermediate
            // application. Therefore, we have to add the last
            // check in the above conjunction.

            Node child = children.next();
            if (!proof.getOpenGoal(child).isLinked()) {
                queue.add(i, new Pair<>(child, intermChildren.get(i++)));
            }
        }
    }

    /**
     * Communicates a non-fatal condition to the caller. Empty string means everything is OK. The
     * message will be displayed to the user in the GUI after the proof has been parsed.
     */
    public String getStatus() {
        return status;
    }

    /**
     * @return errors encountered during replay.
     */
    public Collection<Throwable> getErrors() {
        return errors;
    }

    /**
     * Constructs a taclet application from an intermediate one.
     *
     * @param currInterm The intermediate taclet application to create a "real" application for.
     * @param currGoal The goal on which to apply the taclet app.
     * @return The taclet application corresponding to the supplied intermediate representation.
     * @throws TacletAppConstructionException In case of an error during construction.
     */
    private TacletApp constructTacletApp(TacletAppIntermediate currInterm, Goal currGoal)
            throws TacletAppConstructionException {

        final String tacletName = currInterm.getRuleName();
        final int currFormula = currInterm.getPosInfo().first;
        final PosInTerm currPosInTerm = currInterm.getPosInfo().second;
        final Sequent seq = currGoal.sequent();

        TacletApp ourApp;
        PosInOccurrence pos = null;

        Taclet t = proof.getInitConfig().lookupActiveTaclet(new Name(tacletName));
        if (t == null) {
            ourApp = currGoal.indexOfTaclets().lookup(tacletName);
        } else {
            ourApp = NoPosTacletApp.createNoPosTacletApp(t);
        }

        if (ourApp == null) {
            throw new TacletAppConstructionException(
                "Unknown taclet with name \"" + tacletName + "\"");
        }

        Services services = proof.getServices();

        if (currFormula != 0) { // otherwise we have no pos
            try {
                pos = PosInOccurrence.findInSequent(currGoal.sequent(), currFormula, currPosInTerm);

                /*
                 * part of the fix for #1716: ensure that position of find term
                 * (antecedent/succedent) matches the kind of the taclet.
                 */
                Taclet taclet = ourApp.taclet();
                if (taclet instanceof AntecTaclet && !pos.isInAntec()) {
                    throw new TacletAppConstructionException("The taclet " + taclet.name()
                        + " can not be applied to a formula/term in succedent.");
                } else if (taclet instanceof SuccTaclet && pos.isInAntec()) {
                    throw new TacletAppConstructionException("The taclet " + taclet.name()
                        + " can not be applied to a formula/term in antecedent.");
                }

                ourApp = ((NoPosTacletApp) ourApp).matchFind(pos, services);
                ourApp = ourApp.setPosInOccurrence(pos, services);
            } catch (TacletAppConstructionException e) {
                throw e;
            } catch (Exception e) {
                throw new TacletAppConstructionException("Wrong position information: " + pos, e);
            }
        }

        ourApp = constructInsts(ourApp, currGoal, currInterm.getInsts(), services);

        ImmutableList<IfFormulaInstantiation> ifFormulaList =
            ImmutableSLList.nil();
        for (String ifFormulaStr : currInterm.getIfSeqFormulaList()) {
            ifFormulaList =
                ifFormulaList.append(new IfFormulaInstSeq(seq, Integer.parseInt(ifFormulaStr)));
        }
        for (String ifFormulaStr : currInterm.getIfDirectFormulaList()) {
            // MU 2019: #1487. We have to use the right namespaces to not
            // ignore branch-local functions
            NamespaceSet nss = currGoal.getLocalNamespaces();
            Term term = parseTerm(ifFormulaStr, proof, nss.variables(), nss.programVariables(),
                nss.functions());
            ifFormulaList = ifFormulaList.append(new IfFormulaInstDirect(new SequentFormula(term)));
        }

        if (!ourApp.ifInstsCorrectSize(ifFormulaList)) {
            LOGGER.warn("Proof contains wrong number of \\assumes instatiations for {}",
                tacletName);
            // try to find instantiations automatically
            ImmutableList<TacletApp> instApps = ourApp.findIfFormulaInstantiations(seq, services);
            if (instApps.size() != 1) {
                // none or not a unique result
                throw new TacletAppConstructionException("\nCould not apply " + tacletName
                    + "\nUnknown instantiations for \\assumes. " + instApps.size()
                    + " candidates.\n" + "Perhaps the rule's definition has been changed in KeY.");
            }

            TacletApp newApp = instApps.head();
            ifFormulaList = newApp.ifFormulaInstantiations();
        }

        // TODO: In certain cases, the below method call returns null and
        // induces follow-up NullPointerExceptions. This was encountered
        // in a proof of the TimSort method binarySort with several joins.
        ourApp = ourApp.setIfFormulaInstantiations(ifFormulaList, services);

        if (!ourApp.complete()) {
            ourApp = ourApp.tryToInstantiate(proof.getServices());
        }

        return ourApp;
    }

    /**
     * Constructs a built-in rule application from an intermediate one.
     *
     * @param currInterm The intermediate built-in application to create a "real" application for.
     * @param currGoal The goal on which to apply the built-in app.
     * @return The built-in application corresponding to the supplied intermediate representation.
     * @throws SkipSMTRuleException If the proof has been loaded, but the SMT solvers have not been
     *         run.
     * @throws BuiltInConstructionException In case of an error during construction.
     */
    private IBuiltInRuleApp constructBuiltinApp(BuiltInAppIntermediate currInterm, Goal currGoal)
            throws SkipSMTRuleException, BuiltInConstructionException {

        final String ruleName = currInterm.getRuleName();
        final int currFormula = currInterm.getPosInfo().first;
        final PosInTerm currPosInTerm = currInterm.getPosInfo().second;

        Contract currContract = null;
        ImmutableList<PosInOccurrence> builtinIfInsts = null;

        // Load contracts, if applicable
        if (currInterm.getContract() != null) {
            currContract = proof.getServices().getSpecificationRepository()
                    .getContractByName(currInterm.getContract());
            if (currContract == null) {
                final ProblemLoaderException e =
                    new ProblemLoaderException(loader, "Error loading proof: contract \""
                        + currInterm.getContract() + "\" not found.");
                reportError(ERROR_LOADING_PROOF_LINE + ", goal " + currGoal.node().serialNr()
                    + ", rule " + ruleName + NOT_APPLICABLE, e);
            }
        }

        // Load ifInsts, if applicable
        if (currInterm.getBuiltInIfInsts() != null) {
            builtinIfInsts = ImmutableSLList.nil();
            for (final Pair<Integer, PosInTerm> ifInstP : currInterm.getBuiltInIfInsts()) {
                final int currIfInstFormula = ifInstP.first;
                final PosInTerm currIfInstPosInTerm = ifInstP.second;

                try {
                    final PosInOccurrence ifInst = PosInOccurrence.findInSequent(currGoal.sequent(),
                        currIfInstFormula, currIfInstPosInTerm);
                    builtinIfInsts = builtinIfInsts.append(ifInst);
                } catch (RuntimeException | AssertionError e) {
                    reportError(
                        ERROR_LOADING_PROOF_LINE + "Line " + currInterm.getLineNr() + ", goal "
                            + currGoal.node().serialNr() + ", rule " + ruleName + NOT_APPLICABLE,
                        e);
                }
            }
        }

        if (SMTRuleApp.RULE.name().toString().equals(ruleName)) {
            if (!ProofIndependentSettings.DEFAULT_INSTANCE.getSMTSettings().isEnableOnLoad()) {
                status = SMT_NOT_RUN;
                throw new SkipSMTRuleException();
            }
            boolean error = false;
            final SMTProblem smtProblem = new SMTProblem(currGoal);
            try {
                DefaultSMTSettings settings = new DefaultSMTSettings(
                    proof.getSettings().getSMTSettings(),
                    ProofIndependentSettings.DEFAULT_INSTANCE.getSMTSettings(),
                    proof.getSettings().getNewSMTSettings(),
                    proof);

                ProofIndependentSMTSettings smtSettings = ProofIndependentSettings.DEFAULT_INSTANCE
                        .getSMTSettings();
                SolverTypeCollection active = smtSettings.computeActiveSolverUnion();
                SMTAppIntermediate smtAppIntermediate = (SMTAppIntermediate) currInterm;
                String smtSolver = smtAppIntermediate.getSolver();
                if (smtSolver == null || smtSolver.isEmpty()) {
                    // default to Z3 because this one most likely was used when saving the proof
                    smtSolver = "Z3";
                }
                // try to find the solver that closed the proof
                for (SolverTypeCollection su : smtSettings.getSolverUnions(true)) {
                    if (su.name().equals(smtSolver)) {
                        active = su;
                        break;
                    }
                }
                ArrayList<SMTProblem> problems = new ArrayList<>();
                problems.add(smtProblem);

                SolverLauncher launcher = new SolverLauncher(settings);
                launcher.launch(active.getTypes(), problems,
                    proof.getServices());
            } catch (Exception e) {
                error = true;
            }
            if (error || smtProblem.getFinalResult().isValid() != ThreeValuedTruth.VALID) {
                status = SMT_NOT_RUN;
                throw new SkipSMTRuleException();
            } else {
                String name = smtProblem.getSuccessfulSolver().name();
                ImmutableList<PosInOccurrence> unsatCore = SMTFocusResults.getUnsatCore(smtProblem);
                if (unsatCore != null) {
                    return SMTRuleApp.RULE.createApp(name, unsatCore);
                } else {
                    return SMTRuleApp.RULE.createApp(name);
                }
            }
        }

        IBuiltInRuleApp ourApp = null;
        PosInOccurrence pos = null;

        if (currFormula != 0) { // otherwise we have no pos
            try {
                pos = PosInOccurrence.findInSequent(currGoal.sequent(), currFormula, currPosInTerm);
            } catch (RuntimeException e) {
                throw new BuiltInConstructionException("Wrong position information.", e);
            }
        }

        if (currContract != null) {
            AbstractContractRuleApp contractApp = null;

            BuiltInRule useContractRule;
            if (currContract instanceof OperationContract) {
                useContractRule = UseOperationContractRule.INSTANCE;
                contractApp = (((UseOperationContractRule) useContractRule).createApp(pos))
                        .setContract(currContract);
            } else {
                useContractRule = UseDependencyContractRule.INSTANCE;
                contractApp = (((UseDependencyContractRule) useContractRule).createApp(pos))
                        .setContract(currContract);
                // restore "step" if needed
                var depContractApp = ((UseDependencyContractApp) contractApp);
                if (depContractApp.step() == null) {
                    contractApp = depContractApp.setStep(builtinIfInsts.head());
                }
            }

            if (contractApp.check(currGoal.proof().getServices()) == null) {
                throw new BuiltInConstructionException("Cannot apply contract: " + currContract);
            } else {
                ourApp = contractApp;
            }

            currContract = null;
            if (builtinIfInsts != null) {
                ourApp = ourApp.setIfInsts(builtinIfInsts);
                builtinIfInsts = null;
            }
            return ourApp;
        }

        final ImmutableSet<IBuiltInRuleApp> ruleApps = collectAppsForRule(ruleName, currGoal, pos);
        if (ruleApps.size() != 1) {
            if (ruleApps.size() < 1) {
                throw new BuiltInConstructionException(
                    ruleName + " is missing. Most probably the binary "
                        + "for this built-in rule is not in your path or "
                        + "you do not have the permission to execute it.");
            } else {
                throw new BuiltInConstructionException(ruleName + ": found " + ruleApps.size()
                    + " applications. Don't know what to do !\n" + "@ " + pos);
            }
        }
        ourApp = ruleApps.iterator().next();
        if (ourApp instanceof OneStepSimplifierRuleApp) {
            ((OneStepSimplifierRuleApp) ourApp).restrictAssumeInsts(builtinIfInsts);
        }
        builtinIfInsts = null;
        return ourApp;
    }

    /**
     * Instantiates a Join Rule application.
     *
     * @param joinAppInterm Intermediate join app.
     * @param services The services object.
     * @param currNode The current proof node.
     * @param partnerNodesInfo Information about join partner nodes.
     * @param currNode
     * @param partnerNodesInfo
     * @return The instantiated Join Rule application.
     * @throws SkipSMTRuleException If the proof has been loaded, but the SMT solvers have not been
     *         run.
     * @throws BuiltInConstructionException In case of an error during construction of the builtin
     *         rule app.
     */
    private MergeRuleBuiltInRuleApp instantiateJoinApp(final MergeAppIntermediate joinAppInterm,
            final Node currNode,
            final HashSet<PartnerNode> partnerNodesInfo,
            final Services services) throws SkipSMTRuleException, BuiltInConstructionException {
        final MergeRuleBuiltInRuleApp joinApp =
            (MergeRuleBuiltInRuleApp) constructBuiltinApp(joinAppInterm, currGoal);
        joinApp.setConcreteRule(MergeProcedure.getProcedureByName(joinAppInterm.getJoinProc()));
        joinApp.setDistinguishingFormula(
            MergeRuleUtils.translateToFormula(services, joinAppInterm.getDistinguishingFormula()));

        // Predicate abstraction join rule
        if (joinApp.getConcreteRule() instanceof MergeWithPredicateAbstractionFactory) {
            List<AbstractionPredicate> predicates = new ArrayList<>();

            // It may happen that the abstraction predicates are null -- in this
            // case, it is the expected behavior to create a default lattice
            // with top and bottom elements only, which is accomplished by just
            // supplying an empty list of predicates to the join procedure.
            if (joinAppInterm.getAbstractionPredicates() != null) {
                try {
                    predicates =
                        AbstractionPredicate.fromString(joinAppInterm.getAbstractionPredicates(),
                            services,
                            services.getProof().getOpenGoal(currNode).getLocalNamespaces());
                } catch (ParserException e) {
                    errors.add(e);
                }
            }

            final Class<? extends AbstractPredicateAbstractionLattice> latticeType =
                joinAppInterm.getPredAbstrLatticeType();

            LinkedHashMap<ProgramVariable, AbstractDomainElement> userChoices =
                new LinkedHashMap<>();

            if (joinAppInterm.getUserChoices() != null) {
                final Pattern p = Pattern.compile("\\('(.+?)', `(.+?)`\\)");
                final Matcher m = p.matcher(joinAppInterm.getUserChoices());

                boolean matched = false;
                while (m.find()) {
                    matched = true;

                    for (int i = 1; i < m.groupCount(); i += 2) {
                        assert i + 1 <= m.groupCount() : "Wrong format of join user choices: "
                            + "There should always be pairs of program variables "
                            + "and abstract domain elements.";

                        final String progVarStr = m.group(i);
                        final String abstrElemStr = m.group(i + 1);

                        // Parse the program variable
                        final Pair<Sort, Name> ph =
                            MergeRuleUtils.parsePlaceholder(progVarStr, false, services);

                        final List<AbstractionPredicate> applicablePredicates =
                            predicates.stream()
                                    .filter(pred -> pred.getArgSort().equals(ph.first))
                                    .collect(Collectors.toList());

                        // Parse the abstract domain element
                        final AbstractDomainElement elem = MergeWithPredicateAbstraction
                                .instantiateAbstractDomain(ph.first, applicablePredicates,
                                    latticeType, services)
                                .fromString(abstrElemStr, services);
                        final Named pv =
                            services.getNamespaces().programVariables().lookup(ph.second);

                        assert pv instanceof ProgramVariable
                                && ((ProgramVariable) pv).sort().equals(ph.first)
                                : "Program variable involved in join is not known to the system";

                        userChoices.put((ProgramVariable) pv, elem);
                    }
                }

                if (!matched) {
                    errors.add(new ParserException("Wrong format of join user choices.", null));
                }
            }

            // Instantiate the join procedure
            joinApp.setConcreteRule(
                ((MergeWithPredicateAbstractionFactory) joinApp.getConcreteRule()).instantiate(
                    predicates,
                    latticeType == null ? SimplePredicateAbstractionLattice.class : latticeType,
                    userChoices));

        }

        ImmutableList<MergePartner> joinPartners = ImmutableSLList.nil();
        for (PartnerNode partnerNodeInfo : partnerNodesInfo) {

            SymbolicExecutionStateWithProgCnt ownSEState =
                sequentToSETriple(currNode, joinApp.posInOccurrence(), services);
            var partnerSEState =
                sequentToSETriple(partnerNodeInfo.node, partnerNodeInfo.pio, services);

            assert ownSEState.programCounter().equals(partnerSEState.programCounter())
                    : "Cannot merge incompatible program counters";

            joinPartners = joinPartners.append(
                new MergePartner(proof.getOpenGoal(partnerNodeInfo.node), partnerNodeInfo.pio));
        }

        joinApp.setMergeNode(currNode);
        joinApp.setMergePartners(joinPartners);

        return joinApp;
    }

    // ######## Below: Methods previously listed in DefaultProofFileParser

    /**
     * Stores an error in the list.
     *
     * @param string Description text.
     * @param e Error encountered.
     */
    private void reportError(String string, Throwable e) {
        status = "Errors while reading the proof. Not all branches could be load successfully.";
        errors.add(new ProblemLoaderException(loader, string, e));
    }

    /**
     * Retrieves all registered applications at the given goal and position for the rule
     * corresponding to the given ruleName.
     *
     * @param ruleName Name of the rule to find applications for.
     * @param g Goal to search.
     * @param pos Position of interest in the given goal.
     * @return All matching rule applications at pos in g.
     */
    public static ImmutableSet<IBuiltInRuleApp> collectAppsForRule(String ruleName, Goal g,
            PosInOccurrence pos) {

        ImmutableSet<IBuiltInRuleApp> result = DefaultImmutableSet.nil();

        for (final IBuiltInRuleApp app : g.ruleAppIndex().getBuiltInRules(g, pos)) {
            if (app.rule().name().toString().equals(ruleName)) {
                result = result.add(app);
            }
        }

        return result;
    }

    /**
     * Instantiates schema variables in the given taclet application.
     *
     * @param app The taclet application to instantiate.
     * @param currGoal The corresponding goal.
     * @param loadedInsts Loaded schema variable instantiations.
     * @param services The services object.
     * @return The instantiated taclet.
     */
    public static TacletApp constructInsts(@NonNull TacletApp app, Goal currGoal,
            Collection<String> loadedInsts, Services services) {
        if (loadedInsts == null) {
            return app;
        }
        ImmutableSet<SchemaVariable> uninsts = app.uninstantiatedVars();

        // first pass: add variables
        for (final String s : loadedInsts) {
            int eq = s.indexOf('=');
            final String varname = s.substring(0, eq);

            SchemaVariable sv = lookupName(uninsts, varname);
            if (sv == null) {
                // throw new IllegalStateException(
                // varname+" from \n"+loadedInsts+"\n is not in\n"+uninsts);
                LOGGER.error("{} from {} is not in uninsts", varname, app.rule().name());
                continue;
            }
            final String value = s.substring(eq + 1);
            if (sv instanceof VariableSV vsv) {
                app = parseSV1(app, vsv, value, services);
            }
        }

        // second pass: add everything else
        uninsts = app.uninstantiatedVars();
        for (final String s : loadedInsts) {
            int eq = s.indexOf('=');
            final String varname = s.substring(0, eq);
            final SchemaVariable sv = lookupName(uninsts, varname);
            if (sv == null) {
                continue;
            }

            String value = s.substring(eq + 1);
            app = parseSV2(app, sv, value, currGoal);
        }

        return app;
    }

    /**
     * Finds a schema variable in the given set.
     *
     * @param set The set to search.
     * @param name The name to search for.
     * @return The found schema variable, or null if it is not present in the set.
     */
    private static SchemaVariable lookupName(ImmutableSet<SchemaVariable> set, String name) {
        for (SchemaVariable v : set) {
            if (v.name().toString().equals(name)) {
                return v;
            }
        }
        return null; // handle this better!
    }

    /**
     * Parses a given term in String representation.
     *
     * @param value String to parse.
     * @param proof Proof object (for namespaces and Services object).
     * @param varNS Variable namespace.
     * @param progVarNS Program variable namespace.
     * @return The parsed term.
     * @throws ParserException In case of an error.
     */
    public static Term parseTerm(String value, Proof proof, Namespace<QuantifiableVariable> varNS,
            Namespace<IProgramVariable> progVarNS, Namespace<JFunction> functNS) {
        try {
            return new DefaultTermParser().parse(new StringReader(value), null, proof.getServices(),
                varNS, functNS, proof.getNamespaces().sorts(),
                progVarNS, new AbbrevMap());
        } catch (ParserException e) {
            throw new RuntimeException(
                "Error while parsing value " + value + "\nVar namespace is: " + varNS + "\n", e);
        }
    }

    /**
     * Parses a given term in String representation.
     *
     * @param value String to parse.
     * @param proof Proof object (for namespaces and Services object).
     * @return The parsed term.
     */
    public static Term parseTerm(String value, Proof proof) {
        NamespaceSet nss = proof.getNamespaces();
        return parseTerm(value, proof, nss.variables(), nss.programVariables(), nss.functions());
    }

    /**
     * Instantiates a schema variable in the given taclet application. 1st pass: only VariableSV.
     *
     * @param app Application to instantiate.
     * @param sv VariableSV to instantiate.
     * @param value Name for the instantiated logic variable.
     * @param services The services object.
     * @return An instantiated taclet application, where the schema variable has been instantiated
     *         by a logic variable of the given name.
     */
    public static TacletApp parseSV1(TacletApp app, VariableSV sv, String value,
            Services services) {
        LogicVariable lv = new LogicVariable(new Name(value), app.getRealSort(sv, services));
        Term instance = services.getTermFactory().createTerm(lv);
        return app.addCheckedInstantiation(sv, instance, services, true);
    }

    /**
     * Instantiates a schema variable in the given taclet application. 2nd pass: All other schema
     * variables.
     *
     * @param app Application to instantiate.
     * @param sv Schema variable to instantiate.
     * @param value Name for the instantiated Skolem constant, program element or term..
     * @param targetGoal The goal corresponding to the given application.
     * @return An instantiated taclet application, where the schema variable has been instantiated,
     *         depending on its type, by a Skolem constant, program element, or term of the given
     *         name.
     * @see #parseSV1(TacletApp, VariableSV, String, Services)
     */
    public static TacletApp parseSV2(TacletApp app, SchemaVariable sv, String value,
            Goal targetGoal) {
        final Proof p = targetGoal.proof();
        final Services services = p.getServices();
        TacletApp result;
        if (sv instanceof VariableSV) {
            // ignore -- already done
            result = app;
        } else if (sv instanceof ProgramSV psv) {
            final ProgramElement pe = app.getProgramElement(value, psv, services);
            result = app.addCheckedInstantiation(sv, pe, services, true);
        } else if (sv instanceof SkolemTermSV skolemSv) {
            result = app.createSkolemConstant(value, skolemSv, true, services);
        } else if (sv instanceof ModalOperatorSV msv) {
            result = app.addInstantiation(
                app.instantiations().add(msv, Modality.JavaModalityKind.getKind(value), services),
                services);
        } else {
            Namespace<QuantifiableVariable> varNS = p.getNamespaces().variables();
            Namespace<IProgramVariable> prgVarNS =
                targetGoal.getLocalNamespaces().programVariables();
            Namespace<JFunction> funcNS = targetGoal.getLocalNamespaces().functions();
            varNS = app.extendVarNamespaceForSV(varNS, sv);
            Term instance = parseTerm(value, p, varNS, prgVarNS, funcNS);
            result = app.addCheckedInstantiation(sv, instance, services, true);
        }
        return result;
    }

    /**
     * Signals an error during construction of a taclet app.
     */
    public static class TacletAppConstructionException extends Exception {
        private static final long serialVersionUID = 7859543482157633999L;

        TacletAppConstructionException(String s) {
            super(s);
        }

        TacletAppConstructionException(String s, Throwable cause) {
            super(s, cause);
        }
    }

    /**
     * Signals an error during construction of a built-in rule app.
     */
    public static class BuiltInConstructionException extends Exception {
        private static final long serialVersionUID = -735474220502290816L;

        public BuiltInConstructionException(String s) {
            super(s);
        }

        BuiltInConstructionException(Throwable cause) {
            super(cause);
        }

        public BuiltInConstructionException(String s, Throwable cause) {
            super(s, cause);
        }
    }

    /**
     * Signals that the execution of an SMT solver, that has been used before the now loaded proof
     * was saved, has been skipped.
     */
    static class SkipSMTRuleException extends Exception {
        private static final long serialVersionUID = -2932282883810135168L;
    }

    /**
     * Simple structure containing the results of the replay procedure.
     *
     * @author Dominic Scheurer
     */
    public static class Result {
        private final String status;
        private final List<Throwable> errors;
        private Goal lastSelectedGoal = null;

        public Result(String status, List<Throwable> errors, Goal lastSelectedGoal) {
            this.status = status;
            this.errors = errors;
            this.lastSelectedGoal = lastSelectedGoal;
        }

        public String getStatus() {
            return status;
        }

        public List<Throwable> getErrors() {
            return errors;
        }

        public Goal getLastSelectedGoal() {
            return lastSelectedGoal;
        }
    }
}
