/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package de.uka.ilkd.key.informationflow.rule.executor;

import de.uka.ilkd.key.informationflow.rule.InfFlowContractAppTaclet;
import de.uka.ilkd.key.java.Services;
import de.uka.ilkd.key.logic.*;
import de.uka.ilkd.key.logic.label.TermLabelState;
import de.uka.ilkd.key.proof.Goal;
import de.uka.ilkd.key.proof.StrategyInfoUndoMethod;
import de.uka.ilkd.key.rule.MatchConditions;
import de.uka.ilkd.key.rule.RuleApp;
import de.uka.ilkd.key.rule.Taclet.TacletLabelHint;
import de.uka.ilkd.key.rule.executor.javadl.RewriteTacletExecutor;
import de.uka.ilkd.key.util.properties.Properties;

import org.key_project.util.collection.ImmutableList;
import org.key_project.util.collection.ImmutableSLList;

public class InfFlowContractAppTacletExecutor
        extends RewriteTacletExecutor<InfFlowContractAppTaclet> {
    /**
     * Strategy property which saves the list of formulas which where added by information flow
     * contract applications. This list is used by the macros UseInformationFlowContractMacro and
     * PrepareInfFlowContractPreBranchesMacro to decide how to prepare the formulas resulting from
     * information flow contract applications.
     */
    @SuppressWarnings("unchecked")
    public static final Properties.Property<ImmutableList<Term>> INF_FLOW_CONTRACT_APPL_PROPERTY =
        new Properties.Property<>(
            (Class<ImmutableList<Term>>) (Class<?>) ImmutableList.class,
            "information flow contract applicaton property");


    public InfFlowContractAppTacletExecutor(InfFlowContractAppTaclet taclet) {
        super(taclet);
    }


    @Override
    protected void addToAntec(Semisequent semi, TermLabelState termLabelState,
            TacletLabelHint labelHint, SequentChangeInfo currentSequent, PosInOccurrence pos,
            PosInOccurrence applicationPosInOccurrence, MatchConditions matchCond, Goal goal,
            RuleApp tacletApp, Services services) {
        final ImmutableList<SequentFormula> replacements = instantiateSemisequent(semi,
            termLabelState, labelHint, pos, matchCond, goal, tacletApp, services);
        assert replacements.size() == 1
                : "information flow taclets must have " + "exactly one add!";
        updateStrategyInfo(services.getProof().openEnabledGoals().head(),
            replacements.iterator().next().formula());
        super.addToAntec(semi, termLabelState, labelHint, currentSequent, pos,
            applicationPosInOccurrence, matchCond, goal, tacletApp, services);
    }

    /**
     * Add the contract application formula to the list of the INF_FLOW_CONTRACT_APPL_PROPERTY.
     *
     * @param goal the current goal
     * @param applFormula the information contract application formula added by this taclet
     */
    private void updateStrategyInfo(Goal goal, final Term applFormula) {
        ImmutableList<Term> applFormulas = goal.getStrategyInfo(INF_FLOW_CONTRACT_APPL_PROPERTY);
        if (applFormulas == null) {
            applFormulas = ImmutableSLList.nil();
        }
        applFormulas = applFormulas.append(applFormula);
        StrategyInfoUndoMethod undo = strategyInfos -> {
            ImmutableList<Term> applFormulas1 =
                strategyInfos.get(INF_FLOW_CONTRACT_APPL_PROPERTY);
            strategyInfos.put(INF_FLOW_CONTRACT_APPL_PROPERTY,
                applFormulas1.removeAll(applFormula));
        };
        goal.addStrategyInfo(INF_FLOW_CONTRACT_APPL_PROPERTY, applFormulas, undo);

    }

}
