/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package de.uka.ilkd.key.testgen;

import java.util.HashSet;

import de.uka.ilkd.key.util.KeYConstants;

import org.key_project.logic.sort.Sort;
import org.key_project.util.java.StringUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Creates the RFL.java file, that provides setter and getter methods using the reflection API as
 * well as object creation functions based on the objenesis library.
 *
 * @author mbender
 * @author gladisch
 */
public class ReflectionClassCreator {
    /**
     * Constant for the line break which is used by the operating system.
     * <p>
     * <b>Do not use {@code \n}!</b>
     */
    public static final String NEW_LINE = StringUtil.NEW_LINE;

    public static final String NAME_OF_CLASS = "RFL";

    public static final String ARRAY = "_ARRAY_";
    public static final String SET_PREFIX = "_set_";
    public static final String GET_PREFIX = "_get_";

    // setter and getter methods will be created for these types.
    private static final String[] PRIMITIVE_TYPES =
        { "int", "long", "byte", "char", "boolean", "float", "double" };

    // Default values for primitive types
    private static final String[] PRIM_TYP_DEF_VAL = { "0", "0", "0", "' '", "false", "0", "0" };

    private static final Logger LOGGER = LoggerFactory.getLogger(ReflectionClassCreator.class);

    private final HashSet<Sort> usedObjectSorts;
    private final HashSet<String> usedObjectSortsStrings;


    public ReflectionClassCreator() {
        usedObjectSorts = new HashSet<>();
        usedObjectSortsStrings = new HashSet<>();
    }

    /**
     * Creates the RFL.java file, that provides setter and getter methods using the reflection API
     * as well as object creation functions based on the objenesis library.
     */
    public StringBuilder createClass(boolean staticClass) {
        final HashSet<String> sorts = sortsToString();
        final var result = new StringBuilder();
        result.append(classDecl(staticClass));
        result.append(ghostMapDecls(true));
        result.append(staticInitializer(true));
        result.append(instanceMethod());
        result.append(instances(sorts));
        result.append(getterAndSetter(sorts));
        result.append(footer());
        if (!checkBraces(result)) {
            throw new IllegalStateException("ReflectionClassCreator.createClass(): "
                + "Problem: the number of opening and closing braces of the generated RFL file is not equal!");
        }
        return result;
    }

    /**
     * Constructors, setter-, getter-methods will be created for the added sorts.
     */
    public void addSort(Sort s) {
        usedObjectSorts.add(s);
    }

    /**
     * Constructors, setter-, getter-methods will be created for the added sorts.
     */
    public void addSort(String s) {
        usedObjectSortsStrings.add(s);
    }

    /**
     * @return String representations for all non primitive types
     */
    private HashSet<String> sortsToString() {
        final HashSet<String> result = new HashSet<>();
        for (final Sort var : usedObjectSorts) {
            String sort = var.toString();
            // We only want Object-Types
            if (!" jbyte jint jlong jfloat jdouble jboolean jchar ".contains(" " + sort + " ")) {
                if (" jbyte[] jint[] jlong[] jfloat[] jdouble[] jboolean[] jchar[] "
                        .contains(" " + sort + " ")) {
                    sort = sort.substring(1);
                }
                if (!isPrimitiveType(sort)) {
                    result.add(sort);
                }
            }
        }
        for (String sort : usedObjectSortsStrings) {
            // We only want Object-Types
            if (!" jbyte jint jlong jfloat jdouble jboolean jchar ".contains(" " + sort + " ")) {
                if (" jbyte[] jint[] jlong[] jfloat[] jdouble[] jboolean[] jchar[] "
                        .contains(" " + sort + " ")) {
                    sort = sort.substring(1);
                }
                if (!isPrimitiveType(sort)) {
                    result.add(sort);
                }
            }
        }
        return result;
    }

    /**
     * @return Beginning of the RFL class
     */
    private StringBuilder classDecl(boolean staticClass) {
        final StringBuilder r = new StringBuilder();
        r.append(NEW_LINE);
        r.append("// This file was generated by KeY Version ").append(KeYConstants.VERSION)
                .append(" (www.key-project.org).").append(NEW_LINE).append(NEW_LINE)
                .append(
                    "/** This class enables the test suite to read and write protected and private")
                .append(NEW_LINE)
                .append(
                    " * fields of other classes. It can also simulate ghost fields using a hashmap.")
                .append(NEW_LINE)
                .append(
                    " * Ghostfields are implicit fields that exist in the specification but not in the")
                .append(NEW_LINE)
                .append(
                    " * actual Java class. Futhermore, this class also enables to create an object of ")
                .append(NEW_LINE)
                .append(
                    " * any class even if it has no default constructor. To create objects the ")
                .append(NEW_LINE)
                .append(
                    " * the objenesis library is required and must be provided when compiling and")
                .append(NEW_LINE).append(" * executing the test suite. ").append(NEW_LINE);
        r.append(" * @see http://docs.oracle.com/javase/tutorial/reflect/member/ctorInstance.html")
                .append(NEW_LINE);
        r.append(" * @see http://code.google.com/p/objenesis/").append(NEW_LINE)
                .append(" * @see http://objenesis.org/").append(NEW_LINE);
        r.append(" * @author gladisch").append(NEW_LINE);
        r.append(" * @author mbender").append(NEW_LINE);
        r.append(" */").append(NEW_LINE);
        r.append("public ");
        if (staticClass) {
            r.append("static ");
        }
        r.append("class " + NAME_OF_CLASS + " {").append(NEW_LINE);
        return r;
    }

    /**
     * Writes a hashmap and a utility method for associating ghost/model fiels with objects.
     *
     * @param ghostMapActive becomes are runtime flag that determins if the hashmap should be
     *        enabled or not.
     */
    private StringBuilder ghostMapDecls(boolean ghostMapActive) {
        final StringBuilder r = new StringBuilder();
        r.append(NEW_LINE);

        r.append("  private static final String NoSuchFieldExceptionText;");

        r.append("  public static boolean ghostMapActive;");

        r.append("  public static java.util.HashMap<Integer,Object> ghostModelFields;")
                .append(NEW_LINE).append(NEW_LINE);

        r.append("  public static int getHash(Class<?> c, Object obj, String attr){")
                .append(NEW_LINE);
        r.append("    return c.hashCode() * (obj!=null?obj.hashCode():1) * attr.hashCode();")
                .append(NEW_LINE);
        r.append("  }").append(NEW_LINE).append(NEW_LINE);
        return r;
    }

    /**
     * @return The method that allows to create new instances
     */
    private StringBuilder instanceMethod() {
        final StringBuilder r = new StringBuilder();
        r.append(NEW_LINE).append(NEW_LINE);
        r.append(
            "  /** The Objenesis library can create instances of classes that have no default constructor. */")
                .append(NEW_LINE);
        r.append("  private static org.objenesis.Objenesis objenesis;").append(NEW_LINE)
                .append(NEW_LINE);
        r.append("  private static Object newInstance(Class c) throws Exception {")
                .append(NEW_LINE);
        r.append("    Object res=objenesis.newInstance(c);").append(NEW_LINE);
        r.append("    if (res==null)").append(NEW_LINE);
        r.append("      throw new Exception(\"Couldn't create instance of class:\"+c);")
                .append(NEW_LINE);
        r.append("  return res;").append(NEW_LINE);
        r.append("  }").append(NEW_LINE);
        return r;
    }

    private StringBuilder staticInitializer(boolean ghostMapActive) {
        StringBuilder r = new StringBuilder();
        String tab = "   ";
        r.append(NEW_LINE).append(NEW_LINE);
        r.append(tab).append("static{").append(NEW_LINE);

        r.append(tab).append("objenesis = new org.objenesis.ObjenesisStd();").append(NEW_LINE);

        r.append(tab).append("ghostMapActive = ").append(ghostMapActive).append(";")
                .append(NEW_LINE);

        r.append(tab).append("ghostModelFields = new java.util.HashMap<Integer,Object>();")
                .append(NEW_LINE);

        r.append(tab).append("NoSuchFieldExceptionText =").append(NEW_LINE);
        r.append(tab).append(tab).append(
            "  \"This exception occurs when ghost fields or model fields are used in the code or \" +")
                .append(NEW_LINE);
        r.append(tab).append(tab).append(
            "  \"if mock objects are used that have different fields, than the real objects. \" +")
                .append(NEW_LINE);
        r.append(tab).append(tab).append(
            "  \"The tester should extend the handling of such fields in this generated utility class RFL.java.\";")
                .append(NEW_LINE);

        r.append("}").append(NEW_LINE).append(NEW_LINE);

        return r;
    }

    /**
     * @param sorts
     * @return All calls to create objects for the given sorts
     */
    private StringBuilder instances(final HashSet<String> sorts) {
        final StringBuilder r = new StringBuilder();
        r.append(NEW_LINE).append("  // ---The methods for object creation---").append(NEW_LINE)
                .append(NEW_LINE);
        for (final String sort : sorts) {
            r.append(newRef(sort));
        }
        r.append(NEW_LINE);
        return r;
    }

    /**
     * @return The call to create an object of given type
     */
    private StringBuilder newRef(final String sort) {
        if (sort.indexOf('[') != -1) {
            return newArray(sort);
        } else {
            return newInstance(sort);
        }
    }

    /**
     * Takes a string representing a type e.g. "java.lang.Object[]" and returns a new name without
     * "." and "[]", e.g. "java_lang_Object_ARRAY_". It is used to create correct setter and getter
     * method names. This method is also used in Assignment.toString(boolean rfl) to generate the
     * correct method names.
     */
    public static String cleanTypeName(String s) {
        // WARNING: Make sure this fixed string begins with a SPACE and also
        // ends with a SPACE.
        if (" jbyte jint jlong jfloat jdouble jboolean jchar jbyte[] jint[] jlong[] jfloat[] jdouble[] jboolean[] jchar[] "
                .contains(" " + s + " ")) {
            s = s.substring(1);
        }
        while (s.contains(".")) {
            s = s.substring(0, s.indexOf('.')) + "_" + s.substring(s.indexOf('.') + 1);
        }
        while (s.contains("[]")) {
            s = s.substring(0, s.indexOf("[]")) + ARRAY + s.substring(s.indexOf("[]") + 2);
        }
        return s;
    }

    /**
     * @param sort
     * @return The call to create an object of given type
     */
    private StringBuilder newInstance(final String sort) {
        final StringBuilder r = new StringBuilder();
        r.append(NEW_LINE);
        r.append("  public static ").append(sort).append(" new").append(cleanTypeName(sort))
                .append("() throws java.lang.RuntimeException {").append(NEW_LINE);
        r.append("    try{").append(NEW_LINE);
        r.append("      return (").append(sort).append(")newInstance(").append(sort)
                .append(".class);").append(NEW_LINE);
        r.append("    } catch (java.lang.Throwable e) {").append(NEW_LINE);
        r.append("       throw new java.lang.RuntimeException(e);").append(NEW_LINE);
        r.append("    }").append(NEW_LINE);
        r.append("  }").append(NEW_LINE);
        return r;
    }

    /**
     * @param sort
     * @return The call to create an Array of given type
     */
    private StringBuilder newArray(final String sort) {
        final StringBuilder r = new StringBuilder();
        r.append(NEW_LINE);
        r.append("  public static ").append(sort).append(" new").append(cleanTypeName(sort))
                .append("(int dim){").append(NEW_LINE);
        r.append("    return new ").append(sort.substring(0, sort.length() - 2)).append("[dim];")
                .append(NEW_LINE);
        r.append("  }").append(NEW_LINE);
        return r;
    }

    private boolean isPrimitiveType(String sort) {
        for (String s : PRIMITIVE_TYPES) {
            if (s.equals(sort)) {
                return true;
            }
        }
        return false;
    }

    private StringBuilder getterAndSetter(final HashSet<String> sorts) {
        final StringBuilder result = new StringBuilder();
        result.append(NEW_LINE).append("  // ---Getter and setter for primitive types---")
                .append(NEW_LINE);
        for (int i = 0; i < 7; i++) {
            result.append(NEW_LINE);
            result.append(declareSetter(PRIMITIVE_TYPES[i], true));
            result.append(declareGetter(PRIMITIVE_TYPES[i], PRIM_TYP_DEF_VAL[i], true));
        }
        result.append(NEW_LINE);
        result.append(NEW_LINE).append("  // ---Getter and setter for Reference types---")
                .append(NEW_LINE);
        for (final String sort : sorts) {
            result.append(NEW_LINE);
            result.append(declareSetter(sort, false));
            result.append(declareGetter(sort, "null", false));
        }
        return result;
    }

    private StringBuilder declareSetter(final String sort, final boolean prim) {
        final StringBuilder r = new StringBuilder();
        final String cmd =
            "      " + (prim
                    ? "f.set" + Character.toUpperCase(sort.charAt(0)) + sort.substring(1)
                        + "(obj, val);" + NEW_LINE
                    : "f.set(obj, val);" + NEW_LINE);
        r.append(NEW_LINE);
        r.append("  public static void " + SET_PREFIX).append(cleanTypeName(sort))
                .append("(Class<?> c, Object obj, String attr, ").append(sort)
                .append(" val) throws RuntimeException{").append(NEW_LINE);
        r.append("    try {").append(NEW_LINE);
        r.append("      java.lang.reflect.Field f = c.getDeclaredField(attr);").append(NEW_LINE);
        r.append("      f.setAccessible(true);").append(NEW_LINE);
        r.append(cmd);
        r.append("    } catch(NoSuchFieldException e) {").append(NEW_LINE);
        r.append("      if(ghostMapActive)").append(NEW_LINE);
        r.append("        ghostModelFields.put(getHash(c,obj,attr), val);").append(NEW_LINE);
        r.append("      else").append(NEW_LINE);
        r.append("        throw new RuntimeException(e.toString() + NoSuchFieldExceptionText);")
                .append(NEW_LINE);
        r.append("    } catch(Exception e) {").append(NEW_LINE);
        r.append("      throw new RuntimeException(e);").append(NEW_LINE);
        r.append("    }").append(NEW_LINE);
        r.append("  }").append(NEW_LINE);
        return r;
    }

    private String primToWrapClass(String sort) {
        if (sort.equals("int")) {
            return "Integer";
        } else if (sort.equals("char")) {
            return "Character";
        } else {
            return Character.toUpperCase(sort.charAt(0)) + sort.substring(1);
        }
    }

    private StringBuilder declareGetter(final String sort, final String def, final boolean prim) {
        final StringBuilder r = new StringBuilder();
        final String cmd =
            "      " + (prim
                    ? "return f.get" + Character.toUpperCase(sort.charAt(0)) + sort.substring(1)
                        + "(obj);" + NEW_LINE
                    : "return (" + sort + ") f.get(obj);" + NEW_LINE);
        r.append(NEW_LINE);
        r.append("  public static ").append(sort).append(" ").append(GET_PREFIX)
                .append(cleanTypeName(sort))
                .append("(Class<?> c, Object obj, String attr) throws RuntimeException{")
                .append(NEW_LINE);
        r.append("    ").append(sort).append(" res = ").append(def).append(";").append(NEW_LINE);
        r.append("    try {").append(NEW_LINE);
        r.append("      java.lang.reflect.Field f = c.getDeclaredField(attr);").append(NEW_LINE);
        r.append("      f.setAccessible(true);").append(NEW_LINE);
        r.append(cmd);
        r.append("      } catch(NoSuchFieldException e) {").append(NEW_LINE);
        r.append("      return (").append(prim ? primToWrapClass(sort) : sort)
                .append(")ghostModelFields.get(getHash(c,obj,attr));").append(NEW_LINE);
        r.append("    } catch(Exception e) {").append(NEW_LINE);
        r.append("      throw new RuntimeException(e);").append(NEW_LINE);
        r.append("    }").append(NEW_LINE);
        r.append("  }").append(NEW_LINE);
        return r;
    }

    /**
     * @return the closing bracket and a newline for the end of the class
     */
    private String footer() {
        return "}" + NEW_LINE;
    }

    /**
     * Sanity check. Checks if number of opening and closing braces is equal.
     */
    private boolean checkBraces(final StringBuilder buf) {
        int curly = 0;
        int round = 0;
        int edged = 0;
        for (int i = 0; i < buf.length(); i++) {
            switch (buf.charAt(i)) {
            case '{' -> curly++;
            case '}' -> curly--;
            case '(' -> round++;
            case ')' -> round--;
            case '[' -> edged++;
            case ']' -> edged--;
            }
        }
        if (curly == 0 && round == 0 && edged == 0) {
            return true;
        } else {
            LOGGER.error("Error braces in RFL.java: curly: {} round: {}: egded: {}", curly, round,
                edged);
            return false;
        }
    }

}
