/* This file is part of KeY - https://key-project.org
 * KeY is licensed under the GNU General Public License Version 2
 * SPDX-License-Identifier: GPL-2.0-only */
package de.uka.ilkd.key.symbolic_execution;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.Map.Entry;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import de.uka.ilkd.key.java.*;
import de.uka.ilkd.key.java.reference.MethodReference;
import de.uka.ilkd.key.java.statement.*;
import de.uka.ilkd.key.logic.PosInOccurrence;
import de.uka.ilkd.key.logic.Term;
import de.uka.ilkd.key.logic.op.IProgramMethod;
import de.uka.ilkd.key.logic.op.IProgramVariable;
import de.uka.ilkd.key.proof.Node;
import de.uka.ilkd.key.proof.NodeInfo;
import de.uka.ilkd.key.proof.Proof;
import de.uka.ilkd.key.proof.init.InitConfig;
import de.uka.ilkd.key.proof.init.ProofInputException;
import de.uka.ilkd.key.rule.RuleApp;
import de.uka.ilkd.key.speclang.BlockContract;
import de.uka.ilkd.key.speclang.Contract;
import de.uka.ilkd.key.speclang.LoopSpecification;
import de.uka.ilkd.key.symbolic_execution.model.*;
import de.uka.ilkd.key.symbolic_execution.model.IExecutionTermination.TerminationKind;
import de.uka.ilkd.key.symbolic_execution.object_model.ISymbolicEquivalenceClass;
import de.uka.ilkd.key.symbolic_execution.object_model.ISymbolicLayout;

import org.key_project.logic.sort.Sort;
import org.key_project.util.collection.ImmutableList;
import org.key_project.util.collection.ImmutableSLList;
import org.key_project.util.collection.Pair;
import org.key_project.util.java.CollectionUtil;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Allows to read XML files which contains an symbolic execution tree written via an
 * {@link ExecutionNodeWriter}.
 *
 * @author Martin Hentschel
 * @see ExecutionNodeWriter
 */
public class ExecutionNodeReader {
    /**
     * Reads the given {@link File}.
     *
     * @param file The {@link File} to read.
     * @return The root of the read symbolic execution tree.
     * @throws ParserConfigurationException Occurred Exception.
     * @throws SAXException Occurred Exception.
     * @throws IOException Occurred Exception.
     */
    public IExecutionNode<?> read(File file)
            throws ParserConfigurationException, SAXException, IOException {
        return read(new FileInputStream(file));
    }

    /**
     * Reads from the given {@link InputStream} and closes it.
     *
     * @param in The {@link InputStream} to read from.
     * @return The root of the read symbolic execution tree.
     * @throws ParserConfigurationException Occurred Exception.
     * @throws SAXException Occurred Exception.
     * @throws IOException Occurred Exception.
     */
    public IExecutionNode<?> read(InputStream in)
            throws ParserConfigurationException, SAXException, IOException {
        if (in != null) {
            try (in) {
                // Parse XML file
                SAXParserFactory factory = SAXParserFactory.newInstance();
                factory.setNamespaceAware(true);
                SAXParser saxParser = factory.newSAXParser();
                SEDSAXHandler handler = new SEDSAXHandler();
                saxParser.parse(in, handler);
                // Get root
                IExecutionNode<?> root = handler.getRoot();
                // Construct call stacks
                Set<Entry<AbstractKeYlessExecutionNode<?>, List<String>>> entries =
                    handler.getCallStackPathEntries().entrySet();
                for (Entry<AbstractKeYlessExecutionNode<?>, List<String>> entry : entries) {
                    for (String path : entry.getValue()) {
                        IExecutionNode<?> stackEntry = findNode(root, path);
                        if (stackEntry == null) {
                            throw new SAXException("Can't find call stack entry \"" + path
                                + "\" in parsed symbolic execution tree.");
                        }
                        entry.getKey().addCallStackEntry(stackEntry);
                    }
                }
                // Construct method returns
                Set<Entry<KeYlessMethodCall, List<String>>> methodReturnEntries =
                    handler.getMethodReturnPathEntries().entrySet();
                for (Entry<KeYlessMethodCall, List<String>> entry : methodReturnEntries) {
                    for (String path : entry.getValue()) {
                        IExecutionNode<?> returnEntry = findNode(root, path);
                        if (returnEntry == null) {
                            throw new SAXException("Can't find method return entry \"" + path
                                + "\" in parsed symbolic execution tree.");
                        }
                        if (!(returnEntry instanceof IExecutionBaseMethodReturn<?>)) {
                            throw new SAXException("Expected basemethod return on \"" + path
                                + "\" but is " + returnEntry.getElementType() + ".");
                        }
                        entry.getKey().addMethodReturn((IExecutionBaseMethodReturn<?>) returnEntry);
                    }
                }
                // Construct completed blocks
                Set<Entry<AbstractKeYlessExecutionNode<?>, List<Pair<String, String>>>> completedBlockEntries =
                    handler.getCompletedBlockEntries().entrySet();
                for (Entry<AbstractKeYlessExecutionNode<?>, List<Pair<String, String>>> entry : completedBlockEntries) {
                    for (Pair<String, String> pair : entry.getValue()) {
                        IExecutionNode<?> returnEntry = findNode(root, pair.first);
                        if (returnEntry == null) {
                            throw new SAXException("Can't find completed block entry \""
                                + pair.first + "\" in parsed symbolic execution tree.");
                        } else if (!(returnEntry instanceof IExecutionBlockStartNode<?>)) {
                            throw new SAXException(
                                "Found completed block entry is not an instance of IExecutionBlockStartNode.");
                        }
                        entry.getKey().addCompletedBlock((IExecutionBlockStartNode<?>) returnEntry,
                            pair.second);
                    }
                }
                // Construct block completions
                Set<Entry<AbstractKeYlessExecutionBlockStartNode<?>, List<String>>> blockCompletionEntries =
                    handler.getBlockCompletionEntries().entrySet();
                for (Entry<AbstractKeYlessExecutionBlockStartNode<?>, List<String>> entry : blockCompletionEntries) {
                    for (String path : entry.getValue()) {
                        IExecutionNode<?> returnEntry = findNode(root, path);
                        if (returnEntry == null) {
                            throw new SAXException("Can't find block completion entry \"" + path
                                + "\" in parsed symbolic execution tree.");
                        }
                        entry.getKey().addBlockCompletion(returnEntry);
                    }
                }
                // Construct links
                Set<Entry<AbstractKeYlessExecutionNode<?>, List<String>>> outgoingLinks =
                    handler.getOutgoingLinks().entrySet();
                for (Entry<AbstractKeYlessExecutionNode<?>, List<String>> entry : outgoingLinks) {
                    for (String path : entry.getValue()) {
                        IExecutionNode<?> target = findNode(root, path);
                        if (target == null) {
                            throw new SAXException("Can't find link targets \"" + path
                                + "\" in parsed symbolic execution tree.");
                        }
                        KeYLessLink link = new KeYLessLink();
                        link.setSource(entry.getKey());
                        link.setTarget(target);
                        entry.getKey().addOutgoingLink(link);
                        ((AbstractKeYlessExecutionNode<?>) target).addIncomingLink(link);
                    }
                }
                // Construct terminations
                Set<Entry<KeYlessStart, List<String>>> terminationEntries =
                    handler.getTerminationPathEntries().entrySet();
                for (Entry<KeYlessStart, List<String>> entry : terminationEntries) {
                    for (String path : entry.getValue()) {
                        IExecutionNode<?> terminationEntry = findNode(root, path);
                        if (terminationEntry == null) {
                            throw new SAXException("Can't find termination entry \"" + path
                                + "\" in parsed symbolic execution tree.");
                        }
                        if (!(terminationEntry instanceof IExecutionTermination)) {
                            throw new SAXException("Expected termination on \"" + path
                                + "\" but is " + terminationEntry.getElementType() + ".");
                        }
                        entry.getKey().addTermination((IExecutionTermination) terminationEntry);
                    }
                }
                // Return result
                return root;
            }
        } else {
            return null;
        }
    }

    /**
     * Searches the {@link IExecutionNode} starting at the given root which is defined by the path.
     *
     * @param root The {@link IExecutionNode} to start search.
     * @param path The path.
     * @return The found {@link IExecutionNode}.
     * @throws SAXException If it was not possible to find the node.
     */
    protected IExecutionNode<?> findNode(IExecutionNode<?> root, String path) throws SAXException {
        if (path != null && !path.isEmpty()) {
            StringTokenizer tokenizer =
                new StringTokenizer(path, ExecutionNodeWriter.PATH_SEPARATOR + "");
            while (tokenizer.hasMoreTokens()) {
                String next = tokenizer.nextToken();
                try {
                    int childIndex = Integer.parseInt(next);
                    if (childIndex < 0) {
                        throw new SAXException("Path segment \"" + next + "\" of path \"" + path
                            + "\" is a negative integer.");
                    }
                    IExecutionNode<?>[] children = root.getChildren();
                    if (childIndex >= children.length) {
                        throw new SAXException("Path segment \"" + next + "\" of path \"" + path
                            + "\" is outside of the child array range.");
                    }
                    root = children[childIndex];
                } catch (NumberFormatException e) {
                    throw new SAXException("Path segment \"" + next + "\" of path \"" + path
                        + "\" is no valid integer.", e);
                }
            }
        }
        return root;
    }

    /**
     * {@link DefaultHandler} implementation used in {@link ExecutionNodeReader#read(InputStream)}.
     *
     * @author Martin Hentschel
     */
    private class SEDSAXHandler extends DefaultHandler {
        /**
         * The root of the read symbolic execution tree.
         */
        private IExecutionNode<?> root;

        /**
         * The parent hierarchy filled by {@link #startElement(String, String, String, Attributes)}
         * and emptied by {@link #endElement(String, String, String)}.
         */
        private final Deque<AbstractKeYlessExecutionNode<?>> parentNodeStack =
            new LinkedList<>();

        /**
         * The parent hierarchy of {@link IExecutionVariable} and {@link IExecutionValue} filled by
         * {@link #startElement(String, String, String, Attributes)} and emptied by
         * {@link #endElement(String, String, String)}.
         */
        private final Deque<Object> parentVariableValueStack = new LinkedList<>();

        /**
         * Maps an {@link AbstractKeYlessExecutionNode} to the path entries of its call stack.
         */
        private final Map<AbstractKeYlessExecutionNode<?>, List<String>> callStackPathEntries =
            new LinkedHashMap<>();

        /**
         * Maps an {@link KeYlessMethodCall} to the path entries of its method returns.
         */
        private final Map<KeYlessMethodCall, List<String>> methodReturnPathEntries =
            new LinkedHashMap<>();

        /**
         * Maps an {@link AbstractKeYlessExecutionNode} to its completed block entries
         */
        private final Map<AbstractKeYlessExecutionNode<?>, List<Pair<String, String>>> completedBlockEntries =
            new LinkedHashMap<>();

        /**
         * Maps an {@link AbstractKeYlessExecutionBlockStartNode} to the path entries of its block
         * completions.
         */
        private final Map<AbstractKeYlessExecutionBlockStartNode<?>, List<String>> blockCompletionEntries =
            new LinkedHashMap<>();

        /**
         * Maps an {@link AbstractKeYlessExecutionNode} to the path entries of its outgoing links.
         */
        private final Map<AbstractKeYlessExecutionNode<?>, List<String>> outgoingLinks =
            new LinkedHashMap<>();

        /**
         * Maps an {@link KeYlessStart} to the path entries of its terminations.
         */
        private final Map<KeYlessStart, List<String>> terminationPathEntries =
            new LinkedHashMap<>();

        /**
         * {@inheritDoc}
         */
        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes)
                throws SAXException {
            AbstractKeYlessExecutionNode<?> parent = parentNodeStack.peekFirst();
            if (isConstraint(uri, localName, qName)) {
                Object parentValue = parentVariableValueStack.peekFirst();
                if (parentValue != null) {
                    if (!(parentValue instanceof KeYlessValue)) {
                        throw new SAXException("Can't add constraint to variable.");
                    }
                    KeYlessConstraint constraint = new KeYlessConstraint(getName(attributes));
                    ((KeYlessValue) parentValue).addConstraint(constraint);
                } else {
                    if (!(parent instanceof AbstractKeYlessExecutionNode)) {
                        throw new SAXException("Can't add constraint to non execution node.");
                    }
                    KeYlessConstraint constraint = new KeYlessConstraint(getName(attributes));
                    parent.addConstraint(constraint);
                }
            } else if (isCallStateVariable(uri, localName, qName)) {
                Object parentValue = parentVariableValueStack.peekFirst();
                KeYlessVariable variable = createVariable(
                    parentValue instanceof KeYlessValue ? (KeYlessValue) parentValue : null, uri,
                    localName, qName, attributes);
                if (parentValue != null) {
                    throw new SAXException(
                        "Can't add initial state variable to parent variable or value.");
                }
                if (parent instanceof AbstractKeYlessBaseExecutionNode<?>) {
                    ((AbstractKeYlessBaseExecutionNode<?>) parent).addCallStateVariable(variable);
                } else {
                    throw new SAXException(
                        "Can't add call state variable to parent execution node.");
                }
                parentVariableValueStack.addFirst(variable);
            } else if (isVariable(uri, localName, qName)) {
                Object parentValue = parentVariableValueStack.peekFirst();
                KeYlessVariable variable = createVariable(
                    parentValue instanceof KeYlessValue ? (KeYlessValue) parentValue : null, uri,
                    localName, qName, attributes);
                if (parentValue != null) {
                    ((KeYlessValue) parentValue).addChildVariable(variable);
                } else {
                    if (parent != null) {
                        parent.addVariable(variable);
                    } else {
                        throw new SAXException("Can't add variable to parent execution node.");
                    }
                }
                parentVariableValueStack.addFirst(variable);
            } else if (isValue(uri, localName, qName)) {
                Object parentValue = parentVariableValueStack.peekFirst();
                if (!(parentValue instanceof KeYlessVariable)) {
                    throw new SAXException("Can't add value to parent variable.");
                }
                KeYlessValue value =
                    createValue((KeYlessVariable) parentValue, uri, localName, qName, attributes);
                ((KeYlessVariable) parentValue).addValue(value);
                parentVariableValueStack.addFirst(value);
            } else if (isCallStackEntry(uri, localName, qName)) {
                List<String> callStackEntries =
                    callStackPathEntries.computeIfAbsent(parent, k -> new LinkedList<>());
                callStackEntries.add(getPathInTree(attributes));
            } else if (isMethodReturnEntry(uri, localName, qName)) {
                List<String> methodReturnEntries = methodReturnPathEntries.get(parent);
                if (methodReturnEntries == null) {
                    methodReturnEntries = new LinkedList<>();
                    methodReturnPathEntries.put((KeYlessMethodCall) parent, methodReturnEntries);
                }
                methodReturnEntries.add(0, getPathInTree(attributes));
            } else if (isCompletedBlockEntry(uri, localName, qName)) {
                List<Pair<String, String>> completedBlocks =
                    completedBlockEntries.computeIfAbsent(parent, k -> new LinkedList<>());
                completedBlocks.add(new Pair<>(getPathInTree(attributes),
                    getConditionString(attributes)));
            } else if (isBlockCompletionEntry(uri, localName, qName)) {
                List<String> blockCompletionPathEntries = blockCompletionEntries.get(parent);
                if (blockCompletionPathEntries == null) {
                    blockCompletionPathEntries = new LinkedList<>();
                    blockCompletionEntries.put((AbstractKeYlessExecutionBlockStartNode<?>) parent,
                        blockCompletionPathEntries);
                }
                blockCompletionPathEntries.add(getPathInTree(attributes));
            } else if (isOutgoingLink(uri, localName, qName)) {
                List<String> linkPaths =
                    outgoingLinks.computeIfAbsent(parent, k -> new LinkedList<>());
                linkPaths.add(getPathInTree(attributes));
            } else if (isTerminationEntry(uri, localName, qName)) {
                List<String> terminationEntries = terminationPathEntries.get(parent);
                if (terminationEntries == null) {
                    terminationEntries = new LinkedList<>();
                    terminationPathEntries.put((KeYlessStart) parent, terminationEntries);
                }
                terminationEntries.add(0, getPathInTree(attributes));
            } else if (isMethodReturnValue(uri, localName, qName)) {
                Object parentValue = parentNodeStack.peekFirst();
                if (!(parentValue instanceof KeYlessMethodReturn)) {
                    throw new SAXException("Can't add method return value to parent.");
                }
                KeYlessMethodReturnValue returnValue =
                    createMethodReturnValue(uri, localName, qName, attributes);
                ((KeYlessMethodReturn) parentValue).addReturnValue(returnValue);
            } else {
                AbstractKeYlessExecutionNode<?> child =
                    createExecutionNode(parent, uri, localName, qName, attributes);
                if (root == null) {
                    root = child;
                }
                if (parent != null) {
                    parent.addChild(child);
                }
                parentNodeStack.addFirst(child);
            }
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public void endElement(String uri, String localName, String qName) {
            if (isConstraint(uri, localName, qName)) {
                // Nothing to do.
            } else if (isCallStateVariable(uri, localName, qName)) {
                parentVariableValueStack.removeFirst();
            } else if (isVariable(uri, localName, qName)) {
                parentVariableValueStack.removeFirst();
            } else if (isValue(uri, localName, qName)) {
                parentVariableValueStack.removeFirst();
            } else if (isCallStackEntry(uri, localName, qName)) {
                // Nothing to do.
            } else if (isMethodReturnEntry(uri, localName, qName)) {
                // Nothing to do.
            } else if (isCompletedBlockEntry(uri, localName, qName)) {
                // Nothing to do.
            } else if (isBlockCompletionEntry(uri, localName, qName)) {
                // Nothing to do.
            } else if (isOutgoingLink(uri, localName, qName)) {
                // Nothing to do.
            } else if (isTerminationEntry(uri, localName, qName)) {
                // Nothing to do.
            } else if (isMethodReturnValue(uri, localName, qName)) {
                // Nothing to do.
            } else {
                parentNodeStack.removeFirst();
            }
        }

        /**
         * Returns the root of the read symbolic execution tree.
         *
         * @return The root of the read symbolic execution tree.
         */
        public IExecutionNode<?> getRoot() {
            return root;
        }

        /**
         * Returns the mapping of an {@link AbstractKeYlessExecutionNode} to its call stack entries.
         *
         * @return The mapping of an {@link AbstractKeYlessExecutionNode} to its call stack entries.
         */
        public Map<AbstractKeYlessExecutionNode<?>, List<String>> getCallStackPathEntries() {
            return callStackPathEntries;
        }

        /**
         * Returns the mapping of a {@link KeYlessMethodCall} to its method return entries.
         *
         * @return The mapping of a {@link KeYlessMethodCall} to its method return entries.
         */
        public Map<KeYlessMethodCall, List<String>> getMethodReturnPathEntries() {
            return methodReturnPathEntries;
        }

        /**
         * Returns the mapping of {@link AbstractKeYlessExecutionNode} to its completed block
         * entries.
         *
         * @return The mapping of {@link AbstractKeYlessExecutionNode} to its completed block
         *         entries.
         */
        public Map<AbstractKeYlessExecutionNode<?>, List<Pair<String, String>>> getCompletedBlockEntries() {
            return completedBlockEntries;
        }

        /**
         * Returns the mapping of an {@link AbstractKeYlessExecutionBlockStartNode} to its block
         * completion entries.
         *
         * @return The mapping of an {@link AbstractKeYlessExecutionBlockStartNode} to its block
         *         completion entries.
         */
        public Map<AbstractKeYlessExecutionBlockStartNode<?>, List<String>> getBlockCompletionEntries() {
            return blockCompletionEntries;
        }

        /**
         * Returns the mapping of a {@link KeYlessStart} to its termination entries.
         *
         * @return The mapping of a {@link KeYlessStart} to its termination entries.
         */
        public Map<KeYlessStart, List<String>> getTerminationPathEntries() {
            return terminationPathEntries;
        }

        /**
         * Returns the mapping of an {@link AbstractKeYlessExecutionNode} to its outgoing links.
         *
         * @return The mapping of an {@link AbstractKeYlessExecutionNode} to its outgoing links.
         */
        public Map<AbstractKeYlessExecutionNode<?>, List<String>> getOutgoingLinks() {
            return outgoingLinks;
        }
    }

    /**
     * Checks if the currently parsed tag represents an {@link IExecutionConstraint}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents an {@link IExecutionConstraint}, {@code false} is something
     *         else.
     */
    protected boolean isConstraint(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_CONSTRAINT.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an {@link IExecutionVariable}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents an {@link IExecutionVariable}, {@code false} is something
     *         else.
     */
    protected boolean isVariable(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_VARIABLE.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an {@link IExecutionVariable}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents an {@link IExecutionVariable}, {@code false} is something
     *         else.
     */
    protected boolean isCallStateVariable(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_CALL_STATE_VARIABLE.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an {@link IExecutionMethodReturnValue}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents an {@link IExecutionMethodReturnValue}, {@code false} is
     *         something else.
     */
    protected boolean isMethodReturnValue(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_METHOD_RETURN_VALUE.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an {@link IExecutionValue}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents an {@link IExecutionValue}, {@code false} is something else.
     */
    protected boolean isValue(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_VALUE.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an entry of
     * {@link IExecutionNode#getCallStack()}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents call stack entry, {@code false} is something else.
     */
    protected boolean isCallStackEntry(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_CALL_STACK_ENTRY.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an entry of
     * {@link IExecutionMethodCall#getMethodReturns()}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents method return entry, {@code false} is something else.
     */
    protected boolean isMethodReturnEntry(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_METHOD_RETURN_ENTRY.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an entry of
     * {@link IExecutionNode#getCompletedBlocks()}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents completed block entry, {@code false} is something else.
     */
    protected boolean isCompletedBlockEntry(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_COMPLETED_BLOCK_ENTRY.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an entry of
     * {@link IExecutionNode#getOutgoingLinks()}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents block completion entry, {@code false} is something else.
     */
    protected boolean isOutgoingLink(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_OUTGOING_LINK.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an entry of
     * {@link IExecutionBranchStatement#getBlockCompletions()}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents block completion entry, {@code false} is something else.
     */
    protected boolean isBlockCompletionEntry(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_BLOCK_COMPLETION_ENTRY.equals(qName);
    }

    /**
     * Checks if the currently parsed tag represents an entry of
     * {@link IExecutionStart#getTerminations()}.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @return {@code true} represents termination entry, {@code false} is something else.
     */
    protected boolean isTerminationEntry(String uri, String localName, String qName) {
        return ExecutionNodeWriter.TAG_TERMINATION_ENTRY.equals(qName);
    }

    /**
     * Creates a new {@link IExecutionVariable} with the given content.
     *
     * @param parentValue The parent {@link IExecutionValue}.
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @param attributes The attributes.
     * @return The created {@link IExecutionVariable}.
     */
    protected KeYlessVariable createVariable(IExecutionValue parentValue, String uri,
            String localName, String qName, Attributes attributes) {
        return new KeYlessVariable(parentValue, isArrayIndex(attributes),
            getArrayIndexString(attributes), getName(attributes));
    }

    /**
     * Creates a new {@link IExecutionMethodReturnValue} with the given content.
     *
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @param attributes The attributes.
     * @return The created {@link IExecutionMethodReturnValue}.
     */
    public KeYlessMethodReturnValue createMethodReturnValue(String uri, String localName,
            String qName, Attributes attributes) {
        return new KeYlessMethodReturnValue(getName(attributes), getReturnValueString(attributes),
            getHasCondition(attributes), getConditionString(attributes));
    }

    /**
     * Creates a new {@link IExecutionValue} with the given content.
     *
     * @param parentVariable The parent {@link IExecutionVariable}.
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @param attributes The attributes.
     * @return The created {@link IExecutionValue}.
     */
    protected KeYlessValue createValue(IExecutionVariable parentVariable, String uri,
            String localName, String qName, Attributes attributes) {
        return new KeYlessValue(parentVariable, getTypeString(attributes),
            getValueString(attributes), getName(attributes), isValueUnknown(attributes),
            isValueAnObject(attributes), getConditionString(attributes));
    }

    /**
     * Creates a new {@link IExecutionNode} with the given content.
     *
     * @param parent The parent {@link IExecutionNode}.
     * @param uri The URI.
     * @param localName THe local name.
     * @param qName The qName.
     * @param attributes The attributes.
     * @return The created {@link IExecutionNode}.
     * @throws SAXException Occurred Exception.
     */
    protected AbstractKeYlessExecutionNode<?> createExecutionNode(IExecutionNode<?> parent,
            String uri, String localName, String qName, Attributes attributes) throws SAXException {
        if (ExecutionNodeWriter.TAG_BRANCH_CONDITION.equals(qName)) {
            return new KeYlessBranchCondition(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                getBranchCondition(attributes), isMergedBranchCondition(attributes),
                isBranchConditionComputed(attributes), getAdditionalBranchLabel(attributes));
        } else if (ExecutionNodeWriter.TAG_BRANCH_STATEMENT.equals(qName)) {
            return new KeYlessBranchStatement(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                isBlockOpened(attributes));
        } else if (ExecutionNodeWriter.TAG_LOOP_CONDITION.equals(qName)) {
            return new KeYlessLoopCondition(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                isBlockOpened(attributes));
        } else if (ExecutionNodeWriter.TAG_LOOP_STATEMENT.equals(qName)) {
            return new KeYlessLoopStatement(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                isBlockOpened(attributes));
        } else if (ExecutionNodeWriter.TAG_METHOD_CALL.equals(qName)) {
            return new KeYlessMethodCall(parent, getName(attributes), getPathCondition(attributes),
                isPathConditionChanged(attributes));
        } else if (ExecutionNodeWriter.TAG_METHOD_RETURN.equals(qName)) {
            return new KeYlessMethodReturn(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                getNameIncludingReturnValue(attributes), getSignature(attributes),
                getSignatureIncludingReturnValue(attributes), isReturnValueComputed(attributes),
                getMethodReturnCondition(attributes));
        } else if (ExecutionNodeWriter.TAG_EXCEPTIONAL_METHOD_RETURN.equals(qName)) {
            return new KeYlessExceptionalMethodReturn(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                getSignature(attributes), getMethodReturnCondition(attributes));
        } else if (ExecutionNodeWriter.TAG_START.equals(qName)) {
            return new KeYlessStart(getName(attributes), getPathCondition(attributes),
                isPathConditionChanged(attributes));
        } else if (ExecutionNodeWriter.TAG_STATEMENT.equals(qName)) {
            return new KeYlessStatement(parent, getName(attributes), getPathCondition(attributes),
                isPathConditionChanged(attributes));
        } else if (ExecutionNodeWriter.TAG_TERMINATION.equals(qName)) {
            return new KeYlessTermination(parent, getName(attributes), getPathCondition(attributes),
                isPathConditionChanged(attributes), getTerminationKind(attributes),
                getBranchVerified(attributes));
        } else if (ExecutionNodeWriter.TAG_OPERATION_CONTRACT.equals(qName)) {
            return new KeYlessOperationContract(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                isPreconditionComplied(attributes), isHasNotNullCheck(attributes),
                isNotNullCheckComplied(attributes), getResultTerm(attributes),
                getExceptionTerm(attributes), getSelfTerm(attributes),
                getContractParameters(attributes));
        } else if (ExecutionNodeWriter.TAG_LOOP_INVARIANT.equals(qName)) {
            return new KeYlessLoopInvariant(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                isInitiallyValid(attributes));
        } else if (ExecutionNodeWriter.TAG_BLOCK_CONTRACT.equals(qName)) {
            return new KeYlessBlockContract(parent, getName(attributes),
                getPathCondition(attributes), isPathConditionChanged(attributes),
                isPreconditionComplied(attributes));
        } else if (ExecutionNodeWriter.TAG_JOIN.equals(qName)) {
            return new KeYlessJoin(parent, getName(attributes), getPathCondition(attributes),
                isPathConditionChanged(attributes), isWeakeningVerified(attributes));
        } else {
            throw new SAXException("Unknown tag \"" + qName + "\".");
        }
    }

    /**
     * Returns the additional branch label value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getAdditionalBranchLabel(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_ADDITIONAL_BRANCH_LABEL);
    }

    /**
     * Returns the path in tree value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getPathInTree(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_PATH_IN_TREE);
    }

    /**
     * Returns the name value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getName(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_NAME);
    }

    /**
     * Returns the name value including return value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getNameIncludingReturnValue(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_NAME_INCLUDING_RETURN_VALUE);
    }

    /**
     * Returns the signature value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getSignature(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_SIGNATURE);
    }

    /**
     * Returns the signature value including return value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getSignatureIncludingReturnValue(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_SIGNATURE_INCLUDING_RETURN_VALUE);
    }

    /**
     * Returns the termination kind value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected TerminationKind getTerminationKind(Attributes attributes) {
        return TerminationKind
                .valueOf(attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_TERMINATION_KIND));
    }

    /**
     * Returns the precondition complied value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isPreconditionComplied(Attributes attributes) {
        return Boolean.parseBoolean(
            attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_PRECONDITION_COMPLIED));
    }

    /**
     * Returns the has not null check value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isHasNotNullCheck(Attributes attributes) {
        return Boolean.parseBoolean(
            attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_HAS_NOT_NULL_CHECK));
    }

    /**
     * Returns the block opened value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isBlockOpened(Attributes attributes) {
        return Boolean
                .parseBoolean(attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_BLOCK_OPENED));
    }

    /**
     * Returns the is return value computed value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isReturnValueComputed(Attributes attributes) {
        return Boolean.parseBoolean(
            attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_RETURN_VALUE_COMPUTED));
    }

    /**
     * Returns the is branch condition computed value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isBranchConditionComputed(Attributes attributes) {
        return Boolean.parseBoolean(
            attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_BRANCH_CONDITION_COMPUTED));
    }

    /**
     * Returns the not null check complied value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isNotNullCheckComplied(Attributes attributes) {
        return Boolean.parseBoolean(
            attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_NOT_NULL_CHECK_COMPLIED));
    }

    /**
     * Returns the initially valid value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isInitiallyValid(Attributes attributes) {
        return Boolean
                .parseBoolean(attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_INITIALLY_VALID));
    }

    /**
     * Returns the is value an object value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isValueAnObject(Attributes attributes) {
        return Boolean.parseBoolean(
            attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_IS_VALUE_AN_OBJECT));
    }

    /**
     * Returns if the weakening is verified.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isWeakeningVerified(Attributes attributes) {
        return Boolean.parseBoolean(
            attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_WEAKENING_VERIFIED));
    }

    /**
     * Returns the is value unknown value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isValueUnknown(Attributes attributes) {
        return Boolean
                .parseBoolean(attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_IS_VALUE_UNKNOWN));
    }

    /**
     * Returns the value string value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getValueString(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_VALUE_STRING);
    }

    /**
     * Returns the value condition string value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getConditionString(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_CONDITION_STRING);
    }

    /**
     * Returns the is has condition value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean getHasCondition(Attributes attributes) {
        return Boolean
                .parseBoolean(attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_HAS_CONDITION));
    }

    /**
     * Returns the is branch verified value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean getBranchVerified(Attributes attributes) {
        return Boolean
                .parseBoolean(attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_BRANCH_VERIFIED));
    }

    /**
     * Returns the return value string value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getReturnValueString(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_RETURN_VALUE_STRING);
    }

    /**
     * Returns the type string value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getTypeString(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_TYPE_STRING);
    }

    /**
     * Returns the exception term value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getExceptionTerm(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_EXCEPTION_TERM);
    }

    /**
     * Returns the result term value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getResultTerm(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_RESULT_TERM);
    }

    /**
     * Returns the self term value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getSelfTerm(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_SELF_TERM);
    }

    /**
     * Returns the contract parameters value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getContractParameters(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_CONTRACT_PARAMETERS);
    }

    /**
     * Returns the array index value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getArrayIndexString(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_ARRAY_INDEX);
    }

    /**
     * Returns the is array index value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isArrayIndex(Attributes attributes) {
        return Boolean
                .parseBoolean(attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_IS_ARRAY_INDEX));
    }

    /**
     * Returns the branch condition value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getBranchCondition(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_BRANCH_CONDITION);
    }

    /**
     * Returns the path condition value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getPathCondition(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_PATH_CONDITION);
    }

    /**
     * Returns the method return condition value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected String getMethodReturnCondition(Attributes attributes) {
        return attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_METHOD_RETURN_CONDITION);
    }

    /**
     * Returns the path condition changed value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isPathConditionChanged(Attributes attributes) {
        return Boolean
                .parseBoolean(
                    attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_PATH_CONDITION_CHANGED));
    }

    /**
     * Returns the merged branch condition value.
     *
     * @param attributes The {@link Attributes} which provides the content.
     * @return The value.
     */
    protected boolean isMergedBranchCondition(Attributes attributes) {
        return Boolean.parseBoolean(
            attributes.getValue(ExecutionNodeWriter.ATTRIBUTE_MERGED_BRANCH_CONDITION));
    }

    /**
     * An abstract implementation of {@link IExecutionElement} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static abstract class AbstractKeYlessExecutionElement implements IExecutionElement {
        /**
         * The name.
         */
        private final String name;

        /**
         * Constructor.
         *
         * @param name The name of this node.
         */
        public AbstractKeYlessExecutionElement(String name) {
            this.name = name;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Services getServices() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public InitConfig getInitConfig() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Proof getProof() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public RuleApp getAppliedRuleApp() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Node getProofNode() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public NodeInfo getProofNodeInfo() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getName() {
            return name;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String toString() {
            return getElementType() + " " + getName();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isDisposed() {
            return false;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ITreeSettings getSettings() {
            return null;
        }
    }

    /**
     * An abstract implementation of {@link IExecutionNode} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static abstract class AbstractKeYlessExecutionNode<S extends SourceElement>
            extends AbstractKeYlessExecutionElement implements IExecutionNode<S> {
        /**
         * The parent {@link IExecutionNode}.
         */
        private final IExecutionNode<?> parent;

        /**
         * The children.
         */
        private final List<IExecutionNode<?>> children = new LinkedList<>();

        /**
         * The formated path condition.
         */
        private final String formatedPathCondition;

        /**
         * Is the path condition changed compared to parent?
         */
        private final boolean pathConditionChanged;

        /**
         * The call stack.
         */
        private final List<IExecutionNode<?>> callStack = new LinkedList<>();

        /**
         * The contained constraints.
         */
        private final List<IExecutionConstraint> constraints =
            new LinkedList<>();

        /**
         * The contained variables.
         */
        private final List<IExecutionVariable> variables = new LinkedList<>();

        /**
         * The completed blocks.
         */
        private ImmutableList<IExecutionBlockStartNode<?>> completedBlocks = ImmutableSLList.nil();

        /**
         * The formated conditions under which a block is completed.
         */
        private final Map<IExecutionBlockStartNode<?>, String> formatedCompletedBlockConditions =
            new LinkedHashMap<>();

        /**
         * The contained outgoing links.
         */
        private ImmutableList<IExecutionLink> outgoingLinks = ImmutableSLList.nil();

        /**
         * The contained incoming links.
         */
        private ImmutableList<IExecutionLink> incomingLinks = ImmutableSLList.nil();

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         */
        public AbstractKeYlessExecutionNode(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged) {
            super(name);
            this.parent = parent;
            this.formatedPathCondition = formatedPathCondition;
            this.pathConditionChanged = pathConditionChanged;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionNode<?> getParent() {
            return parent;
        }

        /**
         * Adds the given child.
         *
         * @param child The child to add.
         */
        public void addChild(IExecutionNode<?> child) {
            children.add(child);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionNode<?>[] getChildren() {
            return children.toArray(new IExecutionNode[0]);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getPathCondition() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getFormatedPathCondition() throws ProofInputException {
            return formatedPathCondition;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isPathConditionChanged() {
            return pathConditionChanged;
        }

        /**
         * Adds the given entry to the call stack.
         *
         * @param entry The entry to add to the call stack.
         */
        public void addCallStackEntry(IExecutionNode<?> entry) {
            callStack.add(entry);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionNode<?>[] getCallStack() {
            return callStack.isEmpty() ? null
                    : callStack.toArray(new IExecutionNode[0]);
        }

        /**
         * Adds the given {@link IExecutionConstraint}.
         *
         * @param constraint The {@link IExecutionConstraint} to add.
         */
        public void addConstraint(IExecutionConstraint constraint) {
            constraints.add(constraint);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionConstraint[] getConstraints() {
            return constraints.toArray(new IExecutionConstraint[0]);
        }

        /**
         * Adds the given {@link IExecutionVariable}.
         *
         * @param variable The {@link IExecutionVariable} to add.
         */
        public void addVariable(IExecutionVariable variable) {
            variables.add(variable);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public S getActiveStatement() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public PositionInfo getActivePositionInfo() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionVariable[] getVariables() {
            return variables.toArray(new IExecutionVariable[0]);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionVariable[] getVariables(Term condition) {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public int getLayoutsCount() throws ProofInputException {
            return 0;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ISymbolicLayout getInitialLayout(int configurationIndex) throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ISymbolicLayout getCurrentLayout(int configurationIndex) throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ImmutableList<ISymbolicEquivalenceClass> getLayoutsEquivalenceClasses(
                int configurationIndex) throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public PosInOccurrence getModalityPIO() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ImmutableList<IExecutionBlockStartNode<?>> getCompletedBlocks() {
            return completedBlocks;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getBlockCompletionCondition(IExecutionBlockStartNode<?> completedNode)
                throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getFormatedBlockCompletionCondition(IExecutionBlockStartNode<?> completedNode)
                throws ProofInputException {
            return formatedCompletedBlockConditions.get(completedNode);
        }

        /**
         * Adds the given completed block.
         *
         * @param completedBlock The completed block.
         * @param formatedCondition The formated condition under which the block is completed.
         */
        public void addCompletedBlock(IExecutionBlockStartNode<?> completedBlock,
                String formatedCondition) {
            if (completedBlock != null) {
                completedBlocks = completedBlocks.append(completedBlock);
                formatedCompletedBlockConditions.put(completedBlock, formatedCondition);
            }
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionLink getOutgoingLink(final IExecutionNode<?> target) {
            return CollectionUtil.search(outgoingLinks, element -> element.getTarget() == target);
        }

        /**
         * Adds the outgoing {@link IExecutionLink}.
         *
         * @param link The outgoing {@link IExecutionLink} to add.
         */
        public void addOutgoingLink(IExecutionLink link) {
            outgoingLinks = outgoingLinks.append(link);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ImmutableList<IExecutionLink> getOutgoingLinks() {
            return outgoingLinks;
        }

        /**
         * Adds the incoming {@link IExecutionLink}.
         *
         * @param link The incoming {@link IExecutionLink} to add.
         */
        public void addIncomingLink(IExecutionLink link) {
            incomingLinks = incomingLinks.append(link);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionLink getIncomingLink(final IExecutionNode<?> source) {
            return CollectionUtil.search(incomingLinks, element -> element.getSource() == source);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ImmutableList<IExecutionLink> getIncomingLinks() {
            return incomingLinks;
        }
    }

    /**
     * An abstract implementation of {@link IExecutionBlockStartNode} which is independent from KeY
     * and provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static abstract class AbstractKeYlessExecutionBlockStartNode<S extends SourceElement>
            extends AbstractKeYlessExecutionNode<S> implements IExecutionBlockStartNode<S> {
        /**
         * The block completions.
         */
        private ImmutableList<IExecutionNode<?>> blockCompletions = ImmutableSLList.nil();

        /**
         * Is a block opened?
         */
        private final boolean blockOpened;

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param blockOpened {@code false} block is definitively not opened, {@code true} block is
         *        or might be opened.
         */
        public AbstractKeYlessExecutionBlockStartNode(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged, boolean blockOpened) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
            this.blockOpened = blockOpened;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ImmutableList<IExecutionNode<?>> getBlockCompletions() {
            return blockCompletions;
        }

        /**
         * Adds the given block completion.
         *
         * @param blockCompletion The block completion to add.
         */
        public void addBlockCompletion(IExecutionNode<?> blockCompletion) {
            if (blockCompletion != null) {
                blockCompletions = blockCompletions.append(blockCompletion);
            }
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isBlockOpened() {
            return blockOpened;
        }
    }

    /**
     * An implementation of {@link IExecutionLoopCondition} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessBranchCondition extends AbstractKeYlessExecutionNode<SourceElement>
            implements IExecutionBranchCondition {
        /**
         * The formated branch condition.
         */
        private final String formatedBranchCondition;

        /**
         * Merged branch condition?
         */
        private final boolean mergedBranchCondition;

        /**
         * Indicates if branch condition is computed or not.
         */
        private final boolean branchConditionComputed;

        /**
         * The optional additional branch label.
         */
        private final String additionalBranchLabel;

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param formatedBranchCondition The formated branch condition.
         * @param mergedBranchCondition Merged branch condition?
         * @param branchConditionComputed Is branch condition computed?
         * @param additionalBranchLabel The optional additional branch label.
         */
        public KeYlessBranchCondition(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged,
                String formatedBranchCondition, boolean mergedBranchCondition,
                boolean branchConditionComputed, String additionalBranchLabel) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
            this.formatedBranchCondition = formatedBranchCondition;
            this.mergedBranchCondition = mergedBranchCondition;
            this.branchConditionComputed = branchConditionComputed;
            this.additionalBranchLabel = additionalBranchLabel;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Branch Condition";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getBranchCondition() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getFormatedBranchCondition() {
            return formatedBranchCondition;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isMergedBranchCondition() {
            return mergedBranchCondition;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Node[] getMergedProofNodes() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term[] getMergedBranchCondtions() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isBranchConditionComputed() {
            return branchConditionComputed;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getAdditionalBranchLabel() {
            return additionalBranchLabel;
        }
    }

    /**
     * An implementation of {@link IExecutionStart} which is independent from KeY and provides such
     * only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessStart extends AbstractKeYlessExecutionNode<SourceElement>
            implements IExecutionStart {
        /**
         * The up to now discovered {@link IExecutionTermination}s.
         */
        private ImmutableList<IExecutionTermination> terminations = ImmutableSLList.nil();

        /**
         * Constructor.
         *
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         */
        public KeYlessStart(String name, String formatedPathCondition,
                boolean pathConditionChanged) {
            super(null, name, formatedPathCondition, pathConditionChanged);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Start";
        }

        /**
         * Adds the given {@link IExecutionTermination}.
         *
         * @param termination The {@link IExecutionTermination} to add.
         */
        public void addTermination(IExecutionTermination termination) {
            if (termination != null) {
                terminations = terminations.prepend(termination);
            }
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ImmutableList<IExecutionTermination> getTerminations() {
            return terminations;
        }
    }

    /**
     * An implementation of {@link IExecutionTermination} which is independent from KeY and provides
     * such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessTermination extends AbstractKeYlessExecutionNode<SourceElement>
            implements IExecutionTermination {
        /**
         * The {@link TerminationKind}.
         */
        private final TerminationKind terminationKind;

        /**
         * The branch verified flag.
         */
        private final boolean branchVerified;

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param terminationKind kind of termination
         * @param branchVerified The branch verified flag.
         */
        public KeYlessTermination(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged,
                TerminationKind terminationKind, boolean branchVerified) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
            this.terminationKind = terminationKind;
            this.branchVerified = branchVerified;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IProgramVariable getExceptionVariable() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Sort getExceptionSort() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            switch (getTerminationKind()) {
            case EXCEPTIONAL:
                return "Exceptional Termination";
            case LOOP_BODY:
                return "Loop Body Termination";
            case BLOCK_CONTRACT_EXCEPTIONAL:
                return "Block Contract Exceptional Termination";
            case BLOCK_CONTRACT_NORMAL:
                return "Block Contract Termination";
            default:
                return "Termination";
            }
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public TerminationKind getTerminationKind() {
            return terminationKind;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isBranchVerified() {
            return branchVerified;
        }
    }

    /**
     * An implementation of {@link IExecutionBranchStatement} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessBranchStatement
            extends AbstractKeYlessExecutionBlockStartNode<BranchStatement>
            implements IExecutionBranchStatement {
        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param blockOpened {@code false} block is definitively not opened, {@code true} block is
         *        or might be opened.
         */
        public KeYlessBranchStatement(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged, boolean blockOpened) {
            super(parent, name, formatedPathCondition, pathConditionChanged, blockOpened);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Branch Statement";
        }
    }

    /**
     * An implementation of {@link IExecutionLoopCondition} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessLoopCondition
            extends AbstractKeYlessExecutionBlockStartNode<JavaStatement>
            implements IExecutionLoopCondition {
        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param blockOpened {@code false} block is definitively not opened, {@code true} block is
         *        or might be opened.
         */
        public KeYlessLoopCondition(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged, boolean blockOpened) {
            super(parent, name, formatedPathCondition, pathConditionChanged, blockOpened);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Expression getGuardExpression() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public PositionInfo getGuardExpressionPositionInfo() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Loop Condition";
        }
    }

    /**
     * An implementation of {@link IExecutionLoopStatement} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessLoopStatement
            extends AbstractKeYlessExecutionBlockStartNode<LoopStatement>
            implements IExecutionLoopStatement {
        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param blockOpened {@code false} block is definitively not opened, {@code true} block is
         *        or might be opened.
         */
        public KeYlessLoopStatement(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged, boolean blockOpened) {
            super(parent, name, formatedPathCondition, pathConditionChanged, blockOpened);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Loop Statement";
        }
    }

    /**
     * An implementation of {@link IExecutionMethodCall} which is independent from KeY and provides
     * such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessMethodCall extends AbstractKeYlessExecutionNode<MethodBodyStatement>
            implements IExecutionMethodCall {
        /**
         * The up to now discovered {@link IExecutionBaseMethodReturn<?>}s.
         */
        private ImmutableList<IExecutionBaseMethodReturn<?>> methodReturns = ImmutableSLList.nil();

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         */
        public KeYlessMethodCall(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public MethodReference getMethodReference() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IProgramMethod getProgramMethod() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Method Call";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isImplicitConstructor() {
            return false;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public MethodReference getExplicitConstructorMethodReference() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IProgramMethod getExplicitConstructorProgramMethod() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ImmutableList<IExecutionBaseMethodReturn<?>> getMethodReturns() {
            return methodReturns;
        }

        /**
         * Adds the given {@link IExecutionBaseMethodReturn<?>}.
         *
         * @param methodReturn The {@link IExecutionBaseMethodReturn<?>} to add.
         */
        public void addMethodReturn(IExecutionBaseMethodReturn<?> methodReturn) {
            if (methodReturn != null) {
                methodReturns = methodReturns.prepend(methodReturn);
            }
        }
    }

    /**
     * An implementation of {@link IExecutionBaseMethodReturn} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static abstract class AbstractKeYlessBaseExecutionNode<S extends SourceElement>
            extends AbstractKeYlessExecutionNode<S> implements IExecutionBaseMethodReturn<S> {
        /**
         * The contained call state variables.
         */
        private final List<IExecutionVariable> callStateVariables =
            new LinkedList<>();

        /**
         * The signature.
         */
        private final String signature;

        /**
         * The formated method return condition.
         */
        private final String formatedMethodReturn;

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param signature The signature.
         * @param formatedMethodReturn The formated method return condition.
         */
        public AbstractKeYlessBaseExecutionNode(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged, String signature,
                String formatedMethodReturn) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
            this.signature = signature;
            this.formatedMethodReturn = formatedMethodReturn;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionVariable[] getCallStateVariables() {
            return callStateVariables.toArray(new IExecutionVariable[0]);
        }

        /**
         * Adds the given {@link IExecutionVariable}.
         *
         * @param variable The {@link IExecutionVariable} to add.
         */
        public void addCallStateVariable(IExecutionVariable variable) {
            callStateVariables.add(variable);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionMethodCall getMethodCall() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getSignature() throws ProofInputException {
            return signature;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getMethodReturnCondition() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getFormattedMethodReturnCondition() throws ProofInputException {
            return formatedMethodReturn;
        }
    }

    /**
     * An implementation of {@link IExecutionExceptionalMethodReturn} which is independent from KeY
     * and provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessExceptionalMethodReturn extends
            AbstractKeYlessBaseExecutionNode<Throw> implements IExecutionExceptionalMethodReturn {
        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param signature The signature.
         * @param formatedMethodReturn The formated method return condition.
         */
        public KeYlessExceptionalMethodReturn(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged, String signature,
                String formatedMethodReturn) {
            super(parent, name, formatedPathCondition, pathConditionChanged, signature,
                formatedMethodReturn);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Exceptional Method Return";
        }
    }

    /**
     * An implementation of {@link IExecutionMethodReturn} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessMethodReturn extends AbstractKeYlessBaseExecutionNode<SourceElement>
            implements IExecutionMethodReturn {
        /**
         * The name including the return value.
         */
        private final String nameIncludingReturnValue;

        /**
         * The signature including the return value.
         */
        private final String signatureIncludingReturnValue;

        /**
         * Defines if the return value is computed or not.
         */
        private final boolean returnValueComputed;

        /**
         * The possible return values.
         */
        private final List<IExecutionMethodReturnValue> returnValues =
            new LinkedList<>();

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param formatedPathCondition The formated path condition.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param nameIncludingReturnValue The name including the return value.
         * @param signature The signature.
         * @param signatureIncludingReturnValue The signature including return value.
         * @param returnValueComputed Is the return value computed?
         * @param formatedMethodReturn The formated method return condition.
         */
        public KeYlessMethodReturn(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged,
                String nameIncludingReturnValue, String signature,
                String signatureIncludingReturnValue, boolean returnValueComputed,
                String formatedMethodReturn) {
            super(parent, name, formatedPathCondition, pathConditionChanged, signature,
                formatedMethodReturn);
            this.nameIncludingReturnValue = nameIncludingReturnValue;
            this.signatureIncludingReturnValue = signatureIncludingReturnValue;
            this.returnValueComputed = returnValueComputed;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getNameIncludingReturnValue() throws ProofInputException {
            return nameIncludingReturnValue;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getSignatureIncludingReturnValue() throws ProofInputException {
            return signatureIncludingReturnValue;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Method Return";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isReturnValuesComputed() {
            return returnValueComputed;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionMethodReturnValue[] getReturnValues() throws ProofInputException {
            return returnValues.toArray(new IExecutionMethodReturnValue[0]);
        }

        /**
         * Adds the given {@link IExecutionMethodReturnValue}.
         *
         * @param returnValue The {@link IExecutionMethodReturnValue} to add.
         */
        public void addReturnValue(IExecutionMethodReturnValue returnValue) {
            returnValues.add(returnValue);
        }
    }

    /**
     * An implementation of {@link IExecutionMethodReturn} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessMethodReturnValue extends AbstractKeYlessExecutionElement
            implements IExecutionMethodReturnValue {
        /**
         * The human readable return value.
         */
        private final String returnValueString;

        /**
         * Is a condition available?
         */
        private final boolean hasCondition;

        /**
         * The optional human readable condition.
         */
        private final String conditionString;

        /**
         * Constructor.
         *
         * @param name The name of this node.
         * @param returnValueString The human readable return value.
         * @param hasCondition Is a condition available?
         * @param conditionString The optional human readable condition.
         */
        public KeYlessMethodReturnValue(String name, String returnValueString, boolean hasCondition,
                String conditionString) {
            super(name);
            this.returnValueString = returnValueString;
            this.hasCondition = hasCondition;
            this.conditionString = conditionString;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Return Value";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getReturnValue() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getReturnValueString() throws ProofInputException {
            return returnValueString;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean hasCondition() throws ProofInputException {
            return hasCondition;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getCondition() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getConditionString() throws ProofInputException {
            return conditionString;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public PosInOccurrence getModalityPIO() {
            return null;
        }
    }

    /**
     * An implementation of {@link IExecutionStatement} which is independent from KeY and provides
     * such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessStatement extends AbstractKeYlessExecutionNode<SourceElement>
            implements IExecutionStatement {
        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param formatedPathCondition The formated path condition.
         */
        public KeYlessStatement(IExecutionNode<?> parent, String name, String formatedPathCondition,
                boolean pathConditionChanged) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Statement";
        }
    }

    /**
     * An implementation of {@link IExecutionJoin} which is independent from KeY and provides such
     * only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessJoin extends AbstractKeYlessExecutionNode<SourceElement>
            implements IExecutionJoin {
        /**
         * Is the weakening verified?
         */
        private final boolean weakeningVerified;

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param formatedPathCondition The formated path condition.
         * @param weakeningVerified Is the weakening verified?
         */
        public KeYlessJoin(IExecutionNode<?> parent, String name, String formatedPathCondition,
                boolean pathConditionChanged, boolean weakeningVerified) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
            this.weakeningVerified = weakeningVerified;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Join";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isWeakeningVerified() {
            return weakeningVerified;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isWeakeningVerificationSupported() {
            return true;
        }
    }

    /**
     * An implementation of {@link IExecutionOperationContract} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessOperationContract extends AbstractKeYlessExecutionNode<SourceElement>
            implements IExecutionOperationContract {
        /**
         * Is precondition complied?
         */
        private final boolean preconditionComplied;

        /**
         * Has not null check?
         */
        private final boolean hasNotNullCheck;

        /**
         * Is not null check complied?
         */
        private final boolean notNullCheckComplied;

        /**
         * The formated result term.
         */
        private final String formatedResultTerm;

        /**
         * The formated exception term.
         */
        private final String formatedExceptionTerm;

        /**
         * The formated self term.
         */
        private final String formatedSelfTerm;

        /**
         * The formated contract parameters.
         */
        private final String formatedContractParams;

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param formatedPathCondition The formated path condition.
         * @param preconditionComplied Is precondition complied?
         * @param hasNotNullCheck Has not null check?
         * @param notNullCheckComplied Is not null check complied?
         * @param formatedResultTerm The formated result term.
         * @param formatedExceptionTerm The formated exception term.
         * @param formatedSelfTerm The formated self term.
         * @param formatedContractParams The formated contract parameters.
         */
        public KeYlessOperationContract(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged,
                boolean preconditionComplied, boolean hasNotNullCheck, boolean notNullCheckComplied,
                String formatedResultTerm, String formatedExceptionTerm, String formatedSelfTerm,
                String formatedContractParams) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
            this.preconditionComplied = preconditionComplied;
            this.hasNotNullCheck = hasNotNullCheck;
            this.notNullCheckComplied = notNullCheckComplied;
            this.formatedResultTerm = formatedResultTerm;
            this.formatedExceptionTerm = formatedExceptionTerm;
            this.formatedSelfTerm = formatedSelfTerm;
            this.formatedContractParams = formatedContractParams;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Operation Contract";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Contract getContract() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IProgramMethod getContractProgramMethod() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isPreconditionComplied() {
            return preconditionComplied;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean hasNotNullCheck() {
            return hasNotNullCheck;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isNotNullCheckComplied() {
            return notNullCheckComplied;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getResultTerm() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getExceptionTerm() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getFormatedResultTerm() throws ProofInputException {
            return formatedResultTerm;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getFormatedExceptionTerm() throws ProofInputException {
            return formatedExceptionTerm;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getSelfTerm() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ImmutableList<Term> getContractParams() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getFormatedSelfTerm() throws ProofInputException {
            return formatedSelfTerm;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getFormatedContractParams() throws ProofInputException {
            return formatedContractParams;
        }
    }

    /**
     * An implementation of {@link IExecutionLoopInvariant} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessLoopInvariant extends AbstractKeYlessExecutionNode<SourceElement>
            implements IExecutionLoopInvariant {
        /**
         * Initially valid?
         */
        private final boolean initiallyValid;

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param formatedPathCondition The formated path condition.
         * @param initiallyValid Initially valid?
         */
        public KeYlessLoopInvariant(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged,
                boolean initiallyValid) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
            this.initiallyValid = initiallyValid;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Loop Invariant";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public LoopSpecification getLoopInvariant() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public While getLoopStatement() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isInitiallyValid() {
            return initiallyValid;
        }
    }

    /**
     * An implementation of {@link IExecutionAuxiliaryContract} which is independent from KeY and
     * provides such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessBlockContract extends AbstractKeYlessExecutionNode<SourceElement>
            implements IExecutionAuxiliaryContract {
        /**
         * Precondition complied?
         */
        private final boolean preconditionComplied;

        /**
         * Constructor.
         *
         * @param parent The parent {@link IExecutionNode}.
         * @param name The name of this node.
         * @param pathConditionChanged Is the path condition changed compared to parent?
         * @param formatedPathCondition The formated path condition.
         * @param preconditionComplied Precondition complied?
         */
        public KeYlessBlockContract(IExecutionNode<?> parent, String name,
                String formatedPathCondition, boolean pathConditionChanged,
                boolean preconditionComplied) {
            super(parent, name, formatedPathCondition, pathConditionChanged);
            this.preconditionComplied = preconditionComplied;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Block Contract";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public BlockContract getContract() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public StatementBlock getBlock() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isPreconditionComplied() {
            return preconditionComplied;
        }
    }

    /**
     * An implementation of {@link IExecutionConstraint} which is independent from KeY and provides
     * such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessConstraint extends AbstractKeYlessExecutionElement
            implements IExecutionConstraint {
        /**
         * Constructor.
         *
         * @param name The name.
         */
        public KeYlessConstraint(String name) {
            super(name);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Constraint";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getTerm() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public PosInOccurrence getModalityPIO() {
            return null;
        }
    }

    /**
     * An implementation of {@link IExecutionVariable} which is independent from KeY and provides
     * such only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessVariable extends AbstractKeYlessExecutionElement
            implements IExecutionVariable {
        /**
         * The parent {@link IExecutionValue} if available.
         */
        private final IExecutionValue parentValue;

        /**
         * The is array flag.
         */
        private final boolean isArrayIndex;

        /**
         * The array index.
         */
        private final String arrayIndexString;

        /**
         * The contained values.
         */
        private final List<IExecutionValue> values = new LinkedList<>();

        /**
         * Constructor.
         *
         * @param parentValue The parent {@link IExecutionValue} if available.
         * @param isArrayIndex The is array flag.
         * @param arrayIndexString The array index.
         * @param name The name.
         */
        public KeYlessVariable(IExecutionValue parentValue, boolean isArrayIndex,
                String arrayIndexString, String name) {
            super(name);
            this.parentValue = parentValue;
            this.isArrayIndex = isArrayIndex;
            this.arrayIndexString = arrayIndexString;
        }

        /**
         * Adds the given child {@link IExecutionValue}.
         *
         * @param variable The child {@link IExecutionValue} to add.
         */
        public void addValue(IExecutionValue variable) {
            values.add(variable);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionValue getParentValue() {
            return parentValue;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionValue[] getValues() {
            return values.toArray(new IExecutionValue[0]);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getArrayIndex() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getArrayIndexString() {
            return arrayIndexString;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isArrayIndex() {
            return isArrayIndex;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IProgramVariable getProgramVariable() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Variable";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getAdditionalCondition() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public PosInOccurrence getModalityPIO() {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term createSelectTerm() {
            return null;
        }
    }

    /**
     * An implementation of {@link IExecutionLink} which is independent from KeY and provides such
     * only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYLessLink implements IExecutionLink {
        /**
         * The source.
         */
        private IExecutionNode<?> source;

        /**
         * The target.
         */
        private IExecutionNode<?> target;

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionNode<?> getSource() {
            return source;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionNode<?> getTarget() {
            return target;
        }

        /**
         * Sets the source.
         *
         * @param source The source to set.
         */
        public void setSource(IExecutionNode<?> source) {
            this.source = source;
        }

        /**
         * Sets the target.
         *
         * @param target The target to set.
         */
        public void setTarget(IExecutionNode<?> target) {
            this.target = target;
        }
    }

    /**
     * An implementation of {@link IExecutionValue} which is independent from KeY and provides such
     * only children and default attributes.
     *
     * @author Martin Hentschel
     */
    public static class KeYlessValue extends AbstractKeYlessExecutionElement
            implements IExecutionValue {
        /**
         * The parent {@link IExecutionVariable} if available.
         */
        private final IExecutionVariable variable;

        /**
         * The type string.
         */
        private final String typeString;

        /**
         * The value string.
         */
        private final String valueString;

        /**
         * Is the value unknown?
         */
        private final boolean valueUnknown;

        /**
         * Is the value an object?
         */
        private final boolean valueAnObject;

        /**
         * The child variables.
         */
        private final List<IExecutionVariable> childVariables =
            new LinkedList<>();

        /**
         * The condition as {@link String}.
         */
        private final String conditionString;

        /**
         * The related {@link IExecutionConstraint}s.
         */
        private final List<IExecutionConstraint> constraints =
            new LinkedList<>();

        /**
         * Constructor.
         *
         * @param variable The parent {@link IExecutionVariable}.
         * @param typeString The type string.
         * @param valueString The value string.
         * @param name The name.
         * @param valueUnknown Is the value unknown?
         * @param valueAnObject Is the value an object?
         * @param conditionString The condition as human readable {@link String}.
         */
        public KeYlessValue(IExecutionVariable variable, String typeString, String valueString,
                String name, boolean valueUnknown, boolean valueAnObject, String conditionString) {
            super(name);
            this.variable = variable;
            this.typeString = typeString;
            this.valueString = valueString;
            this.valueUnknown = valueUnknown;
            this.valueAnObject = valueAnObject;
            this.conditionString = conditionString;
        }

        /**
         * Adds the given child {@link IExecutionVariable}.
         *
         * @param variable The child {@link IExecutionVariable} to add.
         */
        public void addChildVariable(IExecutionVariable variable) {
            childVariables.add(variable);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getValueString() throws ProofInputException {
            return valueString;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getTypeString() {
            return typeString;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getConditionString() throws ProofInputException {
            return conditionString;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionVariable getVariable() {
            return variable;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionVariable[] getChildVariables() {
            return childVariables.toArray(new IExecutionVariable[0]);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isValueUnknown() throws ProofInputException {
            return valueUnknown;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean isValueAnObject() throws ProofInputException {
            return valueAnObject;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getValue() throws ProofInputException {
            return null;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getElementType() {
            return "Value";
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Term getCondition() throws ProofInputException {
            return null;
        }

        /**
         * Adds the given {@link IExecutionConstraint}.
         *
         * @param constraint The {@link IExecutionConstraint} to add.
         */
        public void addConstraint(IExecutionConstraint constraint) {
            constraints.add(constraint);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public IExecutionConstraint[] getConstraints() {
            return constraints.toArray(new IExecutionConstraint[0]);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public PosInOccurrence getModalityPIO() {
            return null;
        }
    }
}
