#pragma once

// Debug per probe logic and main probes module

#include "arrays.h"
#include "wire.h"
#include "uintN_t.h"
#include "uint8_t_array_N_t.h"
#include "uint8_t_bytes_t.h"

// UART for data xfers
#include "../uart/uart_mac.c"

// General probes stuff and generated bytes help
#include "debug_probes.h"
#include "probes_cmd_t_bytes_t.h"

// FSM to wait for a command over uart
typedef struct wait_cmd_t
{
  uint1_t rx_ready;
  probes_cmd_t cmd;
  uint1_t done;
}wait_cmd_t;
wait_cmd_t wait_cmd(uint8_t rx, uint1_t rx_valid)
{
  // State regs
  static probes_cmd_t_size_t pos;
  static probes_cmd_t_bytes_t bytes;
  
  // Outputs
  wait_cmd_t o;
  o.done = 0;
  o.rx_ready = 1;
  o.cmd.probe_id = 0;
  
  // Wait for valid data
  if(rx_valid)
  {
    // Shift bytes down to put last byte at top of array
    uint32_t i;
    ARRAY_SHIFT_DOWN(bytes.data, sizeof(probes_cmd_t), 1, i)
    bytes.data[sizeof(probes_cmd_t)-1] = rx;
    
    if(pos<(sizeof(probes_cmd_t)-1))
    {
      pos += 1;
    }
    else
    {
      // last byte, done
      o.done = 1;
      o.cmd = bytes_to_probes_cmd_t(bytes);
      pos = 0;
    }
  }
  return o;  
}

// Each probe has some FSMs
typedef enum probe_state_t
{
  WAIT_EN, // for read/write en
  DATA_TRANSFER // transfer rd/wr data out/in
}probe_state_t;

// Macro for the funcs needed per probe
#define PROBE_FUNCS(num)\
/* FSM to transfer probe bytes by writing into probe byte fifo */ \
typedef struct probe##num##_xfer_t \
{ \
  uint8_t probe_bytes[sizeof(probe##num##_t)]; \
  uint1_t done; \
}probe##num##_xfer_t; \
probe##num##_xfer_t probe##num##_xfer(uint8_t probe_bytes[sizeof(probe##num##_t)]) \
{ \
  static probe##num##_size_t pos; \
  /* Write each byte into 1 byte wide async fifo into uart clock domain*/ \
  \
  probe##num##_xfer_t o; \
  o.done = 0; \
  \
  /* Try to write byte at front of array*/ \
  uint8_t wr_data[1]; \
  wr_data[0] = probe_bytes[0]; \
  uint1_t wr_en = 1; \
  probe##num##_fifo_write_t write = probe##num##_fifo_WRITE_1(wr_data, wr_en); \
  if(write.ready) \
  { \
    /* Shift buffer down, next byte to front*/ \
    uint32_t iter; \
    ARRAY_SHIFT_DOWN(probe_bytes,sizeof(probe##num##_t),1, iter) \
    /* Increment pos */ \
    if(pos<(sizeof(probe##num##_t)-1)) \
    { \
      pos += 1; \
    } \
    else \
    { \
      /* Last byte, done*/ \
      o.done = 1; \
      /* Reset */ \
      pos = 0; \
    } \
  } \
  o.probe_bytes = probe_bytes; \
  return o; \
} \
\
/* FSM to sample to probe when requested and then transfer the data */ \
void probe##num##_process() \
{ \
  static uint8_t probe_bytes[sizeof(probe##num##_t)]; \
  static probe_state_t state; \
  \
  uint1_t rd_en = 0; \
  if(state==WAIT_EN) \
  { \
    /* Same clock wire read*/ \
    probe##num##_t probe; \
    WIRE_READ(probe##num##_t, probe, probe##num) \
    \
    /* Async fifo read of rd_en flag from main scope process at uart clock*/ \
    probe##num##_rd_en_fifo_read_t read = probe##num##_rd_en_fifo_READ_1(1); \
    rd_en = read.data[0] & read.valid; \
    /* Upon read enable begin data transfer*/ \
    if(rd_en) \
    { \
      probe##num##_t_bytes_t bytes = probe##num##_type_to_bytes(probe); \
      probe_bytes = bytes.data; \
      state = DATA_TRANSFER; \
    } \
  } \
  else if(state==DATA_TRANSFER) \
  { \
    /* Start the data xfer fsm and wait for it to complete*/ \
    probe##num##_xfer_t xfer = probe##num##_xfer(probe_bytes); \
    probe_bytes = xfer.probe_bytes; \
    if(xfer.done) \
    { \
      state = WAIT_EN; \
    } \
  } \
  \
  WIRE_WRITE(uint1_t, probe##num##_rd_en, rd_en) \
}

// PROBES STUFF (for now copy-paste per probe)
// 0
#include "probe0.h"
probe0_t probe0;
#include "probe0_clock_crossing.h"
uint1_t probe0_rd_en;
#include "probe0_rd_en_clock_crossing.h"
uint8_t probe0_fifo[1]; 
#include "probe0_fifo_clock_crossing.h"
uint1_t probe0_rd_en_fifo[1];
#include "probe0_rd_en_fifo_clock_crossing.h"
PROBE_FUNCS(0)
// 1
#include "probe1.h"
probe1_t probe1;
#include "probe1_clock_crossing.h"
uint1_t probe1_rd_en;
#include "probe1_rd_en_clock_crossing.h"
uint8_t probe1_fifo[1]; 
#include "probe1_fifo_clock_crossing.h"
uint1_t probe1_rd_en_fifo[1];
#include "probe1_rd_en_fifo_clock_crossing.h"
PROBE_FUNCS(1)

// Macros use in main probes process based on above and below copy pasta

// Send a read enable request to probe
#define PROBE_RD_REQ(num)\
probe##num##_rd_en_fifo_write_t write##num = probe##num##_rd_en_fifo_WRITE_1(rd_en_fifo_wr_data,1); \
if(write##num.ready) \
{ \
  state = PROBE##num##_RD_RESP; \
}

// Send the bytes out over uart 
//(TODO refactor to compare byte_pos to reg instead of repeated compare to constants)
#define PROBE_RD_RESP(num)\
/* Try to read a byte from probe data fifo if uart tx ready */ \
probe##num##_fifo_read_t read##num = probe##num##_fifo_READ_1(data_fifo_rd_en); \
/* Send data into tx uart */ \
tx_word.data = read##num.data[0]; \
tx_word.valid = read##num.valid; \
/* Count off as many bytes as need to be transfered */ \
if(data_fifo_rd_en & tx_word.valid) \
{ \
  if(byte_pos<(sizeof(probe##num##_t)-1)) \
  { \
    byte_pos += 1; \
  } \
  else \
  { \
    /* last byte, done */ \
    state = WAIT_CMD; \
  } \
}

// Main probes fsm
typedef enum probes_state_t
{
  WAIT_CMD,
  PROBE0_RD_REQ,
  PROBE0_RD_RESP,
  //
  PROBE1_RD_REQ,
  PROBE1_RD_RESP
}probes_state_t;
// Probes main module on uart clock for now
#pragma MAIN_MHZ probes uart_module
void probes()
{
  // State regs
  static probes_state_t state;
  static uint32_t byte_pos;
  
  // Read uart mac input signals, rx and tx
  // (shared between states)
  // Read stream of 8b words from mac
  uart_mac_s rx_word;
  WIRE_READ(uart_mac_s, rx_word, uart_rx_mac_word_out)
  // And if tx is ready
  uint1_t tx_ready;
  WIRE_READ(uint1_t, tx_ready, uart_tx_mac_in_ready)
  
  // uart rx and tx output signals
  uint1_t rx_ready = 0;
  uart_mac_s tx_word;
  tx_word.data = 0;
  tx_word.valid = 0;
  
  // Read enable fifo write data constant
  uint1_t rd_en_fifo_wr_data[1];
  rd_en_fifo_wr_data[0] = 1;
  
  // Fifo read when tx ready
  uint1_t data_fifo_rd_en = tx_ready;
  
  if(state==WAIT_CMD)
  {
    // Start the wait for cmd fsm and wait for it to complete
    wait_cmd_t wait_data = wait_cmd(rx_word.data, rx_word.valid);
    rx_ready = wait_data.rx_ready;
    if(wait_data.done)
    {
      // What command to do? (specific to if probe is read or write)
      if     (wait_data.cmd.probe_id == 0) state = PROBE0_RD_REQ;
      else if(wait_data.cmd.probe_id == 1) state = PROBE1_RD_REQ;
    }
    byte_pos = 0;
  }
  // PROBES
  else if(state==PROBE0_RD_REQ) {  PROBE_RD_REQ(0) }
  else if(state==PROBE0_RD_RESP){ PROBE_RD_RESP(0) }
  //
  else if(state==PROBE1_RD_REQ) {  PROBE_RD_REQ(1) }
  else if(state==PROBE1_RD_RESP){ PROBE_RD_RESP(1) }
  
  // Write uart rx and tx output signals
  WIRE_WRITE(uint1_t, uart_rx_mac_out_ready, rx_ready) 
  WIRE_WRITE(uart_mac_s, uart_tx_mac_word_in, tx_word)
}
