#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <math.h>

// Build like:
// g++ mnist_eth_pixels_update.c -I ~/pipelinec_output -o mnist_eth_pixels_update

#include "mnist-neural-network-plain-c-master/mnist_file.c"
#include "mnist-neural-network-plain-c-master/neural_network.c"

/**
 * Downloaded from: http://yann.lecun.com/exdb/mnist/
 */
const char * train_images_file = "mnist-neural-network-plain-c-master/data/train-images-idx3-ubyte";
const char * train_labels_file = "mnist-neural-network-plain-c-master/data/train-labels-idx1-ubyte";
const char * test_images_file = "mnist-neural-network-plain-c-master/data/t10k-images-idx3-ubyte";
const char * test_labels_file = "mnist-neural-network-plain-c-master/data/t10k-labels-idx1-ubyte";

// 'Software' side of ethernet
#include "../eth/eth_sw.c"

// How to send inputs and outputs over eth
#include "pixels_update.h"
#include "type_bytes_t.h/pixels_update_t_bytes_t.h/pixels_update_t_bytes.h" // Autogenerated
void write_update(pixels_update_t* input)
{
 // Copy into buffer
 uint8_t buffer[pixels_update_t_SIZE];
 pixels_update_t_to_bytes(input, buffer);
 // Send buffer
 eth_write(buffer, pixels_update_t_SIZE);  
}
#include "pred_resp.h"
#include "type_bytes_t.h/pred_resp_t_bytes_t.h/pred_resp_t_bytes.h" // Autogenerated
void read_prediction(pred_resp_t* output)
{
  // Read buffer
  uint8_t buffer[pred_resp_t_SIZE];
  size_t rd_size = pred_resp_t_SIZE;
  eth_read(buffer, &rd_size);
  if(rd_size != pred_resp_t_SIZE)
  {
    printf("Did not receive enough bytes! Expected %d, got %ld\n",pred_resp_t_SIZE,rd_size);
  }
  // Copy from buffer
  bytes_to_pred_resp_t(buffer,output);
}

int read_prediction_until_correct(pred_resp_t* output, uint8_t test_label)
{
    int correct = 0;
    int count = 0;
    while(!correct && count < 200) // Rough estimate for test
    {
        read_prediction(output);
        correct = output->pred == test_label;
        count++;
    }
    return correct ? 0 : 1;
}

int main(int argc, char *argv[])
{
    mnist_dataset_t * train_dataset, * test_dataset;
    // Init trained network
    neural_network_t network;
    float weight[MNIST_LABELS*MNIST_IMAGE_SIZE] = 
        #include "trained/weights.c"
    ;
    float bias[MNIST_LABELS] = 
        #include "trained/biases.c"
    ;
    float activations[MNIST_LABELS];
    int i, j;
    for (i = 0; i < MNIST_LABELS; i++) {
        network.b[i] = bias[i];

        for (j = 0; j < MNIST_IMAGE_SIZE; j++) {
            network.W[i][j] = weight[i*MNIST_IMAGE_SIZE + j];
        }
    }

    // Read the datasets from the files
    train_dataset = mnist_get_dataset(train_images_file, train_labels_file);
    test_dataset = mnist_get_dataset(test_images_file, test_labels_file);

    // Init msgs to/from FPGA
    init_eth();
    
    printf("Sending images...\n");

    int test_image_num;
    for(test_image_num=0;test_image_num<128;test_image_num++)
    {
        // Load and image and do software inference to compare to
        mnist_image_t test_image = test_dataset->images[test_image_num];
        uint8_t test_label = test_dataset->labels[test_image_num];
        neural_network_hypothesis(&test_image, &network, activations);
        int predict = 0;
        float max_activation = activations[0];
        for (j = 0; j < MNIST_LABELS; j++) {
            if (max_activation < activations[j]) {
                max_activation = activations[j];
                predict = j;
            }
        }
        //printf("# Inference predicts: %d\n", predict);
        //printf("# Pixels for test image labeled as: %d\n", test_label);
        /*printf("pixels = [\n");
        for (i = 0; i < MNIST_IMAGE_SIZE; i++) {
            uint8_t p = test_image.pixels[i];
            printf("%d, ", p);
        }
        printf("\n");*/

        // Write each pixel update
        uint16_t addr;
        for (addr = 0; addr < MNIST_IMAGE_SIZE; addr += N_PIXELS_PER_UPDATE) {
            pixels_update_t update;
            update.addr = addr;
            int i;
            for(i=0;i<N_PIXELS_PER_UPDATE;i++)
            {
                update.pixels[i] = test_image.pixels[addr+i];
            }
            write_update(&update);
            usleep(0); // Temp for no overflow
        }

        // Read responses until got matching one
        pred_resp_t resp;
        if(read_prediction_until_correct(&resp, predict))
        {
            printf("Failed prediction for image num %d, predicted %d, was %d\n",
                    test_image_num, resp.pred, predict);
        }
    }

    printf("Done %d images\n", test_image_num);

    // Close eth to/from FPGA
	close_eth();

    // Cleanup
    mnist_free_dataset(train_dataset);
    mnist_free_dataset(test_dataset);

    return 0;
}
