#pragma PART "xc7a35ticsg324-1l"

#include "wire.h"
#include "uintN_t.h"

// Include I2S from PMOD
#include "i2s_pmod.c"

/*
https://reference.digilentinc.com/pmod/pmodi2s2/reference-manual
  Quick Start
  To set up a simple 44.1 kHz audio passthrough, three control signals need to be generated by the host system board.
  1. A master clock (MCLK) at a frequency of approximately 22.579 MHz.
  2. A serial clock (SCLK), which fully toggles once per 8 MCLK periods.
  3. A Left/Right Word Select signal, which fully toggles once per 64 SCLK periods.
  The Pmod I2S2's Master/Slave select jumper (JP1) should be placed into the Slave (SLV) position.
  Each of these control signals should be provided to the appropriate pin on both the top and bottom rows of the Pmod I2S2.
  The ADOUT_SDIN pin should be driven by the ADIN_SDOUT signal.
*/
#define SCLK_PERIOD_MCLKS 8
#define LR_PERIOD_SCLKS 64
#pragma MAIN_MHZ app 22.579
void app()
{
  // Registers
  static uint3_t sclk_counter;
  static uint1_t sclk;
  static uint6_t lr_counter;
  static uint1_t lr;
  
  // Read the incoming I2S signals
  i2s_to_app_t from_i2s = read_i2s_pmod();
  
  // Outgoing I2S signals
  app_to_i2s_t to_i2s;
  
  // Basic loopback:
  // Only input is data, connected to output data
  to_i2s.tx_data = from_i2s.rx_data;
  // Outputs clks from registers
  to_i2s.tx_sclk = sclk;
  to_i2s.rx_sclk = sclk;
  to_i2s.tx_lrck = lr;
  to_i2s.rx_lrck = lr;
  
  // Drive I2S clocking derived from current MCLK domain
  
  // SCLK toggling at half period count
  uint1_t sclk_half_toggle = sclk_counter==((SCLK_PERIOD_MCLKS/2)-1);
  // 0->1 SCLK once per period rising edge
  uint1_t sclk_period_toggle = sclk_half_toggle & (sclk==0); 
  if(sclk_half_toggle)
  {
    // Do toggle and reset counter
    sclk = !sclk;
    sclk_counter = 0;
  }
  else
  {
    // No toggle yet, keep counting
    sclk_counter += 1;
  }
  
  // LR toggling happens per SCLK period 
  if(sclk_period_toggle)
  {
    // LR toggling at half period count
    if(lr_counter==((LR_PERIOD_SCLKS/2)-1))
    {
      // Do toggle and reset counter
      lr = !lr;
      lr_counter = 0;
    }
    else
    {
      // No toggle yet, keep counting
      lr_counter += 1;
    }
  }
  
  // Drive the outgoing I2S signals
  write_i2s_pmod(to_i2s);
}
