using System;
using System.Collections.Generic;
using Godot;
using Mirage.Collections;
using Mirage.Logging;
using Mirage.RemoteCalls;
using Mirage.Serialization;

namespace Mirage
{
    /// <summary>
    /// Default NetworkBehaviour that can be used to quickly add RPC and Sync var to nodes
    /// <para>
    /// If you dont need RPC or SyncVar use <see cref="INetworkNode"/> instead
    /// </para>
    /// </summary>
    public abstract partial class NetworkBehaviour : Node, INetworkNode, INetworkNodeWithRpc, INetworkNodeWithSettings, INetworkNodeWithSyncVar
    {
        private static readonly ILogger logger = LogFactory.GetLogger(typeof(NetworkBehaviour));

        public const int COMPONENT_INDEX_NOT_FOUND = -1;

        [ExportGroup("Sync Settings")]
        [Export(PropertyHint.Flags)] public SyncFrom From = SyncSettings.Default.From;
        [Export(PropertyHint.Flags)] public SyncTo To = SyncSettings.Default.To;
        [Export(PropertyHint.Flags)] public SyncTiming Timing = SyncSettings.Default.Timing;
        [Export] public float Interval = SyncSettings.Default.Interval;

        private NetworkIdentity _identity;
        private int? _componentIndex;
        public NetworkNodeSyncVars NetworkNodeSyncVars;
        private readonly List<ISyncObject> syncObjects = new List<ISyncObject>();

        List<ISyncObject> INetworkNodeWithSyncVar.SyncObjects => syncObjects;

        public NetworkIdentity Identity
        {
            get
            {
                if (_identity == null)
                {
                    _identity = this.TryGetNetworkIdentity();
                }
                return _identity;
            }
        }

        /// <summary>
        /// Returns the index of the component on this object
        /// </summary>
        public int ComponentIndex
        {
            get
            {
                if (_componentIndex.HasValue)
                    return _componentIndex.Value;

                // note: FindIndex causes allocations, we search manually instead
                for (var i = 0; i < Identity.NetworkBehaviours.Length; i++)
                {
                    var component = Identity.NetworkBehaviours[i];
                    if (component == this)
                    {
                        _componentIndex = i;
                        return i;
                    }
                }

                // this should never happen
                logger.LogError("Could not find component in GameObject. You should not add/remove components in networked objects dynamically");

                return COMPONENT_INDEX_NOT_FOUND;
            }
        }

        public SyncSettings SyncSettings => new SyncSettings(From, To, Timing, Interval);

        public virtual int GetRpcCount()
        {
            // generated by weaver
            return 0;
        }

        public virtual void RegisterRpc(RemoteCallCollection remoteCallCollection)
        {
            // generated by weaver
        }

        public virtual bool SerializeSyncVars(NetworkWriter writer, bool initial)
        {
            // generated by weaver
            return false;
        }

        public virtual void DeserializeSyncVars(NetworkReader reader, bool initial)
        {
            // generated by weaver
        }

        /// <summary>
        /// this gets called in the constructor by the weaver
        /// for every SyncObject in the component (e.g. SyncLists).
        /// We collect all of them and we synchronize them with OnSerialize/OnDeserialize 
        /// </summary>
        /// <param name="syncObject"></param>
        protected internal void InitSyncObject(ISyncObject syncObject)
        {
            syncObjects.Add(syncObject);
        }

        public struct Id : IEquatable<Id>
        {
            public readonly uint NetId;
            public readonly int ComponentIndex;

            public Id(uint netId, int componentIndex)
            {
                NetId = netId;
                ComponentIndex = componentIndex;
            }

            public Id(NetworkBehaviour behaviour)
            {
                NetId = behaviour.Identity.NetId;
                ComponentIndex = behaviour.ComponentIndex;
            }

            public override int GetHashCode()
            {
                return ((int)NetId * 256) + ComponentIndex;
            }

            public bool Equals(Id other)
            {
                return (NetId == other.NetId) && (ComponentIndex == other.ComponentIndex);
            }

            public override bool Equals(object obj)
            {
                if (obj is Id id)
                    return Equals(id);
                else
                    return false;
            }
        }
    }
}
