/*=========================================================================
 *
 *  Copyright NumFOCUS
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *         https://www.apache.org/licenses/LICENSE-2.0.txt
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *=========================================================================*/
#include "itkImageFileWriter.h"
#include "itkImageFileReader.h"
#include "itkMRCImageIO.h"
#include "itkMRCImageIOFactory.h"
#include "itkTestingMacros.h"
#include "itkVectorImage.h"
#include "itkMetaDataObject.h"
#include "itkTestingHashImageFilter.h"


static_assert(sizeof(itk::MRCHeaderObject::Header) == 1024, " Ill defined MRC Header struct");

namespace
{

template <typename TImageType>
bool
Test(const std::string & inFileName, const std::string & outFileName, const std::string & md5)
{
  using ImageType = TImageType;
  using ImageFileReaderType = itk::ImageFileReader<ImageType>;

  auto reader = ImageFileReaderType::New();
  reader->SetFileName(inFileName);
  reader->UpdateLargestPossibleRegion();

  const typename ImageType::Pointer image = reader->GetOutput();

  using DictionaryType = itk::MetaDataDictionary;
  using MetaDataStringType = itk::MetaDataObject<std::string>;

  // prepare to iterate over the dictionary
  DictionaryType & dic = image->GetMetaDataDictionary();

  DictionaryType::ConstIterator       itr = dic.Begin();
  const DictionaryType::ConstIterator end = dic.End();

  while (itr != end)
  {
    const std::string key = itr->first;

    // check to see if we have a MRC Header
    if (itr->first == itk::MRCImageIO::m_MetaDataHeaderName)
    {
      itk::MRCHeaderObject::ConstPointer header;
      if (itk::ExposeMetaData(dic, itk::MRCImageIO::m_MetaDataHeaderName, header))
      {
        std::cout << "MRC Header: " << std::endl;
        std::cout << header;

        // Use DeepCopy method to improve coverage
        const itk::MRCHeaderObject::Pointer copyHeader = itk::MRCHeaderObject::New();
        copyHeader->DeepCopy(header);
      }
    }
    else
    {
      // just print the strings now
      const itk::MetaDataObjectBase::Pointer entry = itr->second;

      const MetaDataStringType::Pointer entryvalue = dynamic_cast<MetaDataStringType *>(entry.GetPointer());
      if (entryvalue)
      {

        const std::string tagvalue = entryvalue->GetMetaDataObjectValue();

        std::cout << '(' << key << ") ";
        std::cout << " = " << tagvalue << std::endl;
      }
    }
    ++itr;
  }

  using HashFilter = itk::Testing::HashImageFilter<ImageType>;
  auto hasher = HashFilter::New();
  hasher->SetInput(image);
  hasher->Update();

  ITK_TEST_EXPECT_EQUAL(md5, hasher->GetHash());

  using WriterType = itk::ImageFileWriter<ImageType>;
  auto writer = WriterType::New();
  writer->SetInput(image);
  writer->SetFileName(outFileName);
  writer->Update();

  return EXIT_SUCCESS;
}

} // namespace

int
itkMRCImageIOTest2(int argc, char * argv[])
{

  if (argc < 4)
  {
    std::cerr << "Missing Parameters." << std::endl;
    std::cerr << "Usage: " << itkNameOfTestExecutableMacro(argv) << " inputFileName outputFilename md5hash"
              << std::endl;
    return EXIT_FAILURE;
  }

  itk::MRCImageIOFactory::RegisterOneFactory();

  const std::string inputFileName = argv[1];
  const std::string outputFileName = argv[2];
  const std::string md5hash = argv[3];

  return Test<itk::VectorImage<unsigned char, 3>>(inputFileName, outputFileName, md5hash);
}
