/*=========================================================================
 *
 *  Copyright NumFOCUS
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *         https://www.apache.org/licenses/LICENSE-2.0.txt
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *=========================================================================*/
#ifndef itkFullToHalfHermitianImageFilter_hxx
#define itkFullToHalfHermitianImageFilter_hxx


#include "itkImageAlgorithm.h"
#include "itkProgressReporter.h"

namespace itk
{

template <typename TInputImage>
FullToHalfHermitianImageFilter<TInputImage>::FullToHalfHermitianImageFilter()
{
  this->SetActualXDimensionIsOdd(false);
  this->DynamicMultiThreadingOn();
}

template <typename TInputImage>
void
FullToHalfHermitianImageFilter<TInputImage>::GenerateOutputInformation()
{
  // Call the superclass' implementation of this method
  Superclass::GenerateOutputInformation();

  // Get pointers to the input and output
  const typename InputImageType::ConstPointer inputPtr = this->GetInput();
  const typename OutputImageType::Pointer     outputPtr = this->GetOutput();

  if (!inputPtr || !outputPtr)
  {
    return;
  }

  const typename InputImageType::SizeType &  inputSize = inputPtr->GetLargestPossibleRegion().GetSize();
  const typename InputImageType::IndexType & inputStartIndex = inputPtr->GetLargestPossibleRegion().GetIndex();

  typename OutputImageType::SizeType  outputSize;
  typename OutputImageType::IndexType outputStartIndex;

  for (unsigned int i = 0; i < OutputImageType::ImageDimension; ++i)
  {
    outputSize[i] = inputSize[i];
    outputStartIndex[i] = inputStartIndex[i];
  }
  outputSize[0] = (inputSize[0] / 2) + 1;

  const typename OutputImageType::RegionType outputLargestPossibleRegion(outputStartIndex, outputSize);

  outputPtr->SetLargestPossibleRegion(outputLargestPossibleRegion);
  this->SetActualXDimensionIsOdd(inputSize[0] % 2 != 0);
}

template <typename TInputImage>
void
FullToHalfHermitianImageFilter<TInputImage>::GenerateInputRequestedRegion()
{
  Superclass::GenerateInputRequestedRegion();

  // Get pointers to the input and output
  const typename InputImageType::Pointer inputPtr = const_cast<InputImageType *>(this->GetInput());
  if (inputPtr)
  {
    inputPtr->SetRequestedRegionToLargestPossibleRegion();
  }
}

template <typename TInputImage>
void
FullToHalfHermitianImageFilter<TInputImage>::DynamicThreadedGenerateData(
  const OutputImageRegionType & outputRegionForThread)
{
  const typename InputImageType::ConstPointer inputPtr = this->GetInput();
  const typename OutputImageType::Pointer     outputPtr = this->GetOutput();

  if (!inputPtr || !outputPtr)
  {
    return;
  }

  // Copy the non-reflected region.
  ImageAlgorithm::Copy(inputPtr.GetPointer(), outputPtr.GetPointer(), outputRegionForThread, outputRegionForThread);
}

} // end namespace itk

#endif // itkFullToHalfHermitianImageFilter_hxx
