package com.andaily.hb.service.operation.network;

import com.andaily.hb.domain.network.NetworkInstance;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.infrastructure.jpa.NetworkInstanceRepository;
import com.andaily.hb.infrastructure.scheduler.DynamicJob;
import com.andaily.hb.infrastructure.scheduler.JobParamManager;
import com.andaily.hb.service.operation.job.MonitoringInstanceJob;
import org.apache.commons.lang.StringUtils;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import static com.andaily.hb.infrastructure.scheduler.DynamicSchedulerFactory.*;

/**
 * 2023-12-28
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
@Component
public class NetworkInstanceEnabler {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetworkInstanceEnabler.class);

    @Autowired
    private  NetworkInstanceRepository instanceRepository;


    public NetworkInstanceEnabler() {
    }

    public boolean enable(String guid) {
        final NetworkInstance instance = instanceRepository.findByGuid(guid);
        if (instance.enabled()) {
            LOGGER.debug("<{}> Instance[guid={}] already enabled, ignore it", username(), instance.guid());
            return false;
        }

        final boolean addSuccessful = startupMonitoringJob(instance);
        if (!addSuccessful) {
            LOGGER.debug("<{}> NOTE: Add MonitoringJob[jobName={}] failed", username(), instance.jobName());
            return false;
        }

        //update
        instance.enabled(true);
        instanceRepository.save(instance);
        LOGGER.debug("<{}> Update NetworkInstance[guid={}] enabled=true,jobName={}", username(), instance.guid(), instance.jobName());

        return true;
    }

    private boolean startupMonitoringJob(NetworkInstance instance) {
        final String jobName = getAndSetJobName(instance);

        DynamicJob job = new DynamicJob(jobName)
                .cronExpression(instance.frequency().getCronExpression())
                .target(MonitoringInstanceJob.class)
                .addJobData(MonitoringInstanceJob.INSTANCE_GUID, instance.guid());

        return executeStartup(instance, job);
    }

    private boolean executeStartup(NetworkInstance instance, DynamicJob job) {
        boolean result = false;
        try {
            if (existJob(job)) {
                result = resumeJob(job);
                LOGGER.debug("<{}> Resume  [{}] by NetworkInstance[guid={},instanceName={}] result: {}", username(), job, instance.guid(), instance.instanceName(), result);
            } else {
                result = registerJob(job);
                LOGGER.debug("<{}> Register  [{}] by NetworkInstance[guid={},instanceName={}] result: {}", username(), job, instance.guid(), instance.instanceName(), result);
            }
        } catch (SchedulerException e) {
            LOGGER.error("<{}> Register [" + job + "] failed", username(), e);
        }
        return result;
    }

    private String getAndSetJobName(NetworkInstance instance) {
        String jobName = instance.jobName();
        if (StringUtils.isEmpty(jobName)) {
            jobName = JobParamManager.generateMonitoringInstanceJobName(instance.guid());
            instance.jobName(jobName);
        }
        return jobName;
    }

    private String username() {
        return SecurityUtils.currentUsername();
    }
}