package com.andaily.hb.service.operation.network;

import com.andaily.hb.domain.network.NetworkInstance;
import com.andaily.hb.infrastructure.jpa.FrequencyMonitorLogRepository;
import com.andaily.hb.infrastructure.jpa.MonitoringReminderLogRepository;
import com.andaily.hb.infrastructure.jpa.NetworkInstanceRepository;
import com.andaily.hb.infrastructure.scheduler.DynamicJob;
import com.andaily.hb.infrastructure.scheduler.DynamicSchedulerFactory;
import org.apache.commons.lang3.StringUtils;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import static com.andaily.hb.domain.shared.security.SecurityUtils.currentUsername;

/**
 * 2023-12-28
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
@Component
public class NetworkInstanceDeleter {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetworkInstanceDeleter.class);
    @Autowired
    private NetworkInstanceRepository instanceRepository;

    @Autowired
    private FrequencyMonitorLogRepository monitorLogRepository;

    @Autowired
    private MonitoringReminderLogRepository reminderLogRepository;


    public NetworkInstanceDeleter() {
    }

    public boolean delete(String guid) {
        final NetworkInstance instance = instanceRepository.findByGuid(guid);
        if (instance.enabled()) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("<{}> Forbid delete enabled NetworkInstance[guid={}]", currentUsername(), instance.guid());
            }
            return false;
        }

        int row = monitorLogRepository.deleteNetworkInstanceLogs(guid);
        int rRow = reminderLogRepository.deleteNetworkInstanceLogs(guid);

        checkAndRemoveJob(instance);

        //logic delete
        instance.archived(true);
        instanceRepository.save(instance);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("<{}> Archive NetworkInstance[guid={}] and FrequencyMonitorLogs  {},MonitoringReminderLogs {}", currentUsername(), instance.guid(), row, rRow);
        }
        return true;
    }

    private void checkAndRemoveJob(NetworkInstance instance) {
        String jobName = instance.jobName();
        if (StringUtils.isEmpty(jobName)) {
            return;
        }
        DynamicJob job = new DynamicJob(jobName);
        try {
            if (DynamicSchedulerFactory.existJob(job)) {
                final boolean result = DynamicSchedulerFactory.removeJob(job);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("<{}> Remove DynamicJob[{}] result [{}]", currentUsername(), job, result);
                }
            }
        } catch (SchedulerException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("<{}> Remove [" + job + "] failed", currentUsername(), e);
            }
        }
    }
}