package com.andaily.hb.service.operation.job;

import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.log.FrequencyMonitorLog;
import com.andaily.hb.domain.log.reminder.PerMonitoringReminder;
import com.andaily.hb.domain.network.NetworkInstance;
import com.andaily.hb.infrastructure.jpa.ApplicationInstanceRepositoryJpa;
import com.andaily.hb.infrastructure.jpa.FrequencyMonitorLogRepository;
import com.andaily.hb.infrastructure.jpa.NetworkInstanceRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 每一次的 心跳监测 执行类
 *
 * @author Shengzhao Li
 */
@Component
public class PerHeartBeatExecutor {

    private static final Logger LOGGER = LoggerFactory.getLogger(PerHeartBeatExecutor.class);

    @Autowired
    private ApplicationInstanceRepositoryJpa instanceRepository;

    @Autowired
    private NetworkInstanceRepository networkInstanceRepository;

    /**
     * @since 3.0.0
     */
    @Autowired
    private FrequencyMonitorLogRepository monitorLogRepository;


    public PerHeartBeatExecutor() {
    }

    /**
     * Execute heart-beat
     * <p/>
     * 1.Send request and checking response
     * 2.Generate FrequencyMonitorLog
     * 3. If failed will notice
     * <p/>
     * <p/>
     * 执行心跳监测的流程
     * 1. 向指定的 URL 发送请求并检测响应情况
     * 2. 记录每一次的监控日志(FrequencyMonitorLog)
     * 3.若状态变更则发送提醒
     */
    public void execute(String instanceGuid) {

        FrequencyMonitorLog monitorLog;
        final ApplicationInstance instance = instanceRepository.findByGuid(instanceGuid);
        //判断是否存在该实例
        if (instance != null) {
            monitorLog = generateMonitorLog(instance);
        } else {
            NetworkInstance networkInstance = networkInstanceRepository.findByGuid(instanceGuid);
            monitorLog = generateMonitorLog(networkInstance);
        }

        monitorLogRepository.save(monitorLog);
        LOGGER.debug("Generate and persist FrequencyMonitorLog[{}]", monitorLog);
        //reminder
        remind(monitorLog);
    }

    /**
     * 处理 监控异常时的 提醒操作
     */
    private void remind(FrequencyMonitorLog monitorLog) {
        PerMonitoringReminder reminder = new PerMonitoringReminder(monitorLog);
        reminder.remind();
    }

    /**
     * 生成 监控日志
     */
    private FrequencyMonitorLog generateMonitorLog(ApplicationInstance instance) {
        FrequencyMonitorLogGenerator monitorLogGenerator = new FrequencyMonitorLogGenerator(instance);
        return monitorLogGenerator.generate();
    }

    /**
     * 生成  NetworkInstance 监控日志
     *
     * @since 3.0.0
     */
    private FrequencyMonitorLog generateMonitorLog(NetworkInstance instance) {
        NetworkFrequencyMonitorLogGenerator monitorLogGenerator = new NetworkFrequencyMonitorLogGenerator(instance);
        return monitorLogGenerator.generate();
    }
}